/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsutils.h"
#include <RdsShareManager>
#include <RdsShare>
#include <RdsFileManager>
#include <QStringList>
#include <RdsLdapSession>
#include <RdsSid>
#include <QDebug>
#include <QDir>
#include <QProcess>
#include <RdsNtAcl>
#include <sys/stat.h>

///@todo: serverize
#ifndef Q_OS_WIN32
#include <pwd.h>
#include <grp.h>
#include <arpa/inet.h>
#include <assert.h>
#endif

RdsUtils::RdsUtils()
{
}


RdsUtils::~RdsUtils()
{
}

ReturnValue RdsUtils::getSystemPath(const QString &path)
{
	PathType type = pathType(path);

	switch (type)
	{
		case System:
			return(normalizePath(path));
			break;
		case Share:
		{
			RdsShareManager mgr;
			ReturnValue ret = mgr.init();
			if (ret.isError()) return(ret);


			ret = getShareName(path);
			if (ret.isError()) return(ret);

			ret = mgr.share(ret.toString());
			if (ret.isError()) return(ret);

			RdsShare share;
			share = ret;

			ret = share.path();
			if (ret.isError()) return(ret);
			QString sharepath = ret.toString();

			ret = getRelativePath(path);
			if (ret.isError()) return(ret);

			return(normalizePath(sharepath + "/" + ret.toString()));
		}
		break;
		case Volume:
		{
			ReturnValue ret = getVolumeName(path);
			if (ret.isError()) return(ReturnValue(1, "Malformed path: " + path));
			QString name = ret.toString();

			ret = getRelativePath(path);
			if (ret.isError()) return(ReturnValue(1, "Malformed path: " + path));
			QString relativepath = ret.toString();

			return(normalizePath(QString("/volumes/%1/%2").arg(name).arg(relativepath)));
		}
		break;
	}
	return(ReturnValue(1, "Failed to detect path type"));
}

ReturnValue RdsUtils::getSharePath(const QString &path)
{
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);

	QString newpath = ret.toString() + "/";
	RdsShareManager mgr;

	ret = mgr.listShares();
	if (ret.isError()) return(ret);

	QStringList results;

	foreach(QString sharename, ret.toStringList())
	{
		ret = mgr.share(sharename);
		if (ret.isError())
		{
			qWarning() << "Failed to get share:" << sharename;
			continue;
		}

		RdsShare share;
		share = ret;

		ret = share.path();
		if (ret.isError())
		{
			qWarning() << "Failed to get path for share:" << sharename << ret.errString();
			continue;
		}

		QString sharepath = normalizePath(ret.toString());

		if (newpath.startsWith(sharepath + "/"))
		{
			QString relativepath = newpath.mid(sharepath.size());
			if (relativepath == "") relativepath == "/";

			results << normalizePath(QString("@SHARE/%1%2").arg(sharename).arg(relativepath));
		}
	}

	return(results);
}

ReturnValue RdsUtils::getVolumePath(const QString &path)
{
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);

	QString newpath = ret.toString();

	if (!newpath.startsWith("/volumes/")) return(ReturnValue(1, path + " is not a system path"));

	newpath = newpath.mid(9);

	int index = newpath.indexOf("/");
	QString volume;
	if (index == -1)
	{
		volume = newpath;
		newpath = "";
	}
	else
	{
		volume = newpath.left(index);
	}

	return(normalizePath(QString("@VOLUME/%1/%2").arg(volume).arg(newpath.mid(index + 1))));
}


RdsUtils::PathType RdsUtils::pathType(const QString &path)
{
	if (path.startsWith("@SHARE/")) return(Share);
	else if (path.startsWith("@VOLUME/")) return(Volume);
	else return(System);
}

ReturnValue RdsUtils::getRelativePath(const QString &path)
{
	PathType type = pathType(path);
	if (type == System) return(path);

	QString tmp = path;

	if (type == Volume) tmp = path.mid(8);
	else if (type == Share) tmp = path.mid(7);

	int index = tmp.indexOf("/");
	if (index < 0) return(ReturnValue(1, "Malformed path"));

	return(normalizePath(tmp.mid(index + 1)));
}

ReturnValue RdsUtils::getShareName(const QString &path)
{
	if (pathType(path) != Share) return(ReturnValue(1, "The given path is not a share path"));

	QString tmp = path.mid(7);
	int index = tmp.indexOf("/");
	if (index < 0) return(ReturnValue(1, "Malformed path"));

	return(tmp.left(index));
}

ReturnValue RdsUtils::getVolumeName(const QString &path)
{
	if (pathType(path) != Volume) return(ReturnValue(1, "The given path is not a volume path"));

	QString tmp = path.mid(8);
	int index = tmp.indexOf("/");
	if (index < 0) return(ReturnValue(1, "Malformed path"));

	return(tmp.left(index));
}

QString RdsUtils::normalizePath(const QString path)
{
	QStringList pathlist = path.split("/", QString::SkipEmptyParts);
	QString ret;
	foreach(QString s, pathlist)
	{
		if (s != ".." && s != "." && s != "~")
		{
			if ((s == "@SHARE" || s == "@VOLUME") && ret == "")
				ret.append(s);
			else
				ret.append("/").append(s);
		}
	}
	if (ret == "")
		return "/";
	return ret;
}

///TODO Figure out what to do about query injections
ReturnValue RdsUtils::getObjectBySid(RdsSid sid)
{
	QString filter = QString("(objectSid=%1)").arg(sid.toString());

	ReturnValue ret = rdsLdapSession()->search(RdsUtils::baseDn(), filter);
	if (ret.isError())
	{
		return(ret);
	}

	LdapResults results = ret.value<LdapResults>();

	if (results.count() < 1) return(ReturnValue(1, "User not found"));


	return(results.keys()[0]);
}

ReturnValue RdsUtils::getSidOfObject(QString dn)
{
	ReturnValue ret = rdsLdapSession()->read(dn);
	if (ret.isError())
	{
		return(ret);
	}

	LdapResult entry = ret.value<LdapResult>();
	if (!entry.keys().contains("objectsid"))
	{
		return(ReturnValue(2, "Entry does not have an objectSid attribute"));
	}

	LdapValues attribute = entry["objectsid"];
	if (attribute.count() < 1)
	{
		return(ReturnValue(3, "objectSid attribute is empty"));
	}

	return(QVariant::fromValue(RdsSid(attribute[0])));
}

ReturnValue RdsUtils::followSymLink(const QString &p)
{
	QString path = p; // Using this for the return value :P
	for (int j = 0; QFileInfo(path).isSymLink() && j < 16; ++j) // lawl j!
	{
		QString sympath = QFileInfo(path).symLinkTarget();
		if (path == sympath)
			break;
		path = sympath;
	}
	if (QFileInfo(path).isSymLink())
		return ReturnValue(1, "Too many levels of symbolic links in " + path);
	return path;
}

ReturnValue RdsUtils::createDirectory(const QString &path, bool recursive)
{
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);

	QString realpath = ret.toString();

	QDir dir("/");
	bool success = false;

	if (QDir().exists(realpath))
		return(ReturnValue(1, "Directory already exists."));

	if (!recursive)
		success = QDir().mkdir(realpath);
	else
	{
		QStringList parentPath = path.split("/", QString::SkipEmptyParts);
		parentPath.removeLast();
		if(parentPath.size() == 2) //A share path must end with a /, a folder does not
			ret = createDirectory(parentPath.join("/") + "/", true);
		else
			ret = createDirectory(parentPath.join("/"), true);
		if (ret.isError() && ret.errString() != "Directory already exists.")
			return ret;
		success = QDir().mkdir(realpath);
	}

	if (success)
	{
#ifdef __RDS_SERVER
		ret = RdsNtAcl::updateInheritedPermissions(realpath);
		if (ret.isError())
		{
			qWarning() << "Failed to set up permissions for new folder:" << realpath << ret;
		}

		RdsFileManager filemgr;

		ret = getSharePath(path);
		if (!ret.isError())
		{
			RdsShareManager sharemgr;

			foreach(QString share, ret.toStringList())
			{
				sharemgr.addEntity(share);
				filemgr.addEntity(share);
			}
		}

		ret = getVolumePath(path);
		if (!ret.isError())
		{
			filemgr.addEntity(ret.toString());
		}

		ret = getSystemPath(path);
		if (!ret.isError())
		{
			filemgr.addEntity(ret.toString());
		}
#endif
		return(true);
	}
	else
	{
		return(ReturnValue(1, "Failed to create directory:" + path));
	}
}

ReturnValue RdsUtils::remove(const QString &path, bool recursive)
{
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);
	QString realpath = ret.toString();

	if (realpath == "/")
		return(ReturnValue(1, "You may not remove /"));

	if (!recursive && QFileInfo(realpath).isDir())
	{
		if (QProcess::execute("rmdir", QStringList() << realpath) != 0)
			return(ReturnValue(1, "Failed to remove directory:" + path));
	}
	else
	{
		QStringList args = QStringList() << realpath;
		if (recursive) args << "-r";
		if (QProcess::execute("rm", args) != 0)
			return(ReturnValue(1, "Failed to remove directory:" + path));
	}

#ifdef __RDS_SERVER
	RdsFileManager filemgr;

	ret = getSharePath(path);
	if (!ret.isError())
	{
		RdsShareManager sharemgr;

		foreach(QString share, ret.toStringList())
		{
			//qDebug() << "Remving:" << share;
			sharemgr.removeEntity(share);
			filemgr.removeEntity(share);
		}
	}

	ret = getVolumePath(path);
	if (!ret.isError())
	{
		filemgr.removeEntity(ret.toString());
	}

	ret = getSystemPath(path);
	if (!ret.isError())
	{
		filemgr.removeEntity(ret.toString());
	}
#endif

	return(true);
}

ReturnValue RdsUtils::move(const QString &path, const QString &newpath)
{
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);
	QString realpath = ret.toString();
	ret = RdsUtils::getSystemPath(newpath);
	if (ret.isError()) return(ret);
	QString realnewpath = ret.toString();

#ifdef __RDS_SERVER
	QFileInfo info(realnewpath); //we need to check this BEFORE we move the file
	bool destdir = info.isDir();
#endif

	if (QProcess::execute("mv", QStringList() << realpath << realnewpath) == 0)
	{
#ifdef __RDS_SERVER
		//if the file was moved into a directory, we need to append the file name to the destination before we
		QString destpath = newpath;
		if (destdir) destpath = newpath + "/" + QDir(realpath).dirName();

		ret = RdsNtAcl::updateInheritedPermissions(destpath);
		if (ret.isError())
		{
			qWarning() << "Failed to set up permissions for new folder:" << destpath << ret;
		}

		RdsFileManager filemgr;

		ret = getSharePath(path);
		ReturnValue ret2 = getSharePath(destpath);
		if (!ret.isError() && !ret2.isError())
		{
			RdsShareManager sharemgr;
			foreach(QString share, ret.toStringList())
			{
				//qDebug() << "Removing:" << share;
				sharemgr.removeEntity(share);
				filemgr.removeEntity(share);
			}
			foreach(QString share, ret2.toStringList())
			{
				//qDebug() << "Adding:" << share;
				sharemgr.addEntity(share);
				filemgr.addEntity(share);
			}
		}
		else
		{
			qWarning() << "Failed to get share path:" << ret << ret2 << path;
		}

		ret = getVolumePath(path);
		ret2 = getVolumePath(destpath);
		if (!ret.isError())
		{
			filemgr.removeEntity(ret.toString());
		}
		if (!ret2.isError())
		{
			filemgr.addEntity(ret.toString());
		}


		filemgr.removeEntity(newpath);
		filemgr.removeEntity(destpath);
#endif
		return(true);
	}
	else
	{
		return(ReturnValue(1, "Failed to move " + path + " to " + newpath));
	}
}

ReturnValue RdsUtils::rename(const QString &path, const QString &newname)
{
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);
	QString realpath = ret.toString();

	QDir dir(realpath);
	if (!dir.cdUp())
		return(ReturnValue(1, "Unable to get parent directory for " + path));

	QString newpath = dir.absolutePath() + "/" + newname;

	if (QProcess::execute("mv", QStringList() << realpath << newpath) == 0)
	{
#ifdef __RDS_SERVER
		RdsFileManager filemgr;
		ret = getSharePath(path);
		ReturnValue ret2 = getSharePath(newpath);
		if (!ret.isError() && !ret2.isError())
		{
			QStringList shares1 = ret.toStringList();
			QStringList shares2 = ret2.toStringList();

			if (shares1.size() == shares2.size())
			{
				for (int i = 0; i < shares1.size(); i++)
				{
					QString share1 = shares1[i];
					QString share2 = shares2[i];
					//qDebug() << "Renamed:" << share1 << "To:" << share2;
					RdsShareManager sharemgr;
					sharemgr.renameEntity(share1, share2);
					filemgr.renameEntity(share1, share2);
				}
			}
		}
		else
		{
			qWarning() << "Failed to get share path:" << ret << ret2 << path;
		}

		ret = getVolumePath(path);
		ret2 = getVolumePath(newpath);
		if (!ret.isError() && !ret2.isError())
		{
			filemgr.renameEntity(ret.toString(), ret2.toString());
		}

		ret = getSystemPath(path);
		ret2 = getSystemPath(newpath);
		if (!ret.isError() && !ret2.isError())
		{
			filemgr.renameEntity(ret.toString(), ret2.toString());
		}
#endif
		return(true);
	}
	else
	{
		return(ReturnValue(1, "Failed to move " + path + " to " + newpath));
	}
}

ReturnValue RdsUtils::fileContents(const QString &path)
{
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);
	QString realpath = ret.toString();

	QFile file(realpath);
	if (!file.open(QFile::ReadOnly))
		return(ReturnValue(1, "Failed to open file " + path));

	QByteArray data = file.readAll();
	file.close();

	return(data);
}

ReturnValue RdsUtils::setFileContents(const QString &path, const QByteArray &data)
{
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);
	QString realpath = ret.toString();

	//See if the file exists before we set its contents
	bool exists = QFileInfo(realpath).exists();

	QFile file(realpath);
	if (!file.open(QFile::WriteOnly | QFile::Truncate))
		return(ReturnValue(1, "Failed to open file " + path));

	if (file.write(data) != data.size())
		return(ReturnValue(1, "Failed to write file " + path));

	file.close();

	//If we created a new file, then set its NT permissions
	if (!exists)
	{
		ret = RdsNtAcl::updateInheritedPermissions(realpath);
		if (ret.isError())
		{
			qWarning() << "Failed to set up permissions for new file:" << realpath << ret;
		}
	}

#ifdef __RDS_SERVER
	RdsFileManager filemgr;

	ret = getSharePath(path);
	if (!ret.isError())
	{
		RdsShareManager sharemgr;

		foreach(QString share, ret.toStringList())
		{
			sharemgr.updateEntity(share);
			filemgr.updateEntity(share);
		}
	}

	ret = getVolumePath(path);
	if (!ret.isError())
	{
		filemgr.updateEntity(ret.toString());
	}

	ret = getSystemPath(path);
	if (!ret.isError())
	{
		filemgr.updateEntity(ret.toString());
	}
#endif

	return(true);
}

ReturnValue RdsUtils::unixPermissions(const QString &path)
{
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);
	QString realpath = ret.toString();

	struct stat buf;
	memset(&buf, 0, sizeof(struct stat));
	if (stat(qPrintable(realpath), &buf) != 0)
		return(ReturnValue(1, "Failed to stat " + path));

	return(buf.st_mode & 0777);
}

ReturnValue RdsUtils::setUnixPermissions(const QString &path, int mode)
{
#ifdef Q_OS_WIN32
	return ReturnValue(1, "This function is not implemented in Windows");
#else
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);
	QString realpath = ret.toString();

	if (chmod(qPrintable(realpath), mode) != 0)
		return(ReturnValue(1, "Failed to chmod " + path));

#ifdef __RDS_SERVER
	RdsFileManager filemgr;

	ret = getSharePath(path);
	if (!ret.isError())
	{
		RdsShareManager sharemgr;

		foreach(QString share, ret.toStringList())
		{
			sharemgr.updateEntity(share);
			filemgr.updateEntity(share);
		}
	}

	ret = getVolumePath(path);
	if (!ret.isError())
	{
		filemgr.updateEntity(ret.toString());
	}

	ret = getSystemPath(path);
	if (!ret.isError())
	{
		filemgr.updateEntity(ret.toString());
	}
#endif

	return(true);
#endif
}

ReturnValue RdsUtils::owner(const QString &path)
{
#ifdef Q_OS_WIN32
	return ReturnValue(1, "This function is not implemented in Windows");
#else
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);
	QString realpath = ret.toString();

	struct stat buf;
	if (stat(qPrintable(realpath), &buf) != 0)
		return(ReturnValue(1, "Failed to stat " + path));

	struct passwd *pwd = getpwuid(buf.st_uid);
	if (pwd == NULL) return(buf.st_uid); //if we can't look up the name, just return the UID

	return(pwd->pw_name);
#endif
}

ReturnValue RdsUtils::setOwner(const QString &path, const QString &owner)
{
#ifdef Q_OS_WIN32
	return ReturnValue(1, "This function is not implemented in Windows");
#else
	struct passwd *pwd = getpwnam(qPrintable(owner));
	if (pwd == NULL)
		return(ReturnValue(1, "Failed to look up ID for user " + owner));

	return(setOwner(path, pwd->pw_uid));
#endif
}

ReturnValue RdsUtils::setOwner(const QString &path, int owner)
{
#ifdef Q_OS_WIN32
	return ReturnValue(1, "This function is not implemented in Windows");
#else
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);
	QString realpath = ret.toString();

	if (chown(qPrintable(realpath), owner, -1) != 0)
		return(ReturnValue(1, "Failed to change owner of " + path));
#endif
#ifdef __RDS_SERVER
	RdsFileManager filemgr;

	ret = getSharePath(path);
	if (!ret.isError())
	{
		RdsShareManager sharemgr;

		foreach(QString share, ret.toStringList())
		{
			sharemgr.updateEntity(share);
			filemgr.updateEntity(share);
		}
	}

	ret = getVolumePath(path);
	if (!ret.isError())
	{
		filemgr.updateEntity(ret.toString());
	}

	ret = getSystemPath(path);
	if (!ret.isError())
	{
		filemgr.updateEntity(ret.toString());
	}
#endif

	return(true);
}

ReturnValue RdsUtils::group(const QString &path)
{
#ifdef Q_OS_WIN32
	return ReturnValue(1, "This function is not implemented in Windows");
#else
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);
	QString realpath = ret.toString();

	struct stat buf;
	if (stat(qPrintable(realpath), &buf) != 0)
		return(ReturnValue(1, "Failed to stat " + path));

	struct group *grp = getgrgid(buf.st_gid);
	if (grp == NULL) return(buf.st_gid); //if we can't look up the name, just return the UID

	return(grp->gr_name);
#endif
}

ReturnValue RdsUtils::setGroup(const QString &path, const QString &group)
{
#ifdef Q_OS_WIN32
	return ReturnValue(1, "This function is not implemented in Windows");
#else
	struct group *grp = getgrnam(qPrintable(group));
	if (grp == NULL)
		return(ReturnValue(1, "Failed to look up ID for group " + group));

	return(setGroup(path, grp->gr_gid));
#endif
}

ReturnValue RdsUtils::setGroup(const QString &path, int group)
{
#ifdef Q_OS_WIN32
	return ReturnValue(1, "This function is not implemented in Windows");
#else
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);
	QString realpath = ret.toString();

	if (chown(qPrintable(realpath), -1, group) != 0)
		return(ReturnValue(1, "Failed to change group of " + path));
#endif
#ifdef __RDS_SERVER
	RdsFileManager filemgr;

	ret = getSharePath(path);
	if (!ret.isError())
	{
		RdsShareManager sharemgr;

		foreach(QString share, ret.toStringList())
		{
			sharemgr.updateEntity(share);
			filemgr.updateEntity(share);
		}
	}

	ret = getVolumePath(path);
	if (!ret.isError())
	{
		filemgr.updateEntity(ret.toString());
	}

	ret = getSystemPath(path);
	if (!ret.isError())
	{
		filemgr.updateEntity(ret.toString());
	}
#endif

	return(true);
}

ReturnValue RdsUtils::ntPermissions(const QString &path)
{
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);
	QString realpath = ret.toString();

	return(RdsNtAcl::fromFile(realpath));
}

ReturnValue RdsUtils::setNtPermissions(const QString &path, const RdsNtAcl &permissions)
{
	ReturnValue ret = RdsUtils::getSystemPath(path);
	if (ret.isError()) return(ret);
	QString realpath = ret.toString();

	ret = permissions.save(realpath);
	if (ret.isError()) return(ret);

#ifdef __RDS_SERVER
	RdsFileManager filemgr;

	ret = getSharePath(path);
	if (!ret.isError())
	{
		RdsShareManager sharemgr;

		foreach(QString share, ret.toStringList())
		{
			sharemgr.updateEntity(share);
			filemgr.updateEntity(share);
		}
	}

	ret = getVolumePath(path);
	if (!ret.isError())
	{
		filemgr.updateEntity(ret.toString());
	}

	ret = getSystemPath(path);
	if (!ret.isError())
	{
		filemgr.updateEntity(ret.toString());
	}
#endif

	return(ret);
}

ReturnValue RdsUtils::runCommand(const QString &command, const QStringList &args)
{
	QProcess proc;
	proc.setProcessChannelMode(QProcess::MergedChannels);
	proc.start(command, args);
	proc.waitForFinished();
	
	if(proc.exitCode() != 0)
	{
		return ReturnValue(proc.exitCode(), proc.readAll());
	}
	else
	{
		//qDebug() << "Command output:" << proc.readAll();
		return proc.readAll();
	}
}

