/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "aclmodel.h"
#include <RdsAdObject>
#include <RdsUtils>
#include <QDebug>
#include "config.h"

AclModel::AclModel(QObject *parent)
		: QAbstractListModel(parent)
{
	_acl = NULL;
	_advanced = false;

	_unknownicon = QPixmap(findRdsIcon("./icons/16x16/cancel.png"));
	_usericon = QPixmap(findRdsIcon("./icons/16x16/userconfig.png"));
	_groupicon = QPixmap(findRdsIcon("./icons/hi32-group.png")).scaled(16, 16);
}

AclModel::~AclModel()
{
}

void AclModel::setAcl(RdsNtAcl *acl)
{
	_acl = acl;
	reload();
}

void AclModel::setAdvanced(bool advanced)
{
	_advanced = advanced;
	reload();
}

int AclModel::rowCount(const QModelIndex &parent) const
{
	Q_UNUSED(parent);
	if (_acl == NULL) return(0);
	return(_acls.size());
}

int AclModel::columnCount(const QModelIndex &parent) const
{
	Q_UNUSED(parent);
	if (_acl == NULL) return(0);
	if (_advanced) return(4);
	else return(1);
}

QVariant AclModel::data(const QModelIndex &index, int role) const
{
	if (_acl == NULL) return(QVariant());
	if (!index.isValid()) return(QVariant());

	if (role == Qt::DisplayRole)
	{
		if (_advanced)
		{
			switch (index.column())
			{
				case 0:
					return(_acls[index.row()].name);
				case 1:
					return(_acls[index.row()].allow ? "Allow" : "Deny");
				case 2:
					return(_acls[index.row()].inherited ? "Yes" : "No");
				case 3:
				{
					QStringList tmp;
					Acl acl = _acls[index.row()];
					//qDebug() << "Flags:" << acl.flags << acl.name;
					if ((acl.flags & RdsAce::NoPropogate) != 0) tmp << "This Folder Only";
					else
					{
						if ((acl.flags & RdsAce::InheritOnly) == 0) tmp << "This Folder";
						if ((acl.flags & RdsAce::FileInherit) != 0) tmp << "Files";
						if ((acl.flags & RdsAce::FolderInherit) != 0) tmp << "Subfolders";
					}

					return(tmp.join(", "));
				}
			}
		}
		else
		{
			switch (index.column())
			{
				case 0:
					return(_acls[index.row()].name);
			}
		}
	}
	else if ((role == Qt::DecorationRole) && (index.column() == 0))
	{
		switch (_acls[index.row()].type)
		{
			case User:
				return(_usericon);
				break;
			case Group:
				return(_groupicon);
				break;
			default:
				return(_unknownicon);
		}
	}

	return(QVariant());
}

QVariant AclModel::headerData(int section, Qt::Orientation orientation, int role) const
{
	Q_UNUSED(orientation);
	if (_acl == NULL) return(QVariant());
	if (_advanced)
	{
		if (role == Qt::DisplayRole)
		{
			switch (section)
			{
				case 0:
					return("Type");
				case 1:
					return("Name");
				case 2:
					return("Inherited");
				case 3:
					return("Apply To");
				default:
					return(QVariant());
			}
		}
		else
		{
			return(QVariant());
		}
	}
	else
	{
		if (role == Qt::DisplayRole)
		{
			switch (section)
			{
				case 0:
					return("Name");
				default:
					return(QVariant());
			}
		}
		else
		{
			return(QVariant());
		}
	}
}

void AclModel::reload()
{
	_acls.clear();

	if (_advanced)
	{
		foreach(RdsAce ace, _acl->dacl())
		{
			Acl acl;
			acl.primaryace = ace;
			acl.name = ace.sid().toString();
			acl.inherited = ((ace.flags() & RdsAce::Inherited) == RdsAce::Inherited);
			acl.allow = (ace.type() == RdsAce::Allow);
			acl.sid = ace.sid();
			acl.flags = ace.flags();
			resolveAce(ace, acl);

			_acls << acl;
		}
	}
	else
	{
		foreach(RdsAce ace, _acl->dacl())
		{
			bool found = false;
			for (int i = 0; i < _acls.size(); i ++)
			{
				if (_acls[i].sid == ace.sid())
				{
					//qDebug() << "Duplicate!";
					found = true;
				}
			}

			if (found) continue;

			Acl acl;
			acl.primaryace = ace;
			acl.name = ace.sid().toString();
			acl.sid = ace.sid();
			acl.flags = ace.flags();

			resolveAce(ace, acl);

			_acls << acl;
		}
	}

	reset();
}

void AclModel::resolveAce(RdsAce &ace, Acl &acl)
{
	acl.type = Unknown;

	ReturnValue ret = RdsUtils::getObjectBySid(ace.sid());
	if (!ret.isError())
	{
		//qDebug() << "Sid:" << ret.toString() << ace.sid();

		RdsAdObject object(ret.toString());
		ret = object.readAttribute("cn");
		if (!ret.isError())
		{
			LdapValues attr = ret.value<LdapValues>();
			if (attr.size() > 0)
			{
				acl.name = attr[0];

				ret = object.readAttribute("objectClass");
				if (!ret.isError())
				{
					LdapValues attr = ret.value<LdapValues>();
					//qDebug() << acl.name << attr;
					if (attr.contains(QString("user").toAscii()))
					{
						acl.type = User;
						//qDebug() << "User:" << acl.name;
					}
					else if (attr.contains(QString("group").toAscii()))
					{
						acl.type = Group;
						//qDebug() << "Group:" << acl.name;
					}
					else if (attr.contains(QString("foreignSecurityPrincipal").toAscii()))
					{
						acl.type = Group;
						//qDebug() << "Group:" << acl.name;
					}
					else
					{
						acl.type = Group;
					}
				}
				else qWarning() << "Failed to get object type of" << object.dn() << ret;
			}
		}
		else qWarning() << "Failed to get CN of" << object.dn() << ret;
	}
	else qWarning() << "Failed to get DN of" << ace.sid() << ret;
}

AclModel::Acl AclModel::getAclFromIndex(const QModelIndex &index) const
{
	Acl nullacl;
	if (!index.isValid()) return(nullacl);
	if (index.row() > _acls.size()) return(nullacl);

	return(_acls[index.row()]);
}

void AclModel::removeAcl(const Acl &acl)
{
	int row = _acls.indexOf(acl);
	if (row == -1)
	{
		qWarning() << "AclModel::removeAcl Invalid ACL";
		return;
	}

	//qDebug() << "Removing Row:" << row;

	beginRemoveRows(QModelIndex(), row, row);
	_acls.removeAt(row);
	endRemoveRows();
}

void AclModel::addAcl(const AclModel::Acl& acl)
{
	//qDebug() << "Adding ACL" << _acls.size();
	beginInsertRows(QModelIndex(), _acls.size(), _acls.size());
	_acls.append(acl);
	endInsertRows();
}

bool AclModel::Acl::operator==(const AclModel::Acl &other) const
{
	if (primaryace != other.primaryace) return(false);
	if (name != other.name) return(false);
	if (inherited != other.inherited) return(false);
	if (allow != other.allow) return(false);
	if (type != other.type) return(false);
	if (sid != other.sid) return(false);

	return(true);
}
