/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "browsedialog.h"
#include <RdsClient>
#include <QMessageBox>
#include <QInputDialog>
#include <RdsUtils>
#include <QSettings>
#include <QMenu>

BrowseDialog::BrowseDialog(BrowseTypes flags, QWidget* parent, Qt::WFlags fl)
		: QDialog(parent, fl), Ui::BrowseDialog()
{
	setupUi(this);
	
	ReturnValue ret = rdsClient()->getService("FileManager");
	
	if (ret.isError())
	{
		QMessageBox::critical(this, "Failed to get manager", ret.errString());
		return;
	}

	_manager = new RdsFileManager(this);
	*_manager = ret;
	
	if((flags & Volumes) == Volumes) _manager->setShowVolumes(true);
	else _manager->setShowVolumes(false);
	if((flags & Shares) == Shares) _manager->setShowShares(true);
	else _manager->setShowShares(false);
	if((flags & FileSystem) == FileSystem) _manager->setShowFileSystem(true);
	else _manager->setShowFileSystem(false);
	
	_model = new RdsFileModel(_manager, this);
	_sort = new RdsFileSortModel(_model);
	_sort->setSourceModel(_model);
	View->setModel(_sort);
	_selection = View->selectionModel();
	_model->setupSelection(_sort,View);
	View->sortByColumn(0, Qt::AscendingOrder);
	_model->setConstrainSelection(false);
	
	_newfolder = new QAction("New folder", this);
	QObject::connect(_newfolder, SIGNAL(triggered()), this, SLOT(on_NewButton_clicked()));
	_showfilesystem = new QAction("Show File System", this);
	QObject::connect(_showfilesystem, SIGNAL(toggled(bool)), this, SLOT(showFileSystem(bool)));
	_showfilesystem->setCheckable(true);
	
	
	QSettings settings;
	if (settings.value("BrowseShowFilesystem").toBool())
	{
		_showfilesystem->setChecked(true);
		_sort->setHiddenEntities(QStringList());
	}
	else
	{
		_showfilesystem->setChecked(false);
		_sort->setHiddenEntities(QStringList() << "/");
	}
	
	View->setContextMenuPolicy(Qt::CustomContextMenu);
	QObject::connect(View, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(contextMenu(const QPoint&)));
}

BrowseDialog::~BrowseDialog()
{
}

void BrowseDialog::accept()
{
	QDialog::accept();
}

QString BrowseDialog::path()
{
	return(_model->input());
}

void BrowseDialog::on_NewButton_clicked()
{
	if(_model->input() == "")
	{
		QMessageBox::warning(this,"Error","You must select a place to create the folder in.");
		return;
	}
	
	bool ok=false;
	QString name = QInputDialog::getText(this,"New Folder","Enter the name for the new folder", QLineEdit::Normal, "New Folder", &ok);
	
	if(ok)
	{
		if(name == "")
		{
			QMessageBox::warning(this,"Error","You must give the new folder a name.");
			return;
		}
		
		NewButton->setEnabled(false);
		ReturnValue ret = _manager->createDirectory(this,SLOT(newfolder(uint, ReturnValue)), RdsUtils::normalizePath(_model->input() + "/" + name));
	}
}

void BrowseDialog::newfolder(uint id, ReturnValue ret)
{
	Q_UNUSED(id);
	NewButton->setEnabled(true);
	if(ret.isError())
	{
		QMessageBox::warning(this,"Error","Failed to create directory: " + ret.errString());
		return;
	}
}

void BrowseDialog::showFileSystem(bool show)
{
	QSettings settings;
	settings.setValue("BrowseShowFileSystem",show);
	if (show)
	{
		_sort->setHiddenEntities(QStringList());
	}
	else
	{
		_sort->setHiddenEntities(QStringList() << "/");
	}
	
	_sort->setFilterRegExp(_sort->filterRegExp());
}

void BrowseDialog::contextMenu(const QPoint& point)
{
	Q_UNUSED(point);
	QMenu menu;
	menu.addAction(_newfolder);
	menu.addSeparator();
	menu.addAction(_showfilesystem);
	menu.exec(QCursor::pos());
}
