/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/


#include "computereditwidget.h"
#include <RdsComputer>
#include <QMessageBox>
#include <QDebug>
#include <RdsEntityManager>
#include <RdsDhcpManager>
#include <RdsDhcpHost>
#include <RdsDaemonManager>

ComputerEditWidget::ComputerEditWidget(RdsEntityManager *manager, QWidget* parent)
		: RdsEntityWidget(manager, parent), Ui::ComputerEditWidget()
{
	setupUi(this);
	setApplyButton(ApplyButton);
	setDiscardButton(DiscardButton);
	
	//General
	addWidget("cn", Name, NoMulti, "^[0-9a-zA-Z][^+\\;,=]*$", "The name must not contain any of these characters (+ \\ ; , =).");
	addWidget("description",Description,NoMulti);
	addWidget("location",Location);
	addWidget("dnsName",DnsName);
	addWidget("operatingSystem",OperatingSystem);
	addWidget("version",Version);
	addWidget("servicePack",ServicePack);
	//addWidget("ip",Ip,"ip",SIGNAL(changed()), NoMulti);
	
	//Group Tab
	addWidget("groups",GroupList, "groups",SIGNAL(changed()),NoMulti,QStringList());
	addWidget("primaryGroup", PrimaryGroup,"dn",SIGNAL(changed()),RdsEntityWidget::None,"");
	
	QObject::connect(manager, SIGNAL(entityRenamed(QString, QString)), this, SLOT(entityRenamed(QString, QString)));
}

ComputerEditWidget::~ComputerEditWidget()
{
}

#define GET_DATA(name,function) ret = function; \
	if(ret.isError() && !ret.errString().endsWith("attribute does not exist")) err = ret; \
	fields[name] = ret

ReturnValue ComputerEditWidget::getData()
{
	QVariantMap fields;
	
	if (input() == "") return(ReturnValue(1, "Invalid Computer"));
	
	RdsComputer computer(input());
	
	ReturnValue err = true;
	ReturnValue ret;
	
	GET_DATA("cn",computer.cn());
	GET_DATA("description",computer.description());
	GET_DATA("location",computer.location());
	GET_DATA("dnsName",computer.hostName());
	GET_DATA("operatingSystem",computer.operatingSystem());
	GET_DATA("version",computer.operatingSystemVersion());
	GET_DATA("servicePack",computer.servicePack());
	GET_DATA("primaryGroup", computer.primaryGroup());
	GET_DATA("groups", computer.groups());
	QStringList tmp = fields["groups"].toStringList();
	tmp.sort();
	fields["groups"] = tmp;
	
	RdsDhcpManager dhcp;
	ret = dhcp.init();
	if(ret.isError())
	{
		err = ret;
	}
	else
	{
		ret = dhcp.listAllHosts();
		//qDebug() <<"HOSTS:" << ret;
		
		if(ret.isError())
		{
			err = ret;
		}
		else
		{
			QStringList list = ret.toStringList();
			foreach(QString id, list)
			{
				//qDebug() << id << "/h" + fields["cn"].toString().toLower() << id.toLower().endsWith("/h" + fields["cn"].toString().toLower());
				if(id.toLower().endsWith("/h" + fields["cn"].toString().toLower()))
				{
					ret = dhcp.values(id);
					if(ret.isError())
					{
						err = ret;
					}
					else
					{
						RdsDhcpHost host = ret;
						ret = host.value("fixed-address");
						fields["ip"] = ret;
					}
				}
			}
		}
	}
	
	if (err.isError())
	{
		QMessageBox msg(QMessageBox::Warning, "Error", "There was an error opening this computer: " + ret.errString(), QMessageBox::Ok);
		QAbstractButton *editbutton = msg.addButton("Edit Anyway", QMessageBox::RejectRole);
		msg.exec();
		if (msg.clickedButton() == editbutton)
		{
			if (QMessageBox::warning(this, "Warning", "Editing a computer in this state may be dangerous. Continue?", QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::Yes)
			{
				return(fields);
			}
		}
		return(err);
	}
	

	return(fields);
}

#define SET_FIELD(field,function) if (name == field) \
{ \
		ret = function; \
		if (ret.isError()) err = ret; \
}

#define SET_FIELD_STRING(field,function) \
	SET_FIELD(field,computer.function(value.toString()));

bool ComputerEditWidget::setData(QString id, QVariantMap fields)
{
	RdsComputer computer(id);

	ReturnValue err = true;
	ReturnValue ret;
	
	foreach(QString name, fields.keys())
	{
		QVariant value = fields[name];
		
		SET_FIELD_STRING("description",setDescription);
		SET_FIELD_STRING("location",setLocation);
		
		if (name == "groups")
		{
			//qDebug() << "Changing Groups:" << value.toStringList(); 
			ret = computer.groups();
			if(!ret.isError())
			{
				QStringList newgroups = value.toStringList();
				QStringList oldgroups = ret.toStringList();
				
				//Add new groups
				foreach(QString group, newgroups)
				{
					if(!oldgroups.contains(group))
					{
						ret = computer.joinGroup(group);
						//qDebug() << "Joining Group:" << group;
						if(ret.isError()) err = ret;
					}
				}
				
				foreach(QString group, oldgroups)
				{
					if(!newgroups.contains(group))
					{
						ret = computer.leaveGroup(group);
						//qDebug() << "Leaving Group:" << group;
						if(ret.isError()) err = ret;
					}
				}
			}
			else
			{
				err = ret;
			}
			
		}
		if(name == "ip")
		{
			RdsDhcpManager dhcp;
			ret = dhcp.init();
			if(ret.isError())
			{
				err = ret;
			}
			else
			{
				ret = dhcp.listAllHosts();
				//qDebug() <<"HOSTS:" << ret;
				
				if(ret.isError())
				{
					err = ret;
				}
				else
				{
					bool found = false;
					QStringList list = ret.toStringList();
					foreach(QString id, list)
					{
						//qDebug() << id << "/h" + Name->text().toLower() << id.toLower().endsWith("/h" + Name->text().toLower());
						if(id.toLower().endsWith("/h" + Name->text().toLower()))
						{
							found = true;
							if(value.toString() == "")
							{
								QString parent = id.left(id.lastIndexOf("/"));
								//qDebug() << "Parent:" << parent;
								ret = dhcp.values(parent);
								if(ret.isError())
								{
									err = ret;
								}
								else
								{
									RdsDhcpHost host = ret;
									ret = host.removeHost(id.mid(id.lastIndexOf("/") + 2));
									if(ret.isError()) err = ret;
									else 
									{
										dhcp.removeEntity(id);
										RdsDaemonManager dmgr;
										dmgr.init();
										ret = dmgr.restartService("dhcp");
										if(ret.isError())
										{
											QMessageBox::critical(this, "Error", "Failed to restart the DHCP server:\n" + ret.errString());
										}
									}
								}
							}
							else
							{
								//qDebug() << "Values:" << id;
								ret = dhcp.values(id);
								if(ret.isError())
								{
									err = ret;
								}
								else
								{
									RdsDhcpHost host = ret;
									ret = host.setValue("fixed-address", value.toString());
									if(ret.isError()) err = ret;
									else
									{
										dhcp.updateEntity(id);
										RdsDaemonManager dmgr;
										dmgr.init();
										ret = dmgr.restartService("dhcp");
										if(ret.isError())
										{
											QMessageBox::critical(this, "Error", "Failed to restart the DHCP server:\n" + ret.errString());
										}
									}
								}
							}
							
							break;
						}
					}
					
					if((value.toString() != "") && !found)
					{
						ret = dhcp.values();
						if(ret.isError())
						{
							err = ret;
						}
						else
						{
							RdsDhcpValues values = ret;
							ret = values.addHost(Name->text().toLower());
						
							if(ret.isError())
							{
								err = ret;
							}
							else
							{
								dhcp.addEntity("root/m/h" + Name->text().toLower());
								RdsDhcpHost host = ret;
								ret = host.setValue("fixed-address", value.toString());
								if(ret.isError()) err = ret;
								else 
								{
									dhcp.entityUpdated("root/m/h" + Name->text().toLower());
									RdsDaemonManager dmgr;
									dmgr.init();
									ret = dmgr.restartService("dhcp");
									if(ret.isError())
									{
										QMessageBox::critical(this, "Error", "Failed to restart the DHCP server:\n" + ret.errString());
									}
								}
							}
						}
					}
				}
			}
		}
		
		SET_FIELD_STRING("primaryGroup",setPrimaryGroup);
		
		if (name == "cn")
		{
		//qDebug() << "setting cn" << value;
			ret = computer.setCn(value.toString());
			if (ret.isError())
			{
				err = ret;
			}
			else
			{
				QString dn = ret.toString();

				QStringList outlist = outputs();
				for (int i = 0; i < outlist.size(); i++)
				{
					if (outlist[i] == id) outlist[i] = dn;
				}

				setInput(dn);
				setOutputs(outlist);

				manager()->renameEntity(id, dn);
			}
		}
	}
	
	if (err.isError())
	{
		qWarning() << "Failed to save computer:" << id << err.errString();
		return(false);
	}
	else return(true);
}

void ComputerEditWidget::entityUpdated(QString id)
{
	if (id != input()) return;

	if (unsavedChanges())
	{
		if (QMessageBox::question(this, "Computer Changed",
		                          "Another user has made changes to the item you are editing. Would you like to overwrite your changes?",
		                          QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::Yes)
		{
			RdsEntityWidget::entityUpdated(id);
		}
	}
	else
	{
		RdsEntityWidget::entityUpdated(id);
	}
}

void ComputerEditWidget::entityRenamed(QString oldid, QString newid)
{
	if (oldid != input()) return;

	RdsComputer computer(newid);
	ReturnValue ret = computer.cn();
	if (ret.isError()) return;

	QString cn = ret.toString();

	QStringList outlist = outputs();
	for (int i = 0; i < outlist.size(); i++)
	{
		if (outlist[i] == input()) outlist[i] = newid;
	}

	setInput(newid);
	setOutputs(outlist);

	setField("cn", cn);
}


