/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "config.h"
#include "dnsrecordeditwidget.h"
#include "rdsdnsmodel.h"
#include <QMessageBox>
#include <QDebug>
#include <RdsDnsManager>
#include <RdsDnsZone>
#include <RdsDnsRecord>
#include <RdsUtils>
#include <RdsDaemonManager>
#include <RdsFileManager>

DnsRecordEditWidget::DnsRecordEditWidget(RdsDnsManager *manager, QWidget* parent)
		: RdsEntityWidget(manager, parent), Ui::DnsRecordEditWidget(),
		_zone(0),
		_newid(0)
{
	setupUi(this);
	setApplyButton(ApplyButton);
	setDiscardButton(DiscardButton);

	for (int i = 1; true; ++i)
	{
		QString t = RdsDnsZone::typeToString((RdsDnsZone::RecordType)i);
		if (t.isEmpty())
			break;
		Type->addItem(t, i);
	}

	addWidget("name", Name, NoMulti);
	addWidget("type", Type);
	addWidget("value", Value, "list", SIGNAL(changed()));
	Value->setSeperator(QRegExp(",[ ]*"), ", ");
	Value->setUpDownArrows(true);

	QObject::connect(manager, SIGNAL(entityRenamed(QString, QString)), this, SLOT(entityRenamed(QString, QString)));
}

DnsRecordEditWidget::~DnsRecordEditWidget()
{
}

void DnsRecordEditWidget::entityRenamed(QString oldid, QString newid)
{
	if (oldid != input()) return;

	QStringList outlist = outputs();
	for (int i = 0; i < outlist.size(); i++)
	{
		if (outlist[i] == input()) outlist[i] = newid;
	}

	setInput(newid);
	setOutputs(outlist);
}

#define GET_DATA(value, getter) \
	ret = _zone->getter(); \
	if (ret.isError()) \
		return ret; \
	fields[value] = ret;

ReturnValue DnsRecordEditWidget::getData()
{
	QVariantMap fields;
	ReturnValue ret;

	delete _zone;
	_zone = 0;
	_newname = QString();
	_newid = 0;

	RdsDnsManager mgr;
	mgr.init();
	QString zone = input();
	zone = zone.left(zone.indexOf('#'));
	zone = zone.left(zone.indexOf(':'));
	ret = mgr.zone(zone);
	if (ret.isError())
		return ret;
	_zone = new RdsDnsZone(ret);

	QString origin = input();
	origin = origin.left(origin.indexOf('#'));
	QStringList originPath = origin.split(':', QString::SkipEmptyParts);
	origin = QString();
	foreach(QString path, originPath)
	{
		origin.prepend(path);
		origin.prepend('.');
	}
	origin = origin.mid(1);

	QString key = input();
	key = key.left(key.lastIndexOf('#'));
	key = key.mid(key.indexOf('#') + 1);

	QString typeStr = input();
	typeStr = typeStr.mid(typeStr.lastIndexOf('#') + 1);
	RdsDnsZone::RecordType type = (RdsDnsZone::RecordType)typeStr.toInt();

	ret = _zone->records((int)type, key, origin);
	if (ret.isError())
		return ret;
	RdsDnsRecordList records = ret.value<RdsDnsRecordList>();

	if (records.count())
	{
		fields["name"] = records.first().key();
		QStringList values;
		foreach (RdsDnsRecord record, records)
		{
			values << record.value();
		}
		fields["value"] = values;
		fields["type"] = RdsDnsZone::typeToString(records.first().type());
	}
	else
	{
		return ReturnValue(1, "Got an empty list of records from dns");
	}

	return(fields);
}

bool DnsRecordEditWidget::setData(QString id, QVariantMap fields)
{
	ReturnValue ret;

	QString zone = id;
	zone = zone.left(zone.indexOf('#'));
	zone = zone.left(zone.indexOf(':'));

	QString origin = id;
	origin = origin.left(origin.indexOf('#'));
	QStringList originPath = origin.split(':', QString::SkipEmptyParts);
	origin = QString();
	foreach(QString path, originPath)
	{
		origin.prepend(path);
		origin.prepend('.');
	}
	origin = origin.mid(1);

	QString key = id;
	key = key.left(key.lastIndexOf('#'));
	key = key.mid(key.indexOf('#') + 1);

	QString typeStr = id;
	typeStr = typeStr.mid(typeStr.lastIndexOf('#') + 1);
	RdsDnsZone::RecordType type = (RdsDnsZone::RecordType)typeStr.toInt();

	// get the old record
	ret = _zone->records((int)type, key, origin);
	if (ret.isError())
	{
		qCritical() << "Error in record saving (records):" << ret;
		return false;
	}
	RdsDnsRecordList records = ret.value<RdsDnsRecordList>();

	if (!records.count())
	{
		qCritical() << "Error in record saving (records was empty)";
		return false;
	}

	// remove the old records, we're editing not adding
	QStringList values;
	foreach(RdsDnsRecord record, records)
	{
		_zone->removeRecord(record);
		values << record.value();
	}
	RdsDnsRecord record = records.first();

	// apply the changes to the local record object
	if (fields.contains("name"))
	{
		QString key = fields.value("name").toString();
		if (key == zone || key.isEmpty() || (key + '.') == zone)
			key = zone;

		record.setKey(key);
		_newname = key;
	}
	if (fields.contains("value"))
	{
		values = fields.value("value").toStringList();
	}
	if (fields.contains("type"))
	{
		record.setType(RdsDnsZone::stringToType(fields.value("type").toString()));
		_newid = (int)RdsDnsZone::stringToType(fields.value("type").toString());
	}

	// check if the new record already exists!
	ret = _zone->record((int)record.type(), record.key(), record.origin());
	if (!ret.isError())
	{
		qCritical() << "Error in record saving: Changing record into one that already exists!";
		return false;
	}

	// put the values in	
	ret = _zone->addRecord(record.type(), record.key(), values, record.origin(), record.ttl());
	if (ret.isError())
	{
		qCritical() << "Error in record saving (addRecord):" << ret;
		return false;
	}

	return true;
}

void DnsRecordEditWidget::entityUpdated(QString id)
{
	if (id != input()) return;

	if (unsavedChanges())
	{
		if (QMessageBox::question(this, "Dns Changed",
		                          "Another user has made changes to the item you are editing. Would you like to overwrite your changes?",
		                          QMessageBox::Yes | QMessageBox::No, QMessageBox::No) == QMessageBox::Yes)
		{
			RdsEntityWidget::entityUpdated(id);
		}
	}
	else
	{
		RdsEntityWidget::entityUpdated(id);
	}
}

#define CHECK_FOR_ERROR(function, errormsg) \
	ret = function; \
	if (ret.isError()) \
	{ \
		qCritical() << errormsg << ret; \
		QMessageBox::critical(this, "Error", errormsg + ret.errString()); \
		return false; \
	}

bool DnsRecordEditWidget::endSetData()
{
	ReturnValue ret;

	CHECK_FOR_ERROR(_zone->save(), "Failed to save the record object: ");

	if (!_newname.isEmpty() || _newid != 0)
	{
		QString zone = input();
		zone = zone.left(zone.indexOf(':'));
		zone = zone.left(zone.indexOf('#'));

		RdsDnsRecord record(input());

		if (!_newname.isEmpty())
			record.setKey(_newname);
		if (_newid != 0)
			record.setType((RdsDnsZone::RecordType)_newid);

		RdsDnsManager mgr;
		CHECK_FOR_ERROR(mgr.init(), "Failed to get RdsDnsManager service: ");
		CHECK_FOR_ERROR(mgr.renameEntity(input(), record.id(zone)), "Failed to rename entity: ");

		entityRenamed(input(), record.id(zone));
		_newname = QString();
		_newid = 0;
	}

	RdsDaemonManager mgr;

	CHECK_FOR_ERROR(mgr.init(), "Failed to get RdsDaemonManager service: ");
	CHECK_FOR_ERROR(mgr.reloadService("Dns"), "Failed to restart dns: ");

	return(true);
}

bool DnsRecordEditWidget::validate()
{
	QString zone = input();
	zone = zone.left(zone.indexOf('#'));
	zone = zone.left(zone.indexOf(':'));

	QString key = Name->text();
	QString origin = input();
	origin = origin.left(origin.indexOf('#'));
	QStringList originPath = origin.split(':', QString::SkipEmptyParts);
	origin = QString();
	foreach(QString path, originPath)
	{
		origin.prepend(path);
		origin.prepend('.');
	}
	origin = origin.mid(1);

	if ((key == zone || (key + '.') == zone || key.isEmpty()) && (origin == zone))
	{
		return true;
	}

	if (key.contains(QRegExp("[^A-Za-z0-9-_]")))
	{
		QMessageBox::critical(this, "Error", "Records may only contain letters, numbers, underscores, and hyphens.");
		return false;
	}
	if (key.isEmpty())
		return false;

	if (!RdsDnsModel::validateRecord(Value->list(), RdsDnsZone::stringToType(Type->currentText())))
		return false;

	return true;
}

