/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "storagemanagerwidget.h"
#include <QMessageBox>
#include <RdsClient>
#include <QDebug>
#include <QInputDialog>
#include <QMenu>
#include <QSettings>
#include "storageeditorwidget.h"
#include "freespacedelegate.h"
#include <RdsVolume>
#include "initializevolumedialog.h"
#include "unuseddevicedialog.h"
#include "config.h"

StorageManagerWidget::StorageManagerWidget(QWidget* parent, Qt::WFlags fl)
		: QWidget(parent, fl), Ui::StorageManagerWidget()
{
	setupUi(this);
	AddButton->setIcon(QPixmap(findRdsIcon("icons/32x32/add.png")));
	_edit = NULL;

	ReturnValue ret = rdsClient()->getService("VolumeManager");
	if (ret.isError())
	{
		QMessageBox::critical(this, "Failed to get volume manager", ret.errString());
		return;
	}

	_manager = new RdsVolumeManager(this);
	*_manager = ret;

	_model = new RdsStorageModel(_manager, this);
	_sort = new RdsStorageSortModel(_model);
	_sort->setSourceModel(_model);
	View->setModel(_sort);
	View->sortByColumn(0, Qt::AscendingOrder);
	_selection = View->selectionModel();
	_model->setupSelection(_sort, View);
	View->setItemDelegateForColumn(4, new FreeSpaceDelegate(_sort, this));
	View->header()->resizeSection(0,200);

	QObject::connect(_model, SIGNAL(inputChanged(QString, QString)), this, SLOT(inputChanged(QString, QString)));
	QObject::connect(_model, SIGNAL(outputsChanged(QStringList)), this, SLOT(outputsChanged(QStringList)));
	QObject::connect(_model, SIGNAL(entityUpdateFinished(QString)), this, SLOT(volumeUpdate(QString)));

	StorageEditorWidget *w = new StorageEditorWidget(_manager, EditFrame);
	EditLayout->addWidget(w);
	_edit = w;
	w->setInput("");


	//Actions and Menu stuff
	View->setContextMenuPolicy(Qt::CustomContextMenu);
	QObject::connect(View, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(contextMenu(const QPoint&)));


	_adddeviceaction = new QAction("Add Unused Device", this);
	QObject::connect(_adddeviceaction, SIGNAL(triggered()), this, SLOT(addUnusedDevice()));
	_initializeaction = new QAction("Initialize", this);
	QObject::connect(_initializeaction, SIGNAL(triggered()), this, SLOT(initialize()));
	_initializeaction->setEnabled(false);
	_ejectaction = new QAction("Eject", this);
	QObject::connect(_ejectaction, SIGNAL(triggered()), this, SLOT(eject()));
	_ejectaction->setIcon(QPixmap(findRdsIcon("./icons/16x16/eject.png")));
	_ejectaction->setEnabled(false);
	_removeaction = new QAction("Remove", this);
	QObject::connect(_removeaction, SIGNAL(triggered()), this, SLOT(remove()));
	_removeaction->setEnabled(false);

	EjectButton->setDefaultAction(_ejectaction);

	QMenu *addmenu = new QMenu(this);
	addmenu->addAction(_adddeviceaction);
	AddButton->setMenu(addmenu);
}

StorageManagerWidget::~StorageManagerWidget()
{

}

void StorageManagerWidget::inputChanged(QString id, QString type)
{
	//qDebug() << "Input Changed:" << id << type;
	if (_edit != NULL)
	{
		if (_edit->unsavedChanges() && _edit->isEnabled())
		{
			if (QMessageBox::question(this, "Unsaved Changes", "The User you were editing has unsaved changes. Would you like to apply them?", QMessageBox::Apply | QMessageBox::Discard, QMessageBox::Discard) == QMessageBox::Apply)
			{
				_edit->apply();
			}
		}

		delete _edit;
		_edit = NULL;
	}

	if (type == "volume")
	{
		StorageEditorWidget *w = new StorageEditorWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput(id);

		volumeUpdate(_model->input());
	}
	else
	{
		StorageEditorWidget *w = new StorageEditorWidget(_manager, EditFrame);
		EditLayout->addWidget(w);
		_edit = w;
		w->setInput("");
	}
}

void StorageManagerWidget::outputsChanged(QStringList outputs)
{
	//qDebug() << "Output Changed" << outputs;
	if (_edit != NULL) _edit->setOutputs(outputs);
}

void StorageManagerWidget::on_AddButton_clicked()
{
	AddButton->showMenu();
}

void StorageManagerWidget::remove()
{
	//qDebug() << "Removing:" << _model->input();
	RdsEntityModel::Cache *cache = (RdsEntityModel::Cache *)_model->getCache(_model->input());
	if (cache == NULL) return;

	ReturnValue ret = _manager->removeVolume(cache->name());
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to remove volume: " + ret.errString());
		return;
	}

	if ((cache->metadata().contains("status")) && (cache->metadata()["status"].toString().toLower() == "disconnected"))
	{
		inputChanged("","");
		_manager->removeEntity(_model->input());
	}
	else
	{
		_manager->updateEntity(_model->input());
	}
}

void StorageManagerWidget::contextMenu(const QPoint & pos)
{
	Q_UNUSED(pos);
	QMenu menu;
	menu.addAction(_adddeviceaction);
	menu.addAction(_ejectaction);
	menu.addAction(_initializeaction);
	menu.addAction(_removeaction);

	menu.exec(QCursor::pos());
}

void StorageManagerWidget::eject()
{
	ReturnValue ret = _manager->volume(_model->input());
	if (ret.isError())
	{
		QMessageBox::warning(this, "Error", "Failed to get volume: " + ret.errString());
		return;
	}
	RdsVolume volume;
	volume = ret;

	RdsEntityModel::Cache *cache = _model->getCache(_model->input());
	bool mounted = false, ejectable = false;
	if (cache->metadata().contains("mounted")) mounted = cache->metadata()["mounted"].toBool();
	if (cache->metadata().contains("ejectable")) ejectable = cache->metadata()["ejectable"].toBool();

	if (mounted)
	{
		EjectButton->setEnabled(false);
		ret = volume.unmount();
		EjectButton->setEnabled(true);
		if (ret.isError())
		{
			QMessageBox::warning(this, "Error", "Failed to unmount volume: " + ret.errString());
			return;
		}

		_manager->updateEntity(_model->input());

		if (ejectable)
		{
			EjectButton->setEnabled(false);
			ret = volume.eject();
			EjectButton->setEnabled(true);
			if (ret.isError())
			{
				QMessageBox::warning(this, "Error", "Failed to eject volume: " + ret.errString());
				return;
			}
		}
	}
	else
	{
		EjectButton->setEnabled(false);
		ret = volume.mount();
		EjectButton->setEnabled(true);
		if (ret.isError())
		{
			QMessageBox::warning(this, "Error", "Failed to mount volume: " + ret.errString());
			return;
		}
		_manager->updateEntity(_model->input());
	}

}

void StorageManagerWidget::initialize()
{
	InitializeVolumeDialog dialog;
	ReturnValue ret = dialog.initializeVolume(_manager, _model->input());
	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to initialize volume: " + ret.errString());
		return;
	}

	if (!ret.toBool()) return;

	dialog.exec();
	ret = dialog.result();

	if (ret.isError())
	{
		QMessageBox::critical(this, "Error", "Failed to initialize volume: " + ret.errString());
		return;
	}
}

void StorageManagerWidget::volumeUpdate(QString id)
{
	//qDebug() << "Volume Updated:" << id << _model->input();
	if (_model->input() != id) return;

	RdsEntityModel::Cache *cache = _model->getCache(_model->input());
	bool mounted = false, ejectable = false, disabled = false, added = false;
	if (cache->metadata().contains("mounted")) mounted = cache->metadata()["mounted"].toBool();
	if (cache->metadata().contains("ejectable")) ejectable = cache->metadata()["ejectable"].toBool();
	if (cache->metadata().contains("status")) disabled = (cache->metadata()["status"].toString().toLower() == "disconnected");
	if (cache->metadata().contains("added")) added = cache->metadata()["added"].toBool();

	_removeaction->setEnabled(added);
	//qDebug() << "Remove is:" << added;

	if (disabled)
	{
		_ejectaction->setIcon(QPixmap(findRdsIcon("./icons/16x16/plug.png")));
		_ejectaction->setText("Activate");
		_initializeaction->setEnabled(false);
	}
	else
	{
		_ejectaction->setEnabled(true);
		if (mounted)
		{
			_initializeaction->setEnabled(false);
			if (ejectable)
			{
				_ejectaction->setIcon(QPixmap(findRdsIcon("./icons/16x16/eject.png")));
				_ejectaction->setText("Eject");
			}
			else
			{
				_ejectaction->setIcon(QPixmap(findRdsIcon("./icons/16x16/unplug.png")));
				_ejectaction->setText("Deactivate");
			}
		}
		else
		{
			_ejectaction->setIcon(QPixmap(findRdsIcon("./icons/16x16/plug.png")));
			_ejectaction->setText("Activate");
			_initializeaction->setEnabled(true);
		}
	}
}

void StorageManagerWidget::addUnusedDevice()
{
	UnusedDeviceDialog *dialog = new UnusedDeviceDialog(this);
	dialog->show();
}
