/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "config.h"
#include <QCoreApplication>
#include <QString>
#include <QSettings>
#include <QDebug>
#include <AuthToken>
#include <iostream>

#include <RdsClient>
#include <RdsDaemonManager>

#include "qxtcommandoptions.h"

using namespace std;

int main(int argc, char *argv[])
{
	QCoreApplication app(argc, argv);
	QxtCommandOptions options;
	bool socket = true;

	options.add("list", "List available services");
	options.alias("list", "l");
	options.add("help", "Show this help message");
	options.alias("help", "h");

	options.addSection("Connection options");
	options.add("host", "The hostname or IP to connect to", QxtCommandOptions::Required);
	options.add("port", "Port to connect to the RDS server", QxtCommandOptions::Required);
	options.add("service", "A custom service name to connect to", QxtCommandOptions::Required);
	options.add("user", "Username to authenticate with", QxtCommandOptions::Required);
	options.add("pass", "Password to authenticate with", QxtCommandOptions::Required);

	options.parse(argc, argv);

	if (options.count("help") || options.showUnrecognizedWarning())
	{
		options.showUsage();
		return -1;
	}

	QString service = getenv("RDS_SERVICE");
	if (options.count("service"))
		service = options.value("service").toString();
	if (service == "")
		service = "Rds";

	QString host = getenv("RDS_SERVER");
	if (options.count("host"))
		host = options.value("host").toString();
	if (host == "")
		host = "localhost";
	else
		socket = false;

	int port = 0;
	QString tmp = getenv("RDS_PORT");
	if (tmp.toInt() != 0) port = tmp.toInt();
	if (options.count("port"))
		port = options.value("port").toInt();
	if (port == 0)
		port = 10123;
	else
		socket = false;

	QSettings settings(RDS_CONF_PATH + "/rds.conf", QSettings::IniFormat);
	QString user = settings.value("adminuser").toString();
	if (options.count("user"))
		user = options.value("user").toString();
	QString pass = settings.value("adminpw").toString();
	if (options.count("pass"))
		pass = options.value("pass").toString();

	RdsClient client;
	QString url;
	if (socket)
		url = QString("socket:///var/run/rdsd.socket:%3").arg(service);
	else
		url = QString("tcp://%1:%2/%3").arg(host).arg(port).arg(service);
	QtRpc::AuthToken token(user, pass);
	ReturnValue ret = client.connect(url, token);
	if (ret.isError())
	{
		qCritical() << "Failed to connect to server:" << ret.errString();
		return(1);
	}

	RdsClient::setGlobalInstance(&client);
	RdsDaemonManager dmgr;
	ret = dmgr.init();
	if (ret.isError())
	{
		qCritical() << "Failed to get the DaemonManager object:" << ret.errString();
		return(1);
	}

	if (options.count("list"))
	{
		ret = dmgr.list();
		if (ret.isError())
		{
			qCritical() << "Failed to list available services:" << ret.errString(); // I do that to remove the space
			return(1);
		}
		foreach(QString tmp, ret.toStringList())
		{
			cout << qPrintable(tmp) << endl;
		}
		return(0);
	}
	else if (options.positional().size() != 2)
	{
		qWarning() << "Usage: rdscontrol start|stop|reload|restart|reloadconfig service [--list] [--help]";
		return(1);
	}

	QString action = options.positional()[0].toLower();
	QString daemon = options.positional()[1];

	if (action == "stop")
	{
		ret = dmgr.stopService(daemon);
		if (ret.isError())
		{
			qCritical() << "Failed to stop" << (daemon + ":") << ret.errString(); // I do that to remove the space
			return(1);
		}
	}
	else if (action == "start")
	{
		ret = dmgr.startService(daemon);
		if (ret.isError())
		{
			qCritical() << "Failed to start" << (daemon + ":") << ret.errString(); // I do that to remove the space
			return(1);
		}
	}
	else if (action == "restart")
	{
		ret = dmgr.restartService(daemon);
		if (ret.isError())
		{
			qCritical() << "Failed to restart" << (daemon + ":") << ret.errString(); // I do that to remove the space
			return(1);
		}
	}
	else if (action == "reload")
	{
		ret = dmgr.reloadService(daemon);
		if (ret.isError())
		{
			qCritical() << "Failed to reload" << (daemon + ":") << ret.errString(); // I do that to remove the space
			return(1);
		}
	}
	else if (action == "reloadconfig")
	{
		ret = dmgr.reloadConfig(daemon);
		if (ret.isError())
		{
			qCritical() << "Failed to reload the config for" << (daemon + ":") << ret.errString(); // I do that to remove the space
			return(1);
		}
	}
	else
	{
		qWarning() << "Usage: rdscontrol start|stop|reload|restart|reloadconfig service [--lsit] [--help]";
		return(1);
	}

	return 0;
}
