/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsservice.h"
#include <QDebug>
#include <QProcess>
#include <QTextStream>
#include <QDateTime>
#include <QHostInfo>
#include <QFile>
#include <RdsSettings>
#include <RdsLdapSession>
#include <RdsUtils>
#include <RdsUser>
#include <RdsSid>
#include <RdsNetworkSettings>
#include "rdsprovisionator.h"
#include "rdssambaldapsession.h"
#include "logmanager.h"
#include "config.h"


using namespace QtRpc;

RdsService::RdsService(QObject *parent)
		: ServiceProxy(parent)
{
}


RdsService::~RdsService()
{
}

ReturnValue RdsService::auth(AuthToken token)
{
	QString user = token.clientData()["username"].toString();
	QString pass = token.clientData()["password"].toString();

	if (!token.clientData().contains("protocolversions"))
	{
		return(ReturnValue(999, "The client you are connecting with is too old. Please update your client."));
	}

	if ((getProtocolProperty("protocol").toString() == "socket") && (getProtocolProperty("uid").toInt() == 0))
	{
		//Do nothing, the user is authenticating through the socket as root
	}
	else if (rdssettings()->value("provisioned").toBool())
	{
		//qDebug() << "The server is provisioned";

		///FIXME possible injection point
		ReturnValue ret = rdsLdapSession()->search(RdsUtils::baseDn(), QString("(sAMAccountName=%1)").arg(user));
		if (ret.isError()) return(ret);
		LdapResults results = ret.value<LdapResults>();
		QStringList keys = results.keys();
		if (keys.size() == 0) return(ReturnValue(1, "Failed to find user"));

		QString dn = keys[0];

		RdsLdapSession tmp;
		ret = tmp.simpleBind(dn, pass);
		if (ret.isError()) return(ret);

		RdsUser user(dn);
		ret = user.groups();
		if (ret.isError()) return(ret);

		bool isadmin = false;
		foreach(QString group, ret.toStringList())
		{
			RdsAdObject object(group);
			ret = object.sid();
			if (ret.isError())
			{
				qWarning() << "Failed to get SID of group" << group << ret;
				continue;
			}

			RdsSid sid = ret.value<RdsSid>();
			if (sid.rid() == 512)
			{
				isadmin = true;
			}
		}

		if (!isadmin)
		{
			return(ReturnValue(1, "Access Denied. You must be a domain administrator to log in."));
		}
	}
	else
	{
		qInfo() << "The server is not provisioned, not checking username or password";
	}

	token.serverData()["username"] = token.clientData()["username"].toString();
	token.serverData()["password"] = token.clientData()["password"].toString();
	token.serverData()["authenticated"] = true;

	return(rdsProtocolVersions());
}

ReturnValue RdsService::isProvisioned()
{
	return(rdssettings()->value("provisioned").toBool());
}

ReturnValue RdsService::ldapSession()
{
	if (rdssettings()->value("provisioned").toBool())
	{
		RdsSambaLdapSession *ldap = new RdsSambaLdapSession();
		*ldap = *rdsLdapSession();
		return(ldap);
	}
	else
	{
		return(ReturnValue(1, "Not Provisioned"));
	}
}

ReturnValue RdsService::domainData()
{
	QVariantMap data;
	data["basedn"] = RdsUtils::baseDn();
	data["realm"] = RdsUtils::realm();
	data["domain"] = RdsUtils::domain();

	return(data);
}

ReturnValue RdsService::provisionator()
{
	return(new RdsProvisionator());
}

ReturnValue RdsService::status()
{
	QVariantMap settings;

	RdsNetworkSettings net;
	net.loadNetworkConfig();
	ReturnValue ret = net.getDefaultInterface();
	if (!ret.isError()) settings["interface"] = ret;
	QString interface = ret.toString();
	ret = net.getInterfaceMac(interface);
	if (!ret.isError()) settings["mac"] = ret;
	ret = net.getInterfaceIp(interface);
	if (!ret.isError()) settings["ip"] = ret;
	ret = net.getInterfaceNetmask(interface);
	if (!ret.isError()) settings["netmask"] = ret;
	ret = net.getDefaultGateway();
	if (!ret.isError()) settings["gateway"] = ret;
	ret = net.getDnsServers();
	if (!ret.isError()) settings["dns"] = ret;
	ret = net.getDomain();
	if (!ret.isError()) settings["realm"] = ret;
	QStringList parts = ret.toString().split(".");
	if (parts.size() > 0)
	{
		settings["domain"] = parts[0].toUpper();
	}
	ret = RdsUtils::fileContents(RDS_TIMEZONE);
	if (!ret.isError()) settings["timezone"] = ret.toString().replace("\n", "");
	settings["datetime"] = QDateTime::currentDateTime();
	RdsNetworkSettings::Interface iface = net.getInterface(interface);

	char hostname[128];
	gethostname(hostname, 128);
	settings["hostname"] = hostname;

	settings["interfaces"] = net.listAllInterfaces().toStringList();

	ret = RdsUtils::fileContents("/etc/lsb-release");
	if (!ret.isError())
	{
		QString txt = ret.toString();
		QTextStream stream(&txt);
		QString line;
		while ((line = stream.readLine()) != QString::Null())
		{
			if (line.startsWith("DISTRIB_DESCRIPTION"))
			{
				QString tmp = line.mid(line.indexOf("\"") + 1);
				tmp = tmp.mid(0, tmp.indexOf("\""));

				settings["os"] = tmp;
			}
		}
	}

	QProcess uname;
	uname.start("uname", QStringList() << "-r");
	uname.waitForFinished(5000);
	settings["kernel"] = QString(uname.readAll()).replace("\n", "");

	QProcess uptime;
	uptime.start("cat", QStringList() << "/proc/uptime");
	uptime.waitForFinished(5000);
	parts = QString(uptime.readAll()).split(QRegExp("\\s"), QString::SkipEmptyParts);
	if (parts.size() > 0)
		settings["uptime"] = parts[0].trimmed().toDouble();

	QProcess cpu;
	cpu.start("cat", QStringList() << "/proc/cpuinfo");
	cpu.waitForFinished(5000);
	if (cpu.exitCode() == 0)
	{
		QTextStream stream(&cpu);
		QString line;
		int numcpu = 0;
		while ((line = stream.readLine()) != QString::Null())
		{
			parts = line.split(":");
			if (parts.size() != 2) continue;
			QString label = parts[0].trimmed();
			QString value = parts[1].trimmed();

			if (label == "model name")
			{
				numcpu++;
				settings["cputype"] = value;
			}
			else if (label == "cpu MHz")
			{
				settings["cpuspeed"] = value;
			}

		}

		settings["cpucores"] = numcpu;
	}

	QProcess mem;
	mem.start("free", QStringList() << "-m");
	mem.waitForFinished(5000);
	if (mem.exitCode() == 0)
	{
		QTextStream stream(&mem);
		QString line;
		while ((line = stream.readLine()) != QString::Null())
		{
			parts = line.split(QRegExp("\\s"), QString::SkipEmptyParts);
			QString label = parts[0].trimmed();

			if (label == "Mem:")
			{
				if (parts.size() < 2) continue;
				settings["mem"] = parts[1];
			}
			else if (label == "-/+")
			{
				if (parts.size() < 4) continue;
				settings["freemem"] = parts[3];
			}
			else if (label == "Swap:")
			{
				if (parts.size() < 3) continue;
				settings["swap"] = parts[2];
			}
		}
	}

	return(settings);
}

ReturnValue RdsService::dateTime()
{
	return(QDateTime::currentDateTime());
}

ReturnValue RdsService::setDateTime(QDateTime datetime)
{
	//set the date and time
	if (QProcess::execute("date", QStringList() << "-s" << datetime.toString("MM/dd/yyyy hh:mm:ss")) != 0)
		return ReturnValue(2, "Failed to set date and time");

	//sync the hardware clock to the system clock, and set it to UTC
	if (QProcess::execute("hwclock", QStringList() << "--systohc" << "--utc") != 0)
		qWarning() <<  "Failed to set the hardware clock";

	return(true);
}

ReturnValue RdsService::timeZone()
{
	return(RdsUtils::fileContents(RDS_TIMEZONE));
}

ReturnValue RdsService::setTimeZone(QString timezone)
{
	//Because QFile::copy sucks
	if (QProcess::execute("cp", QStringList() << QString(RDS_TIMEZONE_FOLDER).arg(timezone) << "/etc/localtime") != 0)
		return ReturnValue(2, "Failed to copy timezone data.");

	//Another timezone file
	QFile tzfile(RDS_TIMEZONE);
	if (!tzfile.open(QFile::WriteOnly | QFile::Truncate))
		return ReturnValue(2, "Failed to open timezone file for writing.");

	tzfile.write(timezone.toAscii());
	tzfile.close();

	return(true);
}

ReturnValue RdsService::listTimeZones()
{
	QFile file("/usr/share/zoneinfo/zone.tab");
	if (!file.open(QFile::ReadOnly))
	{

	}
	QStringList zones;
	QTextStream stream(&file);
	QString line;

	while ((line = stream.readLine()) != QString::Null())
	{
		if (line.startsWith("#")) continue;
		QStringList parts = line.split(QRegExp("\\s"), QString::SkipEmptyParts);
		if (parts.size() < 3) continue;

		zones << parts[2];
	}

	return(zones);
}

ReturnValue RdsService::listLogs()
{
	QStringList logs;
	QProcess p;
	p.start("find", QStringList() << "/var/log/" << "-type"  << "f");
	p.waitForFinished(10000);

	QTextStream stream(&p);
	QString line;
	while ((line = stream.readLine()) != QString::Null())
	{
		if (line.endsWith(".gz")) continue; //skip compressed logfiles
		logs << line;
	}

	return(logs);
}

ReturnValue RdsService::shutdown()
{
	QProcess::execute("halt");
	return(true);
}

ReturnValue RdsService::reboot()
{
	QProcess::execute("reboot");
	return(true);
}

QtRpc::ReturnValue RdsService::buildVersion()
{
	return(RDS_VERSION);
}

ReturnValue RdsService::hostname()
{
	return(QHostInfo::localHostName());
}

ReturnValue RdsService::listInterfaces()
{
	return(RdsNetworkSettings().listAllInterfaces().toStringList());
}

