/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsmultijob.h"
#include "rdsmultijob_p.h"

QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsMultiJob);

RdsMultiJob::RdsMultiJob(QObject *parent)
		: RdsJob(parent)
{
	setCanPause(true);
}

RdsMultiJob::RdsMultiJob(const RdsMultiJob& other)
{
	setCanPause(true);
	operator=(other);
}

RdsMultiJob::~RdsMultiJob()
{
}

ReturnValue RdsMultiJob::addJob(RdsJob* job)
{
	RDS_JOB_PROXY(RdsMultiJob, addJob(job));

	QMutexLocker locker(&qxt_d().mutex);
	if (!job)
		return ReturnValue(1, "Invalid job");

	qxt_d().jobList << job;
	return true;
}

ReturnValue RdsMultiJob::startInternal()
{
	setCanPause(true);
	qxt_d().jobRunner = new RdsMultiJobRunner(this);
	return qxt_d().jobRunner->start(qxt_d().jobList);
}

ReturnValue RdsMultiJob::stopInternal()
{
	return qxt_d().jobRunner->stop();
}

ReturnValue RdsMultiJob::pauseInternal()
{
	return qxt_d().jobRunner->pause();
}

ReturnValue RdsMultiJob::resumeInternal()
{
	return qxt_d().jobRunner->resume();
}

RdsMultiJobRunner::RdsMultiJobRunner(RdsMultiJob* parentJob)
		: _totalJobs(0),
		_currentJob(0),
		_parentJob(parentJob),
		_isPaused(false)
{
}

RdsMultiJob& RdsMultiJob::operator=(const RdsMultiJob & other)
{
	RdsJob::operator=(other);
	qxt_d().jobList = other.qxt_d().jobList;
	return *this;
}

ReturnValue RdsMultiJobRunner::start(const QList<RdsJob*>& list)
{
	_jobs = list;
	_totalJobs = list.count();
	ReturnValue ret = activateJob(_jobs.takeFirst());
	return ret;
}

ReturnValue RdsMultiJobRunner::stop()
{
	ReturnValue ret = _currentJob->stop();
	_parentJob->finish(ReturnValue(1, "Canceled"));
	return ret;
}

ReturnValue RdsMultiJobRunner::pause()
{
	_isPaused = true;
	if (_currentJob->canPause().toBool())
	{
		_currentJob->pause();
	}
}

ReturnValue RdsMultiJobRunner::resume()
{
	_isPaused = false;
	if (_currentJob)
	{
		return _currentJob->resume();
	}
	else
	{
		RdsJob* newJob = _jobs.takeFirst();
		ReturnValue ret = activateJob(newJob);
		if (ret.isError())
		{
			_parentJob->log("Failed to start next job in queue");
			_parentJob->finish(ret);
		}
		return ret;
	}
}

ReturnValue RdsMultiJobRunner::activateJob(RdsJob* job)
{
	if (_currentJob)
		_currentJob->QObject::disconnect(this);
	_currentJob = job;

	QObject::connect(_currentJob, SIGNAL(progressChanged(int, const QString&, const QString&)), this, SLOT(progressChangedSlot(int, const QString&, const QString&)));
	QObject::connect(_currentJob, SIGNAL(statusChanged(RdsJob::Status)), this, SLOT(statusChangedSlot(RdsJob::Status)));
	QObject::connect(_currentJob, SIGNAL(newLogMessages(const QList<RdsJob::LogEntry> &)), this, SLOT(newLogMessages(const QList<RdsJob::LogEntry> &)));
	QObject::connect(_currentJob, SIGNAL(finished(const ReturnValue&)), this, SLOT(finishedSlot(const ReturnValue&)));

	_parentJob->setProgressText(QString("Starting %1...").arg(_currentJob->name().toString()));
	_parentJob->setProgressDetails(QString("Starting %1...").arg(_currentJob->name().toString()));
	_parentJob->log("Starting job: " + _currentJob->name().toString());
	return _currentJob->start();
}

void RdsMultiJobRunner::progressChangedSlot(int percent, const QString& progressText, const QString& progressDetails)
{
	percent = (percent / _totalJobs) + ((_totalJobs - 1 - _jobs.count()) * (100 / _totalJobs));
	QString text;
	if (_currentJob)
	{
		text = QString("Running %1: %2").arg(_currentJob->name().toString()).arg(progressText);
	}
	else
	{
		if (percent == 100)
			text = "Finishing";
		else
			text = "Running";
	}

	if (percent != _parentJob->progress())
		_parentJob->setProgress(percent);

	if (text != _parentJob->progressText())
		_parentJob->setProgressText(text);

	if (progressDetails != _parentJob->progressDetails())
		_parentJob->setProgressDetails(progressDetails);
}

void RdsMultiJobRunner::statusChangedSlot(RdsJob::Status status)
{
	switch (status)
	{
		case RdsJob::Stopped:
			// wut?
			break;
		case RdsJob::Running:
			_parentJob->setStatus(RdsJob::Running);
			break;
		case RdsJob::Paused:
			_parentJob->setStatus(RdsJob::Paused);
			break;
		case RdsJob::Finished:
		{
			_currentJob = 0;
			if (_jobs.count() == 0)
			{
				_parentJob->finish(true);
				return;
			}
			if (_isPaused)
			{
				_parentJob->setStatus(RdsJob::Paused);
				break;
			}
			RdsJob* newJob = _jobs.takeFirst();
			ReturnValue ret = activateJob(newJob);
			if (ret.isError())
			{
				_parentJob->log("Failed to start next job in queue");
				_parentJob->finish(ret);
			}
			break;
		}
		case RdsJob::Failed:
			// handled in finishedSlot
			break;
	}
}

void RdsMultiJobRunner::newLogMessages(const QList<RdsJob::LogEntry> &logs)
{
	foreach(RdsJob::LogEntry log, logs)
	{
		_parentJob->log(log.log, log.level);
	}
}

void RdsMultiJobRunner::finishedSlot(const ReturnValue& ret)
{
	// on success, the next job is started from the statusChangedSlot above
	if (ret.isError())
	{
		_parentJob->log("A job has failed, canceling the batch");
		_parentJob->finish(ret);
	}
}



