/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsschedulemanager.h"
#include "rdsschedulemanager_p.h"
#include <QTimer>
#include <RdsEntity>

QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsScheduleManager);

RdsScheduler& RdsScheduleManagerPrivate::scheduler()
{
	static RdsScheduler scheduler;
	return scheduler;
}

RdsScheduleManager::RdsScheduleManager()
		: RdsEntityManager()
{
}

RdsScheduleManager::RdsScheduleManager(const RdsScheduleManager& other)
{
	operator=(other);
}

RdsScheduleManager::~RdsScheduleManager()
{
}

ReturnValue RdsScheduleManager::listEntities(const QString &base, bool loadmore) const
{
	QMutexLocker locker(&RdsScheduleManagerPrivate::scheduler().mutex);
	ReturnValue ret;
	if ((base == "") || (base == "root"))
	{
		RdsEntity entity;
		entity.setId("root");
		entity.setType("root");
		entity.setVisible(false);
		entity.setName("");
		entity.setParent("");

		foreach(QString name, RdsScheduleManagerPrivate::scheduler().eventHash.keys())
		{
			ret = listEntities("root/" + name, loadmore);
			if (!ret.isError()) entity.children() << ret.value<RdsEntity>();
		}
		return ReturnValue::fromValue<RdsEntity>(entity);
	}
	else if (base.startsWith("root/"))
	{
		QString name = base.mid(5);
		RdsEntity entity;
		entity.setId(base);
		entity.setType("event");
		entity.setVisible(true);
		entity.setName(name);
		entity.setParent("root");
		if (eventExists(name).toBool())
			return ReturnValue::fromValue<RdsEntity>(entity);
	}
	return ReturnValue(1, "Entity not found");
}

ReturnValue RdsScheduleManager::auth(QtRpc::AuthToken token)
{
	createInternalObject();
	if (token.serverData().contains("authenticated") && (token.serverData().value("authenticated").toBool() == true))
		return(true);
	else
		return(ReturnValue(1, "Not Authenticated"));
}

ReturnValue RdsScheduleManager::addEvent(const QString& name, const RdsSchedule& schedule, QObject* obj, const char* slot)
{
	QMutexLocker locker(&RdsScheduleManagerPrivate::scheduler().mutex);
	RdsScheduledEvent event(name, schedule, obj, slot);
	RdsScheduleManagerPrivate::scheduler().events.removeAll(event);
	RdsScheduleManagerPrivate::scheduler().events << event;
	RdsScheduleManagerPrivate::scheduler().eventHash.insert(name, &(RdsScheduleManagerPrivate::scheduler().events.last()));
	RdsScheduleManagerPrivate::scheduler().processSchedule();
	return true;
}

ReturnValue RdsScheduleManager::removeEvent(const QString& name)
{
	QMutexLocker locker(&RdsScheduleManagerPrivate::scheduler().mutex);
	if (!RdsScheduleManagerPrivate::scheduler().eventHash.contains(name))
		return false;
	RdsScheduledEvent event = *RdsScheduleManagerPrivate::scheduler().eventHash.value(name);
	RdsScheduleManagerPrivate::scheduler().events.removeAll(event);
	RdsScheduleManagerPrivate::scheduler().eventHash.remove(name);
	RdsScheduleManagerPrivate::scheduler().processSchedule();
	return true;
}

ReturnValue RdsScheduleManager::eventExists(const QString& name) const
{
	QMutexLocker locker(&RdsScheduleManagerPrivate::scheduler().mutex);
	return RdsScheduleManagerPrivate::scheduler().eventHash.contains(name);
}

RdsScheduleManager& RdsScheduleManager::operator=(const RdsScheduleManager & other)
{
	return *this;
}

RdsScheduler::RdsScheduler()
		: mutex(QMutex::Recursive)
{
}

void RdsScheduler::processSchedule()
{
	QMutexLocker locker(const_cast<QMutex*>(&mutex));
	for (QList<RdsScheduledEvent>::iterator i = events.begin(); i != events.end(); ++i)
	{
		if (i->nextEvent() == QDateTime())
			continue;
		if (QDateTime::currentDateTime().secsTo(i->nextEvent()) <= 0)
			executeTask(&(*i));
	}
	qSort(events);
	if (events.first().nextEvent() == QDateTime())
		return;
	int next = QDateTime::currentDateTime().secsTo(events.first().nextEvent());
	next = (next < 86400 ? next : 86400);
	QTimer::singleShot(next * 1000, this, SLOT(processSchedule()));
}

void RdsScheduler::executeTask(RdsScheduledEvent* event)
{
	QMutexLocker locker(const_cast<QMutex*>(&mutex));
	while (event->schedule().nextEvent() <= event->_nextEvent)
		usleep(10000);

	event->_nextEvent = event->schedule().nextEvent();

	bool success = false;
	if (event->slot().args().count() == 0)
		success = QMetaObject::invokeMethod(event->object().data(), qPrintable(event->slot().name()), Qt::QueuedConnection);
	else
		success = QMetaObject::invokeMethod(event->object().data(), qPrintable(event->slot().name()), Qt::QueuedConnection, Q_ARG(QString, event->name()));
	if (!success)
	{
		qCritical() << "Scheduled event" << event->name() << event->slot().args().count() << "failed to invokeMethod the slot" << event->slot() << "on" << event->object().data();
	}
}

RdsScheduledEvent::RdsScheduledEvent(const QString& name, const RdsSchedule& schedule, QObject* object, const char* slot)
		: _name(name),
		_schedule(schedule),
		_object(object),
		_slot(slot),
		_nextEvent(schedule.nextEvent())
{
}

RdsScheduledEvent::RdsScheduledEvent(const RdsScheduledEvent& other)
{
	operator=(other);
}

RdsScheduledEvent::~RdsScheduledEvent()
{
	RdsScheduleManagerPrivate::scheduler().eventHash.remove(_name);
}

QString RdsScheduledEvent::name()
{
	return _name;
}

void RdsScheduledEvent::setName(const QString& name)
{
	RdsScheduleManagerPrivate::scheduler().eventHash.insert(
	    name,
	    RdsScheduleManagerPrivate::scheduler().eventHash.take(_name)
	);
	_name = name;
}

RdsSchedule RdsScheduledEvent::schedule()
{
	return _schedule;
}

void RdsScheduledEvent::setSchedule(const RdsSchedule& schedule)
{
	_schedule = schedule;
	_nextEvent = _schedule.nextEvent();
}

QWeakPointer<QObject> RdsScheduledEvent::object()
{
	return _object;
}

void RdsScheduledEvent::setObject(QWeakPointer<QObject> object)
{
	_object = object;
}

QtRpc::Signature RdsScheduledEvent::slot()
{
	return _slot;
}

void RdsScheduledEvent::setSlot(const QtRpc::Signature& slot)
{
	_slot = slot;
}

QDateTime RdsScheduledEvent::nextEvent()
{
	return _nextEvent;
}

RdsScheduledEvent& RdsScheduledEvent::operator=(const RdsScheduledEvent & other)
{
	_name = other._name;
	_schedule = other._schedule;
	_object = other._object;
	_slot = other._slot;
	_nextEvent = other._nextEvent;
	return *this;
}

bool RdsScheduledEvent::operator==(const RdsScheduledEvent& other) const
{
	return (_name == other._name);
}
bool RdsScheduledEvent::operator<(const RdsScheduledEvent& other) const
{
	return (_nextEvent < other._nextEvent);
}
bool RdsScheduledEvent::operator<=(const RdsScheduledEvent& other) const
{
	return (_nextEvent <= other._nextEvent);
}

