/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#ifndef ORGANIZATIONALUNIT_H
#define ORGANIZATIONALUNIT_H

#include <qxtpimpl.h>
#include <QString>
#include <ReturnValue>
#include <RdsAdObject>
#include <rdsglobal.h>

class RdsOrganizationalUnitPrivate;

/**
	@author Brendan Powers <brendan@resara.com>
*/
class RDS_SHARED_EXPORT RdsOrganizationalUnit : public RdsAdObject
{
	friend RDS_SHARED_EXPORT QDataStream& operator<<(QDataStream& d, const RdsOrganizationalUnit& object);
	friend RDS_SHARED_EXPORT QDataStream& operator>>(QDataStream& d, RdsOrganizationalUnit& object);
	QXT_DECLARE_PRIVATE(RdsOrganizationalUnit);
public:
	RdsOrganizationalUnit();
	RdsOrganizationalUnit(QString path);
	RdsOrganizationalUnit(const RdsOrganizationalUnit& other);
	~RdsOrganizationalUnit();
	RdsOrganizationalUnit& operator=(const RdsOrganizationalUnit& other);

	/**
	 *        Gets the dn of the ou
	 * @return QString containing the dn
	 */
	QString dn() const;
	/**
	 *        Gets the dn of where the users are
	 * @return QString containing the dn
	 */
	QString userDn() const;
	/**
	 *        Gets the dn of where the groups are
	 * @return QString containing the dn
	 */
	QString groupDn() const;
	/**
	 *        Gets the dn of where the computers are
	 * @return QString containing the dn
	 */
	QString computerDn() const;

	/**
	 *        Gets a list of all the organizationalUnits in the ou
	 * @return QStringList of the ou's DNs on success, error on failure
	 */
	ReturnValue listOus() const;
	/**
	 *        Gets a list of all the users
	 * @return QStringList of the DNs of the users on success, error on failure
	 */
	ReturnValue listUsers() const;
	/**
	 *        Gets a list of group DNs
	 * @return QStringList of DNs of the groups on success, error on failure
	 */
	ReturnValue listGroups() const;
	/**
	 *        Gets a list of the computer DNs
	 * @return QStringList of computer DNs on success, error on failure.
	 */
	ReturnValue listComputers() const;

	/**
	 *        Creates an organizationalUnit
	 * @param name the name of the ou
	 * @return QString containing the dn of the new ou on success, error on failure.
	 */
	ReturnValue createOu(QString name);
	/**
	 *        Removes an ou
	 * @param name the name of the ou to remove
	 * @return true on success, error on failure
	 */
	ReturnValue deleteOu(QString name);
	/**
	 *        Gets the name of the ou from the "ou" attribute
	 * @return QString containing the ou name on success, error on failure
	 */
	ReturnValue ou() const;
	/**
	 *        Re-names the ou with cn
	 * @param cn the new name of the ou
	 * @return QString containing the new dn of the ou on success, error on failure.
	 */
	ReturnValue setOu(QString cn);
	/**
	 *        Gets the "description" attribute for the ou
	 * @return QString on success, error on failure
	 */
	ReturnValue description() const;
	/**
	 *        Sets the description of the ou
	 * @param desc the description
	 * @return true on success, error on failure
	 */
	ReturnValue setDescription(QString desc);
	/**
	 *        Removes the ou
	 * @return true on success, error on failure
	 */
	ReturnValue remove();

	/**
	 *        Creates a user
	 * @param name name of the user
	 * @param account account name of the user
	 * @return QString of the DN of the user on success, error on failure
	 */
	ReturnValue createUser(QString name, QString account);

	/**
	 *        Creates a group
	 * @param name name of the group
	 * @return QString of the group's DN on success, error on failure
	 */
	ReturnValue createGroup(QString name);

	/**
	 * Get a child OU by it's name
	 * @param ou name of the OU
	 * @return QString of the OU's DN on success, error on failure
	 */
	ReturnValue getOu(const QString &ou);

	/**
	 * Get a user in this OU by their full name
	 * @param fullname fullname of the user
	 * @return QString of the user's DN on success, error on failure
	 */
	ReturnValue getUser(const QString &fullname);

	/**
	 * Get a group in this OU by it's name
	 * @param group name of group
	 * @return QString of the group's DN on success, error on failure
	 */
	ReturnValue getGroup(const QString &group);

	/**
	 * Get a computer in this OU by it's name
	 * @param computer name of the computer
	 * @return QString of the computer's DN on success, error on failure
	 */
	ReturnValue getComputer(const QString &computer);
};

RDS_SHARED_EXPORT QDataStream& operator<<(QDataStream& d, const RdsOrganizationalUnit& object);
RDS_SHARED_EXPORT QDataStream& operator>>(QDataStream& d, RdsOrganizationalUnit& object);
Q_DECLARE_METATYPE(RdsOrganizationalUnit);

#endif
