/***************************************************************************
 *   Copyright (C) 2009 by Chris Vickery   *
 *   chris@resara.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Library General Public License as       *
 *   published by the Free Software Foundation; either version 2 of the    *
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Library General Public     *
 *   License along with this program; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <QtTest>
#include <QStringList>
#include <QDebug>
#include <ReturnValue>
#include <QHostAddress>
#include <QRegExp>
#include <QObject>

#include "RdsNtAcl"
#include "RdsSambaConfigParser"
#include "RdsVolume"
#include "rdsvolumemanager.h"
#include "RdsMountPoint"
#include "RdsNtSecurityDescriptor"
#include "RdsAce"
#include "RdsLdapSession"
#include "RdsUser"
#include "RdsSid"
#include "RdsGuid"
#include "RdsUtils"
#include "RdsDnsZone"
#include "RdsDnsManager"
#include "RdsDnsRecord"
#include "RdsEntity"
#include "RdsDhcpManager"
#include "RdsDhcpValues"
#include "RdsDhcpSubnet"
#include "RdsDhcpSharedNetwork"
#include "RdsDhcpGroup"
#include "RdsDhcpHost"
#include "rdsstoragedevice.h"
// #include "rdsudisks.h"
#include "rdsstoragedevice_p.h"

#ifndef QBENCHMARK
#	define QBENCHMARK
#endif

#define TEST_OUTPUT(x) \
	{\
		ReturnValue ret = x; \
		qDebug() << #x << ret; \
	}

#define TEST_FOR_ERRORS(x)\
	{\
		ReturnValue _test_ret = x; \
		if(_test_ret.isError()) qDebug() << _test_ret.errString(); \
		QVERIFY2(!_test_ret.isError(), qPrintable(_test_ret.errString()));\
	}

#define TEST_GETTER_SETTER(var1, var2, getter, setter)\
	QVERIFY2(var1 . getter () != var2 . getter (), "Testcase fail, the two variables have the same value " #var1 "." #getter);\
	var1 . setter (var2 . getter ());\
	QVERIFY2(var1 . getter () == var2 . getter (), "Getter/setter failure on " #var1 "." #getter);
ReturnValue testForError(const QString &function, const ReturnValue& ret)
{
	Q_ASSERT_X(!ret.isError(), qPrintable(function), qPrintable(ret.errString()));
	return ret;
}
#define TEST_FUNCTION(function) testForError(#function, function);
#define TEST_FUNCTION_ERROR(function) ___ret = function; \
	QVERIFY2(___ret.isError(), qPrintable("\n"#function " did not return an error, and instead returned: \"" + ___ret.toString() + "\""));

class RDSTest//: public QObject
{
// 	Q_OBJECT
public:
	RdsLdapSession session;
	ReturnValue ___ret;

public slots:
	void dnszones();
	void permissions();
	void storageDevice();
	void configReadKey();
	void dhcp();
	void udisks();
	void initTestCase();
	void securityDescriptor();
private slots:
};

void RDSTest::initTestCase()
{
	TEST_FOR_ERRORS(RdsUtils::adminBind(session));
	RdsLdapSession::setGlobalSession(&session);
}

void showEntity(const RdsEntity &entity)
{
	qDebug() << "ID" << entity.id();
	qDebug() << "Type" << entity.type();
	qDebug() << "name" << entity.name();
	qDebug() << "visible" << entity.visible();
	qDebug() << "parent" << entity.parent();
	qDebug() << "metadata" << entity.metadata();
	qDebug() << "children" << entity.children().count();
	foreach(RdsEntity child, entity.children())
	{
		showEntity(child);
	}
}

void RDSTest::udisks()
{
// 	RdsUDisks disks;

}

QString getDhcpFile()
{
	QString output;
	output =
	    "one-lease-per-client	true;\n"
	    "default-lease-time	20000;\n"
	    "max-lease-time	21600;\n"
	    "option	subnet-mask 255.255.0.0;\n"
	    "option	routers 10.0.0.1;\n"
	    "option	domain-name-servers 10.0.0.1;\n"
	    "option	domain-name workstations;\n"
	    "next-server	10.0.4.1;\n"
	    "\n"
	    "shared-network MySharedNetwork {\n"
	    "	default-lease-time 2000;\n"
	    "	subnet 0.0.0.0 netmask 0.0.0.0 {\n"
	    "		#(RDS)name SubnetInSN\n"
	    "		#(RDS) range 10.0.0.10 10.0.0.100\n"
	    "		range 10.0.1.10 10.0.1.100;\n"
	    "	}\n"
	    "}\n"
	    "shared-network EmptySharedNetork {\n"
	    "}\n"
	    "shared-network OLD_PARENT {\n"
	    "	subnet 10.0.1.0 netmask 255.255.255.0 {\n"
	    "		range 10.0.1.10 10.0.1.100;\n"
	    "	}\n"
	    "}\n"
	    "shared-network NEW_PARENT {\n"
	    "}\n"
	    "shared-network WORKSTATIONS {\n"
	    "	subnet 10.0.0.0 netmask 255.0.0.0 {\n"
	    "		#(RDS) range 10.0.4.10 10.0.4.250\n"
	    "		range 10.0.4.10 10.0.4.10;\n"
	    "		range 10.0.4.11 10.0.4.11;\n"
	    "		range 10.0.4.12 10.0.4.12;\n"
	    "		range 10.0.4.226 10.0.4.226;\n"
	    "		range 10.0.4.227 10.0.4.227;\n"
	    "		range 10.0.4.228 10.0.4.228;\n"
	    "		range 10.0.4.229 10.0.4.229;\n"
	    "		range 10.0.4.232 10.0.4.232;\n"
	    "		range 10.0.4.233 10.0.4.233;\n"
	    "		range 10.0.4.234 10.0.4.234;\n"
	    "		range 10.0.4.235 10.0.4.235;\n"
	    "		range 10.0.4.236 10.0.4.236;\n"
	    "		range 10.0.4.237 10.0.4.237;\n"
	    "		range 10.0.4.238 10.0.4.238;\n"
	    "		range 10.0.4.239 10.0.4.239;\n"
	    "		range 10.0.4.240 10.0.4.240;\n"
	    "		range 10.0.4.244 10.0.4.244;\n"
	    "		range 10.0.4.250 10.0.4.250;\n"
	    "	}\n"
	    "}\n";
	return output;
}

void RDSTest::dhcp()
{
	{
		RdsDhcpManager dhcp;
		QTemporaryFile configFile;
		configFile.open();
		configFile.write(getDhcpFile().toLocal8Bit());
		configFile.seek(0);
		dhcp.parse(configFile.readAll());
		TEST_FOR_ERRORS(dhcp.save());
		TEST_FOR_ERRORS(dhcp.reload());
		TEST_FOR_ERRORS(dhcp.reload());
		TEST_FOR_ERRORS(dhcp.save());
		TEST_FOR_ERRORS(dhcp.reload());
		TEST_FOR_ERRORS(dhcp.reload());
		ReturnValue ret = dhcp.listEntities(QString::null, false);
		TEST_FOR_ERRORS(ret);
		RdsEntity entity = ret.value<RdsEntity>();
		showEntity(entity);

		TEST_FOR_ERRORS(dhcp.move("root/m/nOLD_PARENT/sSubnet", "root/m/nNEW_PARENT/sNamedSubnet"));
		ret = dhcp.listEntities("root/m/nNEW_PARENT/sNamedSubnet", false);
		TEST_FOR_ERRORS(ret);
		entity = ret.value<RdsEntity>();
		showEntity(entity);

		return;
		// get the RdsDhcpValues object, which is the root of the configfile
		ret = dhcp.values();
		TEST_FOR_ERRORS(ret);
		RdsDhcpValues dhcpValues = ret;
		qDebug() << dhcpValues.sharedNetworks();
		QVERIFY(dhcpValues.host("SomeRandomHost").isError());
		TEST_FOR_ERRORS(dhcpValues.addHost("SomeRandomHost"));
		ret = dhcpValues.host("SomeRandomHost");
		TEST_FOR_ERRORS(ret);
		RdsDhcpHost dhcpHost = ret;
		TEST_FOR_ERRORS(dhcpHost.setValue("hardware", "ethernet 00:02:B3:AB:DD:1F"));
		TEST_FOR_ERRORS(dhcpHost.setValue("fixed-address", "10.0.0.56"));

		ret = dhcpValues.host("SomeRandomHost");
		TEST_FOR_ERRORS(ret);

		dhcpHost.setName("SomeOtherHost");
		ret = dhcpValues.host("SomeOtherHost");
		TEST_FOR_ERRORS(ret);

		ret = dhcpValues.sharedNetwork("MySharedNetwork");
		TEST_FOR_ERRORS(ret);
		RdsDhcpSharedNetwork dhcpSharedNetwork = ret;

		TEST_FOR_ERRORS(dhcpSharedNetwork.removeSubnet("SubnetInSN"));
		TEST_FOR_ERRORS(dhcpSharedNetwork.addSubnet("New Subnet"));
		ret = dhcpSharedNetwork.subnet("New Subnet");
		TEST_FOR_ERRORS(ret);
		RdsDhcpSubnet dhcpSubnet = ret;

		QList<QPair<QHostAddress, QHostAddress> > ranges;
		ranges << QPair<QHostAddress, QHostAddress>(QHostAddress("10.0.0.20"), QHostAddress("10.0.0.80"));
		ranges << QPair<QHostAddress, QHostAddress>(QHostAddress("10.0.0.90"), QHostAddress("10.0.0.100"));
		TEST_FOR_ERRORS(dhcpSubnet.setRanges(ranges));

		TEST_FOR_ERRORS(dhcpHost.setValue("fixed-address", "10.0.0.51"));
		TEST_FOR_ERRORS(dhcpHost.setValue("option", "subnet-mask 255.255.0.0"));
		TEST_FOR_ERRORS(dhcpHost.setValue("option", "routers 10.0.0.10"));
		TEST_FOR_ERRORS(dhcpHost.setOption("domain-name-servers", "10.0.0.1"));
		TEST_FOR_ERRORS(dhcpHost.setOption("routers", "10.0.0.1"));

		TEST_FOR_ERRORS(dhcp.save());
		TEST_FOR_ERRORS(dhcp.reload());
		TEST_FOR_ERRORS(dhcp.reload());
		TEST_FOR_ERRORS(dhcp.save());
		TEST_FOR_ERRORS(dhcp.reload());
		TEST_FOR_ERRORS(dhcp.reload());

		qDebug() << dhcp.toString();

		RdsDhcpSharedNetwork sharednetwork = TEST_FUNCTION(dhcpValues.sharedNetwork("WORKSTATIONS"));
		QStringList list = sharednetwork.subnets().toStringList();
		QVERIFY2(list.contains("Subnet"), "GROUP 'WORKSTATIONS' DOES NOT CONTAIN SUBNET 'Subnet'");
//-----------Depth------------
		RdsDhcpSubnet subnet = TEST_FUNCTION(sharednetwork.subnet("Subnet"));
		QCOMPARE(subnet.name().toString(), QString("Subnet"));
		QHostAddress address, netmask;
		address = subnet.address().value<QHostAddress>();
		netmask = subnet.netmask().value<QHostAddress>();
		subnet.setAddress(QHostAddress("192.168.1.1"));
		subnet.setNetmask(QHostAddress("255.255.255.0"));
		QCOMPARE(subnet.address().value<QHostAddress>(), QHostAddress("192.168.1.1"));
		QCOMPARE(subnet.netmask().value<QHostAddress>(), QHostAddress("255.255.255.0"));
		subnet.setAddress(address);
		subnet.setNetmask(netmask);

		RdsDhcpSubnet::RangeList rlist = subnet.ranges().value<RdsDhcpSubnet::RangeList>();
		QVERIFY2(rlist.size() == 1, "WHERE DID THESE OTHER RANGES COME FROM?");
		RdsDhcpSubnet::Range rpair = rlist[0];//easy verification later

//make sure parents are updated on name changes
		TEST_FUNCTION(subnet.setName("TitsMcGee"));
		subnet = TEST_FUNCTION(sharednetwork.subnet("TitsMcGee"));
		QCOMPARE(subnet.name().toString(), QString("TitsMcGee"));
		TEST_FUNCTION(subnet.setName("Subnet"));

		ret = TEST_FUNCTION(subnet.ranges());
//----------------------------

	}
}

void RDSTest::dnszones()
{
	{
		RdsDnsZone zone;
		QFile file("/home/chris/zone.conf");
		file.open(QFile::ReadOnly);
		zone.parse(file.readAll());
		foreach(RdsDnsRecord record, zone.listRecords().value<RdsDnsRecordList>())
		{
			qDebug() << "Key:" << record.key() << "Type:" << record.type() << "Value:" << record.value() << "Comments:" << record.comments();
		}
		qDebug() << "\n\n" << zone.toString() << "\n\n";
	}

	{
		RdsDnsManager mgr;
		qDebug() << "Return from save!" << mgr.save();
		ReturnValue ret = mgr.listEntities(QString::null, false);
		TEST_FOR_ERRORS(ret);
		RdsEntity entity = ret.value<RdsEntity>();
		showEntity(entity);
// 	qDebug() << mgr.toString();
	}
	Q_ASSERT(false);
}

void RDSTest::securityDescriptor()
{
	QString secdesc("O:S-1-5-21-2470468500-28526814-1734375037-512G:S-1-5-21-2470468500-28526814-1734375037-513D:AI(A;;RPWPCRCCDCLCLORCWOWDSDDTSW;;;S-1-5-21-2470468500-28526814-1734375037-512)(A;;RPWPCRCCDCLCLORCWOWDSDDTSW;;;SY)(A;;RPWPCRCCDCLCLORCWOWDSDDTSW;;;AO)(A;;RPLCLORC;;;PS)(OA;;CR;ab721a53-1e2f-11d0-9819-00aa0040529b;;PS)(OA;;CR;ab721a54-1e2f-11d0-9819-00aa0040529b;;PS)(OA;;CR;ab721a56-1e2f-11d0-9819-00aa0040529b;;PS)(OA;;RPWP;77b5b886-944a-11d1-aebd-0000f80367c1;;PS)(OA;;RPWP;e45795b2-9455-11d1-aebd-0000f80367c1;;PS)(OA;;RPWP;e45795b3-9455-11d1-aebd-0000f80367c1;;PS)(OA;;RP;037088f8-0ae1-11d2-b422-00a0c968f939;;S-1-5-21-2470468500-28526814-1734375037-553)(OA;;RP;4c164200-20c0-11d0-a768-00aa006e0529;;S-1-5-21-2470468500-28526814-1734375037-553)(OA;;RP;bc0ac240-79a9-11d0-9020-00c04fc2d4cf;;S-1-5-21-2470468500-28526814-1734375037-553)(A;;RC;;;AU)(OA;;RP;59ba2f42-79a2-11d0-9020-00c04fc2d3cf;;AU)(OA;;RP;77b5b886-944a-11d1-aebd-0000f80367c1;;AU)(OA;;RP;e45795b3-9455-11d1-aebd-0000f80367c1;;AU)(OA;;RP;e48d0154-bcf8-11d1-8702-00c04fb96050;;AU)(OA;;CR;ab721a53-1e2f-11d0-9819-00aa0040529b;;WD)(OA;;RP;5f202010-79a5-11d0-9020-00c04fc2d4cf;;S-1-5-21-2470468500-28526814-1734375037-553)(OA;;RPWP;bf967a7f-0de6-11d0-a285-00aa003049e2;;S-1-5-21-2470468500-28526814-1734375037-517)(OA;;RP;46a9b11d-60ae-405a-b7e8-ff8a58d456d2;;S-1-5-32-560)(OA;;RPWP;6db69a1c-9422-11d1-aebd-0000f80367c1;;S-1-5-32-561)(OA;;RPWP;5805bc62-bdc9-4428-a5e2-856a0f4c185e;;S-1-5-32-561)(OA;CIIOID;RP;4c164200-20c0-11d0-a768-00aa006e0529;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;CIIOID;RP;4c164200-20c0-11d0-a768-00aa006e0529;bf967aba-0de6-11d0-a285-00aa003049e2;RU)(OA;CIIOID;RP;5f202010-79a5-11d0-9020-00c04fc2d4cf;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;CIIOID;RP;5f202010-79a5-11d0-9020-00c04fc2d4cf;bf967aba-0de6-11d0-a285-00aa003049e2;RU)(OA;CIIOID;RP;bc0ac240-79a9-11d0-9020-00c04fc2d4cf;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;CIIOID;RP;bc0ac240-79a9-11d0-9020-00c04fc2d4cf;bf967aba-0de6-11d0-a285-00aa003049e2;RU)(OA;CIIOID;RP;59ba2f42-79a2-11d0-9020-00c04fc2d3cf;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;CIIOID;RP;59ba2f42-79a2-11d0-9020-00c04fc2d3cf;bf967aba-0de6-11d0-a285-00aa003049e2;RU)(OA;CIIOID;RP;037088f8-0ae1-11d2-b422-00a0c968f939;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;CIIOID;RP;037088f8-0ae1-11d2-b422-00a0c968f939;bf967aba-0de6-11d0-a285-00aa003049e2;RU)(OA;CIIOID;RP;b7c69e6d-2cc7-11d2-854e-00a0c983f608;bf967a86-0de6-11d0-a285-00aa003049e2;ED)(OA;CIIOID;RP;b7c69e6d-2cc7-11d2-854e-00a0c983f608;bf967a9c-0de6-11d0-a285-00aa003049e2;ED)(OA;CIIOID;RP;b7c69e6d-2cc7-11d2-854e-00a0c983f608;bf967aba-0de6-11d0-a285-00aa003049e2;ED)(OA;CIIOID;RPLCLORC;;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)(OA;CIIOID;RPLCLORC;;bf967a9c-0de6-11d0-a285-00aa003049e2;RU)(OA;CIIOID;RPLCLORC;;bf967aba-0de6-11d0-a285-00aa003049e2;RU)(OA;CIIOID;RPWPCR;91e647de-d96f-4b70-9557-d63ff4f3ccd8;;PS)(A;CIID;RPWPCRCCDCLCLORCWOWDSDDTSW;;;S-1-5-21-2470468500-28526814-1734375037-519)(A;CIID;LC;;;RU)(A;CIID;RPWPCRCCLCLORCWOWDSDSW;;;BA)S:AI(OU;CIIDSA;WP;f30e3bbe-9ff0-11d1-b603-0000f80367c1;bf967aa5-0de6-11d0-a285-00aa003049e2;WD)(OU;CIIDSA;WP;f30e3bbf-9ff0-11d1-b603-0000f80367c1;bf967aa5-0de6-11d0-a285-00aa003049e2;WD)");
	QString aceString("(A;;RPWPCRCCDCLCLORCWOWDSDDTSW;;;S-1-5-21-2470468500-28526814-1734375037-512)");
// 	aceString = "(OU;CIIDSA;WP;f30e3bbf-9ff0-11d1-b603-0000f80367c1;bf967aa5-0de6-11d0-a285-00aa003049e2;WD)";
	RdsNtAcl aclll = RdsNtAcl::fromFile("/media/bleh").value<RdsNtAcl>();
// 	qDebug() << aclll.toBinary().toHex();
	RdsNtSecurityDescriptor desc;
// 	qDebug() << desc.toBinary().toHex();
// 	qDebug() << "FROM USERS";
	RdsUser us("CN=Administrator,CN=Users,DC=resara,DC=local");
	ReturnValue ret = us.read(QStringList() << "nTSecurityDescriptor");
	TEST_FOR_ERRORS(ret);
	LdapResult base = ret.value<LdapResult>();
	QVERIFY(base.keys().count() == 1);
	foreach(QString attribute, base.keys())
	{
		QByteArray data = base[attribute][0];
		desc = RdsNtSecurityDescriptor(data);
// 		qDebug() << data.toHex();
// 		qDebug() << desc.toBinary().toHex();
	}
	return;
// 	RdsNtSecurityDescriptor desc(secdesc);
	RdsAce ace(aceString);
	QVERIFY(ace == RdsAce(ace.toString()));
	QVERIFY(ace == RdsAce(ace.toBinary()));
	QVERIFY(ace == ace.toString());
	QVERIFY(ace == aceString);
	QVERIFY(ace == desc.dacl().at(0));
	QVERIFY(ace == RdsAce(desc.dacl().at(0).toString()));
	QVERIFY(ace == desc.dacl().at(0).toString());
	QVERIFY(desc == RdsNtSecurityDescriptor(desc.toString()));
	QVERIFY(desc == RdsNtSecurityDescriptor(desc.toBinary()));
	QVERIFY(desc == desc.toString());
	QVERIFY(desc == secdesc);

	return;
	RdsNtSecurityDescriptor desc2;
	RdsAce ace2;
	QByteArray buff;
	{
		QDataStream stream(&buff, QIODevice::WriteOnly);
		stream << ace;
		stream << desc;
	}
	qDebug() << "Buff count!" << buff.count();
	{
		QDataStream stream(buff);
		stream >> ace2;
		stream >> desc2;
	}

	QVERIFY(ace2 == RdsAce(ace.toString()));
	QVERIFY(ace2 == ace.toString());
	QVERIFY(ace2 == aceString);
	QVERIFY(ace2 == desc.dacl().at(0));
	QVERIFY(ace2 == RdsAce(desc.dacl().at(0).toString()));
	QVERIFY(ace2 == desc.dacl().at(0).toString());
	QVERIFY(desc2 == RdsNtSecurityDescriptor(desc.toString()));
	QVERIFY(desc2 == desc.toString());
	QVERIFY(desc2 == secdesc);


	RdsUser u("CN=Administrator,CN=Users,DC=resara,DC=local");
	ret = u.read(QStringList() << "nTSecurityDescriptor");
	TEST_FOR_ERRORS(ret);
	base = ret.value<LdapResult>();
	QVERIFY(base.keys().count() == 1);
	foreach(QString attribute, base.keys())
	{
		QByteArray data = base[attribute][0];
		data.replace(3, 1, QByteArray::fromHex("ff"));
		RdsNtSecurityDescriptor desc(base[attribute][0]);

// 		qDebug() << data.toHex();
// 		qDebug();
// 		qDebug() << desc.toBinary().toHex();
		QVERIFY(desc == RdsNtSecurityDescriptor(data));
		QVERIFY(desc.toBinary() == data);
		QVERIFY(desc.toBinary() == RdsNtSecurityDescriptor(desc.toBinary()).toBinary());
	}
	desc2.setOwner("S-1-5-21-2470468500-28526814-1734375037-517");
	desc2.setGroup("S-1-5-21-2470468500-28526814-1734375037-553");
	desc2.setDaclFlags(desc2.daclFlags() ^(RdsNtSecurityDescriptor::Protected | RdsNtSecurityDescriptor::AutoInherited));
	desc2.setSaclFlags(desc2.saclFlags() ^(RdsNtSecurityDescriptor::Protected | RdsNtSecurityDescriptor::AutoInheritReq));
	QList<RdsAce> acelist = desc2.dacl();
	acelist.removeAt(1);
	acelist.removeAt(3);
	acelist.removeAt(5);
	acelist.removeAt(9);
	desc2.setDacl(acelist);
	acelist = desc2.sacl();
	acelist.removeAt(0);
	desc2.setSacl(acelist);

	ace = "(A;;RPWPCRCCDCLCLORCWOWDSDDTSW;;;S-1-5-21-2470468500-28526814-1734375037-512)";
	ace2 = "(OA;CIIOID;RP;4c164200-20c0-11d0-a768-00aa006e0529;4828cc14-1437-45bc-9b07-ad6f015e5f28;RU)";

	TEST_GETTER_SETTER(desc, desc2, owner, setOwner);
	TEST_GETTER_SETTER(desc, desc2, group, setGroup);
	TEST_GETTER_SETTER(desc, desc2, daclFlags, setDaclFlags);
	TEST_GETTER_SETTER(desc, desc2, saclFlags, setSaclFlags);
	TEST_GETTER_SETTER(desc, desc2, dacl, setDacl);
	TEST_GETTER_SETTER(desc, desc2, sacl, setSacl);
	QVERIFY(desc == desc2);

	TEST_GETTER_SETTER(ace, ace2, type, setType);
	TEST_GETTER_SETTER(ace, ace2, flags, setFlags);
	TEST_GETTER_SETTER(ace, ace2, access, setAccess);
	TEST_GETTER_SETTER(ace, ace2, object, setObject);
	TEST_GETTER_SETTER(ace, ace2, inheritedObject, setInheritedObject);
	TEST_GETTER_SETTER(ace, ace2, sid, setSid);
	QVERIFY(ace == ace2);

	RdsGuid guid("12345678-abcd-1234-abcd-123456789010");
	RdsGuid guid2("abcdefaf-1234-abcd-1234-abcdefabcdef");
	TEST_GETTER_SETTER(guid, guid2, timeLow, setTimeLow);
	TEST_GETTER_SETTER(guid, guid2, timeMiddle, setTimeMiddle);
	TEST_GETTER_SETTER(guid, guid2, timeHiAndVersion, setTimeHiAndVersion);
	TEST_GETTER_SETTER(guid, guid2, clockSequence, setClockSequence);
	TEST_GETTER_SETTER(guid, guid2, node, setNode);
	QVERIFY(guid == guid2);
}

void RDSTest::storageDevice()
{
	RdsVolume v("0180d899-3479-4f40-a89f-c38669d01245");
	RdsVolume volume("usbMOFUKAH");
	TEST_OUTPUT(volume.initialize());

	return;
	/*
	RdsVolume volume("20b3153b-582c-4bfa-ac69-60aaf23bee4e");
	TEST_OUTPUT(volume.isMounted());
	TEST_OUTPUT(volume.mountPoint());
	TEST_OUTPUT(volume.size());
	TEST_OUTPUT(volume.fsType());
	TEST_OUTPUT(volume.name());
	TEST_OUTPUT(volume.setName("rkondata"));
	TEST_OUTPUT(volume.name());
	TEST_OUTPUT(volume.id());
	TEST_OUTPUT(volume.storageDevice());
	QStringList list = (volume.list()).toStringList();
	qDebug() << list;
	RdsStorageDevice device(QString("/org/freedesktop/UDisks/devices/sde"));
	TEST_OUTPUT(device.listVolumes());
	TEST_OUTPUT(device.initialize(true, "ext3"));
	TEST_OUTPUT(device.listVolumes());
	{
		RdsVolume volume("/dev/sde1");
		TEST_OUTPUT(volume.storageDevice());
	}
	*/
// 	TEST_OUTPUT(RdsStorageDevicePrivate::createSinglePartition(RdsStorageDevicePrivate::getDevice("/dev/sde")));
// 	TEST_OUTPUT(RdsStorageDevicePrivate::createFilesystem("/dev/sde1", "ext3"));
}

void RDSTest::configReadKey()
{
// 	if (QFile::exists("/tmp/testconf.conf"))
// 		QFile::remove("/tmp/testconf.conf");
	RdsSambaConfigParser config;
	qDebug() << "\n\n" << config.toString() << "\n\n";
	/*
	QMap<QString, QMap<QString, QString> > hash;
	QMap<QString, QMap<QString, QString> > rdshash;
	for (int i = 0; i < 5; ++i) // use 1 section each with 5 keys, each testing 5 values
	{
		for (int j = 0; j < 5; ++j) //assign a bunch of values
		{
			hash["section"][QString("key%1").arg(i)] = QString("value%1").arg(j);
			config.setValue("section", QString("key%1").arg(i), QString("value%1").arg(j));
			QCOMPARE(config.getValue("section", QString("key%1").arg(i)).toString(), QString("value%1").arg(j));
		}
	}
	for (int i = 0; i < 5; ++i) // use 1 section each with 5 keys, each testing 5 values
	{
		for (int j = 0; j < 5; ++j) //assign a bunch of values
		{
			rdshash["section"][QString("key%1").arg(i)] = QString("value%1").arg(j);
			config.setRdsValue("section", QString("key%1").arg(i), QString("value%1").arg(j));
			QCOMPARE(config.getRdsValue("section", QString("key%1").arg(i)).toString(), QString("value%1").arg(j));
		}
	}

	QCOMPARE(config.toMap(), hash);
	QVERIFY(config.getValue("section", "no").isNull());
	QVERIFY(config.getValue("doesntexist", "no").isNull());

	config.disableKey("section", "key1");
	QVERIFY(config.getValue("section", "key1").isNull());
	config.setValue("section", "key1", "value1");
	QCOMPARE(config.getValue("section", "key1").toString(), QString("value1"));
	config.disableKey("section", "key1");
	QVERIFY(config.getValue("section", "key1").isNull());
	config.enableKey("section", "key1");
	QCOMPARE(config.getValue("section", "key1").toString(), QString("value1"));

	config.disableRdsKey("section", "key1");
	QVERIFY(config.getRdsValue("section", "key1").isNull());
	config.setRdsValue("section", "key1", "value1");
	QCOMPARE(config.getRdsValue("section", "key1").toString(), QString("value1"));
	config.disableRdsKey("section", "key1");
	QVERIFY(config.getRdsValue("section", "key1").isNull());
	config.enableRdsKey("section", "key1");
	QCOMPARE(config.getRdsValue("section", "key1").toString(), QString("value1"));

	QCOMPARE(config.listSections().toStringList(), QStringList(hash.keys()));
	QCOMPARE(config.listKeys("section").toStringList(), QStringList(hash["section"].keys()));
	config.disableSection("section");
	hash = QMap<QString, QMap<QString, QString> >();
	QCOMPARE(config.toMap(), hash);
	config.enableSection("section");
	QVERIFY(config.sectionExists("section").toBool());
	config.renameSection("section", "newsection");
	QVERIFY(!config.sectionExists("section").toBool());
	QVERIFY(config.sectionExists("newsection").toBool());

	for (int i = 0; i < 5; ++i)
	{
		config.enableKey("newsection", "key" + QString::number(i));
		config.enableRdsKey("newsection", "key" + QString::number(i));
	}
	QVERIFY(config.keyExists("newsection", "key1").toBool());
	config.removeKey("newsection", "key1");
	QVERIFY(!config.keyExists("newsection", "key1").toBool());
	QVERIFY(config.rdsKeyExists("newsection", "key1").toBool());
	config.removeRdsKey("newsection", "key1");
	QVERIFY(!config.rdsKeyExists("newsection", "key1").toBool());
	QVERIFY(config.getValue("section", "no").isNull());
	QVERIFY(config.getValue("doesntexist", "no").isNull());

	config.removeSection("newsection");
	*/
}

void RDSTest::permissions()
{
// 	if (QProcess::execute("/usr/local/samba/bin/getntacl", QStringList() << "--as-sddl" << "/media/TestTest/") != 0)
// 	{
// 		qWarning() << "Cannot run permissions test case";
// 		return;
// 	}
	QStringList testPerms;
	testPerms << "O:S-1-5-21-1627705053-1474398778-3305600706-500G:S-1-5-21-1627705053-1474398778-3305600706-513D:(D;OIIO;RPWPCRDCLCLORCWOWDSDDTSW;;;S-1-5-21-1627705053-1474398778-3305600706-1012)";
	testPerms << "O:S-1-5-21-1627705053-1474398778-3305600706-500G:S-1-5-21-1627705053-1474398778-3305600706-513D:(A;OICI;0x001f01ff;;;S-1-5-21-1627705053-1474398778-3305600706-500)(A;OICI;0x001f01ff;;;SY)";
	testPerms << "O:S-1-5-21-1627705053-1474398778-3305600706-500G:S-1-5-21-1627705053-1474398778-3305600706-513D:(A;OICI;0x001f01ff;;;S-1-5-21-1627705053-1474398778-3305600706-500)(A;OICI;0x001200a9;;;S-1-5-21-1627705053-1474398778-3305600706-513)(A;;0x001f01ff;;;SY)";
	testPerms << "O:S-1-5-21-1627705053-1474398778-3305600706-500G:S-1-5-21-1627705053-1474398778-3305600706-513D:(A;OICI;0x00180000;;;S-1-5-21-1627705053-1474398778-3305600706-1012)";
	testPerms << "O:S-1-5-21-1627705053-1474398778-3305600706-500G:S-1-5-21-1627705053-1474398778-3305600706-513D:(D;;LC;;;S-1-5-21-1627705053-1474398778-3305600706-1012)";
	testPerms << "O:S-1-5-21-1627705053-1474398778-3305600706-500G:S-1-5-21-1627705053-1474398778-3305600706-513D:(A;CINP;0x00180000;;;S-1-5-21-1627705053-1474398778-3305600706-1012)";

	{
		RdsNtAcl perm = RdsNtAcl::fromFile("/media/TestTest").value<RdsNtAcl>();
		qDebug() << "/media/TestTest" << perm.toString();
	}
	{
		RdsNtAcl perm = RdsNtAcl::fromFile("/media/FuckYou").value<RdsNtAcl>();
		qDebug() << "/media/FuckYou" << perm.toString();
	}
	QBENCHMARK
	{
		foreach(QString str, testPerms)
		{
			RdsNtAcl perm(str);
			QVERIFY(perm == str);
			TEST_FOR_ERRORS(perm.save("/media/TestTest"));
			QVERIFY2(RdsNtAcl::fromFile("/media/TestTest") == perm, qPrintable(RdsNtAcl::fromFile("/media/TestTest").toString() + " == " + perm.toString()));
		}
	}

	QBENCHMARK
	{
		RdsNtAcl perm = RdsNtAcl::fromFile("/media/TestTest").value<RdsNtAcl>();
		QCOMPARE(RdsNtAcl::getNtAclString("/media/TestTest"), perm.toString());
	}
	{
		RdsNtAcl perm = RdsNtAcl::fromFile("/media/TestTest").value<RdsNtAcl>();
		TEST_FOR_ERRORS(perm.save("/media/FuckYou"));
		QCOMPARE(RdsNtAcl::getNtAclString("/media/FuckYou"), perm.toString());
		QCOMPARE(RdsNtAcl::getNtAclString("/media/FuckYou"), RdsNtAcl::getNtAclString("/media/TestTest"));
	}
}


/*
QTEST_MAIN(RDSTest);
/*/
#include <sys/vfs.h>
#include <errno.h>

int main(int argc, char** argv)
{
	RDSTest test;
// 	test.initTestCase();
	test.dhcp();

// 	RdsVolumeManager manager;
	return 0;
}

// */
#include "main.moc"


