/***************************************************************************
 *   Copyright (C) 2009 by Resara LLC   *
 *   brendan@resara.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "rdsdnstester.h"
#include <RdsDnsRecord>

REGISTER_TEST(RdsDnsTester);

#define LOCAL_ZONE_FILE \
	"$ORIGIN .\n" \
	"$TTL 604800	; 1 week\n" \
	"testzone.		IN SOA	resara.local. hostmaster.resara.local. (\n" \
	"				2009112018 ; serial\n" \
	"				172800     ; refresh (2 days)\n" \
	"				14400      ; retry (4 hours)\n" \
	"				3628800    ; expire (6 weeks)\n" \
	"				604800     ; minimum (1 week)\n" \
	"				)\n" \
	"			NS	devserver.resara.local.\n" \
	"		address	A	10.0.4.1\n" \
	"		texty	TXT	text;comment\n"

#define LOCAL_JESU_FILE \
	"$ORIGIN .\n$TTL 604800	; 1 week\n" \
	"jesu.		IN SOA	resara.local. hostmaster.resara.local. (\n" \
	"				2009112018 ; serial\n" \
	"				172800     ; refresh (2 days)\n" \
	"				14400      ; retry (4 hours)\n" \
	"				3628800    ; expire (6 weeks)\n" \
	"				604800     ; minimum (1 week)\n" \
	"				)\n" \
	"			NS	devserver.resara.local.\n" \
	"		address	A	10.0.4.1\n" \
	"		texty	TXT	text;comment\n"

RdsDnsTester::RdsDnsTester(QObject *parent)
		: QObject(parent)
{
}


RdsDnsTester::~RdsDnsTester()
{
}

void RdsDnsTester::init()
{
	RdsDnsZone zone;
	zone = TEST_FUNCTION(manager.addZone("testzone"));
	TEST_FUNCTION(zone.parse(LOCAL_ZONE_FILE));
	ReturnValue ret = TEST_FUNCTION(zone.isValid());
	QVERIFY2(ret.toBool(), qPrintable(zone.errorString().toString()));
	TEST_FUNCTION(zone.save());
}

void RdsDnsTester::cleanup()
{
// 	ReturnValue ret = TEST_FUNCTION(manager.zones());
// 	QStringList list = ret.toStringList();
// 	qDebug() << list.size();
// 	foreach(QString zone, list)
// 	qDebug() << zone;

//	manager.removeZone("Jesu");
	manager.removeZone("autobots");
	manager.removeZone("testzone");
	qDebug() << "\n\nTEST FINISHED\n";
}

void RdsDnsTester::constructorTest()
{
	RdsDnsManager manager2;
	manager2 = manager;
	RdsDnsZone zone;

	RdsDnsZone zone1(zone), zone2;
	zone2 = zone;
	QCOMPARE(zone1.domainName().toString(), zone.domainName().toString());
	QCOMPARE(zone2.domainName().toString(), zone.domainName().toString());


	ReturnValue ret = TEST_FUNCTION(manager.zones());
	QStringList list = ret.toStringList();
	qDebug() << list.size();
	foreach(QString zone, list)
	qDebug() << zone;


	//being thorough
	TEST_FUNCTION(manager.namedConf());
	TEST_FUNCTION(manager.namedConfLocal());
	TEST_FUNCTION(manager.namedConfOptions());


}

void RdsDnsTester::zoneGetSetTest()
{
	RdsDnsZone zone;

	zone = TEST_FUNCTION(manager.zone("testzone"));
	//qDebug() << "HEY FUKCER" << zone.toString().toString();

	ReturnValue ret = TEST_FUNCTION(zone.primaryNameServer());
	QCOMPARE(ret.toString(), QString("resara.local."));
	TEST_FUNCTION(zone.setPrimaryNameServer("bumblebee."));

	ret = TEST_FUNCTION(zone.adminNameServer());
	QCOMPARE(ret.toString(), QString("hostmaster.resara.local."));
	TEST_FUNCTION(zone.setAdminNameServer("optimus.prime."));

	ret = TEST_FUNCTION(zone.domainName());
	QCOMPARE(ret.toString(), QString("testzone"));
	TEST_FUNCTION(zone.setDomainName("autobots"));

	ret = TEST_FUNCTION(zone.soaValues());
	QVariantList vlist = ret.toList();

	QVERIFY2(vlist.size() == 5, "WRONG NUMBER OF SOA VALUES");
	//QVERIFY(vlist.contains("2009112020"));
	QVERIFY(vlist.contains("172800"));
	QVERIFY(vlist.contains("14400"));
	QVERIFY(vlist.contains("3628800"));
	QVERIFY(vlist.contains("604800"));

	TEST_FUNCTION(zone.setSoaValues(QStringList() << "1" << "10" << "100" << "1000" << "10000"));


	ret = TEST_FUNCTION(zone.listRecords());
	RdsDnsRecordList rlist = ret.value<RdsDnsRecordList>();
//	qDebug() << "rlist size" << rlist.size();
	foreach(RdsDnsRecord record, rlist)
	{
		//qDebug() << "Record" << record.type << "key" << record.key << "value" << record.value << "origin" << record.origin << record.comments;
		QCOMPARE(record.origin(), QString("."));
		switch (record.type())
		{
			case RdsDnsZone::NS :
				QCOMPARE(record.key(), QString("IN"));
				QCOMPARE(record.value(), QString("devserver.resara.local."));
				QVERIFY(record.comments().isEmpty());
				break;
			case RdsDnsZone::A :
				QCOMPARE(record.key(), QString("address"));
				QCOMPARE(record.value(), QString("10.0.4.1"));
				QVERIFY(record.comments().isEmpty());
				break;
			case RdsDnsZone::TXT :
				QCOMPARE(record.key(), QString("texty"));
				QCOMPARE(record.value(), QString("text"));
				QVERIFY(record.comments().size() == 1);
				QVERIFY(record.comments().contains("comment"));
				break;
			default:
				QFAIL("DID YOU CHANGE THE RECORDS FOR testzone IN LOCAL_ZONE_FILE");
		}
	}
	qDebug() << "LIST ORIGINS TESTING BEGINNING";
	ret = TEST_FUNCTION(zone.listOrigins());
	QStringList list = ret.toStringList();
	qDebug() << "list size" << list.size();
	for (int i = 0; i < list.size(); ++i)
		qDebug() << "origin" << list[i];
	qDebug() << "LIST ORIGINS TESTING ENDING";

	ret = TEST_FUNCTION(zone.isValid());
	QVERIFY(ret.toBool());

	ret = TEST_FUNCTION(zone.errorString());
	QCOMPARE(ret.toString(), QString("No error"));

	ret = TEST_FUNCTION(zone.toString());
	qDebug() << ret.toString();

	//check save()
	TEST_FUNCTION(zone.save());
	zone = TEST_FUNCTION(manager.zone("autobots"));
	ret = TEST_FUNCTION(zone.isValid());
	QVERIFY2(ret.toBool(), qPrintable(zone.errorString().toString()));

	ret = TEST_FUNCTION(zone.primaryNameServer());
	QCOMPARE(ret.toString(), QString("bumblebee."));

	ret = TEST_FUNCTION(zone.adminNameServer());
	QCOMPARE(ret.toString(), QString("optimus.prime."));

	ret = TEST_FUNCTION(zone.domainName());
	QCOMPARE(ret.toString(), QString("autobots"));

	ret = TEST_FUNCTION(zone.listRecords());
	RdsDnsRecordList rlist2 = ret.value<RdsDnsRecordList>();
	QCOMPARE(rlist2, rlist);

	ret = TEST_FUNCTION(zone.soaValues());
	vlist = ret.toList();

	QVERIFY2(vlist.size() == 5, "WRONG NUMBER OF SOA VALUES");
	QVERIFY(vlist.contains("3"));
	QVERIFY(vlist.contains("10"));
	QVERIFY(vlist.contains("100"));
	QVERIFY(vlist.contains("1000"));
	QVERIFY(vlist.contains("10000"));

	TEST_FUNCTION(zone.setSoaValues(QStringList() << "1" << "10" << "100"));
	TEST_FUNCTION_ERROR(zone.save());
}

void RdsDnsTester::zoneRecordTest()
{
	RdsDnsZone zone;
	zone = TEST_FUNCTION(manager.zone("testzone"));

	ReturnValue ret = TEST_FUNCTION(zone.toString());
	qDebug() << ret.toString();

	ret = TEST_FUNCTION(zone.addRecord(RdsDnsZone::A, "addr", "10.0.45.5"));
	QVERIFY(ret.toBool());

	ret = TEST_FUNCTION(zone.addRecord(RdsDnsZone::TXT, "texty", "byebye"));
	QVERIFY(!ret.toBool());
	TEST_FUNCTION(zone.save());
	ret = TEST_FUNCTION(zone.toString());
	//qDebug() << ret.toString();

	RdsDnsRecord record;
	record.setType(RdsDnsZone::TXT);
	record.setKey("TestRecord");
	record.setValue("replaced test record text");
	ret = TEST_FUNCTION(zone.addRecord(record));
	QVERIFY(ret.toBool());

	ret = TEST_FUNCTION(zone.toString());
	//qDebug() << ret.toString();

	ret = TEST_FUNCTION_ERROR(zone.addRecord(RdsDnsZone::TXT, "another TestRecord", "another test record text"));

	//do some other tests before removing
	ret = TEST_FUNCTION(zone.record(RdsDnsZone::TXT, "TestRecord"));
	QCOMPARE(ret.value<RdsDnsRecord>().value(), QString("replaced test record text"));

	//ret = TEST_FUNCTION(zone.record(RdsDnsZone::TXT, "another TestRecord"));
	//QCOMPARE(ret.value<RdsDnsZone::DnsRecord>().value, QString("another test record text"));

	ret = TEST_FUNCTION(zone.record(RdsDnsZone::TXT, "dne"));
	QCOMPARE(ret.value<RdsDnsRecord>().type(), RdsDnsZone::Invalid);

	ret = TEST_FUNCTION(zone.listRecordsByType(RdsDnsZone::TXT));
	RdsDnsRecordList rlist = ret.value<RdsDnsRecordList>();
	//qDebug() << "rlist size" << rlist.size();
	for (int i = 0; i < rlist.size(); ++i)
	{
		//qDebug() << "Record" << rlist[i].type << rlist[i].key << rlist[i].value;
		QCOMPARE(rlist[i].type(), RdsDnsZone::TXT);
	}

	ret = TEST_FUNCTION(zone.listRecordsByKey("TestRecord"));
	rlist = ret.value<RdsDnsRecordList>();
	//qDebug() << "rlist size" << rlist.size();
	for (int i = 0; i < rlist.size(); ++i)
	{
		//qDebug() << "Record" << rlist[i].type << rlist[i].key << rlist[i].value;
		QCOMPARE(rlist[i].key(), QString("TestRecord"));
		QCOMPARE(rlist[i].value(), QString("replaced test record text"));
	}

	ret = TEST_FUNCTION(zone.listRecordsByOrigin("."));
	rlist = ret.value<RdsDnsRecordList>();
	//qDebug() << "rlist size" << rlist.size();
	for (int i = 0; i < rlist.size(); ++i)
	{
		//qDebug() << "Record" << rlist[i].type << rlist[i].key << rlist[i].value;
		QCOMPARE(rlist[i].origin(), QString("."));
	}

	qDebug() << "LIST ORIGINS TESTING BEGINNING";
	ret = TEST_FUNCTION(zone.listOrigins());
	QStringList list = ret.toStringList();
	qDebug() << "list size" << list.size();
	for (int i = 0; i < list.size(); ++i)
		qDebug() << "origin" << list[i];
	qDebug() << "LIST ORIGINS TESTING ENDING";

	//save
	TEST_FUNCTION(zone.save());
	//qDebug() << zone.toString().toString();

	qDebug() << "LIST ORIGINS TESTING BEGINNING";
	ret = TEST_FUNCTION(zone.listOrigins("local"));
	list = ret.toStringList();
	qDebug() << "list size" << list.size();
	for (int i = 0; i < list.size(); ++i)
		qDebug() << "origin" << list[i];
	qDebug() << "LIST ORIGINS TESTING ENDING";


	ret = TEST_FUNCTION(zone.record(RdsDnsZone::A, "addr"));
	record = ret.value<RdsDnsRecord>();
	QVERIFY(record.type() != RdsDnsZone::Invalid);
	QCOMPARE(record.value(), QString("10.0.45.5"));
	//compare origins?

	ret = TEST_FUNCTION(zone.record(RdsDnsZone::NS, "IN"));
	record = ret.value<RdsDnsRecord>();
	QVERIFY(record.type() != RdsDnsZone::Invalid);
	QCOMPARE(record.value(), QString("devserver.resara.local."));
	QCOMPARE(record.origin(), QString("."));


	ret = TEST_FUNCTION(zone.record(RdsDnsZone::TXT, "texty"));
	record = ret.value<RdsDnsRecord>();
	QVERIFY(record.type() != RdsDnsZone::Invalid);
	QCOMPARE(record.value(), QString("byebye"));
	QCOMPARE(record.origin(), QString("."));

	//now remove
	//ret = TEST_FUNCTION(zone.removeRecord(RdsDnsZone::TXT, "another TestRecord"));
	//QVERIFY(ret.toBool());

	ret = TEST_FUNCTION(zone.removeRecord(RdsDnsZone::A, "addr"));
	QVERIFY(ret.toBool());

	ret = TEST_FUNCTION(zone.removeRecord(record));//texty
	QVERIFY(ret.toBool());

	ret = TEST_FUNCTION(zone.removeRecord(record));
	QVERIFY(!ret.toBool());//should return false if the record was already removed


	//qDebug() << zone.toString().errString();

}

void RdsDnsTester::managerAddRemoveTest()
{
	RdsDnsZone testzone, jesu;
	testzone = TEST_FUNCTION(manager.zone("testzone"));
	ReturnValue ret = TEST_FUNCTION(testzone.toString());

	qDebug() << "BEFORE BEFOREBEFORE BEFOREBEFORE BEFORE\n" << manager.namedConfLocal().toString();



	jesu = TEST_FUNCTION(manager.addZone("jesu"));

	ret = TEST_FUNCTION(manager.zones());
	QStringList list = ret.toStringList();
	foreach(QString z, list) qDebug() << z;

	TEST_FUNCTION(jesu.parse(LOCAL_JESU_FILE));
	ret = TEST_FUNCTION(jesu.isValid());
	QVERIFY2(ret.toBool(), qPrintable(jesu.errorString().toString()));
	TEST_FUNCTION(jesu.save());
	jesu = TEST_FUNCTION(manager.zone("jesu"));

	// Right now you have to save with the zone to make the zone write out, zones and the base manager are treated separately
	qDebug() << "BEFORE SAVE/RELOAD" << manager.namedConfLocal().toString();


	//should be failing? getting "failed to remove 'home/chris/bind.rdsbak': no such file or directory"
	// Don't worry about that error for now
	TEST_FUNCTION(manager.save());
	TEST_FUNCTION(manager.reload());

	ret = TEST_FUNCTION(manager.zones());
	list = ret.toStringList();
	foreach(QString z, list) qDebug() << z;
	QVERIFY(list.contains("jesu"));
	QVERIFY(list.contains("testzone"));

	jesu = TEST_FUNCTION(manager.zone("jesu"));
	ret = TEST_FUNCTION(jesu.isValid());
	QVERIFY2(ret.toBool(), qPrintable(jesu.errorString().toString()));

	testzone = TEST_FUNCTION(manager.zone("testzone"));
	ret = TEST_FUNCTION(testzone.isValid());
	QVERIFY2(ret.toBool(), qPrintable(testzone.errorString().toString()));

	TEST_FUNCTION(manager.removeZone("jesu"));

	TEST_FUNCTION_ERROR(manager.zone("jesu"));

	TEST_FUNCTION(manager.save());
	TEST_FUNCTION(manager.reload());

	TEST_FUNCTION_ERROR(manager.zone("jesu"));
	TEST_FUNCTION(manager.zone("testzone"));

	ret = TEST_FUNCTION(manager.zones());
	list = ret.toStringList();
	foreach(QString z, list) qDebug() << z;
	QVERIFY(!list.contains("jesu"));
	QVERIFY(list.contains("testzone"));
}


void RdsDnsTester::makeSureItsNotBrokenTest()//random things
{
	ReturnValue ret = TEST_FUNCTION(manager.zones());
	QStringList list = ret.toStringList();
	foreach(QString z, list) qDebug() << z;
	QVERIFY(!list.contains("jesu"));
	QVERIFY(list.contains("testzone"));

	RdsDnsZone zone;
	zone = TEST_FUNCTION(manager.zone("testzone"));

	qDebug() << "LIST ORIGINS TESTING BEGINNING";
	ret = TEST_FUNCTION(zone.listOrigins("."));
	list = ret.toStringList();
	qDebug() << "list size" << list.size();
	for (int i = 0; i < list.size(); ++i)
		qDebug() << "origin" << list[i];
	qDebug() << "LIST ORIGINS TESTING ENDING";


}

