/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  backend-ris.c defines the RIS output backend of refdbd
  markus@mhoenicka.de 6-21-00

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

#include <string.h>
#include <syslog.h> /* for definitions of log message priorities */
#include <stdio.h>
#include <dbi/dbi.h>

#include "refdb.h"
#include "linklist.h"
#include "backend.h"
#include "backend-ris.h"
#include "strfncs.h"
#include "refdbd.h"
#include "cgi.h"
#include "dbfncs.h"
#include "connect.h"

extern char refdblib[]; /* location of shareable files */

extern int n_log_level; /* numeric version of log_level */

/* forward declaration of local functions */
char* print_field_ris(const char* item, struct renderinfo* ptr_rendinfo, const char* start_string);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  prepare_render_ris(): writes a header for the ris output of a
                         query

  int prepare_render_ris returns 0 if successful, > 0 if failed

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int prepare_render_ris(struct renderinfo* ptr_rendinfo) {
  /* we just make sure that we start with a clean string */
  (*(ptr_rendinfo->ptr_ref))[0] = '\0';

  return 0;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  finish_render_ris(): writes a footer for the ris output of a query

  int finish_render_ris returns 0 if successful, > 0 if failed

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int finish_render_ris(struct renderinfo* ptr_rendinfo) {
  int n_cgi;
  char* new_ref;
  char* html_foot;

  n_cgi = (ptr_rendinfo->ref_format == 7) ? 1:0;

  if (n_cgi) {
    html_foot = load_html("refdbq_foot", refdblib);
    if (!html_foot) {
      LOG_PRINT(LOG_WARNING, get_status_msg(802));
      return 802;
    }
    if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), html_foot, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
      free(html_foot);
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return 801;
    }
    else {
      *(ptr_rendinfo->ptr_ref) = new_ref;
    }
    free(html_foot);
  }
  /* else: nothing to do */

  return 0;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  render_ris() renders a RIS dataset for RIS export

  int render_ris returns 0 if successful,> 0 if failed

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
int render_ris(struct renderinfo* ptr_rendinfo) {
  int i;
  int n_cgi;
  int errcode; /* receives error code for periodical requests */
  char have_pubyear = 0;
  const char* item;
  const char* type;
  char id[32] = "";
  char* new_ref;
  char date_buffer[256];
  struct REPRINT reprint;
  dbi_result dbires;
  dbi_conn conn;

  conn = dbi_result_get_conn(ptr_rendinfo->dbires);

  n_cgi = (ptr_rendinfo->ref_format == 7) ? 1:0;

  if (get_refdb_id(ptr_rendinfo->dbires, id) == NULL) {
    LOG_PRINT(LOG_WARNING, get_status_msg(234));
    return 234;
  }

  if (n_cgi) {
    /* misuse date_buffer */
    sprintf(date_buffer, "<tr><td class=\"result\"><textarea name=\"updateid%s\" rows=\"25\" cols=\"80\">", id);
    if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), date_buffer, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return 801;
    }
    else {
      *(ptr_rendinfo->ptr_ref) = new_ref;
    }
  }

  /*----------------------------------------------------------------*/
  /* type */
  type = get_refdb_type(ptr_rendinfo->dbires);

  if (print_field_ris(type, ptr_rendinfo, "\nTY  - ") == NULL) {
    return 801;
  }

  /*----------------------------------------------------------------*/
  /* ID */
  item = get_refdb_citekey(ptr_rendinfo->dbires);
  if (item && *item && strstr((ptr_rendinfo->ptr_biblio_info)->format_string, "ID") == NULL) {
    if (print_field_ris(item, ptr_rendinfo, "\nID  - ") == NULL) {
      return 801;
    }
  }
  else { /* ID list must contain numeric IDs, not citekeys
	    item should never be NULL or empty*/
    if (print_field_ris(id, ptr_rendinfo, "\nID  - ") == NULL) {
      return 801;
    }
  }


  /* take a shortcut if the user wants only the IDs. We still make a valid RIS entry - all other tags are optional! */
  if (strstr((ptr_rendinfo->ptr_biblio_info)->format_string, "ID") != NULL) {
    strcat(*(ptr_rendinfo->ptr_ref), "\nER  - \n");
    return 0;
  }

  /*----------------------------------------------------------------*/
  /* authors */
  /* get authors */
  for (i = 1; i < 4; i++) {
    struct AUTHOR_INFO ainfo;

    dbires = request_authors(conn, i, NULL /* all roles */, NULL, 0, my_dbi_result_get_idval(ptr_rendinfo->dbires, "refdb_id"));
    if (dbires == NULL) {
      return 234;
    }

    /* fetch all authors of this article */
    while (get_author_parts(dbires, &ainfo) != NULL) {
      if (i == 1) {
	new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), "\nAU  - ", ptr_rendinfo->ptr_ref_len, 0);
      }
      else if (i == 2) {
	if (has_part_data(type)
	    || !strcmp(ainfo.role, "editor")) {
	  new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), "\nA2  - ", ptr_rendinfo->ptr_ref_len, 0);
	}
	else {
	  new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), "\nAU  - ", ptr_rendinfo->ptr_ref_len, 0);
	}
      }
      else {
	new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), "\nA3  - ", ptr_rendinfo->ptr_ref_len, 0);
      }

      if (new_ref == NULL) {
	clean_request(dbires);
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }

      if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), ainfo.name, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
	clean_request(dbires);
	LOG_PRINT(LOG_CRIT, get_status_msg(801));
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }
    }
    clean_request(dbires);
  }

  /*----------------------------------------------------------------*/
  /* title */
  if ((item = get_refdb_title_copy(ptr_rendinfo->dbires)) != NULL) {
    if (print_field_ris(item, ptr_rendinfo, "\nTI  - ") == NULL) {
      free((char*)item);
      return 801;
    }
    free((char*)item);
  }

  /*----------------------------------------------------------------*/
  /* journal (full)*/
  if ((item = get_periodical(conn, date_buffer, NULL, 0, &errcode, 0, my_dbi_result_get_idval(ptr_rendinfo->dbires, "refdb_id"), NULL /* no frequency required */)) != NULL && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nJF  - ") == NULL) {
      return 801;
    }
  }
  
  /*----------------------------------------------------------------*/
  /* journal (abbrev)*/
  if ((item = get_periodical(conn, date_buffer, NULL, 3, &errcode, 0, my_dbi_result_get_idval(ptr_rendinfo->dbires, "refdb_id"), NULL /* no frequency required */)) != NULL && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nJO  - ") == NULL) {
      return 801;
    }
  }
  
  /*----------------------------------------------------------------*/
  /* journal (custabbrev1) */
  if ((item = get_periodical(conn, date_buffer, NULL, 1, &errcode, 0, my_dbi_result_get_idval(ptr_rendinfo->dbires, "refdb_id"), NULL /* no frequency required */)) != NULL && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nJ1  - ") == NULL) {
      return 801;
    }
  }
  
  /*----------------------------------------------------------------*/
  /* journal (custabbrev2)*/
  if ((item = get_periodical(conn, date_buffer, NULL, 2, &errcode, 0, my_dbi_result_get_idval(ptr_rendinfo->dbires, "refdb_id"), NULL /* no frequency required */)) != NULL && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nJ2  - ") == NULL) {
      return 801;
    }
  }
  
  /*----------------------------------------------------------------*/
  /* keywords */
  dbires = request_keywords(conn, my_dbi_result_get_idval(ptr_rendinfo->dbires, "refdb_id"), 0, 0);
  if (dbires == NULL) {
    return 234;
  }

  /* fetch all keywords of this article */
  while ((item = get_keyword(dbires)) != NULL) {
    if (print_field_ris(item, ptr_rendinfo, "\nKW  - ") == NULL) {
      clean_request(dbires);
      return 801;
    }
  }
  clean_request(dbires);

  /*----------------------------------------------------------------*/
  /* volume */
  if ((item = get_refdb_volume(ptr_rendinfo->dbires)) != NULL) {
    if (!strcmp(type, "BOOK")
	|| !strcmp(type, "ADVS")
	|| !strcmp(type, "MUSIC")
	|| !strcmp(type, "NEWS")) {
      if (print_field_ris(item, ptr_rendinfo, "\nIS  - ") == NULL) {
	return 801;
      }
    }
    else if (!strcmp(type, "CHAP")) {
      if (print_field_ris(item, ptr_rendinfo, "\nM2  - ") == NULL) {
	return 801;
      }
    }
    else {
      if (print_field_ris(item, ptr_rendinfo, "\nVL  - ") == NULL) {
	return 801;
      }
    }
  }

  /*----------------------------------------------------------------*/
  /* title series */
  if ((item = get_refdb_title_series_copy(ptr_rendinfo->dbires)) != NULL) {
    if (print_field_ris(item, ptr_rendinfo, "\nT3  - ") == NULL) {
      free((char*)item);
      return 801;
    }
    free((char*)item);
  }

  /*----------------------------------------------------------------*/
  /* start page */
  if ((item = get_refdb_startpage(ptr_rendinfo->dbires)) != NULL) {
    if (print_field_ris(item, ptr_rendinfo, "\nSP  - ") == NULL) {
      return 801;
    }
  }

  /*----------------------------------------------------------------*/
  /* end page */
  if ((item = get_refdb_endpage(ptr_rendinfo->dbires)) != NULL) {
    if (print_field_ris(item, ptr_rendinfo, "\nEP  - ") == NULL) {
      return 801;
    }
  }

  /*----------------------------------------------------------------*/
  /* abstract */
  if ((item = get_refdb_abstract_copy(ptr_rendinfo->dbires)) != NULL) {
    if (print_field_ris(item, ptr_rendinfo, "\nN2  - ") == NULL) {
      free((char*)item);
      return 801;
    }
    free((char*)item);
  }

  /*----------------------------------------------------------------*/
  /* reprint info, incl availability */
  if (get_reprint(ptr_rendinfo->dbires, &reprint, ptr_rendinfo->username, 2)
      && reprint.reprint != NULL && *(reprint.reprint)) {
    if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), "\nRP  - ", ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
      LOG_PRINT(LOG_WARNING, get_status_msg(801));
      return 801;
    }
    else {
      *(ptr_rendinfo->ptr_ref) = new_ref;
    }
    if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), reprint.reprint, ptr_rendinfo->ptr_ref_len, 0)) == NULL) { 
      LOG_PRINT(LOG_WARNING, get_status_msg(801));
      return 801;
    }
    else {
      *(ptr_rendinfo->ptr_ref) = new_ref;
    }
    if (reprint.date != NULL && *(reprint.date)) {
      if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), " (", ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
	LOG_PRINT(LOG_WARNING, get_status_msg(801));
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }
      if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), reprint.date, ptr_rendinfo->ptr_ref_len, 0)) == NULL) { 
	LOG_PRINT(LOG_WARNING, get_status_msg(801));
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }
      if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), ")", ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
	LOG_PRINT(LOG_WARNING, get_status_msg(801));
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }
    }
  }

  if (reprint.avail != NULL && *(reprint.avail)) {
    if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), "\nAV  - ", ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
      LOG_PRINT(LOG_WARNING, get_status_msg(801));
      return 801;
    }
    else {
      *(ptr_rendinfo->ptr_ref) = new_ref;
    }
    if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), reprint.avail, ptr_rendinfo->ptr_ref_len, 0)) == NULL) { 
      LOG_PRINT(LOG_WARNING, get_status_msg(801));
      return 801;
    }
    else {
      *(ptr_rendinfo->ptr_ref) = new_ref;
    }
  }

  /*----------------------------------------------------------------*/
  /* issue */
  if ((item = get_refdb_issue(ptr_rendinfo->dbires)) != NULL) {
    if (!strcmp(type, "BOOK")
	|| !strcmp(type, "CHAP")
	|| !strcmp(type, "ART")
	|| !strcmp(type, "ADVS")
	|| !strcmp(type, "MUSIC")
	|| !strcmp(type, "NEWS")) {
      if (print_field_ris(item, ptr_rendinfo, "\nVL  - ") == NULL) {
	return 801;
      }
    }
    else {
      if (print_field_ris(item, ptr_rendinfo, "\nIS  - ") == NULL) {
	return 801;
      }
    }
  }

  /*----------------------------------------------------------------*/
  /* book title */
  if ((item = get_refdb_booktitle_copy(ptr_rendinfo->dbires)) != NULL) {
    if (!strcmp(type, "BOOK")
	|| !strcmp(type, "UNPB")) {
      if (print_field_ris(item, ptr_rendinfo, "\nBT  - ") == NULL) {
	free((char*)item);
	return 801;
      }
    }
    else if (has_part_data(type)) {
      if (print_field_ris(item, ptr_rendinfo, "\nT2  - ") == NULL) {
	free((char*)item);
	return 801;
      }
    }
    else {
      if (print_field_ris(item, ptr_rendinfo, "\nT1  - ") == NULL) {
	free((char*)item);
	return 801;
      }
    }
    free((char*)item);
  }

  /*----------------------------------------------------------------*/
  /* city */
  if ((item = get_refdb_city(ptr_rendinfo->dbires)) != NULL) {
    if (print_field_ris(item, ptr_rendinfo, "\nCY  - ") == NULL) {
      return 801;
    }
  }

  /*----------------------------------------------------------------*/
  /* publisher */
  if ((item = get_refdb_publisher(ptr_rendinfo->dbires)) != NULL) {
    if (print_field_ris(item, ptr_rendinfo, "\nPB  - ") == NULL) {
      return 801;
    }
  }

  /*----------------------------------------------------------------*/
  /* address */
  if ((item = get_refdb_address_copy(ptr_rendinfo->dbires)) != NULL) {
    if (print_field_ris(item, ptr_rendinfo, "\nAD  - ") == NULL) {
      free((char*)item);
      return 801;
    }
    free((char*)item);
  }

  /*----------------------------------------------------------------*/
  /* URL, L1 through L4 */

  /* loop over all link types */
  for (i=0; i<6;i++) {
    dbires = request_ulinks(conn, my_dbi_result_get_idval(ptr_rendinfo->dbires, "refdb_id"), 0 /* ref entry */, i /* link type */, 0 /* is_temp */, ptr_rendinfo->username);
    if (dbires == NULL) {
      return 234;
    }

    while ((item = get_ulink(dbires)) != NULL) {
      if (i == 0) {
	if (print_field_ris(item, ptr_rendinfo, "\nUR  - ") == NULL) {
	  clean_request(dbires);
	  return 801;
	}
      }
      else if (i == 1) {
	if (print_field_ris(item, ptr_rendinfo, "\nL1  - ") == NULL) {
	  clean_request(dbires);
	  return 801;
	}
      }
      else if (i == 2) {
	if (print_field_ris(item, ptr_rendinfo, "\nL2  - ") == NULL) {
	  clean_request(dbires);
	  return 801;
	}
      }
      else if (i == 3) {
	if (print_field_ris(item, ptr_rendinfo, "\nL3  - ") == NULL) {
	  clean_request(dbires);
	  return 801;
	}
      }
      else if (i == 4) {
	if (print_field_ris(item, ptr_rendinfo, "\nL4  - ") == NULL) {
	  clean_request(dbires);
	  return 801;
	}
      }
      else if (i == 5) {
	/* DOI needs special treatment as the RIS "designers" crammed
	   it in wherever there was a free slot */
	if (!strcmp(type, "ADVS")) {
	  if (print_field_ris(item, ptr_rendinfo, "\nM2  - ") == NULL) {
	    clean_request(dbires);
	    return 801;
	  }
	}
	if (!strcmp(type, "COMP")
	    || !strcmp(type, "PAT")) {
	  if (print_field_ris(item, ptr_rendinfo, "\nT3  - ") == NULL) {
	    clean_request(dbires);
	    return 801;
	  }
	}
	if (!strcmp(type, "MPCT")
	    || !strcmp(type, "SOUND")
	    || !strcmp(type, "VIDEO")) {
	  if (print_field_ris(item, ptr_rendinfo, "\nIS  - ") == NULL) {
	    clean_request(dbires);
	    return 801;
	  }
	}
	else {
	  if (print_field_ris(item, ptr_rendinfo, "\nM3  - ") == NULL) {
	    clean_request(dbires);
	    return 801;
	  }
	}
      }
    }

    clean_request(dbires);
  } /* end for */

  /*----------------------------------------------------------------*/
  /* notes */
  if ((item = get_notes_copy(ptr_rendinfo->dbires, ptr_rendinfo->username)) != NULL) {
    if (print_field_ris(item, ptr_rendinfo, "\nN1  - ") == NULL) {
      free((char*)item);
      return 801;
    }
    free((char*)item);
  }

  /*----------------------------------------------------------------*/
  /* pubyear */
  if (get_refdb_pubyear(ptr_rendinfo->dbires, date_buffer) != NULL
      && strcmp(date_buffer, "nd")) {
    have_pubyear = 1;
    if (print_field_ris(date_buffer, ptr_rendinfo, "\nPY  - ") == NULL) {
      return 801;
    }
  }

  /*----------------------------------------------------------------*/
  /* pubyear - other information */
  if ((item = get_refdb_pyother_info(ptr_rendinfo->dbires)) != NULL && *item) {
/*     printf("item went to:%s<<\n", item); */
    if (!have_pubyear && strcmp(item, "//")) {
      if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), "\nPY  - ", ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
	LOG_PRINT(LOG_WARNING, get_status_msg(801));
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }
    }
    if (have_pubyear && strcmp(item, "//")) {
      if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), (char*)item, ptr_rendinfo->ptr_ref_len, 0)) == NULL) { 
	LOG_PRINT(LOG_WARNING, get_status_msg(801));
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }
    }
    else if (have_pubyear) { /* empty otherinfo, i.e.'//' */
      if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), "///", ptr_rendinfo->ptr_ref_len, 0)) == NULL) { 
	LOG_PRINT(LOG_WARNING, get_status_msg(801));
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }
    }      
  }
  else if (have_pubyear) {
    /* add /// to make the date entry valid */
    if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), "///", ptr_rendinfo->ptr_ref_len, 0)) == NULL) { 
      LOG_PRINT(LOG_WARNING, get_status_msg(801));
      return 801;
    }
    else {
      *(ptr_rendinfo->ptr_ref) = new_ref;
    }
  }    

  /*----------------------------------------------------------------*/
  /* date secondary */
  have_pubyear = 0; /* reset */
  if (get_refdb_secyear(ptr_rendinfo->dbires, date_buffer) != NULL
      && strcmp(date_buffer, "nd")) {
    have_pubyear = 1;
    if (print_field_ris(date_buffer, ptr_rendinfo, "\nY2  - ") == NULL) {
      return 801;
    }
  }

  /*----------------------------------------------------------------*/
  /* secyear - other information */
  if ((item = get_refdb_secother_info(ptr_rendinfo->dbires)) != NULL && *item) {
    if (!have_pubyear && strcmp(item, "///")) {
      if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), "\nY2  - ", ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
	LOG_PRINT(LOG_WARNING, get_status_msg(801));
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }
    }
    if (have_pubyear || strcmp(item, "///")) {
      if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), (char*)item, ptr_rendinfo->ptr_ref_len, 0)) == NULL) { 
	LOG_PRINT(LOG_WARNING, get_status_msg(801));
	return 801;
      }
      else {
	*(ptr_rendinfo->ptr_ref) = new_ref;
      }
    }
  }
  else if (have_pubyear) {
    /* add /// to make the date entry valid */
    if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), "///", ptr_rendinfo->ptr_ref_len, 0)) == NULL) { 
      LOG_PRINT(LOG_WARNING, get_status_msg(801));
      return 801;
    }
    else {
      *(ptr_rendinfo->ptr_ref) = new_ref;
    }
  }    

  /*----------------------------------------------------------------*/
  /* ISBN/ISSN */
  if ((item = get_refdb_issn(ptr_rendinfo->dbires)) != NULL) {
    if (print_field_ris(item, ptr_rendinfo, "\nSN  - ") == NULL) {
      return 801;
    }
  }

  /*----------------------------------------------------------------*/
  /* user1 */
  if ((item = get_refdb_user1(ptr_rendinfo->dbires)) != NULL) {
    if (print_field_ris(item, ptr_rendinfo, "\nU1  - ") == NULL) {
      return 801;
    }
  }

  /*----------------------------------------------------------------*/
  /* user2 */
  if ((item = get_refdb_user2(ptr_rendinfo->dbires)) != NULL) {
    if (print_field_ris(item, ptr_rendinfo, "\nU2  - ") == NULL) {
      return 801;
    }
  }

  /*----------------------------------------------------------------*/
  /* user3 */
  if ((item = get_refdb_user3(ptr_rendinfo->dbires)) != NULL) {
    if (print_field_ris(item, ptr_rendinfo, "\nU3  - ") == NULL) {
      return 801;
    }
  }

  /*----------------------------------------------------------------*/
  /* user4 */
  if ((item = get_refdb_user4(ptr_rendinfo->dbires)) != NULL) {
    if (print_field_ris(item, ptr_rendinfo, "\nU4  - ") == NULL) {
      return 801;
    }
  }

  /*----------------------------------------------------------------*/
  /* user5 */
  if ((item = get_refdb_user5(ptr_rendinfo->dbires)) != NULL) {
    if (print_field_ris(item, ptr_rendinfo, "\nU5  - ") == NULL) {
      return 801;
    }
  }

  /*----------------------------------------------------------------*/
  /* misc fields - the true horror of roundtripping */
  if ((item = get_refdb_typeofwork(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (!strcmp(type, "THES")) {
      if (print_field_ris(item, ptr_rendinfo, "\nM2  - ") == NULL) {
	return 801;
      }
    }
    else if (!strcmp(type, "PAT")) {
      if (print_field_ris(item, ptr_rendinfo, "\nM3  - ") == NULL) {
	return 801;
      }
    }
    else {
      if (print_field_ris(item, ptr_rendinfo, "\nM1  - ") == NULL) {
	return 801;
      }
    }
  }

  if ((item = get_refdb_area(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nM1  - ") == NULL) {
      return 801;
    }
  }

  if ((item = get_refdb_ostype(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nM2  - ") == NULL) {
      return 801;
    }
  }

  if ((item = get_refdb_degree(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nM1  - ") == NULL) {
      return 801;
    }
  }

  if ((item = get_refdb_runningtime(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nM2  - ") == NULL) {
      return 801;
    }
  }

  if ((item = get_refdb_classcodeintl(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nM1  - ") == NULL) {
      return 801;
    }
  }

  if ((item = get_refdb_classcodeus(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nM2  - ") == NULL) {
      return 801;
    }
  }

  if ((item = get_refdb_senderemail(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nM1  - ") == NULL) {
      return 801;
    }
  }

  if ((item = get_refdb_recipientemail(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nM2  - ") == NULL) {
      return 801;
    }
  }

  if ((item = get_refdb_mediatype(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (!strcmp(type, "MAP")) {
      if (print_field_ris(item, ptr_rendinfo, "\nM2  - ") == NULL) {
	return 801;
      }
    }
    else if (!strcmp(type, "ELEC")
	     || !strcmp(type, "JOUR")) {
      if (print_field_ris(item, ptr_rendinfo, "\nM1  - ") == NULL) {
	return 801;
      }
    }
    else {
      if (print_field_ris(item, ptr_rendinfo, "\nM3  - ") == NULL) {
	return 801;
      }
    }
  }

  if ((item = get_refdb_numvolumes(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nM1  - ") == NULL) {
      return 801;
    }
  }

  if ((item = get_refdb_edition(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nVL  - ") == NULL) {
      return 801;
    }
  }

  if ((item = get_refdb_computer(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nM1  - ") == NULL) {
      return 801;
    }
  }

  if ((item = get_refdb_conferencelocation(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nM2  - ") == NULL) {
      return 801;
    }
  }

  if ((item = get_refdb_registrynum(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nM1  - ") == NULL) {
      return 801;
    }
  }

  if ((item = get_refdb_classification(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nM1  - ") == NULL) {
      return 801;
    }
  }

  if ((item = get_refdb_section(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nM2  - ") == NULL) {
      return 801;
    }
  }

  if ((item = get_refdb_pamphletnum(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nM1  - ") == NULL) {
      return 801;
    }
  }

  if ((item = get_refdb_chapternum(ptr_rendinfo->dbires)) != NULL
      && *item) {
    if (print_field_ris(item, ptr_rendinfo, "\nIS  - ") == NULL) {
      return 801;
    }
  }


  /*----------------------------------------------------------------*/
  /* The End */
  if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), "\nER  - \n", ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
    LOG_PRINT(LOG_WARNING, get_status_msg(801));
    return 801;
  }
  else {
    *(ptr_rendinfo->ptr_ref) = new_ref;
  }

  if (n_cgi) {
    if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), "</textarea></td></tr>", ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
      LOG_PRINT(LOG_WARNING, get_status_msg(801));
      return 801;
    }
    else {
      *(ptr_rendinfo->ptr_ref) = new_ref;
    }
  }

  return 0;
}

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  print_field_ris(): writes simple field contents to HTML output

  char* print_field_ris returns a ptr to the modified string. Due to
                         reallocation this may be different from the
			 ptr passed to this function in the ptr_rendinfo
			 structure which is also updated accordingly

  const char* item ptr to the contents of the field to write

  struct renderinfo* ptr_rendinfo ptr to a structure with the info
                             how the reference should be rendered

  const char* start_string ptr to a string to be printed before the 
                         field contents

 ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
char* print_field_ris(const char* item, struct renderinfo* ptr_rendinfo, const char* start_string) {
  char* new_ref;

  if (item != NULL) {
    if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), (char*)start_string, ptr_rendinfo->ptr_ref_len, 0)) == NULL) {
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return NULL;
    }
    else {
      *(ptr_rendinfo->ptr_ref) = new_ref;
    }
    if ((new_ref = mstrcat(*(ptr_rendinfo->ptr_ref), (char*)item, ptr_rendinfo->ptr_ref_len, 0)) == NULL) { 
      LOG_PRINT(LOG_CRIT, get_status_msg(801));
      return NULL;
    }
    else {
      *(ptr_rendinfo->ptr_ref) = new_ref;
    }
  }
  return *(ptr_rendinfo->ptr_ref);
}
