/***************************************************************************
    file	         : kb_dbtcp.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<time.h>
#include	<sys/types.h>

#ifdef 		_WIN32
#include 	<windows.h>
#endif

#include	<qarray.h>
#include	<qstring.h>
#include	<qdict.h>
#include	<qintdict.h>


#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_databuffer.h"
#include	"kb_database.h"
#include	"kb_serverinfo.h"
#include	"kb_build.h"

#include	"kb_libloader.h"

extern	"C"
{
#include 	<define.h>
#include 	<socket.h>
#include 	<my_error.h>
#include 	<my_string.h>
#include	<my_array.h>
#include	<protocol.h>
}	;


#define		_C(x)	(char *)(const char *)(x)

struct	DBTCPTypeMap
{	int		ident		;	/* DBTCP identifier	*/
	KB::IType	itype		;	/* Internal type	*/
	char		mtype	[16]	;	/* DBTCP type name	*/
	uint		flags		;	/* Various flags	*/
}	;


static	DBTCPTypeMap	rkl_typeMap[] =
{
/*	ident			itype		mtype		flags			*/
{	1,			KB::ITBool,	"Bool",		0,			},
{	2,			KB::ITFixed,	"Integer",	0			},
{	3,			KB::ITFixed,	"Integer",	0			},
{	4,			KB::ITFixed,	"Integer",	0			},
{	5,			KB::ITFloat,	"Float",	FF_LENGTH|FF_PREC	},
{	7,			KB::ITFloat,	"Float",	FF_LENGTH|FF_PREC	},

{	8,		 	KB::ITDateTime,	"DateTime",	0			},

{	10,			KB::ITString,	"Text",		FF_LENGTH		},
{	12,			KB::ITString,	"Memo",		FF_LENGTH		},

{	9,			KB::ITBinary,	"Binary",	FF_LENGTH		},
{	11,			KB::ITBinary,	"LongBinary",	0			},
{	0,			KB::ITUnknown,	"",		0			}
}	;

static	DBTCPTypeMap	sys_typeMap[] =
{
/*	ident			itype		mtype		flags			*/
{	78,			KB::ITFixed,	"Integer",	0			},
{	100,		 	KB::ITDateTime,	"DateTime",	0			},

{	67,			KB::ITString,	"Text",		FF_LENGTH		},
{	0,			KB::ITUnknown,	"",		0			}
}	;




/*  KBDBTCPType								*/
/*  -----------								*/

class	KBDBTCPType : public KBType
{
	DBTCPTypeMap	*m_typeInfo	;

public	:

	KBDBTCPType  (DBTCPTypeMap *, uint, uint, bool) ;

	virtual	bool    isValid		(const QString &, KBError  &, const QString & = QString::null) ;
	virtual	void	getQueryText	(KBDataArray   *, KBShared *, KBDataBuffer  &, QTextCodec *) ;

	virtual	void	escapeText	(KBDataArray   *, KBDataBuffer &) ;
	virtual	void	escapeText	(QCString      &, KBDataBuffer &) ;
}	;



/*  KBDBTCP								*/
/*  -------								*/
/*  Implementation class for insertface to the DBTCP database		*/

class	KBDBTCP : public KBServer
{
	dbftp_result		*dbHandle	;

	bool			m_readOnly	;
	bool			m_rekallTables	;
	DBTCPTypeMap		*m_typeMap	;
	QIntDict<DBTCPTypeMap>	m_dIdentToType	;

	bool			doListTablesSys	(KBTableDetailsList &, bool, uint = KB::IsAny) ;
	bool			doListTablesRkl (KBTableDetailsList &, bool, uint = KB::IsAny) ;
	bool			doListFieldsSys	(KBTableSpec   &) ;
	bool			doListFieldsRkl	(KBTableSpec   &) ;


protected :

	virtual	bool	 	doConnect    	(KBServerInfo  *) ;
	virtual	bool		doListTables 	(KBTableDetailsList &, uint = KB::IsAny) ;
	virtual	bool		doListFields 	(KBTableSpec   &) ;
	virtual	bool		doCreateTable	(KBTableSpec   &,  bool, bool = false) ;
	virtual	bool		doRenameTable	(cchar *, cchar *, bool) ;
	virtual	bool		doDropTable  	(cchar *,	   bool) ;

public	:

	KBDBTCP	() ;
virtual~KBDBTCP () ;


	virtual	KBSQLSelect	*qrySelect 	(bool, const QString &, bool) ;
	virtual	KBSQLUpdate	*qryUpdate 	(bool, const QString &, const QString &) ;
	virtual	KBSQLInsert	*qryInsert 	(bool, const QString &, const QString &) ;
	virtual	KBSQLDelete	*qryDelete 	(bool, const QString &, const QString &) ;

	virtual	bool		command		(bool, const QString &, uint, KBValue *, KBSQLSelect ** = 0) ;

	virtual	QString		listTypes	() ;
	virtual	QString		mapExpression	(const QString &) ;
	virtual	bool		tableExists	(const QString &, bool &) ;

	bool	execSQL
	(	const QString	&,
		QString		&,
		uint		,
		const KBValue	*,
		QTextCodec	*,
		cchar		*,
		KBError		&
	)	;

	inline	dbftp_result	*handle	()
	{
		return	dbHandle	;
	}
}	;


/*  KBDBTCPQrySelect							*/
/*  ----------------							*/
/*  Implementation class for select queries on the DBTCP database	*/

class	KBDBTCPQrySelect : public KBSQLSelect
{
	KBDBTCP		*m_server	;
	QStringList	m_fields	;
	int		m_crow		;

	bool		rowExists	(uint, bool = false) ;

public	:
	KBDBTCPQrySelect (KBDBTCP *, bool, const QString &) ;
virtual~KBDBTCPQrySelect () ;

	virtual	bool	execute		(uint, const KBValue *) ;
	virtual	KBValue	getField	(uint, uint, KBValue::VTrans) ;
	virtual	QString	getFieldName	(uint) ;
}	;


/*  KBDBTCPQryUpdate							*/
/*  ----------------							*/
/*  Implementation class for update queries on the DBTCP database	*/

class	KBDBTCPQryUpdate : public KBSQLUpdate
{
	KBDBTCP	*m_server	;

public	:
	KBDBTCPQryUpdate (KBDBTCP *, bool, const QString &, const QString &) ;
virtual~KBDBTCPQryUpdate () ;

	virtual	bool	execute	 (uint, const KBValue * ) ;

}	;

/*  KBDBTCPQryInsert							*/
/*  ----------------							*/
/*  Implementation class for insert queries on the DBTCP database	*/

class	KBDBTCPQryInsert : public KBSQLInsert
{
	KBDBTCP		*m_server	;
	QString		m_autocol	;
	KBValue		m_newKey	;

public	:
	KBDBTCPQryInsert (KBDBTCP *, bool, const QString &, const QString &) ;
virtual~KBDBTCPQryInsert () ;

	virtual	bool	execute	  (uint, const KBValue *) ;
	virtual	bool	getNewKey (const QString &, KBValue &, bool) ;
}	;

/*  KBDBTCPQryDelete							*/
/*  ----------------							*/
/*  Implementation class for delete queries on the DBTCP database	*/

class	KBDBTCPQryDelete : public KBSQLDelete
{
	KBDBTCP		*m_server	;

public	:
	KBDBTCPQryDelete (KBDBTCP *, bool, const QString &, const QString &) ;
virtual~KBDBTCPQryDelete () ;

	virtual	bool	execute	 (uint, const KBValue * ) ;
}	;


static	QDict	<KBDBTCPType >	typesDict	;

/*  ------------------------------------------------------------------  */

/*  KBDBTCPType								*/
/*  KBDBTCPType	: Constructor for DBTCP type object			*/
/*  typeInfo	: DBTCPTypeMap * : Type information			*/
/*  length	: uint		 : Underlying database length		*/
/*  prec	: uint		 : Underlying database precision	*/
/*  nullOK	: bool		 : True if null is OK			*/
/*  (returns)	: KBDBTCPType	 :					*/

KBDBTCPType::KBDBTCPType
	(	DBTCPTypeMap	*typeInfo,
		uint	  	length,
		uint		prec,
		bool	  	nullOK
	)
	:
	KBType
	(	"DBTCP",
		typeInfo == 0 ? KB::ITUnknown : typeInfo->itype,
		length,
		prec,
		nullOK
	),
	m_typeInfo	(typeInfo)
{
}

/*  KBDBTCPType								*/
/*  isValid	: Test if value is valid for type			*/
/*  value	: const QString & : Value to check			*/
/*  pError	: KBValue &	  : Error return			*/
/*  where	: const QString & : Caller				*/
/*  (returns)	: bool		  : Valid				*/

bool	KBDBTCPType::isValid
	(	const QString	 &value,
		KBError		 &pError,
		const QString	&where
	)
{
	/* *** Specific type checking					*/
	return	KBType::isValid (value, pError, where) ;
}

/*  KBDBTCPType								*/
/*  getQueryText							*/
/*  value	: KBDataArray  * : Raw text of value to convert		*/
/*  d		: KBShared     * : Decoded representation		*/
/*  buffer	: KBDataBuffer & : Results buffer			*/
/*  (returns)	: void		 :					*/

void	KBDBTCPType::getQueryText
	(	KBDataArray	*value,
		KBShared	*d,
		KBDataBuffer	&buffer,
		QTextCodec	*
	)
{
	switch (m_iType)
	{
		case KB::ITDateTime :
			break	;

		case KB::ITFixed    :
			if (value->m_data[0] == 't')
			{	buffer.append ('1') ;
				return	;
			}
			if (value->m_data[0] == 'f')
			{	buffer.append ('0') ;
				return	;
			}
			break	;

		default	:
			break	;
	}

	/* **** Stuff like binary escaping				*/
	KBType::getQueryText (value, d, buffer) ;
}


/*  KBDBTCPType								*/
/*  escapeText	: Escape text characters				*/
/*  value	: KBDataArray  * : Raw text to escape			*/
/*  buffer	: KBDataBuffer & : Escaped text				*/
/*  (returns)	: void		 :					*/

void	KBDBTCPType::escapeText
	(	KBDataArray	*value,
		KBDataBuffer	&buffer
	)
{
	char	   ch	;

	for (uint idx = 0 ; idx < value->m_length ; idx += 1)
		switch (ch = value->m_data[idx])
		{
			case '\'' :
				buffer.append('\'') ;
				buffer.append(ch  ) ;
				break	;

			default	:
				buffer.append(ch  ) ;
				break	;
		}
}

/*  KBDBTCPType								*/
/*  escapeText	: Escape text characters				*/
/*  value	: QCString     & : Raw text to escape			*/
/*  buffer	: KBDataBuffer & : Escaped text				*/
/*  (returns)	: void		 :					*/

void	KBDBTCPType::escapeText
	(	QCString	&value,
		KBDataBuffer	&buffer
	)
{
	char	   ch	;


	for (uint idx = 0 ; idx < value.length() ; idx += 1)
		switch (ch = value[idx])
		{
			case '\'' :
				buffer.append('\'') ;
				buffer.append(ch  ) ;
				break	;

			default	:
				buffer.append(ch  ) ;
				break	;
		}
}




/*  ------------------------------------------------------------------  */

/*  KBDBTCP								*/
/*  KBDBTCP	: Constructor for DBTCP database connection class	*/
/*  (returns)	: KBServer	:					*/

KBDBTCP::KBDBTCP ()
	:
	KBServer ()
{
	/* Assume that the rekall information table is not available;	*/
	/* this will be checked when the connection is made.		*/
	m_rekallTables	= false ;
	dbHandle	= 0	;
}

/*  KBDBTCP								*/
/*  ~KBDBTCP	: Destructor for DBTCP database connection class	*/
/*  (returns)	:		:					*/

KBDBTCP::~KBDBTCP ()
{
	if (dbHandle != 0) free_dbftp_result (dbHandle) ;
}

/*  KBDBTCP								*/
/*  doConnect	: Open connection to database				*/
/*  svInfo	: KBServerInfo *: Server information			*/
/*  (returns)	: bool		: Success				*/

bool	KBDBTCP::doConnect
	(	KBServerInfo	*svInfo
	)
{
	if ((dbHandle = init_dbftp_result()) == 0)
	{
		m_lError = KBError
			   (	KBError::Error,
				TR("Failed to initialise DBTCP"),
				QString::null,
				__ERRLOCN
			   )	;
		return	false	;
	}

	m_readOnly	= svInfo->readOnly  () ;

	QString	host	= svInfo->hostName  ()		;
	int	port	= svInfo->portNumber().toInt()	;

#ifndef	_WIN32
	/* The driver supports SSH tunneling. If there is a tunnel	*/
	/* target then attempt to open the tunnel; if this succeeds	*/
	/* then the host becomes the local host and the port is that	*/
	/* returned by the tunneler.					*/
	fprintf
	(	stderr,
		"KBPgSQL::doConnect: sshTarget=[%s]\n",
		(cchar *)m_sshTarget
	)	;

	if (!m_sshTarget.isEmpty())
	{
		int local = openSSHTunnel(DBTCP_DEFAULT_PORT) ;
		if (local < 0) return false ;

		host	= "127.0.0.1"	;
		port	= local		;
	}
#endif

	QString	dsn	= QString("DSN=%1").arg(svInfo->database()) ;

	if (!svInfo->userName().isEmpty())
		dsn	+= ";UID=" + svInfo->userName() ;
	if (!svInfo->password().isEmpty())
		dsn	+= ";PWD=" + svInfo->password() ;

	if (port == 0) port = DBTCP_DEFAULT_PORT ;

	if (dbftp_connect (dbHandle, _C(host), port, _C(dsn)) != OK)
	{
		m_lError = KBError
			   (	KBError::Error,
				QString(TR("Failed to connect DBTCP to server %1"))
					.arg(svInfo->serverName()),
				dbftp_error_string(dbHandle),
				__ERRLOCN
			   )	;
		return	false	;
	}

	QString	dummy ;
	if (execSQL
		(	"select TableName from __RekallTables where 0 = 1",
			dummy,
			0,
			0,
			0,
			"",
			m_lError
	   )	)
	{
		fprintf
		(	stderr,
			"KBDBTCP::doConnect: using rekall tables\n"
		)	;

		m_rekallTables	= true ;
	}


	if (m_rekallTables)
		m_typeMap = rkl_typeMap ;
	else	m_typeMap = sys_typeMap ;

	m_dIdentToType.clear () ;

	for (uint idx = 0 ; m_typeMap[idx].mtype[0] != 0 ; idx += 1)
	{
		DBTCPTypeMap *m = &m_typeMap[idx] ;
		m_dIdentToType.insert (m->ident, m) ;
	}

	return	true	;
}

/*  KBDBTCP								*/
/*  execSQL	: Execute SQL statement					*/
/*  rawSql	: const QString &: Statement with placeholders		*/
/*  subSql	: QString &	 : Substituted statement for logging	*/
/*  nvals	: uint		 : Number of substitution values	*/
/*  values	: KBValue *	 : Substitution values			*/
/*  codec	: QTextCodec *	 : Non-default codec			*/
/*  emsg	: cchar *	 : Error text on error			*/
/*  pError	: KBError &	 : Error return				*/
/*  (returns)	: bool		 : Success				*/

bool	KBDBTCP::execSQL
	(	const QString	&rawSql,
		QString		&subSql,
		uint		nvals,
		const KBValue	*values,
		QTextCodec	*codec,
		cchar		*emsg,
		KBError		&pError
	)
{
	KBDataBuffer	exeSql	;
	bool	rc	= true	;

	if (!subPlaceList (rawSql, nvals, values, exeSql, codec, pError))
		return false	;

	subSql	= subPlaceList (rawSql, nvals, values, pError) ;
	if (subSql == QString::null)
		return false	;


	if (dbftp_sql (dbHandle, _C(exeSql.data())) != OK)
	{
		pError	= KBError
			  (	KBError::Error,
				TR(emsg),
				QString	("%1\n%2")
					.arg(subSql)
					.arg(dbftp_error_string(dbHandle)),
				__ERRLOCN
			  )	;
		rc	= false	;
	}

//	printQuery (rawSql, nvals, values, rc) ;
	printQuery (subSql, nvals, values, rc) ;
	return	rc ;
}

/*  KBDBTCP								*/
/*  qrySelect	: Open a select query					*/
/*  data	: bool		 : Query for data			*/
/*  select	: const QString& : Select query				*/
/*  update	: bool		 : Select for update			*/
/*  (returns)	: KBSQLSelect *  : Select query class or NULL on error	*/

KBSQLSelect
	*KBDBTCP::qrySelect
	(	bool		data,
		const QString	&select,
		bool
	)
{
	return	new KBDBTCPQrySelect (this, data, select) ;
}


/*  KBDBTCP								*/
/*  qryUpdate	: Open an update query					*/
/*  data	: bool		  : Query for data			*/
/*  update	: const QString & : Update query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  (returns)	: KNQryUpdate *   : Update query class or NULL on error	*/

KBSQLUpdate
	*KBDBTCP::qryUpdate
	(	bool		data,
		const QString	&update,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting update query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	new KBDBTCPQryUpdate (this, data, update, tabName) ;
}

/*  KBDBTCP								*/
/*  qryInsert	: Open an insert query					*/
/*  data	: bool		  : Query for data			*/
/*  insert	: const QString & : Insert query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  (returns)	: KNQryInsert *   : Insert query class or NULL on error	*/

KBSQLInsert
	*KBDBTCP::qryInsert
	(	bool		data,
		const QString	&insert,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting insert query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	new KBDBTCPQryInsert (this, data, insert, tabName) ;
}

/*  KBDBTCP								*/
/*  qryDelete	: Open an delete query					*/
/*  data	: bool		  : Query for data			*/
/*  _delete	: const QString & : Delete query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  (returns)	: KNQryDelete *   : Delete query class or NULL on error	*/

KBSQLDelete
	*KBDBTCP::qryDelete
	(	bool		data,
		const QString	&_delete,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting delete query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	new KBDBTCPQryDelete (this, data, _delete, tabName) ;
}

/*  KBDBTCP								*/
/*  command	: Execute arbitrary SQL					*/
/*  data	: bool		  : Querying for data			*/
/*  query	: const QString & : Query text				*/
/*  nvals	: uint		  : Number of substitution values	*/
/*  values	: KBValue *	  : Substitution values			*/
/*  (returns)	: bool		  : Success				*/

bool	KBDBTCP::command
	(	bool		,
		const QString	&query,
		uint		,
		KBValue		*,
		KBSQLSelect	**
	)
{
	m_lError = KBError
		   (	KBError::Error,
			TR("Driver does not support arbitrary SQL"),
			query,
			__ERRLOCN
		   )	;
	return	false	;
}


/*  KBDBTCP								*/
/*  listTypes	: Get list of types with information flags		*/
/*  (returns)	: QString	: List as bar-separated string		*/

QString	KBDBTCP::listTypes ()
{
	static	QString	typeList ;

	if (typeList.isNull())
	{
		typeList = "Primary Key,0|Foreign Key,0" ;

		for (uint idx = 0 ; m_typeMap[idx].mtype[0] != 0 ; idx += 1)
		{
			DBTCPTypeMap *m = &m_typeMap[idx] ;

			if ((m->flags & FF_NOCREATE) == 0)
				typeList += QString("|%1,%2").arg(m->mtype).arg(m->flags) ;
		}
	}

	return	typeList ;
}

/*  KBDBTCP								*/
/*  tableExists	: See if named table exists				*/
/*  table	: const QString & : Table name				*/
/*  exists	: bool &	  : True if table exists		*/
/*  (returns)	: bool		  : Success				*/

bool	KBDBTCP::tableExists
	(	const QString	&,
		bool		&exists
	)
{
	exists = false	; 
	return	 true	;
}

/*  KBDBTCP								*/
/*  doListTablesSys							*/
/*		: List tables in database using system tables		*/
/*  tabList	: KBTableDetailsList &					*/
/*				: Result list				*/
/*  alltables	: bool		: List all tables			*/
/*  type	: uint		: Get only this type			*/
/*  (returns)	: bool		: Success				*/

bool	KBDBTCP::doListTablesSys
	(	KBTableDetailsList		&tabList,
		bool				,
		uint	
	)
{
	QString	dummy	;

	/* This version access the MSysObjects table to get all table	*/
	/* names.							*/
	if (!execSQL
		(	"select Name from MSysObjects where type = 1",
			dummy,
			0,
			0,
			0,
			"Error retrieving list of tables",
			m_lError
	   )	)
		return	false	;

	while (dbftp_fetch_row (dbHandle) == OK)
	{
		QString	tabName	= dbftp_fetch_value (dbHandle, 0) ;

		tabList.append
		(	KBTableDetails
			(	tabName,
				KB::IsTable,
				QP_SELECT|QP_INSERT|QP_UPDATE|QP_DELETE
		)	)	;
	}

	return	true ;
}

/*  KBDBTCP								*/
/*  doListTablesRkl							*/
/*		: List tables in database using rekall table		*/
/*  tabList	: KBTableDetailsList &					*/
/*				: Result list				*/
/*  alltables	: bool		: List all tables			*/
/*  type	: uint		: Get only this type			*/
/*  (returns)	: bool		: Success				*/

bool	KBDBTCP::doListTablesRkl
	(	KBTableDetailsList		&tabList,
		bool				,
		uint	
	)
{
	QString	dummy	;

	/* Other version, gets the names from the pre-prepared		*/
	/* __RekallTables in the database.				*/
	if (!execSQL
		(	"select distinct TableName from __RekallTables",
			dummy,
			0,
			0,
			0,
			"Error retrieving list of tables",
			m_lError
	   )	)
		return	false	;

	while (dbftp_fetch_row (dbHandle) == OK)
	{
		QString	tabName	= dbftp_fetch_value (dbHandle, 0) ;

		tabList.append
		(	KBTableDetails
			(	tabName,
				KB::IsTable,
				QP_SELECT|QP_INSERT|QP_UPDATE|QP_DELETE
		)	)	;
	}

	return	true ;
}

/*  KBDBTCP								*/
/*  doListTables: List tables in database				*/
/*  tabList	: KBTableDetailsList &					*/
/*				: Result list				*/
/*  type	: uint		: Get only this type			*/
/*  (returns)	: bool		: Success				*/

bool	KBDBTCP::doListTables
	(	KBTableDetailsList	&tabList,
		uint			type
	)
{
	return	m_rekallTables ?
			doListTablesRkl (tabList, m_showAllTables, type) :
			doListTablesSys (tabList, m_showAllTables, type) ;
}

/*  KBDBTCP								*/
/*  doListFieldsSys							*/
/*		: List fields in table, default method			*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  (returns)	: bool		: Success				*/

bool	KBDBTCP::doListFieldsSys
	(	KBTableSpec	&tabSpec
	)
{
	QString	dummy	;

	tabSpec.m_prefKey   = -1    ;
	tabSpec.m_keepsCase = false ;

	/* Jet SQL does not seem to have a "describe table" function,	*/
	/* so do the best we can with a dodgy select.			*/
	if (!execSQL
		(	"select * from [" + tabSpec.m_name + "] where 0 = 1",
			dummy,
			0,
			0,
			0,
			"Error retrieving list of columns",
			m_lError
	   )	)
		return	false	;

	uint	nCols	= dbftp_num_field (dbHandle) ;
	for (uint col = 0 ; col < nCols ; col += 1)
	{
		QString		name	= dbftp_field_name (dbHandle, col) ;
		int		type	= dbftp_field_type (dbHandle, col) ;
		int		len	= dbftp_field_len  (dbHandle, col) ;

		DBTCPTypeMap	*m	= m_dIdentToType.find(type) ;

		QString		mtype	;
		KB::IType	itype	;

		if (m != 0)
		{
			mtype	= m->mtype	;
			itype	= m->itype	;
		}
		else
		{
			mtype	= QString("<Unknown %1>").arg(type) ;
			itype	= KB::ITUnknown	;
		}

		KBFieldSpec *fs = new KBFieldSpec
					(	col,
						name,
						mtype,
						itype,
						0,
						len,
						0
					)	;

		fs->m_dbType	   = new KBDBTCPType (m, len, 0, 0) ;

		tabSpec.m_fldList.append (fs) ;
	}


	return	true ;
}

/*  KBDBTCP								*/
/*  doListFieldsRlk							*/
/*		: List fields in table, using rekall table		*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  (returns)	: bool		: Success				*/

bool	KBDBTCP::doListFieldsRkl
	(	KBTableSpec	&tabSpec
	)
{
	QString	dummy	;
	KBValue	tabName	(tabSpec.m_name) ;

	tabSpec.m_prefKey   = -1    ;
	tabSpec.m_keepsCase = false ;


	/* Interrogate the pre-prepared table on the server to get	*/
	/* information about the table.					*/
	if (!execSQL
		(	"select FieldName, FieldCode, FieldSize,  "
			"	FieldAttr, FieldIndx, FieldReqd   "
			"from	__RekallTables			  "
			"where	TableName = ?			  ",
			dummy,
			1,
			&tabName,
			0,
			"Error retrieving list of columns",
			m_lError
	   )	)
		return	false	;


	while (dbftp_fetch_row (dbHandle) == OK)
	{
		QString	fldName	= dbftp_fetch_value(dbHandle, 0) ;
		int	fldCode	= atoi(dbftp_fetch_value(dbHandle, 1)) ;
		int	fldSize	= atoi(dbftp_fetch_value(dbHandle, 2)) ;
		int	fldAttr	= atoi(dbftp_fetch_value(dbHandle, 3)) ;
		int	fldIndx	= atoi(dbftp_fetch_value(dbHandle, 4)) ;
		bool	fldReqd	= atoi(dbftp_fetch_value(dbHandle, 5)) != 0 ;
		QString	fldDefv	= dbftp_fetch_value(dbHandle, 6) ;


		DBTCPTypeMap	*m	= m_dIdentToType.find(fldCode) ;

		QString		mtype	;
		KB::IType	itype	;
		uint		flags	= 0	;

		if (m != 0)
		{
			mtype	= m->mtype	;
			itype	= m->itype	;
		}
		else
		{
			mtype	= QString("<Unknown %1>").arg(fldCode) ;
			itype	= KB::ITUnknown	;
		}

		if (fldIndx == 1)
			flags	|= KBFieldSpec::Indexed	 ;
		if (fldIndx == 2)
			flags	|= KBFieldSpec::Primary	 | KBFieldSpec::Indexed ;
		if (fldReqd)
			flags	|= KBFieldSpec::NotNull	 ;
		if ((fldAttr & 16) != 0)
			flags	|= KBFieldSpec::InsAvail | KBFieldSpec::Serial | KBFieldSpec::ReadOnly ;

		const uint PrefFlags = KBFieldSpec::Primary|KBFieldSpec::InsAvail ;
		const uint PkeyFlags = KBFieldSpec::Primary|KBFieldSpec::InsAvail|KBFieldSpec::Serial ;

		if ((flags & PrefFlags) == PrefFlags)
		{
			tabSpec.m_prefKey = tabSpec.m_fldList.count() ;
		}
		if ((flags & PkeyFlags) == PkeyFlags)
		{
			mtype		  = "Primary Key" ;
		}

		if (fldDefv[0] == '=') fldDefv = fldDefv.mid(1) ;

		KBFieldSpec *fs = new KBFieldSpec
				  (	tabSpec.m_fldList.count(),
					fldName,
					mtype,
					itype,
					flags,
					fldSize,
					0
				  )	;
		fs->m_dbType	   = new KBDBTCPType (m, fldSize, 0, fldReqd) ;
		fs->m_defval = fldDefv	;
		tabSpec.m_fldList.append (fs) ;
	}


	return	true ;
}

/*  KBDBTCP								*/
/*  doListFields: List fields in table					*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  (returns)	: bool		: Success				*/

bool	KBDBTCP::doListFields
	(	KBTableSpec	&tabSpec
	)
{
	return	m_rekallTables ?
			doListFieldsRkl (tabSpec) :
			doListFieldsSys (tabSpec) ;
}

/*  KBDBTCP								*/
/*  doCreateTable: Create a new table					*/
/*  tabSpec	 : KBTableSpec &: Table specification			*/
/*  assoc	 : bool		: Create associated objects		*/
/*  best	 : bool		: Use best match			*/
/*  (returns)	 : bool		: Success				*/

bool	KBDBTCP::doCreateTable
	(	KBTableSpec	&,
		bool		,
		bool
	)
{
	m_lError = KBError
		   (	KBError::Error,
			"Not implemented",
			"create",
			__ERRLOCN
		   )	;
	return	false	;
}

/*  KBDBTCP								*/
/*  doRenameTable: Rename a  table					*/
/*  oldName	 : cchar *	: Current table name			*/
/*  newName	 : cchar *	: New table name			*/
/*  assoc	 : bool		: Rename associated stuff - none here	*/
/*  (returns)	 : bool		: Success				*/

bool	KBDBTCP::doRenameTable
	(	cchar	*,
		cchar	*,
		bool
	)
{
	m_lError = KBError
		   (	KBError::Error,
			"Not implemented",
			"rename",
			__ERRLOCN
		   )	;
	return	false	;
}

/*  KBDBTCP								*/
/*  doDropTable	: Drop a table						*/
/*  table	: cchar *	: Table name				*/
/*  assoc	: bool		: Drop associated stuff - none here	*/
/*  (returns)	: bool		: Success				*/

bool	KBDBTCP::doDropTable
	(	cchar	*,
		bool
	)
{
	m_lError = KBError
		   (	KBError::Error,
			"Not implemented",
			"drop",
			__ERRLOCN
		   )	;
	return	false	;
}

QString	KBDBTCP::mapExpression
	(	const QString	&expr
	)
{
	static	QString	spec ("_") ;

	return	doMapExpression (expr, "[", "]", spec) ;
}

/*  ------------------------------------------------------------------- */

/*  KBDBTCPQrySelect							*/
/*  KBDBTCPQrySelect							*/
/*		: Constructor for select query object			*/
/*  server	: KBDBTCP *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Select query			*/
/*  (returns)	: KBDBTCPQrySelect:					*/

KBDBTCPQrySelect::KBDBTCPQrySelect
	(	KBDBTCP		*server,
		bool		data,
		const QString	&query
	)	
	:
	KBSQLSelect	(server, data, query),
	m_server	(server)
{
	m_nRows		= 0 ;
	m_nFields	= 0 ;
	m_crow	 	= 0 ;
}

/*  KBDBTCPQrySelect							*/
/*  ~KBDBTCPQrySelect							*/
/*		: Destructor for select query object			*/
/*  (returns)	:		:					*/

KBDBTCPQrySelect::~KBDBTCPQrySelect ()
{
}

/*  KBDBTCPQrySelect							*/
/*  execute	: Execute query						*/
/*  nvals	: uint		: Number of substitution values		*/
/*  values	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBDBTCPQrySelect::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	if (!m_server->execSQL
		(	m_rawQuery,
			m_subQuery,
			nvals,
			values,
			m_codec,
			"Select query failed",
			m_lError
		))	return	false ;

	if (dbftp_fetch_fields (m_server->handle()))
	{
		m_lError = KBError
			   (	KBError::Error,
				TR("Failed to get select query column names"),
				dbftp_error_string(m_server->handle()),
				__ERRLOCN
			   )	;
		return	false	;
	}

	m_nRows		= RowsUnknown ;
	m_nFields	= dbftp_num_field (m_server->handle()) ;
	m_crow		= -1 ;

//	m_fields.clear	() ;

	if (m_types == 0)
	{
	m_types	= new KBType *[m_nFields] ;

	for (uint fld = 0 ; fld < m_nFields ; fld += 1)
	{
		m_fields.append (dbftp_field_name (m_server->handle(), fld)) ;
		m_types[fld] = &_kbString ;
	}
	}

	return	true	;
}

bool	KBDBTCPQrySelect::rowExists
	(	uint		qrow,
		bool
	)
{
	if (m_nRows != KBSQLSelect::RowsUnknown)
		return	(int)qrow < m_nRows ;

	while (m_crow < (int)qrow)
	{
		if (dbftp_fetch_row (m_server->handle()) != OK)
			break	;

		m_crow	+= 1 ;

		KBValue	*values	= new KBValue[getNumFields()] ;

		for (uint colno = 0 ; colno < getNumFields() ; colno += 1)
		{
			char *v =  dbftp_fetch_value(m_server->handle(), colno) ;
			if ((v != 0) && (v[0] != 0))
				values[colno] = v ;
		}

		putInCache (m_crow, values) ;
	}

	return	(m_crow >= 0) && ((int)qrow <= m_crow)	;
}

/*  KBDBTCPQrySelect							*/
/*  getField	: Get a specified field from the query results		*/
/*  qrow	: uint		  : Row number				*/
/*  qcol	: uint		  : Column number			*/
/*  vtrans	: KBValue::VTrans : Translation mode			*/
/*  (returns)	: KBValue	  : Value				*/

KBValue	KBDBTCPQrySelect::getField
	(	uint		qrow,
		uint		qcol,
		KBValue::VTrans
	)
{
	if (rowExists (qrow))
	{
		KBValue	value	;
		return	getFromCache (qrow, qcol, value) ? value : KBValue() ;
	}

	return	KBValue() ;
}

/*  KBDBTCPQrySelect							*/
/*  getFieldname: Get a specified field name from the query results	*/
/*  qcol	: uint		: Column number				*/
/*  (returns)	: QString	: Name					*/

QString	KBDBTCPQrySelect::getFieldName
	(	uint	qcol
	)
{
	return	qcol >= m_nFields ? QString() : m_fields[qcol] ;
}


/*  KBDBTCPQryUpdate							*/
/*  KBDBTCPQryUpdate							*/
/*		: Constructor for update query object			*/
/*  server	: KBDBTCP *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Update query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  (returns)	: KBDBTCPQryUpdate:					*/

KBDBTCPQryUpdate::KBDBTCPQryUpdate
	(	KBDBTCP		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLUpdate (server, data, query, tabName),
	m_server    (server)
{
	m_nRows	= 0 ;
}

/*  KBDBTCPQryUpdate							*/
/*  execute	: Execute query						*/
/*  nvals	: uint		: Number of substitution values		*/
/*  values	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBDBTCPQryUpdate::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	if (!m_server->execSQL
		(	m_rawQuery,
			m_subQuery,
			nvals,
			values,
			m_codec,
			"Update query failed",
			m_lError
		))	return	false ;

	m_nRows	= 1	;
	return	true	;
}

/*  KBDBTCPQryUpdate							*/
/*  ~KBDBTCPQryUpdate							*/
/*		: Destructor for update query object			*/
/*  (returns)	:		:					*/

KBDBTCPQryUpdate::~KBDBTCPQryUpdate ()
{
}


/*  KBDBTCPQryInsert							*/
/*  KBDBTCPQryInsert							*/
/*		: Constructor for insert query object			*/
/*  server	: KBDBTCP *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  tabName	: const QString & : Table being updated			*/
/*  query	: const QString & : Insert query			*/
/*  (returns)	: KBDBTCPQryInsert:					*/

KBDBTCPQryInsert::KBDBTCPQryInsert
	(	KBDBTCP		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLInsert (server, data, query, tabName),
	m_server    (server)
{
	m_nRows	= 0 ;
}

/*  KBDBTCPQryInsert							*/
/*  ~KBDBTCPQryInsert							*/
/*		: Destructor for insert query object			*/
/*  (returns)	:		:					*/

KBDBTCPQryInsert::~KBDBTCPQryInsert ()
{
}

/*  KBDBTCPQryInsert							*/
/*  execute	: Execute query						*/
/*  nvals	: uint		: Number of substitution values		*/
/*  values	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBDBTCPQryInsert::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	QString	dummy	;

	m_newKey = KBValue() ;

	if (!m_server->execSQL
		(	m_rawQuery,
			m_subQuery,
			nvals,
			values,
			m_codec,
			"Insert query failed",
			m_lError
		))	return	false ;

	if (!m_server->execSQL
		(	"select @@IDENTITY",
			dummy,
			0,
			0,
			0,
			"Error retrieving inserted key",
			m_lError
		))	return	false	;

	if (dbftp_fetch_row (m_server->handle()) != OK)
	{
		m_lError = KBError
			   (	KBError::Error,
				TR("Error retrieving inserted key"),
				dbftp_error_string(m_server->handle()),
				__ERRLOCN
			   )	;
		return	false	;

	}

	m_newKey = KBValue (dbftp_fetch_value (m_server->handle(), 0), &_kbFixed) ;
	m_nRows	 = 1	;
	return	 true	;
}

/*  KBDBTCPQryInsert							*/
/*  getNewKey	: Get new insert key					*/
/*  primary	: const QString & : Key column name			*/
/*  _newKey	: KBValue &	  : New key				*/
/*  prior	: bool		  : Pre-insert call			*/
/*  (returns)	: bool		  : Success				*/

bool	KBDBTCPQryInsert::getNewKey
	(	const QString	&,
		KBValue		&_newKey,
		bool		prior
	)
{
	if (prior)
	{
		_newKey	 = KBValue () ;
		return	true ; 
	}

	_newKey = m_newKey ;
	return	true ;
}

/*  KBDBTCPQryDelete							*/
/*  KBDBTCPQryDelete							*/
/*		: Constructor for delete query object			*/
/*  server	: KBDBTCP *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Delete query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  (returns)	: KBDBTCPQryDelete:					*/

KBDBTCPQryDelete::KBDBTCPQryDelete
	(	KBDBTCP		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLDelete (server, data, query, tabName),
	m_server    (server)
{
	m_nRows	= 0 ;
}

/*  KBDBTCPQryDelete							*/
/*  ~KBDBTCPQryDelete							*/
/*		: Destructor for delete query object			*/
/*  (returns)	:		:					*/

KBDBTCPQryDelete::~KBDBTCPQryDelete ()
{
}

/*  KBDBTCPQryDelete							*/
/*  execute	: Execute query						*/
/*  nvals	: uint		: Number of substitution values		*/
/*  values	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBDBTCPQryDelete::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	if (!m_server->execSQL
		(	m_rawQuery,
			m_subQuery,
			nvals,
			values,
			m_codec,
			"Delete query failed",
			m_lError
		))	return	false ;

	m_nRows	= 1	;
	return	true	;
}


#ifndef _WIN32
KBFACTORY
(	KBDBTCPFactory,
	"driver_dbtcp"
)

KBFACTORYIMPL
(	KBDBTCPFactory,
	driver_dbtcp,
	"Rekall DBTCP driver",
	"Plugin",
	"0",
	"7",
	"0"
)
#else
class	KBDBTCPFactory : public KBFactory
{
public:
	inline	KBDBTCPFactory() : KBFactory ()
	{
	}
	virtual	QObject	*create(QObject * = 0, const char *	= 0, const char * = 0, 
		const QStringList &	= QStringList());
};

extern	"C"	__declspec(dllexport) void *init_libkbase_driver_dbtcp()
{							
	return	new KBDBTCPFactory;				
}							
#endif

QObject	*KBDBTCPFactory::create
	(	QObject		  *parent,
		cchar		  *object,
		cchar		  *,
		const QStringList &
	)
{
	if ((parent != 0) && !parent->inherits ("QWidget"))
	{
		fprintf	(stderr, "KBDBTCPFactory: parent does not inherit QWidget\n") ;
		return	0  ;
	}

	if (strcmp (object, "driver"  ) == 0) return new KBDBTCP () ;

	if (strcmp (object, "advanced") == 0) return 0 ;

	return	0 ;
}

cchar	*KBDBTCPFactory::ident ()
{
	return	__KB_BUILD_IDENT	;
}
