/***************************************************************************
    file	         : kb_xbase.cpp
    copyright            : (C) 1999,2000,2001,2002,2003,2004 by Mike Richardson
			   (C) 2000,2001,2002,2003,2004 by theKompany.com
			   (C) 2001,2002,2003,2004 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<time.h>

#include	<time.h>
#include	<sys/types.h>

#include	<qarray.h>
#include	<qdict.h>
#include	<qstring.h>
#include	<qintdict.h>


#include	<xbase/xbase.h>

#include	"xbsql.h"

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_databuffer.h"
#include	"kb_database.h"
#include	"kb_serverinfo.h"
#include	"kb_build.h"

#include	"kb_libloader.h"

#include	"kb_xbase.h"
#include	"kb_xbadvanced.h"



static	XBSQLTypeMap	typeMap[] =
{
{	'L',		KB::ITFixed,	"Bool",		0,			1,	0,  0	},
{	'N',		KB::ITFixed,	"Number",	FF_LENGTH,		0,	10, 0	},
{	'F',		KB::ITFloat,	"Float",	FF_LENGTH|FF_PREC,	0,	12, 2	},
{	'D',		KB::ITDate,	"Date",		0,			8,	0,  0	},
{	'C',		KB::ITString,	"Char",		FF_LENGTH,		0,	50, 0	},
{	'M',		KB::ITBinary,	"Binary",	0,			10,	0,  0	}
}	;



static	QIntDict<XBSQLTypeMap>	dIdentToType	;
static	QDict   <KBXBSQLType >	typesDict	;


/*  convKBXB	: Convert internal values to XBSQL values		*/
/*  kbval	: KBValue *	: Internal value vector			*/
/*  nvals	: uint		: Number of values to convert		*/
/*  codec	: QTextCodec *	: Non-default text codec		*/
/*  (returns)	: XBSQLValue *	: XBSQL value vector			*/

static	XBSQLValue *convKBXB
	(	const KBValue	*kbval,
		uint		nvals,
		QTextCodec	*codec
	)
{
	XBSQLValue *values	= new XBSQLValue[nvals == 0 ? 1 : nvals] ;
	XBSQLValue *xbval	= values ;

	while (nvals > 0)
	{
		uint	dlen	= kbval->dataLength() ;
		cchar	*data	= kbval->isNull    () ? "" : kbval->dataPtr() ;

		if (!kbval->isNull())
		{
			switch (kbval->getType()->getIType())
			{
				case KB::ITUnknown	:
					*xbval	= 0  ;
					break	;

				case KB::ITString	:
					if (codec != 0)
					{
						QString  text  = QString::fromUtf8  (data, dlen) ;
						QCString local = codec->fromUnicode (text) ;
						*xbval	= (cchar *)local ;
					}
					else	*xbval	= data	;
					break	;

				case KB::ITFixed	:
					*xbval	= atoi   (data) ;
					break	;

				case KB::ITFloat	:
					*xbval	= strtod (data, 0) ;
					break	;

				case KB::ITDate		:
					*xbval	= (cchar *)kbval->getText("Date:%Y%m%d") ;
					xbval->tag = XBSQL::VDate ;
					break	;

				case KB::ITBinary	:
					xbval->setBinary    (dlen, data) ;
					break	;

				default	:
					*xbval	= 0 ;
					break	;
			}
		}
		else	*xbval	= ""	;

		kbval	+= 1 ;
		xbval	+= 1 ;
		nvals	-= 1 ;
	}

	return	values	;
}

/*  ------------------------------------------------------------------  */

/*  KBXBSQLType								*/
/*  KBXBSQLType	: Constructor for PostgreSQL type object		*/
/*  typeInfo	: XBSQLTypeMap * : Type information			*/
/*  length	: uint		 : Database field length		*/
/*  prec	: uint		 : Precision				*/
/*  nullOK	: bool		 : True if null is OK			*/
/*  xType	: xbShort	 : XBase type				*/
/*  (returns)	: KBXBSQLType	 :					*/

KBXBSQLType::KBXBSQLType
	(	XBSQLTypeMap	*typeInfo,
		uint	  	length,
		uint	  	prec,
		bool	  	nullOK
	)
	:
	KBType
	(	"XBase",
		typeInfo == 0 ? KB::ITUnknown : typeInfo->itype,
		length,
		prec,
		nullOK
	),
	m_typeInfo (typeInfo)
{
}

/*  KBXBSQLType								*/
/*  isValid	: Test if value is valid for type			*/
/*  value	: const QString & : Value to check			*/
/*  pError	: KBError &	  : Error return			*/
/*  where	: const QString & : Caller				*/
/*  (returns)	: bool		  : Value				*/

bool	KBXBSQLType::isValid
	(	const QString	&value,
		KBError		&pError,
		const QString	&caller
	)
{
	return	KBType::isValid (value, pError, caller) ;
}

/*  KBXBSQLType								*/
/*  getQueryText: Get text for insertion into a query			*/
/*  value	: KBDataArray  * : Raw text of value to convert		*/
/*  d		: KBShared     * : Decoded representation		*/
/*  buffer	: KBDataBuffer & : Result buffer			*/
/*  codec	: QTextCodec *	 : Non-default codec			*/
/*  (returns)	: void		 :					*/

void	KBXBSQLType::getQueryText
	(	KBDataArray	*value,
		KBShared	*d,
		KBDataBuffer	&buffer,
		QTextCodec	*codec
	)
{
	QCString res = "'" ;

	if (value != 0)
		switch (m_iType)
		{
			case KB::ITDate		:
			case KB::ITDateTime	:
				buffer.append (((KBDateTime *)d)->format("%Y%m%d")) ;
				buffer.append ('\'') ;
				return	;

			case KB::ITTime		:
				buffer.append ("null") ;
				return	;

			default	:
				break	;
		}

	KBType::getQueryText (value, d, buffer, codec) ;
}

/*  ------------------------------------------------------------------  */

/*  KBXBSQL								*/
/*  KBXBSQL	: Constructor for PostgreSQL database connection class	*/
/*  (returns)	: KBServer	:					*/

KBXBSQL::KBXBSQL ()
	:
	KBServer ()
{
	m_xBase	  	 = 0	    ;
	m_startat 	 = time (0) ;
	m_keyseq  	 = 0	    ;
	m_mapExpressions = false    ;
}

/*  KBXBSQL								*/
/*  ~KBXBSQL	: Destructor for PostgresQL database connection class	*/
/*  (returns)	:		:					*/

KBXBSQL::~KBXBSQL ()
{
	fprintf	(stderr, "KBXBSQL::~KBXBSQL() m_xBase=%p\n", (void *)m_xBase) ; 
	if (m_xBase != 0) delete m_xBase ;
}

/*  KBXBSQL								*/
/*  doConnect	: Open connection to database				*/
/*  svInfo	: KBServerInfo *: Server information			*/
/*  (returns)	: bool		: Success				*/

bool	KBXBSQL::doConnect
	(	KBServerInfo	*svInfo
	)
{
	fprintf
	(	stderr,
		"KBXBSQL::doConnect: [%s]\n",
		(cchar *)m_database
	)	;

	m_readOnly = svInfo->readOnly () ;

	if (m_database.isEmpty() || (m_database == "."))
		m_database = svInfo->getDBPath() ;

	fprintf
	(	stderr,
		"KBXBSQL::doConnect: [%s]\n",
		(cchar *)m_database
	)	;

	/* Check that we are not already connected. Note that the	*/
	/* notConnected routine sets an error message.			*/
	if (m_xBase != 0)
	{	m_lError = KBError
			   (	KBError::Error,
				"Already connected to XBase database",
				QString::null,
				__ERRLOCN
			   )	;
		return	false	;
	}

	m_xBase	= new XBaseSQL (m_database) ;

	if (svInfo->advanced() != 0)
		if (svInfo->advanced()->isType("xbase"))
		{
			KBXBAdvanced *a = (KBXBAdvanced *)svInfo->advanced() ;
			m_xBase->setClosePack     (a->m_packOnClose  ) ;
			m_xBase->setCaseSensitive (a->m_caseSensitive) ;
			m_xBase->setGoSlow	  (a->m_goSlow	     ) ;
			m_mapExpressions = a->m_mapExpressions	;
		}
		else	/* If the type is wrong then show a warning.	*/
			/* This should never happen unless someone	*/
			/* hand edits the XML database file.		*/
			KBError::EError
			(	TR("Driver error"),
				TR("Invalid advanced options, ignoring"),
				__ERRLOCN
			)	;

	return	true	;
}

QString	KBXBSQL::getNewKey ()
{
	QString	 newKey	= QString().sprintf("%09ld.%09ld", m_startat, m_keyseq) ;
	m_keyseq += 1   ;
	return	 newKey	;
}


/*  KBXBSQL								*/
/*  qrySelect	: Open a select query					*/
/*  data	: bool		 : Query for data			*/
/*  select	: const QString& : Select query				*/
/*  update	: bool		 : Select for update			*/
/*  (returns)	: KBSQLSelect *  : Select query class or NULL on error	*/

KBSQLSelect
	*KBXBSQL::qrySelect
	(	bool		data,
		const QString	&select,
		bool
	)
{
	return	new KBXBSQLQrySelect
		(	this,
			data,
			select
		)	;
}


/*  KBXBSQL								*/
/*  qryUpdate	: Open an update query					*/
/*  data	: bool		 : Query for data			*/
/*  update	: const QString &: Update query				*/
/*  tabName	: const QString &: Table name				*/
/*  (returns)	: KNQryUpdate *  : Update query class or NULL on error	*/

KBSQLUpdate
	*KBXBSQL::qryUpdate
	(	bool		data,
		const QString	&update,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting update query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	new KBXBSQLQryUpdate
		(	this,
			data,
			update,
			tabName
		)	;
}

/*  KBXBSQL								*/
/*  qryInsert	: Open an insert query					*/
/*  data	: bool		 : Query for data			*/
/*  insert	: const QString &: Insert query				*/
/*  tabName	: const QString &: Table name				*/
/*  (returns)	: KNQryInsert *  : Insert query class or NULL on error	*/

KBSQLInsert
	*KBXBSQL::qryInsert
	(	bool		data,
		const QString	&insert,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting insert query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	new KBXBSQLQryInsert
		(	this,
			data,
			insert,
			tabName
		)	;
}

/*  KBXBSQL								*/
/*  qryDelete	: Open a delete query					*/
/*  data	: bool		 : Query for data			*/
/*  _delete	: const QString &: Delete query				*/
/*  tabName	: const QString &: Table name				*/
/*  (returns)	: KNQryDelete *  : Delete query class or NULL on error	*/

KBSQLDelete
	*KBXBSQL::qryDelete
	(	bool		data,
		const QString	&_delete,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting delete query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	new KBXBSQLQryDelete
		(	this,
			data,
			_delete,
			tabName
		)	;
}

/*  KBXBSQL								*/
/*  command	: Execute arbitrary SQL					*/
/*  data	: bool		  : Querying for data			*/
/*  rawqry	: const QString & : Query text				*/
/*  nvals	: uint		  : Number of substitution values	*/
/*  values	: KBValue *	  : Substitution values			*/
/*  select	: KBSQLSelect **  : Return for data queries		*/
/*  (returns)	: bool		  : Success				*/

bool	KBXBSQL::command
	(	bool		data,
		const QString	&rawqry,
		uint		nvals,
		KBValue		*values,
		KBSQLSelect	**
	)
{
	XBSQLValue* xbvals = convKBXB 	  (values, nvals, getCodec(data)) ;
	QString	    subqry = subPlaceList (rawqry, nvals, values, m_lError) ;

	bool	    parsed ;
	XBSQLQuery  *query = getXBase()->openQuery (rawqry, parsed) ;

	if (query == 0)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	"Error parsing query",
				getXBase()->lastError(),
				__ERRLOCN
			   )	;

		printQuery (rawqry, nvals, values, false) ;

		delete [] xbvals;
		return	false	;
	}

	if (!query->execute (nvals, xbvals))
	{
		m_lError = KBError
			   (	KBError::Error,
			 	"Error executing query",
				getXBase()->lastError(),
				__ERRLOCN
			   )	;

		printQuery (rawqry, nvals, values, false) ;

		delete [] xbvals;
		return	false	;
	}

	printQuery (rawqry, nvals, values, true) ;
	delete	[] xbvals;
	delete	query	 ;
	return	true	 ;
}


/*  KBXBSQL								*/
/*  tableExists	: See if named table exists				*/
/*  table	: const QString & : Table name				*/
/*  exists	: bool &	  : True if table exists		*/
/*  (returns)	: bool		  : Success				*/

bool	KBXBSQL::tableExists
	(	const QString	&table,
		bool		&exists
	)
{
	XBSQLTableSet	*tabSet	;

	if ((tabSet = m_xBase->getTableSet ()) == 0)
	{
		m_lError = KBError
			   (	KBError::Error,
				"Unable to get list of tables in database",
				m_xBase->lastError(),
				__ERRLOCN
			   )	;
		return	false	;
	}

	exists	= false	;

	for (int idx = 0 ; idx < tabSet->getNumTables() ; idx += 1)
{
#ifndef	_WIN32
		if (tabSet->getTableName(idx) == table)
#else
		if (QString(tabSet->getTableName(idx)).lower() == table.lower())
#endif
		{	exists	= true	;
			break	;
		}
}
	return	true	;
}

/*  KBXBSQL								*/
/*  doListTables List tables in database				*/
/*  tabList	: KBTableDetailsList &					*/
/*				: Result list				*/
/*  type	: uint		: Type flags				*/
/*  (returns)	: bool		: Success				*/

bool	KBXBSQL::doListTables
	(	KBTableDetailsList	&tabList,
		uint			type
	)
{
	XBSQLTableSet	*tabSet	;

	if ((tabSet = m_xBase->getTableSet ()) == 0)
	{
		m_lError = KBError
			   (	KBError::Error,
				"Unable to get list of tables in database",
				m_xBase->lastError(),
				__ERRLOCN
			   )	;
		return	false	;
	}

	if ((type & KB::IsTable) == 0) return true ;

	for (int idx = 0 ; idx < tabSet->getNumTables() ; idx += 1)
	{
		QString	tabName	= tabSet->getTableName (idx) ;

		if (!m_showAllTables)
			if (tabName.left(8) == "__Rekall")
				continue ;
		tabList.append
		(	KBTableDetails
			(	tabName,
				KB::IsTable,
				QP_SELECT|QP_INSERT|QP_UPDATE|QP_DELETE
		)	)	;
	}

	return	true	;
}


/*  KBXBSQL								*/
/*  doListField: List fields in table					*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  (returns)	: bool		: Success				*/

bool	KBXBSQL::doListFields
	(	KBTableSpec	&tabSpec
	)
{
	XBSQLFieldSet	*fldSet	;
	QString		idxName	;

	if ((fldSet = m_xBase->getFieldSet (tabSpec.m_name)) == 0)
	{
		m_lError = KBError
			   (	KBError::Error,
				"Unable to get list of fields in table",
				m_xBase->lastError(),
				__ERRLOCN
			   )	;
		return	false	;
	}

	tabSpec.m_prefKey   = -1   ;
	tabSpec.m_keepsCase = true ;

	for (int idx = 0 ; idx < fldSet->getNumFields() ; idx += 1)
	{
		xbShort		type	= fldSet->getFieldType   (idx ) ;
		uint		fldLen	= fldSet->getFieldLength (idx ) ;
		uint		fldPrec	= fldSet->getFieldPrec   (idx ) ;
		XBSQL::Index	indexed	= fldSet->fieldIndexed   (idx ) ;

		XBSQLTypeMap	*ptr	= dIdentToType.find      (type) ;
		cchar		*ftype	= ptr == 0 ? "<Unknown>" : ptr->xtype ;

		bool		primary	= false	;
		uint		flags	= 0	;

		if ( (idx    ==  0) &&
		     (ptr    !=  0) &&
		     (fldLen == 22) && (ptr->itype == KB::ITString) )
		{
			ftype	= "Primary Key" ;
			primary	= true		;
			tabSpec.m_prefKey = 0	;
		}

		if (primary)
			flags	|= KBFieldSpec::Primary |
				   KBFieldSpec::NotNull |
				   KBFieldSpec::Unique  |
				   KBFieldSpec::InsAvail;

		switch (indexed)
		{
			case XBSQL::IndexUnique	   :
				flags	|= KBFieldSpec::Indexed |
					   KBFieldSpec::Unique  ;
				break	;

			case XBSQL::IndexNotUnique :
				flags	|= KBFieldSpec::Indexed ;
				break	;

			default	:
				break	;
		}

		KBFieldSpec *s = new KBFieldSpec
				 (	(uint)idx,
					fldSet->getFieldName   (idx),
					ftype,
					ptr == 0 ? KB::ITUnknown : ptr->itype,
					flags,
					fldLen,
					fldPrec
				 )	;

		tabSpec.m_fldList.append (s) ;
	}

	delete	fldSet	;
	return	true	;
}

/*  KBXBSQL								*/
/*  doCreateTable: Create a new table					*/
/*  tabSpec	 : KBTableSpec &: Table specification			*/
/*  assoc	 : bool		: Create associated stuff		*/
/*  best	 : bool		: Use best match			*/
/*  (returns)	 : bool		: Success				*/

bool	KBXBSQL::doCreateTable
	(	KBTableSpec	&tabSpec,
		bool		,
		bool
	)
{
	xbSchema	*schema	 = new xbSchema	   [tabSpec.m_fldList.count() + 1] ;
	XBSQL::Index	*index	 = new XBSQL::Index[tabSpec.m_fldList.count() + 1] ;
	int	 	pKeyCol = -1 ;

	/* Run though the field specification list, assembling a	*/
	/* corresponding set of schema.					*/
	for (uint idx = 0 ; idx < tabSpec.m_fldList.count() ; idx += 1)
	{
		KBFieldSpec	*fSpec	= tabSpec.m_fldList.at(idx) ;
		xbSchema	&s	= schema[idx] ;
		XBSQLTypeMap	*mapp	= 0 ;

		if	((fSpec->m_flags & KBFieldSpec::Unique ) != 0)
		{
			index[idx] = XBSQL::IndexUnique    ;
		}
		else if ((fSpec->m_flags & KBFieldSpec::Indexed) != 0)
		{
			index[idx] = XBSQL::IndexNotUnique ;
		}
		else	index[idx] = XBSQL::IndexNone      ;

		/* Special case. If the field is "Primary Key" then we	*/
		/* create a text column into which we can store unique	*/
		/* values which we generate as primary keys. This is a	*/
		/* fudge, but XBase has no suitable notion. We do the	*/
		/* same for "Foreign Key" ...				*/
		QString	ftype	= fSpec->m_typeName ;
		if (ftype == "Primary Key")
		{
			strncpy	(s.FieldName, fSpec->m_name, sizeof(s.FieldName)) ;
			s.FieldName[sizeof(s.FieldName) - 1] = 0 ;
			s.Type	   = 'C' ;
			s.FieldLen = 22	 ;
			s.NoOfDecs = 0	 ;
			pKeyCol	   = idx ;
			continue   ;
		}
		if (ftype == "Foreign Key")
		{
			strncpy	(s.FieldName, fSpec->m_name, sizeof(s.FieldName)) ;
			s.FieldName[sizeof(s.FieldName) - 1] = 0 ;
			s.Type	   = 'C' ;
			s.FieldLen = 22	 ;
			s.NoOfDecs = 0	 ;
			continue   ;
		}

		/* Map the types used when creating the object and	*/
		/* design dictionary tables.				*/
		if	(ftype == "_Text"   ) ftype = "Char"    ;
		else if (ftype == "_Binary" ) ftype = "Binary"	;
		else if (ftype == "_Integer")
		{
			ftype = "Number"     ;
			fSpec->m_length = 10 ;
		}


		for (uint typ = 0 ; typ < sizeof(typeMap)/sizeof(XBSQLTypeMap) ; typ += 1)
			if (typeMap[typ].xtype == ftype)
				if ((typeMap[typ].flags & FF_NOCREATE) == 0)
				{
					mapp	= &typeMap[typ] ;
					break	;
				}
		if (mapp == 0)
		{	m_lError = KBError
				   (	KBError::Fault,
				  	"Error mapping column type",
				  	QString ("Type %1 for column %2 not known")
						.arg(ftype)
						.arg(fSpec->m_name),
				  	__ERRLOCN
				   )	;
			delete [] schema;
			delete [] index ;
			return	false	;
		}

		strncpy	(s.FieldName, fSpec->m_name, sizeof(s.FieldName)) ;
		s.FieldName[sizeof(s.FieldName) - 1] = 0 ;
		s.Type	   = mapp ->ident  ;

		if ((mapp->flags & FF_LENGTH) == 0)
			s.FieldLen = mapp ->length   ;
		else	s.FieldLen = fSpec->m_length ;

		if ((mapp->flags & FF_PREC  ) == 0)
			s.NoOfDecs = 0 ;
		else	s.NoOfDecs = fSpec->m_prec ;
	}

	if (pKeyCol > 0)
	{
		m_lError = KBError
			   (	KBError::Error,
			  	"Primary key column must be the first",
			  	QString::null,
			  	__ERRLOCN
			   )	;
		delete [] schema;
		delete [] index ;
		return	false	;
	}

	xbSchema &s 	= schema[tabSpec.m_fldList.count()] ;
	s.FieldName[0]	= 0 ;
	s.Type		= 0 ;
	s.FieldLen	= 0 ;
	s.NoOfDecs 	= 0 ;

	if (!m_xBase->createTable (tabSpec.m_name, schema, index))
	{
		m_lError = KBError
			   (	KBError::Error,
				"Unable to create new table",
				m_xBase->lastError(),
				__ERRLOCN
			   )	;
		delete [] schema;
		delete [] index ;
		return	false	;
	}
	delete [] schema;
	delete [] index ;
	return	true	;
}

/*  KBXBSQL								*/
/*  doRenameTable: Rename a  table					*/
/*  oldName	 : cchar *	: Current table name			*/
/*  newName	 : cchar *	: New table name			*/
/*  assoc	 : bool		: Rename associated stuff - none here	*/
/*  (returns)	 : bool		: Success				*/

bool	KBXBSQL::doRenameTable
	(	cchar	*oldName,
		cchar	*newName,
		bool	
	)
{
	if (!m_xBase->renameTable (oldName, newName))
	{
		m_lError = KBError
			   (	KBError::Fault,
			  	QString ("Failed to rename table \"%1\" as \"%2\"")
					.arg(oldName)
					.arg(newName),
			  	m_xBase->lastError(),
			  	__ERRLOCN
			   )	;
		return	false	;
	}

	return	true	;
}

/*  KBXBSQL								*/
/*  doDropTable	: Drop a table						*/
/*  table	: cchar *	: Table name				*/
/*  assoc	: bool		: Drop associated stuff - none here	*/
/*  (returns)	: bool		: Success				*/

bool	KBXBSQL::doDropTable
	(	cchar	*table,
		bool	
	)
{
	if (!m_xBase->dropTable (table))
	{
		m_lError = KBError
			   (	KBError::Fault,
			  	QString("Failed to delete table \"%1\"").arg(table),
			  	m_xBase->lastError(),
			  	__ERRLOCN
			   )	;
		return	false	;
	}

	return	true	;
}

/*  KBServer								*/
/*  listTypes	: Return a list of all column types			*/
/*  (returns)	: QString	: Types					*/

QString	KBXBSQL::listTypes ()
{
	static	QString	typeList ;

	if (typeList.isNull())
	{
		typeList = "Primary Key,0|Foreign Key,0" ;

		for (uint idx = 0 ; idx < sizeof(typeMap)/sizeof(XBSQLTypeMap) ; idx += 1)
		{
			XBSQLTypeMap *m = &typeMap[idx] ;

			if ((m->flags & FF_NOCREATE) == 0)
				typeList += QString("|%1,%2,%3,%4")
						.arg(m->xtype  )
						.arg(m->flags  )
						.arg(m->deflen )
						.arg(m->defprec) ;
		}
	}

	return	typeList ;
}

/*  KBXBSQL								*/
/*  optionFlags	: Get server option flags				*/
/*  (returns)	: uint		: Flags					*/

uint	KBXBSQL::optionFlags ()
{
	return	0	;
}

/*  KBXBSQL								*/
/*  mapExpression: Map expressions with square brackets			*/
/*  expr	 : const QString & : Expression				*/
/*  (returns)	 : QString	   : Mapped expression			*/

QString	KBXBSQL::mapExpression
	(	const QString	&expr
	)
{
	static	QString	spec ("_") ;

	return	m_mapExpressions ?
			doMapExpression (expr, "[", "]", spec) :
			expr ;
}


/*  ------------------------------------------------------------------- */


/*  KBXBSQLQrySelect							*/
/*  KBXBSQLQrySelect							*/
/*		: Constructor for select query object			*/
/*  server	: KBXBSQL *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Select query text			*/
/*  codec	: QTextCodec *	  : Non-default codec			*/
/*  (returns)	: KBXBSQLQrySelect:					*/

KBXBSQLQrySelect::KBXBSQLQrySelect
	(	KBXBSQL		*server,
		bool		data,
		const QString	&select
	)	
	:
	KBSQLSelect (server, data, select),
	m_server    (server)
{
	m_nRows	   = 0 ;
	m_nFields  = 0 ;
	m_lastDump = 0 ;
	m_subQuery = m_rawQuery ;

	QCString qryText = select.utf8() ;

	if ((m_xSelect = m_server->getXBase()->openSelect (qryText)) == 0)
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Error in XBase select query"),
				QString	("%1:\n%2")
					.arg(m_server->getXBase()->lastError())
					.arg(select),
				__ERRLOCN
			   )	;
}

/*  KBXBSQLQrySelect							*/
/*  ~KBXBSQLQrySelect							*/
/*		: Destructor for select query object			*/
/*  (returns)	:		:					*/

KBXBSQLQrySelect::~KBXBSQLQrySelect ()
{
	if (m_xSelect != 0)
		delete m_xSelect ;

//	if (m_types != 0)
//		for (uint idx = 0 ; idx < m_nFields ; idx += 1)
//			m_types[idx]->deref() ;
}

/*  KBXBSQLQrySelect							*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBXBSQLQrySelect::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	if (m_xSelect == 0)
		return false ;

	XBSQLValue* xbvals = convKBXB (values, nvals, m_codec) ;

	/* Substitute the place holders. This is done purely so that	*/
	/* meaningful SQL can appear in the logging window.		*/
	m_subQuery = m_server->subPlaceList (m_rawQuery, nvals, values, m_lError) ;


	if (!m_xSelect->execute (nvals, xbvals))
	{
		m_lError = KBError
			   (	KBError::Error,
			 	"Error executing select query",
				m_server->getXBase()->lastError(),
				__ERRLOCN
			   )	;
		delete [] xbvals;

		m_server->printQuery (m_rawQuery, nvals, values, false) ;
		return	false	;
	}

	m_server->printQuery (m_rawQuery, nvals, values, true) ;

	m_nRows	  = m_xSelect->getNumRows   () ;
	m_nFields = m_xSelect->getNumFields () ;

	if (m_types == 0)
	{
	m_types	  = new KBType *[m_nFields] ;


//	fprintf	(stderr, "KBXBSQLQrySelect::execute: [%s]\n", (cchar *)m_rawQuery) ;

	for (uint idx = 0 ; idx < m_nFields ; idx += 1)
	{
		xbShort	     ftype  = m_xSelect->getFieldType   (idx)	;
		int	     length = m_xSelect->getFieldLength (idx)	;
		XBSQLTypeMap *ptr   = dIdentToType.find (ftype)  ;

//		fprintf
//		(	stderr,
//			"KBXBSQLQrySelect::execute: [%c][%p]\n",
//			ftype,
//			(void *)ptr
//		)	;

		if (ftype == 'M')
			length = 0x00ffffff ;

		m_types[idx] = new KBXBSQLType
			       (	ptr,
					length,
					0,
					true
			       ) ;
	}
	}

	m_lastDump = 0  ;

	delete [] xbvals;
	return	true	;
}


/*  KBXBSQLQrySelect							*/
/*  getField	: Get a specified field from the query results		*/
/*  qrow	: uint		  : Row number				*/
/*  qcol	: uint		  : Column number			*/
/*  vtrans	: KBValue::VTrans : Translation mode			*/
/*  (returns)	: KBValue	  : Value				*/

KBValue	KBXBSQLQrySelect::getField
	(	uint		qrow,
		uint		qcol,
		KBValue::VTrans
	)
{
	/* First check that the request is within the range of rows	*/
	/* and fields returned by the query.				*/
	if ((int)qrow >= m_nRows  ) return KBValue() ;
	if (     qcol >= m_nFields) return KBValue() ;

	const XBSQLValue &value = m_xSelect->getField (qrow, qcol) ;

	/* Binary data gets special treatment as it is does not have a	*/
	/* zero terminator .....					*/
	if (value.tag == XBSQL::VMemo)
		return	KBValue (value.text, value.len, m_types[qcol]) ;

	return	KBValue (value.getText(), m_types[qcol], m_codec) ;
}

/*  KBXBSQLQrySelect							*/
/*  getFieldName: Get a specified field name from the query results	*/
/*  qcol	: uint		: Column number				*/
/*  (returns)	: QString	: name					*/

QString	KBXBSQLQrySelect::getFieldName
	(	uint	qcol
	)
{
	/* First check that the request is within the range of rows	*/
	/* and fields returned by the query.				*/
	if (qcol >= m_nFields) return QString() ;
	return	m_xSelect->getFieldName (qcol)  ;
}

/*  KBXBSQLQrySelect							*/
/*  dumRow	: Dump a specified row from the cache			*/
/*  row		: uint		: Row number				*/
/*  (returns)	: void		:					*/

void	KBXBSQLQrySelect::dumpRow
	(	uint		row
	)
{
//	fprintf
//	(	stderr,
//		"KBXBSQLSelect::dumpRow: [%u]\n",
//		row
//	)	;

	if (m_xSelect != 0) m_xSelect->dumprow (row) ;
}

/*  KBXBSQLQrySelect							*/
/*  dumRow	: Dump rows up to a specified row number		*/
/*  row		: uint		: Row number				*/
/*  (returns)	: void		:					*/

void	KBXBSQLQrySelect::dumpRowsTo
	(	uint		row
	)
{
	if (m_xSelect != 0)
		while (m_lastDump < row)
		{
//			fprintf
//			(	stderr,
//				"KBXBSQLSelect::dumpRowsTo: [%u]->[%u]\n",
//				row,
//				m_lastDump
//			)	;

			m_xSelect->dumprow (m_lastDump) ;
			m_lastDump += 1 ;
		}
}

/*  KBXBSQLQrySelect							*/
/*  dumpAllRows	: Clear the cache					*/
/*  (returns)	: void		:					*/

void	KBXBSQLQrySelect::dumpAllRows ()
{
	dumpRowsTo ((uint)m_nRows) ;
}


/*  ------------------------------------------------------------------  */

/*  KBXBSQLQryUpdate							*/
/*  KBXBSQLQryUpdate							*/
/*		: Constructor for update query object			*/
/*  server	: KBXBSQL *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Update query text			*/
/*  tabName	: const QString & : Table being updated			*/
/*  codec	: QTextCodec *	  : Non-default codec			*/
/*  (returns)	: KBXBSQLQryUpdate:					*/

KBXBSQLQryUpdate::KBXBSQLQryUpdate
	(	KBXBSQL		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLUpdate (server, data, query, tabName),
	m_server    (server)
{
	m_nRows	   = 0 ;
	m_subQuery = m_rawQuery ;

	QCString qryText = query.utf8() ;

	if ((m_xUpdate = m_server->getXBase()->openUpdate (qryText)) == 0)
		m_lError = KBError
			   (	KBError::Error,
			 	"Error in XBase update query",
				m_server->getXBase()->lastError(),
				__ERRLOCN
			   )	;
}

/*  KBXBSQLQryUpdate							*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBXBSQLQryUpdate::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	if (m_xUpdate == 0)
		return	false	;

	XBSQLValue* xbvals = convKBXB (values, nvals, m_codec) ;

	/* Substitute the place holders. This is done purely so that	*/
	/* meaningful SQL can appear in the logging window.		*/
	m_subQuery = m_server->subPlaceList (m_rawQuery, nvals, values, m_lError) ;


	if (!m_xUpdate->execute (nvals, xbvals))
	{
		m_lError = KBError
			   (	KBError::Error,
			 	"Error executing update query",
				m_server->getXBase()->lastError(),
				__ERRLOCN
			   )	;
		delete [] xbvals;

		m_server->printQuery (m_rawQuery, nvals, values, false) ;
		return	false	;
	}

	m_server->printQuery (m_rawQuery, nvals, values, true) ;
	m_nRows	 = m_xUpdate->getNumRows () ;

	delete [] xbvals;
	return	 true	;
}

/*  KBXBSQLQryUpdate							*/
/*  ~KBXBSQLQryUpdate							*/
/*		: Destructor for update query object			*/
/*  (returns)	:		:					*/

KBXBSQLQryUpdate::~KBXBSQLQryUpdate ()
{
	if (m_xUpdate != 0) delete m_xUpdate ;
}


/*  KBXBSQLQryInsert							*/
/*  KBXBSQLQryInsert							*/
/*		: Constructor for insert query object			*/
/*  server	: KBXBSQL *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Insert query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  codec	: QTextCodec *	  : Non-default codec			*/
/*  (returns)	: KBXBSQLQryInsert:					*/

KBXBSQLQryInsert::KBXBSQLQryInsert
	(	KBXBSQL		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLInsert (server, data, query, tabName),
	m_server    (server)
{
	m_nRows	   = 0 ;
	m_subQuery = m_rawQuery ;

	QCString qryText = query.utf8() ;

	if ((m_xInsert = m_server->getXBase()->openInsert (qryText)) == 0)
		m_lError = KBError
			   (	KBError::Error,
			 	"Error in XBase insert query",
				m_server->getXBase()->lastError(),
				__ERRLOCN
			   )	;
}

/*  KBXBSQLQryInsert							*/
/*  ~KBXBSQLQryInsert							*/
/*		: Destructor for insert query object			*/
/*  (returns)	:		:					*/

KBXBSQLQryInsert::~KBXBSQLQryInsert ()
{
	if (m_xInsert != 0) delete m_xInsert ;
}

/*  KBXBSQLQryInsert							*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBXBSQLQryInsert::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	if (m_xInsert == 0)
		return	false	;

	XBSQLValue* xbvals = convKBXB (values, nvals, m_codec) ;

	/* Substitute the place holders. This is done purely so that	*/
	/* meaningful SQL can appear in the logging window.		*/
	m_subQuery = m_server->subPlaceList (m_rawQuery, nvals, values, m_lError) ;


	if (!m_xInsert->execute (nvals, xbvals))
	{
		m_lError = KBError
			   (	KBError::Error,
			 	"Error executing insert query",
				m_server->getXBase()->lastError(),
				__ERRLOCN
			   )	;
		delete [] xbvals;

		m_server->printQuery (m_rawQuery, nvals, values, false) ;
		return	false	;
	}

	m_server->printQuery (m_rawQuery, nvals, values, true) ;
	m_nRows	 = m_xInsert->getNumRows () ;

	delete [] xbvals;
	return	 true	;
}

/*  KBXBSQL								*/
/*  getNewKey	: Get new primary key value				*/
/*  primary	: const QString & : Key column name			*/
/*  _newKey	: KBValue &	  : New key value			*/
/*  prior	: bool		  : Pri-insert call			*/
/*  (returns)	: bool		  : Success				*/

bool	KBXBSQLQryInsert::getNewKey
	(	const QString	&,
		KBValue		&_newKey,
		bool		prior
	)
{
	if (prior)
	{
		_newKey	 = m_server->getNewKey() ;
		return	true ; 
	}

	m_lError = KBError
		   (	KBError::Error,
		  	"Calling getNewKey post-insert",
		  	m_tabName,
		  	__ERRLOCN
		   )	;

	return	false	    ;
}

/*  KBXBSQLQryDelete							*/
/*  KBXBSQLQryDelete							*/
/*		: Constructor for delete query object			*/
/*  server	: KBXBSQL *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Delete query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  codec	: QTextCodec *	  : Non-default codec			*/
/*  (returns)	: KBXBSQLQryDelete:					*/

KBXBSQLQryDelete::KBXBSQLQryDelete
	(	KBXBSQL		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLDelete (server, data, query, tabName),
	m_server    (server)
{
	m_nRows	   = 0 ;
	m_subQuery = m_rawQuery ;

	QCString qryText = query.utf8() ;

	if ((m_xDelete = server->getXBase()->openDelete (qryText)) == 0)
		m_lError = KBError
			   (	KBError::Error,
			 	"Error in XBase delete query",
				m_server->getXBase()->lastError(),
				__ERRLOCN
			   )	;
}

/*  KBXBSQLQryDelete							*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBXBSQLQryDelete::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	if (m_xDelete == 0)
		return	false	;

	XBSQLValue* xbvals = convKBXB (values, nvals, m_codec) ;

	/* Substitute the place holders. This is done purely so that	*/
	/* meaningful SQL can appear in the logging window.		*/
	m_subQuery = m_server->subPlaceList (m_rawQuery, nvals, values, m_lError) ;


	if (!m_xDelete->execute (nvals, xbvals))
	{
		m_lError = KBError
			   (	KBError::Error,
			 	"Error executing update query",
				m_server->getXBase()->lastError(),
				__ERRLOCN
			   )	;
		delete [] xbvals;

		m_server->printQuery (m_rawQuery, nvals, values, false) ;
		return	false	;
	}


	m_server->printQuery (m_rawQuery, nvals, values, true) ;
	m_nRows	 = m_xDelete->getNumRows () ;

	delete [] xbvals;
	return	 true	;
}

/*  KBXBSQLQryDelete							*/
/*  ~KBXBSQLQryDelete							*/
/*		: Destructor for delete query object			*/
/*  (returns)	:		:					*/

KBXBSQLQryDelete::~KBXBSQLQryDelete ()
{
	if (m_xDelete != 0) delete m_xDelete ;
}

#ifndef _WIN32

KBFACTORY
(	KBXBSQLFactory,
	"driver_xbase"
)

KBFACTORYIMPL
(	KBXBSQLFactory,
	driver_xbase,
	"Rekall Xbase/XBSQK driver",
	"Plugin",
	"0",
	"7",
	"0"
)

#else

class	KBXBSQLFactory : public KBFactory
{
public:
	inline	KBXBSQLFactory() : KBFactory ()
	{
	}
	virtual	QObject	*create(QObject * = 0, const char *	= 0, const char * = 0, 
		const QStringList &	= QStringList());
	virtual const char* ident();
};

extern	"C"	__declspec(dllexport) void *init_libkbase_driver_xbase()			
{							
	return	new KBXBSQLFactory;				
}							
#endif

QObject	*KBXBSQLFactory::create
	(	QObject		  *parent,
		cchar		  *object,
		cchar		  *,
		const QStringList &
	)
{
	if (dIdentToType.count() == 0)
		for (uint idx = 0 ; idx < sizeof(typeMap)/sizeof(XBSQLTypeMap) ; idx += 1)
		{
			XBSQLTypeMap *m = &typeMap[idx] ;
			dIdentToType.insert (m->ident, m) ;
		}

	if ((parent != 0) && !parent->inherits ("QWidget"))
	{
		fprintf	(stderr, "KBXBSQLFactory: parent does not inherit QWidget\n") ;
		return	0  ;
	}

	if (strcmp (object, "driver"  ) == 0) return new KBXBSQL () ;

	if (strcmp (object, "advanced") == 0) return new KBXBAdvanced () ;

	return	0 ;
}

cchar	*KBXBSQLFactory::ident ()
{
	return	__KB_BUILD_IDENT	;
}
