/***************************************************************************
    file	         : kb_qryparser.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<qdict.h>

#include	"kb_qryparser.h"


static	QString	tokenChars
	(	"abcdefghijklmnopqrstuvwxyz"
		"ABCDEFGHIJKLMNOPQRSTUVWXYZ"
		"0123456789"
		"_"
		"$"
	)	;

/*  KBQryParser								*/
/*  KBQryParser	: Constructor for simple query parser			*/
/*  select	: KBSelect &	: Select object for results		*/
/*  (returns)	: KBQryParser	:					*/

KBQryParser::KBQryParser
	(	KBSelect	&select
	)
	:
	m_select (select)
{
}

void	KBQryParser::setError
	(	const QString	&error
	)
{
	m_error	= error + ": .... " + m_token + " " + m_buffer.mid(m_buffPtr) ;

	if (m_error.length() > 80)
	{	m_error.truncate (80) ;
		m_error.append   (" ....") ;
	}
}

/*  KBQryParser								*/
/*  nextToken	: Get next token from SQL query string			*/
/*  (returns)	: bool		: Token found				*/

bool	KBQryParser::nextToken ()
{
	bool	inQuote	= false	;
	m_token	= QString::null	;

	/* Loop iterates over the string. The "inQuote" flag is set if	*/
	/* we are in a quoted string.					*/
	while (m_buffPtr < m_buffer.length())
	{
		QChar	ch = m_buffer.at(m_buffPtr) ;

		if (inQuote)
		{
			/* In a quoted string. The next character is	*/
			/* appended to the toke whatever, then end the	*/
			/* token if we hit a further quote.		*/
			m_token.append (ch) ;
			m_buffPtr  += 1     ;

			if (ch == '\'') break ;

			/* Assume that the string escape character is	*/
			/* backquote. If this brings us to the end of	*/
			/* the m_buffer then silently drop out ....	*/
			if (ch == '\\')
			{
				if (m_buffPtr >= m_buffer.length()) break ;

				/* ... otherwise decide how many	*/
				/* characters to gobble. This is a bit	*/
				/* simplistic, but will probably do.	*/
				QChar	ch	= m_buffer.at(m_buffPtr) ;
				uint	gobble	= 1 ;

				if	(ch.isDigit()) gobble = 3 ;
				else if	(ch == 'x'   ) gobble = 3 ;
				else if	(ch == 'X'   ) gobble = 3 ;

				while ((gobble > 0) && (m_buffPtr < m_buffer.length()))
				{	m_token.append (m_buffer.at(m_buffPtr)) ;
					m_buffPtr	+= 1 ;
					gobble	-= 1 ;
				}
			}

			continue ;
		}

		/* Next check for single quote. This terminates the	*/
		/* current token, or starts a new quoted token.		*/
		if (ch == '\'')
		{
			if (!m_token.isEmpty()) break ;
			m_token.append (ch) ;
			m_buffPtr += 1	    ;
			inQuote	   = true   ;
			continue  ;
		}

		/* Check for characters which can be part of a multi-	*/
		/* character token. These are appended to the current	*/
		/* token.						*/
		if (tokenChars.find (ch) >= 0)
		{
			m_token.append (ch) ;
			m_buffPtr += 1	    ;
			continue	    ;
		}

		/* Non-whitespace characters are now treated as tokens	*/
		/* in their own right .....				*/
		if (!ch.isSpace())
		{
			if (m_token.isEmpty())
			{	m_token.append (ch) ;
				m_buffPtr += 1 ;
			}
			break	;
		}

		/* Whitespace terminates the current token if there is	*/
		/* one, otherwise we will just skip over it.		*/
		if (!m_token.isEmpty()) break ;

		m_buffPtr	 += 1	  ;
	}

	/* See if the token is a keyword; if so then convert it to	*/
	/* lower case for convenience.					*/
	if (isKeyword())
		m_token = m_token.lower() ;

	/* Gather up any following whitespace so that we will be able	*/
	/* to reconstruct expressions properly. The only whitespace	*/
	/* that should be lost is at the very start of a query.		*/
	m_white	= "" ;		;
	while ((m_buffPtr < m_buffer.length()) && m_buffer.at(m_buffPtr).isSpace())
	{
		m_white	  += m_buffer.at(m_buffPtr) ;
		m_buffPtr += 1 ;
	}

	return	!m_token.isEmpty () ;
}

/*  KBQryParser								*/
/*  isKeyword	: Check if token is a keyword				*/
/*  (returns)	: bool		  : True if keyword			*/

bool	KBQryParser::isKeyword ()
{
	static	cchar	*keyList[] =
	{
		"select",
		"from",
		"join",
		"left",
		"right",
		"inner",
		"outer",
		"on",
		"where",
		"and",
		"order",
		"by",
		"asc",
		"desc",
		"group",
		"having",
		"and",
		"distinct",
		0
	}	;
	static	QDict<void>	keyDict	;

	if (keyDict.count() == 0)
		for (cchar **keyPtr = &keyList[0] ; *keyPtr != 0 ; keyPtr += 1)
			keyDict.insert (*keyPtr, (void *)1) ;

	return	keyDict.find (m_token.lower()) != 0 ;
}

/*  KBQryParser								*/
/*  parseExpr	: Parse out an expression				*/
/*  orderOK	: bool		: Ordering expression			*/
/*  allowAnd	: bool		: Treat "and" as part of expression	*/
/*  (returns)	: QString	: Expression or null if none		*/

QString	KBQryParser::parseExpr
	(	bool	orderOK,
		bool	allowAnd
	)
{
	QString	expr	;
	int	paren	= 0 ;

	/* Parse until we hit a comma or a keyword (but possibly accept	*/
	/* "asc", "desc" or "and"). Everything up to that point is	*/
	/* accumulated in the expression.				*/
	while (!m_token.isEmpty())
	{
		if (m_token == '(') paren += 1 ;
		if (m_token == ')') paren -= 1 ;

		if (paren == 0)
		{
			if (m_token == ",")
				break ;

			if (isKeyword())
			{
				if	((m_token == "asc") || (m_token == "desc"))
				{
					if (orderOK) nextToken () ;
					break ;
				}
				else if (m_token == "and")
				{
					if (!allowAnd) break ;
				}
				else	break	;
			}
		}

		expr += m_token + m_white ;
		nextToken ()  ;
	}

	return	expr	;
}

/*  KBQryParser								*/
/*  parseExprList: Parse an expression list				*/
/*  eList	 : QStringList & : Return list				*/	
/*  sep		 : cchar *	 : Separator				*/
/*  order	 : bool		 : Ordering expression			*/
/*  (returns)	 : void		 :					*/

void	KBQryParser::parseExprList
	(	void		(KBQryParser::*append)(const QString &),
		cchar		*sep,
		bool		order
	)
{
	for (;;)
	{
		QString expr = parseExpr (order, false) ;
		if (expr.isEmpty()) break   ;

		(this->*append) (expr) ;

		if (m_token != sep) break ;
		nextToken() ;
	}
}

/*  KBQryParser								*/
/*  parseTableList							*/
/*		: Parse list of tables					*/
/*  (returns)	: bool		: Success				*/

bool	KBQryParser::parseTableList ()
{
	bool	gotAny	= false ;

	/* The main loop iterates over the table entries. There should	*/
	/* be table or table/alias pairs, separated by commas or by	*/
	/* join conditions. Actually, this will let some rubbish like	*/
	/* "select ... from inner join ..." through but thats OK.	*/
	while (!m_token.isEmpty()) 
	{
		QString	table	;
		QString	alias	;
		QString	jtype	;
		QString	jexpr	;
		bool	join	= false	;

		/* If the next word is a keyword then it must be the	*/
		/* start of a join condition ...			*/
		if (isKeyword())
		{
			if	((m_token == "left") || (m_token == "right"))
			{
				join	= true    ;
				jtype	= m_token ;

				nextToken () ;
				if (m_token != "outer")
				{	setError (TR("Expected 'outer'")) ;
					return	false ;
				}
				nextToken () ;
				if (m_token != "join" )
				{	setError (TR("Expected 'join'")) ;
					return	false ;
				}

				nextToken () ;
			}
			else if (m_token == "inner")
			{
				join	= true    ;
				jtype	= "inner" ;

				nextToken () ;
				if (m_token != "join")
				{	setError (TR("Expected 'join'")) ;
					return	false ;
				}

				nextToken () ;
			}
			else	break	;
		}

		/* Must now have a table name, possibly followed by a	*/
		/* table alias. We then make sure that we can find a	*/
		/* unique key column on the table.			*/
		table	= m_token	;

		if (nextToken ())
			if ((m_token != ",") && !isKeyword())
			{	alias	= m_token ;
				nextToken ()	  ;
			}
		
		/* If we started with a join then the join condition	*/
		/* should follow the keyword "on"; "and" is allowed as	*/
		/* part of the expression.				*/
		if (join)
		{
			if (m_token != "on")
			{	setError (TR("Expected 'on'")) ;
				return	false ;
			}

			nextToken () ;

			jexpr = parseExpr (false, true) ;
			if (jexpr.isEmpty())
			{	setError (TR("Expected join condition")) ;
				return	false ;
			}
		}

		m_select.appendTable (table, alias, jtype, jexpr) ;
		gotAny	= true ;

		if (m_token == ",") nextToken () ;
	}

	return	gotAny	;
}

void	KBQryParser::addSelectExpr
	(	const QString	&expr
	)
{
	m_select.appendExpr (expr) ;
}

void	KBQryParser::addSelectWhere
	(	const QString	&expr
	)
{
	m_select.appendWhere (expr) ;
}

void	KBQryParser::addSelectGroup
	(	const QString	&expr
	)
{
	m_select.appendGroup (expr) ;
}

void	KBQryParser::addSelectHaving
	(	const QString	&expr
	)
{
	m_select.appendHaving (expr) ;
}

void	KBQryParser::addSelectOrder
	(	const QString	&expr
	)
{
	m_select.appendOrder (expr) ;
}

/*  KBQryParser								*/
/*  parseQuery	: Main parse routine					*/
/*  query	: const QString & : Query to parse			*/
/*  (returns)	: bool		  : Success				*/

bool	KBQryParser::parseQuery
	(	const QString	&query
	)
{
	m_select.reset () ;
	m_buffer  = query ;
	m_buffPtr = 0 	  ;

	/* Prime the parser by getting the first token, and verify that	*/
	/* it is the keyword "select" ...				*/
	m_buffPtr = 0 ;
	if (!nextToken ())
	{	setError (TR("Query is empty")) ;
		return	false ;
	}

	if (m_token.lower() != "select")
	{	setError (TR("Query does not start with 'select'")) ;
		return	false ;
	}

	/* ... whicn should be followed by a comma-separated list of	*/
	/* expressions and thence the keyword "from".			*/
	nextToken     () ;
	if (m_token.lower() == "distinct")
	{	m_select.setDistinct (true) ;
		nextToken () ;
	}
	parseExprList (&KBQryParser::addSelectExpr, ",", false) ;

	if (m_token.lower() != "from")
	{	setError (TR("Expected 'from'")) ;
		return	false ;
	}

	/* The list of tables should follow. This may either be comma	*/
	/* separated, a series of joins, or a combination.		*/
	nextToken () ;
	if (!parseTableList ())
		return	false ;

	if (m_token.lower() == "where")
	{
		nextToken () ;
		parseExprList (&KBQryParser::addSelectWhere, "and", false) ;
	}

	if (m_token.lower() == "group")
	{
		nextToken () ;
		if (m_token.lower() != "by")
		{	setError (TR("Expected 'by'")) ;
			return	false ;
		}

		nextToken () ;
		parseExprList (&KBQryParser::addSelectGroup, ",", false) ;
	}

	if (m_token.lower() == "having")
	{
		nextToken () ;
		parseExprList (&KBQryParser::addSelectHaving, ",", false) ;
	}

	if (m_token.lower() == "order")
	{
		nextToken () ;
		if (m_token.lower() != "by")
		{	setError (TR("Expected 'by'")) ;
			return	false ;
		}

		nextToken () ;
		parseExprList (&KBQryParser::addSelectOrder, ",", true) ;
	}

	return	true	;
}

