/***************************************************************************
    file	         : kb_dblink.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	"kb_dblink.h"
#include	"kb_dbinfo.h"


/*  KBDBLink								*/
/*  KBDBLink	: Constructor for server link class			*/
/*  (returns)	: KBDBLink	:					*/

KBDBLink::KBDBLink ()
{
	m_svInfo	= 0 	;
	m_disabled	= false	;
	m_nOp		= 0	;
	m_opMap		= 0	;
}

/*  KBDBLink								*/
/*  ~KBDBLink	: Destructor for server link class			*/
/*  (returns)	:		:					*/

KBDBLink::~KBDBLink ()
{
	if (m_svInfo != 0) m_svInfo->detachLink (this) ;
}

/*  KBDBLink								*/
/*  checkLinked	: Check linked to server				*/
/*  errLno	: int		: Error line number if not connected	*/
/*  (returns)	: bool		: Connected				*/

bool	KBDBLink::checkLinked
	(	int	errLno
	)
{
	if (m_svInfo == 0)
	{
		m_lError  = KBError
			  (	KBError::Fault,
				TR("Not linked to a server"),
				QString::null,
				__FILE__,
				errLno
			  )	;
		return	false	;
	}

	return	true	;
}

/*  KBDBLink								*/
/*  connect	: Connect to a server					*/
/*  dbInfo	: KBDBInfo *	: Database inforamtion			*/
/*  svName	: const QString&: Server name				*/
/*  verify	: bool		: Verify that a connection opens	*/
/*  (returns)	: bool		: Success				*/

bool	KBDBLink::connect
	(	KBDBInfo		*dbInfo,
		const QString		&svName,
		bool			verify
	)
{
	if (m_svInfo != 0)
	{
		m_lError  = KBError
			  (	KBError::Warning,
				QString(TR("Already connected to server \"%1\""))
					.arg(m_svInfo->serverName()),
				QString::null,
				__ERRLOCN
			  ) 	;

		return	false 	;
	}

	if ((m_svInfo = dbInfo->findServer (svName)) == 0)
	{
		m_lError  = KBError
			  (	KBError::Error,
				QString(TR("Server \"%1\" not known")).arg(svName),
				QString::null,
				__ERRLOCN
			  ) 	;
		return	false	;
	}

	if (m_svInfo->disabled())
	{
		m_disabled= true ;
		m_lError  = KBError
			  (	KBError::Error,
				QString(TR("Server \"%1\" is disabled")).arg(svName),
				QString::null,
				__ERRLOCN
			  ) 	;
		return	false	;
	}

	m_disabled = false ;
	m_svInfo->attachLink (this) ;

	if (verify)
		if (m_svInfo->getServer (m_lError) == 0)
			return	false ;

	return	true ;
}

/*  KBDBLink								*/
/*  connect	: Connect to a server					*/
/*  location	: KBLocation &	: Database information			*/
/*  svName	: const QString&: Server name				*/
/*  verify	: bool		: Verify that a connection opens	*/
/*  (returns)	: bool		: Success				*/

bool	KBDBLink::connect
	(	const KBLocation	&location,
		const QString		&svName,
		bool			verify
	)
{
	if (svName == "Self")
		return	connect (location.dbInfo, location.docLocn, verify) ;

	return	connect (location.dbInfo, svName, verify) ;
}

/*  KBDBLink								*/
/*  disconnect	: Disconnect from server				*/
/*  (returns)	: void		:					*/

void	KBDBLink::disconnect ()
{
	if (m_svInfo != 0)
	{
		m_svInfo->detachLink (this) ;
		m_svInfo	= 0 ;
		m_disabled	= 0 ;
	}
}

/*  KBDBLink								*/
/*  qrySelect	: Instantiate a select query				*/
/*  data	: bool		  : Data query				*/
/*  query	: const QString & : Query text				*/
/*  update	: bool		  : Query is for update			*/
/*  (returns)	: KBSQLSelect *	  : Select query or null on error	*/

KBSQLSelect
	*KBDBLink::qrySelect
	(	bool		data,
		const QString	&query,
		bool		update
	)
{
	KBServer    *s ;
	KBSQLSelect *q ;

	if (!checkLinked (__LINE__)) return 0 ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return 0 ;

	if ((q = s->qrySelect (data, query, update)) == 0)
	{	m_lError = s->lastError () ;
		return	 0 ;
	}

	return	q ;
}

/*  KBDBLink								*/
/*  qryUpdate	: Instantiate an update query				*/
/*  data	: bool		  : Data query				*/
/*  query	: const QString & : Query text				*/
/*  tabName	: const QString & : Table name				*/
/*  (returns)	: KBSQLUpdate *	  : Update query or null on error	*/

KBSQLUpdate
	*KBDBLink::qryUpdate
	(	bool		data,
		const QString	&query,
		const QString	&tabName
	)
{
	KBServer    *s ;
	KBSQLUpdate *u ;

	if (!checkLinked (__LINE__)) return 0 ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return 0 ;

	if ((u = s->qryUpdate (data, query, tabName)) == 0)
	{	m_lError = s->lastError () ;
		return	 0 ;
	}

	return	u ;
}

/*  KBDBLink								*/
/*  qryInsert	: Instantiate an insert query				*/
/*  data	: bool		  : Data query				*/
/*  query	: const QString & : Query text				*/
/*  tabName	: const QString & : Table name				*/
/*  (returns)	: KBSQLInsert *	  : Insert query or null on error	*/

KBSQLInsert
	*KBDBLink::qryInsert
	(	bool		data,
		const QString	&query,
		const QString	&tabName
	)
{
	KBServer    *s ;
	KBSQLInsert *i ;

	if (!checkLinked (__LINE__)) return 0 ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return 0 ;

	if ((i = s->qryInsert (data, query, tabName)) == 0)
	{	m_lError = s->lastError () ;
		return	 0 ;
	}

	return	i ;
}

/*  KBDBLink								*/
/*  qryDelete	: Instantiate a delete query				*/
/*  data	: bool		  : Data query				*/
/*  query	: const QString & : Query text				*/
/*  tabName	: const QString & : Table name				*/
/*  (returns)	: KBSQLDelete *	  : Delete query or null on error	*/

KBSQLDelete
	*KBDBLink::qryDelete
	(	bool		data,
		const QString	&query,
		const QString	&tabName
	)
{
	KBServer    *s ;
	KBSQLDelete *d ;

	if (!checkLinked (__LINE__)) return 0 ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return 0 ;

	if ((d = s->qryDelete (data, query, tabName)) == 0)
	{	m_lError = s->lastError () ;
		return	 0 ;
	}

	return	d ;
}

/*  KBDBLink								*/
/*  qryCursor	: Instantiate a cursor					*/
/*  data	: bool		  : Data query				*/
/*  query	: const QString & : Query text				*/
/*  cursor	: const QString & : Cursor name				*/
/*  (returns)	: KBSQLCursor *	  : Cursor or null on error		*/

KBSQLCursor
	*KBDBLink::qryCursor
	(	bool		data,
		const QString	&query,
		const QString	&cursor
	)
{
	KBServer    *s ;
	KBSQLCursor *c ;

	if (!checkLinked (__LINE__)) return 0 ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return 0 ;

	if ((c = s->qryCursor (data, query, cursor)) == 0)
	{	m_lError = s->lastError () ;
		return	 0 ;
	}

	return	c ;
}

/*  KBDBLink								*/
/*  command	: Execute arbitrary SQL					*/
/*  data	: bool		  : Data query				*/
/*  query	: const QString & : SQL text				*/
/*  nvals	: uint		  : Number of substitution values	*/
/*  values	: KBValue *	  : Vector of substitution values	*/
/*  select	: KBSQLSelect **  : Return for data queries		*/
/*  (returns)	: bool		  : Success				*/

bool	KBDBLink::command
	(	bool		data,
		const QString	&query,
		uint		nvals,
		KBValue		*values,
		KBSQLSelect	**select
	)
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return false ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return false ;

	if (!s->command (data, query, nvals, values, select))
	{	m_lError = s->lastError () ;
		return	 false	;
	}

	return	true	;
}

/*  KBDBLink								*/
/*  transaction	: Execute transaction control				*/
/*  op		: KBServer::Transaction	: Operation			*/
/*  activeCookie: void *&		: Pass/return active cookie	*/
/*  (returns)	: bool			: Success			*/

bool	KBDBLink::transaction
	(	KBServer::Transaction	op,
		void			**activeCookie
	)
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return false ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return false ;

	if (!s->transaction (op, activeCookie))
	{	m_lError = s->lastError () ;
		return	 false	;
	}

	return	true	;
}

/*  KBDBLink								*/
/*  tableExists	: See if a table exists					*/
/*  table	: const QString &: Table name				*/
/*  exists	: bool &	 : Actual result			*/
/*  (returns)	: bool		 : Success				*/

bool	KBDBLink::tableExists
	(	const QString	&table,
		bool		&exists
	)
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return false ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return false ;

	if (!s->tableExists (table, exists))
	{	m_lError = s->lastError () ;
		return	 false ;
	}

	return	true	;
}

/*  KBDBLink								*/
/*  viewExists	: See if a view exists					*/
/*  view	: const QString &: View name				*/
/*  exists	: bool &	 : Actual result			*/
/*  (returns)	: bool		 : Success				*/

bool	KBDBLink::viewExists
	(	const QString	&view,
		bool		&exists
	)
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return false ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return false ;

	if (!s->viewExists (view, exists))
	{	m_lError = s->lastError () ;
		return	 false ;
	}

	return	true	;
}

/*  KBDBLink								*/
/*  sequenceExists							*/
/*		: See if a sequence exists				*/
/*  sequence	: const QString &: Sequence name				*/
/*  exists	: bool &	 : Actual result			*/
/*  (returns)	: bool		 : Success				*/

bool	KBDBLink::sequenceExists
	(	const QString	&sequence,
		bool		&exists
	)
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return false ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return false ;

	if (!s->sequenceExists (sequence, exists))
	{	m_lError = s->lastError () ;
		return	 false ;
	}

	return	true	;
}

/*  KBDBLink								*/
/*  listTables	: Get a list of all tables (views, ...)			*/
/*  list	: KBTableDetailsList &					*/
/*				: Result list				*/
/*  type	: uint		: Type flags				*/
/*  (returns)	: bool		: Success				*/

bool	KBDBLink::listTables
	(	KBTableDetailsList	&list,
		uint			type
	)
{
	KBServer    *s ;

//	fprintf
//	(	stderr,
//		"KBDBLink::listTables: type=%02x\n",
//		type
//	)	;

	if (!checkLinked (__LINE__)) return false ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return false ;

	if (!s->listTables (list, type))
	{	m_lError = s->lastError () ;
		return	 false ;
	}

	return	true	;
}

/*  KBDBLink								*/
/*  listFields	: Get a list of fields in a table			*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  (returns)	: bool		: Success				*/

bool	KBDBLink::listFields
	(	KBTableSpec	&tabSpec
	)
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return false ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return false ;

	if (!s->listFields (tabSpec))
	{	m_lError = s->lastError () ;
		return	 false ;
	}

	return	true	;
}

/*  KBDBLink								*/
/*  flushTableCache							*/
/*		: Flush all cached table information			*/
/*  (returns)	: void		:					*/

void	KBDBLink::flushTableCache ()
{
	KBServer    *s ;
	if (!checkLinked (__LINE__)) return ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return ;
	s->flushTableCache () ;
}

/*  KBDBLink								*/
/*  createTable	: Create a new table					*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  assoc	: bool		: Create associated stuff		*/
/*  (returns)	: bool		: Success				*/

bool	KBDBLink::createTable
	(	KBTableSpec	&tabSpec,
		bool		assoc
	)
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return false ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return false ;

	if (!s->createTable (tabSpec, assoc))
	{	m_lError = s->lastError () ;
		return	 false ;
	}

	return	true	;
}

/*  KBDBLink								*/
/*  renameTable	: Rename a table					*/
/*  oldName	: const QString&: Current table name			*/
/*  newName	: const QString&: New table name			*/
/*  assoc	: bool		: Create associated stuff		*/
/*  (returns)	: bool		: Success				*/

bool	KBDBLink::renameTable
	(	const QString	&oldName,
		const QString	&newName,
		bool		assoc
	)
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return false ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return false ;

	if (!s->renameTable (oldName, newName, assoc))
	{	m_lError = s->lastError () ;
		return	 false ;
	}

	return	true	;
}

/*  KBDBLink								*/
/*  dropTable	: Drop (ie., delete) a table				*/
/*  table	: const QString&: Table name				*/
/*  assoc	: bool		: Create associated stuff		*/
/*  (returns)	: bool		: Success				*/

bool	KBDBLink::dropTable 
	(	const QString	&table,
		bool		assoc
	)
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return false ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return false ;

	if (!s->dropTable (table, assoc))
	{	m_lError = s->lastError () ;
		return	 false ;
	}

	return	true	;
}

/*  KBDBLink								*/
/*  createView	: Create a new view					*/
/*  tabSpec	: KBViewSpec &	: View specification			*/
/*  assoc	: bool		: Create associated stuff		*/
/*  (returns)	: bool		: Success				*/

bool	KBDBLink::createView
	(	KBTableSpec	&tabSpec
	)
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return false ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return false ;

	if (!s->createView (tabSpec))
	{	m_lError = s->lastError () ;
		return	 false ;
	}

	return	true	;
}

/*  KBDBLink								*/
/*  renameView	: Rename a view						*/
/*  oldName	: const QString&: Current view name			*/
/*  newName	: const QString&: New view name				*/
/*  (returns)	: bool		: Success				*/

bool	KBDBLink::renameView
	(	const QString	&oldName,
		const QString	&newName
	)
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return false ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return false ;

	if (!s->renameView (oldName, newName))
	{	m_lError = s->lastError () ;
		return	 false ;
	}

	return	true	;
}

/*  KBDBLink								*/
/*  dropView	: Drop (ie., delete) a view				*/
/*  view	: const QString&: View name				*/
/*  (returns)	: bool		: Success				*/

bool	KBDBLink::dropView 
	(	const QString	&view
	)
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return false ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return false ;

	if (!s->dropView (view))
	{	m_lError = s->lastError () ;
		return	 false ;
	}

	return	true	;
}

/*  KBDBLink								*/
/*  descSequence: Describe a sequence					*/
/*  seqSpec	: KBSequenceSpec & : Sequence specification		*/
/*  (returns)	: bool		   : Success				*/

bool	KBDBLink::descSequence
	(	KBSequenceSpec	&seqSpec
	)
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return false ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return false ;

	if (!s->descSequence (seqSpec))
	{	m_lError = s->lastError () ;
		return	 false ;
	}

	return	true	;
}

/*  KBDBLink								*/
/*  createSequence							*/
/*		: Create a new sequence					*/
/*  seqSpec	: KBSequenceSpec & : Sequence specification		*/
/*  (returns)	: bool		   : Success				*/

bool	KBDBLink::createSequence
	(	KBSequenceSpec	&seqSpec
	)
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return false ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return false ;

	if (!s->createSequence (seqSpec))
	{	m_lError = s->lastError () ;
		return	 false ;
	}

	return	true	;
}

/*  KBDBLink								*/
/*  renameSequence							*/
/*		: Rename a sequence					*/
/*  oldName	: const QString&: Current sequence name			*/
/*  newName	: const QString&: New sequence name			*/
/*  (returns)	: bool		: Success				*/

bool	KBDBLink::renameSequence
	(	const QString	&oldName,
		const QString	&newName
	)
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return false ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return false ;

	if (!s->renameSequence (oldName, newName))
	{	m_lError = s->lastError () ;
		return	 false ;
	}

	return	true	;
}

/*  KBDBLink								*/
/*  dropSequence: Drop (ie., delete) a sequence				*/
/*  sequence	: const QString & : Sequence name			*/
/*  (returns)	: bool		  : Success				*/

bool	KBDBLink::dropSequence 
	(	const QString	&sequence
	)
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return false ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return false ;

	if (!s->dropSequence (sequence))
	{	m_lError = s->lastError () ;
		return	 false ;
	}

	return	true	;
}

/*  KBDBLink	:							*/
/*  listTypes	: Get list of server data types				*/
/*  (returns)	: QString	: Bar-separated types string		*/

QString	KBDBLink::listTypes  ()
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return "" ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return "" ;

	return	s->listTypes () ;
}

/*  KBDBLink	:							*/
/*  placeHolder	: Get placeholder text					*/
/*  slot	: uint		: Placeholder number			*/
/*  (returns)	: QString	: Placeholder text			*/

QString	KBDBLink::placeHolder
	(	uint	slot
	) 
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return "?" ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return "?" ;

	return	s->placeHolder (slot) ;
}

/*  KBDBLink	:							*/
/*  keepsCase	: Check if server preserves case			*/
/*  (returns)	: bool		: True if so				*/

bool	KBDBLink::keepsCase ()
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return true ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return true ;

	return	s->keepsCase () ;
}

/*  KBDBLink	:							*/
/*  fixCase	: Fix to lower case if not case preserving		*/
/*  text	: const QString & : Text to process			*/
/*  (returns)	: QString	  : Possibly modified text		*/

QString	KBDBLink::fixCase
	(	const QString	&text
	)
{
	KBServer    *s ;

	if (!checkLinked (__LINE__)) return text ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return text ;

	return	s->keepsCase () ? text : text.lower() ;
}

/*  KBDBLink	:							*/
/*  rekallPrefix: Modify name with RDBMS-specific prefix		*/
/*  name	: const QString & : Name to modify			*/
/*  (returns)	: QString	  : Modified name			*/

QString	KBDBLink::rekallPrefix
	(	const QString	&name
	)
{
	KBServer    *s ;

	/* Bad news if called with no connection, but we need to	*/
	/* return something.						*/
	if (!checkLinked (__LINE__)) return name ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return name ;

	return	s->rekallPrefix (name) ;
}

/*  KBDBLink								*/
/*  mapExpression: Map SQL expression for driver			*/
/*  expr	 : const QString & : Expression to map			*/
/*  (returns)	 : QString	   : Mapped expression			*/

QString	KBDBLink::mapExpression
	(	const QString	&expr
	)
{
	KBServer    *s ;

	/* Bad news if called with no connection, but we need to	*/
	/* return something.						*/
	if (!checkLinked (__LINE__)) return expr ;
	if ((s = m_svInfo->getServer (m_lError)) == 0) return expr ;

	return	s->mapExpression (expr) ;
}

/*  KBDBLink								*/
/*  mapOperatorDefault							*/
/*		: Default map operator for use in SQL expression	*/
/*  op		: KBServer::Operator	: Operator			*/
/*  def		: ccahr *		: Final fallback		*/
/*  (returns)	: cchar *		: Operator as string		*/

static	cchar	*mapOperatorDefault
	(	KBServer::Operator	op,
		cchar			*def
	)
{
	/* Pick up the default operator map and use that. Hopefully we	*/
	/* should never get called but it means we can still so		*/
	/* something sensible in the absense of a connection.		*/
	cchar	**opMap	;
	uint	nOp	= KBServer::defOperatorMap (opMap) ;

	return	(uint)op >= nOp ? def : opMap[(uint)op] ;
}

/*  KBDBLink								*/
/*  mapOperator	: Map operator for use in SQL expression		*/
/*  op		: KBServer::Operator	: Operator			*/
/*  def		: ccahr *		: Final fallback		*/
/*  (returns)	: cchar *		: Operator as string		*/

cchar	*KBDBLink::mapOperator
	(	KBServer::Operator	op,
		cchar			*def
	)
{
	/* We keep a copy of the driver's map the first time we are	*/
	/* called with a connection, in the interest of efficiency. If	*/
	/* there is no connection then fall back on the defaults.	*/
	if (m_opMap == 0)
	{
		KBServer    *s ;

		if (!checkLinked (__LINE__))
			return mapOperatorDefault (op, def) ;

		if ((s = m_svInfo->getServer (m_lError)) == 0)
			return mapOperatorDefault (op, def) ;

		m_nOp	= s->operatorMap (m_opMap) ;
	}

	return	(uint)op >= m_nOp ? def : m_opMap[(uint)op] ;
}

/*  KBDBLink	:							*/
/*  server	: Get underlying server object				*/
/*  (returns)	: KBServer *	: Server or null			*/

KBServer *KBDBLink::server ()
{
	/* This imethod is used where the client needs direct access	*/
	/* to the server object, for instance to access the getSyntax()	*/
	/* method, which uses varargs.					*/
	if (!checkLinked (__LINE__))  return 0 ;
	return	m_svInfo->getServer (m_lError) ;
}

/*  KBDBLink	 :							*/
/*  hasDesignDict: Check if server/database has a design dictionary	*/
/*  (returns)	 : bool		: True if present			*/

bool	KBDBLink::hasDesignDict ()
{
	return	m_svInfo == 0 ? false : m_svInfo->hasDesignDict() ;
}

/*  KBDBLink	:							*/
/*  copyLink	: Copy an existing link					*/
/*  oldLink	: KBDBLink *	: Extant link				*/
/*  verify	: bool		: Verify that a connection opens	*/
/*  (returns)	: bool		: Success				*/

bool	KBDBLink::copyLink
	(	KBDBLink	&oldLink,
		bool		verify
	)
{
	if ((m_svInfo = oldLink.m_svInfo) == 0)
		return	true	;

	m_svInfo->attachLink (this) ;

	if (verify)
		if (m_svInfo->getServer (m_lError) == 0)
			return	false ;

	return	true	;
}

/*  KBDBLink								*/
/*  getDBType	: Get database type					*/
/*  (returns)	: QString	: Database type				*/

QString	KBDBLink::getDBType ()
{
	return	m_svInfo == 0 ? QString::null : m_svInfo->dbType() ;
}

/*  KBDBLink	:							*/
/*  hasObjectTable							*/
/*		: Check if server/database has an object table		*/
/*  (returns)	: bool		: True if present			*/

bool	KBDBLink::hasObjectTable ()
{
	return	m_svInfo == 0 ? false : m_svInfo->hasObjectTable() ;
}


QString	KBDBLink::objectTableName ()
{
	return	rekallPrefix(OBJECTTABNAME) ;
}

QString	KBDBLink::designTableName  ()
{
	return	rekallPrefix(DESIGNTABNAME) ;
}

uint	KBDBLink::getColumnWidth
	(	const QString	&tabName,
		const QString	&colName
	)
{
	if ((m_svInfo == 0) || (m_svInfo->tableInfoSet() == 0))
		return 0 ;

	return	m_svInfo->tableInfoSet()->designValue
		(	tabName,
			colName,
			TI_WIDTH
		)
		.toInt() ;
}

void	KBDBLink::setColumnWidth
	(	const QString	&tabName,
		const QString	&colName,
		uint		width
	)
{
	if ((m_svInfo != 0) && (m_svInfo->tableInfoSet() != 0))
		m_svInfo->tableInfoSet()->setDesignValue
		(	tabName,
			colName,
			TI_WIDTH,
			QString("%1").arg(width)
		)	;
}
