/***************************************************************************
    file	         : kb_type.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>

#include	<qstring.h>
#include	<qdict.h>
#include	<qintdict.h>

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_databuffer.h"


LIBCOMMON_API KBStaticType _kbUnknown	(KB::ITUnknown, 	0, 0, true)	;
LIBCOMMON_API KBStaticType _kbRaw	(KB::ITRaw, 	  	0, 0, true)	;
LIBCOMMON_API KBStaticType _kbFixed 	(KB::ITFixed,   	0, 0, true)	;
LIBCOMMON_API KBStaticType _kbFloat	(KB::ITFloat,		0, 0, true)	;
LIBCOMMON_API KBStaticType _kbDate	(KB::ITDate,    	0, 0, true)	;
LIBCOMMON_API KBStaticType _kbTime	(KB::ITTime,    	0, 0, true)	;
LIBCOMMON_API KBStaticType _kbDateTime	(KB::ITDateTime,	0, 0, true)	;
LIBCOMMON_API KBStaticType _kbString	(KB::ITString,  	0, 0, true)	;
LIBCOMMON_API KBStaticType _kbBinary	(KB::ITBinary,  	0, 0, true)	;
LIBCOMMON_API KBStaticType _kbBool	(KB::ITBool,	  	0, 0, true)	;
LIBCOMMON_API KBStaticType _kbDriver	(KB::ITDriver,		0, 0, true)	;


static	int	numTypeObjects	;

/*  KBType								*/
/*  KBType	: Constructor for database value type base class	*/
/*  id		: cchar *	: Driver ID				*/
/*  iType	: Internal	: Internal type				*/
/*  length	: uint		: Underlying database length		*/
/*  proc	: uint		: Precision where applicable		*/
/*  nullOK	: bool		: True if value may be null		*/
/*  (returns)	: KBType	:					*/
	
KBType::KBType
	(	cchar		*id,
		KB::IType	iType,
		uint		length,
		uint		prec,
		bool		nullOK
	)
	:
	m_id		(id),
	m_iType		(iType),
	m_length	(length),
	m_prec		(prec),
	m_nullOK	(nullOK)
{
	numTypeObjects += 1 ;
//	fprintf
//	(	stderr,
//		"KBType::KBType: normal : %d %u\n",
//		iType,
//		numTypeObjects
//	)	;
}

/*  KBType								*/
/*  KBType	: Constructor for database value type base class	*/
/*  id		: cchar *	: Driver ID				*/
/*  iType	: Internal	: Internal type				*/
/*  length	: uint		: Underlying database length		*/
/*  proc	: uint		: Precision where applicable		*/
/*  nullOK	: bool		: True if value may be null		*/
/*  locked	: bool		: True if shared object is locked	*/
/*  (returns)	: KBType	:					*/
	
KBType::KBType
	(	cchar		*id,
		KB::IType	iType,
		uint		length,
		uint		prec,
		bool		nullOK,
		bool		locked
	)
	:
	KBShared	(locked),
	m_id		(id),
	m_iType		(iType ),
	m_length	(length),
	m_prec		(prec  ),
	m_nullOK	(nullOK)
{
	numTypeObjects += 1 ;
//	fprintf
//	(	stderr,
//		"KBType::KBType: locked : %d %u %d\n",
//		iType,
//		numTypeObjects,
//		locked
//	)	;
}

/*  KBType								*/
/*  KBType	: Destructor for database value type base class		*/
/*  (returns)	:		:					*/

KBType::~KBType ()
{
	numTypeObjects -= 1 ;
//	fprintf
//	(	stderr,
//		"KBType::KBType: delete : %d %u\n",
//		iType,
//		numTypeObjects
//	)	;
}

/*  KBType								*/
/*  isValid	: Test if value is valid for type			*/
/*  value	: const QString & : Raw Value to check			*/
/*  pError	: KBError &	  : Error return			*/
/*  what	: const QString & : What is being checked		*/
/*  (returns)	: bool		  : True if valid			*/

bool	KBType::isValid
	(	const QString	&value,
		KBError		&pError,
		const QString	&what
	)
{
	bool	ok	;

	/* Check whether the field can be null, and if not and it is	*/
	/* empty, then return an error.					*/
	if (value.isNull())
		if (!m_nullOK)
		{
			pError	= KBError
				  (	KBError::Error,
					TR("Value may not be empty"),
					what,
					__ERRLOCN
				  )	;
			return	false	;
		}
		else	return	true	;


	switch (m_iType)
	{
		case KB::ITFixed	:
			/* Integer of some sort. Check that the text	*/
			/* decodes to an integer with nothing left over	*/
			value.toInt (&ok) ;

			if (!ok)
			{
				pError	= KBError
					  (	KBError::Error,
						TR("Value is not a valid number"),
						QString("%1: %2").arg(what).arg(value),
						__ERRLOCN
					  )	;
				return	false	;
			}

			return	true	;

		case KB::ITFloat	:
			/* Floating point of some sort. This is much	*/
			/* the same as for an integer.			*/
			value.toDouble (&ok) ;

			if (!ok)
			{
				pError	= KBError
					  (	KBError::Error,
						TR("Value is not a valid double"),
						QString("%1: %2").arg(what).arg(value),
						__ERRLOCN
					  )	;
				return	false	;
			}

			return	true	;


		case KB::ITString	:
		case KB::ITBinary	:
			return	true	;

		case KB::ITBool		:
			/* We will be quite flexible about this. We	*/
			/* will recognise words like "true" and "no",	*/
			/* otherwise numbers will be true if non-zero	*/
			/* and strings true if non-empty.		*/
			return	true	;

		case KB::ITDate		:
		case KB::ITTime		:
		case KB::ITDateTime	:
			/* See if the value can be decoded into a date	*/
			/* time structure, then check on the bits if	*/
			/* appropriate.					*/
			{
				KBDateTime dt (value)	;
				QString	   em		;

				if	(!dt.isValid())
					em	= TR("Value is not a date/time") ;
				else if ((m_iType == KB::ITDate) && dt.hasTime())
					em 	= TR("Date has unexpected time part") ;
				else if ((m_iType == KB::ITTime) && dt.hasDate())
					em 	= TR("Time has unexpected date part") ;

				if (!em.isNull())
				{
					pError	= KBError
						  (	KBError::Fault,
							em,
							what,
							__ERRLOCN
						  )	;
					return	false	;
				}
			}
			return	true	;

		case KB::ITDriver :
			/* Assume valid. Generally the drivers should	*/
			/* override this.				*/
			return	true	;


		default	:
			/* Anything use is an error ...			*/
			break	;
	}

	pError	= KBError
		  (	KBError::Fault,
			QString (TR("Unknown internal type")),
			QString	(TR("Got type %1 for %2")).arg((int)m_iType).arg(what),
			__ERRLOCN
		  )	;
	return	false	;
}

/*  KBType								*/
/*  escapeText	: Escape text characters				*/
/*  value	: KBDataArray  * : Raw text to escape			*/
/*  buffer	: KBDataBuffer & : Escaped text				*/
/*  (returns)	: void		 :					*/

void	KBType::escapeText
	(	KBDataArray	*value,
		KBDataBuffer	&buffer
	)
{
	char	   ch	;

	for (uint idx = 0 ; idx < value->m_length ; idx += 1)
		switch (ch = value->m_data[idx])
		{
			case '\'' :
			case '\\' :
				buffer.append('\\') ;
				buffer.append(ch  ) ;
				break	;

			default	:
				buffer.append(ch  ) ;
				break	;
		}
}

/*  KBType								*/
/*  escapeText	: Escape text characters				*/
/*  value	: QCString     & : Raw text to escape			*/
/*  buffer	: KBDataBuffer & : Escaped text				*/
/*  (returns)	: void		 :					*/

void	KBType::escapeText
	(	QCString	&value,
		KBDataBuffer	&buffer
	)
{
	char	   ch	;

	for (uint idx = 0 ; idx < value.length() ; idx += 1)
		switch (ch = value[idx])
		{
			case '\'' :
			case '\\' :
				buffer.append('\\') ;
				buffer.append(ch  ) ;
				break	;

			default	:
				buffer.append(ch  ) ;
				break	;
		}
}

/*  KBType								*/
/*  getQueryText: Get text for insertion into a query			*/
/*  value	: KBDataArray *  : Raw text to escape			*/
/*  d		: KBShared *	 : Decoded representation		*/
/*  buffer	: KBDataBuffer & : Escaped text				*/
/*  codec	: QTextCodec *	 : Non-default codec			*/
/*  (returns)	: void		 :					*/

void	KBType::getQueryText
	(	KBDataArray	*value,
		KBShared	*d,
		KBDataBuffer	&buffer,
		QTextCodec	*codec
	)
{
	if (value == 0)
	{	buffer.append	("null") ;
		return	;
	}

//	fprintf
//	(	stderr,
//		"KBType::getQueryText: [%8s][%s]\n",
//		(cchar *)getDescrip(),
//		value->m_data
//	)	;

	switch (m_iType)
	{
		case KB::ITString	:
			/* String. The value will be enclosed in a	*/
			/* pair of quotes.				*/
			break	;

		case KB::ITFixed	:
		case KB::ITFloat	:
			/* Also easy. The text of the integer or double	*/
			/* can be used as-is.				*/
			buffer.append (value->m_data, value->m_length) ;
			return	;

		case KB::ITDate		:
		case KB::ITTime		:
		case KB::ITDateTime	:
			/* Convert the date and time to the default	*/
			/* format and return it enclosed in quotes.	*/
			/* Empty strings are treated as nulls.		*/
			{
			QCString date = ((KBDateTime *)d)->defFormat(m_iType).utf8() ;
			buffer.append ('\''  ) ;
			buffer.append (date.data(), date.length()) ;
			buffer.append ('\''  ) ;
			}
			return	;


		case KB::ITBinary	:
			/* Although this is a valid type, it should	*/
			/* always be handled in a driver-specific	*/
			/* derived class. Grumble and use the raw text.	*/
			KBError::EError
			(	"KBType::getQueryText: Unexpected binary type",
				QString::null,
				__ERRLOCN
			)	;
			break	;

		case KB::ITDriver	:
			/* Similarly for driver-specific types ...	*/
			KBError::EError
			(	"KBType::getQueryText: Unexpected driver type",
				QString::null,
				__ERRLOCN
			)	;
			break	;

		case KB::ITBool		:
			/* Default is to generate zero for false and	*/
			/* one for true. May be overridden by the	*/
			/* database drivers.				*/
			{
				QString	cs = QString(value->m_data).lower() ;
				bool	ok ;
				bool	r  ;

				if 	(cs == "yes"  ) r = true  ;
				else if (cs == "true" ) r = true  ;
				else if (cs == "t"    ) r = true  ;
				else if (cs == "no"   ) r = false ;
				else if (cs == "false") r = false ;
				else if (cs == "f"    ) r = false ;
				else
				{
					r = cs.toInt(&ok) != 0 ;
					if (!ok) r = cs.length() > 0 ;
				}

				buffer.append (r ? '1' : '0') ;
				return	;
			}

		case KB::ITRaw		:
			/* Raw is passed as-is to the query, This is	*/
			/* used for stuff like database defaults.	*/
			buffer.append (value->m_data, value->m_length) ;
			return	;

		default	:
			KBError::EFault
			(	QString (TR("KBType::getQueryText: Unexpected type %1")).arg((int)m_iType),
				QString::null,
				__ERRLOCN
			)	;
			buffer.append	("null") ;
			return	;
	}

	buffer.append ('\'') ;

	if (codec != 0)
	{
		QString  text	= QString::fromUtf8  (value->m_data, value->m_length) ;
		QCString local  = codec->fromUnicode (text) ;

		escapeText (local, buffer) ;
	}
	else	escapeText (value, buffer) ;

	buffer.append ('\'') ;
}

/*  KBType								*/
/*  getQueryText: Get text for insertion into a query			*/
/*  value	: KBDataArray *  : Raw text to escape			*/
/*  d		: KBShared *	 : Decoded representation		*/
/*  (returns)	: void		 :					*/

QString	KBType::getQueryText
	(	KBDataArray	*value,
		KBShared	*d
	)
{
	if (value == 0)
		return	"null"	;

//	fprintf
//	(	stderr,
//		"KBType::getQueryText: [%8s][%s]\n",
//		(cchar *)getDescrip(),
//		value->m_data
//	)	;

	switch (m_iType)
	{
		case KB::ITString	:
			/* String. The value will be enclosed in a	*/
			/* pair of quotes.				*/
			break	;

		case KB::ITFixed	:
		case KB::ITFloat	:
			/* Also easy. The text of the integer or double	*/
			/* can be used as-is.				*/
			return	value->m_data ;

		case KB::ITDate		:
		case KB::ITTime		:
		case KB::ITDateTime	:
			/* Convert the date and time to the default	*/
			/* format and return it enclosed in quotes.	*/
			{
				QCString date = ((KBDateTime *)d)->defFormat(m_iType).utf8() ;
				return	 QString("'%1'").arg(date) ;
			}


		case KB::ITBinary	:
			/* Although this is a valid type, it should	*/
			/* always be handled in a driver-specific	*/
			/* derived class. 				*/
			return	"[Binary data]" ;

		case KB::ITDriver	:
			/* Similarly for driver-specific types ...	*/
			return	"[Driver data]" ;

		case KB::ITBool		:
			/* Default is to generate zero for false and	*/
			/* one for true. May be overridden by the	*/
			/* database drivers.				*/
			{
				QString	cs = QString(value->m_data).lower() ;
				bool	ok ;
				bool	r  ;

				if 	(cs == "yes"  ) r = true  ;
				else if (cs == "true" ) r = true  ;
				else if (cs == "t"    ) r = true  ;
				else if (cs == "no"   ) r = false ;
				else if (cs == "false") r = false ;
				else if (cs == "f"    ) r = false ;
				else
				{
					r = cs.toInt(&ok) != 0 ;
					if (!ok) r = cs.length() > 0 ;
				}

				return	r ? "1" : "0" ;
			}

		case KB::ITRaw		:
			/* Raw is passed as-is to the query, This is	*/
			/* used for stuff like database defaults.	*/
			return	value->m_data ;

		default	:
			KBError::EFault
			(	QString (TR("KBType::getQueryText: Unexpected type %1")).arg((int)m_iType),
				QString::null,
				__ERRLOCN
			)	;
			return	"[Unknown type]" ;
	}

	KBDataBuffer	tmp	;
	escapeText (value, tmp) ;
	return	QString("'%1'").arg(tmp.data())	 ;
}

	
static	QString	*__ITUnknown  ;
static	QString	*__ITRaw      ;
static	QString	*__ITFixed    ;
static	QString	*__ITFloat    ;
static	QString	*__ITDecimal  ;
static	QString	*__ITDate     ;
static	QString	*__ITTime     ;
static	QString	*__ITDateTime ;
static	QString	*__ITString   ;
static	QString	*__ITBinary   ;
static	QString	*__ITBool     ;
static	QString	*__ITDriver   ;
static	QString	*__ITInvalid  ;


QString	KBType::getDescrip
	(	bool		all
	)
{
	if (__ITUnknown == 0)
	{
		__ITUnknown  = new QString(TR("Unknown" )) ;
		__ITRaw      = new QString(TR("Raw"     )) ;
		__ITFixed    = new QString(TR("Fixed"   )) ;
		__ITFloat    = new QString(TR("Float"   )) ;
		__ITDecimal  = new QString(TR("Decimal" )) ;
		__ITDate     = new QString(TR("Date"    )) ;
		__ITTime     = new QString(TR("Time"    )) ;
		__ITDateTime = new QString(TR("DateTime")) ;
		__ITString   = new QString(TR("String"  )) ;
		__ITBinary   = new QString(TR("Binary"  )) ;
		__ITBool     = new QString(TR("Bool"    )) ;
		__ITDriver   = new QString(TR("Driver"  )) ;
		__ITInvalid  = new QString(TR("Invalid" )) ;
	}

	QString	t ;

	switch (m_iType)
	{
		case KB::ITUnknown  : t = *__ITUnknown	; break ;
		case KB::ITRaw 	    : t = *__ITRaw	; break ;
		case KB::ITFixed    : t = *__ITFixed	; break ;
		case KB::ITFloat    : t = *__ITFloat	; break ;
		case KB::ITDecimal  : t = *__ITDecimal	; break ;
		case KB::ITDate     : t = *__ITDate	; break ;
		case KB::ITTime     : t = *__ITTime	; break ;
		case KB::ITDateTime : t = *__ITDateTime	; break ;
		case KB::ITString   : t = *__ITString	; break ;
		case KB::ITBinary   : t = *__ITBinary	; break ;
		case KB::ITBool     : t = *__ITBool	; break ;
		case KB::ITDriver   : t = *__ITDriver	; break ;
		default		    : t = *__ITInvalid	; break ;
	}

	if (all)
		t = QString("%1: (%2,%3)").arg(t).arg(m_length).arg(m_prec) ;

	return	t ;
}


/*  KBStaticType								*/
/*  KBStaticType: Constructor for static value type base class		*/
/*  iType	: Internal	: Internal type				*/
/*  length	: uint		: Underlying database length		*/
/*  proc	: uint		: Precision where applicable		*/
/*  nullOK	: bool		: True if value may be null		*/
/*  (returns)	: KBType	:					*/
	
KBStaticType::KBStaticType
	(	KB::IType iType,
		uint	  length,
		uint	  prec,
		bool	  nullOK
	)
	:
	KBType	("Static", iType, length, prec, nullOK, true)
{
}

/*  ------------------------------------------------------------------  */

KBType	*KBType::typeToType
	(	KB::IType	itype
	)
{
	QIntDict<KBType> ttMap ;

	if (ttMap.count() == 0)
	{
		ttMap.insert (KB::ITUnknown,	&_kbUnknown ) ;
		ttMap.insert (KB::ITRaw,	&_kbRaw	    ) ;
		ttMap.insert (KB::ITFixed,	&_kbFixed   ) ;
		ttMap.insert (KB::ITFloat,	&_kbFloat   ) ;
		ttMap.insert (KB::ITDate,	&_kbDate    ) ;
		ttMap.insert (KB::ITTime,	&_kbTime    ) ;
		ttMap.insert (KB::ITDateTime, 	&_kbDateTime) ;
		ttMap.insert (KB::ITString,	&_kbString  ) ;
		ttMap.insert (KB::ITBinary,	&_kbBinary  ) ;
		ttMap.insert (KB::ITBool,	&_kbBool    ) ;
		ttMap.insert (KB::ITDriver,	&_kbDriver  ) ;
	}

	KBType	*res = ttMap.find (itype) ;
	return	res == 0 ? &_kbUnknown : res ;
}


/*  ------------------------------------------------------------------  */

/*  getNumTypeObjects							*/
/*		: Return the number of type objects			*/
/*  (returns)	: int		: Number of type objects		*/

LIBCOMMON_API int	getNumTypeObjects ()
{
	/* This is used for debugging, to check that we don't forget to	*/
	/* remove orphaned type objects.				*/
	return	numTypeObjects ;
}
