/***************************************************************************
    file	         : kb_dispscroller.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#if		__KB_EMBEDDED
#include	<qpe/qpeapplication.h>
#endif

#include	<qpainter.h>
#include	<qobjectlist.h>

#ifndef 	_WIN32
#include	"kb_dispscroller.moc"
#else
#include 	"kb_dispscroller.h"
#endif

#include	"kb_item.h"
#include	"kb_display.h"
#include	"kb_dispwidget.h"
#include	"kb_options.h"
#include	"kb_ascii.h"
#include	"kb_report.h"
#include	"kb_attrnav.h"
#include	"kb_writer.h"



/*  KBDispScrollArea							*/
/*  KBDispScrollArea							*/
/*		: Display scrollview constructor			*/
/*  parent	: QWidget *	: Parent widget				*/
/*  display	: KBDisplay *	: Owning display object			*/
/*  rulerX	: KBRuler *	: Associated X ruler			*/
/*  rulerY	: KBRuler *	: Associated Y ruler			*/
/*  showBar	: uint		: Show row scroll bar			*/
/*  stretch	: bool		: Form in stretch mode			*/
/*  (returns)	: KBDispScrollArea:					*/

KBDispScrollArea::KBDispScrollArea
	(	QWidget		*parent,
		KBDisplay	*display,
		KBRuler		*rulerX,
		KBRuler		*rulerY,
		uint		showBar,
		bool		stretch
	)
	:
	QScrollView	(parent, 0, WResizeNoErase|WRepaintNoErase|WStaticContents),
	m_display	(display),
	m_rulerX	(rulerX),
	m_rulerY	(rulerY),
	m_showBar	(0),
	m_stretch	(false)
{
#if	__KB_EMBEDDED
	QPEApplication::setStylusOperation
	(	this->viewport(),
		QPEApplication::RightOnHold
	)	;
	QPEApplication::setStylusOperation
	(	this,
		QPEApplication::RightOnHold
	)	;
#endif
	m_inSetRange	= false	;
	m_rLabel	= 0	;
	m_vBar		= 0	;
	m_rLabel_d	= 0	;
	m_vBar_d	= 0	;
	m_showing	= KB::ShowAsUnknown ;

	enableClipper	(true)	  ;
	setShowbar	(showBar) ;
	setStretchable	(stretch) ;

	connect	(&m_drawTimer1, SIGNAL(timeout()), SLOT(drawTimeout())) ;
	connect	(&m_drawTimer2, SIGNAL(timeout()), SLOT(drawTimeout())) ;
}

/*  KBDispScrollArea							*/
/*  ~KBDispScrollArea							*/
/*		: Display scrollview destructor				*/
/*  (returns)	:		:					*/

KBDispScrollArea::~KBDispScrollArea ()
{
	m_display->displayIsGone () ;
}

void	KBDispScrollArea::addChild
	(	QWidget		*child,
		int		x,
		int		y
	)
{
	KBReport *report = m_display->getOwner()->isReport() ;

	if (report != 0)
	{
		int	left	;
		int	top	;

		report->margins (left, top) ;

		x	+= (int)(left * pixelsPerMM()) ;
		y	+= (int)(top  * pixelsPerMM()) ;
	}

	QScrollView::addChild (child, x, y) ;
}

/*  KBDispScrollArea							*/
/*  setShowbar	: Enable/display display of row scroll bar		*/
/*  showBar	: uint		: Show row scroll bar			*/
/*  (returns)	: void		:					*/

void	KBDispScrollArea::setShowbar
	(	uint	showBar
	)
{
	if (showBar == m_showBar) return ;

	m_showBar = showBar ;

	if (m_showBar == 0)
	{
		DELOBJ	(m_vBar    ) ;
		DELOBJ	(m_vBar_d  ) ;
		DELOBJ	(m_rLabel  ) ;
		DELOBJ	(m_rLabel_d) ;
		return	;
	}

	if (m_vBar == 0)
	{
		/* Setting visible. We need two sets, since in design	*/
		/* mode they must be anchored to the viewport, while	*/
		/* in data mode to the scroller itself.			*/
		m_vBar_d   = new QScrollBar  (QScrollBar::Vertical, viewport()) ;
		m_rLabel_d = new KBRecordNav (viewport(), m_vBar_d->sizeHint().width()) ;
		addChild (m_vBar_d  ) ;
		addChild (m_rLabel_d) ;

		m_vBar	   = new QScrollBar  (QScrollBar::Vertical, this) ;
		m_rLabel   = new KBRecordNav (this, m_vBar->sizeHint().width()) ;

		/* Wire up the controls. The one in design mode is for	*/
		/* visual effect only.					*/
		connect
		(	m_vBar,
			SIGNAL(valueChanged(int)), 
			SLOT  (vbarMoved())
		)	;
		connect
		(	m_rLabel,
			SIGNAL(action(KB::Action)),
			SLOT  (slotAction(KB::Action))
		)	;
	}

	/* We must be in design mode, so maybe show the design widgets	*/
	/* and hide the data mode ones.					*/
	if ((m_showBar & NAV_SCROLL) == 0)
		m_vBar_d  ->hide () ;
	else	m_vBar_d  ->show () ;

	if ((m_showBar & NAV_MINI  ) == 0)
		m_rLabel_d->hide () ;
	else	m_rLabel_d->show () ;

	m_vBar    ->hide () ;
	m_rLabel  ->hide () ;

	sizeAdjusted   () ;
}

/*  KBDispScrollArea							*/
/*  setSetstretchable							*/
/*		: Enable/display display stretching			*/
/*  stretch	: bool		: True to enable			*/
/*  (returns)	: void		:					*/

void	KBDispScrollArea::setStretchable
	(	bool	stretch
	)
{
	m_stretch = stretch ;
}

/*  KBDispScrollArea							*/
/*  showAs	: Set or clear design mode				*/
/*  mode	: KB::ShowAs	: Design mode				*/
/*  (returns)	: void		:					*/

void	KBDispScrollArea::showAs
	(	KB::ShowAs	mode
	)
{
	m_showing = mode ;

#if	! __KB_RUNTIME
	if (mode == KB::ShowAsDesign)
		setVScrollBarMode (Auto) ;
	else
#endif
		setVScrollBarMode (m_showBar ? AlwaysOff : Auto) ;

	if (m_showBar != 0)
	{
#if	! __KB_RUNTIME
		/* If showing the scroll bar and record label then show	*/
		/* and hide the appropriate pair.			*/
		if (mode == KB::ShowAsDesign)
		{
			if ((m_showBar & NAV_SCROLL) == 0)
				m_vBar_d  ->hide () ;
			else	m_vBar_d  ->show () ;

			if ((m_showBar & NAV_MINI  ) == 0)
				m_rLabel_d->hide () ;
			else	m_rLabel_d->show () ;

			m_vBar    ->hide () ;
			m_rLabel  ->hide () ;
		}
		else
#endif
		{
			if ((m_showBar & NAV_SCROLL) == 0)
				m_vBar    ->hide () ;
			else	m_vBar    ->show () ;

			if ((m_showBar & NAV_MINI  ) == 0)
				m_rLabel  ->hide () ;
			else	m_rLabel  ->show () ;

			m_vBar_d  ->hide () ;
			m_rLabel_d->hide () ;

			m_vBar	  ->raise() ;
		}
	}

	setEnabled	(true)	;
viewport()->erase() ;
	repaintContents (contentsX(), contentsY(), visibleWidth(), visibleHeight()) ;
}

/*  KBDispScrollArea							*/
/*  morphDestoyed: Catch destuction of morphed control			*/
/*  (returns)	 : void		:					*/

void	KBDispScrollArea::morphDestroyed ()
{
//	fprintf
//	(	stderr,
//		"KBDispScrollArea::morphDestroyed [%p]\n",
//		(void *)sender()
//	)	;
	m_morphList.removeRef ((KBItem *)sender()) ;
}

/*  KBDispScrollArea							*/
/*  addMorph	: Add a morphed control					*/
/*  morph	: KBItem *	: Morphed item				*/
/*  (returns)	: void		:					*/

void	KBDispScrollArea::addMorph
	(	KBItem		*morph
	)
{
//	fprintf
//	(	stderr,
//		"KBDispScrollArea::addMorph [%p]\n",
//		(void *)morph
//	)	;
	m_morphList.append (morph) ;

	connect	(morph, SIGNAL(destroyed()), SLOT(morphDestroyed())) ;
}

/*  KBDispScrollArea							*/
/*  remMorph	: Remove a morphed control					*/
/*  morph	: KBItem *	: Morphed item				*/
/*  (returns)	: void		:					*/

void	KBDispScrollArea::remMorph
	(	KBItem		*morph
	)
{
	if (m_morphList.removeRef (morph))
		disconnect (morph, SIGNAL(destroyed()), this, SLOT(morphDestroyed())) ;
}

/*  KBDispScrollArea							*/
/*  updateMorph	: Update a morphed control				*/
/*  morph	: KBItem *	: Morphed item				*/
/*  mRect	: const QRect &	: Area occupied by control		*/
/*  (returns)	: void		:					*/

void	KBDispScrollArea::updateMorph
	(	KBItem		*,
		const QRect	&mRect
	)
{
	viewport()->update (mRect) ;
}

/*  KBDispScrollArea							*/
/*  updateMorph	: Update a morphed control				*/
/*  morph	: KBItem *	: Morphed item				*/
/*  drop	: uint		: Control display row			*/
/*  (returns)	: void		:					*/

void	KBDispScrollArea::updateMorph
	(	KBItem		*morph,
		uint		drow
	)
{
	QPainter p  (viewport  ()) ;
	QPoint	 o = contentsToViewport(QPoint(0,0)) ;
	p.translate (o.x(), o.y()) ;

	morph->repaintMorph  (&p, drow) ;
}

void	KBDispScrollArea::contentsWheelEvent
	(	QWheelEvent	*e
	)
{
	QScrollView::contentsWheelEvent (e) ;
}

/*  KBDispScrollArea							*/
/*  contentsMousePressEvent						*/
/*		: Handler for mouse press events			*/
/*  e		: QMouseEvent *	: Mouse event				*/
/*  (returns)	: void		:					*/

void	KBDispScrollArea::contentsMousePressEvent
	(	QMouseEvent	*e
	)
{
	m_mouseDownAt = e->pos () ;

	if (e->button() == QMouseEvent::LeftButton)
	{
		LITER
		(	KBItem,
			m_morphList,
			morph,
			if (morph->mouseClickHit (e->pos()))
				return ;
		)

#if	! __KB_RUNTIME
		if (m_display->markStartEvent (this, e))
			return	;
#endif
	}

	QScrollView::contentsMousePressEvent (e) ;
}

/*  KBDispScrollArea							*/
/*  contentsMouseMoveEvent						*/
/*		: Handle mouse movement during marking			*/
/*  e		: QMouseEvent *	: Mouse event				*/
/*  (returns)	: void		:					*/

void	KBDispScrollArea::contentsMouseMoveEvent
	(	QMouseEvent *e
	)
{
#if	! __KB_RUNTIME
	if ((e->state() & Qt::LeftButton) != 0)
		m_display->markMoveEvent (this, e) ;

	/* Lastly, a belt-and-braces check, since we seem otherwise to	*/
	/* miss some mouse release events.				*/
	if ((e->state() & Qt::LeftButton) == 0)
		m_display->markEndEvent (this, e) ;
#endif
}

/*  KBDispScrollArea							*/
/*  contentsMouseReleaseEvent						*/
/*		: Handle mouse release to end marking			*/
/*  e		: QMouseEvent *	: Mouse event				*/
/*  (returns)	: void		:					*/

void	KBDispScrollArea::contentsMouseReleaseEvent
	(	QMouseEvent *e
	)
{
#if	! __KB_RUNTIME
	if (e->button() == Qt::RightButton)
	{
		QPoint	mouseUpAt = e->pos() ;
		int	dx	  = mouseUpAt.x() - m_mouseDownAt.x() ;
		int	dy	  = mouseUpAt.y() - m_mouseDownAt.y() ;

		if (((dx * dx) + (dy * dy)) < 25)
		{
			m_display->contextEvent (this, e) ;
			return ;
		}
	}

	m_display->markEndEvent (this, e) ;
#endif
}

/*  KBDispScrollArea							*/
/*  contentsMouseDoubleClickEvent					*/
/*		: Handler for mouse double click events			*/
/*  e		: QMouseEvent *	: Mouse event				*/
/*  (returns)	: void		:					*/

void	KBDispScrollArea::contentsMouseDoubleClickEvent
	(	QMouseEvent	*e
	)
{
	/* Debugging hook. Double-click with the control and shift keys	*/
	/* down prints the widget tree.					*/
	if ((e->state() & (ShiftButton|ControlButton)) == (ShiftButton|ControlButton))
	{
		extern	void	printWidgetTree (QWidget *, uint, int) ;
		printWidgetTree (this, 0, -1) ;
		return	;
	}

#if	! __KB_RUNTIME
	m_display->contextEvent (this, e) ;
#endif
}

/*  KBDispScrollArea							*/
/*  drawContents: Handle painting for area marker			*/
/*  p		: QPainter *	: Painter				*/
/*  clipx, ...	: int, ...	: Clipping rectangle			*/
/*  (returns)	: void		:					*/

void	KBDispScrollArea::drawContents
	(	QPainter *p,
		int	 clipx,
		int	 clipy,
		int	 clipw,
		int	 cliph
	)
{
//	fprintf
//	(	stderr,
//		"KBDispScrollArea::drawContents (%d,%d,%d,%d) [%f,%f]\n",
//		clipx, clipy, clipw, cliph,
//		p->translationX(),
//		p->translationY()
//	)	;

	QScrollView::drawContents (p, clipx, clipy, clipw, cliph) ;
	QRect	rect		  (   clipx, clipy, clipw, cliph) ;

#if	! __KB_RUNTIME
	/* NOTE: Have to call erase else we get some spurious broder	*/
	/* lines wheh switching design->display mode, due to resizing.	*/
	/* Probably a buglette elsewehere.				*/
	if (m_showing == KB::ShowAsDesign)
	{
		p->eraseRect (rect) ;
		m_display->drawDisplay (p, rect) ;
	}
#endif

	if (m_rulerX != 0) m_rulerX->setOffset (contentsX()) ;
	if (m_rulerY != 0) m_rulerY->setOffset (contentsY()) ;

	/* QT seems to generate draws for the entire visible area when	*/
	/* shrinking. This means lots of unnecessary morph repaints,	*/
	/* which is a real pain in table data view. So, bunch stuff up	*/
	/* with a timer.						*/
	if (m_drawRect.isValid())
		m_drawRect |= rect ;
	else	m_drawRect  = rect ;

	m_drawTimer1.start (100, true) ;
	if (!m_drawTimer2.isActive()) m_drawTimer2.start (500, true) ;

//	fprintf	(stderr, "KBDispScrollArea: drawRect now %s\n", _TEXT(drawRect)) ;

//	LITER
//	(	KBItem,
//		morphList,
//		morph,
//		morph->repaintMorph (p, rect) ;
//	)
}

void	KBDispScrollArea::drawTimeout ()
{
	QPainter p  (viewport()) ;
	QPoint	 tr = contentsToViewport(QPoint(0,0)) ;

	p.translate (tr.x(), tr.y()) ;

	LITER
	(	KBItem,
		m_morphList,
		morph,
		morph->repaintMorph (&p, m_drawRect) ;
	)

//	fprintf
//	(	stderr,
//		"KBDispScrollArea: updated      %s %s  [%f,%f]\n",
//		_TEXT(drawRect),
//		_TEXT(contentsToViewport(QPoint(0,0))),
//		p.translationX(),
//		p.translationY()
//	)	;

	m_drawRect = QRect() ;
	m_drawTimer1.stop () ;
	m_drawTimer2.stop () ;
}

/*  KBDispScrollArea							*/
/*  setVBarGeometry							*/
/*		: Set geometry of the vertical scroll bar		*/
/*  vbar	: QScrollBar &	: Vertical scroll bar			*/
/*  x, y	: int, int	: Default x and y			*/
/*  w, h	: int, int	: Default w and h			*/
/*  (returns)	: void		:					*/

void	KBDispScrollArea::setVBarGeometry
	(	QScrollBar	&vbar,
		int		x,
		int		y,
		int		w,
		int		h
	)
{
	if (h > height() - w) h -= w ;
	vbar.setGeometry (x, y, w, h) ;
}

/*  KBDispScrollArea							*/
/*  setHBarGeometry							*/
/*		: Set geometry of the horizontal scroll bar		*/
/*  hbar	: QScrollBar &	: Horizontal scroll bar			*/
/*  x, y	: int, int	: Default x and y			*/
/*  w, h	: int, int	: Default w and h			*/
/*  (returns)	: void		:					*/

void	KBDispScrollArea::setHBarGeometry
	(	QScrollBar	&hbar,
		int		x,
		int		y,
		int		w,
		int		h
	)
{
	if (w > width() - h)
		w -= h ;

	if (m_showBar && (m_showing == KB::ShowAsData))
	{
		x += m_rLabel->size().width() ;
		w -= m_rLabel->size().width() ;
	}

	hbar .setGeometry (x, y, w, h) ;
}


/*  KBDispScrollArea							*/
/*  resizeEvent	: Handle a resize event					*/
/*  e		: QResizeEvent * : The event				*/
/*  (returns)	: void		 :					*/

void	KBDispScrollArea::resizeEvent
	(	QResizeEvent	*e
	)
{
	QScrollView::resizeEvent (e) ;

	/* If the display is stretchable and we are _not_ in design	*/
	/* mode, then resize the contents to be exactly the same size	*/
	/* as the visible area, and notify the display.			*/
	if (m_showing == KB::ShowAsData)
	{
		int	vw	= visibleWidth () ;
		int	vh	= visibleHeight() ;
		int	wsfix	= frameWidth   () ;

		if (m_stretch)
		{
			QSize	bs	= m_display->getBaseSize () ;

			if (vw < bs.width ()) vw = bs.width () ;
//			if (vh < bs.height()) vh = bs.height() ;

			resizeContents	(vw, vh) ;

			m_display->resizeEvent
			(	visibleWidth () - wsfix * 2,
				visibleHeight() - wsfix * 2
			)	;
		}
	}

	sizeAdjusted () ;
}

/*  KBDispScrollArea							*/
/*  sizeAdjusted: Size has been adjusted				*/
/*  (returns)	: void		 :					*/

void	KBDispScrollArea::sizeAdjusted ()
{
	/* If we are showing our own scrollbar then we have to fix its	*/
	/* geometry outselves; we must also move the record count label	*/
	/* Where it goes also depends on whether we are in design mode	*/
	/* or not.							*/
	if (m_showBar)
	{
		int	w	;
		int	h	;
		int	wsfix	= frameWidth() ;

#if	! __KB_RUNTIME
		if (m_showing == KB::ShowAsDesign)
		{
			/* In design mode, our scrollbar and label are	*/
			/* aligned to the notional area occupied by the	*/
			/* topmost form widget.				*/
			QSize	topSize	= m_display->getTopSize () ;
			w	= topSize.width  () ;
			h	= topSize.height () ;
		}
		else
#endif
		{	/* Not in design mode so our bar and label go	*/
			/* along the edges of the scrollview.		*/
			w	= width () ;
			h	= height() ;
		}


		QScrollBar *hBar = horizontalScrollBar   () ;
		int	   sbw   = m_vBar->sizeHint().width() ;
		int	   sbh   = hBar == 0 ? sbw : hBar->sizeHint().height() ;


		/* The design mode scrollbar and label are adjusted so	*/
		/* that they stay fixed relative to the viewport ...	*/
		moveChild (m_vBar_d,   w - sbw - wsfix * 2, 0) ;
		moveChild (m_rLabel_d, 0, h - sbh - wsfix * 2) ;
		m_vBar_d  ->resize (sbw,  h - sbh - wsfix * 2) ;

		/* ... while the data mode scroller and label are kept	*/
		/* at the edjes of the scroll view.			*/
		m_vBar  ->setGeometry
		(	w - sbw - wsfix, // - contentsX(),
			wsfix,		 // - contentsY(),
			sbw,
			h - sbh - wsfix *2
		)	;
		m_rLabel->move   (wsfix, h - m_rLabel->size().height() - wsfix) ;

#if	! __KB_RUNTIME
		if (m_showing == KB::ShowAsDesign)
		{
			m_vBar_d  ->raise  () ;
			m_rLabel_d->raise  () ;
		}
		else
#endif
		{
			m_vBar    ->raise  () ;
			m_rLabel  ->raise  () ;
		}
	}
}

/*  KBDispScrollArea							*/
/*  setRowRange	: Set range of rows being displayed			*/
/*  extra	: uint		: Extra insertion rows			*/
/*  curQRow	: uint		: Current query row			*/
/*  curDRow	: uint		: Index of topmost displayed row	*/
/*  dispRows	: uint		: Number of rows actually visible	*/
/*  (returns)	: void		:					*/

void	KBDispScrollArea::setRowRange
	(	uint	totalRows,
		uint	extra,
		uint	curQRow,
		uint	curDRow,
		uint	dispRows
	)
{
	if (m_showBar)
	{
		/* Note the scrollbar-moved feedback is disabled while	*/
		/* setting the range. This is because range setting is	*/
		/* invoked from the block object, which will already	*/
		/* have scrolled its contents.				*/
		m_inSetRange = true  ;
		m_vBar->setRange   (0, totalRows + extra - dispRows) ;
		m_vBar->setSteps   (1, dispRows <= 1 ? 1 : dispRows  - 1) ;
		m_vBar->setValue   (curDRow) ;
		m_inSetRange = false ;

		m_rLabel->setRecord(curQRow, totalRows) ;
	}
}

/*  KBDispScrollArea							*/
/*  vbarMoved	: Notification of scrollbar change			*/
/*  (returns)	: void		:					*/

void	KBDispScrollArea::vbarMoved ()
{
	/* If we are not setting the value then we need to notify the	*/
	/* owning block. But, it is possible that the block may reject	*/
	/* the scroll and move us back, and there does not seem to be	*/
	/* a "allow this scroll" hook in QScrollBar. Now, if we allow	*/
	/* a reset _before_ the scroll bar move-and-redraw completes,	*/
	/* we get some nasty recursive error loops. So, set a timer to	*/
	/* give time for things to settle down ....			*/
	if (!m_inSetRange)
		m_scrollTimer.singleShot (200, this, SLOT(scrollTimeout())) ;
}

/*  KBDispScrollArea							*/
/*  scrollTimeout: Timeout on scroll bar movement			*/
/*  (returns)	 : void		:					*/

void	KBDispScrollArea::scrollTimeout ()
{
	int	value = m_vBar->value() ;
	m_display->scrollToRow (value < 0 ? 0 : value) ;
}


/*  KBDispScrollArea							*/
/*  slotAction	: Action requested					*/
/*  action	: KB::Action	: Action in question			*/
/*  (returns)	 : void		:					*/

void	KBDispScrollArea::slotAction
	(	KB::Action	action
	)
{
	m_display->doAction (action) ;
}

/*  KBDispScrollArea							*/
/*  effectiveSize : Get effective display size				*/
/*  topSize	  : QSize	: Top object size			*/
/*  (returns)	  : QSize	: Effective size			*/

QSize	KBDispScrollArea::effectiveSize
	(	QSize		topSize
	)
{
#if	! __KB_RUNTIME
	if (m_showing == KB::ShowAsDesign)
	{
		int	w = topSize.width () ;
		int	h = topSize.height() ;

		if (m_showBar)
		{	w -= verticalScrollBar  ()->sizeHint().width () ;
			h -= horizontalScrollBar()->sizeHint().height() ;
		}

		return	QSize (w, h) ;
	}
#endif

	if (m_stretch)
	{
		int	w = visibleWidth () ;
		int	h = visibleHeight() ;

		if (m_showBar)
			w -= verticalScrollBar  ()->sizeHint().width () ;

		if (m_showBar || horizontalScrollBar()->isVisible())
			h -= horizontalScrollBar()->sizeHint().height() ;

		return	QSize (w, h) ;
	}

	return	topSize	;
}

/*  KBDispScrollArea							*/
/*  o		: QObject *	: Actually pointer at the widget	*/
/*  e		: QEvent *	: Event to process			*/
/*  (returns)	: bool		: False to continue processing		*/

bool	KBDispScrollArea::eventFilter
	(	QObject		*o,
		QEvent		*e
	)
{
	/* NOTE: It seems that the QScrollView installs an event filter	*/
	/* on the viewport itself, so we do not need install it. The	*/
	/* other point is to call QScrollView::eventFilter if we are	*/
	/* not consuming the event.					*/

	/* Pass key presses to the display object and thence the block	*/
	/* navigator. This is used to bring the current control back	*/
	/* into view if it has been scrolled out of site.		*/
	if ((o == viewport()) && (e->type() == QEvent::KeyPress))
		return	m_display->keyStroke ((QKeyEvent *)e) ;


	/* Trap wheel events. These are passed to the display object	*/
	/* to scroll, and are then accepted and consumed.		*/
	if (e->type() == QEvent::Wheel)
	{
		QWheelEvent *w = (QWheelEvent *)e ;
		m_display->scrollBy (- w->delta() / 120) ;
		w->accept() 	;
		return	true	;
	}

	return	QScrollView::eventFilter (o, e) ;
}

/*  KBDispScrollArea							*/
/*  moveTags	: Move design tags					*/
/*  widget	: KBDispWidget * : Widget requiring move		*/
/*  y		: uint		 : Y position of tag			*/
/*  (returns)	: void		 :					*/

void	KBDispScrollArea::moveTags
	(	KBDispWidget	*widget,
		uint		y
	)
{
	QLabel	*tag	= widget->getTagLabel (viewport()) ;
	QPoint	pos	= viewportToContents  (QPoint(widget->width(), y)) ;

	if (tag != 0)
	{
		KBReport *report = m_display->getOwner()->isReport() ;

		int	x = widget->width() ;
		int	y = pos.y() ;

		if (report != 0)
		{
			int	left	;
			int	top	;

			report->margins (left, top) ;

			y -= (int)(top * pixelsPerMM()) ;
		}

		moveChild (tag, x, y) ;
	}


	QObjectList *children = widget->queryList("KBDispWidget", 0, true, false) ;
	if (children == 0) return ;

	QObjectListIt iter (*children) ;
	QObject	      *obj ;

	while ((obj = iter.current()) != 0)
	{
		KBDispWidget *dw  = (KBDispWidget *)obj	  ;
		moveTags (dw, y + dw->y()) ;
		*++iter ;
	}

	delete	children ;
}

/*  KBDispScrollArea							*/
/*  makeVisible	: Make a control area visible in the viewport		*/
/*  rRect	: const QRect &		: Control area			*/
/*  reason	: QFocusEvent::Reason	: Associated focus reason	*/
/*  (returns)	: void			:				*/

void	KBDispScrollArea::makeVisible
	(	const QRect		&rRect,
		QFocusEvent::Reason	reason
	)
{
	int	contX	= contentsX	() ;
	int	contY	= contentsY	() ;
	int	visW	= visibleWidth	() ;
	int	visH	= visibleHeight	() ;
	bool	move	= false	;

	if (verticalScrollBar  () != 0) visW -= verticalScrollBar  ()->width () ;
	if (horizontalScrollBar() != 0) visH == horizontalScrollBar()->height() ;
//	fprintf
//	(	stderr,
//		"KBDispScrollArea::makeVisible: rl=%d rr=%d cx=%d vw=%d\n",
//		rRect.left (),
//		rRect.right(),
//		contX,
//		visW
//	)	;

	if ((rRect.right () >= contX + visW) || (rRect.left() < contX))
	{
		if (reason == QFocusEvent::Backtab)
			contX	= rRect.right() - visW	;
		else	contX	= rRect.left ()		;

		move	= true	;
	}

	if ((rRect.bottom() >= contY + visH) || (rRect.top () < contY))
	{
		if (reason == QFocusEvent::Backtab)
			contY	= rRect.bottom() - visH ;
		else	contY	= rRect.top   ()	;

		move	= true	;
	}

	if (move)
	{
		fprintf
		(	stderr,
			"KBDispScrollArea::makeVisible: to (%d,%d\n",
			contX,
			contY
		)	;

		setContentsPos (contX, contY) ;
	}
}

/*  KBDispScrollArea							*/
/*  focusNextPrevChild							*/
/*		: Next/prev focus change handler			*/
/*  next	: bool		: Moving to next control		*/
/*  (returns)	: void		:					*/

bool	KBDispScrollArea::focusNextPrevChild
	(	bool
	)
{
	/* This method is overridden since we handle all this ourselves	*/
	/* Otherwise, we get problems as controls are morphed back and	*/
	/* forth.							*/
	return	true ;
}

/*  KBDispScrollArea							*/
/*  setContentsPos							*/
/*		: Set contents position in viewport			*/
/* x, y		: int, int	: Top-left position			*/
/*  (returns)	: void		:					*/

void	KBDispScrollArea::setContentsPos
	(	int		x,
		int		y
	)
{
	/* Override for debugging only ....				*/
	fprintf
	(	stderr,
		"KBDispScrollArea::setContentsPos: (%d,%d)\n",
		x, y
	)	;

	QScrollView::setContentsPos (x, y) ;
}


/*  ------------------------------------------------------------------  */

/*  KBDispScroller							*/
/*  KBDispScroller							*/
/*		: Display scrollview constructor			*/
/*  parent	: QWidget *	: Parent widget				*/
/*  display	: KBDisplay *	: Owning display object			*/
/*  showBar	: uint		: Show row scroll bar			*/
/*  stretch	: bool		: Form in stretch mode			*/
/*  showRuler	: bool		: Show design ruler			*/
/*  (returns)	: KBDispScrollArea:					*/

KBDispScroller::KBDispScroller
	(	QWidget		*parent,
		KBDisplay	*display,
		uint		showBar,
		bool		stretch,
		bool		showRuler
	)
	:
	QWidget		(parent),
	m_rulerX	(showRuler ? new KBRuler (this, true ) : 0),
	m_rulerY	(showRuler ? new KBRuler (this, false) : 0),
	m_scroller	(this, display, m_rulerX, m_rulerY, showBar, stretch),
	m_layout	(this)
{
	if (showRuler)
	{
		double	step	= 10	;
		int	frac	= 5	;
		int	lincr	= 10	;

		if (KBOptions::getDesignInches())
		{
			step	= 25.4	;
			frac	= 10	;
			lincr	= 1	;
		}


		m_rulerX->setFrame	(m_scroller.frameWidth()) ;
		m_rulerX->setStep	(step, pixelsPerMM(), frac, lincr) ;

		m_rulerY->setFrame	(m_scroller.frameWidth()) ;
		m_rulerY->setStep	(step, pixelsPerMM(), frac, lincr) ;

		m_layout.addWidget 	(m_rulerX,    0, 1) ;
		m_layout.addWidget 	(m_rulerY,    1, 0) ;
		m_layout.addWidget 	(&m_scroller, 1, 1) ;
	}
	else	m_layout.addWidget	(&m_scroller, 0, 0) ;
}

KBDispScroller::~KBDispScroller ()
{
}

/*  KBDispScroller							*/
/*  showAs	: Set or clear design mode				*/
/*  mode	: KB::ShowAs	: Design mode				*/
/*  (returns)	: void		:					*/

void	KBDispScroller::showAs
	(	KB::ShowAs	mode
	)
{
	m_scroller.showAs (mode) ;
}

/*  KBDispScroller							*/
/*  setSetstretchable							*/
/*		: Enable/display display stretching			*/
/*  _stretch	: bool		: True to enable			*/
/*  (returns)	: void		:					*/

void	KBDispScroller::setStretchable
	(	bool	stretch
	)
{
	m_scroller.setStretchable (stretch) ;
}

/*  KBDispScrollArea							*/
/*  setShowbar	: Enable/display display of row scroll bar		*/
/*  _showBar	: uint		: True to enable			*/
/*  (returns)	: void		:					*/

void	KBDispScroller::setShowbar
	(	uint	showBar
	)
{
	m_scroller.setShowbar (showBar) ;
}

/*  KBDispScroller							*/
/*  setRowRange	: Set range of rows being displayed			*/
/*  totalRows	: uint		: Total number of rows			*/
/*  extra	: uint		: Extra insertion rows			*/
/*  curQRow	: uint		: Current query row			*/
/*  curDRow	: uint		: Index of topmost displayed row	*/
/*  dispRows	: uint		: Number of rows actually visible	*/
/*  (returns)	: void		:					*/

void	KBDispScroller::setRowRange
	(	uint	totalRows,
		uint	extra,
		uint	curQRow,
		uint	curDRow,
		uint	dispRows
	)
{
	m_scroller.setRowRange (totalRows, extra, curQRow, curDRow, dispRows) ;
}

/*  KBDispScroller							*/
/*  effectiveSize : Get effective display size				*/
/*  topSize	  : QSize	: Top object size			*/
/*  (returns)	  : QSize	: Effective size			*/

QSize	KBDispScroller::effectiveSize
	(	QSize		topSize
	)
{
	return	m_scroller.effectiveSize (topSize) ;
}

/*  KBDispScroller							*/
/*  sizeAdjusted: Size has been adjusted				*/
/*  (returns)	: void		 :					*/

void	KBDispScroller::sizeAdjusted ()
{
	m_scroller.sizeAdjusted () ;
}

