/***************************************************************************
    file	         : kb_parse.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<stdarg.h>


#include	<qpopupmenu.h>
#include	<qfile.h>
#include	<qtextstream.h>
#include	<qregexp.h>

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_object.h"
#include	"kb_slot.h"
#include	"kb_location.h"
#include	"kb_macro.h"

//#define	__KB_NINST
#include	"kb_nodereg.h"
//#undef	__KB_NINST

#include	"kb_parse.h"

#include	"kb_form.h"
#include	"kb_report.h"

class	KBAttrSortList	: public QList<KBAttr>
{

protected	:

	virtual	int	compareItems (QCollection::Item, QCollection::Item) ;
}	;

int	KBAttrSortList::compareItems
	(	QCollection::Item	item1,
		QCollection::Item	item2
	)
{
	return	(int)((KBAttr *)item1)->getOrder() -
		(int)((KBAttr *)item2)->getOrder() ;
}

/*  ------------------------------------------------------------------  */

static	QList<NodeSpec>		*nodeSpecs	;
static	QDict<uint>		nodeFlags	;
static	QDict<QString>		dummyDict	;


/*  KBNodeReg								*/
/*  KBNodeReg	: Constructor for node registration class		*/
/*  nodeName	: cchar *	: XML node name				*/
/*  msgText	: cchar *	: Text for popup menu			*/
/*  nodeFunc	: MKNODE	: Node creation function		*/
/*  flags	: uint		: Applicability flags			*/
/*  (returns)	:		:					*/

KBNodeReg::KBNodeReg
	(	cchar		*nodeName,
		cchar		*msgText,
		MKNODE		nodeFunc,
		uint		flags
	)
{
//	fprintf
//	(	stderr,
//		"KBNodeReg::KBNodeReg: [%12s] [%s]\n",
//		nodeName,
//		(cchar *)msgText
//	)	;

	NodeSpec *ns = new NodeSpec 	;

	ns->nodeName  = nodeName	;
	ns->msgText   = msgText 	;
	ns->popupFunc = 0		;
	ns->nodeFunc  = nodeFunc	;
	ns->flags     = flags		;

	if (nodeSpecs == 0) nodeSpecs = new QList<NodeSpec> ;
	nodeSpecs->append (ns)		;
}

/*  KBNodeReg								*/
/*  KBNodeReg	: Constructor for node registration class		*/
/*  nodeName	: cchar *	: XML node name				*/
/*  popupFunc	: MKPOPUP	: Popup menu creation function		*/
/*  nodeFunc	: MKNODE	: Node creation function		*/
/*  flags	: uint		: Applicability flags			*/
/*  (returns)	:		:					*/

KBNodeReg::KBNodeReg
	(	cchar		*nodeName,
		MKPOPUP		popupFunc,
		MKNODE		nodeFunc,
		uint		flags
	)
{
//	fprintf
//	(	stderr,
//		"KBNodeReg::KBNodeReg: [%12s] [%p]\n",
//		nodeName,
//		(void  *)popupFunc
//	)	;

	NodeSpec *ns = new NodeSpec 	;

	ns->nodeName  = nodeName	;
	ns->msgText   = QString::null	;
	ns->popupFunc = popupFunc	;
	ns->nodeFunc  = nodeFunc	;
	ns->flags     = flags		;

	if (nodeSpecs == 0) nodeSpecs = new QList<NodeSpec> ;
	nodeSpecs->append (ns)		;
}

/*  GetNodeFlags: Get flags for named node type				*/
/*  name	: const QString & : Node name				*/
/*  (returns)	: uint		  : Flags, zero if node not found	*/

LIBKBASE_API
	uint	GetNodeFlags
	(	const QString	&name
	)
{
	uint	*flags = nodeFlags.find (name)	;

	if (flags == 0)
	{
		if (name == "KBForm"  ) return KF_FORM   ;
		if (name == "KBReport") return KF_REPORT ;
		return	0 ;
	}

	return	*flags	;
}


#if	! __KB_RUNTIME && ! __KB_EMBEDDED

/*  exportSGML	: Export node attribute information for documentation	*/
/*  node	: KBNode *	: Node to be exported			*/
/*  me		: bool		: Only export for derived class		*/
/*  (returns)	: void		:					*/

LIBKBASE_API
	void	exportSGML
	(	KBNode		*node,
		bool		me
	)
{
	QFile	af (QString("/tmp/%1.sgml").arg(node->getElement())) ;
	if (!af.open (IO_WriteOnly))
	{	delete	node	;
		return	;
	}

	const QList<KBAttr>	&attribs= node->getAttribs() ;
	KBAttrSortList		sorted	;

	LITER
	(	KBAttr,
		attribs,
		attr,
		sorted.inSort (attr)
	)
	

	QTextStream as (&af) ;

	/* The output is generated directly in the format needed for	*/
	/* the DocBook-XML manual source.				*/
	as << "<informaltable>\n"
	   << "<tgroup cols=\"2\">\n"
	   << "<colspec colname=\"a\" colwidth=\"1*\"/>\n"
	   << "<colspec colname=\"b\" colwidth=\"5*\"/>\n"
	   << "<tbody>\n"
	;

	QRegExp	tag ("<[^>]*>") ;


	/* Get the attributes in the order that thay appear in the	*/
	/* "all properties" group. This means that they will be in the	*/
	/* correct order, and hidden attributes will be passed over.	*/
	LITER
	(	KBAttr,
		sorted,
		attr,

		/* If only showing "my" attributes then check the	*/
		/* attribute owner name agains the node element name	*/
		/* and ignore it if they mismatch.			*/
		if (me && (attr->getOwnerName() != node->getElement()))
			continue ;

		as	<< QString
		   	   (	"<row>\n"
				"<entry><emphasis>%1</emphasis></entry>\n"
				"<entry>%2</entry>\n"
				"</row>\n"
				"<row>\n"
				"<entry></entry>\n"
				"<entry>%3</entry>\n"
				"</row>\n"
			   )
			   .arg	(attr->getName  ())
			   .arg	(attr->getLegend())
			   .arg	(attr->getDescription().replace(tag, ""))
			;
	)

	as << "</tbody>\n"
	   << "</tgroup>\n"
	   << "</informaltable>\n"
	;

	af.close ()	;
}

/*  exportSGML	: Export node attribute information for documentation	*/
/*  spec	: NodeSpec *	: Node specification			*/
/*  me		: bool		: Only export for derived class		*/
/*  (returns)	: void		:					*/

LIBKBASE_API
	void	exportSGML
	(	const NodeSpec	*spec,
		bool		me
	)
{

	KBNode	*node	= (*spec->nodeFunc) (0, dummyDict, 0) ;
	if (node->isObject() != 0) exportSGML (node, me) ;
	delete	node	;
}

LIBKBASE_API
	void	exportSGML
	(	bool		me
	)
{
	for (uint idx = 0 ; idx < nodeSpecs->count() ; idx += 1)
		exportSGML (nodeSpecs->at(idx), me) ;

	KBNode		*node	;
	KBLocation	locn	;

	node	= new KBForm   (locn, dummyDict) ;
	exportSGML (node, me)	;
	delete	node		;

	node	= new KBReport (locn, dummyDict) ;
	exportSGML (node, me)	;
	delete	node		;
}

#endif

/*  LoadNodeFunc: Load node definition functions			*/
/*  pNodeFns	: uint		   : Number checked previously		*/
/*  mask	: int		   : Node type mask			*/
/*  nodeDict	: QDict<NodeSpec>& : Node definition dictionary		*/
/*  (returns)	: uint		   : Number now checked			*/

LIBKBASE_API
	uint LoadNodeFuncs
	(	uint		pNodeFns,
		int		mask,
		QDict<NodeSpec>	&nodeDict
	)
{
#if	0
	static	bool	first	= true	;

	if (first)
	{
		registerKBaseNodes ()	;
		first	= false		;
	}
#endif

#if	! __KB_RUNTIME && ! __KB_EMBEDDED
	cchar	*ee	= getenv ("REKALLEXPORT_DOC") ;
#endif

	while (pNodeFns < nodeSpecs->count())
	{
		const NodeSpec *spec = nodeSpecs->at(pNodeFns) ;

		if ((spec->flags & mask ) != 0)
		{	nodeDict .insert (spec->nodeName, spec) ;
			nodeFlags.insert (spec->nodeName, &spec->flags) ;
		}

		pNodeFns += 1 ;

#if	! __KB_RUNTIME && ! __KB_EMBEDDED
		/* This is a hack! Used to generate documentation by	*/
		/* driving the property dialog for the node to output	*/
		/* the attributes. Yeck!				*/
		if (ee != 0) exportSGML (spec, false) ;
#endif
	}

	return	pNodeFns ;
}

/*  ------------------------------------------------------------------  */

/*  KBSAXHandler							*/
/*  KBSAXHandler: Constructor for base parser class			*/
/*  what	: cchar *	 : What is being parsed			*/
/*  location	: KBLocation &	 : Database location			*/
/*  parent	: KBNode *	 : Initial parent node			*/
/*  (returns)	: KBSAXHandler	 :					*/

KBSAXHandler::KBSAXHandler
	(	cchar		*what,
		KBLocation	&location,
		KBNode		*parent
	)
	:
	m_what		(what),
	m_location	(location)
{
	m_kbTOS		= parent	;
	m_kbTop		= 0		;
	m_kbSlot	= 0		;
	m_kbEvent	= 0		;
	m_kbMacro	= 0		;
	m_state		= Normal	;
	m_bGotErr	= false		;
}

/*  KBSAXHandler							*/
/*  parseFile	: Parse document					*/
/*  xmlSource	: QXmlInputSource & : XML input source			*/
/*  (returns)	: bool		    : Success				*/

bool	KBSAXHandler::parse
	(	QXmlInputSource	&xmlSource
	)
{
	QXmlSimpleReader xmlReader		;

	xmlReader.setContentHandler(this     )	;
	xmlReader.parse		   (xmlSource)	;

	/* See if an error was logged. If so then delete any parse tree	*/
	/* that was forthcoming, then return failure.			*/
	if (m_bGotErr)
	{
		if (m_kbTop != 0) delete m_kbTop ;
		return	false	;
	}

	/* See if any parse tree at all resulted. It will be missing if	*/
	/* the document is empty.					*/
	if (m_kbTop == 0)
	{
		m_lError  = KBError
			  (	KBError::Error,
			  	QString(TR("%1 is empty")).arg(m_what),
			  	QString::null,
			  	__ERRLOCN
			  )	;
		return	false	;
	}

	return	true	;
}

/*  KBSAXHandler							*/
/*  parseFile	: Parse document					*/
/*  document	: const QString &: Document path			*/
/*  (returns)	: bool		 : Success				*/

bool	KBSAXHandler::parseFile
	(	const QString	&document
	)
{
	/* Check that the document actually exists, if not then we can	*/
	/* harldy parse it!						*/
	if (!QFile::exists (document))
	{
		m_lError  = KBError
			  (	KBError::Error,
			  	QString(TR("%1 \"%2\" does not exist"))
			  		.arg(m_what)
			  		.arg(document),
			  	QString::null,
			  	__ERRLOCN
			  )	;
		return	false	;
	}

//	fprintf	(stderr, "About to parse \"%s\"\n", (cchar *)document) ;

	/* Set up the Qt SAX parser and parse the document ...		*/
	QFile		 xmlFile   (document )	;
	QXmlInputSource	 xmlSource (xmlFile  )	;

	return	parse	(xmlSource) ;
}

/*  KBSAXHandler							*/
/*  parseText	: Parse document					*/
/*  document	: QByteArray &	: Document text				*/
/*  (returns)	: bool		: Success				*/

bool	KBSAXHandler::parseText
	(	QByteArray	&document
	)
{
//	fprintf	(stderr, "About to parse text ...\n") ;

	/* Set up the Qt SAX parser and parse the document ...		*/
	QTextStream	 xmlText   (document, IO_ReadOnly) ;
	QXmlInputSource	 xmlSource (xmlText ) ;

	return	parse	(xmlSource) ;
}

/*  KBSAXHandler							*/
/*  setErrMessage: Set an error message					*/
/*  e		 : const QXmlParseException &				*/ 
/*				: SAX parser exception information	*/
/*  (returns)	 : void		:					*/

void	KBSAXHandler::setErrMessage
	(	const QXmlParseException &e
	)
{
	m_lError = KBError
		   (	KBError::Error,
			QString  (TR("%1 parsing error at line %2, column %3"))
			   		.arg(m_what)
					.arg(e.lineNumber  ())
					.arg(e.columnNumber()),
			e.message(),
			__ERRLOCN
		   )	  ;
	m_bGotErr  = true ;
}

/*  KBSAXHandler							*/
/*  setErrMessage: Set an error message					*/
/*  msg		 : const QString&: Error message			*/
/*  arg		 : const QString&: Error message argument		*/
/*  (returns)	 : void		 :					*/

void	KBSAXHandler::setErrMessage
	(	const QString	&msg,
		const QString	&arg
	)
{
	m_lError  = KBError
		  (	KBError::Error,
			QString(TR("Error parsing %1"))	.arg(m_what),
			QString(TR(msg))		.arg(arg),
			__ERRLOCN
		  )	;
	m_bGotErr = true;
}

/*  KBSAXHandler							*/
/*  setErrMessage: Set an error message					*/
/*  msg		 : const QString&: Error message			*/
/*  state	 : ParseState	 : Error state				*/
/*  (returns)	 : void		 :					*/

void	KBSAXHandler::setErrMessage
	(	const QString	&msg,
		ParseState	state
	)
{
	QString	arg	;

	switch (state)
	{
		case Normal	 : arg = "Normal"	; break ;
		case InEvent	 : arg = "InEvent"	; break ;
		case InSlot	 : arg = "InSlot"	; break ;
		case InSlotLink	 : arg = "InSlotLink"	; break ;
		case InSlotCode	 : arg = "InSlotCode"	; break ;
		case InMacro	 : arg = "InMacro"	; break ;
		case InMacroInstr: arg = "InMacroInstr"	; break ;
		case InMacroArg	 : arg = "InMacroArg"	; break ;
		default		 :
			arg = QString("Unknown (%1)").arg(state) ;
			break	 ;
	}

	setErrMessage (msg, QString(": state %1").arg(arg)) ;
	m_bGotErr = true;
}

/*  KBSAXHandler							*/
/*  endElement	: Handle end of element					*/
/*  URI		: const QString &	: Namespace URI			*/
/*  localName	: const QString &	:				*/
/*  qName	: const QString &	: Element name			*/
/*  (returns)	: bool			: Success			*/

bool	KBSAXHandler::endElement
	(	const	QString		&,
		const	QString		&,
		const	QString		&
	)
{
//	fprintf
//	(	stderr,
//		"KBSAXHandler::endElement: [%s] [%p][%p][%p]\n",
//		(cchar *)qName,
//		(void  *)m_kbTOS,
//		(void  *)m_kbSlot,
//		(void  *)m_kbEvent
//	)	;

	if (m_kbTOS == 0)
	{	setErrMessage (TR("Internal error: "), TR("Empty parse stack")) ;
		return	false	 ;
	}

	/* Check for special states related to processing slots, macros	*/
	/* and events.							*/
	switch (m_state)
	{
		case InSlot	 :
			/* End of a slot. Tidy the code and return to	*/
			/* the normal state.				*/
			m_kbSlot->tidy ()	;
			m_kbSlot  = 0		;
			m_state	  = Normal	;
			return	  true 		;

		case InSlotLink	  :
			/* End of a slot link. Just pop back to the	*/
			/* main slot state.				*/
			m_state	  = InSlot	;
			return	  true		;

		case InSlotCode   :
			/* End of slot code. Just pop back to the main	*/
			/* slot state.					*/
			m_state	  = InSlot	;
			return	  true		;

		case InMacro	  :
			/* End of a macro. Store the instructions in	*/
			/* assocated event and return to the normal	*/
			/* state.					*/
			m_kbEvent->setMacro (m_kbMacro) ;
			m_kbMacro = 0		;
			m_kbEvent = 0		;
			m_state	  = Normal	;
			return	  true		;
		
		case InMacroInstr :
			/* End of a macro instruction. Append it to the	*/
			/* macro (checking for errors), then clean up	*/
			/* and return to the main macro state.		*/
			if (!m_kbMacro->append
				(	m_macroAction,
					m_macroArgs,
					m_macroComment,
					m_lError
				))
			{
				setErrMessage
				(	QString(TR("Invalid macro '%1' ignored")).arg(m_macroAction),
					""
				)	;
				return	false	;
			}

			m_macroAction  = QString::null ;
			m_macroComment = QString::null ;
			m_macroArgs.clear() 	;
			m_state	  = InMacro	;
			return	  true		;

		case InMacroArg	   :
			/* End of a macro argument. Add the accumulated	*/
			/* characters to the macro arguments and return	*/
			/* to the macro instruction state.		*/
			m_macroArgs.append (m_chars)	;
			m_chars   = QString::null	;
			m_state	  = InMacroInstr;
			return	  true 		;

		case InEvent	   :
			/* End of an event. Tidy up the stored text and	*/
			/* return to the normal state.			*/
			m_kbEvent->tidy ()	;
			m_kbEvent = 0		;
			m_state	  = Normal	;
			return	  true		;

		default	:
			break	;
	}


	m_kbTOS = m_kbTOS->endParse () ;
	return	true ;
}

/*  KBSAXHandler							*/
/*  processNode	: Process an XML node					*/
/*  qName	: const QString	&	 : Node name			*/
/*  aList	: const QDict<QString>	 : Attributes			*/
/*  nDict	: const QDict<NodeSpec> &: Node specifications		*/
/*  (returns)	: bool			 : Success			*/

bool	KBSAXHandler::processNode
	(	const QString		&qName,
		const QDict<QString>	&aList,
		const QDict<NodeSpec>	&nDict
	)
{
//	fprintf
//	(	stderr,
//		"KBSAXHandler::processNode: [%s]\n",
//		(cchar *)qName
//	)	;

	m_chars	     = QString::null ;


	/* SLOTS							*/
	/* -----							*/

	/* First there is some special processing to handle slots, 	*/
	/* which appear as child nodes of objects in the XML. The first	*/
	/* and main case is the "slot" element itself. This cannot be	*/
	/* a top-level element, cannot be nested, and must have a	*/
	/* "name" attribute.						*/
	if (qName == "slot")
	{
		if ((m_kbTOS == 0) || (m_kbTOS->isObject() == 0))
		{	setErrMessage (TR("Slot outside object"), "") ;
			return	false ;
		}

		const QString	*name	= aList.find("name") ;
		if (name == 0)
		{	setErrMessage (TR("Missing slot name"), "") ;
			return	false ;
		}

		if (m_state != Normal)
		{	setErrMessage (TR("Nested slot elements"), m_state) ;
			return	false ;
		}

		m_kbSlot = new KBSlot (m_kbTOS->isObject(), *name) ;
		m_state	 = InSlot ;
		return	 true	  ;
	}

	/* Next is a "slotlink" which specifies a linkage from the slot	*/
	/* to another target object. This must be inside a slot, and	*/
	/* must have all four attributes (link name, target name,	*/
	/* target enabled and enabled) specified.			*/
	if (qName == "slotlink")
	{
		if (m_state != InSlot)
		{	setErrMessage (TR("Slot link outside slot"), m_state) ;
			return	false ;
		}

		const QString	*name	= aList.find("name"   ) ;
		const QString	*target	= aList.find("target" ) ;
		const QString	*event	= aList.find("event"  ) ;
		const QString	*enabled= aList.find("enabled") ;

		if ((name == 0) || (target == 0) || (event == 0) || (enabled == 0))
		{	setErrMessage (TR("Malformed slot linkage"), "") ;
			return	false ;
		}

		m_kbSlot->addLinkage
		(	*name,
			*target,
			*event,
			enabled->toInt()
		)	;

		m_state = InSlotLink ;
		return	true	     ;
	}

	/* Lastly there is the code of the slot. The code appears as	*/
	/* embedded text, so provided that we are inside a slot, set	*/
	/* the in-slot-code flag.					*/
	if (qName == "slotcode")
	{
		if (m_state != InSlot)
		{	setErrMessage (TR("Slot code outside slot"), m_state) ;
			return	false ;
		}

		m_state	= InSlotCode  ;
		return	true	      ;
	}


	/* EVENTS							*/
	/* ------							*/

	/* Now similarly for nested event code. This cannot be a	*/
	/* top-level element, cannot be nested, and must have a "name"	*/
	/* attribute.							*/
	if (qName == "event")
	{
		if ((m_kbTOS == 0) || (m_kbTOS->isObject() == 0))
		{	setErrMessage (TR("Event outside object"), "") ;
			return	false ;
		}

		const QString	*name	= aList.find("name") ;
		if (name == 0)
		{	setErrMessage (TR("Missing event name"), "") ;
			return	false ;
		}

		if (m_kbEvent != 0)
		{	setErrMessage (TR("Nested event elements"), "") ;
			return	false ;
		}

		KBAttr	*attr = m_kbTOS->getAttr(*name) ;
		if ((attr == 0) || (attr->isEvent() == 0))
		{	setErrMessage (TR("No such event attribute"), "") ;
			return	false ;
		}

		m_kbEvent = attr->isEvent() ;
		m_state	  = InEvent	    ;
		return	true ;
	}

	/* MACROS							*/
	/* ------							*/

	if (qName == "macro")
	{
		if ((m_kbTOS == 0) || (m_kbTOS->isObject() == 0))
		{	setErrMessage (TR("Macro outside object"), "") ;
			return	false ;
		}

		const QString	*name	= aList.find("name") ;
		if (name == 0)
		{	setErrMessage (TR("Missing macro name"), "") ;
			return	false ;
		}

		if (m_kbMacro != 0)
		{	setErrMessage (TR("Nested macro elements"), "") ;
			return	false ;
		}

		KBAttr	*attr = m_kbTOS->getAttr(*name) ;
		if ((attr == 0) || (attr->isEvent() == 0))
		{	setErrMessage (TR("No such macro attribute"), "") ;
			return	false ;
		}

		m_kbEvent = attr->isEvent() ;
		m_kbMacro = new KBMacroExec (m_location.dbInfo, m_location.docLocn) ;
		m_state	  = InMacro 	    ;
		return	true ;
	}

	if (qName == "instruction")
	{
		if (m_state != InMacro)
		{	setErrMessage (TR("Macro code outside macro"), m_state) ;
			return	false ;
		}

		const QString	*action	= aList.find("action" ) ;
		const QString	*comment= aList.find("comment") ;

		if (action == 0)
		{	setErrMessage (TR("Unnamed macro code"), "") ;
			return	false ;
		}

		m_macroAction	= *action ;
		m_macroComment	= comment == 0 ? QString::null : *comment ;
		m_macroArgs.clear ()	  ;
		m_state	= InMacroInstr	  ;
		return	true ;
	}

	if (qName == "arg")
	{
		if (m_state != InMacroInstr)
		{	setErrMessage (TR("Macro argument outside macro"), m_state) ;
			return	false ;
		}

		m_state	= InMacroArg  ;
		return	true	      ;
	}


	/* Done the special stuff, now do the normal processing of	*/
	/* KBNode-derived objects.					*/
	NodeSpec *spec = nDict.find (qName) ;

	if (spec == 0)
	{
		setErrMessage (TR("Unknown XML element \"%1\""), qName) ;
		return	false	 ;
	}


	m_kbTOS = (*spec->nodeFunc) (m_kbTOS, aList, 0) ;
	m_kbTOS->startParse () ;

	if (m_kbTop == 0) m_kbTop = m_kbTOS ;
	return	true ;

}

/*  KBSAXHandler							*/
/*  characters	: Handle a block of characters				*/
/*  code	: const QString & : Characters				*/
/*  (returns)	: bool		  : Success				*/

bool	KBSAXHandler::characters
	(	const QString		&code
	)
{
	if (m_state == InSlotCode)
	{
//		fprintf	(stderr, "Add slot code [%s]\n", (cchar *)code) ;
		m_kbSlot ->setCode (code, true) ;
		return	true ;
	}

	if (m_state == InEvent)
	{
//		fprintf	(stderr, "Add event code [%s]\n", (cchar *)code) ;
		m_kbEvent->setCode (code, true) ;
		return	true ;
	}

	m_chars	+= code	;
	return	true	;
}

bool	KBSAXHandler::error
	(	const QXmlParseException &e
	)
{
	setErrMessage (e) ;
	return	true	;
}

bool	KBSAXHandler::warning
	(	const QXmlParseException &e
	)
{
	setErrMessage (e) ;
	return	true	;
}

bool	KBSAXHandler::fatalError
	(	const QXmlParseException &e
	)
{
	setErrMessage (e) ;
	return	false	 ;
}


/*  ------------------------------------------------------------------  */

/*  makeDesignMenu							*/
/*		: Insert node entries into popup design menu		*/
/*  nodeSpec	: QDict<NodeSpec>& : Set of possible node specificators	*/
/*  popup	: QPopupMenu *	   : Popup menu being constructed	*/
/*  receiver	: QObject *	   : Received for menu signals		*/
/*  flags	: uint		   : Inclusion flags			*/
/*  (returns)	: void		   :					*/

LIBKBASE_API
	void	makeDesignMenu
	(	QDict<NodeSpec>	&nodeSpec,
		QPopupMenu	*popup,
		QObject		*receiver,
		uint		flags
	)
{
	QDictIterator<NodeSpec> nsIter	(nodeSpec) ;
	NodeSpec		*nsItem	;

	QList<NodeSpec>		extra	;
	bool			sep	= false	;

	/* Non-data entries, for instance for buttons and labels, are	*/
	/* inserted first. We also skip any that do not have the	*/
	/* appropriate (typically KF_FORM or KF_REPORT) flag set.	*/
	for (nsIter.toFirst () ; (nsItem = nsIter.current()) != 0 ; nsIter += 1)
	{
		if ((nsItem->flags & KF_DATA ) != 0)
		{
			continue ;
		}

		/* Items flagged as extra are noted for inclusion later	*/
		/* in their own sub-menu.				*/
		if ((nsItem->flags & KF_EXTRA) != 0)
		{
			extra.append (nsItem) ;
			continue ;
		}

//		fprintf
//		(	stderr,
//			"makeDesignMenu: [%12s] [%p] [%08x:%08x]\n",
//			(cchar *)nsItem->msgText,
//			(void  *)nsItem->popupFunc,
//			nsItem->flags,
//			flags
//		)	;

		if ((nsItem->flags &   flags) == 0)
			continue ;

		/* Node spec defines its own callback function to build	*/
		/* the popup menu, so use that.				*/
		if (nsItem->popupFunc != 0)
		{
			(*nsItem->popupFunc) (popup, receiver) ;
			continue	;
		}

		/* If there is no text then this node is parsed but	*/
		/* does not have a popup menu item.			*/
		if (nsItem->msgText == 0)
			continue	;

		/* Node spec defines the menu text so insert straight	*/
		/* into the popup menu. Set the separator flag so that	*/
		/* a separator will appear between non-data and data	*/
		/* entries.						*/
		popup->insertItem
		(	TR(nsItem->msgText),
			receiver,
			SLOT (newNode(int)),
			0,
			(int)nsItem
		)	;

		sep	= true	;
	}

	/* Now scan for data items. This is essentially the same as the	*/
	/* loop above ...						*/
	for (nsIter.toFirst () ; (nsItem = nsIter.current()) != 0 ; nsIter += 1)
	{
		if ((nsItem->flags & KF_DATA ) == 0)
		{
			continue ;
		}

		if ((nsItem->flags & KF_EXTRA) != 0)
		{
			extra.append (nsItem) ;
			continue ;
		}

//		fprintf
//		(	stderr,
//			"makeDesignMenu: [%12s] [%p] [%08x:%08x]\n",
//			(cchar *)nsItem->msgText,
//			(void  *)nsItem->popupFunc,
//			nsItem->flags,
//			flags
//		)	;

		if ((nsItem->flags &   flags) == 0)
			continue ;

		if (nsItem->popupFunc != 0)
		{
			(*nsItem->popupFunc) (popup, receiver) ;
			continue ;
		}

		if (nsItem->msgText == 0)
			continue	;

		/* Add a separator into the menu before the first data	*/
		/* item added.						*/
		if (sep)
		{	popup->insertSeparator () ;
			sep	= false	;
		}

		popup->insertItem
		(	nsItem->msgText,
			receiver,
			SLOT (newNode(int)),
			0,
			(int)nsItem
		)	;
	}

	if (extra.count() > 0)
	{
		QPopupMenu *em = new QPopupMenu (popup) ;

		LITER
		(	NodeSpec,
			extra,
			exItem,

			em->insertItem
			(	exItem->msgText,
				receiver,
				SLOT (newNode(int)),
				0,
				(int)exItem
			)	;
		)

		popup->insertItem (TR("Extra"), em) ;
	}
}

