/***************************************************************************
    file	         : kb_qtpixmap.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<errno.h>

#include	<qapplication.h>
#include	<qlist.h>
#include	<qstring.h>
#include	<qpixmap.h>
#include	<qfile.h>
#include	<qfileinfo.h>
#include	<qdatastream.h>
#include	<qdragobject.h>
#include	<qurl.h>

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"

#include	"kb_pixmap.h"
#include	"kb_block.h"
#include	"kb_display.h"
#include	"kb_writer.h"
#include	"kb_qtpixmap.h"


extern	const	QPixmap	&scalePixmap	(const QPixmap &, const QRect &, uint) ;


/*  KBCtrlPixmap							*/
/*  KBCtrlPixmap: Constructor for image wrapper				*/
/*  display	: KBDisplay *	: Parent display			*/
/*  pmap	: KBPixmap *	: Parent pixmap				*/
/*  drow	: uint		: Row number in pixmap			*/
/*  (returns)	: KBCtrlPixmap	:					*/

KBCtrlPixmap::KBCtrlPixmap
	(	KBDisplay	*display,
		KBPixmap	*pmap,
		uint		drow
	)
	:
	QLabel	  (display->getDisplayWidget()),
	KBControl (this, display, pmap, drow),
	m_pmap	  (pmap)
{
	setAcceptDrops (true)	;
	redraw	  ()		;
}

/*  KBCtrlPixmap							*/
/*  ~KBCtrlPixmap: Desctructor for field class				*/
/*  (returns)	:		:					*/

KBCtrlPixmap::~KBCtrlPixmap ()
{
}

/*  KBCtrlPixmap								*/
/*  eventFilter	: Widget event filter					*/
/*  o		: QObject *	: Object originating event		*/
/*  e		: QEvent *	: Event in question			*/
/*  (returns)	: bool		: False to continue processing		*/

bool	KBCtrlPixmap::eventFilter
	(	QObject	*o,
		QEvent	*e
	)
{
	if (e->type() == QEvent::MouseButtonDblClick)
		if (m_showing == KB::ShowAsData)
		{
			m_pmap->contextMenu ((QMouseEvent *)e, m_drow) ;
			return	true ;
		}

	return	KBControl::eventFilter (o, e) ;
}

/*  KBCtrlPixmap								*/
/*  redraw	: Redraw to get profile and frame correct		*/
/*  (returns)	: bool		:					*/

void	KBCtrlPixmap::redraw ()
{
	ctrlSetFrame (this) ;
}

/*  KBCtrlPixmap							*/
/*  showName	: Show control name					*/
/*  (returns)	: void		:					*/

void	KBCtrlPixmap::showName ()
{
#if	! __KB_RUNTIME
	if ((m_drow == 0) && (m_showing == KB::ShowAsDesign))
		setText (m_pmap->getName()) ;
#endif
}

/*  KBCtrlPixmap							*/
/*  showAs	: Set or clear design mode				*/
/*  mode	: KB::ShowAs	: Design mode				*/
/*  (returns)	: void		:					*/

void	KBCtrlPixmap::showAs
	(	KB::ShowAs	mode
	)
{
	KBControl::showAs (mode) ;

#if	! __KB_RUNTIME
	if ((mode == KB::ShowAsDesign) && (m_drow == 0))
	{	showName () ;
		return ;
	}
#endif
	setText  ("") ;
}

/*  KBCtrlPixmap							*/
/*  setMonitor	: Set monitor display value				*/
/*  value	: const KBValue&: Value					*/
/*  (returns)	: void		:					*/

void	KBCtrlPixmap::setMonitor
	(	const KBValue &v
	)
{
	KBControl::setMonitor (v.isEmpty() ? TR("[Null Image]") : TR("[Image]")) ;
}

/*  KBCtrlPixmap							*/
/*  setValue	: Set field value					*/
/*  value	: const KBValue &: Value				*/
/*  fromDB	: bool		 : Set from database			*/
/*  (returns)	: void		 :					*/

void	KBCtrlPixmap::setValue
	(	const KBValue	&value
	)
{
	/* The value should contain an image (ie., it should be of	*/
	/* type KB::Binary). If not, then something strange will appear	*/
	/* First check is for empty (or null) data in which case the	*/
	/* image is simply cleared.					*/
	KBControl::setValue (value) ;
	if (value.isEmpty())
	{
		clear	() ;
		return	;
	}

	QPixmap	pm ;

	pm.loadFromData
	(	(const unsigned char *)value.dataPtr(),
		value.dataLength()
	)	;

	setPixmap 
	(	scalePixmap
		(	pm,
			m_pmap->geometry(),
			m_pmap->autoSize()
	)	)	;
}

/*  KBCtrlPixmap							*/
/*  getValue	: Get field value					*/
/*  (returns)	: const KBValue &: Value				*/

KBValue	KBCtrlPixmap::getValue ()
{
	return	KBValue (m_curVal, m_pmap->getFieldType()) ;
}

/*  KBCtrlPixmap							*/
/*  clearValue	: Clear value for specified query and display row	*/
/*  query	: bool		: Clear for query			*/
/*  (returns)	: void		:					*/

void	KBCtrlPixmap::clearValue
	(	bool
	)
{
//	fprintf
//	(	stderr,
//		"KBCtrlPixmap::clearValue:\n"
//	)	;

	m_curVal = KBValue() ;
	clear	 () ;
}

/*  KBCtrlPixmap							*/
/*  changed	: Test if field value has been changed			*/
/*  (returns)	: bool		: TRUE if changed			*/

bool	KBCtrlPixmap::changed ()
{
//	fprintf
//	(	stderr,
//		"KBCtrlPixmap::changed: [%d][%d] [%d]\n",
//		m_curVal.isEmpty   (),
//		m_curVal.dataLength(),
//		getIniValue().getRawText().isEmpty()
//	)	;

	/* Special case: This handles the situation where the inital	*/
	/* value is null, otherwise we get empty != null.		*/
	if (m_curVal.isEmpty() && getIniValue().getRawText().isEmpty())
		return false ;

	return	m_curVal != getIniValue() ;
}

/*  KBCtrlPixmap							*/
/*  isEmpty	: See if control is empty				*/
/*  (returns)	: bool		: Empty					*/

bool	KBCtrlPixmap::isEmpty ()
{
	return	m_curVal.isEmpty() ;
}

/*  KBCtrlPixmap							*/
/*  isValid	: See if control is valid				*/
/*  alowNull	: bool		: Ignore not-null check			*/
/*  (returns)	: bool		: Empty					*/

bool	KBCtrlPixmap::isValid
	(	bool
	)
{
	/* Always true. The checking is actually done when a new image	*/
	/* is loaded.							*/
	return	true	;
}

/*  KBCtrlPixmap							*/
/*  saveImage	: Save image from a file				*/
/*  path	: const QString & : Path to file			*/
/*  format	: const char *	  : Format in which to save		*/
/*  (returns)	: void		  :					*/

void	KBCtrlPixmap::saveImage
	(	const QString	&path,
		const char	*format
	)
{
	/* Open and read the file into a byte array, whence it can be	*/
	/* stored and loaded into the control.				*/
	QPixmap	*pm	= pixmap () ;

	if (pm == 0)
	{
		KBError::EError
		(	"Cannot retrieve image",
			QString::null,
			__ERRLOCN
		)	;
		return	;
	}

	if (!pm->save (path, format))
	{
		KBError::EError
		(	"Unable to save image in file",
			QString::null,
			__ERRLOCN
		)	;
		return	;
	}
}

/*  KBCtrlPixmap							*/
/*  loadImage	: Load image from a file				*/
/*  path	: const QString & : Path to file			*/
/*  (returns)	: void		  :					*/

void	KBCtrlPixmap::loadImage
	(	const QString	&path
	)
{
	KBValue	value	;

	/* Ignore this if we are read-only ...				*/
	if (m_pmap->isReadOnly())
		return	;

	/* ... or if we cannot start an update.				*/
	if (!m_pmap->startUpdate (m_pmap->getBlock()->getCurDRow() + m_drow))
		return	;

	if (!path.isNull())
	{
		/* Check that the file exists and is not a directory,	*/
		/* and that is will fit in the amount of space that is	*/
		/* available in the underlying database field.		*/
		QFileInfo info	(path)	;

		if (!info.exists ())
		{
			KBError::EError
			(	QString (TR("File \"%1\" does not exist"))
					.arg(path),
				QString::null,
				__ERRLOCN
			)	;
			return	;
		}
		if (info.isDir ())
		{
			KBError::EError
			(	QString (TR("\"%1\" is a directory"))
					.arg(path),
				QString::null,
				__ERRLOCN
			)	;
			return	;
		}
		if (info.size() > m_pmap->getFieldType()->getLength())
		{
			KBError::EError
			(
				QString (TR("Image \"%1\" is too large for database field"))
					.arg(path),
				QString (TR("Image size %1, field size %2"))
					.arg(info.size())
					.arg(m_pmap->getFieldType()->getLength()),
				__ERRLOCN
			)	;
			return	;
		}

		/* Open and read the file into a byte array, whence it	*/
		/* can be stored and loaded into the control.		*/
		QFile  imgFile (path) ;

		if (!imgFile.open (IO_ReadOnly))
		{
			KBError::EError
			(	QString  (TR("Cannot open \"%1\"")).arg(path),
				strerror (errno),
				__ERRLOCN
			)	;
			return	;
		}

		value	= KBValue (imgFile.readAll(), m_pmap->getFieldType()) ;
	}
	else	value	= KBValue () ;

	QPixmap	pixMap	;
	if (!value.isNull()) pixMap.loadFromData ((const unsigned char *)value.dataPtr(), value.dataLength()) ;
	setPixmap (pixMap) ;

	m_pmap->userChange    (m_pmap->getBlock()->getCurDRow() + m_drow) ;
	KBControl::setValue (value) ;
}

#if	! __KB_EMBEDDED

/*  KBCtrlPixmap							*/
/*  dragEnterEvent: Handle drag entering the widget			*/
/*  e		  : QDragEnterEvent * : The enter event			*/
/*  (returns)	  : void	      :					*/

void	KBCtrlPixmap::dragEnterEvent
	(	QDragEnterEvent *e
	)
{
	e->accept (QUriDrag::canDecode (e)) ;
}

/*  KBCtrlPixmap							*/
/*  dropEvent	: Handle dropping into the image			*/
/*  e		: QDropEvent *	: The event				*/
/*  (returns)	: void		:					*/

void	KBCtrlPixmap::dropEvent
	(	QDropEvent	*e
	)
{
	QStringList	files	;

	if (QUriDrag::decodeLocalFiles (e, files))
	{
		/* Clean up the URL. This should change if we make the	*/
		/* code network transparent.				*/
		QUrl	  url	(files[0]  )	;
		QString	  path	(url.path())	;
		int	  nlo	;
		if ((nlo = path.findRev ('\n')) >= 0) path.truncate (nlo) ;
		if ((nlo = path.findRev ('\r')) >= 0) path.truncate (nlo) ;

		loadImage (path) ;
		return	  ;
	}
}
#endif


/*  KBCtrlPixmap							*/
/*  giveFocus	: Give focus to the control				*/
/*  (returns)	: void							*/

void	KBCtrlPixmap::giveFocus ()
{
	QLabel::setFocus () ;
}

/*  KBCtrlPixmap							*/
/*  write	: Write field 						*/
/*  writer	: KBWriter *	: Output writer				*/
/*  rect	: QRect		: Field area				*/
/*  value	: KBValue &	: Value for reports			*/
/*  fSubs	: bool		: Substitution flag			*/
/*  extra	: int &		: Return extra space			*/
/*  (returns)	: bool		: Success				*/

bool	KBCtrlPixmap::write
	(	KBWriter	*writer,
		QRect		rect,
		const KBValue	&value,
		bool		fSubs,
		int		&extra
	)
{
	if (writer->asReport())
	{
		QPixmap	pm ;

		pm.loadFromData
		(	(const unsigned char *)value.dataPtr(),
			value.dataLength()
		)	;

		new KBWriterPixmap
		(	writer,
			rect,
			scalePixmap (pm, rect, m_pmap->autoSize())
		)	;

		extra	= 0	;
		return	true	;
	}

	return	KBControl::write (writer, rect, value, fSubs, extra) ;
}
