/***************************************************************************
    file	         : kb_select.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	"kb_classes.h"
#include	"kb_dblink.h"
#include	"kb_select.h"
#include	"kb_node.h"
#include	"kb_table.h"
#include	"kb_qrybase.h"
#include	"kb_queryset.h"
#include	"kb_qrylevel.h"




static	QString	tokenChars
	(	"abcdefghijklmnopqrstuvwxyz"
		"ABCDEFGHIJKLMNOPQRSTUVWXYZ"
		"0123456789"
		"_"
		"$"
		"[]"
	)	;

static	QString	wrapBold
	(	const QString	&text,
		bool		pretty
	)
{
	if (pretty)
		return	QString("<b>%1</b>").arg(text) ;

	return	text	;
}


/*  KBSelect								*/
/*  singleExpression						STATIC	*/
/*		: See if expression really is one expression		*/
/*  expr	: const QString & : Expression in question		*/
/*  (returns)	: bool		  : True if so				*/

bool	KBSelect::singleExpression
	(	const QString	&expr
	)
{
	uint	pos	= 0 	;
	int	nest	= 0	;
	bool	inquote	= false	;

	if (expr.stripWhiteSpace() == "*")
		return	false	;

	while (pos < expr.length())
	{
		QChar	ch = expr.at(pos) ;

		if (inquote)
		{
			if (ch == '\\')
			{
				pos += 2 ;
				continue ;
			}

			if (ch == '\'') inquote = false ;
			pos += 1 ;
			continue ;
		}

		if (ch == '\'')
		{
			pos	+= 1	;
			inquote	 = true ;
			continue ;
		}

		if (ch == '(')
		{
			pos	+= 1	;
			nest	+= 1	;
			continue	;
		}

		if (ch == ')')
		{
			pos	+= 1	;
			nest	-= 1	;
			continue	;
		}

		if ((ch == ',') && (nest <= 0))
			return	false	;

		pos += 1 ;
	}

	return	true	;
}

/*  ------------------------------------------------------------------  */

/*  KBSelectTable							*/
/*  KBSelectTable: Constuctor for select query table entry		*/
/*  (returns)	 : KBSelectTable :					*/

KBSelectTable::KBSelectTable ()
	:
	m_joinType	(None)
{
}

/*  KBSelectTable							*/
/*  KBSelectTable: Constuctor for select query table entry		*/
/*  tabName	 : const QString & : Table name				*/
/*  alias	 : const QString & : Table alias			*/
/*  primary	 : const QString & : Primary column			*/
/*  (returns)	 : KBSelectTable   :					*/

KBSelectTable::KBSelectTable
	(	const QString	&tabName,
		const QString	&alias,
		const QString	&primary
	)
	:
	m_tabName	(tabName),
	m_alias		(alias),
	m_joinType	(None),
	m_primary	(primary)
{
}

/*  KBSelectTable							*/
/*  KBSelectTable: Constuctor for select query table entry		*/
/*  tabName	 : const QString & : Table name				*/
/*  alias	 : const QString & : Table alias			*/
/*  joinType	 : JoinType	   : Join type				*/
/*  joinExpr	 : const QString & : Join expression			*/
/*  primary	 : const QString & : Primary column			*/
/*  (returns)	 : KBSelectTable   :					*/

KBSelectTable::KBSelectTable
	(	const QString	&tabName,
		const QString	&alias,
		JoinType	joinType,
		const QString	&joinExpr,
		const QString	&primary
	)
	:
	m_tabName	(tabName),
	m_alias		(alias),
	m_joinType	(joinType),
	m_joinExpr	(joinExpr),
	m_primary	(primary)
{
	if (m_joinExpr.isEmpty()) m_joinType = None ;
}

/*  KBSelectTable							*/
/*  KBSelectTable: Constuctor for select query table entry		*/
/*  tabName	 : const QString & : Table name				*/
/*  alias	 : const QString & : Table alias			*/
/*  joinType	 : JoinType	   : Join type				*/
/*  joinExpr	 : const QString & : Join expression			*/
/*  primary	 : const QString & : Primary column			*/
/*  (returns)	 : KBSelectTable   :					*/

KBSelectTable::KBSelectTable
	(	const QString	&tabName,
		const QString	&alias,
		const QString	&joinType,
		const QString	&joinExpr,
		const QString	&primary
	)
	:
	m_tabName	(tabName),
	m_alias		(alias),
	m_joinType	(joinType == "left"  ? LeftOuter  :
			 joinType == "right" ? RightOuter :
					       Inner 
			),
	m_joinExpr	(joinExpr),
	m_primary	(primary)
{
	if (m_joinExpr.isEmpty()) m_joinType = None ;
}

/*  KBSelectTable							*/
/*  print	 : Debug print						*/
/*  (returns)	 : void		:					*/

void	KBSelectTable::print ()
{
	fprintf
	(	stderr,
		"Table  [%s,%s,%s,%s,%s]\n",
		(cchar *)m_tabName,
		(cchar *)m_alias,
		(cchar *)m_joinType,
		(cchar *)m_joinExpr,
		(cchar *)m_primary
	) ;
}

/*  KBSelectTable							*/
/*  hasOuterJoin : See if table has an outer join			*/
/*  (returns)	 : bool		: True if so				*/

bool	KBSelectTable::hasOuterJoin ()
{
	return	(m_joinType == RightOuter) || (m_joinType == LeftOuter) ;
}

/*  KBSelectTable							*/
/*  hasInnerJoin : See if table has an inner join			*/
/*  (returns)	 : bool		: True if so				*/

bool	KBSelectTable::hasInnerJoin ()
{
	return	(m_joinType == Inner) ;
}

/*  KBSelectTable							*/
/*  hasAnyJoin	 : See if table has an any join				*/
/*  (returns)	 : bool		: True if so				*/

bool	KBSelectTable::hasAnyJoin ()
{
	return	(m_joinType != None) ;
}

/*  KBSelectTable							*/
/*  tableText	: Get text for table with possible alias		*/
/*  dbLink	: KBDBlink *	: Database link				*/ 
/*  (returns)	: QString	: Text					*/

QString	KBSelectTable::tableText
	(	KBDBLink	*dbLink
	)
{
	QString	tabName	= m_tabName	;

	if (dbLink != 0)
		tabName	= dbLink->mapExpression (tabName) ;

	if (!m_alias.isEmpty())
		return	QString("%1 %2").arg(tabName).arg(m_alias) ;

	return	tabName ;
}

/*  KBSelectTable							*/
/*  joinType	 : Get text for table join type				*/
/*  (returns)	 : QString	: Text					*/

QString	KBSelectTable::joinType ()
{
	switch (m_joinType)
	{
		case LeftOuter  : return "left outer join"  ;
		case RightOuter : return "right outer join" ;
		case Inner	: return "inner join"	    ;
		default	   	: break  ;
	}

	return	"unknown join" ;
}

/*  KBSelectTable							*/
/*  joinExpr	: Get text for table join expression			*/
/*  dbLink	: KBDBLink *	: Database connection			*/ 
/*  (returns)	: QString	: Text					*/

QString	KBSelectTable::joinExpr
	(	KBDBLink	*dbLink
	)
{
	return	dbLink == 0 ? m_joinExpr : dbLink->mapExpression (m_joinExpr) ;
}

KBTable	*KBSelectTable::makeTable
	(	KBNode		*parent
	)
{
	KBTable	*table	= new KBTable
			  (	parent,
				m_tabName,
				m_alias,
				m_primary,
				"", "", "", "", "", ""
			  )	;

	table->setPrimary (QString::null, KBTable::Auto) ;
	return	table	;
}



/*  ------------------------------------------------------------------  */

/*  KBSelectExpr							*/
/*  KBSelectExpr: Constructor for expression in select			*/
/*  (returns)	: KBSelectExpr	:					*/

KBSelectExpr::KBSelectExpr ()
{
}

/*  KBSelectExpr							*/
/*  KBSelectExpr: Constructor for expression in select			*/
/*  expr	: const QString & : Expression itself			*/
/*  alias	: const QString & : Possible alias			*/
/*  (returns)	: KBSelectExpr	  :					*/

KBSelectExpr::KBSelectExpr
	(	const QString	&expr,
		const QString	&alias
	)
	:
	m_expr	(expr),
	m_alias	(alias)
{
}

/*  KBSelectExpr							*/
/*  print	 : Debug print						*/
/*  (returns)	 : void		:					*/

void	KBSelectExpr::print ()
{
	fprintf
	(	stderr,
		"Expr   [%s,%s]\n",
		(cchar *)m_expr,
		(cchar *)m_alias
	)	;
}

/*  KBSelectExpr							*/
/*  exprText	: Get expression text					*/
/*  dbLink	: KBDBlink *	: Database link				*/ 
/*  (returns)	: QString	: Text					*/

QString	KBSelectExpr::exprText
	(	KBDBLink	*dbLink
	)
	const
{
	QString	expr	= m_expr	;

	if (dbLink != 0)
		expr	= dbLink->mapExpression (expr) ;

	if (!m_alias.isEmpty())
		return	QString("%1 as %2").arg(expr).arg(m_alias) ;

	return	expr	;
}


/*  ------------------------------------------------------------------  */

/*  KBSelect								*/
/*  KBSelect	: Constoctor for selct query object			*/
/*  (returns)	: KBSelect	:					*/

KBSelect::KBSelect ()
{
	m_distinct  = false ;
	m_forUpdate = false ;
	m_limit	    = 0	;
	m_offset    = 0 ;
}

/*  KBSelect								*/
/*  appendTable	: Add a table to the select				*/
/*  tabName	: const QString & : Table name				*/
/*  alias	: const QString & : Table alias				*/
/*  (returns)	: KBSelectTable   :					*/

void	KBSelect::appendTable
	(	const QString	&tabName,
		const QString	&alias
	)
{
	m_tableList.append
	(	KBSelectTable
		(	tabName,
			alias
	)	)	;
}

/*  KBSelect								*/
/*  appendTable	: Add a table to the select				*/
/*  tabName	: const QString & : Table name				*/
/*  alias	: const QString & : Table alias				*/
/*  joinType	: JoinType	  : Join type				*/
/*  joinExpr	: const QString & : Join expression			*/
/*  (returns)	: KBSelectTable   :					*/

void	KBSelect::appendTable
	(	const QString		&tabName,
		const QString		&alias,
		KBSelectTable::JoinType	joinType,
		const QString		&joinExpr
	)
{
	m_tableList.append
	(	KBSelectTable
		(	tabName,
			alias,
			joinType,
			joinExpr
	)	)	;
}

/*  KBSelect								*/
/*  appendTable	: Add a table to the select				*/
/*  tabName	: const QString & : Table name				*/
/*  alias	: const QString & : Table alias				*/
/*  joinType	: const QString & : Join type				*/
/*  joinExpr	: const QString & : Join expression			*/
/*  (returns)	: KBSelectTable   :					*/

void	KBSelect::appendTable
	(	const QString	&tabName,
		const QString	&alias,
		const QString	&joinType,
		const QString	&joinExpr
	)
{
	m_tableList.append
	(	KBSelectTable
		(	tabName,
			alias,
			joinType,
			joinExpr
	)	)	;
}

/*  KBSelect								*/
/*  appendExpr	: Add an expression to the select			*/
/*  expr	: const QString & : Expression				*/
/*  alias	: const QString & : Possible alias			*/
/*  (returns)	: void		  :					*/

void	KBSelect::appendExpr
	(	const QString	&expr,
		const QString	&alias
	)
{
	m_exprList.append (KBSelectExpr (expr, alias)) ;
}

/*  KBSelect								*/
/*  appendGroup	: Add a group expression to the select			*/
/*  group	: const QString & : Expression				*/
/*  (returns)	: void		  :					*/

void	KBSelect::appendGroup
	(	const QString	&group
	)
{
	m_groupList.append (group) ;
}

/*  KBSelect								*/
/*  appendHaving: Add a having expression to the select			*/
/*  having	: const QString & : Expression				*/
/*  (returns)	: void		  :					*/

void	KBSelect::appendHaving
	(	const QString	&having
	)
{
	m_havingList.append (having) ;
}

/*  KBSelect								*/
/*  appendWhere	: Add a where expression to the select			*/
/*  where	: const QString & : Expression				*/
/*  (returns)	: void		  :					*/

void	KBSelect::appendWhere
	(	const QString	&where
	)
{
	m_whereList.append (where) ;
}

/*  KBSelect								*/
/*  appendOrder	: Add a order expression to the select			*/
/*  order	: const QString & : Expression				*/
/*  (returns)	: void		  :					*/

void	KBSelect::appendOrder
	(	const QString	&order
	)
{
	m_orderList.append (order) ;
}

/*  KBSelect								*/
/*  getQueryText: Get text of the query					*/
/*  pretty	: bool		: Return in pretty rich-text format	*/
/*  ellipsis	: bool		: Add ellipsis in pretty formt		*/
/*  dbLink	: KBDBLink *	: Database link				*/
/*  (returns)	: QString	: Text					*/

QString	KBSelect::getQueryText
	(	bool		pretty,
		bool		ellipsis,
		KBDBLink	*dbLink
	)
{
	QString	SQL	;
	QString	sep	;
	bool	hasOuterJoin	= false	;

	for (uint idx1 = 0 ; idx1 < m_tableList.count() ; idx1 += 1)
		if (m_tableList[idx1].hasOuterJoin())
		{	hasOuterJoin = true ;
			break	;
		}

	SQL	= wrapBold ("select", pretty) ;
	if (m_distinct) SQL += wrapBold (" distinct", pretty) ;
	sep	= " "	;

	for (uint idx2 = 0 ; idx2 < m_exprList.count() ; idx2 += 1)
	{	SQL	+= sep	;
		SQL	+= m_exprList[idx2].exprText(dbLink) ;
		sep	 = ", "	;
	}

	if (pretty)
	{	if (ellipsis) SQL += "..." ;
		SQL	+= "<blockquote> " ;
	}

	SQL    += wrapBold (" from", pretty) ;
	sep	= " "	  ;

	if (hasOuterJoin)
	{
		for (uint idx3 = 0 ; idx3 < m_tableList.count() ; idx3 += 1)
			if (!m_tableList[idx3].hasAnyJoin())
			{	SQL	+= sep  ;
				SQL	+= m_tableList[idx3].tableText(dbLink) ;
				sep	 = ", " ;
			}

		if (pretty) SQL += "<br/>" ;

		for (uint idx4 = 0 ; idx4 < m_tableList.count() ; idx4 += 1)
			if (m_tableList[idx4].hasAnyJoin())
			{	SQL	+= " "  ;
				SQL	+= wrapBold (m_tableList[idx4].joinType(), pretty) ;
				SQL	+= " "  ;
				SQL	+= m_tableList[idx4].tableText(dbLink) ;
				SQL	+= wrapBold (" on ", pretty) ;
				SQL	+= m_tableList[idx4].joinExpr (dbLink) ;
				if (pretty) SQL += "<br/>" ;
			}

		sep	= wrapBold(" where ", pretty) ;
	}
	else
	{
		for (uint idx5 = 0 ; idx5 < m_tableList.count() ; idx5 += 1)
		{	SQL	+= sep  ;
			SQL	+= m_tableList[idx5].tableText(dbLink) ;
			sep	 = ", " ;
		}
		if (pretty) SQL += "<br/>" ;

		sep	= wrapBold(" where ", pretty) ;
		for (uint idx6 = 0 ; idx6 < m_tableList.count() ; idx6 += 1)
			if (!m_tableList[idx6].joinExpr(0).isEmpty())
			{	SQL	+= sep  ;
				SQL	+= m_tableList[idx6].joinExpr(dbLink) ;
				sep	 = wrapBold(" and ", pretty) ;
				if (pretty) SQL += "<br/>" ;
			}

	}

	for (uint idx7 = 0 ; idx7 < m_whereList.count() ; idx7 += 1)
	{	SQL	+= sep ;
		SQL	+= m_whereList[idx7].exprText(dbLink) ;
		sep	 = wrapBold	 (" and ", pretty) ;
		if (pretty) SQL += "<br/>" ;
	}

	sep	= wrapBold(" group by ", pretty) ;
	for (uint idx8 = 0 ; idx8 < m_groupList.count() ; idx8 += 1)
	{	SQL	+= sep ;
		SQL	+= m_groupList[idx8].exprText(dbLink) ;
		sep	 = ", " ;
	}
	if (pretty && (m_groupList.count() > 0)) SQL += "<br/>" ;

	sep	= wrapBold(" having ", pretty) ;
	for (uint idx9 = 0 ; idx9 < m_havingList.count() ; idx9 += 1)
	{	SQL	+= sep ;
		SQL	+= m_havingList[idx9].exprText(dbLink) ;
		sep	 = ", " ;
	}
	if (pretty && (m_havingList.count() > 0)) SQL += "<br/>" ;

	sep	= wrapBold(" order by ", pretty) ;
	for (uint idx10 = 0 ; idx10 < m_orderList.count() ; idx10 += 1)
	{	SQL	+= sep ;
		SQL	+= m_orderList[idx10].exprText(dbLink) ;
		sep	 = ", " ;
	}
	if (pretty && (m_orderList.count() > 0)) SQL += "<br/>" ;

	if (m_forUpdate)
		SQL	+= wrapBold (" for update", pretty) ;

	if (m_limit > 0)
	{
		if	(pretty)
		{
			SQL	+= wrapBold (" limit ", pretty) + QString::number(m_limit) ;
		}
		else if ((dbLink != 0) && (dbLink->server() != 0))
		{
			QString	text	;
			if (dbLink->server()->getSyntax (text, KBServer::Limit, m_limit, 0))
				SQL	+= text	;
			else	dbLink->server()->lastError().DISPLAY() ;
		}
		else	SQL	+= QString(" limit %1 offset %2").arg(m_limit).arg(0) ;
	}

	if (pretty)
		SQL	+= "</blockquote> " ;

	return	SQL	;
}

/*  KBSelect								*/
/*  nextToken	: Get next token from SQL query string			*/
/*  (returns)	: bool		: Token found				*/

bool	KBSelect::nextToken ()
{
	bool	inQuote	= false	;
	m_token	= QString::null	;

	/* Loop iterates over the string. The "inQuote" flag is set if	*/
	/* we are in a quoted string.					*/
	while (m_buffPtr < m_buffer.length())
	{
		QChar	ch = m_buffer.at(m_buffPtr) ;

		if (inQuote)
		{
			/* In a quoted string. The next character is	*/
			/* appended to the toke whatever, then end the	*/
			/* token if we hit a further quote.		*/
			m_token.append (ch) ;
			m_buffPtr += 1 ;

			if (ch == '\'') break ;

			/* Assume that the string escape character is	*/
			/* backslash. If this brings us to the end of	*/
			/* the buffer then silently drop out ....	*/
			if (ch == '\\')
			{
				if (m_buffPtr >= m_buffer.length()) break ;

				/* ... otherwise decide how many	*/
				/* characters to gobble. This is a bit	*/
				/* simplistic, but will probably do.	*/
				QChar	ch	= m_buffer.at(m_buffPtr) ;
				uint	gobble	= 1 ;

				if	(ch.isDigit()) gobble = 3 ;
				else if	(ch == 'x'   ) gobble = 3 ;
				else if	(ch == 'X'   ) gobble = 3 ;

				while ((gobble > 0) && (m_buffPtr < m_buffer.length()))
				{	m_token.append (m_buffer.at(m_buffPtr)) ;
					m_buffPtr += 1 ;
					gobble	  -= 1 ;
				}
			}

			continue ;
		}

		/* Next check for single quote. This terminates the	*/
		/* current token, or starts a new quoted token.		*/
		if (ch == '\'')
		{
			if (!m_token.isEmpty()) break ;
			m_token.append (ch) ;
			m_buffPtr += 1	  ;
			inQuote	   = true ;
			continue  ;
		}

		/* Check for characters which can be part of a multi-	*/
		/* character token. These are appended to the current	*/
		/* token.						*/
		if (tokenChars.find (ch) >= 0)
		{
			m_token.append (ch) ;
			m_buffPtr += 1	  ;
			continue	  ;
		}

		/* Non-whitespace characters are now treated as tokens	*/
		/* in their own right .....				*/
		if (!ch.isSpace())
		{
			if (m_token.isEmpty())
			{	m_token.append (ch) ;
				m_buffPtr += 1 ;
			}
			break	;
		}

		/* Whitespace terminates the current token if there is	*/
		/* one, otherwise we will just skip over it.		*/
		if (!m_token.isEmpty()) break ;

		m_buffPtr += 1 ;
	}

	/* See if the token is a keyword; if so then convert it to	*/
	/* lower case for convenience.					*/
	if (isKeyword())
		m_token = m_token.lower() ;

	/* Gather up any following whitespace so that we will be able	*/
	/* to reconstruct expressions properly. The only whitespace	*/
	/* that should be lost is at the very start of a query.		*/
	m_white	= "" ;		;
	while ((m_buffPtr < m_buffer.length()) && m_buffer.at(m_buffPtr).isSpace())
	{
		m_white	  += m_buffer.at(m_buffPtr) ;
		m_buffPtr += 1 ;
	}

	return	!m_token.isEmpty () ;
}

/*  KBSelect								*/
/*  isKeyword	: Check if token is a keyword				*/
/*  (returns)	: bool		  : True if keyword			*/

bool	KBSelect::isKeyword ()
{
	static	cchar	*keyList[] =
	{
		"select",
		"from",
		"join",
		"left",
		"right",
		"inner",
		"outer",
		"on",
		"where",
		"and",
		"order",
		"by",
		"asc",
		"desc",
		"group",
		"having",
		"and",
		"distinct",
		"limit",
		"offset",
		0
	}	;
	static	QDict<void>	keyDict	;

	if (keyDict.count() == 0)
		for (cchar **keyPtr = &keyList[0] ; *keyPtr != 0 ; keyPtr += 1)
			keyDict.insert (*keyPtr, (void *)1) ;

	return	keyDict.find (m_token.lower()) != 0 ;
}

void	KBSelect::setParseError
	(	const QString	&error
	)
{
	m_error	= KBError
		  (	KBError::Error,
			TR("Error parsing SQL query"),
			error,
			__ERRLOCN
		  )	;
}

/*  KBSelect								*/
/*  parseExpr	: Parse out an expression				*/
/*  orderOK	: bool		: Ordering expression			*/
/*  allowAnd	: bool		: Treat "and" as part of expression	*/
/*  (returns)	: QString	: Expression or null if none		*/

QString	KBSelect::parseExpr
	(	bool	orderOK,
		bool	allowAnd
	)
{
	QString	expr	;
	int	paren	= 0 ;

	/* Parse until we hit a comma or a keyword (but possibly accept	*/
	/* "asc", "desc" or "and". Everything up to that point is	*/
	/* accumulated in the expression.				*/
	while (!m_token.isEmpty())
	{
//		fprintf
//		(	stderr,
//			"KBSelect::parseExpr: token=[%s] p=%d\n",
//			(cchar *)m_token,
//			paren
//		)	;

		if (m_token == "(") paren += 1 ;
		if (m_token == ")") paren -= 1 ;

		if (paren == 0)
		{
			if (m_token == ",")
				break ;

			if (isKeyword())
			{
				if	((m_token == "asc") || (m_token == "desc"))
				{
					if (orderOK) nextToken () ;
					break ;
				}
				else if (m_token == "and")
				{
					if (!allowAnd) break ;
				}
				else	break	;
			}
		}

		expr += m_token + m_white ;
		nextToken ()  ;
	}

//	fprintf	(stderr, "KBSelect::parseExpr: expr =[%s]\n", (cchar *)expr) ;
	return	expr	;
}

/*  KBSelect								*/
/*  parseExprList: Parse an expression list				*/
/*  eList	 : QValueList<KBSelectExpr> &				*/
/*				 : Return list				*/	
/*  sep		 : cchar *	 : Separator				*/
/*  order	 : bool		 : Ordering expression			*/
/*  (returns)	 : void		 :					*/

void	KBSelect::parseExprList
	(	QValueList<KBSelectExpr> &eList,
		cchar			 *sep,
		bool			 order
	)
{
	for (;;)
	{
		QString expr = parseExpr (order, false) ;
		if (expr.isEmpty()) break   ;

		eList.append (expr) ;

		if (m_token != sep) break ;
		nextToken() ;
	}
}

/*  KBSelect								*/
/*  parseTableList							*/
/*		: Parse list of tables					*/
/*  dbLink	: KBDBLink *	: Database connection			*/
/*  (returns)	: bool		: Success				*/

bool	KBSelect::parseTableList
	(	KBDBLink	*dbLink
	)
{
	/* The main loop iterates over the table entries. There should	*/
	/* be table or table/alias pairs, separated by commas or by	*/
	/* join conditions. Actually, this will let some rubbish like	*/
	/* "select ... from inner join ..." through but thats OK.	*/

	while (!m_token.isEmpty()) 
	{
		QString	table	;
		QString	alias	;
		QString	jtype	;
		QString	jexpr	;
		bool	join	= false	;

		/* If the next word is a keywork then it must be the	*/
		/* start of a join condition ...			*/
		if (isKeyword())
		{
			if	((m_token == "left") || (m_token == "right"))
			{
				join	= true    ;
				jtype	= m_token ;

				nextToken () ;
				if (m_token != "outer")
				{	setParseError ("Expected 'outer'") ;
					return	false ;
				}
				nextToken () ;
				if (m_token != "join")
				{	setParseError ("Expected 'join'") ;
					return	false ;
				}

				nextToken () ;
			}
			else if (m_token == "inner")
			{
				join	= true    ;
				jtype	= "inner" ;

				nextToken () ;
				if (m_token != "join")
				{	setParseError ("Expected 'join'") ;
					return	false ;
				}

				nextToken () ;
			}
			else	break	;
		}

		/* Must now have a table name, possibly followed by a	*/
		/* table alias. We then make sure that we can find a	*/
		/* unique key column on the table.			*/
		table	= m_token	;

		if (nextToken ())
			if ((m_token != ",") && !isKeyword())
			{	alias	= m_token ;
				nextToken ()	;
			}


		KBTableSpec tabSpec (table) ;

		if (dbLink != 0)
		{
			if (!dbLink->listFields (tabSpec))
			{
				m_error	= dbLink->lastError() ;
				return	  false	  ;
			}

			if (tabSpec.m_prefKey < 0)
			{
				m_error	= KBError
					  (	KBError::Error,
						TR("Unable to determine unique key column"),
						QString("Table: %1").arg(table),
						__ERRLOCN
					  )	;
				return	false	;
			}
		}

		/* If we started with a join then the join condition	*/
		/* should follow the keyword "on"; "and" is allowed as	*/
		/* part of the expression.				*/
		if (join)
		{
			if (m_token != "on")
			{	setParseError ("Expected 'on'") ;
				return	false ;
			}
			nextToken () ;

			jexpr = parseExpr (false, true) ;
			if (jexpr.isEmpty())
			{	setParseError ("Expected join condition") ;
				return	false ;
			}
		}

		m_tableList.append
		(	KBSelectTable
			(	table,
				alias,
				jtype,
				jexpr,
				tabSpec.m_prefKey >= 0 ?
					tabSpec.m_fldList.at(tabSpec.m_prefKey)->m_name :
					QString::null
			)
		)	;

		if (m_token == ",") nextToken () ;
	}

	if (m_tableList.count() == 0)
	{	setParseError (TR("Error parsing list of tables")) ;
		return	false ;
	}

	return	true ;
}

/*  KBSelect								*/
/*  parseQuery	: Parse a select query					*/
/*  query	: const QString & : Query text				*/
/*  dbInfo	: KBDBInfo *	  : Database information		*/
/*  (returns)	: bool		  : Success				*/

bool	KBSelect::parseQuery
	(	const QString	&query,
		KBDBLink	*dbLink
	)
{
	fprintf
	(	stderr,
		"KBSelect::parseQuery [%s] [%p]\n",
		(cchar *)query,
		(void  *)dbLink
	)	;

	reset	() ;

	/* Prime the parser by getting the first token, and verify that	*/
	/* it is the keyword "select" ...				*/
	m_buffer	= query ;
	m_buffPtr	= 0	;
	if (!nextToken   ())
	{
		setParseError (TR("Query is empty")) ;
		return	false ;
	}

	if (m_token.lower() != "select")
	{
		setParseError (TR("Query does not start with 'select'")) ;
		return	false ;
	}

	/* ... which should be followed by a comma-separated list of	*/
	/* expressions and thence the keyword "from". This may be	*/
	/* prefixed with the work "distinct".				*/
	nextToken () ;

	if (m_token.lower() == "distinct")
	{	m_distinct = true  ;
		nextToken () ;
	}
	else	m_distinct = false ;


	parseExprList (m_exprList, ",", false) ;

	if (m_token.lower() != "from")
	{
		setParseError (TR("Expected 'from' in query")) ;
		return	false ;
	}

	/* The list of tables should follow. This may either be comma	*/
	/* separated, a series of joins, or a combination.		*/
	nextToken () ;
	if (!parseTableList (dbLink))
	{
		return	false ;
	}

	if (m_token.lower() == "where")
	{
		nextToken () ;
		parseExprList (m_whereList, "and", false) ;
	}

	if (m_token.lower() == "group")
	{
		nextToken () ;
		if (m_token.lower() != "by")
		{
			setParseError (TR("Expected 'by' after 'group'")) ;
			return	false ;
		}

		nextToken () ;
		parseExprList (m_groupList, ",", false) ;
	}

	if (m_token.lower() == "having")
	{
		nextToken () ;
		parseExprList (m_havingList, ",", false) ;
	}

	if (m_token.lower() == "order")
	{
		nextToken () ;
		if (m_token.lower() != "by")
		{
			setParseError (TR("Expected 'by' after 'order'")) ;
			return	false ;
		}

		nextToken () ;
		parseExprList (m_orderList, ",", true) ;
	}


//	for (uint idx0 = 0 ; idx0 < m_tableList .count() ; idx0 += 1)
//		m_tableList [idx0].print() ;
//
//	for (uint idx1 = 0 ; idx1 < m_exprList  .count() ; idx1 += 1)
//		m_exprList  [idx1].print() ;
//
//	for (uint idx2 = 0 ; idx2 < m_whereList .count() ; idx2 += 1)
//		m_whereList [idx2].print() ;
//
//	for (uint idx3 = 0 ; idx3 < m_orderList .count() ; idx3 += 1)
//		m_orderList [idx3].print() ;
//
//	for (uint idx4 = 0 ; idx4 < m_groupList .count() ; idx4 += 1)
//		m_groupList [idx4].print() ;
//
//	for (uint idx5 = 0 ; idx5 < m_havingList.count() ; idx5 += 1)
//		m_havingList[idx5].print() ;

	/* See if we have a limit or limit/offset pair. The "limit"	*/
	/* case is extended to check for commas, to handle the MySQL	*/
	/* "limit M,N" syntax.						*/
	if (m_token.lower() == "limit")
	{
		nextToken () ;
		m_limit	= m_token.toInt() ;
		nextToken () ;

		if (m_token == ",")
		{	nextToken() ;
			m_limit	= m_token.toInt() ;
			nextToken () ;
		}
	}

	if (m_token.lower() == "offset")
	{
		nextToken () ;
		// m_offset = m_token.toInt() ;
		nextToken () ;
	}

	if (!m_token.isNull())
	{
		setParseError
		(	TR(QString("Unexpected token '%1'").arg(m_token))
		)	;
		return	false	;
	}

	return	true	;
}

/*  KBSelect								*/
/*  parseExprList							*/
/*		: Parse an expression list				*/
/*  exprList	: const QString & : Query text				*/
/*  dbInfo	: KBDBInfo *	  : Database information		*/
/*  (returns)	: bool		  : Success				*/

bool	KBSelect::parseExprList
	(	const QString	&exprList,
		KBDBLink	*dbLink
	)
{
	fprintf
	(	stderr,
		"KBSelect::parseExprList [%s] [%p]\n",
		(cchar *)exprList,
		(void  *)dbLink
	)	;

	reset	() ;

	/* Prime the parser by getting the first token ...		*/
	m_buffer	= exprList ;
	m_buffPtr	= 0	;
	if (!nextToken ())
	{
		setParseError (TR("Expression list is empty")) ;
		return	false ;
	}


	/* ... which should be followed by a comma-separated list of	*/
	/* expressions and This may be prefixed with the work		*/
	/* "distinct".							*/
	if (m_token.lower() == "distinct")
	{	m_distinct = true  ;
		nextToken () ;
	}
	else	m_distinct = false ;


	parseExprList (m_exprList, ",", false) ;

	if (!m_token.isEmpty())
	{
		setParseError
		(	QString(TR("Unexpected '%1' in expression list")).arg(m_token)
		)	;
		return	false ;
	}

	return	true	;
}

/*  KBSelect								*/
/*  getQueryText: Get text of the query					*/
/*  dbLink	: KBDBLink *	: Database link				*/
/*  (returns)	: QString	: Raw query text			*/

QString	KBSelect::getQueryText
	(	KBDBLink	*dbLink
	)
{
	return	getQueryText	(false, false, dbLink) ;
}

/*  KBSelect								*/
/*  getPrettyText: Get prettified text of the query			*/
/*  ellipsis	 : bool		: Add ellipsis after expression list	*/
/*  dbLink	: KBDBLink *	: Database link				*/
/*  (returns)	 : QString	: Prettified rich-text query		*/

QString	KBSelect::getPrettyText
	(	bool		ellipsis,
		KBDBLink	*dbLink
	)
{
	return	getQueryText	(true, ellipsis, dbLink) ;
}

/*  KBSelect								*/
/*  reset	: Reset to empty state					*/
/*  (returns)	: void		:					*/

void	KBSelect::reset ()
{
	m_tableList .clear ()	;
	m_exprList  .clear ()	;
	m_whereList .clear ()	;
	m_groupList .clear ()	;
	m_havingList.clear ()	;
	m_orderList .clear ()	;
}

/*  KBSelect								*/
/*  makeQryLevel: Make query level object for query			*/
/*  qryBase	: KBQryBase  *	  : Parent object for tables		*/
/*  dbLink	: KBDBLink   *	  : Database connection			*/
/*  topTable	: const QSTring & : Top table				*/
/*  qryTable	: KBTable    *&	  : Return topmost table		*/
/*  (returns)	: KBQryLevel *	  : New query level			*/

KBQryLevel
	*KBSelect::makeQryLevel
	(	KBQryBase	*qryBase,
		KBDBLink	&dbLink,
		const QString	&topTable,
		KBTable		*&qryTable
	)
{
	KBTable	*rootTable	;

	qryTable = rootTable = m_tableList[0].makeTable  (qryBase) ;

	for (uint idx = 1 ; idx < m_tableList.count() ; idx += 1)
	{
		KBTable *table	= m_tableList[idx].makeTable (rootTable) ;

		table->setJType (m_tableList[idx].joinType()) ;
		table->setJExpr (m_tableList[idx].joinExpr(&dbLink)) ;

		if (m_tableList[idx].tableName() == topTable)
			qryTable = table ;
	}
	
	KBQryLevel *qryLevel = new KBQryLevel
			       (	qryBase->getParent(),
					0,
					dbLink,
					0,
					rootTable,
					qryTable
			       )	;

	qryLevel->setDistinct  (m_distinct) ;

	QString	where	;
	QString	order	;
	QString	group	;
	QString	having	;

	for (uint idx2 = 0 ; idx2 < m_whereList .count() ; idx2 += 1)
	{
		if (idx2 > 0) where += " and " ;
		where	+= m_whereList [idx2].exprText(&dbLink) ;
	}

	for (uint idx3 = 0 ; idx3 < m_orderList .count() ; idx3 += 1)
	{
		if (idx3 > 0) where += ", " ;
		m_orderList [idx3].exprText(&dbLink) ;
	}

	for (uint idx4 = 0 ; idx4 < m_groupList .count() ; idx4 += 1)
	{
		if (idx4 > 0) where += ", " ;
		m_groupList [idx4].exprText(&dbLink) ;
	}

	for (uint idx5 = 0 ; idx5 < m_havingList.count() ; idx5 += 1)
	{
		if (idx5 > 0) where += ", " ;
		m_havingList[idx5].exprText(&dbLink) ;
	}

	qryLevel->setGlobalWhere  (where ) ;
	qryLevel->setGlobalOrder  (order ) ;
	qryLevel->setGlobalGroup  (group ) ;
	qryLevel->setGlobalHaving (having) ;

	return	qryLevel ;
}

/*  KBSelect								*/
/*  getComment	: Get brief descriptive comment				*/
/*  (returns)	: QString	: Comment				*/

QString	KBSelect::getComment ()
{
	switch (m_tableList.count())
	{
		case 0	:
			return	"SQL"	;

		case 1	:
			return	QString("SQL: %1").arg(m_tableList[0].tableText(0)) ;

		default	:	
			break	;
	}

	return	QString("SQL: %1, ...").arg(m_tableList[0].tableText(0)) ;
}

QStringList
	KBSelect::tableList ()
{
	QStringList tables ;

	for (uint idx = 0 ; idx < m_tableList.count() ; idx += 1)
		tables.append (m_tableList[idx].tableName()) ;

	return	tables ;
}

