/***************************************************************************
    file	         : kb_copytable.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<qdom.h>

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_location.h"

#include	"kb_database.h"
#include	"kb_dbinfo.h"
#include	"kb_dblink.h"

#include	"kb_param.h"
#include	"kb_select.h"

#include	"kb_copybase.h"
#include	"kb_copytable.h"



void	KBCopyCompare::addKey
	(	const QString	&key
	)
{
	m_count	+= 1 ;
	if (m_count < 32) m_keys.append (key) ;
}

QString	KBCopyCompare::keys ()
{
	QString	res = m_keys.join(",") ;
	if (m_count > m_keys.count()) res += ",...." ;
	return	res ;
}


/*  KBCopyTable								*/
/*  KBCopyTable	: Constructor for table copier				*/
/*  srce	: bool		: Source, else destination		*/
/*  location	: KBLocation &	: Database location			*/
/*  (returns)	: KBCopyTable	:					*/

KBCopyTable::KBCopyTable
	(	bool		srce,
		KBLocation	&location
	)
	:
	m_srce	   (srce),
	m_location (location)
{
	m_select	= 0	;
	m_insert	= 0	;
	m_update	= 0	;
	m_count		= 0	;
	m_compare	= 0	;
	m_copies	= 0	;
	m_qExec		= false	;
}


/*  KBCopyTable								*/
/*  ~KBCopyTable: Destructor for table copier				*/
/*  (returns)	:		:					*/

KBCopyTable::~KBCopyTable ()
{
	LITER
	(	KBType,
		m_types,
		type,
		type->deref()
	)

	DELOBJ	(m_select ) ;
	DELOBJ	(m_insert ) ;
	DELOBJ	(m_update ) ;
	DELOBJ	(m_count  ) ;
	DELOBJ	(m_compare) ;

	if (m_copies != 0) delete [] m_copies ;
}

/*  KBCopyTable								*/
/*  tag		: Get tag for XML output				*/
/*  (returns)	: cchar *	: Tag					*/

cchar	*KBCopyTable::tag ()
{
	return	"table"	;
}

/*  KBCopyTable								*/
/*  set		: Set copier from definition				*/
/*  copy	: QDomElement & : Definition parent			*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBCopyTable::set
	(	QDomElement	&copy,
		KBError		&
	)
{
	QDomElement	element	= copy.namedItem(tag()).toElement() ;

	if (element.isNull ()) return true ;
//	{
//		pError	= KBError
//			  (	KBError::Error,
//				QString("Document lacks %1 table part").arg(srce ? "source" : "destination"),
//				QString::null,
//				__ERRLOCN
//			  )	;
//		return	false	;
//	}

	reset	()	;

	setServer   (element.attribute ("server"  )) ;
	setTable    (element.attribute ("table"   )) ;
	setWhere    (element.attribute ("where"   )) ;
	setOrder    (element.attribute ("order"   )) ;
	setOption   (element.attribute ("option"  ).toInt(),
		     element.attribute ("optfield")) ;

	QDomNodeList	dFields	= element.elementsByTagName ("field") ;
	for (uint idx = 0 ; idx < dFields.length() ; idx += 1)
		m_fields.append (dFields.item(idx).toElement().attribute("name")) ;

	return	true	;
}

/*  KBCopyTable								*/
/*  valid	: Check validity					*/
/*  error	: KBError &	: Error return				*/
/*  (returns)	: bool		: Valid					*/

bool	KBCopyTable::valid
	(	KBError		&error
	)
{
	if (m_server.isEmpty())
	{
		error	= KBError
			  (	KBError::Error,
				TR("Server not set in table copier"),
				QString::null,
				__ERRLOCN
			  )	;
		return	false	;
	}
	if (m_table .isEmpty())
	{
		error	= KBError
			  (	KBError::Error,
				TR("Table not set in table copier"),
				QString::null,
				__ERRLOCN
			  )	;
		return	false	;
	}
	if (m_fields.count() == 0)
	{
		error	= KBError
			  (	KBError::Error,
				TR("No fields set in table copier"),
				QString::null,
				__ERRLOCN
			  )	;
		return	false	;
	}

	if ((m_option == OptUpdate) || (m_option == OptUpdIns))
	{
		m_uIndex	= 999999 ;
		for (uint idx = 0 ; idx < m_fields.count() ; idx += 1)
			if (*m_fields.at(idx) == m_optField)
			{	m_uIndex = idx ;
				break	 ;
			}

		if (m_uIndex == 999999)
		{
			error	= KBError
				  (	KBError::Error,
					TR("Update field is not in list of fields"),
					QString::null,
					__ERRLOCN
				  )	;
			return	false	;
		}
	}

	return	true	;
}

/*  KBCopyTable								*/
/*  def		: Get table copy definition				*/
/*  copy	: QDomElement &	: Element to which to attach		*/
/*  (returns)	: void		:					*/

void	KBCopyTable::def
	(	QDomElement	&copy
	)
{
	QDomElement	element	;

	copy   .appendChild  (element = copy.ownerDocument().createElement (tag())) ;

	element.setAttribute ("server",   m_server  ) ;
	element.setAttribute ("table",    m_table   ) ;
	element.setAttribute ("where",    m_where   ) ;
	element.setAttribute ("order",    m_order   ) ;
	element.setAttribute ("option",   m_option  ) ;
	element.setAttribute ("optfield", m_optField) ;

	for (uint idx = 0 ; idx < m_fields.count() ; idx += 1)
	{
		QDomElement	dField	;
		element.appendChild  (dField = element.ownerDocument().createElement ("field")) ;
		dField.setAttribute ("name", *m_fields.at(idx)) ;
	}
}

/*  KBCopyTable								*/
/*  getColumnNames							*/
/*		: Get column names					*/
/*  names	: QStringList &	: List of column names			*/
/*  (returns)	: void		:					*/

void	KBCopyTable::getColumnNames
	(	QStringList	&names
	)
{
	names	= m_fields ;
}

/*  KBCopyTable								*/
/*  prepare	: Prepare for execution					*/
/*  paramDict	: const QDict<QString> &				*/
/*				: Paramater dictionary			*/
/*  other	: KBCopyBase *	: Other half of copier			*/
/*  (returns)	: bool		: Success				*/

bool	KBCopyTable::prepare
	(	const QDict<QString>	&paramDict,
		KBCopyBase		*other
	)
{
	m_dbLink.disconnect () ;
	DELOBJ	(m_select ) ;
	DELOBJ	(m_insert ) ;
	DELOBJ	(m_update ) ;
	DELOBJ	(m_count  ) ;
	DELOBJ	(m_compare) ;

	m_useWhere = paramSub (m_where, paramDict) ;
	m_useOrder = paramSub (m_order, paramDict) ;

	/* Connect to the database server. If this fails then grab the	*/
	/* error and return false.					*/
	if (!m_dbLink.connect (m_location, m_server))
	{	m_lError = m_dbLink.lastError () ;
		return	 false	;
	}

	/* If this is the source then just build a select for all the	*/
	/* requested fields.						*/
	if (m_srce)
	{
		KBSelect query	;

		query.appendTable (m_table, QString::null) ;

		for (uint idx = 0 ; idx < m_fields.count() ; idx += 1)
			query.appendExpr  (*m_fields.at(idx)) ;

		if (!m_useWhere.isEmpty())
			query.appendWhere (m_useWhere) ;
		if (!m_useOrder.isEmpty())
			query.appendOrder (m_useOrder) ;

//		QString	query	= "select" ;
//		cchar	*sep	= " "	   ;
//
//		for (uint idx = 0 ; idx < fields.count() ; idx += 1)
//		{
//			query	+= sep	  ;
//			query	+= *fields.at(idx) ;
//			sep	 = ", "	  ;
//		}
//
//		query	+= " from " + table ;
//
//		if (!useWhere.isEmpty())
//			query	+= " where "    + useWhere ;
//		if (!useOrder.isEmpty())
//			query	+= " order by " + useOrder ;
//
		if ((m_select = m_dbLink.qrySelect (true, query.getQueryText (&m_dbLink))) == 0)
		{	m_lError = m_dbLink.lastError () ;
			return	 false	;
		}

		m_qExec	= false	;
		return	true	;
	}

	/* Insert is a little more complicated, since we need to deal	*/
	/* with the table primary key if the user has not included it.	*/
	/* Also, we need to get the types associated with each column,	*/
	/* plus <Auto> columns. First scan to see if we need to handle	*/
	/* the primary key.						*/
	KBTableSpec	tabSpec (m_table) ;
	if (!m_dbLink.listFields (tabSpec))
	{	m_lError = m_dbLink.lastError () ;
		return	 false	;

	}

	QStringList	srceNames	  ;
	other->getColumnNames (srceNames) ;

	KBType		*pkType	= 0	  ;

	if (tabSpec.m_prefKey >= 0)
	{
		/* Table has a primary key. If the primary key is	*/
		/* serial (eg., MySQL auto_increment) then we need not	*/
		/* handle it since either (a) the user has included it	*/
		/* or (b) it will be generated in the server. If not	*/
		/* serial then see if the user has included it.		*/
		KBFieldSpec *pSpec = tabSpec.m_fldList.at(tabSpec.m_prefKey) ;

		m_pkName = pSpec->m_name ;

		if ((pSpec->m_flags & KBFieldSpec::Serial) == 0)
		{
			m_getPKey = true	;
			pkType	= KBType::typeToType (pSpec->m_typeIntl) ;

			for (uint idx = 0 ; idx < m_fields.count() ; idx += 1)
				if (*m_fields.at(idx) == m_pkName)
				{	m_getPKey	= false ;
					pkType	= 0	;
					break	;
				}
		}
		else	m_getPKey = false ;
	}
	else	/* No primary key so we must assume that the user does	*/
		/* the right thing (tm).				*/
		m_getPKey = false	;


	/* Build up queries for insert, update and select (for data	*/
	/* comparisoms). Build all three at one go since the loop is	*/
	/* common and its not worth testing at this level which are	*/
	/* needed.							*/
	QString	qryins	= "insert into " + m_dbLink.mapExpression(m_table) + " (" ;
	QString	qryupd	= "update " 	 + m_dbLink.mapExpression(m_table) + " set " ;
	QString qrycmp	= "select "	 ;

	cchar	*sep	= ""	   ;

	for (uint idx1 = 0 ; idx1 < m_fields.count() ; idx1 += 1)
	{
		QString	fName = *m_fields.at(idx1) ;

		/* If the name is "<Auto>" then get the name from the	*/
		/* source ....						*/
		if (fName == "<Auto>") fName = srceNames[idx1] ;

		
		KBFieldSpec   *fSpec = tabSpec.findField (fName) ;

		if (fSpec == 0)
		{
			m_lError = KBError
				   (	KBError::Error,
					TR("Destination column not found"),
					QString(TR("Table: %1, column: %2"))
						.arg(m_table)
						.arg(fName),
					__ERRLOCN
				   )	;
			return	 false	;
		}

		KBType	*type	= fSpec->m_dbType ;
		if (type == 0) type = KBType::typeToType (fSpec->m_typeIntl) ;

		m_types.append (type) ;
		type->ref () ;

		qryins	+= sep	 ;
		qryins	+= m_dbLink.mapExpression(fName) ;
		qryupd	+= sep	 ;
		qryupd	+= m_dbLink.mapExpression(fName) + " = " +  + m_dbLink.placeHolder (idx1) ;
		qrycmp  += sep	 ;
		qrycmp	+= m_dbLink.mapExpression(fName) ;
		sep	 = ", "	 ;
	}

	qryupd	+= " where " + 
		   m_dbLink.mapExpression(m_optField) +
		   " = "     +
		   m_dbLink.placeHolder  (m_fields.count()) ;

	qrycmp	+= " from "  +
		   m_dbLink.mapExpression(m_table) +
		   " where " + 
		   m_dbLink.mapExpression(m_optField) +
		   " = "     +
		   m_dbLink.placeHolder  (0) ;

	if (m_getPKey)
	{
		m_types.append (pkType)	;
		qryins	+= sep		;
		qryins	+= m_dbLink.mapExpression(m_pkName)	;
		sep	 = ", "		;
	}

	qryins	+= ") values ("	  ;
	sep	 = "" ;
 
	for (uint idx2 = 0 ; idx2 < m_fields.count() ; idx2 += 1)
	{
		qryins	+= sep	  ;
		qryins	+= m_dbLink.placeHolder (idx2) ;
		sep	 = ", "	  ;
	}

	if (m_getPKey) qryins += ", " + m_dbLink.placeHolder (m_fields.count()) ;

	qryins	+= ")" ;

	/* Append							*/
	/* Replace							*/
	/* Update/Insert						*/
	/* Insert new							*/
	/*	These cases all need (or may need) an insert query	*/
	if
	(	(m_option == OptAppend) || (m_option == OptReplace) ||
		(m_option == OptUpdIns) || (m_option == OptInsNew )
	)
		if ((m_insert = m_dbLink.qryInsert (true, qryins, m_table)) == 0)
		{	m_lError = m_dbLink.lastError ()  ;
			return	 false	;
		}

	/*  Update							*/
	/*  Update/Insert						*/
	/*	These two require an update query			*/
	if ((m_option == OptUpdate) || (m_option == OptUpdIns))
		if ((m_update = m_dbLink.qryUpdate (true, qryupd, m_table)) == 0)
		{	m_lError = m_dbLink.lastError () ;
			return	 false	;
		}

	/* Insert new							*/
	/*	Construct a query used to find out if the record exists	*/
	if (m_option == OptInsNew)
		if ((m_count = m_dbLink.qrySelect
				(	true,
					QString	("select count(*) from %1 where %2 = %3")
						.arg(m_dbLink.mapExpression(m_table   ))
						.arg(m_dbLink.mapExpression(m_optField))
						.arg(m_dbLink.placeHolder  (0         ))
				)) == 0)
		{	m_lError = m_dbLink.lastError () ;
			return	 false	;
		}

	/*  Compare							*/
	/*	Construct a query used to retrieve the extant records	*/
	if (m_option == OptCompare)
		if ((m_compare = m_dbLink.qrySelect (true, qrycmp)) == 0)
		{	m_lError = m_dbLink.lastError () ;
			return	 false	;
		}


	m_copies = new KBValue [m_fields.count() + 1] ;

	m_rDeleted	= 0	;
	m_rUpdated	= 0	;
	m_rInserted	= 0	;

	m_rSame    .clear ()	;
	m_rDiffer  .clear ()	;
	m_rMissing .clear ()	;
	m_rMultiple.clear ()	;

	return	 true	;
}

/*  KBCopyTable								*/
/*  getNumCols	: Get number of columns in copy				*/
/*  (returns)	: int		: Number of columns			*/

int	KBCopyTable::getNumCols ()
{
	return	m_fields.count ()	;
}

/*  KBCopyTable								*/
/*  getRow	: Get next row						*/
/*  values	: KBValue *	: Value vector				*/
/*  nCols	: uint		: Number of columns available		*/
/*  ok		: bool &	: Execute OK				*/
/*  (returns)	: int		: Actual number of columns		*/

int	KBCopyTable::getRow
	(	KBValue		*values,
		uint		,
		bool		&ok
	)
{
	/* Check that this is a source copier, if not then there is an	*/
	/* implementation error.					*/
	if (!m_srce)
	{
		m_lError = KBError
			   (	KBError::Fault,
				TR("Attempt to fetch row from destination copier"),
				QString::null,
				__ERRLOCN
			   )		;
		ok	 = false	;
		return	-1 ;
	}

	/* See if the select query has been executed, if not then do so	*/
	/* now and reset the current row number. If there is an error	*/
	/* then return with "ok" set false (error occurred) and a false	*/
	/* result (no row available).					*/
	if (!m_qExec)
	{
		if (!m_select->execute (0, 0))
		{	m_lError = m_select->lastError () ;
			ok	 = false ;
			return	 -1 ;
		}

		m_qExec	= true	;
		m_sRow	= 0	;
	}

	/* If the row number has exceeded then number of rows available	*/
	/* then all rows have been retrieved. The "ok" value is set	*/
	/* true as there is no error, but return false to indicate the	*/
	/* end of the data.						*/
	if (!m_select->rowExists (m_sRow, true))
	{
		ok	= true	;
		return	-1	;
	}

	for (uint sCol = 0; sCol < m_select->getNumFields() ; sCol += 1)
		values[sCol] = m_select->getField (m_sRow, sCol) ;

	m_sRow += 1	;
	ok	= true	;
	return	m_select->getNumFields() ;
}

/*  KBCopyTable								*/
/*  putRow	: Put a row of data					*/
/*  values	: KBValue *	: Value vector				*/
/*  aCols	: uint		: Actual number of columns		*/
/*  (returns)	: bool		: Success				*/

bool	KBCopyTable::putRow
	(	KBValue		*values,
		uint		
	)
{
	/* Check that this is a destination copier. An error here is an	*/
	/* implementation problem.					*/
	if (m_srce)
	{
		m_lError = KBError
			   (	KBError::Fault,
				TR("Attempt to insert row into source copier"),
				QString::null,
				__ERRLOCN
			   )	;
		return	 false	;
	}

	/* If this is the first call and the option is to replace then	*/
	/* now is the time to do a delete.				*/
	if (values == 0)
	{
		if (m_option == OptReplace)
		{
			KBSQLDelete *qryDel = m_dbLink.qryDelete (true, "delete from " + m_table, m_table) ;
			if (qryDel == 0)
			{	m_lError = m_dbLink.lastError() ;
				return	 false	;
			}

			if (!qryDel->execute (0, 0))
			{	m_lError = qryDel->lastError() ;
				delete	 qryDel	;
				return	 false	;
			}

			m_rDeleted	= qryDel->getNumRows () ;
			delete	qryDel	;
			m_qExec	= true	;
		}

		return	true	;
	}

//	for (uint idx = 0 ; idx < fields.count() ; idx += 1)
//	{	if (idx > 0) fprintf (stderr, "|") ;
//		fprintf	(stderr, "%s", (cchar *)values[idx].getRawText()) ;
//	}
//	fprintf	(stderr, "\n") ;

	/* Comparisom is not really a put at all! Retrieve the 		*/
	/* corresponding row from the (nominal) destination, check that	*/
	/* we get just one record back, and compare values piecemeal.	*/
	if (m_option == OptCompare)
	{
		if (!m_compare->execute (1, &values[m_uIndex]))
		{
			m_lError = m_update->lastError () ;
			return	 false	;
		}

		if (!m_compare->rowExists(0))
		{
			m_rMissing .addKey (values[m_uIndex].getRawText()) ;
			return	true	;
		}

		if ( m_compare->rowExists(1))
		{
			m_rMultiple.addKey (values[m_uIndex].getRawText()) ;
			return	true	;
		}

		bool	same = true ;

		for (uint idx = 0 ; idx < m_fields.count() ; idx += 1)
		{
			/* Retrieve the current and target values. We	*/
			/* consider two null values to be the same.	*/
			KBValue  &v1	= values[idx] ;
			KBValue   v2	= m_compare->getField(0, idx) ;
 
			if (v1.isNull() && v2.isNull()) continue ;

			/* OK, there are various cases so start by	*/
			/* by picking up the raw text and the types,	*/
			/* and test which are numeric an any form.	*/
			QString	  s1	= v1.getRawText() ;
			QString	  s2	= v2.getRawText() ;

			KB::IType t1	= v1.getType()->getIType() ;
			KB::IType t2	= v2.getType()->getIType() ;

			bool	  f1	= (t1 == KB::ITFloat) || (t1 == KB::ITDecimal) ;
			bool	  f2	= (t2 == KB::ITFloat) || (t2 == KB::ITDecimal) ;

			/* Hack: If either value is floating then	*/
			/* normalise to two decimal places ..... hmmm	*/
			if (f1 || f2)
			{
				QString	tmp ;
				tmp.sprintf ("%.2f", s1.toDouble()) ; s1 = tmp	  ;
				tmp.sprintf ("%.2f", s2.toDouble()) ; s2 = tmp	  ;
			}

//			fprintf
//			(	stderr,
//				"KBCopyTable::putRow: %s [%s][%s] [%s][%s]\n",
//				(cchar *)m_fields[idx],
//				(cchar *)v1.getRawText(),
//				(cchar *)v2.getRawText(),
//				(cchar *)s1,
//				(cchar *)s2
//			)	;

			if (s1 != s2)
			{
				fprintf
				(	stderr,
					"KBCopyTable::putRow: differ: %s k=[%s] [%s][%s] (%d,%d)\n",
					(cchar *)values[m_uIndex].getRawText(),
					(cchar *)m_fields[idx],
					(cchar *)v1.getRawText(),
					(cchar *)v2.getRawText(),
					t1,
					t2
				)	;
				same	= false	;
			}
		}

		if (same)
			m_rSame  .addKey (values[m_uIndex].getRawText()) ;
		else	m_rDiffer.addKey (values[m_uIndex].getRawText()) ;
		return	true	;
	}


	/* If inserting only new records, ie., those that do not already	*/
	/* exist, then see if the key already exists in the target. if it	*/
	/* does then ignore this record. Otherwise, drop through; this will	*/
	/* then go through to the insert at the bottom of this method.		*/
	if (m_option == OptInsNew)
	{
		if (!m_count->execute (1, &values[m_uIndex]))
		{
			m_lError = m_update->lastError () ;
			return	 false	;
		}

		m_count->rowExists (0) ;
		KBValue	res	= m_count->getField (0, 0) ;
		if (res.getRawText().toInt() > 0)
		{
			fprintf
			(	stderr,
				"KBCopyTable::putRow: row exists for key [%s]\n",
				(cchar *)res.getRawText()
			)	;
			return	true	;
		}
	}


	/* If updating or updating/inserting then do the update query.	*/
	/* We finish if either this updates at least one row or if the	*/
	/* option is just update.					*/
	if ((m_option == OptUpdate) || (m_option == OptUpdIns))
	{
		values[m_fields.count()] = values[m_uIndex] ;

		if (!m_update->execute (m_fields.count() + 1, values))
		{
			m_lError = m_update->lastError () ;
			return	 false	;
		}

		m_rUpdated	+= m_update->getNumRows() ;

		if ((m_update->getNumRows() > 0) || (m_option == OptUpdate))
			return	true	;
	}

	/* Drop through for append and replace options, plus the	*/
	/* update/insert option where no rows were updated.		*/


	/* Copy the values to a local array, setting the types for each	*/
	/* to the types we determined from the table earlier. Then add	*/
	/* the primary key value to this is required.			*/
	for (uint idx = 0 ; idx < m_fields.count() ; idx += 1)
		m_copies[idx] = KBValue(values[idx], m_types.at(idx)) ;

	if (m_getPKey)
		if (!m_insert->getNewKey (m_pkName, m_copies[m_fields.count()], true))
		{
			m_lError = m_insert->lastError () ;
			return	 false	;
		}

	if (!m_insert->execute (m_fields.count() + (m_getPKey ? 1 : 0), m_copies))
	{
		m_lError = m_insert->lastError () ;
		return	 false	;
	}

	m_rInserted	+= 1	;
	return	true	;
}

/*  KBCopyTable								*/
/*  finish	: Finish execution					*/
/*  report	: QString &	: Completion report			*/
/*  (returns)	: bool		: Success				*/

bool	KBCopyTable::finish
	(	QString		&report
	)
{
	QStringList	res	;
	report		= "" ;

	if (m_option == OptCompare)
	{
		res.append (QString(TR("%1 rows the same: %2")).arg(m_rSame    .count()).arg(m_rSame    .keys())) ;
		res.append (QString(TR("%1 rows differ: %2"  )).arg(m_rDiffer  .count()).arg(m_rDiffer  .keys())) ;
		res.append (QString(TR("%1 rows missing: %2" )).arg(m_rMissing .count()).arg(m_rMissing .keys())) ;
		res.append (QString(TR("%1 rows multiple: %2")).arg(m_rMultiple.count()).arg(m_rMultiple.keys())) ;
	}
	else
	{
		if (m_rDeleted > 0)
			res.append (QString(TR("%1 rows deleted" )).arg(m_rDeleted )) ;
		if (m_rUpdated > 0)
			res.append (QString(TR("%1 rows updated" )).arg(m_rUpdated )) ;
		if (m_rInserted > 0)
			res.append (QString(TR("%1 rows inserted")).arg(m_rInserted)) ;
	}

	m_dbLink.disconnect () ;
	DELOBJ	(m_select ) ;
	DELOBJ	(m_insert ) ;
	DELOBJ	(m_update ) ;
	DELOBJ	(m_count  ) ;
	DELOBJ	(m_compare) ;

	report	= res.join ("\n") ;
	return	true	;
}

/*  KBCopyTable								*/
/*  (returns)	: void		:					*/

void	KBCopyTable::reset ()
{
	m_server = QString::null;
	m_table	 = QString::null;
	m_qExec	 = false	;

	m_fields.clear ()	;

	DELOBJ	(m_select ) 	;
	DELOBJ	(m_insert ) 	;
	DELOBJ	(m_update )	;
	DELOBJ	(m_count  )	;
	DELOBJ	(m_compare)	;
	m_dbLink.disconnect ()	;
}

/*  KBCopyTable								*/
/*  setServer	: Set server name					*/
/*  _server	: const QString & : Server name				*/
/*  (returns)	: void		  :					*/

void	KBCopyTable::setServer
	(	const QString	&_server
	)
{
	m_server = _server	;
}

/*  KBCopyTable								*/
/*  setTable	: Set table name					*/
/*  _table	: const QString & : Table name				*/
/*  (returns)	: void		  :					*/

void	KBCopyTable::setTable
	(	const QString	&_table
	)
{
	m_table	= _table	;
}

/*  KBCopyTable								*/
/*  addField	: Add field name					*/
/*  _field	: const QString & : Field name				*/
/*  (returns)	: void		  :					*/

void	KBCopyTable::addField
	(	const QString	&_field
	)
{
	m_fields.append (_field)	;
}

/*  KBCopyTable								*/
/*  setWhere	: Set where expression					*/
/*  _where	: const QString & : Expression				*/
/*  (returns)	: void		  :					*/

void	KBCopyTable::setWhere
	(	const QString	&_where
	)
{
	m_where	= _where	;
}

/*  KBCopyTable								*/
/*  setOrder	: Set order expression					*/
/*  _order	: const QString & : Expression				*/
/*  (returns)	: void		  :					*/

void	KBCopyTable::setOrder
	(	const QString	&_order
	)
{
	m_order	= _order	;
}

/*  KBCopyTable								*/
/*  setOption	: Set destination option				*/
/*  _option	: uint		  : Option				*/
/*  _optField	: const QString & : Associated field			*/
/*  (returns)	: void		  :					*/

void	KBCopyTable::setOption
	(	uint		_option,
		const QString	&_optField
	)
{
	m_option   = _option	;
	m_optField = _optField	;
}

