/***************************************************************************
    file	         : kb_dictionary.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<errno.h>

#include	<qdict.h>
#include	<qfile.h>
#include	<qdir.h>
#include	<qmessagebox.h>


#include	"kb_classes.h"
#include	"kb_dictionary.h"
#include	"kb_locator.h"
#include	"kb_dom.h"

#if		__KB_KDE

#include	<klocale.h>
#include	<kglobal.h>

#endif		// __KB_KDE

/*  KBDictionary:							*/
/*  lookup	: Get dictionary entry for element/attribute		*/
/*  element	: const QString & : Element name			*/
/*  attrib	: const QString & : Attribute name			*/
/*  (returns)	: KBDictEntry *	  : Entry				*/

KBDictEntry
	*KBDictionary::lookup
	(	const QString	&element,
		const QString	&attrib
	)
{
	QString		tag	;
	KBDictEntry	*entry	;

	/* We try "element.attrib" and "*.attrib" in that order. If	*/
	/* neither is present, then return null.			*/
	tag = QString("%1.%2").arg(element).arg(attrib)  ;
	if ((entry = m_dict.find (tag)) != 0) return entry ;

	tag = QString("%1.%2").arg(    "*").arg(attrib)  ;
	if ((entry = m_dict.find (tag)) != 0) return entry ;

	fprintf
	(	stderr, "=====> No attribute dictionary for [%s][%s]\n",
		(cchar *)element,
		(cchar *)attrib
	)	;
	return	0 ;
}

QString	KBDictionary::getAttrLegend
	(	const QString	&element,
		const QString	&attrib
	)
{
	KBDictEntry *entry = lookup(element, attrib) ;
	return	entry 	   !=   0 ? entry->m_legend :
		element[0] == '_' ? QString::null   :
				    QString("%1.%2").arg(element).arg(attrib) ;
}

QString	KBDictionary::getAttrDescription
	(	const QString	&element,
		const QString	&attrib
	)
{
	KBDictEntry *entry = lookup(element, attrib) ;
	return	entry      !=   0 ? "<qt>" + entry->m_descrip + "</qt>" :
		element[0] == '_' ? QString::null :
				    QString("%1.%2").arg(element).arg(attrib) ;
}

QString	KBDictionary::getAttrNullcheck
	(	const QString	&element,
		const QString	&attrib
	)
{
	KBDictEntry *entry = lookup(element, attrib) ;
	if ((entry == 0) || entry->m_nullchk.isNull())
		return	QString("%1 has not been set").arg(attrib) ;

	return	entry->m_nullchk ;
}

QString	KBDictionary::getAttrExtra
	(	const QString	&element,
		const QString	&attrib,
		const QString	&key
	)
{
	KBDictEntry *entry = lookup(element, attrib) ;
	if ((entry == 0) || (entry->m_extra == 0))
		return QString::null ;

	QString	*res	= entry->m_extra->find(key) ;
	if (res == 0) return QString::null ;

	return	*res	;
}

QList<KBDictEntry>
	KBDictionary::getMatching
	(	const QString	&element,
		const QString	&attrib
	)
{
	QList	     <KBDictEntry> matched	 ;
	QDictIterator<KBDictEntry> iter (m_dict) ;
	KBDictEntry		   *entry	 ;

	QString	tag	= QString("%1.%2").arg(element).arg(attrib) ;

	while ((entry = iter.current()) != 0)
	{
		if (iter.currentKey().find(tag) == 0)
			matched.append (entry) ;
		iter	+= 1 ;
	}

	return	matched	;
}
	
/*  KBDictionary							*/
/*  KBDictionary: Dictionary constructor				*/
/*  dName	: QString	: Dictionary name			*/
/*  (returns)	: KBDictionary	:					*/

KBDictionary::KBDictionary
	(	QString		dName
	)
{
	QString	dPath	;
	QDir	dDir	;

#ifndef _WIN32
	dPath	= locateDir ("appdata", QString("dict/%1.dict") .arg(dName)) ;
	dPath  += "/dict";
#else
	dPath	= locateDir ("appdata", QString("dict\\%1.dict").arg(dName)) ;
	dPath  += "\\dict";
#endif

	fprintf
	(	stderr,
		"KBDictionary::KBDictionary: [%s] -> [%s]\n",
		(cchar *)dName,
		(cchar *)dPath
	)	;

	dDir.setPath	   (dPath) ;
	dDir.setNameFilter (QString("%1*.dict").arg(dName)) ;
	dDir.setFilter     (QDir::Files|QDir::Readable) ;
	dDir.setSorting    (QDir::Name) ;

	const QFileInfoList   *info = dDir.entryInfoList() ;
	if (info == 0)
	{
		QMessageBox::warning(0, "Location Error", "No dictionary directory found!!\n");
		return	;
	}

	QFileInfoListIterator dIter (*info) ;
	QFileInfo *fi ;

#if	__KB_KDE
	KLocale	*locale	= KGlobal::locale () ;
	fprintf
	(	stderr,
		"KBDictionary::KBDictionary: [%s] language [%s]\n",
		(cchar *)dName,
		(cchar *)locale->language()
	)	;
#endif

	while ((fi = dIter.current()) != 0)
	{
		QString	name	= fi->filePath() ; 
		loadFile (name)	;

#if	__KB_KDE
		QString	name2	= fi->dirPath () + "/" + locale->language() + "/" + fi->fileName() ;
		if (QFileInfo(name2).exists()) loadFile (name2) ;
#endif
		dIter += 1 ;
	}

	/* Final operation is to scan the dictionary and set unset	*/
	/* legends and descriptions to the entry tag. This means that	*/
	/* something shows in the property dialogs when the dictionary	*/
	/* is missing an entry.						*/
	QDictIterator<KBDictEntry> eIter(m_dict) ;
	KBDictEntry *entry ;

	while ((entry = eIter.current()) != 0)
	{
		if (entry->m_legend .isNull()) entry->m_legend  = eIter.currentKey() ;
		if (entry->m_descrip.isNull()) entry->m_descrip = eIter.currentKey() ;
		eIter += 1 ;
	}

	m_dict.setAutoDelete (true) ;
}

/*  KBDictionary							*/
/*  loadFile	: Load a dictionary file				*/
/*  name	: const QString & : File name				*/
/*  (returns)	: void		  :					*/

void	KBDictionary::loadFile
	(	const QString	&name
	)
{
	fprintf
	(	stderr,
		"KBDictionary::loadFile: [%s]\n",
		(cchar *)name
	)	;

	QFile	dFile	(name) ;

	if (!dFile.open (IO_ReadOnly))
	{
		KBError::EError
		(	"Cannot open dictionary",
			QString("%1: %2").arg(name).arg(strerror(errno)),
			__ERRLOCN
		)	;
		return	;
	}

	QDomDocument	dXML	;
	if (!dXML.setContent (&dFile))
	{
		KBError::EError
		(	"Cannot parse dictionary",
			name,
			__ERRLOCN
		)	;
		return	;
	}

	QDomNodeList	dNodes	= dXML.elementsByTagName ("attribute") ;

	for (uint idx = 0 ; idx < dNodes.length() ; idx += 1)
	{
		QDomNode	dAttr	 = dNodes.item  (idx) ;
		QDomElement	dElement = dAttr.toElement () ;

		if (dElement.isNull()) continue ;

		QString		kElement = dElement.attribute ("element") ;
		QString		kName	 = dElement.attribute ("name"   ) ;

		if (kElement.isEmpty() || kName.isEmpty()) continue ;

		QString		kLegend	 ;
		QString		kDescrip ;
		QString		kNullchk ;

		/* See if the entry already exists, if not then create	*/
		/* it in the empty state.				*/
		QString		tag	= QString("%1.%2").arg(kElement).arg(kName) ;
		KBDictEntry	*entry	;

		if ((entry = m_dict.find (tag)) == 0)
		{
			entry = new KBDictEntry	   ;
			m_dict.insert (tag, entry) ;
		}

		/* Scan through the children of the dictionary		*/
		/* element, looking for cases that we know about.	*/
		for (QDomNode child = dAttr.firstChild() ; !child.isNull() ; child = child.nextSibling())
		{
			QDomElement e	= child.toElement() ;
			QString	    tag	= e    .tagName  () ;

			/* First three cases are the main ones, and	*/
			/* set the corresponding values in the		*/
			/* dictionary entry.				*/
			if (tag == "legend"     )
			{	entry->m_legend  = e.text() ;
				continue ;
			}
			if (tag == "description")
			{	entry->m_descrip = e.text() ;
				continue ;
			}
			if (tag == "nullcheck"  )
			{	entry->m_nullchk = e.text() ;
				continue ;
			}

			/* The other case is an "extra" value, which	*/
			/* is stored under a tag.			*/
			if (tag == "extra"	)
			{
				if (entry->m_extra == 0)
					entry->m_extra = new QDict<QString> ;

				entry->m_extra->insert
				(	e.attribute("tag"),
					new QString(e.text().stripWhiteSpace())
				)	;
				continue;
			}
		}
	}
}
