/***************************************************************************
    file	         : tk_action.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>

#include	<qguardedptr.h>
#include	<qpopupmenu.h>
#include	<qtoolbar.h>
#include	<qregexp.h>
#include	<qiconset.h>
#include	<qcursor.h>


#include	"tk_toolbutton.h"
#include	"tk_config.h"

#ifndef 	_WIN32
#include	"tk_action.moc"
#else
#include 	"tk_action.h"
#endif


/*  TKActionPlugin							*/
/*  --------------							*/
/*  This class is used to record an instance of an action being 	*/
/*  inserted into a popup menu or being instantiated as a toolbar	*/
/*  button. In addition, it has methods to transparently handle		*/
/*  operations such as enabling or disabling.				*/

class LIBTKKDE_API	TKActionPlugin
{
	QGuardedPtr<QPopupMenu>		m_menu		;
	QGuardedPtr<TKToolBarButton>	m_button	;
	QGuardedPtr<QWidget>		m_widget	;
	int				m_id		;

public	:

	inline	TKActionPlugin
		(	QPopupMenu	*menu,
			int		id
		)
		:
		m_menu		(menu),
		m_button	(0),
		m_widget	(0),
		m_id		(id)
	{
	}

	inline	TKActionPlugin
		(	TKToolBarButton	*button
		)
		:
		m_menu		(0),
		m_button	(button),
		m_widget	(button),
		m_id		(0)
	{
	}

	inline	TKActionPlugin
		(	QWidget		*widget
		)
		:
		m_menu		(0),
		m_button	(0),
		m_widget	(widget),
		m_id		(0)
	{
	}

	~TKActionPlugin 	() ;

	bool	refersTo	(const QObject *) ;
	void	setEnabled	(bool) ;

	inline	QPopupMenu	*menu	() { return m_menu	; }
	inline	QToolButton	*button	() { return m_button	; }
	inline	QWidget		*widget	() { return m_widget	; }
	inline	int		id	() { return m_id	; }
}	;


TKActionPlugin::~TKActionPlugin ()
{
}

bool	TKActionPlugin::refersTo
	(	const QObject	*obj
	)
{
	return	(obj == m_menu) || (obj == m_button) ;
}

/*  TKActionPlugin							*/
/*  setEnabled	: Enable or disable the associate control		*/
/*  enabled	: bool		: True to enable			*/
/*  (returns)	: void		:					*/

void	TKActionPlugin::setEnabled
	(	bool	enabled
	)
{
//	fprintf
//	(	stderr,
//		"TKActionPlugin::setEnabled(%d) [%p/%d] [%p]\n",
//		enabled,
//		(void *)m_menu,
//		m_id,
//		(void *)m_button
//	)	;

	if (m_menu  ) m_menu  ->setItemEnabled (m_id, enabled) ;
	if (m_button) m_button->setEnabled     (      enabled) ;
}




static	int	toolButtonId = -2 ;
extern	QPixmap	getDesktopIcon	(const QString &) ;
extern	QPixmap	getSmallIcon	(const QString &) ;
extern	QPixmap	getBarIcon	(const QString &) ;



/*  TKAction								*/
/*  TKAction	: Constructor for a basic action			*/
/*  text	: const QString & : Text for menu entry			*/
/*  iconName	: const QString & : Name of icon			*/
/*  accel	: int		  : Keyboard accelerator		*/
/*  received	: QObject *	  : Receiver for action signals		*/
/*  slot	: const char *	  : Slot ton which to direct signals	*/
/*  parent	: QObject *	  : Parent object			*/
/*  name	: const char *	  : Possible object name		*/
/*  (returns)	: TKAction	  :					*/

TKAction::TKAction
	(	const QString	&text,
		const QString	&iconName,
		int		accel,
		const QObject	*receiver,
		const char	*slot,
		QObject		*parent,
		const char	*name
	)
	:
	QObject		(parent, name),
	m_text		(text),
	m_iconName	(iconName),
	m_accel		(accel)
{
	m_enabled	= true	;
	m_code		= 0	;
	m_group		= 0	;

	m_plugins.setAutoDelete (true) ;

	if (QObject::parent() && QObject::parent()->inherits("TKActionCollection"))
		m_collection = (TKActionCollection *)(QObject::parent()) ;
	else	m_collection = 0 ;

	if (m_collection != 0) m_collection->insert (this) ;

	/* The activated signal if triggered when a plugin for this	*/
	/* action is activated; we bounce the signal on to our receiver	*/
	/* with the addition of the code.				*/
	connect	(this, SIGNAL(activated(int)), receiver, slot) ;
}

TKAction::TKAction
	(	const QString	&text,
		int		accel,
		QObject		*parent,
		const char	*name
	)
	:
	QObject	(parent, name),
	m_text	(text),
	m_accel	(accel)
{
	m_enabled	= true	;
	m_code		= 0	;
	m_group		= 0	;

	m_plugins.setAutoDelete (true) ;

	if (QObject::parent() && QObject::parent()->inherits("TKActionCollection"))
		m_collection = (TKActionCollection *)(QObject::parent()) ;
	else	m_collection = 0 ;

	if (m_collection != 0) m_collection->insert (this) ;
}

/*  TKAction								*/
/*  ~TKAction	: Destructor for a basic action				*/
/*  (returns)	:		:					*/

TKAction::~TKAction ()
{
	/* If the action is part of a collection then remove it. Note	*/
	/* that we use "take" and not "remove" as we don't want the	*/
	/* collection to delete us!					*/
	if (m_collection != 0)
		m_collection->take (this) ;

	TKActionPlugin *plugin ;
	while ((plugin = m_plugins.first()) != 0)
	{
		QPopupMenu  *menu   = plugin->menu  () ;
		QWidget	    *widget = plugin->widget() ;

		if (widget != 0)
		{
			delete	widget	;
		}
		if (menu != 0)
		{
			menu->removeItem (plugin->id()) ;
		}	

		m_plugins.remove (plugin) ;
	}
}

void	TKAction::setIcon
	(	const QString	&iconName
	)
{
	m_iconName = iconName ;
}

/*  TKAction								*/
/*  plug	: Plug the action into a container			*/
/*  container	: QWidget *	: Container in question			*/
/*  index	: int		: Slot in containter			*/
/*  (returns)	: int		: Index in contained			*/

int	TKAction::plug
	(	QWidget		*container,
		int		index
	)
{
	int	id	;

	if (container->inherits("QPopupMenu"))
	{
		QPopupMenu *popup = (QPopupMenu *)(container) ;

		if (!m_iconName.isEmpty())
		{
			QIconSet iconSet (getSmallIcon (m_iconName)) ;

			id	= popup->insertItem
				  (	iconSet,
					m_text,
					this,
					SLOT(slotActivated()),
					0,
					-1,
					index
				  )	;

		}
		else
			id	= popup->insertItem
				  (	m_text,
					this,
					SLOT(slotActivated()),
					0,
					-1,
					index
				  )	;

		m_plugins.append (new TKActionPlugin (popup, id)) ;
		popup->setItemEnabled (id, m_enabled) ;

		connect
		(	popup,
			SIGNAL(destroyed()),
			this,
			SLOT  (pluginDestroyed())
		)	;

		return	m_plugins.count() - 1 ;
	}

	if (container->inherits ("QToolBar"))
	{
		QToolBar	*toolbar = (QToolBar *)(container) ;
		TKToolBarButton	*button	 = new TKToolBarButton
					   (	m_iconName,
					   	m_text.replace(QRegExp("&"), ""),
					   	"group",
					   	this,
					   	SLOT(slotActivated()),
					   	toolbar,
					   	name()
					   )	;

		m_plugins.append (new TKActionPlugin (button)) ;
		button->setEnabled (m_enabled) ;
		toolButtonId -= 1 ;

		connect
		(	button,
			SIGNAL(destroyed()),
			this,
			SLOT  (pluginDestroyed())
		)	;

		return	m_plugins.count() - 1 ;
	}

	return	-1 ;
}

void	TKAction::unplug
	(	QWidget	 *container
	)
{
	for (TKActionPlugin *plugin  = m_plugins.first() ;
			     plugin != 0 ;
			     plugin  = m_plugins.next ())
	{
		QPopupMenu  *menu   = plugin->menu  () ;
		QWidget	    *widget = plugin->widget() ;
		bool	    gotit   = false ;

		if (widget == container)
		{
			delete	widget	;
			gotit	= true	;
		}
		if (menu   == container)
		{
			menu->removeItem (plugin->id()) ;
			gotit	= true	;
		}	

		if (gotit)
		{	m_plugins.remove (plugin) ;
			break	;
		}
	}
}

void	TKAction::unplugAll ()
{
	for (TKActionPlugin *plugin  = m_plugins.first() ;
			     plugin != 0 ;
			     plugin  = m_plugins.next ())
	{
		QPopupMenu  *menu   = plugin->menu  () ;
		QWidget     *widget = plugin->widget() ;

		if (widget != 0)
			delete	widget	;
		if (menu   != 0)
			menu->removeItem (plugin->id()) ;
	}

	m_plugins.clear () ;
}

/*  TKAction								*/
/*  pluginDestroyed							*/
/*		: Catch destruction of plugin				*/
/*  (returns)	: void		:					*/

void	TKAction::pluginDestroyed ()
{
	const QObject	*obj	= sender() ;

	QListIterator<TKActionPlugin> iter (m_plugins) ;
	TKActionPlugin *plugin ;

	while ((plugin = iter.current()) != 0)
	{
		iter	+= 1 ;
		if (plugin->refersTo(obj)) m_plugins.removeRef (plugin) ;
	}
}

/*  TKAction								*/
/*  setEnabled	: Enable or disable the action				*/
/*  enabled	: bool		: True to enable			*/
/*  (returns)	: void		:					*/

void	TKAction::setEnabled
	(	bool	enabled
	)
{
//	fprintf
//	(	stderr,
//		"TKAction::setEnabled(%s) <- %d\n",
//		(const char *)m_text,
//		enabled
//	)	;

	m_enabled	= enabled ;

	for (TKActionPlugin *plugin  = m_plugins.first() ;
			     plugin != 0 ;
			     plugin  = m_plugins.next ())
		plugin->setEnabled (m_enabled) ;
}

/*  TKAction								*/
/*  setEnabled	: Enable or disable the action if group matches		*/
/*  group	: int		: Required group			*/
/*  enabled	: bool		: True to enable			*/
/*  (returns)	: void		:					*/

void	TKAction::setEnabled
	(	int	group,
		bool	enabled
	)
{
	if (group == m_group) setEnabled (enabled) ;
}

/*  TKAction								*/
/*  slotActivated: Handled for activated signal from plugin		*/
/*  (returns)	 : void		:					*/

void	TKAction::slotActivated ()
{
	emit	activated (m_code) ;
}


void	TKAction::setToolTip
	(	const QString	&
	)
{
}


/*  ------------------------------------------------------------------  */

/*  TKToggleAction							*/
/*  TKToggleAction							*/
/*		: Constructor for a toggle action			*/
/*  text	: const QString & : Text for menu entry			*/
/*  iconName	: const QString & : Name of icon			*/
/*  accel	: int		  : Keyboard accelerator		*/
/*  received	: QObject *	  : Receiver for action signals		*/
/*  slot	: const char *	  : Slot ton which to direct signals	*/
/*  parent	: QObject *	  : Parent object			*/
/*  name	: const char *	  : Possible object name		*/
/*  (returns)	: TKAction	  :					*/

TKToggleAction::TKToggleAction
	(	const QString	&text,
		const QString	&iconName,
		int		accel,
		const QObject	*receiver,
		const char	*slot,
		QObject		*parent,
		const char	*name
	)
	:
	TKAction (text, iconName, accel, receiver, slot, parent, name)
{
	m_checked	= false ;
}

/*  TKToggleAction							*/
/*  ~TKToggleAction							*/
/*		: Destructor for a toggle action			*/
/*  (returns)	:		:					*/

TKToggleAction::~TKToggleAction ()
{
}

/*  TKToggleAction								*/
/*  plug	: Plug the action into a container			*/
/*  container	: QWidget *	: Container in question			*/
/*  index	: int		: Slot in containter			*/
/*  (returns)	: int		: Index in contained			*/

int	TKToggleAction::plug
	(	QWidget		*container,
		int		index
	)
{
	int	_index	= TKAction::plug (container, index) ;
	if (_index < 0) return _index ;

	if (container->inherits("QPopupMenu"))
	{
		QPopupMenu *popup  = (QPopupMenu *)(container) ;
		popup->setItemChecked (m_plugins.last()->id(), m_checked) ;
		return	_index	;
	}

	if (container->inherits ("QToolBar"))
	{
		m_plugins.last()->button()->setToggleButton   (true)	  ;
		m_plugins.last()->button()->setOn             (m_checked) ;
		return	_index	;
	}

	return	_index	;
}

/*  TKToggleAction							*/
/*  setChecked	: Set checked state of plugins				*/
/*  checked	: bool		: Required state			*/
/*  (returns)	: void		:					*/

void	TKToggleAction::setChecked
	(	bool	checked
	)
{
	m_checked = checked ;

	for (TKActionPlugin *plugin  = m_plugins.first() ;
			     plugin != 0 ;
			     plugin  = m_plugins.next ())
		if (plugin->button() != 0)
			plugin->button()->setOn (m_checked) ;
		else	plugin->menu  ()->setItemChecked (plugin->id(), m_checked) ;
}

/*  TKToggleAction							*/
/*  setChecked	: Set checked state of plugin if in group		*/
/*  group	: int		: Required group			*/
/*  checked	: bool		: Required state			*/
/*  (returns)	: void		:					*/

void	TKToggleAction::setChecked
	(	int	group,
		bool	checked
	)
{
	if (group == m_group) setChecked (checked) ;
}

/*  TKToggleAction							*/
/*  slotActivated: Handle toggle activation				*/
/*  (returns)	 : void		:					*/

void	TKToggleAction::slotActivated ()
{
	setChecked (!isChecked ()) ;
	emit	activated (m_code) ;
	emit	toggled   (isChecked(), m_code) ;
}

/*  ------------------------------------------------------------------  */

TKSelectAction::TKSelectAction
	(	const QString	&text,
		const QString	&,
		int		,
		const QObject	*receiver,
		const char	*slot,
		QObject		*parent,
		const char 	*name
	)
	:
	TKActionMenu (text, parent, name)
{
	m_select.setAutoDelete (true) ;

	connect	(this, SIGNAL(selected(const QString &)), receiver, slot) ;
}

int	TKSelectAction::plug
	(	QWidget		*container,
		int		index
	)
{
	int	id	= TKActionMenu::plug (container, index) ;

	if (id >= 0)
	{
		for (TKAction *action  = m_select.first() ;
			       action != 0 ;
			       action  = m_select.next())
		{
			action->plug (m_popup, index) ;
			index	+= 1 ;
		}

		return	   id	    ;		
	}

	return	-1 ;
}

void	TKSelectAction::slotActivated ()
{
	TKAction *action = (TKAction *)sender() ;

	if (m_select.findRef (action) >= 0)
	{
//		fprintf	(stderr, "select: %d: %s\n",
//				 action,
//				 (const char *)(action == 0 ? QString::null : action->text())) ;

		emit selected (action->text()) ;
	}
}

void	TKSelectAction::setItems
	(	const QStringList	&items
	)
{
	m_select.clear () ;

	for (uint idx = 0 ; idx < items.count() ; idx += 1)
	{
		const QString	&item	= items[idx] ;

		for (TKAction *action1  = m_select.first() ;
			       action1 != 0 ;
			       action1  = m_select.next ())
			if (action1->text() == item)
			{
				m_select.removeRef (action1) ;
				delete	action1	;
				break	;
			}

		TKAction *action2 = new TKAction
				   (	item,
					0,
					0,
					this,
					SLOT(slotActivated()),
					this
				   )	;

		insert		(action2) ;
		m_select.append (action2) ;
	}
}

void	TKSelectAction::setCurrentItem
	(	int		
	)
{

}

QPopupMenu
	*TKSelectAction::popupMenu ()
{
	return	m_popup	;
}

/*  ------------------------------------------------------------------  */

TKRecentFilesAction::TKRecentFilesAction
	(	const QString	&text,
		const QString	&,
		int		,
		const QObject	*receiver,
		const char	*slot,
		QObject		*parent,
		const char 	*name
	)
	:
	TKActionMenu (text, parent, name)
{
	connect	(this, SIGNAL(urlSelected(const TKURL &)), receiver, slot) ;
}

/*  TKRecentFilesAction							*/
/*  ~TKRecentFilesAction						*/
/*		: Destructor for recent files action			*/
/*  (returns)	:		:					*/

TKRecentFilesAction::~TKRecentFilesAction ()
{
}

void	TKRecentFilesAction::slotActivated ()
{
	TKAction *action = (TKAction *)sender() ;

	if (m_files.findRef (action) >= 0)
	{
//		fprintf	(stderr, "recent: %d: %s\n",
//				 action,
//				 (const char *)(action == 0 ? QString::null : action->text())) ;

		emit urlSelected (TKURL(action->text())) ;
	}
}

int	TKRecentFilesAction::plug
	(	QWidget		*container,
		int		index
	)
{
	int	id	= TKActionMenu::plug (container, index) ;

	if (id >= 0)
	{
		for (TKAction *action  = m_files.first() ;
			       action != 0 ;
			       action  = m_files.next())
		{
			action->plug (m_popup, index) ;
			index	+= 1 ;
		}

		return	   id	    ;		
	}

	return	-1 ;
}

void	TKRecentFilesAction::loadEntries
	(	TKConfig	*config
	)
{
	config->setGroup ("RecentFiles") ;

	for (int idx = 10 ; idx >= 1 ; idx -= 1)
	{
		QString	file	= config->readEntry (QString("File%1").arg(idx)) ;
		if (!file.isEmpty ()) addURL (TKURL(file)) ;
	}
}

void	TKRecentFilesAction::saveEntries
	(	TKConfig	*config
	)
{
	uint	idx	= 1 ;

	config->setGroup ("RecentFiles") ;

	for (TKAction *action  = m_files.first() ;
		       action != 0 ;
		       action  = m_files.next ())
	{
		config->writeEntry (QString("File%1").arg(idx), action->text()) ;
		idx	+= 1 ;
	}

	while (idx <= 10)
	{
		config->writeEntry (QString("File%1").arg(idx), QString("")) ;
		idx	+= 1 ;
	}
}

void TKRecentFilesAction::addURL(const TKURL &url)
{
	for (TKAction *action1  = m_files.first(); action1 != 0; action1  = m_files.next ())
		if (action1->text() == url.path())
		{
			m_files.removeRef (action1) ;
			delete	action1	;
			break	;
		}

	TKAction *action2 = new TKAction
			   (	url.path(),
				0,
				0,
				this,
				SLOT(slotActivated()),
				this
			   )	;

	insert		(action2, 0) ;
	m_files.prepend	(action2) ;
}

void	TKRecentFilesAction::removeURL
	(	const TKURL	&url
	)
{
	for (TKAction *action1  = m_files.first(); action1 != 0; action1  = m_files.next ())
		if (action1->text() == url.path())
		{
			m_files.removeRef (action1) ;
			delete	action1	;
			break	;
		}

}

void	TKRecentFilesAction::setCurrentItem
	(	int		
	)
{

}

/*  ------------------------------------------------------------------  */

TKActionMenu::TKActionMenu
	(	const QString	&text,
		QObject		*parent,
		const char	*name
	)
	:
	TKAction (text, 0, parent, name)
{
	m_popup	= new QPopupMenu () ;
}

TKActionMenu::~TKActionMenu ()
{
	delete	m_popup	;
}

QPopupMenu
	*TKActionMenu::popup ()
{
	return	m_popup	;
}

/*  TKActionMenu							*/
/*  plug	: Plug the action menu into a container			*/
/*  container	: QWidget *	: Container in question			*/
/*  index	: int		: Index in containter			*/
/*  (returns)	: int		: Index in container			*/

int	TKActionMenu::plug
	(	QWidget		*container,
		int		index
	)
{
	if (container->inherits("QPopupMenu"))
	{
		QPopupMenu *popup   = (QPopupMenu *)(container) ;
		int	   id	    = popup->insertItem (m_text, m_popup, -1, index) ;

		m_plugins.append (new TKActionPlugin (popup, id)) ;
		popup->setItemEnabled (id, m_enabled) ;

		connect
		(	m_popup,
			SIGNAL(destroyed()),
			this,
			SLOT  (pluginDestroyed())
		)	;

		return	   id	    ;		
	}

	if (container->inherits ("QToolBar"))
	{
		QToolBar	*toolbar = (QToolBar *)(container) ;
		TKToolBarButton	*button	 = new TKToolBarButton
					   (	m_iconName,
					   	m_text.replace(QRegExp("&"), ""),
					   	"group",
					   	this,
					   	SLOT(slotActivated()),
					   	toolbar,
					   	name()
					   )	;

		m_plugins.append (new TKActionPlugin (button)) ;
		button->setEnabled (m_enabled) ;
		toolButtonId -= 1 ;

		connect
		(	button,
			SIGNAL(destroyed()),
			this,
			SLOT  (pluginDestroyed())
		)	;

		return	m_plugins.count() - 1 ;
	}


	return	-1 ;
}

void	TKActionMenu::slotActivated ()
{
	m_popup->exec(QCursor::pos()) ;
}


void	TKActionMenu::insert
	(	TKAction	*action,
		int		index
	)
{
	action->plug   (m_popup, index) ;
}

void	TKActionMenu::remove
	(	TKAction	*action
	)
{
	action->unplug (m_popup) ;
}

/*  ------------------------------------------------------------------  */

TKWidgetAction::TKWidgetAction
	(	QWidget		*widget,
		QObject		*parent,
		const char	*name
	)
	:
	TKAction ("", 0, parent, name),
	m_widget (widget)
{
}

int	TKWidgetAction::plug
	(	QWidget		*container,
		int		
	)
{
	if (container->inherits ("QToolBar"))
	{
		QToolBar	*toolbar = (QToolBar *)(container) ;

		m_plugins.append (new TKActionPlugin (m_widget)) ;
		m_widget->reparent   (toolbar, QPoint())	 ;
		m_widget->setEnabled (m_enabled) ;

		connect
		(	m_widget,
			SIGNAL(destroyed()),
			this,
			SLOT  (pluginDestroyed())
		)	;

		return	m_plugins.count() - 1 ;
	}

	return	-1 ;
}

/*  ------------------------------------------------------------------  */

/*  TKActionCollection							*/
/*  TKActionCollection							*/
/*		: Constructor for an action collection			*/
/*  parent	: QObject *	     : Parent object if any		*/
/*  name	: const char *	     : Possible object name		*/
/*  (returns)	: TKActionCollection :					*/

TKActionCollection::TKActionCollection
	(	QObject		*parent,
		const char	*name
	)
	:
	QObject	(parent, name)
{
}

/*  TKActionCollection							*/
/*  ~TKActionCollection							*/
/*		: Destructor for an action collection			*/
/*  (returns)	:		:					*/

TKActionCollection::~TKActionCollection ()
{
	QDictIterator<TKAction> iter (m_actions) ;
	TKAction *action ;

	for ( ; (action = iter.current()) != 0 ; iter += 1)
		if (action->collection() == this)
			action->setCollection (0) ;
}

void	TKActionCollection::insert
	(	TKAction	*action
	)
{
	m_actions.insert (action->name(), action) ;
	action->setCollection (this) ;
}

void	TKActionCollection::remove
	(	TKAction	*action
	)
{
	m_actions.remove (action->name()) ;
}

void	TKActionCollection::take
	(	TKAction	*action
	)
{
	m_actions.take   (action->name()) ;
}

TKAction*TKActionCollection::action
	(	const char	*name
	)
{
	return	m_actions.find (name) ;
}



TKURL::TKURL
	(	const QUrl	&url
	)
	:
	QUrl	(url)
{
}

TKURL::TKURL
	(	const QString	&url
	)
	:
	QUrl	(url)
{
}

int	TKStdAccel::key
	(	TKStdAccel::StdAccel	
	)
{
	return	0	;
}
