/***************************************************************************
    file	         : kb_querydlg.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<qheader.h>
#include 	<qpainter.h>
#include 	<qcursor.h>

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_gui.h"

#include	"kb_dbinfo.h"
#include	"kb_dblink.h"
#include	"kb_node.h"
#include	"kb_query.h"
#include	"kb_table.h"
#include	"kb_qryexpr.h"
#include	"kb_prompt.h"

#ifndef 	_WIN32
#include	"kb_querydlg.moc"
#else
#include 	"kb_querydlg.h"
#endif

#include	"kb_primarydlg.h"
#include	"kb_qryjoindlg.h"
#include	"kb_select.h"

#include	"tk_messagebox.h"


static	cchar	*exprUsage[] =
{	"",
	"Sort Ascending",
	"Sort Descending",
	"Where",
	"Group By",
	"Having",
	0
}	;


static	int	usageToCode
	(	const QString	&usage
	)
{
	for (uint idx = 0 ; exprUsage[idx] != 0 ; idx += 1)
		if (exprUsage[idx] == usage) return idx ;

	return	0 ;
}



/*  ------------------------------------------------------------------  */

/*  KBQryExprs								*/
/*  KBQryExprs	: Constructor for query expression list			*/
/*  parent	: QWidget *	: Parent widget				*/
/*  (returns)	: KBQryExprs	:					*/

KBQueryExprs::KBQueryExprs
	(	QWidget	*parent
	)
	:
	KBEditListView (false, parent)
{
}

/*  KBQryExprs								*/
/*  fillCombo	: Fill combo box					*/
/*  comboBox	: QComboBox &	  : Combo box in question		*/
/*  		: uint		  : Column number			*/
/*  curVal	: const QString & : Current text value			*/
/*  (returns)	: void		  :					*/

void	KBQueryExprs::fillCombo
	(	QComboBox	&comboBox,
		uint		,
		const QString	&curVal
	)
{
	uint	at	= 0 ;

	comboBox.clear	() ;

	for (uint idx = 0 ; exprUsage[idx] != 0 ; idx += 1)
	{
		comboBox.insertItem (exprUsage[idx]) ;
		if (exprUsage[idx] == curVal) at = idx ;
	}

	comboBox.setCurrentItem (at) ;
}



/*  ------------------------------------------------------------------  */

/*  KBQueryDlg								*/
/*  KBQueryDlg	: Constructor for query design dialog			*/
/*  parent	: QWidget *	: Parent widget				*/
/*  location	: KBLocation &	: Location information			*/
/*  qryRoot	: KBQuery *	: Document root				*/
/*  designGUI	: KBaseGUI *	: Design GUI				*/
/*  (returns)	: KBQueryDlg	:					*/

KBQueryDlg::KBQueryDlg
	(	QWidget		*parent,
		KBLocation	&location,
		KBQuery		*qryRoot,
		KBaseGUI	*designGUI
	)
	:
	KBQueryDlgBase	(parent),
	parent		(parent),
	location	(location),
	qryRoot		(qryRoot),
	designGUI	(designGUI),

	tblPicker	(this),

#if	__KB_EMBEDDED
	vLayPick	(&tblPicker),
#else
	hLayMain	(&tblPicker),
	vLayPick	(&hLayMain),
#endif

	serverList	(&tblPicker),
	allTables	(&tblPicker),
	qryDisplay	(&tblPicker),
	qrySpace	(&qryDisplay, this),

	expressions	(this),
	SQLView		(this)
{
#if	__KB_EMBEDDED
	addPage		(&tblPicker,	TR("Tables"	)) ;
	addPage		(&expressions,	TR("Expressions")) ;
	addPage		(&SQLView,	"SQL") ;
#endif

	vLayPick   .addWidget   (&serverList) ;
	vLayPick   .addWidget   (&allTables ) ;
#if	__KB_EMBEDDED
	vLayPick   .addWidget   (&qryDisplay, 1) ;
#else
	hLayMain   .addWidget   (&qryDisplay, 1) ;
#endif
	expressions.addColumn		(TR("Usage"),      100) ;
	expressions.addColumn		(TR("Expression"), 500) ;
	expressions.addColumn		(TR("Alias"),      100) ;
	expressions.setEditType 	(0, KBEditListView::EdComboBox) ;

	qryDisplay .show        () ;

#if	__KB_EMBEDDED
	allTables  .setFixedHeight (serverList.sizeHint().height() * 3) ;
#else
	serverList .setFixedWidth  (160) ;
	allTables  .setFixedWidth  (160) ;
#endif

	KBServerInfo *self  = location.dbInfo->findServer (location.docLocn) ;
	if (!self ->dbType().isEmpty())
		serverList.insertItem ("Self") ;

	KBServerInfo *files = location.dbInfo->findServer (KBLocation::m_pFile) ;
	if (!files->dbType().isEmpty())
		serverList.insertItem (KBLocation::m_pFile) ;

	QListIterator<KBServerInfo> *svIter = location.dbInfo->getServerIter() ;
	KBServerInfo		    *svInfo ;

	while ((svInfo = svIter->current ()) != 0)
	{
		serverList.insertItem (svInfo->serverName()) ;
		(*svIter) += 1 ;
	}

	delete	svInfo	;

	popup = new QPopupMenu (this) ;
	popup->insertItem ("Cancel")  ;
	popup->insertItem ("Delete",    this, SLOT(clickDropTable())) ;
	popup->insertItem ("Set Alias", this, SLOT(setAlias      ())) ;
	popup->insertItem ("Set Key",   this, SLOT(setKey        ())) ;

	connect (&serverList, SIGNAL(activated(int)), SLOT(serverSelected(int))) ;
	connect	(&allTables,  SIGNAL(selected (int)), SLOT(clickAddTable    ())) ;

	connect	(&qryDisplay, SIGNAL(resized        (KBResizeFrame *, QSize)),
		 	      SLOT  (displayResize  (KBResizeFrame *, QSize))) ;
	connect	(&qrySpace,   SIGNAL(windowActivated(QWidget *)),
		 	      SLOT  (tableSelected  (QWidget *))) ;

	connect (&expressions,SIGNAL(changed	    (uint,uint)), SLOT(exprChanged(uint,uint))) ;
	connect (&expressions,SIGNAL(inserted	    (uint)),      SLOT(exprChanged())) ;
	connect (&expressions,SIGNAL(deleted	    (uint)),      SLOT(exprChanged())) ;


	connect (&exprTimer,  SIGNAL(timeout	    ()),	  SLOT(updateExprs())) ;

	qryDisplay.setFrameStyle(QFrame::Box|QFrame::Plain) ;
	qrySpace.move	  (2, 2) ;

	tabList.setAutoDelete (true) ;
	buildDisplay  ()    ;

	trackTable  = 0	    ;
	curTable    = 0	    ;
	loadSQL	      ()    ;

#if	! __KB_EMBEDDED
	QValueList<int> sl		;
	sl.append	(250)		;
	sl.append	(100)		;
	sl.append	(200)		;
	resize		(sizeHint())	;
	setSizes	(sl)		;
	setResizeMode	(&SQLView, QSplitter::KeepSize) ;
#endif
}

/*  KBQueryDlg								*/
/*  ~KBQueryDlg	: Destructor for query design dialog			*/
/*  (returns)	:		:					*/

KBQueryDlg::~KBQueryDlg ()
{
}

/*  KBQueryDlg								*/
/*  setChanged	: Set changed on update					*/
/*  (returns)	: void		 :					*/

void	KBQueryDlg::setChanged ()
{
	designGUI->setEnabled (KB::GRDelta, true) ;
	qryRoot  ->setChanged (true) ;
}

/*  KBQueryDlg								*/
/*  sizeHint	: Return preferred size					*/
/*  (returns)	: QSize		: Preferred size			*/

QSize	KBQueryDlg::sizeHint ()
{
	return	QSize (600, 400) ;
}

/*  KBQueryDlg								*/
/*  resizeEvent	: Handle resize event					*/
/*  e		: QResizeEvent * : The event				*/
/*  (returns)	: void		 :					*/

void	KBQueryDlg::resizeEvent
	(	QResizeEvent	*e
	)
{
#if	__KB_EMBEDDED
	TKTabWidget::resizeEvent (e) ;
#else
	QSplitter  ::resizeEvent (e) ;
#endif
//	int ers = expressions.width()
//				- expressions.columnWidth(0)
//				- expressions.columnWidth(1) ;
//
//	if (ers > 16) expressions.setColumnWidth (2, ers) ;
}

QValueList<int>
	KBQueryDlg::exprSizes ()
{
	QValueList<int>	sizes	;
	sizes.append (expressions.columnWidth (0)) ;
	sizes.append (expressions.columnWidth (1)) ;
	sizes.append (expressions.columnWidth (2)) ;
	return		sizes	;
}

void	KBQueryDlg::setExprSizes
	(	QValueList<int>	sizes
	)
{
	if (sizes[0] > 16) expressions.setColumnWidth (0, QMIN(sizes[0],  200)) ;
	if (sizes[1] > 16) expressions.setColumnWidth (1, QMIN(sizes[1], 1200)) ;
	if (sizes[2] > 16) expressions.setColumnWidth (2, QMIN(sizes[2],  300)) ;
}

/*  KBQueryDlg								*/
/*  displayResize: Handle resize of display widget			*/
/*  frame	 : KBResizeFrame * : Frame				*/
/*  s		 : QSize	   : New size				*/
/*  (returns)	 : void		   :					*/

void	KBQueryDlg::displayResize
	(	KBResizeFrame	*,
		QSize		s
	)
{
	qrySpace.resize (s - QSize(4,4)) ;
}

/*  KBQueryDlg								*/
/*  buildDisplay: Build list view and graphic displays			*/
/*  (returns)	: void		:					*/

void	KBQueryDlg::buildDisplay ()
{
	/* Get the server name from the query and look through the list	*/
	/* of known servers to verify that it is there and to show the	*/
	/* appropriate one ...						*/
	QString			svName	   ;	
	QList<KBTable  >	qryTabList ;
	QList<KBQryExpr>	exprList   ;
	KBEditListViewItem	*last	   = 0 ;

	qryRoot->getQueryInfo (svName, qryTabList, exprList) ;

	serverIndex	= -1 	   ;

	for (int idx = 0 ; idx < serverList.count() ; idx += 1)
		if (serverList.text(idx) == svName)
		{	serverIndex = idx ;
			break ;
		}

	if ((serverIndex > 0) || (svName == "Self"))
	{
		/* Found the server, so load the table and join stuff.	*/
		/* Iterate over the tables and add them, then add any	*/
		/* expressions.						*/
		serverList.setCurrentItem (serverIndex) ;
		serverConnect () ;

		LITER
		(	KBTable,
			qryTabList,
			table,
			tabList.append (new KBTableAlias (this, table))
		)

		LITER
		(	KBQryExpr,
			exprList,
			expr,

			KBQryExpr::Usage usage = expr->getUsage() ;
			QString		 alias = expr->getAlias() ;

			if ((uint)usage >= sizeof(exprUsage)/sizeof(exprUsage[0]) - 1)
				usage = KBQryExpr::AsExprOnly ;

			if (usage != KBQryExpr::AsExprOnly) alias = "" ;

			last  = new KBEditListViewItem
				(	&expressions,
					last,
					exprUsage[usage],
					expr->getExpr (),
					alias
				)
		)
	}
	else if (!svName.isEmpty())
	{
		/* Server not found. In this case grumble, and clear	*/
		/* the tables from the query object, since thay are not	*/
		/* referred to from any KBTableAlias objects.		*/
		TKMessageBox::sorry
		(	0,
			QString (TR("Server \"%1\" is not in this database")).arg(svName),
			TR("Query server error")
		)	;
		serverIndex = 0 ;

		LITER (KBTable, qryTabList, t, delete t)

		serverList.setCurrentItem (0) ;
	}

	new KBEditListViewItem  (&expressions, last, "") ;
	tblPicker .show() ;
	qrySpace  .show() ;
}

/*  KBQueryDlg								*/
/*  serverConnect: Connect to selected server				*/
/*  (returns)	 : void		:					*/

void	KBQueryDlg::serverConnect ()
{
	dbLink.disconnect () ;
	allTables.clear   () ;

	if (!dbLink.connect (location, serverList.currentText()))
	{
		dbLink.lastError().DISPLAY () ;
		return	;
	}

	KBTableDetailsList tabList	;

	if (!dbLink.listTables (tabList))
	{
		dbLink.lastError().DISPLAY() ;
		return	;
	}

	for (uint idx = 0 ; idx < tabList.count() ; idx += 1)
		allTables.insertItem (tabList[idx].m_name) ;
}

/*  KBQueryDlg								*/
/*  nameIsFree	: See if name is free for use				*/
/*  name	: const QString & : Name to be checked			*/
/*  checkAll	: bool		  : Check against all			*/
/*  (returns)	: bool		  : Name is free			*/

bool	KBQueryDlg::nameIsFree
	(	const QString	&name,
		bool		checkAll
	)
{
	LITER
	(	KBTableAlias,
		tabList,
		table,

		/* If the current search item is the item being edited	*/
		/* then maybe skip it, otherwise see if the name	*/
		/* conflicts with the name by which the table is	*/
		/* identified in the SQL statement.			*/
		if (checkAll || (table != curTable))
			if (table->getTable()->getName () == name)
				return	false	;
	)
	return	true	;
}

/*  KBQueryDlg								*/
/*  setAlias	: Set table alias					*/
/*  (returns)	: void		:					*/

void	KBQueryDlg::setAlias ()
{
	if (curTable == 0) return ;

	QString	table	= curTable->getTable()->getTable() ;
	QString	alias	= curTable->getTable()->getAlias() ;

	if (!doPrompt
		(	TR("Alias"),
			QString(TR("Enter alias for table %1")).arg(table),
			alias
	   )	)	return ;

	/* If the first alias is blank then verify that this is the	*/
	/* only entry for that table. On the other hand, if it is not	*/
	/* blank then it must be different to all other tables and	*/
	/* aliases.							*/
	if (alias.isEmpty())
	{
		if (!nameIsFree (table, false))
		{
			TKMessageBox::sorry
			(	0,
				QString (TR("Table \"%1\" appears more than once: an alias is required"))
					.arg(table),
				TR("Alias required")
			)	;
			return	;
		}
	}
	else
	{
		if (!nameIsFree (alias, false))
		{
			TKMessageBox::sorry
			(	0,
				QString (TR("Alias \"%1\" already used as a table or alias name"))
					.arg(alias),
				TR("Unique alias required")
			)	;
			return	;
		}
	}

	/* Update the alias, except that if the alias is the same as	*/
	/* the table name then clear it.				*/
	if (alias != table)
		curTable->setAlias (alias) ;
	else	curTable->setAlias (""   ) ;

	loadSQL	   () ;
	setChanged () ;
}

/*  KBQueryDlg								*/
/*  setKey	: Set table key						*/
/*  (returns)	: void		:					*/

void	KBQueryDlg::setKey ()
{
	if (curTable == 0)  return ;

	QString		    column ;
	QStringList	    names  ;
	KBTable::UniqueType utype  ;
	QString		    uexpr  ;
	KBTableSpec	    tabSpec(curTable->getTable()->getTable()) ;

	if (!dbLink.listFields (tabSpec))
	{
		dbLink.lastError().DISPLAY() ;
		return	;
	}

	utype = curTable->getTable()->getUnique (names, uexpr) ;

	KBQryPrimaryDlg pDlg (tabSpec, names, utype, uexpr) ;
	if (!pDlg.exec()) return ;

	utype  = pDlg.retrieve (column, uexpr) ;
	curTable->setPrimary   (column, utype) ;
	setChanged () ;
}

/*  KBQueryDlg								*/
/*  serverSelected: Handle server selection				*/
/*  index	  : int		: Server list index			*/
/*  (returns)	  : void	:					*/

void	KBQueryDlg::serverSelected
	(	int	index
	)
{
	if (index == serverIndex) return ;

	if (tabList.count() > 0)
		if (TKMessageBox::questionYesNo
	   	(	0,
			TR("This will clear all tables and relationships: continue?"),
			TR("Change server")
		)
		!= TKMessageBox::Yes)
	{
		serverList.setCurrentItem (serverIndex) ;
		return	;
	}

	LITER
	(	KBTableAlias,
		tabList,
		t,

		delete	t->getTable()
	)

	tabList.clear ()    ;
	serverIndex = index ;
	serverConnect ()    ;

	qryRoot->setServer (serverList.currentText()) ;
	loadSQL	      ()    ;
	repaintLinks  ()    ;
}

/*  KBQueryDlg								*/
/*  getUniqeueAlias: Get unique alias for a table			*/
/*  name	   : const QString & : Table name			*/
/*  (returns)	   : QString	     : Unique alias			*/

QString	KBQueryDlg::getUniqueAlias
	(	const QString	&name
	)
{
	uint		mIdx	= 0 ;
	QString		alias	;

	if (nameIsFree  (name, true)) return "" ;

	for (;;)
	{
		mIdx   += 1	;
		alias	= QString("%1_%2").arg(name).arg(mIdx) ;
		if (nameIsFree (alias, true)) return alias ;
	}
}

/*  KBQueryDlg								*/
/*  findTable	: Find table alias by widget				*/
/*  widget	: QWidget *	 : Widget to seach for			*/
/*  (returns)	: KBTableAlias * : Table alias or null if not found	*/

KBTableAlias
	*KBQueryDlg::findTable
	(	QWidget		*widget
	)
{
	LITER
	(	KBTableAlias,
		tabList,
		table,
		if ((QWidget *)table == widget) return table ;
	)

	return	0 ;
}

/*  KBQueryDlg								*/
/*  findTable	: Find table alias by global position			*/
/*  p		: QPoint	 : Global position			*/
/*  field	: QString &	 : Field under point if found		*/
/*  (returns)	: KBTableAlias * : Table alias or null if not found	*/

KBTableAlias
	*KBQueryDlg::findTable
	(	QPoint		p,
		QString		&field
	)
{
	LITER
	(	KBTableAlias,
		tabList,
		table,
		if (table->hit (p, field)) return table ;
	)

	return	0 ;
}

/*  KBQueryDlg								*/
/*  getPrimary	: Get primary key for named table			*/
/*  tabName	: const QString & : Table name				*/
/*  colName	: QString &	  : Return column name			*/
/*  (returns)	: UniqueType	  : Uniqueness type			*/

KBTable::UniqueType
	KBQueryDlg::getPrimary
	(	const QString	&tabName,
		QString		&colName
	)
{
	KBTableSpec	tabSpec	(tabName) ;
	KBFieldSpec	*fSpec	;

	if (!dbLink.listFields (tabSpec))
	{
		dbLink.lastError().DISPLAY() ;
		colName	= QString::null	     ;
		return	KBTable::AnySingle   ;
	}


	fSpec	= tabSpec.findPrimary() ;
	if (fSpec != 0)
	{	colName	= fSpec->m_name	     ;
		return	KBTable::PrimaryKey  ;
	}

	fSpec	= tabSpec.findUnique () ;
	if (fSpec != 0)
	{	colName	= fSpec->m_name	     ;
		return	KBTable::AnyUnique   ;
	}

	colName	= QString::null	     ;
	return	KBTable::AnySingle   ;
}

/*  KBQueryDlg								*/
/*  clickAddTable: Handle request to add a table			*/
/*  (returns)	 : void		:					*/

void	KBQueryDlg::clickAddTable ()
{
	if (allTables.currentItem () < 0) return ;

	QString	tName	= allTables.text (allTables.currentItem()) ;
	QString	aName	= getUniqueAlias (tName) ;

	QString		    uName ;
	KBTable::UniqueType uType = getPrimary (tName, uName) ;

	/* Construct a new table object and set the primary key if	*/
	/* there is one.						*/
	KBTable	*table	= new KBTable  (qryRoot, tName, aName, "", "", "", "", "", "", "") ;
	table->setPrimary (uName, uType) ;

	tabList.append (curTable = new KBTableAlias (this, table)) ;
	loadSQL    () ;
	setChanged () ;
}

/*  KBQueryDlg	  :							*/
/*  clickDropTable: Request to drop a table				*/
/*  (returns)	  : void	:					*/

void	KBQueryDlg::clickDropTable ()
{
	if (curTable != 0)
	{
		KBTableAlias *t = curTable ;
		curTable	= 0	   ;

		delete	t->getTable()	   ;
		tabList.remove (t)	   ;
		
		loadSQL 	() ;
		repaintLinks	() ;
		setChanged	() ;
	}
}

/*  KBQueryDlg	 :							*/
/*  tableSelected: User selects table in graphic view			*/
/*  widget	 : QWidget *	: Table					*/
/*  (returns)	 : void		:					*/

void	KBQueryDlg::tableSelected
	(	QWidget		*widget
	)
{
	if (widget != 0) curTable = findTable (widget) ;
}

/*  KBQueryDlg								*/
/*  loadSQL	: Load SQL into SQL view window				*/
/*  (returns)	: void		:					*/

void	KBQueryDlg::loadSQL ()
{

	/* NOTE: This is done by mostly by interrogating the KBQuery	*/
	/* object and its KBTable children, rather than by extracting	*/
	/* data directly from the local list. It is done this way as a	*/
	/* check that changes at the dialog box level are being		*/
	/* correctly propogated to the table nodes.			*/
	KBSelect	select	 ;
	QString		svName   ;
	QList<KBTable>	qryList	 ;
	QList<KBTable>	tabList	 ;
	QList<KBQryExpr>exprList ;
	KBError		error	 ;

	qryRoot->getQueryInfo (svName, qryList, exprList) ;


	/* No tables, no SQL ...					*/
	if (qryList.count() == 0)
	{
		SQLView.setText ("") ;
		return	;
	}

	/* Block up into a flat query (exactly as will be done in data	*/
	/* view. This has the side effect of setting the table join	*/
	/* expressions.							*/
	if (!KBTable::blockUp (qryList, QString::null, tabList, error))
	{
		error  .DISPLAY ()   ;
		SQLView.setText ("") ;
		return	;
	}

	LITER
	(	KBQryExpr,
		exprList,
		expr,

		switch (expr->getUsage())
		{
			case KBQryExpr::AsExprOnly :
				select.appendExpr  (expr->getExpr(), expr->getAlias()) ;
				break	;

			case KBQryExpr::AsWhere	   :
				select.appendWhere (expr->getExpr()) ;
				break	;

			case KBQryExpr::AsSortAsc  :
				select.appendOrder (expr->getExpr() + " asc" ) ;
				break	;

			case KBQryExpr::AsSortDesc :
				select.appendOrder (expr->getExpr() + " desc") ;
				break	;

			case KBQryExpr::AsGroup	   :
				select.appendGroup (expr->getExpr()) ;
				break	;

			case KBQryExpr::AsHaving   :
				select.appendHaving(expr->getExpr()) ;
				break	;

			default	:
				break	;
		}
	)

	LITER
	(	KBTable,
		tabList,
		table,
		table->addToSelect (select, true) ;
	)

	SQLView.setText (select.getPrettyText(true, &dbLink)) ;
}


/*  KBQueryDlg	:							*/
/*  exprChanged	: User has added or deleted an expression		*/
/*  (returns)	: void		:					*/

void	KBQueryDlg::exprChanged ()
{
	updateExprs (false) ;
	setChanged  () ;
}

/*  KBQueryDlg	:							*/
/*  exprChanged	: User has added or deleted an expression		*/
/*  rowNo	: uint		: Row number				*/
/*  colNo	: uint		: Column number				*/
/*  (returns)	: void		:					*/

void	KBQueryDlg::exprChanged
	(	uint		rowNo,
		uint		
	)
{
	QListViewItem	*item = expressions.firstChild() ;
	while ((item != 0) && (rowNo > 0))
	{	item	= item->nextSibling() ;
		rowNo  -= 1 ;
	}

	if ((item != 0) && (item->text(0) != ""))
		item->setText (2, "") ;

	updateExprs (false) ;
	setChanged  () ;

	exprTimer.start (500) ;
}

#if	0
/*  KBQueryDlg	:							*/
/*  hasCommas	: See if expression is really several expressions	*/
/*  expr	: const QString & : Expression in question		*/
/*  (returns)	: bool		  : True if so				*/

bool	KBQueryDlg::hasCommas
	(	const QString	&expr
	)
{
	uint	pos	= 0 	;
	int	nest	= 0	;
	bool	inquote	= false	;

	while (pos < expr.length())
	{
		QChar	ch = expr.at(pos) ;

		if (inquote)
		{
			if (ch == '\\')
			{
				pos += 2 ;
				continue ;
			}

			if (ch == '\'') inquote = false ;
			pos += 1 ;
			continue ;
		}

		if (ch == '\'')
		{
			pos	+= 1	;
			inquote	 = true ;
			continue ;
		}

		if (ch == '(')
		{
			pos	+= 1	;
			nest	+= 1	;
			continue	;
		}

		if (ch == ')')
		{
			pos	+= 1	;
			nest	-= 1	;
			continue	;
		}

		if ((ch == ',') && (nest <= 0))
			return	true	;

		pos += 1 ;
	}

	return	false ;
}
#endif

/*  KBQueryDlg								*/
/*  updateExprs	: Update expressions to match dialog			*/
/*  checkCommas	: bool		: Check for spurious commas		*/
/*  (returns)	: bool		: All expressions OK			*/

bool	KBQueryDlg::updateExprs
	(	bool	checkCommas
	)
{
	QString			svName	   ;	
	QList<KBTable  >	qryTabList ;
	QList<KBQryExpr>	exprList   ;
	bool			rc	   = true  ;
	bool			commas	   = false ;

	exprTimer.stop() ;

	qryRoot->getQueryInfo (svName, qryTabList, exprList) ;
	exprList.setAutoDelete(true) ;
	exprList.clear	      ()     ;


	for (QListViewItem *lvi  = expressions.firstChild () ;
			    lvi != 0 ;
			    lvi  = lvi->nextSibling())
	{
		if (lvi->text(1).isEmpty())
			continue ;

		if	(lvi->text(1).stripWhiteSpace() == "*")
		{	lvi->setText (1, "*") ;
			lvi->setText (2,  "") ;
		}
		else if (checkCommas && !KBSelect::singleExpression (lvi->text(1)))
		{
			if (!commas)
			{
				KBError::EError
				(	QString (TR("Expressions may not contain multiple columns, please edit")),
					lvi->text(1),
					__ERRLOCN
				)	;
				commas	= true	;
				rc	= false	;
			}
		}

		new KBQryExpr
		(	qryRoot,
			lvi->text(1),
			lvi->text(2),
			(KBQryExpr::Usage)usageToCode(lvi->text(0))
		)	;
	}

	loadSQL () ;
	return	rc ;
}

/*  KBQueryDlg								*/
/*  updateExprs	: Update expressions to match dialog			*/
/*  (returns)	: bool		: All expressions OK			*/

bool	KBQueryDlg::updateExprs ()
{
	return	updateExprs (false) ;
}

/*  KBQueryDlg								*/
/*  saveDocument: Save current document state				*/
/*  (returns)	: bool		: Success				*/

bool	KBQueryDlg::saveDocument ()
{
	exprTimer  .stop	() ;
	expressions.cancelEdit	() ;


	if (!updateExprs (true)) return false ;

	/* Log the geometries of the table listboxes. Note that we have	*/
	/* to get the position via the parent, since they are embedded	*/
	/* in a workspace, while the size comes from list listboxes	*/
	/* themselves.							*/
	LITER
	(	KBTableAlias,
		tabList,
		table,
		table->getTable()->setGeometry
		(
			QRect
			(	table->geometry().topLeft(),
				table->size()
		)	)	;
	)

	return	true	;
}


/*  KBQueryDlg								*/
/*  repaintLinks: Repaint all relationship links			*/
/*  p		: QPintEvent *	: Paint event				*/
/*  (returns)	: void		:					*/

void	KBQueryDlg::repaintLinks
	(	QPaintEvent	*
	)
{
	QPainter painter (qrySpace.spaceWidget()) ;

	LITER
	(	KBTableAlias,
		tabList,
		table,

		QString		pid	= table->getTable()->getParent() ;
		KBTableAlias	*parent = 0 ;

		if (pid.isEmpty()) continue ;

		LITER
		(	KBTableAlias,
			tabList,
			table,

			if (table->getTable()->getIdent() == pid)
			{	parent	= table ;
				break	;
			}
		)

		if (parent == 0) continue ;

		QString	field	= table->getTable()->getField () ;
		QString	field2	= table->getTable()->getField2() ;

		if (!field.isEmpty() && !field2.isEmpty())
		{
			QRect	pr	= parent->geometry() ;
			QRect	tr	= table ->geometry() ;
			bool	ps	= false	;
			bool	ts	= false	;

			if	(pr.right() <= tr. left())
			{	ps	= true	;
				ts	= false	;
			}
			else if	(pr.left () >= tr.right())
			{	ps	= false	;
				ts	= true	;
			}

			bool	pkp	;
			bool	tkp	;
			QPoint	pp1	= parent->getPosition (field2, ps, pkp) ;
			QPoint	tp1	= table ->getPosition (field,  ts, tkp) ;
			QPoint	pp2	= ps ? pp1 + QPoint(12, 0) : pp1 - QPoint(12, 0) ;
			QPoint	tp2	= ts ? tp1 + QPoint(12, 0) : tp1 - QPoint(12, 0) ;

			static	QPen	 pen1 (black, 1) ;
			static	QPen	 pen3 (black, 3) ;

			painter.setPen   (pen3	  ) ;
			painter.drawLine (pp1, pp2) ;
			painter.drawLine (tp1, tp2) ;
			painter.setPen   (pen1	  ) ;
			painter.drawLine (pp2, tp2) ;

			if (!pkp)
			{	painter.drawLine (pp1 + QPoint(0, 6), pp2) ;
				painter.drawLine (pp1 - QPoint(0, 6), pp2) ;
			}
			if (!tkp)
			{	painter.drawLine (tp1 + QPoint(0, 6), tp2) ;
				painter.drawLine (tp1 - QPoint(0, 6), tp2) ;
			}
		}
	)
}

/*  KBQueryDlg								*/
/*  repaintLinks: Repaint all relationship links on timeout		*/
/*  (returns)	: void		:					*/

void	KBQueryDlg::repaintLinks ()
{
	qrySpace.spaceWidget()->update() ;
}

void	KBQueryDlg::linkProperties
	(	QPoint		mp
	)
{
	KBTableAlias	*owner	= 0	    ;
	KBTableAlias	*other	= 0	    ;
	int		best	= 0x7ffffff ;

	LITER
	(	KBTableAlias,
		tabList,
		table,

		QString		pid	= table->getTable()->getParent() ;
		KBTableAlias	*parent = 0 ;

		if (pid.isEmpty()) continue ;

		LITER
		(	KBTableAlias,
			tabList,
			table,

			if (table->getTable()->getIdent() == pid)
			{	parent	= table ;
				break	;
			}
		)

		if (parent == 0) continue ;

		QRect	pr	= parent->geometry() ;
		QRect	tr	= table ->geometry() ;

		bool	ps	= false	;
		bool	ts	= false	;

		if	(pr.right() <= tr. left())
		{	ps	= true	;
			ts	= false	;
		}
		else if	(pr.left () >= tr.right())
		{	ps	= false	;
			ts	= true	;
		}

		bool	pkp	;
		bool	tkp	;
		QPoint	pp1	= parent->getPosition (table->getTable()->getField2(), ps, pkp) ;
		QPoint	tp1	= table ->getPosition (table->getTable()->getField (), ts, tkp) ;
		QPoint	pp2	= ps ? pp1 + QPoint(12, 0) : pp1 - QPoint(12, 0) ;
		QPoint	tp2	= ts ? tp1 + QPoint(12, 0) : tp1 - QPoint(12, 0) ;

		QRect	b	= QRect  (pp2, tp2).normalize() ;
		int	ml	= (b.center() - mp).manhattanLength() ;

		if (b.contains(mp) && (ml < best))
		{
			best	= ml	;
			owner	= table	;
			other	= parent;
		}
	)

	if (owner == 0) return ;

//	fprintf
//	(	stderr,
//		"KBQueryDlg::linkProperties: pointers: owner=%p other=%p\n",
//		(void  *)owner->getTable(),
//		(void  *)other->getTable()
//	)	;
//	fprintf
//	(	stderr,
//		"KBQueryDlg::linkProperties: names:    owner=%s other=%s\n",
//		(cchar *)owner->getTable()->getTable(),
//		(cchar *)other->getTable()->getTable()
//	)	;

	KBQryJoinDlg jDlg
		     (	owner->getTable()->getTable (),
			owner->getTable()->getField (),
			other->getTable()->getTable (),
			owner->getTable()->getField2(),
			owner->getTable()->getJType ()
		     )  ;

	if (!jDlg.exec ()) return ;

	QString	jtype	;
	bool	doDel	= jDlg.getResults (jtype) ;


	if (doDel)
	{
		KBTable	*table	= owner->getTable() ;
		table->setParent ("") ;
		table->setField  ("") ;
		table->setField2 ("") ;
		table->setJType  ("") ;

		loadSQL	      () ;
		repaintLinks  () ;
		setChanged    () ;

		return	;
	}

	if (owner->getTable()->getJType () != jtype)
	{
		owner->getTable()->setJType (jtype) ;

		loadSQL	   () ;
		setChanged () ;
	}
}

/*  KBQueryDlg								*/
/*  hasAncestor	: See if a table has a specified ancestor		*/
/*  table	: KBTable *	: Table in question			*/
/*  ancestor	: KBTable *	: Putative ancestor			*/
/*  (returns)	: bool		: True if ancestor really is		*/

bool	KBQueryDlg::hasAncestor
	(	KBTable		*table,
		KBTable		*ancestor
	)
{
	KBTable	*parent	= 0	;
	QString	pid		= table->getParent() ;

	if (pid.isEmpty()) return false ;

	LITER
	(	KBTableAlias,
		tabList,
		t,

		if (t->getTable()->getIdent() == pid)
		{
			parent	= t->getTable() ;
			break	;
		}
	)

	return	parent == 0	   ? false :
		parent == ancestor ? true  : hasAncestor (parent, ancestor) ;
}

/*  KBQueryDlg								*/
/*  eventFilter	: Event filter method					*/
/*  o		: QObject *	: Object being filtered			*/
/*  e		: QEvent  *	: Event					*/
/*  (returns)	: bool		: Event consumed			*/

bool	KBQueryDlg::eventFilter
	(	QObject		*,
		QEvent		*e
	)
{
	/* Trap moves and resizes of field lists in order to keep the	*/
	/* links correct. We don't bother to optimise anything here, it	*/
	/* seems a total repaint is OK.					*/
	if ((e->type() == QEvent::Move) || (e->type() == QEvent::Resize))
	{
		repaintLinks() ;
		setChanged  () ;
		return	false  ;
	}

	return	false	;
}

/*  KBQueryDlg								*/
/*  startLinking: Start table linking operation				*/
/*  table	: KBTableAlias * : Table from which tracking starts	*/
/*  field	: QString	 : Field in table			*/
/*  (returns)	: void		 :					*/

void	KBQueryDlg::startLinking
	(	KBTableAlias	*table,
		QString		field
	)
{
	trackTable = table ;
	trackField = field ;
	grabMouse  (QCursor(PointingHandCursor)) ;
}

/*  KBQueryDlg								*/
/*  showContextMenu							*/
/*		: Show context menu for a table				*/
/*  table	: KBTableAlias * : Table from which tracking starts	*/
/*  (returns)	: void		 :					*/

void	KBQueryDlg::showContextMenu
	(	KBTableAlias	*table
	)
{
	curTable = table ;
	popup->popup (QCursor::pos()) ;
}

/*  KBQueryDlg								*/
/*  mouseReleaseEvent							*/
/*		: Handle release of mouse button			*/
/*  e		: QMouseEvent *	: Mouse event				*/
/*  (returns)	: void		:					*/

void	KBQueryDlg::mouseReleaseEvent
	(	QMouseEvent	*e
	)
{
	/* If not tracking then just do the default ....		*/
	if (trackTable == 0)
	{
		QWidget::mouseReleaseEvent (e) ;
		return	;
	}

	releaseMouse () ;

	/* See if the mouse has been released over a field in a table	*/
	/* list box, eccept in the case if being released over the	*/
	/* table where we started ...					*/
	QString		toField   ;
	KBTableAlias	*toTable  = findTable (e->globalPos(), toField) ;

	if ((toTable != 0) && (toTable != trackTable))
	{
		fprintf	(stderr, "--->[%s][%s][%s] -> [%s][%s][%s]\n",
				 (cchar *)trackTable->getTable()->getTable(),
				 (cchar *)trackField,
				 (cchar *)trackTable->getKeyField(),
				 (cchar *)toTable->getTable()->getTable(),
				 (cchar *)toField,
				 (cchar *)toTable->getKeyField()
			) ;

		KBTableAlias	*pTable	= 0 ;
		KBTableAlias	*cTable	= 0 ;
		QString		pField	;
		QString		cField	;

		/* One end has to be the primary key; if neither end	*/
		/* fulfills this then nothing happens.			*/
		/* NOTE: This should probably be relaxed.		*/
		if	(trackTable->getKeyField() == trackField)
		{
			pTable	= toTable	;
			pField	= toField	;
			cTable	= trackTable	;
			cField	= trackField	;
		}
		else if	(toTable   ->getKeyField() == toField   )
		{
			pTable	= trackTable	;
			pField	= trackField	;
			cTable	= toTable	;
			cField	= toField	;
		}

		if (pTable != 0)
		{
			if (!hasAncestor (pTable->getTable(), cTable->getTable()))
			{
				cTable->getTable()->setParent (pTable->getTable()->getIdent()) ;
				cTable->getTable()->setField  (cField) ;
				cTable->getTable()->setField2 (pField) ;
			}
			else	TKMessageBox::sorry
				(	0,
					TR("Cannot create a relationship loop"),
					TR("Query Error")
				)	;
		}

		trackTable = 0	;

		loadSQL      () ;
		repaintLinks () ;
		setChanged   () ;

		return		;
	}

	/* See if the mouse has been released over an expression field	*/
	/* in the expressions list ....					*/
	QPoint			vpPos	= expressions.viewport()->mapFromGlobal(e->globalPos()) ;
	KBEditListViewItem	*item	= (KBEditListViewItem *)expressions.itemAt(vpPos) ;

	if ((item != 0) && (trackTable != 0))
	{
		item->putText
		(	1,
			QString	("%1 %2.%3")
				.arg(item->text(1))
				.arg(trackTable->getTable()->getName())
				.arg(trackField)
		)	;
	
		if (item->nextSibling() == 0)
			new KBEditListViewItem (&expressions, item, "") ;


		trackTable = 0	;

		updateExprs (false) ;
		setChanged  ()  ;

		return		;
	}

	trackTable = 0	;
}

