/***************************************************************************
    file	         : tkc_pydebugbase.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<qdict.h>



#ifndef 	_WIN32
#include	"tkc_pydebugbase.moc"
#else
#include	"tkc_pydebugbase.h"
#endif


static	bool		inited		   ;	/* Debugger initialised	*/
static	unsigned int	debugCnt	   ;	/* Debug set count	*/
static	unsigned int	debugLevel	   ;	/* Debug nesting	*/

#if	! __PY22PLUS
						/* DEBUGGER module	*/
static	PyObject	*pyTKCDebModule	   ;	/* The python module	*/
static	PyObject	*pyTKCDebDict	   ;	/* ... its dictionary	*/
static	PyObject	*pyTKCDebFuncTrace ;	/* ... function trace	*/
static	PyObject	*pyTKCDebLineTrace ;	/* ... line trace	*/
static	PyObject	*pyTKCDebProfTrace ;	/* ... profile trace	*/

						/* Python SYS module	*/
static	PyObject	*pySysModule	   ;	/* The sys module	*/
static	PyObject	*pySysSetTrace	   ;	/* Set trace method	*/
static	PyObject	*pySysSetProfile   ;	/* Set profile method	*/

#endif

#if	__PY22PLUS

static	PyObject	*pyMsgExcept 	   ;
static	PyObject	*pyMsgCall   	   ;
static	PyObject	*pyMsgLine   	   ;

#endif

static	TKCPyType typeMap[] =
{
/*	Python Type		TKCPyType		Name		Expandable	Simple	*/
{	&PyBuffer_Type,		TKCPyType::Buffer,	"Buffer",	false,		false	},
{	&PyCFunction_Type,	TKCPyType::CFunction,	"CFunction",	false,		false	},
{	&PyClass_Type,		TKCPyType::Class,	"Class",	true,		true	},
{	&PyCObject_Type,	TKCPyType::CObject,	"CObject",	false,		false	},
{	&PyCode_Type,		TKCPyType::Code,	"Code",		true,		false	},
{	&PyComplex_Type,	TKCPyType::Complex,	"Complex",	false,		true	},
{	&PyDict_Type,		TKCPyType::Dict,	"Dict",		true,		false	},
{	&PyFile_Type,		TKCPyType::File,	"File",		false,		true	},
{	&PyFloat_Type,		TKCPyType::Float,	"Float",	false,		true	},
{	&PyFrame_Type,		TKCPyType::Frame,	"Frame",	true,		false	},
{	&PyFunction_Type,	TKCPyType::Function,	"Function",	true,		true	},
{	&PyInstance_Type,	TKCPyType::Instance,	"Instance",	true,		true	},
{	&PyInt_Type,		TKCPyType::Int,		"Int",		false,		true	},
{	&PyList_Type,		TKCPyType::List,	"List",		true,		false	},
{	&PyLong_Type,		TKCPyType::Long,	"Long",		false,		true	},
{	&PyMethod_Type,		TKCPyType::Method,	"Method",	false,		true	},
{	&PyModule_Type,		TKCPyType::Module,	"Module",	true,		true	},
{	&PyRange_Type,		TKCPyType::Range,	"Range",	false,		true	},
{	&PySlice_Type,		TKCPyType::Slice,	"Slice",	false,		false	},
{	&PyString_Type,		TKCPyType::String,	"String",	false,		true	},
{	&PyTuple_Type,		TKCPyType::Tuple,	"Tuple",	true,		false	},
{	&PyType_Type,		TKCPyType::Type,	"Type",		false,		true	},
#ifdef	_PYTHON2
{	&PyCell_Type,		TKCPyType::Cell,	"Cell",		false,		false	},
{	&PyUnicode_Type,	TKCPyType::Unicode,	"Uincode",	false,		false	},
#endif	// _PYTHON2
{	0,			TKCPyType::Null,	0,		false,		false	},
}	;

static	TKCPyType typeNull    =
{	0,			TKCPyType::Null,	"Null",		false,		false	} ;
static	TKCPyType typeUnknown =	
{	0,			TKCPyType::Unknown,	"Unknown",	false,		false	} ;
static	TKCPyType typeNone    =
{	0,			TKCPyType::Nothing,	"None",		false,		true	} ;


/*  ------------------------------------------------------------------  */


/*  TKCPyTracePoint							*/
/*  TKCPyTracePoint							*/
/*		: Constructor for tracepoint				*/
/*  pyObj	: PyObject *	  : Code or module object		*/
/*  userPtr	: void *	  : Used supplied pointer		*/
/*  lineNo	: uint		  : Line number for module		*/
/*  (returns)	: TKCPyTracePoint :					*/

TKCPyTracePoint::TKCPyTracePoint
	(	PyObject	*pyObj,
		void		*userPtr,
		uint		lineNo
	)
	:
	pyObj	(pyObj),
	userPtr	(userPtr),
	lineNo	(lineNo)
{
	if (PyModule_Check(pyObj)) fName = PyModule_GetFilename (pyObj) ;
}


/*  ------------------------------------------------------------------  */

TKCPyCookie::~TKCPyCookie ()
{
}

/*  ------------------------------------------------------------------  */


#if	! __PY22PLUS

/*  The first set of trace hook functions is for Python prior to	*/
/*  version 2.2, where the interface is at the python level.		*/

/*  funcTraceHook: Python trace hook function				*/
/*  self	 :  PyObject *	:					*/
/*  args	 :  PyObject *	: Argument tuple			*/
/*  (returns)	 :  PyObject *	: Always Py_None			*/

static	PyObject *funcTraceHook
	(	 PyObject	*,
		 PyObject	*args
	)
{
//	fprintf	(stderr, "funcTraceHook called ......\n") ;
	return	TKCPyDebugBase::self()->funcTraceHook (args) ;
}

/*  lineTraceHook: Python trace hook function				*/
/*  self	 :  PyObject *	:					*/
/*  args	 :  PyObject *	: Argument tuple			*/
/*  (returns)	 :  PyObject *	: Always Py_None			*/

static	PyObject *lineTraceHook
	(	 PyObject	*,
		 PyObject	*args
	)
{
//	fprintf	(stderr, "lineTraceHook called ......\n") ;
	return	TKCPyDebugBase::self()->lineTraceHook (args) ;
}

/*  profTraceHook: Python profile hook function				*/
/*  self	 :  PyObject *	:					*/
/*  args	 :  PyObject *	: Argument tuple			*/
/*  (returns)	 :  PyObject *	: Always Py_None			*/

static	PyObject *profTraceHook
	(	 PyObject	*,
		 PyObject	*args
	)
{
//	fprintf	(stderr, "profTraceHook called ......\n") ;
	return	TKCPyDebugBase::self()->profTraceHook (args) ;
}



static	PyMethodDef __funcTraceHook =
{
	(char *)"funcTraceHook",
	funcTraceHook,
	METH_VARARGS,
	(char *)0
}	;
static	PyMethodDef __lineTraceHook =
{
	(char *)"lineTraceHook",
	lineTraceHook,
	METH_VARARGS,
	(char *)0
}	;
static	PyMethodDef __profTraceHook =
{
	(char *)"profTraceHook",
	profTraceHook,
	METH_VARARGS,
	(char *)0
}	;

/*  setFuncTraceHook							*/
/*		: Set function trace hook				*/
/*  (returns)	: void		:					*/

static	void	setFuncTraceHook ()
{
	PyObject	  *argvec = PyTuple_New (1) ;

	Py_INCREF	  (pyTKCDebFuncTrace) ;

	PyTuple_SetItem   (argvec, 0, pyTKCDebFuncTrace) ;
	PyEval_CallObject (pySysSetTrace, argvec) ;

	Py_DECREF	  (argvec) ;

	fprintf
	(	stderr,
		"***** TKCDebug function tracer installed [%p]\n",
		(void *)pyTKCDebFuncTrace
	)	;
}

/*  clearFuncTraceHook							*/
/*		: Clear function trace hook				*/
/*  (returns)	: void		:					*/

static	void	clearFuncTraceHook ()
{
	PyObject	  *argvec = PyTuple_New (1) ;

	Py_INCREF	  (Py_None) ;

	PyTuple_SetItem   (argvec, 0, Py_None)    ;
	PyEval_CallObject (pySysSetTrace, argvec) ;

	Py_DECREF	  (argvec) ;

	fprintf	(stderr, "***** TKCDebug function tracer removed\n") ;
}

/*  setProfTraceHook							*/
/*		: Set profile trace hook				*/
/*  (returns)	: void		:					*/

static	void	setProfTraceHook ()
{
	PyObject	  *argvec = PyTuple_New (1) ;

	Py_INCREF	  (pyTKCDebProfTrace) ;

	PyTuple_SetItem   (argvec, 0, pyTKCDebProfTrace) ;
	PyEval_CallObject (pySysSetProfile, argvec) ;

	Py_DECREF	  (argvec) ;

	fprintf
	(	stderr,
		"***** TKCDebug profile tracer installed [%p]\n",
		(void *)pyTKCDebProfTrace
	)	;
}

/*  clearProfTraceHook							*/
/*		: Clear profile trace hook				*/
/*  (returns)	: void		:					*/

static	void	clearProfTraceHook ()
{
	PyObject	  *argvec = PyTuple_New (1) ;

	Py_INCREF	  (Py_None) ;

	PyTuple_SetItem   (argvec, 0, Py_None)      ;
	PyEval_CallObject (pySysSetProfile, argvec) ;

	Py_DECREF	  (argvec) ;

	fprintf	(stderr, "TKCDebug profile tracer removed\n") ;
}

/*  TKCPyDebugBase							*/
/*  enable	: Enable trace points					*/
/*  (returns)	: void		:					*/

void	TKCPyDebugBase::enable ()
{
	if (pySysSetTrace   == 0) return ;
	if ((debugCnt += 1) != 1) return ;

	setFuncTraceHook () ;
}

/*  TKCPyDebugBase							*/
/*  disable	: Enable trace points					*/
/*  (returns)	: void		:					*/

void	TKCPyDebugBase::disable ()
{
	if ((pySysSetTrace == 0) || (debugCnt == 0)) return ;
	if ((debugCnt -= 1) != 0) return ;

	clearFuncTraceHook () ;
}

/*  TKCPyDebugBase							*/
/*  trapExceptions							*/
/*		: Toggle exception trapping				*/
/*  trap	: bool		: True to trap				*/
/*  (returns)	: void;		:					*/

void	TKCPyDebugBase::trapExceptions
	(	bool	trap
	)
{
	if (trap)
		setProfTraceHook   () ;
	else	clearProfTraceHook () ;
}

#endif


/*  ------------------------------------------------------------------  */

/*  The second set is for version 2.2 onwards where there is a direct	*/
/*  C interface. This is *all* much easier!				*/

#if	__PY22PLUS

/*  pythonTraceHook							*/
/*		: Low level hook function				*/
/*  object	: PyObject *	  : Not used				*/
/*  frame	: PyFrameObject * : Frame at point of call		*/
/*  what	: int		  : Code for trace event		*/
/*  arg		: PyObject *	  : Other arguments			*/
/*  (returns)	: int		  : Non-zero to abort execution		*/

static	int	pythonTraceHook
	(	PyObject	*,
		PyFrameObject	*frame,
		int		what,
		PyObject	*arg
	)
{
//	fprintf	(stderr, "pythonTraceHook called ......\n") ;
	return	TKCPyDebugBase::self()->pythonTraceHook (frame, what, arg) ;
}

/*  TKCPyDebugBase							*/
/*  enable	: Enable trace points					*/
/*  (returns)	: void		:					*/

void	TKCPyDebugBase::enable ()
{
	if ((debugCnt += 1) != 1) return ;

	PyEval_SetTrace	(::pythonTraceHook, 0) ;
	fprintf	(stderr, "***** TKCDebug tracer installed\n") ;
}

/*  TKCPyDebugBase							*/
/*  disable	: Enable trace points					*/
/*  (returns)	: void		:					*/

void	TKCPyDebugBase::disable ()
{
	if ((debugCnt == 0)	) return ;
	if ((debugCnt -= 1) != 0) return ;

	PyEval_SetTrace	(0, 0) ;
	fprintf	(stderr, "***** TKCDebug tracer removed\n") ;
}

/*  TKCPyDebugBase							*/
/*  trapExceptions							*/
/*		: Toggle exception trapping				*/
/*  trap	: bool		: True to trap				*/
/*  (returns)	: void		:					*/

void	TKCPyDebugBase::trapExceptions
	(	bool	trap
	)
{
	/* For Python 2.2 onwards, exceptions are reported via the	*/
	/* normal trace hook, which must be installed.			*/
	if (trap)
		enable  () ;
	else	disable	() ;
}

#endif


/*  TKCPyDebugBase							*/
/*  inDebugger	: Called as debugger is entered/exited			*/
/*  enter	: bool		: Entering				*/
/*  (returns)	: void		:					*/

void	TKCPyDebugBase::inDebugger
	(	bool	enter
	)
{
	if (enter)
	{
		debugLevel += 1 ;
		return	;
	}

	if (debugLevel == 0)
	{
		fprintf	(stderr, "TKCPyDebugBase::inDebugger: ***ZERO***\n") ;
		return	;
	}

	debugLevel -= 1 ;
}


/*  ------------------------------------------------------------------  */

TKCPyDebugBase		*TKCPyDebugBase::debugger 	;
TKCPyTraceOpt		TKCPyDebugBase::traceOpt	;
QList<TKCPyTracePoint>	TKCPyDebugBase::tracePoints	;



TKCPyTracePoint
	*TKCPyDebugBase::findTracePoint
	(	PyObject	*pyObj,
		uint		lineNo
	)
{
	for (uint idx = 0 ; idx < tracePoints.count() ; idx += 1)
	{	TKCPyTracePoint *tp = tracePoints.at(idx) ;
		if ((tp->pyObj == pyObj) && (tp->lineNo == lineNo))
			return	tp ;
	}

	return	0 ;
}

/*  TKCPyDebugBase							*/
/*  TKCPyDebugBase: Base python debugger class constructor		*/
/*  (returns	  : TKCPyDebugBase :					*/

TKCPyDebugBase::TKCPyDebugBase ()
{
	/* See if the debuger has already been instantiated. We do not	*/
	/* want more than one ...					*/
	if (debugger != 0)
	{	TKCPyDebugError ("Attempt to create multiple debuggers") ;
		return	;
	}

	debugger = this ;
}

/*  TKCPyDebugBase							*/
/*  ~TKCPyDebugBase: Base python debugger class destructor		*/
/*  (returns	   :		 :					*/

TKCPyDebugBase::~TKCPyDebugBase ()
{
	if (debugger == this) debugger = 0 ;
}

/*  TKCPyDebugBase							*/
/*  init	: Initialise debugger					*/
/*  (returns)	: QString	: Error message or QString::null if OK	*/

QString	TKCPyDebugBase::init ()
{
	if (inited)
		return	"Debugger already initialised" ;

	inited		= true		;
	traceOpt	= OptContinue	;

#if	! __PY22PLUS
	PyObject	*_pyTKCDebModule	;
	PyObject	*_pyTKCDebDict		;
	PyObject	*_pyTKCDebFuncTrace	;
	PyObject	*_pyTKCDebLineTrace	;
	PyObject	*_pyTKCDebProfTrace	;
	PyObject	*_pySysModule		;
	PyObject	*_pySysSetTrace		;
	PyObject	*_pySysSetProfile	;


	if ((_pyTKCDebModule	= PyImport_AddModule ((char *)"TKCPyDebugBase")) == 0)
		return	"Cannot add module TKCDebug" ;
	if ((_pyTKCDebDict	= PyModule_GetDict   (_pyTKCDebModule))     == 0)
		return	"Cannot get TKCDebug dictionary" ;
	if ((_pyTKCDebFuncTrace	= PyCFunction_New    (&__funcTraceHook, 0)) == 0)
		return	"Cannot create function trace hook" ;
	if ((_pyTKCDebLineTrace	= PyCFunction_New    (&__lineTraceHook, 0)) == 0)
		return	"Cannot create line trace hook" ;
	if ((_pyTKCDebProfTrace	= PyCFunction_New    (&__profTraceHook, 0)) == 0)
		return	"Cannot create exception trace hook" ;

	Py_INCREF	(_pyTKCDebModule   ) ;
	Py_INCREF	(_pyTKCDebDict     ) ;
	Py_INCREF	(_pyTKCDebFuncTrace) ;
	Py_INCREF	(_pyTKCDebLineTrace) ;
	Py_INCREF	(_pyTKCDebProfTrace) ;

	PyDict_SetItemString (_pyTKCDebDict, (char *)"funcTraceHook", _pyTKCDebFuncTrace) ;
	PyDict_SetItemString (_pyTKCDebDict, (char *)"lineTraceHook", _pyTKCDebLineTrace) ;
	PyDict_SetItemString (_pyTKCDebDict, (char *)"profTraceHook", _pyTKCDebProfTrace) ;


	if ((_pySysModule     = PyImport_ImportModule  ((char *)"sys")) == 0)
		return	"Unable to access Python sys module" ;
	if ((_pySysSetTrace   = PyObject_GetAttrString (_pySysModule, (char *)"settrace"  )) == 0)
		return	"Unable to locate sys.settrace function" ;
	if ((_pySysSetProfile = PyObject_GetAttrString (_pySysModule, (char *)"setprofile")) == 0)
		return	"Unable to locate sys.setprofile function" ;
	if (!PyCallable_Check (_pySysSetTrace))
		return	"Located sys.settrace is not callable" ;

	pyTKCDebModule	  = _pyTKCDebModule	;
	pyTKCDebDict	  = _pyTKCDebDict   	;
	pyTKCDebFuncTrace = _pyTKCDebFuncTrace  ;
	pyTKCDebLineTrace = _pyTKCDebLineTrace  ;
	pyTKCDebProfTrace = _pyTKCDebProfTrace  ;
	pySysModule	  = _pySysModule	;
	pySysSetTrace	  = _pySysSetTrace  	;
	pySysSetProfile	  = _pySysSetProfile  	;
#endif

#if	__PY22PLUS
	pyMsgExcept = PyString_FromString ("exception") ;
	pyMsgCall   = PyString_FromString ("call"     ) ;
	pyMsgLine   = PyString_FromString ("line"     ) ;
#endif

	fprintf	(stderr, "TKCDebug initialised\n") ;
	return	 QString::null ;
}

/*  TKCPyDebugBase							*/
/*  setTracePoint: Set trace point					*/
/*  pyObj	 : PyObject *	: Code/Module to trace			*/
/*  userPtr	 : void *	: User supplied pointer			*/
/*  lineNo	 : uint		: Line number				*/
/*  (returns)	 : void		:					*/

void	TKCPyDebugBase::setTracePoint
	(	PyObject	*pyObj,
		void		*userPtr,
		uint		lineNo
	)
{
	if (PyModule_Check (pyObj) || PyCode_Check (pyObj))
		if (findTracePoint (pyObj, lineNo) == 0)
		{
			fprintf	(stderr, "Set trace: %08x:%d\n", (uint)pyObj, lineNo) ;
			tracePoints.append (new TKCPyTracePoint (pyObj, userPtr, lineNo)) ;
			enable	() ;
		}
}

/*  TKCPyDebugBase							*/
/*  clearTracePoint							*/
/*		 : Clear trace point					*/
/*  pyObj	 : PyObject *	: Code/Module to clear			*/
/*  lineNo	 : uint		: Line number				*/
/*  (returns)	 : void		:					*/

void	TKCPyDebugBase::clearTracePoint
	(	PyObject	*pyObj,
		uint		lineNo
	)
{
	TKCPyTracePoint *tp ;

	if (PyModule_Check (pyObj) || PyCode_Check (pyObj))
		if ((tp = findTracePoint (pyObj, lineNo)) != 0)
		{
			fprintf	(stderr, "Rem trace: %08x:%d\n", (uint)pyObj, lineNo) ;
			tracePoints.remove (tp) ;
			delete	tp ;
			disable	() ;
		}
}

/*  TKCPyDebugBase							*/
/*  codeTraced	: See if code object is traced				*/
/*  pyCode	: PyCodeObject *    : Code object			*/
/*  (returns)	: TKCPyTracePoint * : Trace point data or null		*/

TKCPyTracePoint
	*TKCPyDebugBase::codeTraced
	(	PyCodeObject	*pyCode
	)
{
	TKCPyTracePoint *tp ;
	for (uint idx = 0 ; idx < tracePoints.count() ; idx += 1)
		if ((tp = tracePoints.at(idx))->pyObj == (PyObject *)pyCode)
			return	tp ;

	return	0 ;
}

/*  TKCPyDebugBase							*/
/*  moduleTraced: See if module containing code is traced		*/
/*  pyCode	: PyCodeObject *    : Code object			*/
/*  (returns)	: TKCPyTracePoint * : Trace point data or null		*/

TKCPyTracePoint
	*TKCPyDebugBase::moduleTraced
	(	PyCodeObject	*pyCode
	)
{
	QString	fName	= getPythonString (pyCode->co_filename) ;

//	fprintf	(stderr, "moduleTraced(%s): ", (const char *)fName) ;

	for (uint idx = 0 ; idx < tracePoints.count() ; idx += 1)
	{	TKCPyTracePoint *tp = tracePoints.at(idx) ;
		if ((tp->fName == fName) && (tp->lineNo != 0))
		{
//			fprintf	(stderr, "at line %u\n", tp->lineNo) ;
			return	tp	;
		}
	}

//	fprintf	(stderr, "no\n") ;
	return	0 ;
}

/*  TKCPyDebugBase							*/
/*  moduleTraced: See if module is traced at specified line		*/
/*  pyCode	: PyCodeObject *    : Code object			*/
/*  lineNo	: uint		    : Line number			*/
/*  (returns)	: TKCPyTracePoint * : Trace point data or null		*/

TKCPyTracePoint
	*TKCPyDebugBase::moduleTraced
	(	PyCodeObject	*pyCode,
		uint		lineNo
	)
{
	QString	fName	= getPythonString (pyCode->co_filename) ;

//	fprintf	(stderr, "moduleTraced(%s@%u): ", (const char *)fName, lineNo) ;

	for (uint idx = 0 ; idx < tracePoints.count() ; idx += 1)
	{	TKCPyTracePoint *tp = tracePoints.at(idx) ;
		if ((tp->fName == fName) && (tp->lineNo == lineNo))
		{
//			fprintf	(stderr, "at line %u\n", tp->lineNo) ;
			return	tp	;
		}
	}

//	fprintf	(stderr, "no\n") ;
	return	0 ;
}

/*  TKCPyDebugBase							*/
/*  getPythonType: Get type information for specified python object	*/
/*  pyObj	 : PyObject *	: The object				*/
/*  (returns)	 : TKCPyType	: Type information			*/

const	TKCPyType
	*TKCPyDebugBase::getPythonType
	(	PyObject	*pyObj
	)
{
	if (pyObj == 0      ) return &typeNull ;
	if (pyObj == Py_None) return &typeNone ;

	for (TKCPyType *tp = &typeMap[0] ; tp->typeObj != 0 ; tp += 1)
		if (pyObj->ob_type == tp->typeObj)
			return	tp ;

	return	&typeUnknown	;
}

/*  TKCPyDebugBase							*/
/*  funcTraceHook: Python function call tracing hook			*/
/*  pyFrame	 : PyObject *	 : Call frame				*/
/*  pyMsg	 : PyObject *	 : Trace message			*/
/*  userPtr	 : void *	 : User supplied pointer		*/
/*  (returns)	 : TKCPyTraceOpt : Trace continuation code		*/

TKCPyTraceOpt
	TKCPyDebugBase::funcTraceHook
	(	PyObject	*,
		PyObject	*,
		PyObject	*,
		void		*
	)
{
	fprintf	(stderr, "TKCPyDebugBase::funcTraceHook called ......\n") ;
	return	OptContinue ;
}

/*  TKCPyDebugBase							*/
/*  lineTraceHook: Python line tracing hook				*/
/*  pyFrame	 : PyObject *	 : Call frame				*/
/*  pyMsg	 : PyObject *	 : Trace message			*/
/*  pyArg	 : PyObject *	 : Trace arguments			*/
/*  userPtr	 : void *	 : User supplied pointer		*/
/*  (returns)	 : TKCPyTraceOpt : Trace continuation code		*/

TKCPyTraceOpt
	TKCPyDebugBase::lineTraceHook
	(	PyObject	*,
		PyObject	*,
		PyObject	*,
		void		*
	)
{
	fprintf	(stderr, "TKCPyDebugBase::lineTraceHook called ......\n") ;
	return	OptContinue ;
}

/*  TKCPyDebugBase							*/
/*  profTraceHook: Python exception tracing hook			*/
/*  pyFrame	 : PyObject *	 : Call frame				*/
/*  pyMsg	 : PyObject *	 : Trace message			*/
/*  pyArg	 : PyObject *	 : Trace arguments			*/
/*  userPtr	 : void *	 : User supplied pointer		*/
/*  (returns)	 : TKCPyTraceOpt : Trace continuation code		*/

TKCPyTraceOpt
	TKCPyDebugBase::profTraceHook
	(	PyObject	*,
		PyObject	*,
		PyObject	*,
		void		*
	)
{
	fprintf	(stderr, "TKCPyDebugBase::profTraceHook called ......\n") ;
	return	OptContinue ;
}

TKCPyTraceOpt
	TKCPyDebugBase::doDebugHook
	(	PY_FRAME_TYPE	*,
		const char	*
	)
{
	return	OptContinue ;
}

#if	! __PY22PLUS
/*  TKCPyDebugBase							*/
/*  funcTraceHook: Python trace hook function				*/
/*  args	 :  PyObject *	: Argument tuple			*/
/*  (returns)	 :  PyObject *	: Always Py_None			*/

PyObject*TKCPyDebugBase::funcTraceHook
	(	 PyObject	*args
	)
{
	PyObject	*pyFrame = PyTuple_GetItem (args, 0) ;
	PyObject	*pyMsg   = PyTuple_GetItem (args, 1) ;
	PyObject	*pyArg   = PyTuple_GetItem (args, 2) ;

	PyCodeObject	*pyCode  = ((PyFrameObject *)pyFrame)->f_code ;
	PyObject	*pyRes	 = Py_None ;
	TKCPyTracePoint *tp 	 ;

	fprintf
	(	stderr,
		"funcTraceHook(%s:%s)\n",
		(const char *)getPythonString(pyCode->co_name),
		(const char *)getPythonString(pyMsg)
 	)	;

	/* If this code is not traced then ignore. This happens since	*/
	/* the function entry hook is global. Otherwise, invoke the	*/
	/* debug object hook and see what to do on return ...		*/
	if ((tp = codeTraced (pyCode)) != 0)
		switch (traceOpt = funcTraceHook (pyFrame, pyMsg, pyArg, tp->userPtr))
		{
			case OptAbort :
				/* Aborting execution, returning zero	*/
				/* signals this.			*/
				pyRes	= 0 ;
				break	;

			case OptStep  :
				/* Line stepping requested, we return	*/
				/* the line trace function.		*/
				pyRes	= pyTKCDebLineTrace ;
				break	;

			default	:
				/* Anything else continues function	*/
				/* tracing.				*/
				pyRes	= Py_None	;
				break	;
		}

	/* If the module is traced (ie., there are line tracepoints)	*/
	/* then always continue (unless aborting).			*/
	if (pyRes != 0)
		if (moduleTraced (pyCode) != 0)
			pyRes = pyTKCDebLineTrace ;

	Py_XINCREF(pyRes) ;
	return	pyRes	  ;
}

/*  TKCPyDebugBase							*/
/*  lineTraceHook: Python trace hook function				*/
/*  args	 :  PyObject *	: Argument tuple			*/
/*  (returns)	 :  PyObject *	: Always Py_None			*/

PyObject*TKCPyDebugBase::lineTraceHook
	(	 PyObject	*args
	)
{
	PyObject	*pyFrame = PyTuple_GetItem (args, 0) ;
	PyObject	*pyMsg   = PyTuple_GetItem (args, 1) ;
	PyObject	*pyArg   = PyTuple_GetItem (args, 2) ;

	PyCodeObject	*pyCode  = ((PyFrameObject *)pyFrame)->f_code ;
	PyObject	*pyRes	 = Py_None ;

	fprintf
	(	stderr,
		"lineTraceHook(%s:%d:%s)\n",
		(const char *)getPythonString(pyCode->co_name),
		((PyFrameObject *)pyFrame)->f_lineno,
		(const char *)getPythonString(pyMsg)
	)	;

	TKCPyTracePoint *tp 	 = moduleTraced (pyCode, ((PyFrameObject *)pyFrame)->f_lineno) ;

	/* Take the tracepoint if either we are single stepping or if	*/
	/* the module is tracepointed at the current line.		*/
	if ((traceOpt == OptStep) || (tp != 0))
		switch (traceOpt = lineTraceHook (pyFrame, pyMsg, pyArg, tp == 0 ? 0 : tp->userPtr))
		{
			case OptAbort :
				/* Aborting execution, returning zero	*/
				/* signals this.			*/
				pyRes	= 0 ;
				break	;

			case OptStep  :
				/* Line stepping requested, we return	*/
				/* the line trace function.		*/
				pyRes	= pyTKCDebLineTrace ;
				break	;

			default	:
				/* Anything else continues function	*/
				/* tracing.				*/
				pyRes	= Py_None ;
				break	;
		}

	/* If the module is traced (ie., there are line tracepoints)	*/
	/* then always continue (unless aborting).			*/
	if (pyRes != 0)
		if (moduleTraced (pyCode) != 0)
			pyRes = pyTKCDebLineTrace ;

	Py_XINCREF(pyRes) ;
	return	pyRes	  ;
}

/*  TKCPyDebugBase							*/
/*  profTraceHook: Python trace hook function				*/
/*  args	 :  PyObject *	: Argument tuple			*/
/*  (returns)	 :  PyObject *	: Always Py_None			*/

PyObject*TKCPyDebugBase::profTraceHook
	(	 PyObject	*args
	)
{
	PyObject	*pyFrame = PyTuple_GetItem (args, 0) ;
	PyObject	*pyMsg   = PyTuple_GetItem (args, 1) ;
	PyObject	*pyArg   = PyTuple_GetItem (args, 2) ;

	PyCodeObject	*pyCode  = ((PyFrameObject *)pyFrame)->f_code ;
	fprintf
	(	stderr,
		"profTraceHook(%s:%d:%s)\n",
		(const char *)getPythonString(pyCode->co_name),
		((PyFrameObject *)pyFrame)->f_lineno,
		(const char *)getPythonString(pyMsg)
	)	;

	/* If we are already inside the debugger, then ignore any	*/
	/* exceptions. It seems that these case occur as part of normal	*/
	/* operations in some cases.					*/
	if (debugLevel != 0)
	{
		Py_XINCREF(Py_None) ;
		return	   Py_None  ;
	}

	/* Up to python 2.1.1, exceptions are reported back through	*/
	/* the profiling trace hook .....				*/
	if (getPythonString(pyMsg) == "exception")
		profTraceHook (pyFrame, pyMsg, pyArg, 0) ;

	Py_XINCREF(Py_None) ;
	return	   Py_None  ;
}

#endif

#if	__PY22PLUS

/*  TKCPyDebugBase							*/
/*  pythonTraceHook							*/
/*		: Low level hook function				*/
/*  frame	: PyFrameObject * : Frame at point of call		*/
/*  what	: int		  : Code for trace event		*/
/*  arg		: PyObject *	  : Other arguments			*/
/*  (returns)	: int		  : Non-zero to abort execution		*/

int	TKCPyDebugBase::pythonTraceHook
	(	PyFrameObject	*pyFrame,
		int		what,
		PyObject	*pyArg
	)
{
//	fprintf
//	(	stderr,
//		"pythonTraceHook: %d/%s\n",
//		what,
//		what == PyTrace_CALL	  ? "call"	:
//		what == PyTrace_EXCEPTION ? "exception" :
//		what == PyTrace_LINE	  ? "line"	:
//		what == PyTrace_RETURN    ? "return"	: "unknown"
//	)	;

	/* First case, exceptions. Pass these up via the profTraceHook	*/
	/* method (misnamed for Python 2.2 :)				*/
	if (what == PyTrace_EXCEPTION)
	{
		/* Ignore the exception if we are already inside the	*/
		/* debugger, to avoid recursion problems.		*/
		if (debugLevel == 0)
			profTraceHook ((PyObject *)pyFrame, pyMsgExcept, pyArg, 0) ;
		return	0 ;
	}

	/* Function call. We are only interested in this if the the	*/
	/* function code is being traced.				*/
	if (what == PyTrace_CALL)
	{
		PyCodeObject	*pyCode = pyFrame->f_code     ;
		TKCPyTracePoint *tp	= codeTraced (pyCode) ;

		if (tp == 0) return 0 ;

		traceOpt = funcTraceHook
			   (	(PyObject *)pyFrame,
				pyMsgCall,
				pyArg,
				tp->userPtr
			   )	;
		return	traceOpt == OptAbort ? 1 : 0 ;
	}

	/* Line tracking. Interesting if either execution stepping is	*/
	/* in action or the specific line is breakpointed.		*/
	if (what == PyTrace_LINE)
	{
		PyCodeObject	*pyCode  = pyFrame->f_code   ;
		TKCPyTracePoint *tp 	 = moduleTraced (pyCode, pyFrame->f_lineno) ;

		if ((traceOpt != OptStep) && (tp == 0)) return 0 ;

		traceOpt = lineTraceHook
			   (	(PyObject *)pyFrame,
				pyMsgLine,
				pyArg,
				tp == 0 ? 0 : tp->userPtr
			   )	;
		return	traceOpt == OptAbort ? 1 : 0 ;
	}

	return	0 ;
}

#endif

/*  TKCPyDebugBase							*/
/*  debugHook	: User debug call					*/
/*  frame	: PyObject *	: Callers frame				*/
/*  msg		: cchar *	: Message				*/
/*  (returns)	: PyObject *	: Null to abort				*/

PyObject
	*TKCPyDebugBase::debugHook
	(	PY_FRAME_TYPE	*frame,
		const char	*msg
	)
{
	fprintf
	(	stderr,
		"TKCPyDebugBase::goDebugHook [%s]\n",
		msg
	)	;

	PyObject *pyRes	  ;

	traceOpt = doDebugHook (frame, msg) ;
	pyRes	 = traceOpt == OptAbort ? 0 : Py_None ;

	Py_XINCREF(pyRes) ;
	return	pyRes	  ;
}

/*  TKCPyDebugBase							*/
/*  getPythonString							*/
/*		: Get string for python object				*/
/*  pyobj	: PyObject *	: Python object				*/
/*  returns	: QString	: String				*/

QString	TKCPyDebugBase::getPythonString
	(	PyObject	*pyobj
	)
{
	if ((pyobj == 0) || (pyobj == Py_None))
		return	"<None>" ;

	if (PyString_Check (pyobj))
		return	PyString_AsString (pyobj) ;

	QString	 res	;
	PyObject *str	= PyObject_Str (pyobj) ;
	res	= PyString_AsString (str) ;
	Py_DECREF (str) ;
	return	res	;
}


void	TKCPyDebugBase::loadDictionary
	(	PyObject		*pyDict,
		QDict<TKCPyValue>	&tkcDict
	)
{
	if (PyDict_Check (pyDict)) 
	{
		PyObject *pkey	 ;
		PyObject *pvalue ;
		int	 dictPos = 0 ;

		while (PyDict_Next (pyDict, &dictPos, &pkey, &pvalue))
			tkcDict.insert
			(	getPythonString(pkey),
				TKCPyValue::allocValue (pvalue)
			)	;
	}
}

void	TKCPyDebugBase::getModuleDict
	(	QDict<TKCPyValue>	&tkcDict
	)
{
	loadDictionary (PyImport_GetModuleDict(), tkcDict) ;
}

void	TKCPyDebugBase::getModuleDict
	(	PyObject		*pyObj,
		QDict<TKCPyValue>	&tkcDict
	)
{
	if (PyModule_Check (pyObj)) 
		loadDictionary (PyModule_GetDict (pyObj), tkcDict) ;
}


static	QString		scanDictForObject   (PyObject *, PyObject *) ;
static	QString		scanClassForObject  (PyObject *, PyObject *) ;
static	QString		scanModuleForObject (PyObject *, PyObject *) ;
static	QList<PyObject>	scanList ;


/*  scanDictForObject							*/
/*		: Scan a dictionary for a specified object		*/
/*  pyDict	: PyObject *	: Dictionary				*/
/*  pyObject	: PyObject *	: Object to find			*/
/*  (returns)	: QString	: Entry name or QString::null		*/

static	QString	scanDictForObject
	(	PyObject	*pyDict,
		PyObject	*pyObject
	)
{
	PyObject  *pyKey   ;
	PyObject  *pyValue ;
	QString	  name	   ;
	int	  dictPos  = 0 ;

	/* See if we have been here before, in which case there is no	*/
	/* point in going there again. This also prevents infinite	*/
	/* recursion on circular structures.				*/
	if (scanList.find (pyDict) >= 0)
		return QString::null ;

	scanList.append (pyDict) ;

	/* Scan the dictionary, looking for the object. In certain	*/
	/* cases, we will recurse down ...				*/
	while (PyDict_Next (pyDict, &dictPos, &pyKey, &pyValue))
	{
		if (pyValue == pyObject)
			return	TKCPyDebugBase::getPythonString (pyKey) ;

		if (PyModule_Check (pyValue))
			if ((name = scanModuleForObject (pyValue, pyObject)) != QString::null)
				return	TKCPyDebugBase::getPythonString(pyKey) + "." + name ;

		if (PyClass_Check  (pyValue))
			if ((name = scanClassForObject  (pyValue, pyObject)) != QString::null)
				return	TKCPyDebugBase::getPythonString(pyKey) + "." + name ;

		if (PyDict_Check   (pyValue))
			if ((name = scanDictForObject   (pyValue, pyObject)) != QString::null)
				return	TKCPyDebugBase::getPythonString(pyKey) + "." + name ;

		if (PyFunction_Check (pyValue))
			if (((PyFunctionObject *)pyValue)->func_code == pyObject)
				return	TKCPyDebugBase::getPythonString(pyKey) ;
	}

	return	QString::null ;
}

/*  scanModuleForObject							*/
/*		: Scan a module for a specified object			*/
/*  pyModule	: PyObject *	: Module				*/
/*  pyObject	: PyObject *	: Object to find			*/
/*  (returns)	: QString	: Entry name or QString::null		*/

static	QString	scanModuleForObject
	(	PyObject	*pyModule,
		PyObject	*pyObject
	)
{
	PyObject  *modDict = pyModule == 0 ?
				PyImport_GetModuleDict	() :
				PyModule_GetDict 	(pyModule) ;

	return	  scanDictForObject (modDict, pyObject) ;
}

/*  scanClassForObject							*/
/*		: Scan a class for a specified object			*/
/*  pyClass	: PyObject *	: Class					*/
/*  pyObject	: PyObject *	: Object to find			*/
/*  (returns)	: QString	: Entry name or QString::null		*/

static	QString	scanClassForObject
	(	PyObject	*pyClass,
		PyObject	*pyObject
	)
{
	PyObject *pyDict = ((PyClassObject *)pyClass)->cl_dict ;
	return	  scanDictForObject (pyDict, pyObject) ;
}


/*  TKCPyDebugBase							*/
/*  getObjectName: Try to find path name for object			*/
/*  pyObject	 : PyObject *	: The object				*/
/*  (returns)	 : QString	: Name or null of not found		*/

QString	TKCPyDebugBase::getObjectName
	(	PyObject	*pyObject
	)
{
	QString	res	= scanModuleForObject (0, pyObject) ;
	scanList.clear () ;
	return	res	;
}



