/*
 * Copyright (c) 1998 - 2001 Phil Thompson <phil@river-bank.demon.co.uk>
 *
 * The code generator module for SIP.
 */


#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include "sip.h"


/* Macro expansions for template code. */

#define	MACRO_ESCAPE	'$'
#define	MACRO_CLASS	'C'			/* Class name. */
#define	MACRO_SOURCES	'S'			/* Source files. */
#define	MACRO_OBJECTS	'O'			/* Object files. */
#define	MACRO_CPPSUFF	'c'			/* C++ file suffix. */
#define	MACRO_OBJSUFF	'o'			/* Object file suffix. */
#define	MACRO_CPPMODULE	'm'			/* C++ module name. */
#define	MACRO_PERCENT	'P'			/* Percent sign. */


/* The maximum number of nested version qualifiers in the generated Python. */

#define	MAX_VQ_STACK	10


/* Control what generateSingleArg() actually generates. */

typedef enum {
	Call,
	Declaration,
	Definition
} funcArgType;


/* This defined an entry in a consolidated list of Python classes. */

#define	PCV_CLASSVARS	(1 << 0)

typedef struct _pyClassVers {
	int			generated;	/* If it has been generated. */
	int			detail;		/* Detail flags. */
	int			combmeth;	/* Combined methods. */
	int			combnummeth;	/* Combined numeric methods. */
	int			combseqmeth;	/* Combined sequence methods. */
	int			combmapmeth;	/* Combined mapping methods. */
	classDef		*cd;		/* The class. */
	classList		*supers;	/* The unique superclasses. */
	versionOrList		*vol;		/* The consolidated version. */
	struct _pyClassVers	*next;		/* Next in the list. */
} pyClassVers;


static int currentLineNr;		/* Current output line number. */
static char *currentFileName;		/* Current output file name. */
static versionAnded vqstack[MAX_VQ_STACK];	/* Nested version qualifiers. */
static int vqstackptr = 0;		/* Version qualifier stack pointer. */


static void generateDocumentation(sipSpec *,char *);
static void generateMakefile(sipSpec *,char *,char *);
static void generatePython(sipSpec *,char *);
static void generateModuleProxy(sipSpec *,FILE *);
static void generateModuleProxyHeader(sipSpec *,char *);
static void generatePackageHeader(sipSpec *,char *);
static void generateCpp(sipSpec *,char *,char *);
static void generateVersionProgram(sipSpec *,char *,char *);
static void generateClassCpp(classDef *,sipSpec *,char *,char *);
static void generateClassVersCpp(classVersDef *,sipSpec *,FILE *);
static void generatePythonClassWrappers(sipSpec *,FILE *);
static void generatePythonClassVersWrapper(sipSpec *,pyClassVers *,pyClassVers *,FILE *);
static int generatePythonPushVersion(sipSpec *,versionOrList *,int,FILE *);
static void pushVersion(sipSpec *,versionAnded *,versionQual **,versionQual **,versionQual **);
static void popVersion();
static void generateCppVersionExpr(versionQual *,versionQual *,versionQual *,FILE *);
static void generatePythonClassWrapper(sipSpec *,pyClassVers *,int,FILE *);
static void generateClassFunctions(sipSpec *,classVersDef *,FILE *);
static void generateComplexCode(sipSpec *,classVersDef *,FILE *);
static void generateFunction(sipSpec *,memberDef *,overDef *,classVersDef *,classVersDef *,int,FILE *);
static void generateFunctionBody(sipSpec *,overDef *,classVersDef *,FILE *);
static void generatePythonConstructor(sipSpec *,classVersDef *,FILE *);
static void generateCppCodeBlock(sipSpec *,codeBlock *,char *,FILE *);
static void generateMacroCode(codeBlock *,char *,sipSpec *,char *,char *,FILE *);
static void generateMacroSource(sipSpec *,char *,FILE *);
static void generateMacroObject(sipSpec *,char *,FILE *);
static void generateUsedIncludes(classList *,FILE *);
static void generateClassHeader(sipSpec *,classDef *,char *);
static void generateWrapperClassDeclaration(sipSpec *,classVersDef *,FILE *);
static void generateLimitedWrapperClassDeclaration(sipSpec *,classVersDef *,FILE *);
static void generateResultType(argDef *,FILE *);
static void generateCallTidyUp(funcArgs *,FILE *);
static void generateArgs(funcArgs *,funcArgType,FILE *);
static void generateVariable(argDef *,int,int,FILE *);
static void generateValueType(argDef *,FILE *);
static void generateBasicType(argDef *,FILE *);
static void generateSingleArg(argDef *,int,funcArgType,FILE *);
static void generateType(argDef *,FILE *);
static void generateExpression(valueDef *,FILE *);
static void generateTupleBuilder(sipSpec *,funcArgs *,int,FILE *);
static void generateEmitter(sipSpec *,classVersDef *,visibleList *,FILE *);
static void generateVirtualHandlerDeclaration(overDef *,FILE *);
static void generateVirtualHandlers(sipSpec *,classVersDef *,FILE *);
static void generateVirtualHandler(sipSpec *,virtOverDef *,FILE *);
static void generateVirtualCatcher(classVersDef *,int,virtOverDef *,FILE *);
static void generateProtectedEnums(sipSpec *,classVersDef *,FILE *);
static void generateProtectedDeclarations(sipSpec *,classVersDef *,FILE *);
static void generateProtectedDefinitions(sipSpec *,classVersDef *,FILE *);
static void generateArgCallList(funcArgs *,FILE *fp);
static void generateConstructorCall(sipSpec *,classVersDef *,ctorDef *,int,FILE *);
static void generateHandleResult(sipSpec *,overDef *,int,FILE *);
static void generateFunctionCall(sipSpec *,classVersDef *,overDef *,int,FILE *);
static void generateVariableHandler(sipSpec *,classVersDef *,varDef *,FILE *);
static void generateObjToCppConversion(argDef *,char *,char *,FILE *);
static void generateObjConversion(varDef *,FILE *);
static void generateVarClassConversion(sipSpec *,varDef *,FILE *);
static void generateArgConvertors(sipSpec *,funcArgs *,int,int,char *,FILE *);
static void generateOredVersionStart(sipSpec *,versionOrList *,FILE *);
static void generateOredVersionEnd(sipSpec *,versionOrList *,FILE *);
static void generateVersionStart(sipSpec *,versionAnded *,FILE *);
static void generateVersionEnd(sipSpec *,versionAnded *,FILE *);
static void generateVoidPointers(sipSpec *,classVersDef *,varDef *,FILE *);
static void generateChars(sipSpec *,classVersDef *,varDef *,FILE *);
static void generateStrings(sipSpec *,classVersDef *,varDef *,FILE *);
static void generateEnums(sipSpec *,classVersDef *,enumDef *,FILE *);
static void generateLongs(sipSpec *,classVersDef *,varDef *,FILE *);
static void generateDoubles(sipSpec *,classVersDef *,varDef *,FILE *);
static void generateClasses(sipSpec *,classVersDef *,varDef *,FILE *);
static void generateAccessFunctions(sipSpec *,classVersDef *,varDef *,FILE *);
static void generateSpecialMethodStart(classVersDef *,specialPyMethod *,FILE *);
static void generateSpecialMethodEnd(specialPyMethod *,FILE *);
static void checkForConvertFromCode(sipSpec *,classVersDef *);
static void freeVersionOrList(versionOrList *);
static int getClassVersDetail(classVersDef *);
static int generateArgParser(funcArgs *,int,int,FILE *);
static int samePyClassVers(pyClassVers *,classVersDef *);
static int countVirtuals(classVersDef *);
static int skipOverload(overDef *,memberDef *,classVersDef *,classVersDef *,int);
static FILE *createFile(sipSpec *,char *,char *,char *);
static void closeFile(FILE *);
static void prcode(FILE *,char *,...);


/*
 * Generate the code from a specification.
 */

void generateCode(sipSpec *pt,char *codeDir,stringList *makefiles,char *docFile,char *cppSuffix)
{
	/* Generate the documentation. */

	if (docFile != NULL)
		generateDocumentation(pt,docFile);

	/* Generate the Makefiles. */

	while (makefiles != NULL)
	{
		generateMakefile(pt,makefiles -> s,cppSuffix);

		makefiles = makefiles -> next;
	}

	/* Generate the code. */

	if (codeDir != NULL)
	{
		generateVersionProgram(pt,codeDir,cppSuffix);
		generatePackageHeader(pt,codeDir);
		generateCpp(pt,codeDir,cppSuffix);
		generatePython(pt,codeDir);
	}
}


/*
 * Generate the version generator program.  This program generates a header
 * file that is included by other C++ code and contains #defines corresponding
 * to each of the different versions.  Why not just #include the necessary file
 * that provides the version information and use the C++ test specified as part
 * of the version definition in the .sip files?  The answer is that moc can't
 * handle C++ pre-processor commands so the proxy header file must be run
 * through the C++ pre-processor beforehand.  The code generated by moc is then
 * #included by the main module code.  The net result is that the header file
 * specifying the version information is #included by the main module code and
 * #included a second time - but the second time is a version that has already
 * been run through the C++ pre-processor and has therefore lost it's usual
 * means of protecting itself from being #included twice.  Unless the file
 * follows certain rules (like having no function definitions) it is likely to
 * make the C++ compiler complain.  Therefore the solution is to use a
 * generated file that isn't going to cause complaints.
 */

static void generateVersionProgram(sipSpec *pt,char *codeDir,char *cppSuffix)
{
	char *vfile;
	int v;
	FILE *fp;
	versionQual *vq;

	vfile = concat(codeDir,"/sip_helper",cppSuffix,NULL);
	fp = createFile(pt,vfile,"//","Generate a header file containing the version information.");

	prcode(fp,
"\n"
"\n"
"#include <iostream.h>\n"
"\n"
		);

	generateCppCodeBlock(NULL,pt -> versioncode,NULL,fp);

	prcode(fp,
"\n"
"\n"
"int main(int argc,char **argv)\n"
"{\n"
"	cout << \"// Define simple constants for each version qualifier.\\n\\n\";\n"
		);

	for (vq = pt -> versions; vq != NULL; vq = vq -> next)
		prcode(fp,
"\n"
"	cout << \"#undef SIP_VERS_%s\\n\";\n"
"#if %s\n"
"	cout << \"#define SIP_VERS_%s\\n\";\n"
"#endif\n"
			,vq -> name,vq -> vqcpp,vq -> name);

	prcode(fp,
"\n"
"	return 0;\n"
"}\n"
		);

	closeFile(fp);
	free(vfile);
}


/*
 * Generate the module proxy header file.
 */

static void generateModuleProxyHeader(sipSpec *pt,char *codeDir)
{
	char *hfile, *mname = pt -> module -> name -> text;
	FILE *fp;
	funcArgsList *fl;

	hfile = concat(codeDir,"/sip",mname,"Proxy",mname,".h",NULL);
	fp = createFile(pt,hfile,"//","Module proxy signal/slot handler class.");

	prcode(fp,
"\n"
"#include \"sip%sVersion.h\"\n"
"\n"
"class sipProxy%s: public sipProxy\n"
"{\n"
"	Q_OBJECT\n"
"\n"
"public:\n"
"	char *getProxySlot(char *);\n"
"\n"
"public slots:\n"
		,mname,mname);

	for (fl = pt -> sigargs; fl != NULL; fl = fl -> next)
	{
		generateOredVersionStart(pt,fl -> vol,fp);

		prcode(fp,
"	void proxySlot(");

		generateArgs(fl -> fa,Declaration,fp);

		prcode(fp,");\n"
			);

		generateOredVersionEnd(pt,fl -> vol,fp);
	}

	prcode(fp,
"};\n"
		);

	closeFile(fp);
	free(hfile);
}


/*
 * Generate the documentation.
 */

static void generateDocumentation(sipSpec *pt,char *docFile)
{
	FILE *fp;
	codeBlock *cb;

	fp = createFile(pt,docFile,NULL,NULL);

	for (cb = pt -> docs; cb != NULL; cb = cb -> next)
		fputs(cb -> frag,fp);

	closeFile(fp);
}


/*
 * Generate the code Makefile.
 */

static void generateMakefile(sipSpec *pt,char *makeFile,char *cppSuffix)
{
	char *tn;
	FILE *fp;
	codeBlock *cb;
	mkTemplateDef *mtd;

	/* Find the template. */

	if ((tn = strrchr(makeFile,'/')) != NULL)
		++tn;
	else
		tn = makeFile;

	for (mtd = pt -> mktemplates; mtd != NULL; mtd = mtd -> next)
		if (strcmp(mtd -> name,tn) == 0)
			break;

	if (mtd == NULL)
		fatal("Unknown Makefile template: %s\n",tn);

	fp = createFile(pt,makeFile,NULL,NULL);

	for (cb = mtd -> templ; cb != NULL; cb = cb -> next)
		generateMacroCode(cb,NULL,pt,cppSuffix,mtd -> objext,fp);

	closeFile(fp);
}


/*
 * Generate the Python wrapper code.
 */

static void generatePython(sipSpec *pt,char *codeDir)
{
	char *pyfile, *mname = pt -> module -> name -> text;
	FILE *fp;
	codeBlock *cb;
	moduleDef *mod;

	pyfile = concat(codeDir,"/",mname,".py",NULL);
	fp = createFile(pt,pyfile,"#","Python wrapper code.");

	/* Generate the signal/slot support if needed. */

	if (pt -> qobjclass >= 0)
		prcode(fp,
"\n"
"# Support for signals.\n"
"\n"
"def SLOT(slot):\n"
"	return \"1\" + slot\n"
"\n"
"def SIGNAL(sig):\n"
"	return \"2\" + sig\n"
"\n"
"def PYSIGNAL(sig):\n"
"	return \"9\" + sig\n"
			);

	/*
	 * Now get the different modules.  The first job is to load the C++
	 * code for any imported modules.  If we left it to the dynamic loader
	 * to do when importing the C++ code for this module it would
	 * probably fail because the C++ modules are unlikely to be in a
	 * directory in the standard path.  We can't leave it to any imports
	 * of classes from those modules because a) they may be dependent on
	 * particular versions, and b) the version tests may need this module
	 * loaded to determine the version information.
	 */

	prcode(fp,
"\n"
		);

	for (mod = pt -> moduleList; mod != NULL; mod = mod -> next)
	{
		char *modname;

		/*
		 * Handle the case where the C++ module has a different name.
		 * This make use of "as" which is only in newer Pythons, but as
		 * this feature is for BlackAdder (which has the latest Python)
		 * then we should be Ok.
		 */

		if (mod == pt -> module && pt -> cppmname != mname)
			prcode(fp,
"import lib%sc as lib%sc\n"
				,pt -> cppmname,mod -> name -> text);
		else
			prcode(fp,
"import lib%sc\n"
				,mod -> name -> text);
	}

	prcode(fp,
"\n"
"lib%sc.sipInitModule()\n"
		,mname);

	/* Generate the pre-Python code blocks. */

	for (cb = pt -> prepycode; cb != NULL; cb = cb -> next)
		prcode(fp,"\n%s",cb -> frag);

	prcode(fp,
"\n"
		);

	/* Generate the Python class objects for all versions. */

	generatePythonClassWrappers(pt,fp);

	/* Register the classes. */

	prcode(fp,
"# Register the classes with the C++ module.\n"
"\n"
"lib%sc.sipRegisterClasses()\n"
		,mname);

	/* Generate the Python code blocks. */

	for (cb = pt -> pycode; cb != NULL; cb = cb -> next)
		prcode(fp,"\n%s",cb -> frag);

	closeFile(fp);
	free(pyfile);
}


/*
 * Generate the Python tests for a particular version.
 */

static int generatePythonPushVersion(sipSpec *pt,versionOrList *vol,int indent,FILE *fp)
{
	static versionOrList *prevVol = NULL;
	static int prevIndent = 0;

	int noIntro = TRUE;
	versionOrList *v1, *v2;

	/*
	 * Compare the new condition with the previous one (if any) and use it
	 * if possible.  This is a bit of a hack but does reduce the amount of
	 * Python code a little bit.
	 */

	v1 = prevVol;
	v2 = vol;

	while (v1 != NULL && v2 != NULL)
	{
		if (v1 -> va.lowidx != v2 -> va.lowidx ||
		    v1 -> va.uppidx != v2 -> va.uppidx ||
		    v1 -> va.secondary != v2 -> va.secondary)
			break;

		v1 = v1 -> next;
		v2 = v2 -> next;
	}

	if (v1 == NULL && v2 == NULL)
		return prevIndent;

	/* We need a new one. */

	prevVol = vol;

	while (vol != NULL)
	{
		versionQual *low, *upp, *sec;

		pushVersion(pt,&vol -> va,&low,&upp,&sec);
		popVersion();

		if (low != NULL || upp != NULL || sec != NULL)
		{
			char *sep = "";

			if (noIntro)
			{
				noIntro = FALSE;
				prcode(fp,
"%Iif (",indent);
			}
			else
				prcode(fp,") or (");

			if (low != NULL)
			{
				prcode(fp,"%s",low -> vqpy);
				sep = " and ";
			}

			if (upp != NULL)
			{
				prcode(fp,"%snot %s",sep,upp -> vqpy);
				sep = " and ";
			}


			if (sec != NULL)
				prcode(fp,"%s%s",sep,sec -> vqpy);
		}

		vol = vol -> next;
	}

	if (!noIntro)
	{
		prcode(fp,"):\n"
			);

		++indent;
	}

	prevIndent = indent;

	/* Return the new indent level. */

	return indent;
}


/*
 * Push a new version qualifier onto the stack and return the specific
 * qualifier that needs to be applied to achieve it.
 */

static void pushVersion(sipSpec *pt,versionAnded *va,versionQual **low,versionQual **upp,versionQual **sec)
{
	int lowidx, uppidx;

	if (vqstackptr >= MAX_VQ_STACK)
		fatal("Internal error: increase the value of MAX_VQ_STACK\n");

	/* Assume the new version is the specific version. */

	lowidx = va -> lowidx;
	uppidx = va -> uppidx;
	*sec = va -> secondary;

	if (vqstackptr > 0)
	{
		int i;

		/*
		 * For the primary we only need to look at the previous version
		 * because each range will always be the same or within the
		 * previous one.
		 */

		if (lowidx == vqstack[vqstackptr - 1].lowidx)
			lowidx = 0;

		if (uppidx == vqstack[vqstackptr - 1].uppidx)
			uppidx = pt -> latestversion;

		/* We apply the secondary if it hasn't already been applied. */

		for (i = 0; i < vqstackptr; ++i)
			if (vqstack[i].secondary == va -> secondary)
			{
				/* It's already been applied. */

				*sec = NULL;
				break;
			}
	}

	*low = getVersionQual(pt,lowidx);
	*upp = getVersionQual(pt,uppidx);

	vqstack[vqstackptr++] = *va;
}


/*
 * Pop a version qualifier off the stack.
 */

static void popVersion()
{
	--vqstackptr;
}


/*
 * Generate the Python class objects.
 */

static void generatePythonClassWrappers(sipSpec *pt,FILE *fp)
{
	classDef *cd;
	pyClassVers *list, *pcv;

	/*
	 * Create a list of all the different class versions that have unique
	 * superclasses.
	 */

	list = NULL;

	for (cd = pt -> classes; cd != NULL; cd = cd -> next)
	{
		classVersDef *cvd;

		for (cvd = cd -> cvds; cvd != NULL; cvd = cvd -> next)
		{
			if (isMapped(cvd))
				continue;

			for (pcv = list; pcv != NULL; pcv = pcv -> next)
				if (samePyClassVers(pcv,cvd))
					break;

			/* Create a new one if nothing matches. */

			if (pcv == NULL)
			{
				pcv = sipMalloc(sizeof (pyClassVers));

				pcv -> generated = FALSE;
				pcv -> detail = getClassVersDetail(cvd);
				pcv -> combmeth = cvd -> combmeth;
				pcv -> combnummeth = cvd -> combnummeth;
				pcv -> combseqmeth = cvd -> combseqmeth;
				pcv -> combmapmeth = cvd -> combmapmeth;
				pcv -> cd = cd;
				pcv -> supers = cvd -> supers;
				pcv -> vol = orVersionQual(pt,NULL,&cvd -> version);
				pcv -> next = list;

				list = pcv;
			}
			else if (pcv -> vol != NULL)
				pcv -> vol = orVersionQual(pt,pcv -> vol,&cvd -> version);
		}
	}

	/* Optimize the or expressions. */

	for (pcv = list; pcv != NULL; pcv = pcv -> next)
		reduceOredVersion(pt,&pcv -> vol);

	/* Generate each consolidated version. */

	for (pcv = list; pcv != NULL; pcv = pcv -> next)
		if (pcv -> cd -> module == pt -> module)
			generatePythonClassVersWrapper(pt,list,pcv,fp);
}


/*
 * Return the detail flags for a class version.
 */

static int getClassVersDetail(classVersDef *cvd)
{
	int detail = 0;
	varDef *vd;

	for (vd = cvd -> vars; vd != NULL; vd = vd -> next)
		if (!isStaticVar(vd))
		{
			detail |= PCV_CLASSVARS;
			break;
		}

	return detail;
}


/*
 * Return TRUE if a class version is part of a Python consolidation class
 * version.
 */

static int samePyClassVers(pyClassVers *pcv,classVersDef *cvd)
{
	classList *cl1, *cl2;

	/* They must be the same class. */

	if (pcv -> cd != cvd -> cd)
		return FALSE;

	/* The detail must be the same. */

	if (pcv -> detail != getClassVersDetail(cvd))
		return FALSE;

	/* They must define the same special methods. */

	if (pcv -> combmeth != cvd -> combmeth ||
	    pcv -> combnummeth != cvd -> combnummeth ||
	    pcv -> combseqmeth != cvd -> combseqmeth ||
	    pcv -> combmapmeth != cvd -> combmapmeth)
		return FALSE;

	/* Compare the superclasses. */

	cl1 = pcv -> supers;
	cl2 = cvd -> supers;

	while (cl1 != NULL && cl2 != NULL && cl1 -> c == cl2 -> c)
	{
		cl1 = cl1 -> next;
		cl2 = cl2 -> next;
	}

	return (cl1 == NULL && cl2 == NULL);
}


/*
 * Generate the Python class object for a particular version.
 */

static void generatePythonClassVersWrapper(sipSpec *pt,pyClassVers *list,pyClassVers *pcv,FILE *fp)
{
	/* It may already have been done. */

	if (pcv -> generated)
		return;

	/* If this class is in a separate module then we need to import it. */

	if (pcv -> cd -> module != pt -> module)
	{
		pyClassVers *imp;

		prcode(fp,
"from %s import %s\n"
"\n"
			,pcv -> cd -> module -> name -> text,pcv -> cd -> name -> text);

		/*
		 * Importing is version independent, so mark all variations of
		 * this class as being generated as well.
		 */

		for (imp = list; imp != NULL; imp = imp -> next)
			if (imp -> cd == pcv -> cd)
				imp -> generated = TRUE;
	}
	else
	{
		int indent;
		classList *cl;

		/* Make sure the super-classes have been generated. */

		for (cl = pcv -> supers; cl != NULL; cl = cl -> next)
		{
			pyClassVers *pscv;

			for (pscv = list; pscv != NULL; pscv = pscv -> next)
			{
				if (pscv -> cd != cl -> c)
					continue;

				/* Generate it if the versions overlap. */

				if (pcv -> vol == NULL || pscv -> vol == NULL)
					generatePythonClassVersWrapper(pt,list,pscv,fp);
				else
				{
					versionOrList *v1;

					for (v1 = pscv -> vol; v1 != NULL; v1 = v1 -> next)
					{
						versionOrList *v2;

						for (v2 = pcv -> vol; v2 != NULL; v2 = v2 -> next)
							if (versionsOverlap(&v1 -> va,&v2 -> va))
							{
								generatePythonClassVersWrapper(pt,list,pscv,fp);
								break;
							}

						if (v2 != NULL)
							break;
					}
				}
			}
		}

		indent = generatePythonPushVersion(pt,pcv -> vol,0,fp);
		generatePythonClassWrapper(pt,pcv,indent,fp);

		pcv -> generated = TRUE;
	}
}


/*
 * Generate an expression in C++.
 */

static void generateExpression(valueDef *vd,FILE *fp)
{
	while (vd != NULL)
	{
		if (vd -> vunop != '\0')
			prcode(fp,"%c",vd -> vunop);

		switch (vd -> vtype)
		{
		case qchar_value:
			prcode(fp,"'%c'",vd -> u.vqchar);
			break;

		case string_value:
			prcode(fp,"%N",vd -> u.vstr);
			break;

		case numeric_value:
			prcode(fp,"%l",vd -> u.vnum);
			break;

		case real_value:
			prcode(fp,"%g",vd -> u.vreal);
			break;

		case scoped_value:
			prcode(fp,"%S",vd -> u.vscp);
			break;
		}
 
		if (vd -> vbinop != '\0')
			prcode(fp," %c ",vd -> vbinop);
 
		vd = vd -> next;
	}
}


/*
 * Generate the wrapper for a Python class.
 */

static void generatePythonClassWrapper(sipSpec *pt,pyClassVers *pcv,int indent,FILE *fp)
{
	char *mname = pt -> module -> name -> text;
	char *cname = pcv -> cd -> name -> text;
	int noIntro;
	classList *cl;

	prcode(fp,
"%Iclass %s",indent,cname);

	/* Inherit from all immediate super-classes. */

	noIntro = TRUE;

	for (cl = pcv -> supers; cl != NULL; cl = cl -> next)
	{
		if (noIntro)
		{
			prcode(fp,"(");
			noIntro = FALSE;
		}
		else
			prcode(fp,",");

		prcode(fp,"%s",cl -> c -> name -> text);
	}

	if (!noIntro)
		prcode(fp,")");

	prcode(fp,":\n");

	/* Generate the standard methods. */

	prcode(fp,
"%I	def __init__(self,*args):\n"
"%I		lib%sc.sipCallCtor(%d,self,args)\n"
		,indent
		,indent,mname,pcv -> cd -> classnr);

	if (pcv -> supers == NULL)
		prcode(fp,
"%I	def __del__(self):\n"
"%I		__sipDtor__(self)\n"
			,indent
			,indent);

	if (pcv -> detail & PCV_CLASSVARS)
		prcode(fp,
"%I	def __getattr__(self,name):\n"
"%I		return lib%sc.sipCallGetVar(%d,self,name)\n"
"%I	def __setattr__(self,name,value):\n"
"%I		return lib%sc.sipCallSetVar(%d,self,name,value)\n"
			,indent
			,indent,mname,pcv -> cd -> classnr
			,indent
			,indent,mname,pcv -> cd -> classnr);

	/* Generate the special methods. */

	if (pcv -> combmeth & SPM_METH_REPR)
		prcode(fp,
"%I	def __repr__(self):\n"
"%I		return repr(self.sipThis)\n"
			,indent,indent);

	if (pcv -> combmeth & SPM_METH_STR)
		prcode(fp,
"%I	def __str__(self):\n"
"%I		return str(self.sipThis)\n"
			,indent,indent);

	if (pcv -> combmeth & SPM_METH_CMP)
		prcode(fp,
"%I	def __cmp__(self,other):\n"
"%I		return cmp(self.sipThis,other)\n"
			,indent,indent);

	if (pcv -> combmeth & SPM_METH_HASH)
		prcode(fp,
"%I	def __hash__(self):\n"
"%I		return hash(self.sipThis)\n"
			,indent,indent);

	if (pcv -> combmeth & SPM_METH_CALL)
		prcode(fp,
"%I	def __call__(self,*args):\n"
"%I		return self.sipThis(args)\n"
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_ADD || pcv -> combseqmeth & SPM_SEQ_CONCAT)
		prcode(fp,
"%I	def __add__(self,other):\n"
"%I		return self.sipThis + other\n"
"%I	def __radd__(self,other):\n"
"%I		return other + self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_SUB)
		prcode(fp,
"%I	def __sub__(self,other):\n"
"%I		return self.sipThis - other\n"
"%I	def __rsub__(self,other):\n"
"%I		return other - self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_MUL || pcv -> combseqmeth & SPM_SEQ_REPEAT)
		prcode(fp,
"%I	def __mul__(self,other):\n"
"%I		return self.sipThis * other\n"
"%I	def __rmul__(self,other):\n"
"%I		return other * self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_DIV)
		prcode(fp,
"%I	def __div__(self,other):\n"
"%I		return self.sipThis / other\n"
"%I	def __rdiv__(self,other):\n"
"%I		return other / self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_MOD)
		prcode(fp,
"%I	def __mod__(self,other):\n"
"%I		return self.sipThis % other\n"
"%I	def __rmod__(self,other):\n"
"%I		return other % self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_DIVMOD)
		prcode(fp,
"%I	def __divmod__(self,other):\n"
"%I		return divmod(self.sipThis,other)\n"
"%I	def __rdivmod__(self,other):\n"
"%I		return divmod(other,self.sipThis)\n"
			,indent,indent
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_POW)
		prcode(fp,
"%I	def __pow__(self,other,modulo = None):\n"
"%I		return pow(self.sipThis,other,modulo)\n"
"%I	def __rpow__(self,other):\n"
"%I		return other ** self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_NEG)
		prcode(fp,
"%I	def __neg__(self):\n"
"%I		return -self.sipThis\n"
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_POS)
		prcode(fp,
"%I	def __pos__(self):\n"
"%I		return +self.sipThis\n"
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_ABS)
		prcode(fp,
"%I	def __abs__(self):\n"
"%I		return abs(self.sipThis)\n"
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_NONZERO)
		prcode(fp,
"%I	def __nonzero__(self):\n"
"%I		return not self.sipThis\n"
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_INVERT)
		prcode(fp,
"%I	def __invert__(self):\n"
"%I		return ~self.sipThis\n"
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_LSHIFT)
		prcode(fp,
"%I	def __lshift__(self,other):\n"
"%I		return self.sipThis << other\n"
"%I	def __rlshift__(self,other):\n"
"%I		return other << self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_RSHIFT)
		prcode(fp,
"%I	def __rshift__(self,other):\n"
"%I		return self.sipThis >> other\n"
"%I	def __rrshift__(self,other):\n"
"%I		return other >> self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_AND)
		prcode(fp,
"%I	def __and__(self,other):\n"
"%I		return self.sipThis & other\n"
"%I	def __rand__(self,other):\n"
"%I		return other & self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_XOR)
		prcode(fp,
"%I	def __xor__(self,other):\n"
"%I		return self.sipThis ^ other\n"
"%I	def __rxor__(self,other):\n"
"%I		return other ^ self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_OR)
		prcode(fp,
"%I	def __or__(self,other):\n"
"%I		return self.sipThis | other\n"
"%I	def __ror__(self,other):\n"
"%I		return other | self.sipThis\n"
			,indent,indent
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_COERCE)
		prcode(fp,
"%I	def __coerce__(self,other):\n"
"%I		return coerce(self.sipThis,other)\n"
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_INT)
		prcode(fp,
"%I	def __int__(self):\n"
"%I		return int(self.sipThis)\n"
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_LONG)
		prcode(fp,
"%I	def __long__(self):\n"
"%I		return long(self.sipThis)\n"
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_FLOAT)
		prcode(fp,
"%I	def __float__(self):\n"
"%I		return float(self.sipThis)\n"
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_OCT)
		prcode(fp,
"%I	def __oct__(self):\n"
"%I		return oct(self.sipThis)\n"
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_HEX)
		prcode(fp,
"%I	def __hex__(self):\n"
"%I		return hex(self.sipThis)\n"
			,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_IADD || pcv -> combseqmeth & SPM_SEQ_ICONCAT)
		prcode(fp,
"%I	def __iadd__(self,other):\n"
"%I		self.sipThis += other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_ISUB)
		prcode(fp,
"%I	def __isub__(self,other):\n"
"%I		self.sipThis -= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_IMUL || pcv -> combseqmeth & SPM_SEQ_IREPEAT)
		prcode(fp,
"%I	def __imul__(self,other):\n"
"%I		self.sipThis *= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_IDIV)
		prcode(fp,
"%I	def __idiv__(self,other):\n"
"%I		self.sipThis /= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_IMOD)
		prcode(fp,
"%I	def __imod__(self,other):\n"
"%I		self.sipThis %= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_IPOW)
		prcode(fp,
"%I	def __ipow__(self,other,modulo = None):\n"
"%I		self.sipThis **= other)\n"
"%I		return self\n"
			,indent,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_ILSHIFT)
		prcode(fp,
"%I	def __lshift__(self,other):\n"
"%I		self.sipThis <<= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_IRSHIFT)
		prcode(fp,
"%I	def __rshift__(self,other):\n"
"%I		self.sipThis >>= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_IAND)
		prcode(fp,
"%I	def __and__(self,other):\n"
"%I		self.sipThis &= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_IXOR)
		prcode(fp,
"%I	def __xor__(self,other):\n"
"%I		self.sipThis ^= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (pcv -> combnummeth & SPM_NUM_IOR)
		prcode(fp,
"%I	def __or__(self,other):\n"
"%I		self.sipThis |= other\n"
"%I		return self\n"
			,indent,indent,indent);

	if (pcv -> combseqmeth & SPM_SEQ_LEN || pcv -> combmapmeth & SPM_MAP_LEN)
		prcode(fp,
"%I	def __len__(self):\n"
"%I		return len(self.sipThis)\n"
			,indent,indent);

	if (pcv -> combseqmeth & SPM_SEQ_GETITEM || pcv -> combmapmeth & SPM_MAP_GETITEM)
		prcode(fp,
"%I	def __getitem__(self,key):\n"
"%I		return self.sipThis[key]\n"
			,indent,indent);

	if (pcv -> combseqmeth & SPM_SEQ_SETITEM || pcv -> combmapmeth & SPM_MAP_SETITEM)
		prcode(fp,
"%I	def __setitem__(self,key,value):\n"
"%I		return self.sipThis[key] = value\n"
"%I	def __delitem__(self,key):\n"
"%I		return del self.sipThis[key]\n"
			,indent,indent
			,indent,indent);

	if (pcv -> combseqmeth & SPM_SEQ_GETSLICE)
		prcode(fp,
"%I	def __getslice__(self,i,j):\n"
"%I		return self.sipThis[i:j]\n"
			,indent,indent);

	if (pcv -> combseqmeth & SPM_SEQ_SETSLICE)
		prcode(fp,
"%I	def __setslice__(self,i,j,sequence):\n"
"%I		return self.sipThis[i:j] = sequence\n"
"%I	def __delslice__(self,i,j):\n"
"%I		return del self.sipThis[i:j]\n"
			,indent,indent
			,indent,indent);

	if (pcv -> combseqmeth & SPM_SEQ_CONTAINS)
		prcode(fp,
"%I	def __contains__(self,member):\n"
"%I		return member in self.sipThis\n"
			,indent,indent);

	if (pcv -> combmeth & SPM_METH_RICH)
	{
		prcode(fp,
"%I	def __lt__(self,other):\n"
"%I		return self.sipThis < other\n"
			,indent,indent);

		prcode(fp,
"%I	def __le__(self,other):\n"
"%I		return self.sipThis <= other\n"
			,indent,indent);

		prcode(fp,
"%I	def __eq__(self,other):\n"
"%I		return self.sipThis == other\n"
			,indent,indent);

		prcode(fp,
"%I	def __ne__(self,other):\n"
"%I		return self.sipThis != other\n"
			,indent,indent);

		prcode(fp,
"%I	def __gt__(self,other):\n"
"%I		return self.sipThis > other\n"
			,indent,indent);

		prcode(fp,
"%I	def __ge__(self,other):\n"
"%I		return self.sipThis >= other\n"
			,indent,indent);
	}

	prcode(fp,
"\n"
		);
}


/*
 * Generate the C++ package specific header file.
 */

static void generatePackageHeader(sipSpec *pt,char *codeDir)
{
	char *hfile, *mname = pt -> module -> name -> text;
	FILE *fp;
	nameDef *nd;

	hfile = concat(codeDir,"/sip",mname,"Decl",mname,".h",NULL);
	fp = createFile(pt,hfile,"//","C++ package header file.");

	/* Include files. */

	prcode(fp,
"\n"
"#ifndef _%sDECL%s_H\n"
"#define	_%sDECL%s_H\n"
"\n"
"#ifdef HAVE_CONFIG_H\n"
"#include \"config.h\"\n"
"#endif\n"
"\n"
"#include <sip%s.h>\n"
"\n"
"#include \"sip%sVersion.h\"\n"
"\n"
		,mname,mname,mname,mname,(pt -> sigslots ? "Qt" : ""),mname);

	generateCppCodeBlock(pt,pt -> exphdrcode,NULL,fp);
	generateCppCodeBlock(pt,pt -> hdrcode,NULL,fp);

	/* The names declarations (for all modules). */

	prcode(fp,
"\n"
		);

	for (nd = pt -> namecache; nd != NULL; nd = nd -> next)
		prcode(fp,
"extern SIP_%s char %N[];\n"
			,(nd -> module == pt -> module) ? "EXPORT" : "IMPORT"
			,nd);

	if (pt -> sigslots)
		prcode(fp,
"\n"
"sipProxy *sipNewProxy_%s Py_PROTO(());\n"
			,mname);

	prcode(fp,
"\n"
"#endif\n"
		);

	closeFile(fp);
	free(hfile);
}


/*
 * Generate the C++ code.
 */

static void generateCpp(sipSpec *pt,char *codeDir,char *cppSuffix)
{
	char *cppfile, *mname = pt -> module -> name -> text;
	int nrclasses, noIntro;
	FILE *fp;
	classDef *cd;
	nameDef *nd;
	varDef *vd;
	memberDef *md;
	expFuncDef *ef;

	/* Generate any module proxy header file. */

	if (pt -> sigargs != NULL)
		generateModuleProxyHeader(pt,codeDir);

	cppfile = concat(codeDir,"/",mname,"cmodule",cppSuffix,NULL);
	fp = createFile(pt,cppfile,"//","C++ wrapper code.");

	prcode(fp,
"\n"
"#include \"sip%sDecl%s.h\"\n"
		,mname,mname);

	generateUsedIncludes(pt -> used,fp);

	/* The names definitions. */

	for (nd = pt -> namecache; nd != NULL; nd = nd -> next)
		if (nd -> module == pt -> module)
			prcode(fp,
"char %N[] = \"%s\";\n"
				,nd,nd -> text);

	/* Generate the module proxy. */

	if (pt -> sigargs != NULL)
	{
		generateModuleProxy(pt,fp);

		prcode(fp,
"\n"
"sipProxy *sipNewProxy_%s()\n"
"{\n"
"	return new sipProxy%s();\n"
"}\n"
			,mname,mname);
	}

	/* Generate the C++ code blocks. */

	generateCppCodeBlock(pt,pt -> cppcode,NULL,fp);

	/* Generate the global functions. */

	for (md = pt -> othfuncs; md != NULL; md = md -> next)
		if (md -> module == pt -> module)
		{
			overDef *od;

			prcode(fp,
"\n"
"static PyObject *sipDo_%s(PyObject *,PyObject *sipArgs)\n"
"{\n"
"	int sipArgsParsed = 0;\n"
				,md -> pyname -> text);

			for (od = pt -> overs; od != NULL; od = od -> next)
				if (od -> common == md)
					generateFunctionBody(pt,od,NULL,fp);

			prcode(fp,
"\n"
"	// Report an error if the arguments couldn't be parsed.\n"
"\n"
"	sipNoFunction(sipArgsParsed,%N);\n"
"\n"
"	return NULL;\n"
"}\n"
				,md -> pyname);
		}

	/* Generate the access functions. */

	generateAccessFunctions(pt,NULL,pt -> globals,fp);

	for (cd = pt -> classes; cd != NULL; cd = cd -> next)
	{
		classVersDef *cvd;

		if (cd -> module != pt -> module)
			continue;

		for (cvd = cd -> cvds; cvd != NULL; cvd = cvd -> next)
			generateAccessFunctions(pt,cvd,cvd -> vars,fp);
	}

	/* Generate the module data structures. */

	prcode(fp,
"\n"
"static sipClassDef classesTable[] = {\n"
		);

	nrclasses = 0;

	for (cd = pt -> classes; cd != NULL; cd = cd -> next)
	{
		char *cname = cd -> name -> text;
		int iscond;
		classVersDef *cvd;

		if (cd -> module != pt -> module)
			continue;

		++nrclasses;

		/*
		 * Version control is handled a little differently to make
		 * sure there is a table entry for every possible class.
		 */

		iscond = FALSE;

		for (cvd = cd -> cvds; cvd != NULL; cvd = cvd -> next)
		{
			versionQual *low, *upp;

			low = getVersionQual(pt,cvd -> version.lowidx);
			upp = getVersionQual(pt,cvd -> version.uppidx);

			if (low != NULL || upp != NULL || cvd -> version.secondary != NULL)
			{
				iscond = TRUE;

				prcode(fp,(cvd == cd -> cvds ? "#if " : "#elif "));

				generateCppVersionExpr(low,upp,cvd -> version.secondary,fp);

				prcode(fp,"\n"
					);
			}

			if (isMapped(cvd))
				prcode(fp,
"	{NULL, NULL, NULL, NULL, NULL},\n"
					);
			else
			{
				varDef *vd;

				prcode(fp,
"	{%N, sipNew_%s, &sipClass_%s, sipClassAttrTab_%s, "
					,cd -> name,cname,cname,cname);

				for (vd = cvd -> vars; vd != NULL; vd = vd -> next)
					if (!isStaticVar(vd))
						break;

				if (vd != NULL)
					prcode(fp,"sipClassVarHierTab_%s},\n"
						,cname);
				else
					prcode(fp,"NULL},\n"
						);
			}
		}

		if (iscond)
			prcode(fp,
"#else\n"
"	{NULL, NULL, NULL, NULL, NULL},\n"
"#endif\n"
				);
	}

	prcode(fp,
"};\n"
"\n"
"static sipModuleDef sipModule = {\n"
"	%N,\n"
"	%d,\n"
"	classesTable,\n"
"	0,\n"
"	0\n"
"};\n"
		,pt -> module -> name,nrclasses);


	/* Generate the call class ctor function. */

	prcode(fp,
"\n"
"// The entry point into the bindings for constructors.\n"
"\n"
"static PyObject *callCtor(PyObject *,PyObject *args)\n"
"{\n"
"	return sipCallCtor(&sipModule,args);\n"
"}\n"
		);

	/* Generate the get and set class attribute functions. */

	prcode(fp,
"\n"
"// The entry point into the bindings for getting class variables.\n"
"\n"
"static PyObject *callGetVar(PyObject *,PyObject *args)\n"
"{\n"
"	return sipGetVar(&sipModule,args);\n"
"}\n"
"\n"
"// The entry point into the bindings for setting class variables.\n"
"\n"
"static PyObject *callSetVar(PyObject *,PyObject *args)\n"
"{\n"
"	return sipSetVar(&sipModule,args);\n"
"}\n"
		);

	/* Generate the register Python globals function. */

	prcode(fp,
"\n"
"// Initialise the module.\n"
"\n"
"static PyObject *initModule(PyObject *,PyObject *)\n"
"{\n"
"	if (sipRegisterModule(&sipModule) < 0)\n"
"		return NULL;\n"
		);

	/* Generate the global functions. */

	noIntro = TRUE;

	for (md = pt -> othfuncs; md != NULL; md = md -> next)
		if (md -> module == pt -> module)
		{
			if (noIntro)
			{
				prcode(fp,
"\n"
"	// Add the global functions to the dictionary.\n"
"\n"
"	static PyMethodDef globfuncs[] = {\n"
					);

				noIntro = FALSE;
			}

			prcode(fp,
"		{%N, sipDo_%s, METH_VARARGS, NULL},\n"
				,md -> pyname,md -> pyname -> text);
		}

	if (!noIntro)
		prcode(fp,
"		{NULL}\n"
"	};\n"
"\n"
"	if (sipAddFunctions(sipModule.md_dict,globfuncs) < 0)\n"
"		return NULL;\n"
			);

	/* Generate the global variables. */

	generateVoidPointers(pt,NULL,pt -> globals,fp);
	generateChars(pt,NULL,pt -> globals,fp);
	generateStrings(pt,NULL,pt -> globals,fp);
	generateLongs(pt,NULL,pt -> globals,fp);
	generateDoubles(pt,NULL,pt -> globals,fp);
	generateEnums(pt,NULL,pt -> enums,fp);

	prcode(fp,
"\n"
"	Py_INCREF(Py_None);\n"
"	return Py_None;\n"
"}\n"
		);

	/* Generate the register Python classes function. */

	prcode(fp,
"\n"
"// Final stage after the Python classes have been created.\n"
"\n"
"static PyObject *registerClasses(PyObject *,PyObject *)\n"
"{\n"
"	if (sipRegisterClasses(&sipModule,%d) < 0)\n"
"		return NULL;\n"
		,pt -> qobjclass);

	/* Add the global class instances. */

	generateClasses(pt,NULL,pt -> globals,fp);

	/* Generate all static class variables. */

	for (cd = pt -> classes; cd != NULL; cd = cd -> next)
	{
		classVersDef *cvd;

		if (cd -> module != pt -> module)
			continue;

		for (cvd = cd -> cvds; cvd != NULL; cvd = cvd -> next)
		{
			generateVoidPointers(pt,cvd,cvd -> vars,fp);
			generateChars(pt,cvd,cvd -> vars,fp);
			generateStrings(pt,cvd,cvd -> vars,fp);
			generateLongs(pt,cvd,cvd -> vars,fp);
			generateDoubles(pt,cvd,cvd -> vars,fp);
			generateEnums(pt,cvd,cvd -> enums,fp);
			generateClasses(pt,cvd,cvd -> vars,fp);
		}
	}

	prcode(fp,
"\n"
"	Py_INCREF(Py_None);\n"
"	return Py_None;\n"
"}\n"
		);

	/* Generate the Python module initialisation function. */

	prcode(fp,
"\n"
"// The Python module initialisation function.\n"
"\n"
"extern \"C\" SIP_EXPORT void initlib%sc()\n"
"{\n"
"	static PyMethodDef methods[] = {\n"
"		{(char *)\"sipCallCtor\", callCtor, METH_VARARGS, NULL},\n"
"		{(char *)\"sipCallGetVar\", callGetVar, METH_VARARGS, NULL},\n"
"		{(char *)\"sipCallSetVar\", callSetVar, METH_VARARGS, NULL},\n"
"		{(char *)\"sipInitModule\", initModule, METH_VARARGS, NULL},\n"
"		{(char *)\"sipRegisterClasses\", registerClasses, METH_VARARGS, NULL},\n"
		,pt -> cppmname);

	for (ef = pt -> exposed; ef != NULL; ef = ef -> next)
		prcode(fp,
"		{\"%s\", %s, METH_VARARGS, NULL},\n"
			,ef -> name,ef -> name);

	prcode(fp,
"		{NULL,NULL,0,NULL}\n"
"	};\n"
"\n"
"	// Initialise the module.\n"
"\n"
"	Py_InitModule((char *)\"lib%sc\",methods);\n"
"}\n"
		,pt -> cppmname);

	closeFile(fp);
	free(cppfile);

	/* Generate the class source and header files. */

	for (cd = pt -> classes; cd != NULL; cd = cd -> next)
	{
		if (cd -> module == pt -> module)
			generateClassCpp(cd,pt,codeDir,cppSuffix);

		if (isUsed(cd))
			generateClassHeader(pt,cd,codeDir);
	}
}


/*
 * Generate the access functions for a list of variables.
 */

static void generateAccessFunctions(sipSpec *pt,classVersDef *cvd,varDef *vars,FILE *fp)
{
	char *cname;
	int noIntro;
	varDef *vd;

	noIntro = TRUE;

	for (vd = vars; vd != NULL; vd = vd -> next)
	{
		char *vname = vd -> name -> text;

		if (vd -> module != pt -> module || vd -> accessfunc == NULL)
			continue;

		if (cvd != NULL && !isStaticVar(vd))
			continue;

		if (noIntro)
		{
			if (cvd != NULL)
			{
				cname = classVersName(cvd);
				generateVersionStart(pt,&cvd -> version,fp);
			}

			noIntro = FALSE;
		}

		generateVersionStart(pt,&vd -> version,fp);

		if (cvd != NULL)
			prcode(fp,
"\n"
"// Access function for %s::%s.\n"
"\n"
"static const void *sipAccess_%s_%s()\n"
"{\n"
				,cname,vname,cname,vname);
		else
			prcode(fp,
"\n"
"// Access function for %s.\n"
"\n"
"static const void *sipAccess_%s()\n"
"{\n"
				,vname,vname);

		generateCppCodeBlock(NULL,vd -> accessfunc,NULL,fp);

		prcode(fp,
"}\n"
			);

		generateVersionEnd(pt,&vd -> version,fp);
	}

	if (!noIntro)
		if (cvd != NULL)
			generateVersionEnd(pt,&cvd -> version,fp);
}


/*
 * Generate the code to add a set of class instances to a dictionary.
 */

static void generateClasses(sipSpec *pt,classVersDef *cvd,varDef *vars,FILE *fp)
{
	char *cname;
	int noIntro;
	varDef *vd;

	noIntro = TRUE;

	for (vd = vars; vd != NULL; vd = vd -> next)
	{
		char *vname = vd -> name -> text;
		char *vcname;

		if (vd -> module != pt -> module || vd -> type.atype != class_type)
			continue;

		if (cvd != NULL && !isStaticVar(vd))
			continue;

		if (noIntro)
		{
			if (cvd != NULL)
			{
				cname = classVersName(cvd);
				generateVersionStart(pt,&cvd -> version,fp);
			}

			prcode(fp,
"\n"
"	// Add the class instances to the dictionary.\n"
"\n"
"	static sipClassInstanceDef %sclassInstances[] = {\n"
			,(cvd != NULL ? cname : ""));

			noIntro = FALSE;
		}

		generateVersionStart(pt,&vd -> version,fp);

		vcname = classVersName(vd -> type.u.cvd);

		if (vd -> accessfunc != NULL)
		{
			if (cvd != NULL)
				prcode(fp,
"		{%N, sipAccess_%s_%s, sipClass_%s, SIP_SIMPLE | SIP_ACCFUNC},\n"
					,vd -> name,cname,vname,vcname);
			else
				prcode(fp,
"		{%N, sipAccess_%s, sipClass_%s, SIP_SIMPLE | SIP_ACCFUNC},\n"
					,vd -> name,vname,vcname);
		}
		else if (vd -> type.nrderefs != 0)
		{
			if (cvd != NULL)
				prcode(fp,
"		{%N, &%s::%s, sipClass_%s, SIP_SIMPLE | SIP_INDIRECT},\n"
					,vd -> name,cname,vname,vcname);
			else
				prcode(fp,
"		{%N, &%s, sipClass_%s, SIP_SIMPLE | SIP_INDIRECT},\n"
					,vd -> name,vname,vcname);
		}
		else
		{
			if (cvd != NULL)
				prcode(fp,
"		{%N, &%s::%s, sipClass_%s, SIP_SIMPLE},\n"
					,vd -> name,cname,vname,vcname);
			else
				prcode(fp,
"		{%N, &%s, sipClass_%s, SIP_SIMPLE},\n"
					,vd -> name,vname,vcname);
		}

		generateVersionEnd(pt,&vd -> version,fp);
	}

	if (!noIntro)
	{
		prcode(fp,
"		NULL\n"
"	};\n"
"\n"
"	if (sipAddClassInstances(");

		if (cvd != NULL)
			prcode(fp,"((PyClassObject *)sipClass_%s) -> cl_dict,%s",cname,cname);
		else
			prcode(fp,"sipModule.md_dict,");

		prcode(fp,"classInstances) < 0)\n"
"		return NULL;\n"
			);

		if (cvd != NULL)
			generateVersionEnd(pt,&cvd -> version,fp);
	}
}


/*
 * Generate the code to add a set of void pointers to a dictionary.
 */

static void generateVoidPointers(sipSpec *pt,classVersDef *cvd,varDef *vars,FILE *fp)
{
	char *cname;
	int noIntro;
	varDef *vd;

	noIntro = TRUE;

	for (vd = vars; vd != NULL; vd = vd -> next)
	{
		char *vname = vd -> name -> text;

		if (vd -> module != pt -> module)
			continue;

		if (vd -> type.atype != voidptr_type && vd -> type.atype != struct_type)
			continue;

		if (cvd != NULL && !isStaticVar(vd))
			continue;

		if (noIntro)
		{
			if (cvd != NULL)
			{
				cname = classVersName(cvd);
				generateVersionStart(pt,&cvd -> version,fp);
			}

			prcode(fp,
"\n"
"	// Add the void pointers to the dictionary.\n"
"\n"
"	static sipVoidPtrInstanceDef %svoidPtrInstances[] = {\n"
				,(cvd != NULL ? cname : ""));

			noIntro = FALSE;
		}

		generateVersionStart(pt,&vd -> version,fp);

		if (cvd != NULL)
			prcode(fp,
"		{%N, %s::%s},\n"
				,vd -> name,cname,vname);
		else
			prcode(fp,
"		{%N, %s},\n"
				,vd -> name,vname);

		generateVersionEnd(pt,&vd -> version,fp);
	}

	if (!noIntro)
	{
		prcode(fp,
"		{NULL}\n"
"	};\n"
"\n"
"	if (sipAddVoidPtrInstances(");

		if (cvd != NULL)
			prcode(fp,"((PyClassObject *)sipClass_%s) -> cl_dict,%s",cname,cname);
		else
			prcode(fp,"sipModule.md_dict,");

		prcode(fp,"voidPtrInstances) < 0)\n"
"		return NULL;\n"
			);

		if (cvd != NULL)
			generateVersionEnd(pt,&cvd -> version,fp);
	}
}


/*
 * Generate the code to add a set of characters to a dictionary.
 */

static void generateChars(sipSpec *pt,classVersDef *cvd,varDef *vars,FILE *fp)
{
	char *cname;
	int noIntro;
	varDef *vd;

	noIntro = TRUE;

	for (vd = vars; vd != NULL; vd = vd -> next)
	{
		char *vname = vd -> name -> text;
		argType vtype = vd -> type.atype;

		if (vd -> module != pt -> module)
			continue;

		if (!((vtype == ustring_type || vtype == string_type) && vd -> type.nrderefs == 0))
			continue;

		if (cvd != NULL && !isStaticVar(vd))
			continue;

		if (noIntro)
		{
			if (cvd != NULL)
			{
				cname = classVersName(cvd);
				generateVersionStart(pt,&cvd -> version,fp);
			}

			prcode(fp,
"\n"
"	// Add the chars to the dictionary.\n"
"\n"
"	static sipCharInstanceDef %scharInstances[] = {\n"
				,(cvd != NULL ? cname : ""));

			noIntro = FALSE;
		}

		generateVersionStart(pt,&vd -> version,fp);

		if (cvd != NULL)
			prcode(fp,
"		{%N, %s::%s},\n"
				,vd -> name,cname,vname);
		else
			prcode(fp,
"		{%N, %s},\n"
				,vd -> name,vname);

		generateVersionEnd(pt,&vd -> version,fp);
	}

	if (!noIntro)
	{
		prcode(fp,
"		{NULL}\n"
"	};\n"
"\n"
"	if (sipAddCharInstances(");

		if (cvd != NULL)
			prcode(fp,"((PyClassObject *)sipClass_%s) -> cl_dict,%s",cname,cname);
		else
			prcode(fp,"sipModule.md_dict,");

		prcode(fp,"charInstances) < 0)\n"
"		return NULL;\n"
			);

		if (cvd != NULL)
			generateVersionEnd(pt,&cvd -> version,fp);
	}
}


/*
 * Generate the code to add a set of strings to a dictionary.
 */

static void generateStrings(sipSpec *pt,classVersDef *cvd,varDef *vars,FILE *fp)
{
	char *cname;
	int noIntro;
	varDef *vd;

	noIntro = TRUE;

	for (vd = vars; vd != NULL; vd = vd -> next)
	{
		char *vmname = vd -> name -> module -> name -> text;
		char *vname = vd -> name -> text;
		argType vtype = vd -> type.atype;

		if (vd -> module != pt -> module)
			continue;

		if (!((vtype == ustring_type || vtype == string_type) && vd -> type.nrderefs != 0))
			continue;

		if (cvd != NULL && !isStaticVar(vd))
			continue;

		if (noIntro)
		{
			if (cvd != NULL)
			{
				cname = classVersName(cvd);
				generateVersionStart(pt,&cvd -> version,fp);
			}

			prcode(fp,
"\n"
"	// Add the strings to the dictionary.\n"
"\n"
"	static sipStringInstanceDef %sstringInstances[] = {\n"
				,(cvd != NULL ? cname : ""));

			noIntro = FALSE;
		}

		generateVersionStart(pt,&vd -> version,fp);

		if (cvd != NULL)
			prcode(fp,
"		{%N, %s::%s},\n"
				,vd -> name,cname,vname);
		else
			prcode(fp,
"		{%N, %s},\n"
				,vd -> name,vname);

		generateVersionEnd(pt,&vd -> version,fp);
	}

	if (!noIntro)
	{
		prcode(fp,
"		{NULL}\n"
"	};\n"
"\n"
"	if (sipAddStringInstances(");

		if (cvd != NULL)
			prcode(fp,"((PyClassObject *)sipClass_%s) -> cl_dict,%s",cname,cname);
		else
			prcode(fp,"sipModule.md_dict,");

		prcode(fp,"stringInstances) < 0)\n"
"		return NULL;\n"
			);

		if (cvd != NULL)
			generateVersionEnd(pt,&cvd -> version,fp);
	}
}


/*
 * Generate the code to add a set of enums to a dictionary.
 */

static void generateEnums(sipSpec *pt,classVersDef *cvd,enumDef *enums,FILE *fp)
{
	char *cname;
	int noIntro;
	enumDef *ed;

	/* First the tables of values for each enum. */

	noIntro = TRUE;

	for (ed = enums; ed != NULL; ed = ed -> next)
	{
		enumValueDef *evd;

		if (ed -> module != pt -> module)
			continue;

		if (noIntro)
		{
			noIntro = FALSE;

			if (cvd != NULL)
			{
				cname = classVersName(cvd);
				generateVersionStart(pt,&cvd -> version,fp);
			}

			prcode(fp,
"\n"
"	// Add the enums to the dictionary.\n"
"\n"
"	static sipEnumValueInstanceDef %senumValues[] = {\n"
				,(cvd != NULL ? cname : "")
				);
		}

		generateVersionStart(pt,&ed -> version,fp);

		for (evd = ed -> values; evd != NULL; evd = evd -> next)
		{
			prcode(fp,
"		{%N, ",evd -> name);

			if (cvd != NULL)
				prcode(fp,"%s%s::",(isProtectedEnum(ed) ? "sip" : ""),cname);

			prcode(fp,"%s},\n"
				,evd -> name -> text);
		}

		generateVersionEnd(pt,&ed -> version,fp);
	}

	if (!noIntro)
	{
		prcode(fp,
"		{NULL}\n"
"	};\n"
"\n"
"	if (sipAddEnumInstances("
			);

		if (cvd != NULL)
			prcode(fp,"((PyClassObject *)sipClass_%s) -> cl_dict,%s",cname,cname);
		else
			prcode(fp,"sipModule.md_dict,");

		prcode(fp,"enumValues) < 0)\n"
"		return NULL;\n"
			);

		if (cvd != NULL)
			generateVersionEnd(pt,&cvd -> version,fp);
	}
}


/*
 * Generate the code to add a set of longs to a dictionary.
 */

static void generateLongs(sipSpec *pt,classVersDef *cvd,varDef *vars,FILE *fp)
{
	char *cname;
	int noIntro;
	varDef *vd;

	noIntro = TRUE;

	for (vd = vars; vd != NULL; vd = vd -> next)
	{
		char *vname = vd -> name -> text;
		argType vtype = vd -> type.atype;

		if (vd -> module != pt -> module)
			continue;

		if (!(vtype == enum_type || vtype == ushort_type || vtype == short_type || vtype == uint_type || vtype == int_type || vtype == ulong_type || vtype == long_type || vtype == bool_type))
			continue;

		if (cvd != NULL && !isStaticVar(vd))
			continue;

		if (noIntro)
		{
			if (cvd != NULL)
			{
				cname = classVersName(cvd);
				generateVersionStart(pt,&cvd -> version,fp);
			}

			prcode(fp,
"\n"
"	// Add the longs to the dictionary.\n"
"\n"
"	static sipLongInstanceDef %slongInstances[] = {\n"
				,(cvd != NULL ? cname : ""));

			noIntro = FALSE;
		}

		generateVersionStart(pt,&vd -> version,fp);

		if (cvd != NULL)
			prcode(fp,
"		{%N, %s::%s},\n"
				,vd -> name,cname,vname);
		else
			prcode(fp,
"		{%N, %s},\n"
				,vd -> name,vname);

		generateVersionEnd(pt,&vd -> version,fp);
	}

	if (!noIntro)
	{
		prcode(fp,
"		{NULL}\n"
"	};\n"
"\n"
"	if (sipAddLongInstances(");

		if (cvd != NULL)
			prcode(fp,"((PyClassObject *)sipClass_%s) -> cl_dict,%s",cname,cname);
		else
			prcode(fp,"sipModule.md_dict,");

		prcode(fp,"longInstances) < 0)\n"
"		return NULL;\n"
			);

		if (cvd != NULL)
			generateVersionEnd(pt,&cvd -> version,fp);
	}
}


/*
 * Generate the code to add a set of doubles to a dictionary.
 */

static void generateDoubles(sipSpec *pt,classVersDef *cvd,varDef *vars,FILE *fp)
{
	char *cname;
	int noIntro;
	varDef *vd;

	noIntro = TRUE;

	for (vd = vars; vd != NULL; vd = vd -> next)
	{
		char *vname = vd -> name -> text;
		argType vtype = vd -> type.atype;

		if (vd -> module != pt -> module)
			continue;

		if (!(vtype == float_type || vtype == double_type))
			continue;

		if (cvd != NULL && !isStaticVar(vd))
			continue;

		if (noIntro)
		{
			if (cvd != NULL)
			{
				cname = classVersName(cvd);
				generateVersionStart(pt,&cvd -> version,fp);
			}

			prcode(fp,
"\n"
"	// Add the doubles to the dictionary.\n"
"\n"
"	static sipDoubleInstanceDef %sdoubleInstances[] = {\n"
				,(cvd != NULL ? cname : ""));

			noIntro = FALSE;
		}

		generateVersionStart(pt,&vd -> version,fp);

		if (cvd != NULL)
			prcode(fp,
"		{%N, %s::%s},\n"
				,vd -> name,cname,vname);
		else
			prcode(fp,
"		{%N, %s},\n"
				,vd -> name,vname);

		generateVersionEnd(pt,&vd -> version,fp);
	}

	if (!noIntro)
	{
		prcode(fp,
"		{NULL}\n"
"	};\n"
"\n"
"	if (sipAddDoubleInstances(");

		if (cvd != NULL)
			prcode(fp,"((PyClassObject *)sipClass_%s) -> cl_dict,%s",cname,cname);
		else
			prcode(fp,"sipModule.md_dict,");

		prcode(fp,"doubleInstances) < 0)\n"
"		return NULL;\n"
			);

		if (cvd != NULL)
			generateVersionEnd(pt,&cvd -> version,fp);
	}
}


/*
 * Generate the implementation of the module proxy.
 */

static void generateModuleProxy(sipSpec *pt,FILE *fp)
{
	char *mname = pt -> module -> name -> text;
	funcArgsList *fl;

	prcode(fp,
"\n"
"#include \"sip%sProxy%s.moc\"\n"
"\n"
"char *sipProxy%s::getProxySlot(char *sigargs)\n"
"{\n"
"	static char *tab[] = {\n"
		,mname,mname,mname);

	for (fl = pt -> sigargs; fl != NULL; fl = fl -> next)
	{
		generateOredVersionStart(pt,fl -> vol,fp);

		prcode(fp,
"		SLOT(proxySlot(");

		generateArgs(fl -> fa,Declaration,fp);

		prcode(fp,")),\n"
			);

		generateOredVersionEnd(pt,fl -> vol,fp);
	}

	prcode(fp,
"		NULL\n"
"	};\n"
"\n"
"	return searchProxySlotTable(tab,sigargs);\n"
"}\n"
		);

	for (fl = pt -> sigargs; fl != NULL; fl = fl -> next)
	{
		prcode(fp,
"\n"
			);

		generateOredVersionStart(pt,fl -> vol,fp);

		prcode(fp,
"void sipProxy%s::proxySlot(",mname);

		generateArgs(fl -> fa,Definition,fp);

		prcode(fp,")\n"
"{\n"
"	int relLock = sipCondAcquireLock();\n"
			);

		generateTupleBuilder(pt,fl -> fa,FALSE,fp);

		prcode(fp,
"\n"
"	if (sipArgs == NULL || sipEmitToSlot(&sipRealSlot,sipArgs) < 0)\n"
"		PyErr_Print();\n"
"\n"
"	Py_XDECREF(sipArgs);\n"
"\n"
"	sipCondReleaseLock(relLock);\n"
"}\n"
			);

		generateOredVersionEnd(pt,fl -> vol,fp);
	}
}


/*
 * Generate the C++ code for a class.
 */

static void generateClassCpp(classDef *cd,sipSpec *pt,char *codeDir,char *cppSuffix)
{
	char *cppfile, *cname = cd -> name -> text;
	char *cmname = cd -> module -> name -> module -> name -> text;
	classVersDef *cvd;
	FILE *fp;

	cppfile = concat(codeDir,"/sip",cmname,cname,cppSuffix,NULL);
	fp = createFile(pt,cppfile,"//","C++ class wrapper code.");

	prcode(fp,
"\n"
"#include \"sip%sDecl%s.h\"\n"
"#include \"sip%s%s.h\"\n"
		,cmname,cmname,cmname,cname);

	for (cvd = cd -> cvds; cvd != NULL; cvd = cvd -> next)
		generateClassVersCpp(cvd,pt,fp);

	closeFile(fp);
	free(cppfile);
}


/*
 * Generate the C++ code for a class version.
 */

static void generateClassVersCpp(classVersDef *cvd,sipSpec *pt,FILE *fp)
{
	char *cname = classVersName(cvd);
	int noIntro;

	prcode(fp,
"\n"
"\n"
		);

	generateVersionStart(pt,&cvd -> version,fp);

	/* Generate the type convertors. */

	if (cvd -> convfromcode != NULL)
	{
		prcode(fp,
"\n"
"PyObject *sipConvertFrom_%s(const %s *sipCpp)\n"
"{\n"
"	if (sipCpp == NULL)\n"
"	{\n"
"		Py_INCREF(Py_None);\n"
"		return Py_None;\n"
"	}\n"
"\n"
			,cname,cname);

		generateCppCodeBlock(NULL,cvd -> convfromcode,cname,fp);

		prcode(fp,
"}\n"
			);
	}

	prcode(fp,
"\n"
"PyObject *sipClass_%s;\n"
		,cname);

	if (!isMapped(cvd))
	{
		varDef *vd;
		classVersList *cvl;
		visibleList *vl;
		spmDef *sd;

		/* Generate the Python object structures. */

		prcode(fp,
"\n"
"static void sipDealloc_%s(sipThisType *);\n"
			,cname);

		for (sd = cvd -> spms; sd != NULL; sd = sd -> next)
		{
			generateSpecialMethodStart(cvd,sd -> spm,fp);

			prcode(fp,";\n"
				);

			generateSpecialMethodEnd(sd -> spm,fp);
		}

		/* Define the table for number methods if any. */

		if (cvd -> combnummeth || cvd -> combnummethx)
		{
			prcode(fp,
"\n"
"static PyNumberMethods sipNumMethods_%s = {\n"
				,cname);

			if (cvd -> combnummeth & SPM_NUM_ADD)
				prcode(fp,
"	sip__add__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_SUB)
				prcode(fp,
"	sip__sub__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_MUL)
				prcode(fp,
"	sip__mul__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_DIV)
				prcode(fp,
"	sip__div__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_MOD)
				prcode(fp,
"	sip__mod__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_DIVMOD)
				prcode(fp,
"	sip__divmod__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_POW)
				prcode(fp,
"	sip__pow__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_NEG)
				prcode(fp,
"	sip__neg__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_POS)
				prcode(fp,
"	sip__pos__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_ABS)
				prcode(fp,
"	sip__abs__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_NONZERO)
				prcode(fp,
"	sip__nonzero__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_INVERT)
				prcode(fp,
"	sip__invert__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_LSHIFT)
				prcode(fp,
"	sip__lshift__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_RSHIFT)
				prcode(fp,
"	sip__rshift__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_AND)
				prcode(fp,
"	sip__and__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_XOR)
				prcode(fp,
"	sip__xor__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_OR)
				prcode(fp,
"	sip__or__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_COERCE)
				prcode(fp,
"	sip__coerce__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_INT)
				prcode(fp,
"	sip__int__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_LONG)
				prcode(fp,
"	sip__long__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_FLOAT)
				prcode(fp,
"	sip__float__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_OCT)
				prcode(fp,
"	sip__oct__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_HEX)
				prcode(fp,
"	sip__hex__%s,\n"
					,cname);
		else
				prcode(fp,
"	0\n"
					);

			/* The inplace methods came in with Python 2.0. */

			prcode(fp,
"#if PY_VERSION_HEX >= 0x02000000\n"
				);

			if (cvd -> combnummeth & SPM_NUM_IADD)
				prcode(fp,
"	sip__iadd__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_ISUB)
				prcode(fp,
"	sip__isub__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_IMUL)
				prcode(fp,
"	sip__imul__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_IDIV)
				prcode(fp,
"	sip__idiv__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_IMOD)
				prcode(fp,
"	sip__imod__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_IPOW)
				prcode(fp,
"	sip__ipow__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_ILSHIFT)
				prcode(fp,
"	sip__ilshift__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_IRSHIFT)
				prcode(fp,
"	sip__irshift__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummeth & SPM_NUM_IAND)
				prcode(fp,
"	sip__iand__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummethx & SPM_NUM_IXOR)
				prcode(fp,
"	sip__ixor__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combnummethx & SPM_NUM_IOR)
				prcode(fp,
"	sip__ior__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			prcode(fp,
"#endif\n"
				);

			prcode(fp,
"};\n"
				);
		}

		/* Define the table for sequence methods if any. */

		if (cvd -> combseqmeth || cvd -> combseqmethx)
		{
			prcode(fp,
"\n"
"static PySequenceMethods sipSeqMethods_%s = {\n"
				,cname);

			if (cvd -> combseqmeth & SPM_SEQ_LEN)
				prcode(fp,
"	sip__len__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combseqmeth & SPM_SEQ_CONCAT)
				prcode(fp,
"	sip__add__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combseqmeth & SPM_SEQ_REPEAT)
				prcode(fp,
"	sip__mul__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combseqmeth & SPM_SEQ_GETITEM)
				prcode(fp,
"	sip__getitem__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combseqmeth & SPM_SEQ_SETITEM)
				prcode(fp,
"	sip__setitem__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combseqmeth & SPM_SEQ_GETSLICE)
				prcode(fp,
"	sip__getslice__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combseqmeth & SPM_SEQ_SETSLICE)
				prcode(fp,
"	sip__setslice__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			/* __contains__ came in with Python 1.6. */

			prcode(fp,
"#if PY_VERSION_HEX >= 0x01060000\n"
				);

			if (cvd -> combseqmeth & SPM_SEQ_CONTAINS)
				prcode(fp,
"	sip__contains__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			prcode(fp,
"#endif\n"
				);

			/* __iadd__ and __imul__ came in with Python 2.0. */

			prcode(fp,
"#if PY_VERSION_HEX >= 0x02000000\n"
				);

			if (cvd -> combseqmeth & SPM_SEQ_ICONCAT)
				prcode(fp,
"	sip__iadd__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combseqmeth & SPM_SEQ_IREPEAT)
				prcode(fp,
"	sip__imul__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			prcode(fp,
"#endif\n"
				);

			prcode(fp,
"};\n"
				);
		}

		/* Define the table for mapping methods if any. */

		if (cvd -> combmapmeth || cvd -> combmapmethx)
		{
			prcode(fp,
"\n"
"static PyMappingMethods sipMapMethods_%s = {\n"
				,cname);

			if (cvd -> combmapmeth & SPM_MAP_LEN)
				prcode(fp,
"	sip__len__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combmapmeth & SPM_MAP_GETITEM)
				prcode(fp,
"	sip__getitem__%s,\n"
					,cname);
		else
				prcode(fp,
"	0,\n"
					);

			if (cvd -> combmapmeth & SPM_MAP_SETITEM)
				prcode(fp,
"	sip__setitem__%s,\n"
					,cname);
		else
				prcode(fp,
"	0\n"
					);

			prcode(fp,
"};\n"
				);
		}

		prcode(fp,
"\n"
"static PyTypeObject sipType_%s = {\n"
"	PyObject_HEAD_INIT(&PyType_Type)\n"
"	0,\n"
"	%N,\n"
"	sizeof (sipThisType),\n"
"	0,\n"
"	(destructor)sipDealloc_%s,\n"
"	0,\n"
"	0,\n"
"	0,\n"
			,cname,cvd -> cd -> name,cname);

		if (cvd -> combmeth & SPM_METH_CMP)
			prcode(fp,
"	sip__cmp__%s,\n"
				,cname);
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combmeth & SPM_METH_REPR)
			prcode(fp,
"	sip__repr__%s,\n"
				,cname);
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combnummeth || cvd -> combnummethx)
			prcode(fp,
"	&sipNumMethods_%s,\n"
				,cname);
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combseqmeth || cvd -> combseqmethx)
			prcode(fp,
"	&sipSeqMethods_%s,\n"
				,cname);
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combmapmeth || cvd -> combmapmethx)
			prcode(fp,
"	&sipMapMethods_%s,\n"
				,cname);
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combmeth & SPM_METH_HASH)
			prcode(fp,
"	sip__hash__%s,\n"
				,cname);
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combmeth & SPM_METH_CALL)
			prcode(fp,
"	sip__call__%s,\n"
				,cname);
		else
			prcode(fp,
"	0,\n"
				);

		if (cvd -> combmeth & SPM_METH_STR)
			prcode(fp,
"	sip__str__%s,\n"
				,cname);
		else
			prcode(fp,
"	0,\n"
				);

		prcode(fp,
"	0,\n"
"	0,\n"
"	0,\n"
"	Py_TPFLAGS_DEFAULT,\n"
"	0,\n"
"	0,\n"
"	0,\n"
"	0,\n"
"	0,\n"
			);

		if (cvd -> combmeth == SPM_METH_RICH)
			prcode(fp,
"#if PY_VERSION_HEX >= 0x02010000\n"
"	sip__richcompare__%s,\n"
"#endif\n"
				,cname);

		prcode(fp,
"};\n"
			);

		/* Generate any local class code. */

		generateCppCodeBlock(NULL,cvd -> cppcode,cname,fp);

		generateClassFunctions(pt,cvd,fp);

		/* Generate the variable handlers. */

		for (vd = cvd -> vars; vd != NULL; vd = vd -> next)
			if (!isStaticVar(vd))
				generateVariableHandler(pt,cvd,vd,fp);

		/* Generate the attribute table. */

		prcode(fp,
"\n"
"PyMethodDef sipClassAttrTab_%s[] = {\n"
			,cname);

		/*
		 * Only provide an entry point if there is at least one
		 * overload that is defined in this class and is a non-abstract
		 * function or slot.  We allow private (even though we don't
		 * actually generate code) because we need to intercept the
		 * name before it reaches a more public version further up the
		 * class hierachy.
		 */

		for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
		{
			int isOver;
			overDef *od;
			versionOrList *vol;

			isOver = FALSE;

			for (od = vl -> cv -> overs; od != NULL; od = od -> next)
			{
				if (od -> common != vl -> m)
					continue;

				if (!isSignal(od) && !isAbstract(od) && (isProtected(od) || vl -> cv == cvd))
				{
					if (!isOver)
					{
						vol = orVersionQual(pt,NULL,&od -> version);
						isOver = TRUE;
					}
					else if (vol != NULL)
						vol = orVersionQual(pt,vol,&od -> version);
				}
			}

			if (isOver)
			{
				reduceOredVersion(pt,&vol);

				generateOredVersionStart(pt,vol,fp);

				prcode(fp,
"	{%N, sipDo_%s_%s, METH_VARARGS, NULL},\n"
					,vl -> m -> pyname,cname,vl -> m -> pyname -> text);

				generateOredVersionEnd(pt,vol,fp);
				freeVersionOrList(vol);
			}
		}

		prcode(fp,
"	{NULL,NULL,0,NULL}\n"
"};\n"
			);

		/* Generate the variable tables. */

		noIntro = TRUE;

		for (vd = cvd -> vars; vd != NULL; vd = vd -> next)
		{
			if (isStaticVar(vd))
				continue;

			if (noIntro)
			{
				noIntro = FALSE;

				prcode(fp,
"\n"
"PyMethodDef sipClassVarTab_%s[] = {\n"
					,cname);
			}

			generateVersionStart(pt,&vd -> version,fp);

			prcode(fp,
"	{%N, sipGetSetVar_%s_%s, 0, NULL},\n"
				,vd -> name,cname,vd -> name -> text);

			generateVersionEnd(pt,&vd -> version,fp);
		}

		if (!noIntro)
		{
			prcode(fp,
"	{NULL}\n"
"};\n"
"\n"
"PyMethodDef *sipClassVarHierTab_%s[] = {\n"
				,cname);

			for (cvl = cvd -> hierachy; cvl != NULL; cvl = cvl -> next)
				for (vd = cvl -> cv -> vars; vd != NULL; vd = vd -> next)
					if (!isStaticVar(vd))
					{
						prcode(fp,
"	sipClassVarTab_%s,\n"
							,classVersName(cvl -> cv));

						break;
					}

			prcode(fp,
"	NULL\n"
"};\n"
				);
		}
	}

	/* Generate the type convertors. */

	prcode(fp,
"\n"
"int sipCanConvertTo_%s(PyObject *sipPy)\n"
"{\n"
		,cname);

	if (cvd -> canconvtocode != NULL)
		generateCppCodeBlock(NULL,cvd -> canconvtocode,cname,fp);
	else
		prcode(fp,
"	return sipIsSubClassInstance(sipPy,sipClass_%s);\n"
			,cname);

	prcode(fp,
"}\n"
		);

	prcode(fp,
"\n"
"%s sipConvertTo_%s(PyObject *sipPy,%s **sipCppPtr,int sipWillDeref,int *sipIsErr)\n"
"{\n"
"	if (*sipIsErr || sipPy == NULL)\n"
"		return%s;\n"
"\n"
		,(cvd -> convtocode != NULL ? "int" : "void"),cname,cname
		,(cvd -> convtocode != NULL ? " 0" : ""));

	if (cvd -> convtocode != NULL)
		generateCppCodeBlock(NULL,cvd -> convtocode,cname,fp);
	else
		prcode(fp,
"	if (sipPy == Py_None)\n"
"	{\n"
"		sipCheckNone(sipWillDeref,sipIsErr,%N);\n"
"		*sipCppPtr = NULL;\n"
"\n"
"		return;\n"
"	}\n"
"\n"
"	*sipCppPtr = (%s *)sipConvertToCpp(sipPy,sipClass_%s,sipIsErr);\n"
			,cvd -> cd -> name
			,cname,cname);

	prcode(fp,
"}\n"
		);

	prcode(fp,
"\n"
"%s *sipForceConvertTo_%s(PyObject *valobj,int *iserrp)\n"
"{\n"
"	if (*iserrp || valobj == NULL || valobj == Py_None)\n"
"		return NULL;\n"
"\n"
"	if (sipCanConvertTo_%s(valobj))\n"
"	{\n"
"		%s *val;\n"
"\n"
"		sipConvertTo_%s(valobj,&val,0,iserrp);\n"
"\n"
"		return val;\n"
"	}\n"
"\n"
"	sipBadClass(%N);\n"
"	*iserrp = 1;\n"
"\n"
"	return NULL;\n"
"}\n"
		,cname,cname
		,cname
		,cname
		,cname
		,cvd -> cd -> name);

	generateVersionEnd(pt,&cvd -> version,fp);
}


/*
 * Generate the start of the declaration for a special method handler.
 */

static void generateSpecialMethodStart(classVersDef *cvd,specialPyMethod *spm,FILE *fp)
{
	if (spm -> pyvers != 0)
	{
		char buf[20];

		sprintf(buf,"%08x",spm -> pyvers);

		prcode(fp,
"#if PY_VERSION_HEX >= 0x%s\n"
			,buf);
	}

	prcode(fp,
"static %s sip%s%s(",spm -> rettype,spm -> name,classVersName(cvd));

	if (spm -> nrargs >= 0)
	{
		int i;

		for (i = 0; i < spm -> nrargs; ++i)
		{
			if (i > 0)
				prcode(fp,",");

			prcode(fp,"PyObject *a%d",i);
		}
	}
	else
	{
		char *args;

		switch (spm -> id)
		{
		case SPM_NUM_COERCE:
			args = "PyObject **a0,PyObject **a1";
			break;

		case SPM_SEQ_REPEAT:
		case SPM_SEQ_IREPEAT:
		case SPM_SEQ_GETITEM:
			args = "PyObject *a0,int a1";
			break;

		case SPM_SEQ_SETITEM:
			args = "PyObject *a0,int a1,PyObject *a2";
			break;

		case SPM_SEQ_GETSLICE:
			args = "PyObject *a0,int a1,int a2";
			break;

		case SPM_SEQ_SETSLICE:
			args = "PyObject *a0,int a1,int a2,PyObject *a3";
			break;

		case SPM_METH_RICH:
			args = "PyObject *a0,PyObject *a1,int a2";
			break;
		}

		prcode(fp,"%s",args);
	}

	prcode(fp,")");
}


/*
 * Generate the end of the declaration for a special method handler.
 */

static void generateSpecialMethodEnd(specialPyMethod *spm,FILE *fp)
{
	if (spm -> pyvers != 0)
		prcode(fp,
"#endif\n"
			);
}


/*
 * Generate a variable handler.
 */

static void generateVariableHandler(sipSpec *pt,classVersDef *cvd,varDef *vd,FILE *fp)
{
	char *cname = classVersName(cvd);
	char *vname = vd -> name -> text;
	char *deref;
	argType atype = vd -> type.atype;

	generateVersionStart(pt,&vd -> version,fp);

	prcode(fp,
"\n"
"static PyObject *sipGetSetVar_%s_%s(PyObject *sipThisObj,PyObject *valobj)\n"
"{\n"
		,cname,vname);

	if (atype == class_type)
		prcode(fp,
"	int iserr = 0;\n"
			);

	prcode(fp,
"	");

	generateValueType(&vd -> type,fp);

	prcode(fp,"val;\n"
"	%s *ptr;\n"
"\n"
"	if ((ptr = (%s *)sipGetCppPtr((sipThisType *)sipThisObj,sipClass_%s)) == NULL)\n"
"		return NULL;\n"
"\n"
"	if (valobj == NULL)\n"
"	{\n"
		,cname,cname,cname);

	/* Generate the get handler part. */

	prcode(fp,
"		val = %sptr -> %s;\n"
"\n"
					,((atype == class_type && vd -> type.nrderefs == 0) ? "&" : ""),vname);

	switch (atype)
	{
	case class_type:
		generateVarClassConversion(pt,vd,fp);
		break;

	case bool_type:
		prcode(fp,
"		valobj = sipConvertFromBool((int)val);\n"
			);

		break;

	case ustring_type:
	case string_type:
		if (vd -> type.nrderefs == 0)
			prcode(fp,
"		valobj = PyString_FromStringAndSize(%s&val,1);\n"
				,(atype == ustring_type) ? "(char *)" : "");
		else
			prcode(fp,
"		valobj = PyString_FromString(val);\n"
				);

		break;

	case float_type:
		prcode(fp,
"		valobj = PyFloat_FromDouble((double)val);\n"
			);
		break;

	case double_type:
		prcode(fp,
"		valobj = PyFloat_FromDouble(val);\n"
			);
		break;

	case enum_type:
	case ushort_type:
	case short_type:
	case uint_type:
	case int_type:
	case ulong_type:
		prcode(fp,
"		valobj = PyInt_FromLong((long)val);\n"
			);
		break;

	case long_type:
		prcode(fp,
"		valobj = PyInt_FromLong(val);\n"
			);
		break;

	case struct_type:
	case voidptr_type:
		prcode(fp,
"		valobj = sipConvertFromVoidPtr(val);\n"
			);
		break;
	}

	prcode(fp,
"\n"
"		return valobj;\n"
"	}\n"
		);

	/* Generate the set handler part. */

	prcode(fp,
"\n"
		);

	generateObjToCppConversion(&vd -> type,"val","\t",fp);

	deref = "";

	if (atype == class_type)
	{
		if (vd -> type.nrderefs == 0)
			deref = "*";

		prcode(fp,
"\n"
"	if (iserr)\n"
			);
	}
	else if (atype == ustring_type || atype == string_type)
	{
		if (vd -> type.nrderefs == 0)
			deref = "*";

		prcode(fp,
"\n"
"	if (val == NULL)\n"
			);
	}
	else
		prcode(fp,
"\n"
"	if (PyErr_Occurred() != NULL)\n"
			);

	prcode(fp,
"	{\n"
"		sipBadSetType(%N,%N);\n"
"		return NULL;\n"
"	}\n"
"\n"
"	ptr -> %s = %sval;\n"
"\n"
"	Py_INCREF(Py_None);\n"
"	return Py_None;\n"
"}\n"
		,cvd -> cd -> name,vd -> name,vname,deref);

	generateVersionEnd(pt,&vd -> version,fp);
}


/*
 * Generate an object conversion fragment.
 */

static void generateObjConversion(varDef *vd,FILE *fp)
{
	prcode(fp,
"			");

	generateValueType(&vd -> type,fp);

	prcode(fp,"vc;\n"
"\n"
		);

	generateObjToCppConversion(&vd -> type,"vc","\t\t\t",fp);

	prcode(fp,
"\n"
"			if (val == vc)\n"
		);
}


/*
 * Generate an variable class conversion fragment.
 */

static void generateVarClassConversion(sipSpec *pt,varDef *vd,FILE *fp)
{
	classVersDef *cvd = vd -> type.u.cvd;
	char *cname = classVersName(cvd);

	if (isMapped(cvd))
	{
		checkForConvertFromCode(pt,cvd);

		prcode(fp,
"		valobj = sipConvertFrom_%s(val);\n"
			,cname);
	}
	else
	{
		prcode(fp,
"		valobj = sipMapCppToSelf(val,");

		if (cvd -> convtosubcode == NULL)
			prcode(fp,"sipClass_%s",cname);
		else
			prcode(fp,"sipSubClass_%s(val)",cname);

		prcode(fp,");\n"
			);
	}
}


/*
 * Generate the declaration of a variable that is initialised from a Python
 * object.
 */

static void generateObjToCppConversion(argDef *ad,char *name,char *indent,FILE *fp)
{
	if (ad -> atype == class_type)
	{
		char *cname = classVersName(ad -> u.cvd);

		prcode(fp,
"%s%s = sipForceConvertTo_%s(%sobj,&iserr);\n"
			,indent,name,cname,name);
	}
	else if (ad -> atype == enum_type)
		prcode(fp,
"%s%s = (%E)PyInt_AsLong(%sobj);\n"
			,indent,name,ad -> u.ed,name);
	else
	{
		char *fmt;

		switch (ad -> atype)
		{
		case ustring_type:
			if (ad -> nrderefs == 0)
				fmt = "%s%s = *(unsigned char *)PyString_AsString(%sobj);\n";
			else
				fmt = "%s%s = (unsigned char *)PyString_AsString(%sobj);\n";
			break;

		case string_type:
			if (ad -> nrderefs == 0)
				fmt = "%s%s = *PyString_AsString(%sobj);\n";
			else
				fmt = "%s%s = PyString_AsString(%sobj);\n";
			break;

		case float_type:
			fmt = "%s%s = (float)PyFloat_AsDouble(%sobj);\n";
			break;

		case double_type:
			fmt = "%s%s = PyFloat_AsDouble(%sobj);\n";
			break;

		case bool_type:
			fmt = "%s%s = (bool)PyInt_AsLong(%sobj);\n";
			break;

		case ushort_type:
			fmt = "%s%s = (unsigned short)PyInt_AsLong(%sobj);\n";
			break;

		case short_type:
			fmt = "%s%s = (short)PyInt_AsLong(%sobj);\n";
			break;

		case uint_type:
			fmt = "%s%s = (unsigned)PyInt_AsLong(%sobj);\n";
			break;

		case int_type:
			fmt = "%s%s = (int)PyInt_AsLong(%sobj);\n";
			break;

		case ulong_type:
			fmt = "%s%s = (unsigned long)PyInt_AsLong(%sobj);\n";
			break;

		case long_type:
			fmt = "%s%s = PyInt_AsLong(%sobj);\n";
			break;

		case struct_type:
			prcode(fp,
"%s%s = (struct %s *)sipConvertToVoidPtr(%sobj);\n"
				,indent,name,ad -> u.sname,name);
			break;

		case voidptr_type:
			fmt = "%s%s = sipConvertToVoidPtr(%sobj);\n";
			break;
		}

		if (fmt != NULL)
			prcode(fp,fmt,indent,name,name);
	}
}


/*
 * Generate the member functions for a class.
 */

static void generateClassFunctions(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	char *cname = classVersName(cvd);
	visibleList *vl;
	classList *cl;
	spmDef *sd;

	/* Any complex code. */

	if (isComplex(cvd))
		if (cannotCreate(cvd))
			generateVirtualHandlers(pt,cvd,fp);
		else
			generateComplexCode(pt,cvd,fp);

	/* The member functions. */

	for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
	{
		generateFunction(pt,vl -> m,vl -> cv -> overs,cvd,vl -> cv,TRUE,fp);
		generateFunction(pt,vl -> m,vl -> cv -> overs,cvd,vl -> cv,FALSE,fp);
	}

	/* The sub-class convertor if there is one. */

	if (cvd -> convtosubcode != NULL)
	{
		prcode(fp,
"\n"
"// Convert to a sub-class if possible.\n"
"\n"
"PyObject *sipSubClass_%s(const %s *sipCpp)\n"
"{\n"
"	PyObject *sipClass;\n"
			,cname,cname);

		generateCppCodeBlock(NULL,cvd -> convtosubcode,cname,fp);

		prcode(fp,
"\n"
"	return sipClass;\n"
"}\n"
			);
	}

	/* The cast function. */

	prcode(fp,
"\n"
"// Cast a pointer to a type somewhere in its superclass hierachy.\n"
"\n"
"const void *sipCast_%s(const void *ptr,PyObject *targetClass)\n"
"{\n"
		,cname);

	if (cvd -> supers != NULL)
		prcode(fp,
"	const void *res;\n"
"\n"
			);

	prcode(fp,
"	if (targetClass == sipClass_%s)\n"
"		return ptr;\n"
		,cname);

	for (cl = cvd -> supers; cl != NULL; cl = cl -> next)
	{
		char *sname = cl -> c -> name -> text;

		prcode(fp,
"\n"
"	if ((res = sipCast_%s((%s *)(%s *)ptr,targetClass)) != NULL)\n"
"		return res;\n"
			,sname,sname,cname);
	}

	prcode(fp,
"\n"
"	return NULL;\n"
"}\n"
		);

	/* The dealloc function. */

	prcode(fp,
"\n"
"static void sipDealloc_%s(sipThisType *sipThis)\n"
"{\n"
		,cname);

	if (cvd -> dtorcode != NULL)
		generateCppCodeBlock(NULL,cvd -> dtorcode,cname,fp);

	/* Generate the right dtor call - if there is one. */

	if (!cannotCreate(cvd) && (isComplex(cvd) || !noPublicDtor(cvd)))
	{
		prcode(fp,
"	if (sipThis -> u.cppPtr != NULL)\n"
"	{\n"
			);

		/* Disable the virtual handlers. */

		if (isComplex(cvd))
			prcode(fp,
"		if (!sipIsSimple(sipThis))\n"
"			((sip%s *)sipThis -> u.cppPtr) -> sipPyThis = NULL;\n"
"\n"
				,cname);

		prcode(fp,
"		if (sipIsPyOwned(sipThis))\n"
			);

		if (isComplex(cvd))
		{
			if (noPublicDtor(cvd))
				prcode(fp,
"			if (!sipIsSimple(sipThis))\n"
"				delete (sip%s *)sipThis -> u.cppPtr;\n"
					,cname);
			else
				prcode(fp,
"			if (sipIsSimple(sipThis))\n"
"				delete (%s *)sipThis -> u.cppPtr;\n"
"			else\n"
"				delete (sip%s *)sipThis -> u.cppPtr;\n"
					,cname,cname);
		}
		else
			prcode(fp,
"			delete (%s *)sipThis -> u.cppPtr;\n"
				,cname);

		prcode(fp,
"	}\n"
"\n"
			);
	}

	prcode(fp,
"	sipDeleteThis(sipThis);\n"
"}\n"
		);

	/* The special method implementations. */

	for (sd = cvd -> spms; sd != NULL; sd = sd -> next)
	{
		generateSpecialMethodStart(cvd,sd -> spm,fp);

		prcode(fp,"\n"
"{\n"
			);

		generateCppCodeBlock(NULL,sd -> code,cname,fp);

		prcode(fp,
"}\n"
			);

		generateSpecialMethodEnd(sd -> spm,fp);
	}

	/* The Python constructor. */

	generatePythonConstructor(pt,cvd,fp);
}


/*
 * Generate the code specific to complex classes.
 */

static void generateComplexCode(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	char *cname = classVersName(cvd);
	int noIntro, nrVirts;
	visibleList *vl;
	virtVersDef *vvd;
	ctorDef *ct;

	nrVirts = countVirtuals(cvd);

	/* Generate the wrapper class constructors. */

	for (ct = cvd -> ctors; ct != NULL; ct = ct -> next)
	{
		char *prefix;
		int a;

		if (isPrivateCtor(ct))
			continue;

		generateVersionStart(pt,&ct -> version,fp);

		prcode(fp,
"\n"
"sip%s::sip%s(",cname,cname);

		generateArgs(&ct -> args,Definition,fp);

		prcode(fp,"): %s(",cname);

		prefix = "";

		for (a = 0; a < ct -> args.nrArgs; ++a)
		{
			prcode(fp,"%sa%d",prefix,a);
			prefix = ",";
		}

		prcode(fp,")\n"
"{\n"
"	sipCommonCtor(%s,%d);\n"
"}\n"
			,(nrVirts > 0 ? "sipPyMethods" : "NULL"),nrVirts);

		generateVersionEnd(pt,&ct -> version,fp);
	}

	/* The destructor. */

	prcode(fp,
"\n"
"sip%s::~sip%s()\n"
"{\n"
"	sipCommonDtor(sipPyThis);\n"
"}\n"
		,cname,cname);

	/* Generate the virtual catchers. */
 
	for (vvd = cvd -> vmembers; vvd != NULL; vvd = vvd -> next)
	{
		int virtNr;
		virtOverDef *vod;
 
		generateVersionStart(pt,&vvd -> version,fp);

		virtNr = 0;
 
		for (vod = vvd -> vo; vod != NULL; vod = vod -> next)
			if (!isPrivate(vod -> o))
				generateVirtualCatcher(cvd,virtNr++,vod,fp);
 
		generateVersionEnd(pt,&vvd -> version,fp);
	}

	/* Generate the virtual handlers. */

	generateVirtualHandlers(pt,cvd,fp);

	/* Generate the wrapper around each protected member function. */

	generateProtectedDefinitions(pt,cvd,fp);

	/* Generate the emitter functions. */

	for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
	{
		overDef *od;

		for (od = vl -> cv -> overs; od != NULL; od = od -> next)
		{
			if (od -> common != vl -> m)
				continue;

			if (isSignal(od))
			{
				generateEmitter(pt,cvd,vl,fp);
				break;
			}
		}
	}

	/* Generate the table of signals to support fan-outs. */

	noIntro = TRUE;

	for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
	{
		overDef *od;

		for (od = vl -> cv -> overs; od != NULL; od = od -> next)
		{
			if (od -> common != vl -> m)
				continue;

			if (isSignal(od))
			{
				if (noIntro)
				{
					setHasSigSlots(cvd);

					prcode(fp,
"\n"
"static sipQtSignal sipSignals_%s[] = {\n"
						,cname);

					noIntro = FALSE;
				}

				prcode(fp,
"	{%N, %s_emit_%s},\n"
					,vl -> m -> pyname,cname,vl -> m -> pyname -> text);

				break;
			}
		}
	}

	if (!noIntro)
		prcode(fp,
"	{NULL, NULL}\n"
"};\n"
			);
}


/*
 * Generate the protected enums for a class.
 */

static void generateProtectedEnums(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	enumDef *ed;

	for (ed = cvd -> enums; ed != NULL; ed = ed -> next)
		if (isProtectedEnum(ed))
		{
			char *eol;
			enumValueDef *evd;

			generateVersionStart(pt,&ed -> version,fp);

			prcode(fp,
"\n"
"	// Expose this protected enum.\n"
"\n"
"	enum");

			if (ed -> name != NULL)
				prcode(fp," sip%s",ed -> name);

			prcode(fp," {");

			eol = "\n";

			for (evd = ed -> values; evd != NULL; evd = evd -> next)
			{
				prcode(fp,"%s"
"		%s = %s::%s",eol,evd -> name -> text,classVersName(cvd),evd -> name -> text);

				eol = ",\n";
			}

			prcode(fp,"\n"
"	};\n"
				);

			generateVersionEnd(pt,&ed -> version,fp);
		}
}


/*
 * Generate the catcher for a virtual function.
 */

static void generateVirtualCatcher(classVersDef *cvd,int virtNr,virtOverDef *vod,FILE *fp)
{
	char *cname = classVersName(cvd);
	char *ncname = classVersName(vod -> nearc);
	char *fcname = classVersName(vod -> farc);
	char *pname = vod -> o -> common -> pyname -> text;
	char *mname = vod -> o -> cppname;
	char *cast;
	nameDef *pmname = vod -> o -> common -> pyname;
	overDef *od = vod -> o;

	prcode(fp,
"\n");

	generateResultType(od -> result,fp);

	prcode(fp," sip%s::%s(",cname,mname);
	generateArgs(&od -> args,Definition,fp);
	prcode(fp,")%s\n"
"{\n"
"	int relLock;\n"
"\n"
		,(isConst(od) ? " const" : ""));

	cast = (isConst(od) ? "(sipMethodCache *)" : "");

	if (isAbstract(od))
	{
		prcode(fp,
"	if (sipIsPyMethod(%s&sipPyMethods[%d],sipPyThis,%N,%N,&relLock))\n"
"		%ssip%s::sipVH_%s(&sipPyMethods[%d],sipPyThis,relLock",cast,virtNr,cvd -> cd -> name,pmname,(od -> result != NULL ? "return " : ""),fcname,mname,virtNr);
 
		if (od -> args.nrArgs > 0)
			prcode(fp,",");
 
		generateArgCallList(&od -> args,fp);
 
		prcode(fp,");\n"
			);
	}
	else if (od -> result == NULL)
	{
		prcode(fp,
"	if (sipIsPyMethod(%s&sipPyMethods[%d],sipPyThis,NULL,%N,&relLock))\n"
"		sip%s::sipVH_%s(&sipPyMethods[%d],sipPyThis,relLock",cast,virtNr,pmname,fcname,mname,virtNr);
 
		if (od -> args.nrArgs > 0)
			prcode(fp,",");
 
		generateArgCallList(&od -> args,fp);
 
		prcode(fp,");\n"
"	else\n"
"		%s::%s(",ncname,mname);
 
		generateArgCallList(&od -> args,fp);
 
		prcode(fp,");\n"
			);
	}
	else
	{
		prcode(fp,
"	return sipIsPyMethod(%s&sipPyMethods[%d],sipPyThis,NULL,%N,&relLock) ?\n"
"		sip%s::sipVH_%s(&sipPyMethods[%d],sipPyThis,relLock",cast,virtNr,pmname,fcname,mname,virtNr);

		if (od -> args.nrArgs > 0)
			prcode(fp,",");
 
		generateArgCallList(&od -> args,fp);
 
		prcode(fp,") :\n"
"		%s::%s(",ncname,mname);
 
		generateArgCallList(&od -> args,fp);
 
		prcode(fp,");\n"
			);
	}
 
	prcode(fp,
"}\n"
		);
}


/*
 * Generate the emitter function for a signal.
 */

static void generateEmitter(sipSpec *pt,classVersDef *cvd,visibleList *vl,FILE *fp)
{
	char *cname = classVersName(cvd);
	char *pname = vl -> m -> pyname -> text;
	overDef *od;

	prcode(fp,
"\n"
"int sip%s::sipEmit_%s(PyObject *sipArgs)\n"
"{\n"
"	int sipArgsParsed = 0;\n"
		,cname,pname);

	for (od = vl -> cv -> overs; od != NULL; od = od -> next)
	{
		if (od -> common != vl -> m || !isSignal(od))
			continue;

		/*
		 * Generate the code that parses the args and emits the
		 * appropriate overloaded signal.
		 */

		generateVersionStart(pt,&od -> version,fp);

		prcode(fp,
"\n"
"	{\n"
			);

		generateArgParser(&od -> args,TRUE,FALSE,fp);

		prcode(fp,
"		{\n"
			);

		generateArgConvertors(pt,&od -> args,FALSE,FALSE,"-1",fp);

		prcode(fp,
"			emit %s("
			,od -> cppname);

		generateArgs(&od -> args,Call,fp);

		prcode(fp,");\n"
"\n"
			);

		generateCallTidyUp(&od -> args,fp);

		prcode(fp,
"			return 0;\n"
"		}\n"
"	}\n"
			);

		generateVersionEnd(pt,&od -> version,fp);
	}

	prcode(fp,
"\n"
"	sipNoMethod(sipArgsParsed,%N,%N);\n"
"\n"
"	return -1;\n"
"}\n"
"\n"
"static int %s_emit_%s(sipThisType *w,PyObject *sipArgs)\n"
"{\n"
"	sip%s *ptr;\n"
"\n"
"	if ((ptr = (sip%s *)sipGetComplexCppPtr(w)) == NULL)\n"
"		return -1;\n"
"\n"
"	return ptr -> sipEmit_%s(sipArgs);\n"
"}\n"
		,cvd -> cd -> name,vl -> m -> pyname
		,cname,pname
		,cname
		,cname
		,pname);
}


/*
 * Generate the declarations of the protected wrapper functions for a class.
 */

static void generateProtectedDeclarations(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	int noIntro;
	visibleList *vl;

	noIntro = TRUE;

	for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
	{
		overDef *od;

		for (od = vl -> cv -> overs; od != NULL; od = od -> next)
		{
			if (od -> common != vl -> m || !isProtected(od) || isAbstract(od))
				continue;

			if (noIntro)
			{
				prcode(fp,
"\n"
"	// There is a public member function for every protected member\n"
"	// function visible from this class.\n"
"\n"
					);

				noIntro = FALSE;
			}

			generateVersionStart(pt,&od -> version,fp);

			prcode(fp,
"	");

			generateResultType(od -> result,fp);

			prcode(fp," sipProtect_%s(",od -> cppname);
			generateArgs(&od -> args,Declaration,fp);
			prcode(fp,");\n"
				);

			generateVersionEnd(pt,&od -> version,fp);
		}
	}
}


/*
 * Generate the definitions of the protected wrapper functions for a class.
 */

static void generateProtectedDefinitions(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	visibleList *vl;

	for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
	{
		overDef *od;

		for (od = vl -> cv -> overs; od != NULL; od = od -> next)
		{
			char *mname = od -> cppname;
			int a;

			if (od -> common != vl -> m || !isProtected(od) || isAbstract(od))
				continue;

			generateVersionStart(pt,&od -> version,fp);

			prcode(fp,
"\n"
				);

			generateResultType(od -> result,fp);

			prcode(fp,
" sip%s::sipProtect_%s(",classVersName(cvd),mname);
			generateArgs(&od -> args,Definition,fp);
			prcode(fp,")\n"
"{\n"
				);

			if (od -> result != NULL)
				prcode(fp,
"	return ");
			else
				prcode(fp,
"	");

			prcode(fp,"%s::%s(",classVersName(vl -> cv),mname);

			for (a = 0; a < od -> args.nrArgs; ++a)
				prcode(fp,"%sa%d",(a == 0 ? "" : ","),a);

			prcode(fp,");\n"
"}\n"
				);

			generateVersionEnd(pt,&od -> version,fp);
		}
	}
}


/*
 * Generate the argument list of a simple call.
 */

static void generateArgCallList(funcArgs *fa,FILE *fp)
{
	int a;

	for (a = 0; a < fa -> nrArgs; ++a)
		prcode(fp,"%sa%d",(a == 0 ? "" : ","),a);
}


/*
 * Generate the virtual handlers for a class version.
 */

static void generateVirtualHandlers(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	virtVersDef *vvd;

	for (vvd = cvd -> vmembers; vvd != NULL; vvd = vvd -> next)
	{
		virtOverDef *vod;
 
		generateVersionStart(pt,&vvd -> version,fp);

		for (vod = vvd -> vo; vod != NULL; vod = vod -> next)
			if (!isPrivate(vod -> o) && vod -> farc == cvd)
				generateVirtualHandler(pt,vod,fp);

		generateVersionEnd(pt,&vvd -> version,fp);
	}
}


/*
 * Generate the function that does most of the work to handle a particular
 * virtual function.
 */

static void generateVirtualHandler(sipSpec *pt,virtOverDef *vod,FILE *fp)
{
	char *cname = classVersName(vod -> farc);
	char *indir = "";
	argDef *ad = vod -> o -> result;
	overDef *od = vod -> o;

	prcode(fp,
"\n"
"// The common handler for all classes that inherit this virtual member\n"
"// function.\n"
"\n"
		);

	generateResultType(ad,fp);

	prcode(fp," sip%s::sipVH_%s(const sipMethodCache *pymc,sipThisType *sipThis,int sipRelLock",cname,od -> cppname);

	if (od -> args.nrArgs > 0)
		prcode(fp,",");

	generateArgs(&od -> args,Definition,fp);
	prcode(fp,")\n"
"{\n"
		);

	if (od -> virtcode != NULL)
	{
		generateCppCodeBlock(NULL,od -> virtcode,cname,fp);
		prcode(fp,
"}\n"
			);

		return;
	}

	if (ad != NULL)
	{
		prcode(fp,
"	");

		generateValueType(ad,fp);

		prcode(fp,"res;\n"
			);
	}

	prcode(fp,
"	PyObject *resobj;\n"
		);

	generateTupleBuilder(pt,&od -> args,TRUE,fp);

	/* Generate the call the the Python member function. */

	prcode(fp,
"\n"
"	if (sipArgs == NULL)\n"
"		goto reportError;\n"
"\n"
"	resobj = sipEvalMethod(&pymc -> pyMethod,sipArgs);\n"
"\n"
"	Py_DECREF(sipArgs);\n"
"\n"
"	if (resobj != NULL)\n"
"	{\n"
		);

	/* Handle void functions. */

	if (ad == NULL)
		prcode(fp,
"		Py_DECREF(resobj);\n"
"\n"
"		if (resobj == Py_None)\n"
"			goto releaseLock;\n"
			);
	else if (ad -> atype == class_type)
	{
		char *rcname = classVersName(ad -> u.cvd);

		if (ad -> nrderefs == 0)
			indir = "*";

		prcode(fp,
"		int iserr = 0;\n"
"\n"
			);

		/* Possible memory leak - who deletes res? */

		prcode(fp,
"		res = sipForceConvertTo_%s(resobj,&iserr);\n"
			,rcname);

		if (!isMapped(ad -> u.cvd))
			prcode(fp,
"		sipTransferSelf(resobj,1);\n"
				);

		prcode(fp,
"\n"
"		Py_DECREF(resobj);\n"
"\n"
"		if (!iserr)\n"
"			goto releaseLock;\n"
			);
	}
	else
	{
		/* Convert and check the result. */

		switch (ad -> atype)
		{
		case ustring_type:
			if (ad -> nrderefs == 0)
				prcode(fp,
"		unsigned char *resstr = (unsigned char *)PyString_AsString(resobj);\n"
					);
			else
				prcode(fp,
"		if (resobj == Py_None)\n"
"			res = NULL;\n"
"		else\n"
"			res = (unsigned char *)PyString_AsString(resobj);\n"
					);
			break;

		case string_type:
			if (ad -> nrderefs == 0)
				prcode(fp,
"		char *resstr = PyString_AsString(resobj);\n"
					);
			else
				prcode(fp,
"		if (resobj == Py_None)\n"
"			res = NULL;\n"
"		else\n"
"			res = PyString_AsString(resobj);\n"
					);
			break;

		case ushort_type:
			prcode(fp,
"		res = (unsigned short)PyInt_AsLong(resobj);\n"
				);
			break;

		case short_type:
			prcode(fp,
"		res = (short)PyInt_AsLong(resobj);\n"
				);
			break;

		case uint_type:
			prcode(fp,
"		res = (unsigned)PyInt_AsLong(resobj);\n"
				);
			break;

		case int_type:
			prcode(fp,
"		res = (int)PyInt_AsLong(resobj);\n"
				);
			break;

		case ulong_type:
			prcode(fp,
"		res = (unsigned long)PyInt_AsLong(resobj);\n"
				);
			break;

		case long_type:
			prcode(fp,
"		res = PyInt_AsLong(resobj);\n"
				);
			break;

		case struct_type:
			prcode(fp,
"		res = (struct %s *)sipConvertToVoidPtr(resobj);\n"
				,ad -> u.sname);
			break;

		case voidptr_type:
			prcode(fp,
"		res = sipConvertToVoidPtr(resobj);\n"
				);
			break;

		case bool_type:
			prcode(fp,
"		res = (bool)PyInt_AsLong(resobj);\n"
				);
			break;

		case float_type:
			prcode(fp,
"		res = (float)PyFloat_AsDouble(resobj);\n"
				);
			break;

		case double_type:
			prcode(fp,
"		res = PyFloat_AsDouble(resobj);\n"
				);
			break;

		case enum_type:
			prcode(fp,
"		res = (%E)PyInt_AsLong(resobj);\n"
				,ad -> u.ed);
			break;
		}

		prcode(fp,
"\n"
"		Py_DECREF(resobj);\n"
"\n"
"		if (PyErr_Occurred() == NULL)\n"
"		{\n"
			);

		if ((ad -> atype == ustring_type || ad -> atype == string_type) && ad -> nrderefs == 0)
			prcode(fp,
"			res = *resstr;\n"
				);

		prcode(fp,
"			goto releaseLock;\n"
"		}\n"
			);
	}

	prcode(fp,
"\n"
"		sipBadVirtualResultType(%N,%N);\n"
"	}\n"
"\n"
"reportError:\n"
"	PyErr_Print();\n"
"\n"
"releaseLock:\n"
"	sipCondReleaseLock(sipRelLock);\n"
		,vod -> farc -> cd -> name,vod -> o -> common -> pyname);

	if (ad != NULL)
		prcode(fp,
"\n"
"	return %sres;\n"
			,indir);

	prcode(fp,
"}\n"
		);
}


/*
 * Generate the code to build a tuple of Python arguments.
 */

static void generateTupleBuilder(sipSpec *pt,funcArgs *fa,int isMethod,FILE *fp)
{
	int a, arraylenarg;

	/* Generate the variables. */

	prcode(fp,
"	PyObject *sipArgs;\n"
		);

	for (a = 0; a < fa -> nrArgs; ++a)
		if (fa -> args[a].atype == class_type || fa -> args[a].atype == rxcon_type || fa -> args[a].atype == rxdis_type)
			prcode(fp,
"	PyObject *a%dobj;\n"
				,a);

	/* Convert each class to a Python object. */

	for (a = 0; a < fa -> nrArgs; ++a)
		if (fa -> args[a].atype == class_type)
		{
			classVersDef *cvd = fa -> args[a].u.cvd;
			char *acname = classVersName(cvd), *ptr = "";

			if (isReference(&fa -> args[a]) || fa -> args[a].nrderefs == 0)
				ptr = "&";

			if (isMapped(cvd))
			{
				checkForConvertFromCode(pt,cvd);

				prcode(fp,
"\n"
"	a%dobj = sipConvertFrom_%s(%sa%d);\n"
					,a,acname,ptr,a);
			}
			else
			{
				prcode(fp,
"\n"
"	a%dobj = sipMapCppToSelf(%sa%d,",a,ptr,a);

				if (cvd -> convtosubcode == NULL)
					prcode(fp,"sipClass_%s",acname);
				else
					prcode(fp,"sipSubClass_%s(%sa%d)",acname,ptr,a);

				prcode(fp,");\n"
					);
			}
		}
		else if (fa -> args[a].atype == rxcon_type || fa -> args[a].atype == rxdis_type)
			prcode(fp,
"\n"
"	a%dobj = sipMapCppToSelf(a%d,sipClass_QObject);\n",
				a,a);

	prcode(fp,
"\n"
"	sipArgs = Py_BuildValue(\"(");

	if (isMethod)
		prcode(fp,"O");

	for (a = 0; a < fa -> nrArgs; ++a)
	{
		char *fmt;

		switch (fa -> args[a].atype)
		{
		case ustring_type:
		case string_type:
			if (fa -> args[a].nrderefs == 0)
				fmt = "c";
			else
				fmt = "s";

			break;

		case enum_type:
		case bool_type:
		case uint_type:
		case int_type:
			fmt = "i";
			break;

		case ushort_type:
		case short_type:
			fmt = "h";
			break;

		case ulong_type:
		case long_type:
			fmt = "l";
			break;

		case struct_type:
		case voidptr_type:
			fmt = "v";
			break;

		case float_type:
			fmt = "f";
			break;

		case double_type:
			fmt = "d";
			break;

		case slotcon_type:
		case slotdis_type:
			fmt = "s";
			break;

		case rxcon_type:
		case rxdis_type:
		case class_type:
			fmt = "O";
			break;

		case array_type:
		case uarray_type:
			fmt = "s#";
			break;

		case arraysize_type:
		case arrayusize_type:
			fmt = "";
			arraylenarg = a;
			break;
		}

		prcode(fp,fmt);
	}

	prcode(fp,")\"");

	if (isMethod)
		prcode(fp,",sipThis -> sipSelf");

	for (a = 0; a < fa -> nrArgs; ++a)
	{
		int atype = fa -> args[a].atype;

		if (atype != arraysize_type && atype != arrayusize_type)
			prcode(fp,",a%d",a);

		if (atype == array_type || atype == uarray_type)
			prcode(fp,",a%d",arraylenarg);

		if (atype == class_type || atype == rxcon_type || atype == rxdis_type)
			prcode(fp,"obj");
	}

	prcode(fp,");\n");

	/* Free the class references. */

	for (a = 0; a < fa -> nrArgs; ++a)
		if (fa -> args[a].atype == class_type || fa -> args[a].atype == rxcon_type || fa -> args[a].atype == rxdis_type)
			prcode(fp,
"\n"
"	Py_XDECREF(a%dobj);\n"
				,a);
}


/*
 * Generate the class interface #include directives required by either a class
 * or a module.
 */

static void generateUsedIncludes(classList *cl,FILE *fp)
{
	prcode(fp,
"\n"
		);

	while (cl != NULL)
	{
		prcode(fp,
"#include \"sip%s%s.h\"\n"
			,cl -> c -> module -> name -> text,cl -> c -> name -> text);

		cl = cl -> next;
	}

	prcode(fp,
"\n"
		);
}


/*
 * Generate the header file for the C++ wrapper class.
 */

static void generateClassHeader(sipSpec *pt,classDef *cd,char *codeDir)
{
	char *impexp, *wfile, *cname = cd -> name -> text;
	char *cmname = cd -> module -> name -> module -> name -> text;
	classVersDef *cvd;
	FILE *fp;

	/* See if we are importing or exporting. */

	impexp = (cd -> module == pt -> module) ? "EXPORT" : "IMPORT";

	/* Create the header file. */

	wfile = concat(codeDir,"/sip",cmname,cname,".h",NULL);
	fp = createFile(pt,wfile,"//","C++ wrapper class header file.");

	prcode(fp,
"\n"
"#ifndef _%s%s_H\n"
"#define	_%s%s_H\n"
		,cmname,cname,cmname,cname);

	generateUsedIncludes(cd -> used,fp);

	for (cvd = cd -> cvds; cvd != NULL; cvd = cvd -> next)
	{
		generateVersionStart(pt,&cvd -> version,fp);

		generateCppCodeBlock(NULL,cvd -> hdrcode,cname,fp);

		if (isOpaque(cvd))
			prcode(fp,
"\n"
"class %s;\n"
				,cname);

		if (!isMapped(cvd))
		{
			prcode(fp,
"\n"
"extern SIP_%s PyObject *sipClass_%s;\n"
"extern SIP_%s PyMethodDef sipClassAttrTab_%s[];\n"
				,impexp,cname
				,impexp,cname);

			if (cvd -> vars != NULL)
				prcode(fp,
"extern SIP_%s PyMethodDef sipClassVarTab_%s[];\n"
"extern SIP_%s PyMethodDef *sipClassVarHierTab_%s[];\n"
					,impexp,cname,impexp,cname);

			/* The cast and constructor functions. */

			prcode(fp,
"\n"
"extern SIP_%s const void *sipCast_%s Py_PROTO((const void *,PyObject *));\n"
"extern SIP_%s PyObject *sipNew_%s Py_PROTO((PyObject *,PyObject *));\n"
					,impexp,cname,impexp,cname);

			/* The sub-class convertor. */

			if (cvd -> convtosubcode != NULL)
				prcode(fp,
"extern SIP_%s PyObject *sipSubClass_%s Py_PROTO((const %s *));\n"
					,impexp,cname,cname);

			if (isComplex(cvd))
				if (cannotCreate(cvd))
					generateLimitedWrapperClassDeclaration(pt,cvd,fp);
				else
					generateWrapperClassDeclaration(pt,cvd,fp);
		}

		prcode(fp,
"\n"
"extern SIP_%s int sipCanConvertTo_%s Py_PROTO((PyObject *));\n"
"extern SIP_%s %s sipConvertTo_%s Py_PROTO((PyObject *,%s **,int,int *));\n"
"extern SIP_%s %s *sipForceConvertTo_%s Py_PROTO((PyObject *,int *));\n"
			,impexp,cname
			,impexp,(cvd -> convtocode != NULL ? "int" : "void"),cname,cname
			,impexp,cname,cname);

		if (cvd -> convfromcode != NULL)
			prcode(fp,
"extern SIP_%s PyObject *sipConvertFrom_%s Py_PROTO((const %s *));\n"
				,impexp,cname,cname);

		generateVersionEnd(pt,&cvd -> version,fp);
	}

	prcode(fp,
"\n"
"#endif\n"
		);

	closeFile(fp);
	free(wfile);
}


/*
 * Generate the declaration of a virtual handler.
 */

static void generateVirtualHandlerDeclaration(overDef *od,FILE *fp)
{
	prcode(fp,
"	static ");

	generateResultType(od -> result,fp);

	prcode(fp," sipVH_%s(const sipMethodCache *,sipThisType *,int",od -> cppname);

	if (od -> args.nrArgs > 0)
		prcode(fp,",");

	generateArgs(&od -> args,Declaration,fp);

	prcode(fp,");\n"
		);
}


/*
 * Generate the limited wrapper class declaration that might be needed by a
 * complex class that cannot be created in Python.
 */

static void generateLimitedWrapperClassDeclaration(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	int noIntro;
	virtVersDef *vvd;

	noIntro = TRUE;

	for (vvd = cvd -> vmembers; vvd != NULL; vvd = vvd -> next)
	{
		int noVers;
		virtOverDef *vod;

		noVers = TRUE;

		for (vod = vvd -> vo; vod != NULL; vod = vod -> next)
			if (!isPrivate(vod -> o) && vod -> farc == cvd)
			{
				if (noIntro)
				{
					noIntro = FALSE;

					prcode(fp,
"\n"
"\n"
"class SIP_%s sip%s\n"
"{\n"
"public:\n"
						,(cvd -> cd -> module == pt -> module) ? "EXPORT" : "IMPORT"
						,classVersName(cvd));
				}

				if (noVers)
				{
					noVers = FALSE;

					generateVersionStart(pt,&vvd -> version,fp);
				}

				generateVirtualHandlerDeclaration(vod -> o,fp);
			}

		if (!noVers)
			generateVersionEnd(pt,&vvd -> version,fp);
	}

	if (!noIntro)
		prcode(fp,
"};\n"
			);
}


/*
 * Generate the wrapper class declaration.
 */

static void generateWrapperClassDeclaration(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	char *cname = classVersName(cvd);
	int noIntro, nrVirts;
	ctorDef *ct;
	visibleList *vl;
	virtVersDef *vvd;

	nrVirts = countVirtuals(cvd);

	prcode(fp,
"\n"
"\n"
"class SIP_%s sip%s : public %s\n"
"{\n"
"public:\n"
		,(cvd -> cd -> module == pt -> module) ? "EXPORT" : "IMPORT"
		,cname,cname);

	/* The constructor declarations. */

	for (ct = cvd -> ctors; ct != NULL; ct = ct -> next)
	{
		if (isPrivateCtor(ct))
			continue;

		generateVersionStart(pt,&ct -> version,fp);

		prcode(fp,
"	sip%s(",cname);

		generateArgs(&ct -> args,Declaration,fp);

		prcode(fp,");\n"
			);

		generateVersionEnd(pt,&ct -> version,fp);
	}

	/* The destructor. */

	prcode(fp,
"	~sip%s();\n"
		,cname);

	/* The exposure of protected enums. */

	generateProtectedEnums(pt,cvd,fp);

	/* The catcher around each virtual function in the hierachy. */

	for (vvd = cvd -> vmembers; vvd != NULL; vvd = vvd -> next)
	{
		virtOverDef *vod;
 
		generateVersionStart(pt,&vvd -> version,fp);
 
		for (vod = vvd -> vo; vod != NULL; vod = vod -> next)
		{
			overDef *od = vod -> o;

			if (isPrivate(od))
				continue;
 
			prcode(fp,
"	");
 
			generateResultType(od -> result,fp);
 
			prcode(fp," %s(",od -> cppname);
			generateArgs(&od -> args,Declaration,fp);
 
			prcode(fp,")%s;\n"
				,(isConst(od) ? " const" : ""));

			if (vod -> farc == cvd)
				generateVirtualHandlerDeclaration(od,fp);
		}

		generateVersionEnd(pt,&vvd -> version,fp);
	}

	/* The wrapper around each protected member function. */

	generateProtectedDeclarations(pt,cvd,fp);

	/* The public wrapper around each signal emitter. */

	noIntro = TRUE;

	for (vl = cvd -> visible; vl != NULL; vl = vl -> next)
	{
		overDef *od;

		for (od = vl -> cv -> overs; od != NULL; od = od -> next)
		{
			if (od -> common != vl -> m || !isSignal(od))
				continue;

			if (noIntro)
			{
				prcode(fp,
"\n"
"	// There is a public member function for every Qt signal that can be\n"
"	// emitted by this object.  This function is called by Python to emit\n"
"	// the signal.\n"
"\n"
					);

				noIntro = FALSE;
			}

			prcode(fp,
"	int sipEmit_%s(PyObject *sipArgs);\n"
				,vl -> m -> pyname -> text);

			break;
		}
	}

	/* The private declarations. */

	prcode(fp,
"\n"
"	sipThisType *sipPyThis;\n"
		);

	prcode(fp,
"\n"
"private:\n"
"	sip%s(const sip%s &);\n"
"	sip%s &operator = (const sip%s &);\n"
		,cname,cname,cname,cname);

	if (nrVirts > 0)
		prcode(fp,
"\n"
"	sipMethodCache sipPyMethods[%d];\n"
			,nrVirts);

	prcode(fp,
"};\n"
		);
}


/*
 * Generate the return type of a member function.
 */

static void generateResultType(argDef *ad,FILE *fp)
{
	if (ad == NULL)
		prcode(fp,"void");
	else
		generateSingleArg(ad,-1,Declaration,fp);
}


/*
 * Generate typed arguments.
 */

static void generateArgs(funcArgs *args,funcArgType ftype,FILE *fp)
{
	char *prefix = NULL;
	int a;

	for (a = 0; a < args -> nrArgs; ++a)
	{
		if (prefix == NULL)
			prefix = ",";
		else
			prcode(fp,prefix);

		generateSingleArg(&args -> args[a],a,ftype,fp);
	}
}


/*
 * Generate the basic type of a variable or cast.
 */

static void generateBasicType(argDef *ad,FILE *fp)
{
	if (isConstArg(ad))
		prcode(fp,"const ");

	switch (ad -> atype)
	{
	case ustring_type:
		prcode(fp,"unsigned char");
		break;

	case string_type:
		prcode(fp,"char");
		break;

	case ushort_type:
		prcode(fp,"unsigned short");
		break;

	case short_type:
		prcode(fp,"short");
		break;

	case uint_type:
		prcode(fp,"unsigned");
		break;

	case int_type:
		prcode(fp,"int");
		break;

	case ulong_type:
		prcode(fp,"unsigned long");
		break;

	case long_type:
		prcode(fp,"long");
		break;

	case struct_type:
		prcode(fp,"struct %s",ad -> u.sname);
		break;

	case voidptr_type:
		prcode(fp,"void");
		break;

	case bool_type:
		prcode(fp,"bool");
		break;

	case float_type:
		prcode(fp,"float");
		break;

	case double_type:
		prcode(fp,"double");
		break;

	case enum_type:
		prcode(fp,"%E",ad -> u.ed);
		break;

	case class_type:
		prcode(fp,"%s",classVersName(ad -> u.cvd));
		break;
	}

	if (ad -> nrderefs > 0)
	{
		int i;

		prcode(fp," ");

		for (i = 0; i < ad -> nrderefs; ++i)
			prcode(fp,"*");
	}
}


/*
 * Generate the declaration of a variable (excluding the name) to hold a result
 * from a C++ function call.
 */

static void generateValueType(argDef *ad,FILE *fp)
{
	generateBasicType(ad,fp);

	if (ad -> nrderefs == 0)
	{
		prcode(fp," ");

		if (ad -> atype == class_type)
			prcode(fp,"*");
	}
}


/*
 * Generate a single argument.
 */

static void generateSingleArg(argDef *ad,int argnr,funcArgType ftype,FILE *fp)
{
	int genType, genName, derefPtr, toType;

	/* Break the type down to individual modifications. */

	genType = FALSE;
	genName = FALSE;
	derefPtr = FALSE;
	toType = FALSE;

	switch (ftype)
	{
	case Call:
		genName = TRUE;
		derefPtr = TRUE;
		toType = TRUE;
		break;

	case Declaration:
		genType = TRUE;
		break;

	case Definition:
		genType = TRUE;
		genName = TRUE;
		break;
	}

	if (genType)
		generateType(ad,fp);

	if (genName)
	{
		if (derefPtr && (isReference(ad) || (ad -> atype == class_type && ad -> nrderefs == 0)))
			prcode(fp,"*");

		prcode(fp," ");

		if (toType)
			if (ad -> atype == bool_type)
				prcode(fp,"(bool)");
			else if (ad -> atype == enum_type)
				prcode(fp,"(%E)",ad -> u.ed);

		prcode(fp,"a%d",argnr);
	}
}


/*
 * Generate a C++ type.
 */

static void generateType(argDef *ad,FILE *fp)
{
	int nr_derefs;

	nr_derefs = ad -> nrderefs;

	if (isConstArg(ad))
		prcode(fp,"const ");

	switch (ad -> atype)
	{
	case uarray_type:
		nr_derefs = 1;

		/* Drop through. */

	case ustring_type:
		prcode(fp,"unsigned char");
		break;

	case slotcon_type:
	case slotdis_type:
	case array_type:
		nr_derefs = 1;

		/* Drop through. */

	case string_type:
		prcode(fp,"char");
		break;

	case ushort_type:
		prcode(fp,"unsigned short");
		break;

	case short_type:
		prcode(fp,"short");
		break;

	case arrayusize_type:
	case uint_type:
		prcode(fp,"unsigned");
		break;

	case arraysize_type:
	case int_type:
		prcode(fp,"int");
		break;

	case ulong_type:
		prcode(fp,"unsigned long");
		break;

	case long_type:
		prcode(fp,"long");
		break;

	case struct_type:
		prcode(fp,"struct %s",ad -> u.sname);
		break;

	case voidptr_type:
		prcode(fp,"void");
		break;

	case bool_type:
		prcode(fp,"bool");
		break;

	case float_type:
		prcode(fp,"float");
		break;

	case double_type:
		prcode(fp,"double");
		break;

	case defined_type:
		/*
		 * The only defined types still remaining are arguments to
		 * templates.
		 */

		prcode(fp,"%S",ad -> u.snd);
		break;

	case enum_type:
		prcode(fp,"%E",ad -> u.ed);
		break;

	case rxcon_type:
	case rxdis_type:
		nr_derefs = 1;
		prcode(fp,"QObject");
		break;

	case class_type:
		prcode(fp,"%s",classVersName(ad -> u.cvd));
		break;

	case template_type:
		prcode(fp,"%s<",ad -> u.td -> name);
		generateType(&ad -> u.td -> type,fp);
		prcode(fp,">");
		break;
	}

	if (nr_derefs > 0)
	{
		int i;

		prcode(fp," ");

		for (i = 0; i < nr_derefs; ++i)
			prcode(fp,"*");
	}

	if (isReference(ad))
		prcode(fp,"&");
}


/*
 * Generate the definition of an argument variable and any supporting
 * variables.
 */

static void generateVariable(argDef *ad,int argnr,int secCall,FILE *fp)
{
	char *str0, *str;
	int nr_derefs = ad -> nrderefs;
	argType atype = ad -> atype;

	str0 = "";

	switch (atype)
	{
	case uarray_type:
		nr_derefs = 1;

		/* Drop through. */

	case ustring_type:
		str = "unsigned char";
		break;

	case slotcon_type:
	case slotdis_type:
	case array_type:
		nr_derefs = 1;

		/* Drop through. */

	case string_type:
		str = "char";
		break;

	case ushort_type:
		str = "unsigned short";
		break;

	case short_type:
		str = "short";
		break;

	case arrayusize_type:
	case uint_type:
		str = "unsigned";
		break;

	case arraysize_type:
	case int_type:
		str = "int";
		break;

	case ulong_type:
		str = "unsigned long";
		break;

	case bool_type:
	case enum_type:
	case long_type:
		str = "long";
		break;

	case struct_type:
		str0 = "struct ";
		str = ad -> u.sname;
		break;

	case voidptr_type:
		str = "void";
		break;

	case float_type:
		str = "float";
		break;

	case double_type:
		str = "double";
		break;

	case rxcon_type:
	case rxdis_type:
		nr_derefs = 1;
		str = "QObject";
		break;

	case class_type:
		nr_derefs = 1;
		str = classVersName(ad -> u.cvd);

		if (ad -> defval != NULL && ad -> defval -> vtype == fcall_value)
		{
			int i;

			/* Generate an instance for the default value. */

			prcode(fp,
"		%s a%ddef = %S(",str,argnr,ad -> defval -> u.fcd -> name);

			for (i = 0; i < ad -> defval -> u.fcd -> nrArgs; ++i)
			{
				if (i > 0)
					prcode(fp,",");

				generateExpression(ad -> defval -> u.fcd -> args[i],fp);
			}

			prcode(fp,");\n"
				);
		}

		break;
	}

	prcode(fp,
"		%s%s%s ",(isConstArg(ad) ? "const " : ""),str0,str);

	while (nr_derefs--)
		prcode(fp,"*");

	prcode(fp,"a%d",argnr);

	/* Handle any default value. */

	if (ad -> defval != NULL)
	{
		prcode(fp," = ");

		if (ad -> atype == class_type)
		{
			if (ad -> defval -> vtype == fcall_value)
				prcode(fp,"&a%ddef",argnr);
			else if (ad -> defval -> vtype == scoped_value)
			{
				/* Cast if needed. */

				if (!isConstArg(ad))
					prcode(fp,"(%s *)",classVersName(ad -> u.cvd));

				prcode(fp,"&%S",ad -> defval -> u.vscp);
			}
			else
				prcode(fp,"NULL");
		}
		else
		{
			if (ad -> defval -> vtype == fcall_value)
				fatal("Non-class argument has a constructor as it's default value\n");

			generateExpression(ad -> defval,fp);
		}
	}

	prcode(fp,";\n"
		);

	/* Some types have supporting variables. */

	switch (atype)
	{
	case rxcon_type:
	case rxdis_type:
		prcode(fp,
"		PyObject *a%dobj;\n"
			,argnr);
		break;

	case class_type:
		prcode(fp,
"		PyObject *a%dobj%s;\n"
			,argnr,(ad -> defval == NULL ? "" : " = NULL"));
		break;
	}
}


/*
 * Generate the Python constructor function for the class.
 */

static void generatePythonConstructor(sipSpec *pt,classVersDef *cvd,FILE *fp)
{
	char *cname = classVersName(cvd);
	char *mname = pt -> module -> name -> text;
	ctorDef *ct;

	prcode(fp,
"\n"
"PyObject *sipNew_%s(PyObject *sipSelf,PyObject *sipArgs)\n"
"{\n"
"	static sipExtraType et = {\n"
"		sipCast_%s,NULL,NULL",cname,cname);

	if (pt -> sigslots)
	{
		prcode(fp,",\n"
"		(void *)sipNewProxy_%s,\n"
			,mname);

		if (hasSigSlots(cvd))
			prcode(fp,
"		sipSignals_%s"
				,cname);
		else
			prcode(fp,
"		NULL");
	}

	prcode(fp,"\n"
"	};\n"
"\n"
"	sipThisType *sipThis = NULL;\n"
"	const void *sipNew = NULL;\n"
"	int sipFlags = SIP_PY_OWNED;\n"
"	int sipArgsParsed = 0;\n"
"\n"
"	// See if there is something pending.\n"
"\n"
"	sipNew = sipGetPending(&sipFlags);\n"
		);

	/*
	 * Generate the code that parses the Python arguments and calls the
	 * correct constructor.
	 */

	if (!cannotCreate(cvd))
		for (ct = cvd -> ctors; ct != NULL; ct = ct -> next)
		{
			if (isPrivateCtor(ct))
				continue;

			generateVersionStart(pt,&ct -> version,fp);

			prcode(fp,
"\n"
"	if (sipNew == NULL)\n"
"	{\n"
				);

			/* Use any code supplied in the specification. */

			if (ct -> cppcode != NULL)
				generateCppCodeBlock(NULL,ct -> cppcode,cname,fp);
			else
			{
				int needSecCall;

				needSecCall = generateArgParser(&ct -> args,TRUE,FALSE,fp);
				generateConstructorCall(pt,cvd,ct,FALSE,fp);

				if (needSecCall)
				{
					prcode(fp,
"	}\n"
"\n"
"	if (sipNew == NULL)\n"
"	{\n"
						);

					generateArgParser(&ct -> args,TRUE,TRUE,fp);
					generateConstructorCall(pt,cvd,ct,TRUE,fp);
				}
			}

			prcode(fp,
"	}\n"
				);

			generateVersionEnd(pt,&ct -> version,fp);
		}

	prcode(fp,
"\n"
"	if (sipNew == NULL)\n"
"	{\n"
"		sipNoCtor(sipArgsParsed,%N);\n"
"		return NULL;\n"
"	}\n"
"\n"
"	// Wrap the object.\n"
"\n"
"	if ((sipThis = sipCreateThis(sipSelf,sipNew,&sipType_%s,sipFlags,&et)) == NULL)\n"
"	{\n"
		,cvd -> cd -> name
		,cname);

	if (!cannotCreate(cvd) && (isComplex(cvd) || !noPublicDtor(cvd)))
	{
		prcode(fp,
"		if (sipFlags & SIP_PY_OWNED)\n"
			);

		if (isComplex(cvd))
		{
			if (noPublicDtor(cvd))
				prcode(fp,
"			if (!(sipFlags & SIP_SIMPLE))\n"
"				delete (sip%s *)sipNew;\n"
					,cname);
			else
				prcode(fp,
"			if (sipFlags & SIP_SIMPLE)\n"
"				delete (%s *)sipNew;\n"
"			else\n"
"				delete (sip%s *)sipNew;\n"
					,cname,cname);
		}
		else
			prcode(fp,
"			delete (%s *)sipNew;\n"
				,cname);
	}

	prcode(fp,
"\n"
"		return NULL;\n"
"	}\n"
		);

	if (isComplex(cvd) && !cannotCreate(cvd))
		prcode(fp,
"\n"
"	if (!(sipFlags & SIP_SIMPLE))\n"
"		((sip%s *)sipNew) -> sipPyThis = sipThis;\n"
			,cname);

	prcode(fp,
"\n"
"	Py_INCREF(Py_None);\n"
"	return Py_None;\n"
"}\n"
		);
}


/*
 * Count the number of virtual members in a class.
 */
 
static int countVirtuals(classVersDef *cvd)
{
	int nrvirts;
	virtVersDef *vvd;
 
	nrvirts = 0;
 
	for (vvd = cvd -> vmembers; vvd != NULL; vvd = vvd -> next)
	{
		int thisVers = 0;

		virtOverDef *vod;

		/* Only count members than have a non-private overload. */

		for (vod = vvd -> vo; vod != NULL; vod = vod -> next)
			if (!isPrivate(vod -> o))
				++thisVers;

		if (thisVers > nrvirts)
			nrvirts = thisVers;
	}
 
	return nrvirts;
}
 

/*
 * Generate a single constructor call.
 */

static void generateConstructorCall(sipSpec *pt,classVersDef *cvd,ctorDef *ct,int secCall,FILE *fp)
{
	prcode(fp,
"		{\n"
		);

	generateArgConvertors(pt,&ct -> args,secCall,FALSE,NULL,fp);

	prcode(fp,
"			sipNew = new %s%s(",(isComplex(cvd) ? "sip" : ""),classVersName(cvd));

	generateArgs(&ct -> args,Call,fp);

	prcode(fp,");\n"
		);

	generateCallTidyUp(&ct -> args,fp);

	prcode(fp,
"		}\n"
		);
}


/*
 * See if a member overload should be skipped.
 */

static int skipOverload(overDef *od,memberDef *md,classVersDef *cvd,classVersDef *ccvd,int doStatics)
{
	/* Skip if it's a signal or abstract. */

	if (od -> common != md || isSignal(od) || isAbstract(od))
		return TRUE;

	/* Skip if its not in the current class unless it is protected. */

	if (!isProtected(od) && ccvd != cvd)
		return TRUE;

	/* Skip if it is static and we aren't doing statics, or vice versa. */

	if ((doStatics && !isStatic(od)) || (!doStatics && isStatic(od)))
		return TRUE;

	return FALSE;
}


/*
 * Generate a class member function.
 */

static void generateFunction(sipSpec *pt,memberDef *md,overDef *overs,classVersDef *cvd,classVersDef *ocvd,int doStatics,FILE *fp)
{
	int isOver;
	overDef *od;
	versionOrList *vol;

	/*
	 * Go through the overloads and build and or'ed list of versions for
	 * those that need to be handled.
	 */

	isOver = FALSE;

	for (od = overs; od != NULL; od = od -> next)
	{
		if (skipOverload(od,md,cvd,ocvd,doStatics))
			continue;

		if (!isOver)
		{
			vol = orVersionQual(pt,NULL,&od -> version);
			isOver = TRUE;
		}
		else if (vol != NULL)
			vol = orVersionQual(pt,vol,&od -> version);
	}

	if (isOver)
	{
		char *cname = classVersName(cvd);
		char *pname = md -> pyname -> text;

		reduceOredVersion(pt,&vol);

		generateOredVersionStart(pt,vol,fp);

		/* See if the member function is static. */

		if (doStatics)
			prcode(fp,
"\n"
"static PyObject *sipDo_%s_%s(PyObject *,PyObject *sipArgs)\n"
"{\n"
"	int sipArgsParsed = 0;\n"
				,cname,pname);
		else
			prcode(fp,
"\n"
"static PyObject *sipDo_%s_%s(PyObject *sipThisObj,PyObject *sipArgs)\n"
"{\n"
"	sipThisType *sipThis;\n"
"	int sipArgsParsed = 0;\n"
"\n"
"	if ((sipThis = sipGetThis(sipThisObj,&sipArgs,sipClass_%s)) == NULL)\n"
"		return NULL;\n"
				,cname,pname,cname);

		for (od = overs; od != NULL; od = od -> next)
		{
			if (skipOverload(od,md,cvd,ocvd,doStatics))
				continue;

			generateFunctionBody(pt,od,cvd,fp);
		}

		prcode(fp,
"\n"
"	// Report an error if the arguments couldn't be parsed.\n"
"\n"
"	sipNoMethod(sipArgsParsed,%N,%N);\n"
"\n"
"	return NULL;\n"
"}\n"
			,cvd -> cd -> name,md -> pyname);

		generateOredVersionEnd(pt,vol,fp);
		freeVersionOrList(vol);
	}
}


/*
 * Free an or'ed version list - but not the versions themselves.
 */

static void freeVersionOrList(versionOrList *vol)
{
	while (vol != NULL)
	{
		versionOrList *next;

		next = vol -> next;
		free(vol);
		vol = next;
	}
}


/*
 * Generate the function calls for a particular overload.
 */

static void generateFunctionBody(sipSpec *pt,overDef *od,classVersDef *cvd,FILE *fp)
{
	char *cname;
	int noDecr;

	if (cvd == NULL)
	{
		cname = NULL;
		noDecr = TRUE;
	}
	else
	{
		cname = classVersName(cvd);
		noDecr = isStatic(od);
	}

	generateVersionStart(pt,&od -> version,fp);

	prcode(fp,
"\n"
"	{\n"
		);

	/* Use any code supplied in the specification. */

	if (od -> cppcode != NULL)
		generateCppCodeBlock(NULL,od -> cppcode,cname,fp);
	else if (!isPrivate(od))
	{
		int needSecCall;

		needSecCall = generateArgParser(&od -> args,noDecr,FALSE,fp);
		generateFunctionCall(pt,cvd,od,FALSE,fp);

		if (needSecCall)
		{
			prcode(fp,
"	}\n"
"\n"
"	{\n"
				);

			generateArgParser(&od -> args,noDecr,TRUE,fp);
			generateFunctionCall(pt,cvd,od,TRUE,fp);
		}
	}

	prcode(fp,
"	}\n"
		);

	generateVersionEnd(pt,&od -> version,fp);
}


/*
 * Generate the code to handle the result of a call to a member function.
 */

static void generateHandleResult(sipSpec *pt,overDef *od,int isNew,FILE *fp)
{
	if (od -> result == NULL)
	{
		prcode(fp,
"			Py_INCREF(Py_None);\n"
"			return Py_None;\n"
			);

		return;
	}

	switch (od -> result -> atype)
	{
	case class_type:
		{
			classVersDef *cvd = od -> result -> u.cvd;
			char *rcname = classVersName(cvd);

			if (isMapped(cvd))
			{
				checkForConvertFromCode(pt,cvd);

				prcode(fp,
"			PyObject *resobj = sipConvertFrom_%s(res);\n"
					,rcname);

				if (isNew)
					prcode(fp,
"			delete res;\n"
						);

				prcode(fp,
"\n"
"			return resobj;\n"
					);
			}
			else if (isNew)
			{
				prcode(fp,
"			return sipNewCppToSelf(res,");

				if (cvd -> convtosubcode == NULL)
					prcode(fp,"sipClass_%s",rcname);
				else
					prcode(fp,"sipSubClass_%s(res)",rcname);

				prcode(fp,",SIP_SIMPLE | SIP_PY_OWNED);\n"
					);
			}
			else
			{
				prcode(fp,
"			return sipMapCppToSelf(res,");

				if (cvd -> convtosubcode == NULL)
					prcode(fp,"sipClass_%s",rcname);
				else
					prcode(fp,"sipSubClass_%s(res)",rcname);

				prcode(fp,");\n"
					);
			}
		}
	
		break;

	case bool_type:
		prcode(fp,
"			return sipConvertFromBool((int)res);\n"
			);

		break;

	case ustring_type:
	case string_type:
		if (od -> result -> nrderefs == 0)
			prcode(fp,
"			return PyString_FromStringAndSize(%s&res,1);\n"
				,(od -> result -> atype == ustring_type) ? "(char *)" : "");
		else
			prcode(fp,
"			if (res == NULL)\n"
"			{\n"
"				Py_INCREF(Py_None);\n"
"				return Py_None;\n"
"			}\n"
"\n"
"			return PyString_FromString(res);\n"
				);

		break;

	case enum_type:
	case ushort_type:
	case short_type:
	case uint_type:
	case int_type:
	case ulong_type:
		prcode(fp,
"			return PyInt_FromLong((long)res);\n"
			);

		break;

	case long_type:
		prcode(fp,
"			return PyInt_FromLong(res);\n"
			);

		break;

	case struct_type:
	case voidptr_type:
		prcode(fp,
"			return sipConvertFromVoidPtr(res);\n"
			);

		break;

	case float_type:
		prcode(fp,
"			return PyFloat_FromDouble((double)res);\n"
			);

		break;

	case double_type:
		prcode(fp,
"			return PyFloat_FromDouble(res);\n"
			);

		break;
	}
}


/*
 * Generate a function call.
 */

static void generateFunctionCall(sipSpec *pt,classVersDef *cvd,overDef *od,int secCall,FILE *fp)
{
	char *cname, *mname = od -> cppname;
	int inNew;

	prcode(fp,
"		{\n"
		);

	/*
	 * If we can never create this class from Python, then all protected
	 * methods can never be called.
	 */

	if (isProtected(od) && cannotCreate(cvd))
	{
		prcode(fp,
"			sipGetComplexCppPtr(sipThis);\n"
"			return NULL;\n"
"		}\n"
			);

		return;
	}

	if (od -> result != NULL)
	{
		prcode(fp,
"			");
		generateValueType(od -> result,fp);
		prcode(fp,"res;\n"
			);
	}

	if (cvd != NULL)
	{
		cname = classVersName(cvd);

		if (isProtected(od))
			prcode(fp,
"			sip%s *ptr;\n"
"\n"
"			if ((ptr = (sip%s *)sipGetComplexCppPtr(sipThis)) == NULL)\n"
"				return NULL;\n"
				,cname,cname);
		else if (!isStatic(od))
			prcode(fp,
"			%s *ptr;\n"
"\n"
"			if ((ptr = (%s *)sipGetCppPtr(sipThis,sipClass_%s)) == NULL)\n"
"				return NULL;\n"
				,cname,cname,cname);
	}

	if (od -> result != NULL || cvd != NULL)
		prcode(fp,
"\n"
			);

	generateArgConvertors(pt,&od -> args,secCall,isStatic(od),NULL,fp);

	/* Call any pre-hook. */

	if (od -> prehook != NULL)
		prcode(fp,
"			sipCallHook(\"%s\");\n"
			,od -> prehook);

	/* Save the thread if required. */

	if (isBlocking(od))
		prcode(fp,
"			sipReleaseLock();\n"
			);

	prcode(fp,
"			");

	if (od -> result != NULL)
		prcode(fp,"res = ");

	/*
	 * If the result is a class instance, then construct a copy on the
	 * heap.
	 */

	inNew = FALSE;

	if (od -> result != NULL && od -> result -> atype == class_type)
		if (isReference(od -> result))
			prcode(fp,"&");
		else if (od -> result -> nrderefs == 0)
		{
			prcode(fp,"new %s(",classVersName(od -> result -> u.cvd));
			inNew = TRUE;
		}

	/*
	 * If the function is protected then call the public wrapper.
	 * Otherwise explicitly call the real function and not any version in
	 * the wrapper class in case it is virtual.  This will prevent virtual
	 * loops.  Don't need to worry about indirected objects for protected
	 * functions.
	 */

	if (cvd == NULL)
		prcode(fp,"%s(",mname);
	else if (isProtected(od))
		prcode(fp,"ptr -> sipProtect_%s(",mname);
	else if (isStatic(od))
		prcode(fp,"%s::%s(",cname,mname);
	else
		prcode(fp,"ptr -> %s::%s(",cname,mname);

	generateArgs(&od -> args,Call,fp);

	prcode(fp,")");

	if (inNew)
		prcode(fp,")");

	prcode(fp,";\n"
		);

	/* Restore the thread if required. */

	if (isBlocking(od))
		prcode(fp,
"			sipAcquireLock();\n"
			);

	/* Call any post-hook. */

	if (od -> posthook != NULL)
		prcode(fp,
"			sipCallHook(\"%s\");\n"
			,od -> posthook);

	generateCallTidyUp(&od -> args,fp);

	prcode(fp,
"\n"
		);

	generateHandleResult(pt,od,inNew,fp);

	prcode(fp,
"		}\n"
		);
}


/*
 * Generate the code to convert the remaining unconverted arguments.
 */

static void generateArgConvertors(sipSpec *pt,funcArgs *args,int secCall,int isstat,char *retval,FILE *fp)
{
	char *mname = pt -> module -> name -> text, *this;
	int a, rxconarg, rxdisarg, slotconarg, slotdisarg, noIntro;

	this = (isstat ? "NULL" : "sipThis");

	/* Do class ones together. */

	noIntro = TRUE;
	rxconarg = rxdisarg = -1;

	for (a = 0; a < args -> nrArgs; ++a)
	{
		argDef *ad = &args -> args[a];

		switch (ad -> atype)
		{
		case class_type:
			{
				classVersDef *cvd = ad -> u.cvd;
				char *cname = classVersName(cvd);

				if (noIntro)
				{
					prcode(fp,
"			int iserr = 0;\n"
"\n"
						);

					noIntro = FALSE;
				}

				prcode(fp,
"			");

				if (cvd -> convtocode != NULL)
					prcode(fp,"int istemp%d = ",a);

				prcode(fp,"sipConvertTo_%s(a%dobj,",cname,a);

				/* See if it needs a cast. */

				if (isConstArg(ad))
					prcode(fp,"(%s **)",cname);

				prcode(fp,"&a%d,%s,&iserr);\n"
					,a,(ad -> nrderefs == 0 ? "1" : "0"));

				break;
			}

		case rxcon_type:
			rxconarg = a;
			break;

		case rxdis_type:
			rxdisarg = a;
			break;

		case slotcon_type:
			slotconarg = a;
			break;

		case slotdis_type:
			slotdisarg = a;
			break;
		}
	}

	if (rxconarg >= 0 || rxdisarg >= 0)
		if (noIntro)
		{
			prcode(fp,
"			int iserr = 0;\n"
"\n"
				);

			noIntro = FALSE;
		}

	if (secCall)
	{
		if (rxconarg >= 0)
		{
			prcode(fp,
"			a%d = sipConvertRx(sipNewProxy_%s,%s,\"(",rxconarg,mname,this);

			generateArgs(args -> args[slotconarg].u.sa,Declaration,fp);

			prcode(fp,")\",a%dobj,NULL,&a%d,&iserr);\n"
				,rxconarg,slotconarg);
		}

		if (rxdisarg >= 0)
		{
			prcode(fp,
"			a%d = sipGetRx(%s,\"(",rxdisarg,this);

			generateArgs(args -> args[slotdisarg].u.sa,Declaration,fp);

			prcode(fp,")\",a%dobj,NULL,&a%d,&iserr);\n"
				,rxdisarg,slotdisarg);
		}
	}
	else
	{
		if (rxconarg >= 0)
		{
			prcode(fp,
"			a%d = sipConvertRx(sipNewProxy_%s,%s,\"(",rxconarg,mname,this);

			generateArgs(args -> args[slotconarg].u.sa,Declaration,fp);

			prcode(fp,")\",a%dobj,a%d,&a%d,&iserr);\n"
				,rxconarg,slotconarg,slotconarg);
		}

		if (rxdisarg >= 0)
		{
			prcode(fp,
"			a%d = sipGetRx(%s,\"(",rxdisarg,this);

			generateArgs(args -> args[slotdisarg].u.sa,Declaration,fp);

			prcode(fp,")\",a%dobj,a%d,&a%d,&iserr);\n"
				,rxdisarg,slotdisarg,slotdisarg);
		}
	}

	if (!noIntro)
	{
		if (retval == NULL)
			retval = "NULL";

		prcode(fp,
"\n"
"			if (iserr)\n"
"				return %s;\n"
"\n"
			,retval);
	}
}


/*
 * Generate the argument variables for a member function/constructor.
 */

static int generateArgParser(funcArgs *args,int noDecr,int secCall,FILE *fp)
{
	int a, needSecCall, optargs, arraylenarg;

	/* Assume there isn't a slot. */

	needSecCall = FALSE;

	/* Generate the local variables that will hold the parsed arguments. */

	for (a = 0; a < args -> nrArgs; ++a)
	{
		argType atype = args -> args[a].atype;

		if (atype == slotcon_type || atype == slotdis_type ||
			 atype == rxcon_type || atype == rxdis_type)
			needSecCall = TRUE;

		if (atype == arraysize_type || atype == arrayusize_type)
			arraylenarg = a;

		generateVariable(&args -> args[a],a,secCall,fp);
	}

	if (args -> nrArgs != 0)
		prcode(fp,
"\n"
			);

	/* Generate the call to sipParseArgs(). */

	prcode(fp,
"		if (sipParseArgs(&sipArgsParsed,sipArgs,(char *)\"");

	/* Generate the format string. */

	optargs = FALSE;

	if (noDecr)
		prcode(fp,"-");

	for (a = 0; a < args -> nrArgs; ++a)
	{
		char *fmt;

		if (args -> args[a].defval != NULL && !optargs)
		{
			prcode(fp,"|");
			optargs = TRUE;
		}

		switch (args -> args[a].atype)
		{
		case ustring_type:
		case string_type:
			if (args -> args[a].nrderefs == 0)
				fmt = "c";
			else
				fmt = "s";
			break;

		case uint_type:
		case int_type:
			fmt = "i";
			break;

		case ushort_type:
		case short_type:
			fmt = "h";
			break;

		case enum_type:
		case bool_type:
		case ulong_type:
		case long_type:
			fmt = "l";
			break;

		case struct_type:
		case voidptr_type:
			fmt = "v";
			break;

		case float_type:
			fmt = "f";
			break;

		case double_type:
			fmt = "d";
			break;

		case slotcon_type:
		case slotdis_type:
			fmt = (secCall ? "" : "S");
			break;

		case rxcon_type:
		case rxdis_type:
			fmt = (secCall ? "F" : "R");
			break;

		case class_type:
			fmt = "I";
			break;

		case array_type:
		case uarray_type:
			fmt = "a";
			break;

		case arrayusize_type:
		case arraysize_type:
			fmt = "";
			break;
		}

		prcode(fp,fmt);
	}

	prcode(fp,"\"");

	/* Generate the parameters corresponding to the format string. */

	for (a = 0; a < args -> nrArgs; ++a)
	{
		int atype = args -> args[a].atype;

		switch (atype)
		{
		case class_type:
			prcode(fp,",sipCanConvertTo_%s,&a%dobj",classVersName(args -> args[a].u.cvd),a);
			break;

		case rxcon_type:
		case rxdis_type:
			prcode(fp,",&a%dobj",a);
			break;

		case slotcon_type:
		case slotdis_type:
			if (!secCall)
				prcode(fp,",&a%d",a);

			break;

		case arraysize_type:
		case arrayusize_type:
			break;

		default:
			prcode(fp,",&a%d",a);

			if (atype == array_type || atype == uarray_type)
				prcode(fp,",&a%d",arraylenarg);
		}
	}

	prcode(fp,"))\n");

	return needSecCall;
}


/*
 * Generate the code to tidy up after a call to C++.
 */

static void generateCallTidyUp(funcArgs *args,FILE *fp)
{
	int a;

	for (a = 0; a < args -> nrArgs; ++a)
	{
		classVersDef *cvd;

		if (args -> args[a].atype != class_type)
			continue;

		cvd = args -> args[a].u.cvd;

		if (cvd -> convtocode == NULL &&
		    !isTransferred(&args -> args[a]) &&
		    !isThisTransferred(&args -> args[a]) &&
		    !isTransferredBack(&args -> args[a]))
			continue;

		/* Delete temporary instances created by class convertors. */

		if (cvd -> convtocode != NULL)
			prcode(fp,
"\n"
"			if (istemp%d)\n"
"				delete a%d;\n"
				,a,a);

		if (isTransferred(&args -> args[a]))
			prcode(fp,
"\n"
"			sipTransferSelf(a%dobj,1);\n"
				,a);

		if (isThisTransferred(&args -> args[a]))
			prcode(fp,
"\n"
"			if (a%d)\n"
"				sipFlags &= ~SIP_PY_OWNED;\n"
				,a);

		if (isTransferredBack(&args -> args[a]))
			prcode(fp,
"\n"
"			sipTransferSelf(a%dobj,0);\n"
				,a);
	}
}


/*
 * Generate a C++ code block.
 */

static void generateCppCodeBlock(sipSpec *pt,codeBlock *code,char *cname,FILE *fp)
{
	codeBlock *cb;

	for (cb = code; cb != NULL; cb = cb -> next)
	{
		if (pt != NULL)
			generateVersionStart(pt,&cb -> version,fp);

		prcode(fp,
"#line %d \"%s\"\n"
			,cb -> linenr,cb -> filename);

		generateMacroCode(cb,cname,NULL,NULL,NULL,fp);

		if (pt != NULL)
			generateVersionEnd(pt,&cb -> version,fp);
	}

	if (code != NULL)
	{
		char *bn;

		/* Just use the base name. */

		if ((bn = strrchr(currentFileName,'/')) != NULL)
			++bn;
		else
			bn = currentFileName;

		prcode(fp,
"#line %d \"%s\"\n"
			,currentLineNr + 1,bn);
	}
}


/*
 * Generate a code fragment with macro expansion.
 */

static void generateMacroCode(codeBlock *cb,char *cname,sipSpec *pt,char *cppSuffix,char *objSuffix,FILE *fp)
{
	char *cp;

	for (cp = cb -> frag; *cp != '\0'; ++cp)
	{
		char mc;
		int invalid;

		if (*cp != MACRO_ESCAPE)
		{
			prcode(fp,"%c",*cp);
			continue;
		}

		invalid = FALSE;

		switch (mc = *++cp)
		{
		case MACRO_ESCAPE:
			prcode(fp,"%c",*cp);
			break;

		case MACRO_CLASS:
			if (cname != NULL)
				prcode(fp,"%s",cname);
			else
				invalid = TRUE;

			break;

		case MACRO_CPPSUFF:
			if (cppSuffix != NULL)
				prcode(fp,"%s",cppSuffix);
			else
				invalid = TRUE;

			break;

		case MACRO_OBJSUFF:
			if (objSuffix != NULL)
				prcode(fp,"%s",objSuffix);
			else
				invalid = TRUE;

			break;

		case MACRO_SOURCES:
			if (pt != NULL)
				generateMacroSource(pt,cppSuffix,fp);
			else
				invalid = TRUE;

			break;

		case MACRO_OBJECTS:
			if (pt != NULL)
				generateMacroObject(pt,objSuffix,fp);
			else
				invalid = TRUE;

			break;

		case MACRO_CPPMODULE:
			prcode(fp,"%s",pt -> cppmname);
			break;

		case MACRO_PERCENT:
			prcode(fp,"%c",'%');
			break;

		case '\0':
			--cp;
			break;

		default:
			fatal("Invalid macro: %c%c\n",MACRO_ESCAPE,mc);
		}

		if (invalid)
			fatal("%c%c macro invalid in this context\n",MACRO_ESCAPE,mc);
	}
}


/*
 * Generate the macro expansion for the list of source files.
 */

static void generateMacroSource(sipSpec *pt,char *cppSuffix,FILE *fp)
{
	char *mname = pt -> module -> name -> text;
	classDef *cd;

	prcode(fp,"%scmodule%s sip%sDecl%s.h",mname,cppSuffix,mname,mname);

	if (pt -> sigargs != NULL)
		prcode(fp," \\\n"
"	sip%sProxy%s.h",mname,mname);
 
	/* Dependencies on the source files. */
 
	for (cd = pt -> classes; cd != NULL; cd = cd -> next)
	{
		char *cname = cd -> name -> text;
 
		if (cd -> module == pt -> module)
			prcode(fp," \\\n"
"	sip%s%s.h sip%s%s%s",mname,cname,mname,cname,cppSuffix);
		else if (isUsed(cd))
			prcode(fp," \\\n"
"	sip%s%s.h",cd -> module -> name -> text,cname);
	}
}


/*
 * Generate the macro expansion for the list of object files.
 */

static void generateMacroObject(sipSpec *pt,char *objSuffix,FILE *fp)
{
	char *mname = pt -> module -> name -> text;
	classDef *cd;

	prcode(fp,"%scmodule%s",mname,objSuffix);

	/* Dependencies on the object files. */
 
	for (cd = pt -> classes; cd != NULL; cd = cd -> next)
		if (cd -> module == pt -> module)
			prcode(fp," \\\n"
"	sip%s%s%s",mname,cd -> name -> text,objSuffix);
}


/*
 * Generate the pre-processor code to check an ored set of anded versions.
 */

static void generateOredVersionStart(sipSpec *pt,versionOrList *vol,FILE *fp)
{
	int noIntro = TRUE;

	while (vol != NULL)
	{
		versionQual *low, *upp, *sec;

		pushVersion(pt,&vol -> va,&low,&upp,&sec);
		popVersion();

		if (low != NULL || upp != NULL || sec != NULL)
		{
			if (noIntro)
			{
				noIntro = FALSE;
				prcode(fp,
"#if (");
			}
			else
				prcode(fp,") || (");

			generateCppVersionExpr(low,upp,sec,fp);
		}

		vol = vol -> next;
	}

	if (!noIntro)
		prcode(fp,")\n"
			);
}


/*
 * Generate the pre-processor code to complete a version check.
 */

static void generateOredVersionEnd(sipSpec *pt,versionOrList *vol,FILE *fp)
{
	while (vol != NULL)
	{
		versionQual *low, *upp, *sec;

		pushVersion(pt,&vol -> va,&low,&upp,&sec);
		popVersion();

		if (low != NULL || upp != NULL || sec != NULL)
		{
			prcode(fp,
"#endif\n"
				);

			break;
		}

		vol = vol -> next;
	}
}


/*
 * Generate the pre-processor code to check an anded set of versions.
 */

static void generateVersionStart(sipSpec *pt,versionAnded *version,FILE *fp)
{
	versionQual *low, *upp, *sec;

	pushVersion(pt,version,&low,&upp,&sec);

	if (low != NULL || upp != NULL || sec != NULL)
	{
		prcode(fp,
"#if ");

		generateCppVersionExpr(low,upp,sec,fp);

		prcode(fp,"\n");
	}
}


/*
 * Generate a C++ pre-processor expression for a version.
 */

static void generateCppVersionExpr(versionQual *low,versionQual *upp,versionQual *sec,FILE *fp)
{
	char *sep = "";

	if (low != NULL)
	{
		prcode(fp,"%V",low);
		sep = " && ";
	}

	if (upp != NULL)
	{
		prcode(fp,"%s!%V",sep,upp);
		sep = " && ";
	}

	if (sec != NULL)
		prcode(fp,"%s%V",sep,sec);
}


/*
 * Generate the pre-processor code to complete a version check.
 */

static void generateVersionEnd(sipSpec *pt,versionAnded *version,FILE *fp)
{
	versionQual *low, *upp, *sec;

	popVersion();

	/*
	 * Push it again so that we can work out if anything was generated at
	 * the start of the version.
	 */

	pushVersion(pt,version,&low,&upp,&sec);
	popVersion();

	if (low != NULL || upp != NULL || sec != NULL)
		prcode(fp,
"#endif\n"
			);
}


/*
 * Check that a mapped class has convert-from code.
 */

static void checkForConvertFromCode(sipSpec *pt,classVersDef *cvd)
{
	if (cvd -> convfromcode == NULL)
	{
		fatal("Class %s ",classVersName(cvd));
		fatalVersion(pt,&cvd -> version);
		fatal(" must have a %%ConvertFromClassCode directive\n");
	}
}


/*
 * Create a file with an optional standard header.
 */

static FILE *createFile(sipSpec *pt,char *fname,char *commentStr,char *description)
{
	FILE *fp;

	/* Create the file. */

	if ((fp = fopen(fname,"w")) == NULL)
		fatal("Unable to create file \"%s\"\n",fname);

	currentLineNr = 1;
	currentFileName = fname;

	if (commentStr != NULL)
	{
		int needComment;
		codeBlock *cb;
		time_t now;

		/* Write the header. */

		now = time(NULL);

		prcode(fp,"%s %s\n",commentStr,description);
		prcode(fp,"%s\n",commentStr);
		prcode(fp,"%s Generated by SIP v%s on %s",commentStr,sipVersion,ctime(&now));

		if (pt -> copying != NULL)
			prcode(fp,"%s\n",commentStr);

		needComment = TRUE;

		for (cb = pt -> copying; cb != NULL; cb = cb -> next)
		{
			char *cp;

			for (cp = cb -> frag; *cp != '\0'; ++cp)
			{
				if (needComment)
				{
					needComment = FALSE;
					prcode(fp,"%s ",commentStr);
				}

				prcode(fp,"%c",*cp);

				if (*cp == '\n')
					needComment = TRUE;
			}
		}
	}

	return fp;
}


/*
 * Close a file and report any errors.
 */

static void closeFile(FILE *fp)
{
	if (ferror(fp))
		fatal("Error writing to \"%s\"\n",currentFileName);

	if (fclose(fp))
		fatal("Error closing \"%s\"\n",currentFileName);
}


/*
 * Print formatted code.
 */

static void prcode(FILE *fp,char *fmt,...)
{
	char ch;
	va_list ap;

	va_start(ap,fmt);

	while ((ch = *fmt++) != '\0')
		if (ch == '%')
		{
			ch = *fmt++;

			switch (ch)
			{
			case 'c':
				{
					char c = va_arg(ap,int);

					if (c == '\n')
						++currentLineNr;

					fputc(c,fp);
					break;
				}

			case 's':
				{
					char *cp = va_arg(ap,char *);

					while (*cp != '\0')
					{
						if (*cp == '\n')
							++currentLineNr;

						fputc(*cp,fp);
						++cp;
					}

					break;
				}

			case 'l':
				fprintf(fp,"%ld",va_arg(ap,long));
				break;

			case 'd':
				fprintf(fp,"%d",va_arg(ap,int));
				break;

			case 'g':
				fprintf(fp,"%g",va_arg(ap,double));
				break;

			case '\0':
				fputc('%',fp);
				--fmt;
				break;

			case '\n':
				fputc('\n',fp);
				++currentLineNr;
				break;

			case 'I':
				{
					int indent = va_arg(ap,int);

					while (indent-- > 0)
						fputc('\t',fp);

					break;
				}

			case 'E':
				{
					enumDef *ed = va_arg(ap,enumDef *);

					if (ed -> cvd != NULL)
						fprintf(fp,"%s::",ed -> cvd -> cd -> name -> text);

					fprintf(fp,"%s",ed -> name);

					break;
				}

			case 'N':
				{
					nameDef *nd = va_arg(ap,nameDef *);

					fprintf(fp,"sipName_%s_%s",nd -> module -> name -> text,nd -> text);
					break;
				}

			case 'S':
				{
					scopedNameDef *snd = va_arg(ap,scopedNameDef *);

					while (snd != NULL)
					{
						fprintf(fp,"%s",snd -> name);

						if ((snd = snd -> next) != NULL)
							fprintf(fp,"::");
					}

					break;
				}

			case 'V':
				{
					versionQual *vq = va_arg(ap,versionQual *);

					fprintf(fp,"defined(SIP_VERS_%s)",vq -> name);
					break;
				}

			default:
				fputc(ch,fp);
			}
		}
		else if (ch == '\n')
		{
			fputc('\n',fp);
			++currentLineNr;
		}
		else
			fputc(ch,fp);

	va_end(ap);
}
