/***************************************************************************
    file		: kb_db2cli.cpp
    copyright		: (C) 1999,2000,2001 by Mike Richardson & John Dean
			  (C) 2000,2001 by theKompany.com
    email		: mike@quaking.demon.co.uk & john@rygannon.com                                    
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *     This program is licensed under the terms contained in the file      *
 *     LICENSE which is contained with the source code distribution.       *
 *                                                                         *
 ***************************************************************************/



#include	"kb_db2cli.h"
#include	"kb_serverinfo.h"
#include	"kb_build.h"


static	DB2CLITypeMap typeMap[] =
{
{ SQL_SMALLINT,		"SMALLINT",			KB::ITFixed,	"SmallInt",	 	0		  },
{ SQL_INTEGER,		"INTEGER",			KB::ITFixed,	"Integer",	 	0		  },
{ SQL_BIGINT,		"BIGINT",			KB::ITFixed,	"BigInt",	 	0		  },
{ SQL_REAL,		"REAL",				KB::ITFloat,	"Real",		 	0		  },
{ SQL_DOUBLE,		"DOUBLE",			KB::ITFloat,	"Double",	 	0		  },
{ SQL_DECIMAL,		"DECIMAL(%1,%2)",		KB::ITFloat,	"Decimal",	 	FF_LENGTH|FF_PREC },
{ SQL_CHAR,		"CHAR(%1)",			KB::ITString,	"Char",		 	FF_LENGTH	  },
{ SQL_VARCHAR,		"VARCHAR (%1)",			KB::ITString,	"VarChar",	 	FF_LENGTH	  },
{ SQL_LONGVARCHAR,	"LONG VARCHAR",			KB::ITString,	"LongVarChar",	 	0		  },
{ SQL_BINARY,		"CHAR(%1) FOR BIT DATA",	KB::ITString,	"Char Bit",	 	FF_LENGTH	  },
{ SQL_VARBINARY,	"VARCHAR(%1) FOR BIT DATA",  	KB::ITString,	"VarChar Bit",	 	FF_LENGTH	  },
{ SQL_LONGVARBINARY,	"LONG VARCHAR FOR BIT DATA", 	KB::ITString,	"LongVarChar Bit",	0		  },
{ SQL_BLOB,		"BLOB",				KB::ITBinary,	"Blob",		 	0		  },
{ SQL_CLOB,		"CLOB",				KB::ITBinary,	"Clob",		 	0		  },
{ SQL_TYPE_DATE,	"DATE",				KB::ITDate,	"Date",	 	 	0		  },
{ SQL_TYPE_TIME,	"TIME",				KB::ITTime,	"Time",		 	0		  },
{ SQL_TYPE_TIMESTAMP,	"TIMESTAMP",			KB::ITDateTime,	"Date/Time",	 	0		  }
};

static QIntDict<DB2CLITypeMap> dIdentToType;


/*  getInternalType	: Get corresponding internal type		*/
/*  type		: SQLSMALLINT	: ODBC type			*/
/*  (returns)		: KB::IType	: Internal type			*/

static	KB::IType getInternalType
	(	SQLSMALLINT	type
	)
{
	DB2CLITypeMap *ptr = dIdentToType.find(type);
	return	ptr == 0 ? KB::ITUnknown : ptr->kbType;
}


inline	SQLINTEGER packBinary
	(	char		*,
		SQLINTEGER	bufflen
	)
{
	return	bufflen;
}

#define DBTAG 		"KBDB2CLI"
#define	DBTYPE		KBDB2CLI
#define	QRYSELECT 	KBDB2CLIQrySelect

#include 	"kb_odbcval.cpp"
#include 	"kb_odbccheck.cpp"


/*  KBDB2CLIType														*/
/*  KBDB2CLIType: Constructor for DB2 type object			*/
/*  iType	: KB::IType	: KBase internal type			*/
/*  length	: uint		: Database field length			*/
/*  nullOK	: bool		: True if null is OK			*/
/*  (returns)	: KBDB2CLIType	:					*/

KBDB2CLIType::KBDB2CLIType
	(	SQLSMALLINT	db2cliType,
		uint		length,
		bool		nullOK
	)
	:
	KBType	("DB2CLI", getInternalType(db2cliType), length, 0, nullOK)
{
}

/*  KBDB2CLIType							*/
/*  isValid	: Test if value is valid for type			*/
/*  value	: const QString & : Value to check			*/
/*  pError	: KBError &	  : Error return			*/
/*  where	: const QString & : Caller				*/
/*  (returns)	: bool		  : Value				*/

bool	KBDB2CLIType::isValid
	(	const QString	&value,
		KBError		&pError,
		const QString	&where
	)
{
	return KBType::isValid (value, pError, where);
}

/*  KBDB2CLIType							*/
/*  getQueryText: Get text for insertion into a query			*/
/*  value	: KBDataArray  * : Raw text of value to convert		*/
/*  d		: KBShared     * : Decoded representation		*/
/*  buffer	: KBDataBuffer & : Results buffer			*/
/*  codec	: QTextCodec *	 : Non-default codec			*/
/*  (returns)	: void		 :										*/

void	KBDB2CLIType::getQueryText
	(	KBDataArray	*value,
		KBShared	*d,
		KBDataBuffer	& buffer,
		QTextCodec	*codec
	)
{
	return	KBType::getQueryText(value, d, buffer, codec);
}

/*  ------------------------------------------------------------------  */

/*  KBSQLTypeInfo							*/
/*  loadTypeInfo: Load type information for retrieved values		*/
/*  stmHandle	: SQLHSTMT	: Statement handle			*/
/*  nFields	: uint		; Number of fields retrieved		*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBSQLTypeInfo::loadTypeInfo
	(	SQLHSTMT	stmHandle,
		uint		nFields,
		KBError		&pError
	)
{
	SQLRETURN	db2cliRC ;
	for (uint idx = 0; idx < nFields ; idx++)
	{
		SQLCHAR		colName[101]	;
		SQLSMALLINT	nameLen		;	
		SQLSMALLINT	colType		;
		SQLUINTEGER	colSize		;
		SQLSMALLINT	decimal		;
		SQLSMALLINT	nullable	;

		db2cliRC = SQLDescribeCol
			   (	stmHandle,
				idx + 1,
				colName,
				sizeof(colName),
				&nameLen,
				&colType,
				&colSize,
				&decimal,
				&nullable
			   )	;

		if (!db2cliOK(db2cliRC))
		{
			pError  = KBError
				  (	KBError::Error,
					TR("Error finding DB2 select column type"),
					QString::null,
					__ERRLOCN
				  )	;
			return	false	;
		}


		/* Types retured by SQLDescribeCol are like		*/
		/* SQL_CHAR						*/
		/* SQL_VARCHAR						*/
		/* ...							*/
		switch (colType)
		{
//			case SQL_TYPE_DATE:
//				m_cTypes.append (SQL_C_TYPE_DATE     ) ;
//				break;
//
//			case SQL_TYPE_TIME:
//				m_cTypes.append (SQL_C_TYPE_TIME     ) ;
//				break;
//
//			case SQL_TYPE_TIMESTAMP:
//				m_cTypes.append (SQL_C_TYPE_TIMESTAMP) ;
//				break;

			case SQL_NUMERIC  :
			case SQL_DECIMAL  :
				/* For numeric and decimal types, we	*/
				/* get the driver to do the conversion	*/
				/* for us. Much easier!			*/
				m_cTypes.append (SQL_C_DOUBLE	     )  ;
				break	;

			default:
				m_cTypes.append (SQL_C_DEFAULT	     )  ;
				break;
		}

		/* HACK ALERT!						*/
		/* We need to know if the column is actually an		*/
		/* identity, since if it is then the type should be	*/
		/* marked as null-ok, even though DB2 will say that it	*/
		/* is not nullable. The IBM DB2CLI documentation says	*/
		/* that we cannot find out if a column is an identity	*/
		/* (as of 27-Feb-04, version 8.1.4) so we maintain a	*/
		/* table/identity column name in ::doListFields(), and	*/
		/* use it here. This is OK since we know that rekall	*/
		/* will have called ::doListFields() for the table	*/
		/* before executing a select query if that select is	*/
		/* being used in a form.				*/
		SQLCHAR		tabName[101]	;
		SQLUINTEGER	dummy		;

		db2cliRC = SQLColAttribute
			   (	stmHandle,
				idx + 1,
				SQL_DESC_BASE_TABLE_NAME,
				tabName,
				sizeof(tabName),
				&nameLen,
				&dummy
			   )	;

		if (!db2cliOK(db2cliRC))
		{
			pError  = KBError
				  (	KBError::Error,
					TR("Error finding column attributes"),
					(cchar *)colName,
					__ERRLOCN
				  )	;
			return	false	;
		}

		if (tabName[0] != 0)
		{
			QString	identity = m_kbServer->findIdentity ((cchar *)tabName) ;
			if ((cchar *)colName == identity)
			{
				fprintf
				(	stderr,
					"KBSQLTypeInfo::loadTypeInfo: identity [%s] for [%s]\n",
					(cchar *)identity,
					(cchar *)tabName
				)	;

				nullable = true ;
			}
		}

		m_colNames.append ((cchar *)colName) ;
		m_dbTypes .append (colType) ;
		m_kbTypes[idx] = new KBDB2CLIType(colType, colSize, nullable);

		fprintf(stderr, "DB2: %3d: %5d: %12s: %s\n", idx, colType, tabName, (cchar *)m_kbTypes[idx]->getDescrip(true)) ;
	}

	return	true	;
}


/*  ------------------------------------------------------------------  */

/*  KBDB2CLI								*/
/*  KBDB2CLI	: Constructor for DB2 database connection class		*/
/*  (returns)	: KBDB2CLI	:					*/

KBDB2CLI::KBDB2CLI() : KBServer()
{
	m_EnvHandle = 0		;
	m_ConHandle = 0		;
	m_Connected = false	;
}

/*  KBDB2CLI								*/
/*  ~KBDB2CLI	: Destructor for DB2 database connection class		*/
/*  (returns)	:							*/

KBDB2CLI::~KBDB2CLI()
{
	if (m_Connected)
	{
		SQLDisconnect (m_ConHandle) ;
		SQLFreeHandle (SQL_HANDLE_ENV, m_EnvHandle);
		SQLFreeHandle (SQL_HANDLE_DBC, m_ConHandle);
	}
}

/*  KBDB2CLI								*/
/*  rekallPrefix: Modify name with rekall table prefix			*/
/*  name	: const QString & : Base name				*/
/*  (returns)	: QSTring	  : Modified name			*/

QString KBDB2CLI::rekallPrefix
	(	const QString	&name
	)
{
	return "$$" + name.upper() ;
}

/*  KBDB2CLI								*/
/*  keepsCase	: Indicate whether table/column names preserve case	*/
/*  (returns)	: bool		: True -> case preservation		*/

bool KBDB2CLI::keepsCase()
{
	return	false ;
}

/*  KBDB2CLI								*/
/*  getSyntax	: Get text for syntax element				*/
/*  result	: QString &	: Return resule string			*/
/*  syntax	: Syntax	: Element				*/
/*  ...		: ...		: Arguments				*/
/*  (returns)	: QString	: Text					*/

bool	KBDB2CLI::getSyntax
	(	QString			&result,
		KBServer::Syntax	syntax,
		...
	)
{
	va_list	 ap ;
	va_start (ap, syntax) ;

	if (syntax == Limit)
	{
		int	limit	= va_arg(ap, int) ;
		int	offset	= va_arg(ap, int) ;

		/* Is there any way to do offset ???			*/
		if (offset > 0)
		{
			m_lError = KBError
				   (	KBError::Error,
					QString(TR("Driver does not support %1 with non-zero offsets"))
						.arg(syntaxToText(syntax)),
					QString::null,
					__ERRLOCN
				   )	;
			return	false	;
		}

		result	= QString(" fetch first %1 rows only ").arg(limit) ;
		return	true	;
	}

	m_lError = KBError
		   (	KBError::Error,
			QString(TR("Driver does not support %1")).arg(syntaxToText(syntax)),
			QString::null,
			__ERRLOCN
		   )	;
	return	false	;
}


/*  KBDB2CLI								*/
/*  checkRCOK	: Check whether SQL statement executed correctly	*/
/*  stmHandle	: SQLHSTMT	: Statement handle			*/
/*  db2cliRC	: long		: ODBC return code			*/
/*  where	: cchar *	: First text string for possible error	*/
/*  htype	: SQLSMALLINT	: Handle type							*/
/*  (returns)	: bool		: True if statement executed correctly	*/

bool	KBDB2CLI::checkRCOK
	(	SQLHSTMT	stmHandle,
		SQLRETURN	db2cliRC,
		cchar 		*where,
		SQLSMALLINT	htype
	)
{
	return ::checkRCOK (stmHandle, db2cliRC, where, htype, m_lError) ;
}

/*  KBDB2CLI														*/
/*  checkDataOK	: Check whether SQL statement executed correctly	*/
/*		: Allows SQL_NO_DATA as OK (eg., update and delete)	*/
/*  stmHandle	: SQLHSTMT	: Statement handle			*/
/*  db2cliRC	: long		: ODBC return code			*/
/*  where	: cchar *	: First text string for possible error	*/
/*  (returns)	: bool		: True if statement executed correctly	*/

bool	KBDB2CLI::checkDataOK
	(	SQLHSTMT	stmHandle,
		SQLRETURN	db2cliRC,
		cchar		*where
	)
{
	return ::checkDataOK (stmHandle, db2cliRC, where, m_lError) ;
}

/*  KBDB2CLI								*/
/*  connect	: Open connection to database				*/
/*  svInfo	: KBServerInfo * : Server information			*/
/*  (returns)	: bool		 : Success				*/

bool	KBDB2CLI::doConnect
	(	KBServerInfo	*svInfo
	)
{
	SQLRETURN db2cliRC ;

	m_readOnly = svInfo->readOnly() ;

	/* Check that we are not already connected. Note that the	*/
	/* notConnected routine sets an error message.			*/
	if (m_ConHandle != 0)
	{	
		m_lError = KBError
			   (	KBError::Error,
				TR("Already connected to a DB2 database"),
				QString::null,
				__ERRLOCN
			   )	;
		return	false	;
	}

	/* If the database name contains two parts split by a colon	*/
	/* then use the part before the colon as the DB2INSTANCE value	*/
	/* and the rest as the database name.				*/
	QStringList dbSpec = QStringList::split(':', m_database) ;
	QString	    dbName ;

	if (dbSpec.count() == 2)
	{
		QString	dbInst = QString("DB2INSTANCE=%1").arg(dbSpec[0]) ;

		fprintf	(stderr, "KBDB2CLI::doConnect: [%s]\n", (cchar *)dbInst);
		putenv	(strdup((cchar *)dbInst)) ;

		dbName = dbSpec[1] ;
	}
	else	
		dbName = m_database;

	fprintf
	(	stderr,
		"DB2CLI: connecting to [%s]\n",
		dbName.latin1()
	)	;

	/* Allocate Environment handle and register our version with	*/
	/* the DB2CLI subsystem.					*/
	db2cliRC = SQLAllocHandle (SQL_HANDLE_ENV, SQL_NULL_HANDLE, &m_EnvHandle);

	if (!db2cliOK(db2cliRC))
	{
		m_lError = KBError
			   (	KBError::Error,
				"Failed to allocate DB2 environment handle", 
				QString::null,
				__ERRLOCN
			   )	;
			
		return	false	;
	}

	fprintf
	(	stderr,
		"DB2CLI: got environment handle\n"
	)	;

	SQLSetEnvAttr
	(	m_EnvHandle,
		SQL_ATTR_ODBC_VERSION,
		(SQLPOINTER)SQL_OV_ODBC3,
		0
	)	;

	db2cliRC = SQLSetEnvAttr
		   (	m_EnvHandle,
			SQL_ATTR_OUTPUT_NTS,
			(SQLPOINTER)SQL_TRUE,
			0
		   )	; 

	if (!checkRCOK
		(	m_EnvHandle,
			db2cliRC,
			"Error registering with DB2",
			SQL_HANDLE_ENV
		))
	{
		SQLFreeHandle(SQL_HANDLE_ENV, m_EnvHandle);
			
		m_lError = KBError
			   (	KBError::Error,
				"Error registering with DB2",
				QString::null,
				__ERRLOCN
			   )   	;
		
		return	false	;
	}

	fprintf
	(	stderr,
		"DB2CLI: about to allocate connection handle\n"
	)	;

	/* Next allocate a connection handle, and set a timeout on the	*/
	/* connection. Then we can see about getting the actual		*/
	/* connection itself.						*/
	db2cliRC = SQLAllocHandle (SQL_HANDLE_DBC, m_EnvHandle, &m_ConHandle); 
	if (!checkRCOK
		(	m_EnvHandle,
			db2cliRC,
			"Error getting DB2 connection handle",
			SQL_HANDLE_ENV
		))
	{
		SQLFreeHandle (SQL_HANDLE_ENV, m_EnvHandle) ;
		return false;
	}

	fprintf
	(	stderr,
		"DB2CLI: setting connection attrributes\n"
	)	;

	SQLSetConnectAttr (m_ConHandle, SQL_LOGIN_TIMEOUT, (SQLPOINTER *)5, 0);

	fprintf
	(	stderr,
		"DB2CLI: calling SQLConnect\n"
	)	;

	db2cliRC = SQLConnect
		   (	m_ConHandle,
			(SQLCHAR *)(cchar *)dbName, 	SQL_NTS,
			(SQLCHAR *)(cchar *)m_user, 	SQL_NTS,
			(SQLCHAR *)(cchar *)m_password, SQL_NTS
		   )	;

	if (!checkRCOK
		(	m_ConHandle,
			db2cliRC,
			"Error connecting DB2 data source",
			SQL_HANDLE_DBC
		))
	{
		SQLFreeHandle (SQL_HANDLE_DBC, m_ConHandle);
		SQLFreeHandle (SQL_HANDLE_ENV, m_EnvHandle);
		return	false ;
	}

	fprintf
	(	stderr,
		"DB2CLI: connected OK\n"
	)	;

	m_Connected  = true ;
	return	true  ;
}

/*  KBDB2CLI								*/
/*  qrySelect	: Open a select query					*/
/*  data	: bool		  : Query for data			*/
/*  select	: const QString & : Select query			*/
/*  update	: bool		  : Select for update			*/
/*  (returns)	: KBSQLSelect *	  : Select query class or NULL on error	*/

KBSQLSelect
	*KBDB2CLI::qrySelect
	(	bool		data,
		const QString	&select,
		bool		
	)
{
	return	new KBDB2CLIQrySelect(this, data, select) ;
}


/*  KBDB2CLI								*/
/*  qryUpdate	: Open an update query					*/
/*  data	: bool		  : Query for data			*/
/*  update	: const QString & : Update query			*/
/*  tabName	: const QString & : Table name				*/
/*  (returns)	: KBQryUpdate *	  : Update query class or NULL on error	*/

KBSQLUpdate
	*KBDB2CLI::qryUpdate
	(	bool 		data,
		const QString	&updateStr,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting update query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	new KBDB2CLIQryUpdate
		(	this,
			data,
			updateStr,
			tabName
		)	;
}

/*  KBDB2CLI								*/
/*  qryInsert	: Open an insert query					*/
/*  data	: bool		  : Query for data			*/
/*  insert	: const QString & : Insert query			*/
/*  tabName	: const QString & : Table name				*/
/*  (returns)	: KNQryInsert *	  : Insert query class or NULL on error	*/

KBSQLInsert
	*KBDB2CLI::qryInsert
	(	bool		data,
		const QString	&insertStr,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting insert query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	new KBDB2CLIQryInsert
		(	this,
			data,
			insertStr,
			tabName
		)	;
}

/*  KBDB2CLI								*/
/*  qryDelete	: Open a delete query					*/
/*  data	: bool		  : Query for data			*/
/*  _delete	: const QString & : Delete query			*/
/*  tabName	: const QString & : Table name				*/
/*  (returns)	: KNQryDelete *	  : Delete query class or NULL on error	*/

KBSQLDelete
	*KBDB2CLI::qryDelete
	(	bool		data,
		const QString	&deleteStr,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting delete query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return new KBDB2CLIQryDelete(this, data, deleteStr, tabName);
}

/*  KBDB2CLI								*/
/*  transaction	: Server transaction request				*/
/*  op		: Transaction	: Specific operation			*/
/*  activeCookie: void **	: Pass/return active cookie		*/
/*  (returns)	: bool		: Success				*/

bool	KBDB2CLI::transaction
	(	Transaction	op,
		void		**activeCookie
	)
{
	SQLRETURN db2cliRC	;
	QString	  eMsg		;
	bool	  acon		= false	;

	if (activeCookie != 0) *activeCookie = 0 ;

	switch (op)
	{
		case BeginTransaction	:
			db2cliRC = SQLSetConnectAttr
				   (	m_ConHandle,
					SQL_ATTR_AUTOCOMMIT,
					(SQLPOINTER)SQL_AUTOCOMMIT_OFF,
					SQL_NTS
				   )	;
			eMsg	 = TR("Error setting autocommit off") ;
			break	;

		case CommitTransaction	:
			db2cliRC = SQLEndTran
				   (	SQL_HANDLE_DBC,
					m_ConHandle,
					SQL_COMMIT
				   )	;
			eMsg	 = TR("Error committing work") ;
			acon	 = true	;
			break	;

		case RollbackTransaction:
			db2cliRC = SQLEndTran
				   (	SQL_HANDLE_DBC,
					m_ConHandle,
					SQL_ROLLBACK
				   )	;
			eMsg	 = TR("Error rolling back work") ;
			acon	 = true	;
			break	;

		default	:
			m_lError = KBError
				   (	KBError::Fault,
					TR("Unknown driver transaction operation"),
					QString(TR("Code: %1")).arg(op),
					__ERRLOCN
				   )	;
			return	false	;
	}

	bool	rc = checkRCOK
			(	m_ConHandle,
				db2cliRC,
				eMsg,
				SQL_HANDLE_DBC
			)	;

	if (acon)
		SQLSetConnectAttr
		(	m_ConHandle,
			SQL_ATTR_AUTOCOMMIT,
			(SQLPOINTER)SQL_AUTOCOMMIT_ON,
			SQL_NTS
		)	;

	return	rc	;
}

/*  KBDB2CLI								*/
/*  qryCursor	: Create cursor						*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Query text				*/
/*  cursor	: const QString & : Cursor name				*/
/*  (returns)	: bool		  : Success				*/

KBSQLCursor
	*KBDB2CLI::qryCursor
	(	bool		data,
		const QString	&query,
		const QString	&cursor
	)
{
	return	new KBDB2CLIQryCursor (this, data, query, cursor) ;
}

/*  KBDB2CLI								*/
/*  command	: Execute arbitrary SQL					*/
/*  data	: bool		  : Querying for data			*/
/*  query	: const QString & : Query text				*/
/*  nvals	: uint		  : Number of substitution values	*/
/*  values	: KBValue *	  : Substitution values			*/
/*  select	: KBSQLSelect **  : Return for data queries		*/
/*  (returns)	: bool		  : Success				*/

bool	KBDB2CLI::command
	(	bool		,
		const QString	&query,
		uint		nvals,
		KBValue		*,
		KBSQLSelect	**
	)
{
	if (nvals > 0)
	{
		m_lError = KBError
			   (	KBError::Error,
				"DB2 SQL commands do not support placeholders",
				QString::null,
				__ERRLOCN
			   )	;
		return	false	;
	}

	return	execSQL (query, QString(TR("Command: %1")).arg(query)) ;
}


bool	KBDB2CLI::getStatement
	(	SQLHSTMT	&stmHandle
	)
{
	SQLRETURN db2cliRC = SQLAllocStmt (m_ConHandle, &stmHandle) ;

	if (!db2cliOK(db2cliRC))
	{
		m_lError = KBError
			   (	KBError::Error,
				"Unable to get DB2 statement handle",
				QString::null,
				__ERRLOCN
			   )	;
		stmHandle = 0	;
		return	false	;
	}

	return	true ;
}

/*  KBDB2CLI								*/
/*  tableExists	: See if named table exists				*/
/*  table	: const QString & : Table name				*/
/*  exists	: bool &	  : True if table exists		*/
/*  (returns)	: bool		  : Success				*/

bool	KBDB2CLI::tableExists
	(	const QString	&table,
		bool 		&exists
	)
{
	KBTableDetailsList tabList;

	if (!doListTables (tabList, true, table))
	{
		exists = false	;
		return	false	;
	}

	exists	= tabList.count() > 0	;
	return	true ;
}

/*  KBDB2CLI								*/
/*  doListTables: List tables in database				*/
/*  tabList	: QList<QString &	: Result list			*/
/*  allTables	: bool			: Allow all tables		*/
/*  tabName	: const QString &	: Optional table name		*/
/*  type	: uint			: Type flags			*/
/*  (returns)	: bool			: Success			*/

bool	KBDB2CLI::doListTables
	(	KBTableDetailsList	&tabList,
		bool			allTables,
		const QString		&tabName,
		uint
	)
{
	SQLHSTMT stmHandle ;

	struct
	{
		SQLCHAR		val	[129]	;
		SQLINTEGER	ind		;
	}
		tbName,
		tbType		;

	struct
	{
		SQLCHAR		val	[255]	;
		SQLINTEGER	ind	;
	}
		tbRemarks	;

	SQLRETURN db2cliRC	;

	fprintf
	(	stderr,
		"KBDB2CLI::doListTables: called [%s]\n",
		(cchar *)tabName
	)	;

	/* Create an SQL statement and then execute the SQLTables		*/
	/* routine whicn retrieves the list of tables.				*/

	if (!getStatement(stmHandle)) 
		return	false	;


	SQLCHAR	tableType[] = "TABLE";
	db2cliRC = SQLTables(stmHandle, NULL, 0, NULL, SQL_NTS, NULL, SQL_NTS, tableType, SQL_NTS);

	if (!checkRCOK(stmHandle, db2cliRC, "Failed to retrieve DB2CLI table list"))
	{
		SQLFreeHandle	(SQL_HANDLE_STMT, stmHandle)	;
		return	false	;
	}

	SQLBindCol(stmHandle, 3, SQL_C_CHAR, tbName.val,    129, &tbName   .ind) ;
	SQLBindCol(stmHandle, 4, SQL_C_CHAR, tbType.val,    129, &tbType   .ind) ;
	SQLBindCol(stmHandle, 5, SQL_C_CHAR, tbRemarks.val, 255, &tbRemarks.ind) ;

	db2cliRC = SQLFetch (stmHandle) ;

	while(db2cliOK(db2cliRC))
	{
		QString	thisName ;

		if (tbName.ind == SQL_NULL_DATA)
			thisName = "UnknownTableName"  ;
		else	thisName = (cchar *)tbName.val ;

//		fprintf	
//		(	stderr,	
//			"KBDB2CLI::doListTables: got [%s]\n",
//			(cchar *)thisName
//		)	;

		if (!allTables)
			if (thisName.left(8).lower() == "$$rekall")
				thisName = QString::null ;

		if (!tabName.isEmpty())
			if (thisName.lower() != tabName.lower())
				thisName = QString::null ;

		if (!thisName.isNull())
			tabList.append
			(	KBTableDetails
				(	thisName,
					KB::IsTable, 
					QP_DELETE | QP_INSERT | QP_UPDATE | QP_DELETE
			)	)	;

		tbName   .val[0] = 0 ;
		tbType   .val[0] = 0 ;
		tbRemarks.val[0] = 0 ;

		db2cliRC = SQLFetch(stmHandle);
	}

	SQLFreeHandle(SQL_HANDLE_STMT, stmHandle);

	return true;
}

/*  KBDB2CLI								*/
/*  doListTables: List tables in database				*/
/*  tabList	: QList<QString> & : Result list			*/
/*  type	: uint		   : Type mask				*/
/*  (returns)	: bool		   : Success				*/

bool	KBDB2CLI::doListTables
	(	KBTableDetailsList	&tabList,
		uint			type
	)
{
	return	doListTables(tabList, m_showAllTables, QString::null, type) ;
}

/*  KBDB2CLI								*/
/*  getIdentityCols							*/
/*		: Get list of identify columns in table			*/
/*  tabName	: const QSring & : Table name				*/
/*  identityCols: QStringList  & : Return list for column names		*/
/*  (returns)	: bool		 : Success				*/

bool	KBDB2CLI::getIdentityCols
	(	const QString	&tabName,
		QStringList	&identityCols
	)
{
#define	IDENTITY_VAL_SQL	"SELECT NAME, IDENTITY FROM SYSIBM.SYSCOLUMNS WHERE TBNAME='%1' "

	SQLRETURN db2cliRC;
	SQLHSTMT stmHandle;
	QString identityValSQL = QString(IDENTITY_VAL_SQL).arg(tabName);

	struct
	{
		SQLINTEGER  ind;
		SQLCHAR		val[129];
	}
	colName;
	
	
	struct
	{
		SQLINTEGER	ind;
		SQLCHAR     val[2];
	}
	colIdentity;

	if(!getStatement(stmHandle)) 
		return false;

	db2cliRC = SQLPrepare(stmHandle, (uchar *)identityValSQL.latin1(), identityValSQL.length());

	if(!checkRCOK(stmHandle, db2cliRC, "Error preparing DB2 identity query"))
	{
		SQLFreeHandle(SQL_HANDLE_STMT, stmHandle);
		return false;
	}

	db2cliRC = SQLExecute(stmHandle);

	if(!checkRCOK(stmHandle, db2cliRC, "Error executing DB2 identity query"))
	{
		SQLFreeHandle(SQL_HANDLE_STMT, stmHandle);
		return false ;
	}

	for (db2cliRC = SQLFetch(stmHandle) ; db2cliOK(db2cliRC) ; db2cliRC = SQLFetch(stmHandle))
	{
		db2cliRC = SQLGetData
			   (	stmHandle,
				1,
				SQL_C_CHAR,
				colName.val, 
				sizeof(colName.val),
				&colName.ind
			   )	;

		db2cliRC = SQLGetData
			   (	stmHandle,
				2,
				SQL_C_CHAR,
				colIdentity.val,
				sizeof(colIdentity.val),
				&colIdentity.ind
			   )	;

		fprintf
		(	stderr,
			"KBDB2CLI::getIdentityCols: [%s][%s]\n",
			(cchar *)colName.val,
			(cchar *)colIdentity.val
		)	;

		if (colIdentity.val[0] == 'Y')
			identityCols.append((cchar *)colName.val);
	}

	fprintf
	(	stderr,
		"KBDB2CLI::getIdentityCols: %d columns\n",
		identityCols.count()
	)	;

	SQLFreeHandle (SQL_HANDLE_STMT, stmHandle);
	return	true  ;
}

/*  KBDB2CLI								*/
/*  listFields	: List fields in table					*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  (returns)	: bool		: Success				*/

bool	KBDB2CLI::doListFields
	(	KBTableSpec	&tabSpec
	)
{
	SQLHSTMT stmHandle;
	QString  identityValSQL = QString::null;

	struct
	{
		SQLINTEGER  ind;
		SQLCHAR     val[129];
	}
	colName;
	
	struct
	{
		SQLINTEGER  ind;
		SQLINTEGER  val;
	}
	colLength;
	
	struct
	{
		SQLINTEGER  ind;
		SQLSMALLINT val;
	}
	colType, colPrec, colNullable;

	SQLCHAR	table[101] = "";

	uint idx = 0;
	SQLRETURN db2cliRC;

	tabSpec.m_fldList.clear();
	tabSpec.m_prefKey   = -1;
	tabSpec.m_keepsCase = false;

	if(!getStatement(stmHandle)) 
		return false;

	strncpy((char *)table, tabSpec.m_name, 100);
	table[100] = 0;

	db2cliRC = SQLColumns(stmHandle, 0, 0, 0, 0, table, SQL_NTS, 0, 0);

	if(!checkRCOK(stmHandle, db2cliRC, "Failed to get DB2 column list"))
	{
		SQLFreeHandle(SQL_HANDLE_STMT, stmHandle);

		return false;
	}

	for (db2cliRC = SQLFetch(stmHandle); db2cliOK(db2cliRC); db2cliRC = SQLFetch(stmHandle))
	{
		db2cliRC = SQLGetData(stmHandle, 4, SQL_C_CHAR, colName.val,
			sizeof(colName.val), &colName.ind);

		if(!db2cliOK(db2cliRC))
			strcpy((char *)colName.val, "Unknown");

		SQLGetData
		(	stmHandle,
			5,
			SQL_C_SHORT,
			(SQLPOINTER)&colType.val,
			sizeof(colType.val),
			&colType.ind
		)	;

		SQLGetData
		(	stmHandle,
			7,
			SQL_C_SLONG,
			(SQLPOINTER)&colLength.val, 
			sizeof(colLength.val),
			&colLength.ind
		)	;

		SQLGetData
		(	stmHandle,
			9,
			SQL_C_SHORT,
			(SQLPOINTER)&colPrec.val, 
			sizeof(colPrec.val),
			&colPrec.ind
		)	;

		SQLGetData
		(	stmHandle,
			11,
			SQL_C_SHORT,
			(SQLPOINTER)&colNullable.val,
			sizeof(colNullable.val),
			&colNullable.ind
		)	;


		fprintf
		(	stderr,
			"DB2CLI: field  [%s] [%d] [%d] [%d]\n",
			(cchar *)colName.val,
			(int)colType.val,
			(int)colLength.val,
			(int)colNullable.val
		)	;

		DB2CLITypeMap	*ptr  = dIdentToType.find(colType.val) ;
		QString		ktype = ptr == 0 ? QString("<%1>").arg(colType.val) : QString(ptr->kbName) ;
		KB::IType	itype = ptr == 0 ? KB::ITUnknown : ptr->kbType ;
		uint 		flags = 0 ;

		if (colNullable.val == SQL_NO_NULLS) 
			flags |= KBFieldSpec::NotNull ;

		if (ptr != 0)
		{
			if ((ptr->flags & FF_LENGTH) == 0) colLength.val = 0 ;
			if ((ptr->flags & FF_PREC  ) == 0) colPrec  .val = 0 ;
		}

		KBFieldSpec *s = new KBFieldSpec
				 (	(uint)idx,
				 	(cchar *)colName.val,
				 	ktype,
					itype,
					flags,
					colLength.val,
					colPrec.val
				 )	;

		tabSpec.m_fldList.append(s);
		idx	+= 1;
	}

	SQLFreeHandle (SQL_HANDLE_STMT, stmHandle);

	/* Next step is to loog for the primary key column. If found,	*/
	/* it will be marked as Unique and NotNull (irrespective of	*/
	/* what was found above).					*/
	if(!getStatement(stmHandle)) 
		return false ;

	db2cliRC = SQLPrimaryKeys (stmHandle, 0, 0, 0, 0, table, SQL_NTS);

	if(!checkRCOK(stmHandle, db2cliRC, "Failed to get DB2 primary key list"))
	{
		SQLFreeHandle(SQL_HANDLE_STMT, stmHandle);
		return false ;
	}

	for (db2cliRC = SQLFetch(stmHandle) ; db2cliOK(db2cliRC) ; db2cliRC = SQLFetch(stmHandle))
	{
		db2cliRC = SQLGetData
		(	stmHandle,
			4,
			SQL_C_CHAR,
			colName.val,
			sizeof(colName.val),
			&colName.ind
		)	;

		if(!db2cliOK(db2cliRC)) 
			continue ;

//		fprintf
//		(	stderr,
//			"DB2: primary [%s]->[%s]\n",
//			(cchar *)tabSpec.name,
//			(cchar *)colName.val
//		)	;

		for (idx = 0 ; idx < tabSpec.m_fldList.count() ; idx += 1)
		{
			KBFieldSpec *fspec = tabSpec.m_fldList.at(idx);

			if (fspec->m_name != QString((cchar *)colName.val))
				continue ;

			fprintf (stderr, "    : matched %d\n", idx);

			fspec->m_typeName = "Primary Key" ;
			fspec->m_flags   |= KBFieldSpec::Primary |
					    KBFieldSpec::Unique	 |
					    KBFieldSpec::NotNull ;

			tabSpec.m_prefKey = idx ;
			break	;
		}
	}

	SQLFreeHandle(SQL_HANDLE_STMT, stmHandle);

	/* Check for identity columns. Any such columns are noted as	*/
	/* serial, insert-available, and read-only. They are also noted	*/
	/* in the identities map; see the comments in ::loadTypeInfo()	*/
	/* for details.							*/
	QStringList	identityCols ;
	if (!getIdentityCols (tabSpec.m_name, identityCols))
		return	false	;

	clearIdentity (tabSpec.m_name) ;

	for (uint idx1 = 0 ; idx1 < identityCols.count() ; idx1 += 1)
	{
		KBFieldSpec *spec = tabSpec.findField (identityCols[idx1]);

		if (spec != 0) 
		{
			spec->m_flags |= KBFieldSpec::Serial   |
					 KBFieldSpec::InsAvail |
					 KBFieldSpec::ReadOnly ;
			setIdentity (tabSpec.m_name, spec->m_name) ;
		}
	}

	return true;
}

/*  KBDB2CLI								*/
/*  execSQL	: Simple SQL execution with no placeholders		*/
/*  sql		: const QString & : SQL statement			*/
/*  where	: cchar *	  : Text string for possible error	*/
/*  (returns)	: bool		  : True if statement executed OK	*/

bool	KBDB2CLI::execSQL
	(	const QString	&sql,
		cchar		*where
	)
{
	SQLHSTMT  stmHandle ;
	if (!getStatement(stmHandle)) return false ;

	bool	  rc	    = true ;
	SQLCHAR	  *s	    = (SQLCHAR *)(cchar *)sql ;
	SQLRETURN db2cliRC  = SQLExecDirect (stmHandle, s, strlen((cchar *)s)) ;

	if (!checkRCOK (stmHandle, db2cliRC, where))
		rc = false ;

	SQLFreeHandle (SQL_HANDLE_STMT, stmHandle) ;
	printQuery    (sql, 0, 0, rc) ;
	return rc ;
}

/*  KBDB2CLI								*/
/*  createTable	: Create a new table					*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  assoc	: bool		: Create associated stuff		*/
/*  best	: bool		: Use best match			*/
/*  (returns)	: bool		: Success				*/

bool	KBDB2CLI::doCreateTable
	(	KBTableSpec	&tabSpec,
		bool		,
		bool
	)
{
	cchar *sep = "";
	QString	create = QString("create table %1\n(").arg(tabSpec.m_name);

	/* Main loop generates SQL create for the columns and column	*/
	/* types, plus options like auto-increment and primary key.	*/
	for (uint idx = 0 ; idx < tabSpec.m_fldList.count() ; idx += 1)
	{
		KBFieldSpec *fSpec = tabSpec.m_fldList.at(idx) ;

		/* Special cases. If the field is "Primary Key" then we	*/
		/* create an "Int4" column marked as primary key, not	*/
		/* null and auto increment ...							*/

		QString	ftype = fSpec->m_typeName ;
		
		if(ftype == "Primary Key")
		{
			// We do not need to add NOT NULL clause since IDENTITY columns
			// are defined as NOT NULL by default
			create += QString ("%1\t%2 integer generated by default as identity (start with 0 increment by 1) primary key")
					  .arg(sep).arg(fSpec->m_name);
			sep	= ",\n";
			continue;
		}

		/* ... while a foreign key is also "Int4" not null.	*/

		if(ftype == "Foreign Key")
		{
			create += QString("%1\t%2 integer not null").arg(sep).arg(fSpec->m_name) ;
			sep	= ",\n";
			continue;
		}

		/* Map the types used when creating the object and	*/
		/* design dictionary tables.				*/
		if	(ftype == "_Text"   ) ftype = "VarChar";
		else if (ftype == "_Integer") ftype = "Integer";
		else if (ftype == "_Binary" ) ftype = "Blob";


		/* Scan though the type mapping table looking for the	*/
		/* required type.					*/
		DB2CLITypeMap *mapp = 0 ;
		for (uint slot = 0 ; slot < sizeof(typeMap)/sizeof(typeMap[0]) ; slot += 1)
			if (typeMap[slot].kbName == ftype)
				if ((typeMap[slot].flags & FF_NOCREATE) == 0)
				{	mapp	= &typeMap[slot];
					break	;
				}

		if (mapp == 0)
		{
			m_lError = KBError
				   (	KBError::Fault,
					TR("Error mapping column type"),
				  	QString ("Type %1 for column %2 not known")
						.arg(ftype)
						.arg(fSpec->m_name),
					__ERRLOCN
				   )	;
			return	false	;
		}

		QString	colSpec	;

		if	((mapp->flags & FF_LENGTH) == 0)
		{
			colSpec	= QString(mapp->db2CliName);
		}
		else if	((mapp->flags & FF_PREC  ) == 0)
		{
			colSpec	= QString(mapp->db2CliName).arg(fSpec->m_length);
		}
		else	
			colSpec	= QString(mapp->db2CliName).arg(fSpec->m_length).arg(fSpec->m_prec);


		create	+= QString("%1\t%2 %3").arg(sep).arg(fSpec->m_name).arg(colSpec);

		if((fSpec->m_flags & KBFieldSpec::NotNull) != 0) 
			create += " not null";
		
		if((fSpec->m_flags & KBFieldSpec::Primary) != 0) 
			create += " primary key";

		if((fSpec->m_flags & KBFieldSpec::Unique) != 0) 
			create += " unique";

		sep	= ",\n";
	}


#if	0
	/* Now add any unique columns. Note that we do not handle	*/
	/* multiple-column uniqueness.								*/

	LITER
	(	KBFieldSpec,
		tabSpec.fldList,
		fSpec,

		if ((fSpec->flags & KBFieldSpec::Indexed) != 0)
		{
			create	+= sep	;
			create	+= QString("index  (%1)").arg(fSpec->name);
			sep	= ",\n";
		}
	)
#endif
	create	+= "\n)";

	fprintf(stderr, "DB2:\n%s\n", (cchar *)create);

	return execSQL(create, "Error creating table");
}

/*  KBDB2CLI														*/
/*  renameTable	: Rename a table									*/
/*  oldName		: cchar *	: Current table name					*/
/*  newName		: cchar *	: New table name						*/
/*  assoc		: bool		: Rename associated stuff - none here	*/
/*  (returns)	: bool		: Success								*/

bool KBDB2CLI::doRenameTable(cchar *oldName, cchar *newName, bool)
{
	QString renameStmt = QString("RENAME TABLE %1 TO %2").arg(oldName).arg(newName);

	return execSQL(renameStmt, QString("Failed to rename table \"%1\" as \"%2\"").arg(oldName).arg(newName));
}

/*  KBDB2CLI													*/
/*  dropTable	: Drop a table									*/
/*  table		: cchar *	: Table name						*/
/*  assoc		: bool		: Drop associated stuff - none here	*/
/*  (returns)	: bool		: Success							*/

bool KBDB2CLI::doDropTable(cchar *table, bool)
{
	QString dropStmt = QString("DROP TABLE %1").arg(table);

	return execSQL(dropStmt, QString("Failed to drop table \"%1\"").arg(table));
}

/*  KBServer											*/
/*  listTypes	: Return a list of all column types		*/
/*  (returns)	: QString	: Types						*/

QString	KBDB2CLI::listTypes()
{
	static QString typeList;

	if(typeList.isNull())
	{
		typeList = "Primary Key,0|Foreign Key,0";

		for(uint idx = 0; idx < sizeof(typeMap)/sizeof(typeMap[0]); idx++)
		{
			DB2CLITypeMap* pMap = &typeMap[idx];

			if((pMap->flags & FF_NOCREATE) != 0)
				continue;

			typeList += QString("|%1,%2").arg(pMap->kbName).arg(pMap->flags);
		}
	}

	return typeList;
}

/*  KBDB2CLI								*/
/*  bindParameters							*/
/*		: Bind placeholder parameters to statement		*/
/*  stmHandle	: SQLHSTMT	 : Statement handle			*/
/*  nvals	: uint		 : Number of substitution values	*/
/*  value	: KBValue *	 : Substitution values			*/
/*  vList	: KBODBCValue &  : Substituted values list		*/
/*  codec	: QTextCodec *	 : Non-default codec			*/
/*  (returns)	: bool		 : Success				*/

bool	KBDB2CLI::bindParameters
	(	SQLHSTMT		stmHandle,
		uint			nvals,
		const KBValue		*values,
		QList<KBODBCValue>	&vList,
		QTextCodec		*codec
	)
{
	SQLRETURN db2cliRC;

	for(uint idx = 0 ; idx < nvals ; idx += 1)
	{
		KBODBCValue *ODBCVal = new KBODBCValue(values[idx], codec);

		vList.append(ODBCVal);

		db2cliRC = SQLBindParameter
			   (	stmHandle,
				idx + 1,
				SQL_PARAM_INPUT,
				ODBCVal->vtype(),
				ODBCVal->ptype(),
				0,
				0,
				ODBCVal->vptr (),
				ODBCVal->vlen (),
				ODBCVal->slind()
			   )	;

		if (!checkRCOK
			(	stmHandle,
				db2cliRC,
				"Error binding DB2 parameter"
			))
			return false;
	}

	return true;
}

void	KBDB2CLI::describeParams
	(	SQLHSTMT	stmHandle,
		QValueList<SQLSMALLINT>	&paramTypes
	)
{
	uint		paramNo		;
	SQLSMALLINT	dataType	;
	SQLUINTEGER	paramSize	;
	SQLSMALLINT	decimals	;
	SQLSMALLINT	nullable	;

	paramNo	= 0	;

	while (db2cliOK(SQLDescribeParam
			(	stmHandle,
				paramNo + 1,
				&dataType,
				&paramSize,
				&decimals,
				&nullable
	      )	       ))
	{
		fprintf
		(	stderr,
			"KBDB2CLI::describe: %u->%d\n",
			paramNo,
			dataType
		)	;

		paramTypes.append (dataType) ;
		paramNo	+= 1 ;
	}
}

/*  ------------------------------------------------------------------- */


/*  KBDB2CLIQrySelect							*/
/*  KBDB2CLIQrySelect							*/
/*		: Constructor for select query object			*/
/*  server	: KBDB2CLI *		: Server connection object	*/
/*  data	: bool			: Query for data		*/
/*  select	: const QString &	: Select query text		*/
/*  update	: bool			: Select for update		*/
/*  (returns)	: KBDB2CLIQrySelect	:				*/

KBDB2CLIQrySelect::KBDB2CLIQrySelect
	(	KBDB2CLI	*server,
		bool		data,
		const QString	&select
	)	
	:
	KBSQLSelect	(server, data, select),
	KBSQLTypeInfo	(server, m_types),
	m_pServer	(server)
{
	m_CRow		= -1;
	m_nRows 	= 0	;
	m_nFields	= 0	;

	if (!m_pServer->getStatement(m_stmHandle))
		return	;

	QCString  qryText  = m_rawQuery.utf8();
	SQLRETURN db2cliRC = SQLPrepare (m_stmHandle, (uchar *)qryText.data(), qryText.length()) ;

	if(!m_pServer->checkRCOK(m_stmHandle, db2cliRC, "Error preparing statement from DB2"))
	{
		SQLFreeHandle(SQL_HANDLE_STMT, m_stmHandle);
		m_stmHandle = 0;
		m_lError = m_pServer->lastError();
	}
}

/*  KBDB2CLIQrySelect							*/
/*  ~KBDB2CLIQrySelect							*/
/*		: Destructor for select query object	*/
/*  (returns)	:		:						*/

KBDB2CLIQrySelect::~KBDB2CLIQrySelect ()
{
	if (m_stmHandle != 0)
		SQLFreeHandle(SQL_HANDLE_STMT, m_stmHandle);

}

/*  KBDB2CLIQrySelect										*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBDB2CLIQrySelect::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	SQLRETURN db2cliRC   ;

	if (m_stmHandle == 0)
		return false ;

	SQLCloseCursor (m_stmHandle) ;

	m_dbTypes .clear () ;
	m_cTypes  .clear () ;
	m_colNames.clear () ;

	QList<KBODBCValue> valueList;
	valueList.setAutoDelete(true);

	fprintf
	(	stderr,
		"KBDB2CLIQrySelect::execute: [%s] [%d]\n",
		(cchar *)m_rawQuery,
		nvals

	)	;

	if (!m_pServer->bindParameters
		(	m_stmHandle,
			nvals,
			values,
			valueList,
			m_codec
		))
	{
		m_lError = m_pServer->lastError();
		return false;
	}

	db2cliRC = SQLExecute (m_stmHandle);
	m_pServer->printQuery (m_rawQuery, nvals, values, db2cliOK(db2cliRC)) ;

	if (!m_pServer->checkRCOK
		(	m_stmHandle,
			db2cliRC,
			"Error executing DB2 select query"
		))
	{
		m_lError = m_pServer->lastError();
		return false;
	}

	m_nRows = RowsUnknown	;

	SQLSMALLINT	nFields	;
	SQLNumResultCols(m_stmHandle, &nFields) ;
	m_nFields = nFields	;

	if (m_types == 0)
	{
	m_types = new KBType*[m_nFields];

	if (!loadTypeInfo
		(	m_stmHandle,
			m_nFields,
			m_lError
		))
	{
		return	false	;
	}
	}

	m_CRow = -1;
	return true;
}

#include	"kb_odbcrow.cpp"

/*  KBDB2CLIQrySelect							*/
/*  getField	: Get a specified field from the query results		*/
/*  qrow	: uint		  : Row number				*/
/*  qcol	: uint		  : Column number			*/
/*  vtrans	: KBValue::VTrans : Translation mode			*/
/*  (returns)	: KBValue	  : Value				*/

KBValue	KBDB2CLIQrySelect::getField
	(	uint		qrow,
		uint		qcol,
		KBValue::VTrans
	)
{
	if (rowExists (qrow))
	{
		KBValue	value	;
		return	getFromCache (qrow, qcol, value) ? value : KBValue() ;
	}

	return	KBValue() ;
}

/*  KBDB2CLIQrySelect													*/
/*  getFieldName: Get a specified field name from the query results	*/
/*  qcol	: uint		: Column number				*/
/*  (returns)	: QString	: Name					*/

QString	KBDB2CLIQrySelect::getFieldName
	(	uint	qcol
	)
{
	return	m_colNames[qcol] ;
}

/*  KBDB2CLIQryUpdate							*/
/*  KBDB2CLIQryUpdate							*/
/*		: Constructor for update query object			*/
/*  server	: KBDB2CLI *	    : Server connection object		*/
/*  data	: bool			: Query for data		*/
/*  query	: const QString &   : Update query text			*/
/*  tabName	: const QString &   : Table being updated		*/
/*  codec	: QTextCodec *	    : Non-default codec			*/
/*  (returns)	: KBDB2CLIQryUpdate :							*/

KBDB2CLIQryUpdate::KBDB2CLIQryUpdate
	(	KBDB2CLI	*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLUpdate	(server, data, query, tabName),
	m_pServer	(server)
{
	m_nRows = 0	;

	if (!m_pServer->getStatement (m_stmHandle))
		return ;

	QCString  qryText  = m_rawQuery.utf8() ;
	SQLRETURN db2cliRC = SQLPrepare (m_stmHandle, (uchar *)qryText.data(), qryText.length()) ;

	if (!m_pServer->checkRCOK
		(	m_stmHandle,
			db2cliRC,
			"Error preparing statement for DB2"
		))
	{
		SQLFreeHandle (SQL_HANDLE_STMT, m_stmHandle);
		m_stmHandle   = 0;
		m_lError      = m_pServer->lastError();
		return	;
	}

//	fprintf
//	(	stderr,
//		"DB2: [%s]\n",
//		(cchar *)query
//	)	;
}

/*  KBDB2CLIQryUpdate							*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBDB2CLIQryUpdate::execute
	(	uint 		nvals,
		const KBValue	*values
	)
{
	SQLRETURN db2cliRC;

	if (m_stmHandle == 0)
		return false ;

	SQLCloseCursor(m_stmHandle);

	QList<KBODBCValue> valueList;
	valueList.setAutoDelete(true);

//	fprintf
//	(	stderr,
//		"KBDB2CLIQryUpdate::execute: [%s] [%d]\n",
//		(cchar *)m_rawQuery,
//		nvals
//	)	;

	if (!m_pServer->bindParameters
		(	m_stmHandle,
			nvals,
			values,
			valueList,
			m_codec
		))
	{
		m_lError = m_pServer->lastError();
		return	 false ;
	}

	db2cliRC = SQLExecute (m_stmHandle);
	m_pServer->printQuery (m_rawQuery, nvals, values, db2cliOK(db2cliRC)) ;
	if (!m_pServer->checkDataOK
		(	m_stmHandle,
			db2cliRC,
			"Error executing DB2 update query"
		))
	{
		m_lError = m_pServer->lastError() ;
		return	false ;
	}

	db2cliRC = SQLRowCount (m_stmHandle, (SQLINTEGER*)&m_nRows);
	if (!m_pServer->checkRCOK
		(	m_stmHandle,
			db2cliRC,
			"Error getting row count"
		))
	{
		m_lError = m_pServer->lastError() ;
		return	false ;
	}


	return true;
}

/*  KBDB2CLIQryUpdate							*/
/*  ~KBDB2CLIQryUpdate							*/
/*		: Destructor for update query object			*/
/*  (returns)	:		:					*/

KBDB2CLIQryUpdate::~KBDB2CLIQryUpdate()
{
	if(m_stmHandle != 0)
		SQLFreeHandle(SQL_HANDLE_STMT, m_stmHandle);
}


/*  KBDB2CLIQryInsert							*/
/*  KBDB2CLIQryInsert							*/
/*		: Constructor for insert query object			*/
/*  server	: KBDB2CLI *		: Server connection object	*/
/*  data	: bool			: Query for data		*/
/*  query	: const QString &	: Insert query			*/
/*  tabName	: const QString &	: Table being updated		*/
/*  codec	: QTextCodec *	    	: Non-default codec		*/
/*  (returns)	: KBDB2CLIQryInsert	:				*/

KBDB2CLIQryInsert::KBDB2CLIQryInsert
	(	KBDB2CLI	*server,
		bool		data,
		const QString	&query,
		const QString 	&tabName
	)	
	:
	KBSQLInsert (server, data, query, tabName),
	m_pServer   (server)
{
#define	LAST_ASSIGNED_SQL	"SELECT IDENTITY_VAL_LOCAL() "		\
				"FROM SYSIBM.SYSDUMMY1"
	m_nRows			= 0	;
	m_useIdentityCol	= -1	;
	m_stmHandle		= 0	;
	m_IdentityStmHandle	= 0	;

	/* Prepare the insert query now. If this fails then the error	*/
	/* will be detected when ::execute is called since the handle	*/
	/* will have been returned to null.				*/
	if (!m_pServer->getStatement(m_stmHandle))
		return;

	QCString  qryText  = m_rawQuery.utf8() ;
	SQLRETURN db2cliRC = SQLPrepare (m_stmHandle, (uchar *)qryText.data(), qryText.length()) ;

	if (!m_pServer->checkRCOK(m_stmHandle, db2cliRC, "Error preparing DB2 insert query"))
		goto error ;

	fprintf	(stderr, "DB2: [%s]\n", (cchar *)query) ;

	m_pServer->describeParams (m_stmHandle, m_paramTypes) ;

	/* Similarly prepare a query to retrieve the last identity	*/
	/* value. This *might* be wasted effort, but we will need it	*/
	/* immediately after the insert is executed so that it can be	*/
	/* executed inside an atomic transaction.			*/
	if (!m_pServer->getStatement(m_IdentityStmHandle))
		goto error ;

	db2cliRC = SQLPrepare
		   (	m_IdentityStmHandle,
			(uchar *)LAST_ASSIGNED_SQL,
			strlen(LAST_ASSIGNED_SQL)
		   )	;

	if (!m_pServer->checkRCOK (m_IdentityStmHandle, db2cliRC, "Error preparing DB2 last assigned query"))
		goto error ;

	return	;

	error	:
		m_lError = m_pServer->lastError() ;

		if (m_stmHandle != 0)
		{
			SQLFreeHandle (SQL_HANDLE_STMT, m_stmHandle) ;
			m_stmHandle   = 0 ;
		}
		if (m_IdentityStmHandle != 0)
		{
			SQLFreeHandle (SQL_HANDLE_STMT, m_IdentityStmHandle) ;
			m_IdentityStmHandle   = 0 ;
		}
}

/*  KBDB2CLIQryInsert											*/
/*  ~KBDB2CLIQryInsert	: Destructor for insert query object	*/
/*  (returns)	:		:										*/

KBDB2CLIQryInsert::~KBDB2CLIQryInsert()
{
	if(m_stmHandle != 0)
		SQLFreeHandle(SQL_HANDLE_STMT, m_stmHandle);

	if(m_IdentityStmHandle != 0)
		SQLFreeHandle(SQL_HANDLE_STMT, m_IdentityStmHandle);
}

/*  KBDB2CLIQryInsert							*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBDB2CLIQryInsert::execute
	(	uint 		nvals,
		const KBValue	*values
	)
{
	/* If the statement handle is null then there was an error in	*/
	/* the constructor. The error code will already have been set	*/
	/* up, so return immediately.					*/
	if (m_stmHandle == 0)
		return false;

	SQLCloseCursor(m_stmHandle) ;

	QList<KBODBCValue> valueList ;
	valueList.setAutoDelete (true) ;

	fprintf
	(	stderr,
		"KBDB2CLIQryInsert::execute: [%s] [%d]\n",
		(cchar *)m_rawQuery,
		nvals

	)	;


	/* If the table contains an identity then we will very likely	*/
	/* want to retrieve it. To make sure there are no problems with	*/
	/* concurrent access, we do the insert inside a transaction	*/
	/* which is commited after the identity is retrieved (and there	*/
	/* is a rumour that without this, identity_val_local() returs	*/
	/* null).							*/
	SQLRETURN db2cliRC = SQLSetConnectAttr
		  	     (	m_pServer->connectionHandle(),
				SQL_ATTR_AUTOCOMMIT,
				(SQLPOINTER)SQL_AUTOCOMMIT_OFF,
				SQL_NTS
			     )	;
	
	if(!m_pServer->checkRCOK(m_pServer->connectionHandle(), db2cliRC, "Error disabling autocommit"))
	{
		m_lError = m_pServer->lastError();
		return false;
	}
	
	if(!m_pServer->bindParameters(m_stmHandle, nvals, values, valueList, m_codec))
	{
		m_lError = m_pServer->lastError();
		return false;
	}
	
	db2cliRC = SQLExecute(m_stmHandle);
	m_pServer->printQuery(m_rawQuery, nvals, values, db2cliOK(db2cliRC)) ;

	if(!m_pServer->checkDataOK(m_stmHandle, db2cliRC, "Error executing DB2 insert query"))
	{
		m_lError = m_pServer->lastError();
		SQLEndTran(SQL_HANDLE_DBC, m_pServer->connectionHandle(), SQL_ROLLBACK);
		return false;
	}

	/* Grab the last identity value. If there is an error then end	*/
	/* the transaction as above, but don't worry about any error,	*/
	/* we return the error from the failure to get the identity.	*/
	if (!getLastIdentity())
	{
		SQLEndTran(SQL_HANDLE_DBC, m_pServer->connectionHandle(), SQL_ROLLBACK);
		return false ;
	}	

	db2cliRC = SQLEndTran(SQL_HANDLE_DBC, m_pServer->connectionHandle(), SQL_COMMIT);
	if(!m_pServer->checkRCOK(m_pServer->connectionHandle(), db2cliRC, "Error enabling autocommit"))
	{
		m_lError = m_pServer->lastError();
		return false;
	}

	db2cliRC = SQLRowCount(m_stmHandle, (SQLINTEGER*)&m_nRows) ;
	if(!m_pServer->checkRCOK(m_stmHandle, db2cliRC, "Error getting row count"))
	{
		m_lError = m_pServer->lastError();
		return false;
	}

	/* Return to autocommit enabled. We have to do this since this	*/
	/* operates at the connection level, not the handle level.	*/
	db2cliRC = SQLSetConnectAttr
		   (	m_pServer->connectionHandle(),
			SQL_ATTR_AUTOCOMMIT,
			(SQLPOINTER)SQL_AUTOCOMMIT_ON,
			SQL_NTS
		   )	;

	if(!m_pServer->checkRCOK(m_pServer->connectionHandle(), db2cliRC, "Error setting connection attribute"))
	{
		m_lError = m_pServer->lastError();
		return false;
	}
	
	return true;
}

/*  KBDB2CLIQryInsert							*/
/*  getLastIdentity							*/
/*		: Get last inserted identity value			*/
/*  (returns)	: bool		: Success				*/

bool	KBDB2CLIQryInsert::getLastIdentity ()
{
	SQLRETURN db2cliRC;

	struct
	{
		SQLINTEGER ind	;
		SQLINTEGER val	;
	}
	lastAssignedVal	;


	SQLCloseCursor(m_IdentityStmHandle) ;

	db2cliRC = SQLExecute(m_IdentityStmHandle);
	if(!m_pServer->checkRCOK(m_IdentityStmHandle, db2cliRC, "Error executing DB2 last assigned query"))
	{
		m_lError = m_pServer->lastError();
		return false ;
	}

	db2cliRC = SQLFetch  (m_IdentityStmHandle);
	if(!m_pServer->checkRCOK(m_IdentityStmHandle, db2cliRC, "Error fetching DB2 last assigned query"))
	{
		m_lError = m_pServer->lastError();
		return false ;
	}


	db2cliRC = SQLGetData
		   (	m_IdentityStmHandle,
			1,
			SQL_C_LONG,
			(SQLPOINTER)&lastAssignedVal.val, 
			sizeof(lastAssignedVal.val),
			&lastAssignedVal.ind
		   )	;

	if(!m_pServer->checkRCOK(m_IdentityStmHandle, db2cliRC, "Error getting DB2 last assigned value"))
	{
		m_lError = m_server->lastError();
		return	false ;
	}


	m_lastIdentity = KBValue((int)lastAssignedVal.val) ;
	return	true   ;
}

/*  KBDB2CLIQryInsert							*/
/*  getNewKey	: Get new primary key value				*/
/*  primary	: const QString & : Key column name			*/
/*  newKey	: KBValue &	  : New key value			*/
/*  prior	: bool		  : Pri-insert call			*/
/*  (returns)	: bool		  : Success				*/

bool	KBDB2CLIQryInsert::getNewKey
	(	const QString	&primary,
		KBValue 	&newKey,
		bool		prior
	)
{
	fprintf
	(	stderr,
		"KBDB2CLIQryInsert::getNewKey: [%s][%d]\n",
		(cchar *)primary,
		prior
	)	;

	/* Before we execute the query for the first time, we need to	*/
	/* see whether the primary key column is an INDENTITY column.	*/
	/* If it is then we get the Primary Key ahead of the Insert,	*/
	/* if not then we will retrieve it afterwards.			*/
	if (m_useIdentityCol < 0)
	{
		QStringList	identityCols	;
		if (!m_pServer->getIdentityCols (m_tabName, identityCols))
		{
			m_lError = m_pServer->lastError() ;
			return	 false ;
		}

		m_useIdentityCol = false ;
		for (uint idx = 0 ; idx < identityCols.count() ; idx += 1)
			if (identityCols[idx].lower() == primary.lower())
			{	m_useIdentityCol = true ;
				break ;
			}

	}	

	if (prior)
	{
		fprintf
		(	stderr,
			"KBDB2CLIQryInsert::getNewKey: prior, m_hasIdentityCol: %d\n",
			m_useIdentityCol
		)	;

		if (m_useIdentityCol)
		{
			/* This is an error. We treat identity columns	*/
			/* as read-only, so we cannot provide a value	*/
			/* to insert.					*/
			m_lError = KBError
				   (	KBError::Error,
					TR("Error getting new unique key"),
					TR("GetNewKey called before insert on identity column"),
					__ERRLOCN
				   )	;
		}

		newKey = KBValue() ;
		return true ;
	}

	/* Called after the event. If we are using an identity column	*/
	/* then all is OK.						*/
	if (m_useIdentityCol)
	{
		newKey	= m_lastIdentity ;
		fprintf
		(	stderr,
			"KBDB2CLIQryInsert newKey [%s]\n",
			(cchar *)newKey.getRawText()
		)	;
		return	true	;
	}

	m_lError = KBError
		   (	KBError::Error,
			"Asking for insert key",
			QString ("%1, %2").arg(m_tabName).arg(primary),
			__ERRLOCN
		   )	;
	return	false	;
}

/*  KBDB2CLIQryDelete							*/
/*  KBDB2CLIQryDelete							*/
/*		: Constructor for delete query object			*/
/*  server	: KBDB2CLI *		: Server connection object	*/
/*  data	: bool			: Query for data		*/
/*  query	: const QString &	: Delete query			*/
/*  tabName	: const QString &	: Table being updated		*/
/*  codec	: QTextCodec *		: Non-default codec		*/
/*  (returns)	: KBDB2CLIQryDelete	:				*/

KBDB2CLIQryDelete::KBDB2CLIQryDelete
	(	KBDB2CLI	*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLDelete	(server, data, query, tabName),
	m_pServer	(server)
{
	m_nRows = 0;

	// Enable transaction processing by switching AUTOCOMMIT OFF

	if(!m_pServer->getStatement(m_stmHandle))
		return;

	QCString  qryText  = m_rawQuery.utf8() ;
	SQLRETURN db2cliRC = SQLPrepare (m_stmHandle, (uchar *)qryText.data(), qryText.length()) ;

	if(!m_pServer->checkRCOK(m_stmHandle, db2cliRC, "Error preparing statement from DB2"))
	{
		SQLFreeHandle(SQL_HANDLE_STMT, m_stmHandle);
		m_stmHandle = 0;
		m_lError = m_pServer->lastError();
		return;
	}

	fprintf(stderr, "DB2: [%s]\n", (cchar *)query);
}

/*  KBDB2CLIQryDelete							*/
/*  execute	: Execute the query					*/
/*  nvals	: uint		: Number of substitution values		*/
/*  value	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success							*/

bool KBDB2CLIQryDelete::execute(uint nvals, const KBValue *values)
{
	SQLRETURN db2cliRC;

	if (m_stmHandle == 0)
		return false ;

	SQLCloseCursor(m_stmHandle);

	QList<KBODBCValue> valueList;
	valueList.setAutoDelete(true);

	fprintf
	(	stderr,
		"KBDB2CLIQryDelete::execute: [%s] [%d]\n",
		(cchar *)m_rawQuery,
		nvals

	)	;

	if(!m_pServer->bindParameters(m_stmHandle, nvals, values, valueList, m_codec))
	{
		m_lError = m_pServer->lastError();
		return false;
	}

	db2cliRC = SQLExecute(m_stmHandle);
	m_pServer->printQuery(m_rawQuery, nvals, values, db2cliOK(db2cliRC)) ;

	if(!m_pServer->checkRCOK(m_stmHandle, db2cliRC, "Error executing DB2 delete query"))
	{
		m_lError = m_pServer->lastError();
		return false;
	}


	db2cliRC = SQLRowCount(m_stmHandle, (SQLINTEGER*)&m_nRows);

	if(!m_pServer->checkRCOK(m_stmHandle, db2cliRC, "Error getting row count"))
	{
		m_lError = m_pServer->lastError();
		return false;
	}

	return true;
}

/*  KBDB2CLIQryDelete											*/
/*  ~KBDB2CLIQryDelete	: Destructor for delete query object	*/
/*  (returns)	:		:										*/

KBDB2CLIQryDelete::~KBDB2CLIQryDelete ()
{
	if(m_stmHandle != 0)
		SQLFreeHandle(SQL_HANDLE_STMT, m_stmHandle);
}

/*  ------------------------------------------------------------------  */

/*  KBDB2CLIQryCursor							*/
/*  KBDB2CLIQryCursor							*/
/*		: Constructor for cursor base class			*/
/*  server	: KBServer *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Cursor text				*/
/*  cursor	: const QString & : Cursor name				*/
/*  (returns)	: KBSQLCursor	  :					*/

KBDB2CLIQryCursor::KBDB2CLIQryCursor
	(	KBDB2CLI	*server,
		bool		data,
		const QString	&query,
		const QString	&cursor
	)
	:
	KBSQLCursor	(server, data, query, cursor),
	KBSQLTypeInfo	(server, m_types),
	m_pServer	(server)
{
	fprintf
	(	stderr,
		"KBDB2CLIQryCursor::KBDB2CLIQryCursor: [%s][%s]\n",
		(cchar *)query,
		(cchar *)cursor
	)	;

	if (!m_pServer->getStatement(m_stmHandle))
		return	;

	QCString  qryText  = m_rawQuery.utf8();
	SQLRETURN db2cliRC = SQLPrepare (m_stmHandle, (uchar *)qryText.data(), qryText.length()) ;

	if (!m_pServer->checkRCOK
		(	m_stmHandle,
			db2cliRC,
			"Error preparing DB2 cursor"
		))
	{
		SQLFreeHandle(SQL_HANDLE_STMT, m_stmHandle);
		m_stmHandle = 0;
		m_lError = m_pServer->lastError();
		return	;
	}

	/* Associate the cursor name with the statement. Experiment	*/
	/* shows that we need to do this before the statement is ever	*/
	/* executed.							*/
	QCString  name = m_cursor.utf8();
	db2cliRC = SQLSetCursorName (m_stmHandle, (uchar *)name.data(), SQL_NTS) ;
	if (!m_pServer->checkRCOK
		(	m_stmHandle,
			db2cliRC,
			"Error setting cursor name"
		))
	{
		SQLFreeHandle(SQL_HANDLE_STMT, m_stmHandle);
		m_stmHandle = 0;
		return	;
	}
}

/*  KBDB2CLIQryCursor							*/
/*  ~KBDB2CLIQryCursor: Destructor for cursor base class		*/

KBDB2CLIQryCursor::~KBDB2CLIQryCursor ()
{
	if (m_stmHandle != 0)
		SQLFreeHandle (SQL_HANDLE_STMT, m_stmHandle) ;
}

/*  KBDB2CLIQryCursor							*/
/*  execute	: Open the cursor					*/
/*  nvals	: uint		  : Number of substitution values	*/
/*  values	: KBValue *	  : Substitution values			*/
/*  (returns)	: bool		  : Success				*/

bool	KBDB2CLIQryCursor::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	if (m_stmHandle == 0)
		return	false	;

	close	()	;

	QList<KBODBCValue>   valueList	;
	valueList.setAutoDelete (true)	;
	SQLCloseCursor(m_stmHandle)	;

	if (!m_pServer->bindParameters
		(	m_stmHandle,
			nvals,
			values,
			valueList,
			m_codec
		))
	{
		m_lError = m_pServer->lastError();
		return false;
	}

	SQLRETURN db2cliRC = SQLExecute (m_stmHandle) ;
	if (!m_pServer->checkRCOK
		(	m_stmHandle,
			db2cliRC,
			"Error executing cursor query"
		))
	{
		m_lError = m_pServer->lastError() ;
		return	false	;
	}

	return	true	;
}


/*  KBDB2CLIQryCursor							*/
/*  fetch	: Fetch a row from the cursor				*/
/*  nvals	: uint		: Number of result values		*/
/*  values	: KBValue *	: Result values				*/
/*  got		: bool		: Set true if a row is forthcoming	*/
/*  (returns)	: bool		: Success				*/

bool	KBDB2CLIQryCursor::fetch
	(	uint		nvals,
		KBValue		*values,
		bool		&got
	)
{
	if (m_stmHandle == 0)
		return	false	;

	SQLRETURN db2cliRC = SQLFetch (m_stmHandle) ;
	if (db2cliRC == SQL_NO_DATA_FOUND)
	{
		got	= false	;
		return	true	;
	}

	if (!m_pServer->checkRCOK
		(	m_stmHandle,
			db2cliRC,
			"Error executing cursor fetch"
		))
	{
		m_lError = m_pServer->lastError() ;
		return	false	;
	}

	/* First time round, get the field types. This code is just a	*/
	/* copy of that ion KBDB2CLIQrySelect::execute() and shoule be	*/
	/* pulled out as a separate routine.				*/
	if (m_nFields == 0)
	{
		SQLSMALLINT nFields	;
		SQLNumResultCols (m_stmHandle, &nFields) ;
		m_nFields = nFields	;

		m_types = new KBType*[m_nFields];

		if (!loadTypeInfo
			(	m_stmHandle,
				m_nFields,
				m_lError
			))
		{
			return	false	;
		}
	}

	if (!fetchRowValues
		(	m_pServer,
			m_codec,
			m_stmHandle,
			nvals > m_nFields ? m_nFields : nvals,
			values,
			m_types,
			m_cTypes,
			m_dbTypes,
			m_lError
		))
		return	false	;

	for (uint col = m_nFields ; col < nvals ; col += 1)
		values[col] = KBValue() ;

	return	true	;
}

/*  KBDB2CLIQryCursor							*/
/*  close	: Close the cursor					*/
/*  (returns)	: bool		: Success				*/

bool	KBDB2CLIQryCursor::close ()
{
	m_dbTypes .clear () ;
	m_cTypes  .clear () ;
	m_colNames.clear () ;

	if (m_types != 0)
	{
		for (uint idx = 0 ; idx < m_nFields ; idx += 1)
			m_types[idx]->deref() ;

		delete	[] m_types ;
		m_types	= 0 ;
	}

	return	true	;
}

#ifndef _WIN32

KBFACTORY
(	KBDB2CLIFactory,
	"driver_db2cli"
)

KBFACTORYIMPL
(	KBDB2CLIFactory,
	driver_db2cli,
	"Rekall DB2CLI driver",
	"Plugin",
	"0",
	"7",
	"0"
)

#else

class KBDB2CLIFactory : public KBFactory
{
public:

	inline KBDB2CLIFactory() : KBFactory()
	{
	}

	virtual QObject* create(QObject* = 0, const char* = 0, const char* = 0, const QStringList& = QStringList());
	virtual const char* ident();
};

extern "C" __declspec(dllexport) void* init_libkbase_driver_db2cli()
{
	return new KBDB2CLIFactory;
}

#endif

QObject	*KBDB2CLIFactory::create
	(	QObject			*parent,
		cchar			*object,
		cchar			*,
		const QStringList 	&
	)
{
	if (dIdentToType.count() == 0)
	{
		for(uint idx = 0 ; idx < sizeof(typeMap)/sizeof(DB2CLITypeMap) ; idx++)
		{
			DB2CLITypeMap *m = &typeMap[idx];
			dIdentToType.insert(m->symbolicType, m);
		}
	}

	if ((parent != 0) && !parent->inherits("QWidget"))
	{
		fprintf	(stderr, "KBDB2CLIFactory: parent does not inherit QWidget\n");
		return	0 ;
	}
	
	if (strcmp (object,   "driver") == 0)
		return	new KBDB2CLI() ;

	if (strcmp (object, "advanced") == 0)
		return	0 ;

	return	0 ;
}

cchar *KBDB2CLIFactory::ident()
{
	return	__KB_BUILD_IDENT;
}
