/***************************************************************************
    file	         : kb_unisql.cpp
    copyright            : (C) 1999,2000,2001 by Mike Richardson
			   (C) 2000,2001 by theKompany.com
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *     This program is licensed under the terms contained in the file      *
 *     LICENSE which is contained with the source code distribution.       *
 *                                                                         *
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<time.h>
#include	<errno.h>
#include	<sys/types.h>

#ifdef 		_WIN32
#include 	<windows.h>
#endif

#include	<qlayout.h>

#ifndef		_WIN32
#include	"kb_unisql.moc"
#else
#include	"kb_unisql.h"
#endif

#include	"kb_unisqladv.h"
#include	"kb_libloader.h"


static	UniSQLTypeMap	rkl_typeMap[] =
{
/*	ident			itype		mtype		flags			*/
{	"1",			KB::ITBool,	"Bool",		0,			},
{	"2",			KB::ITFixed,	"Integer",	0			},
{	"3",			KB::ITFixed,	"Integer",	0			},
{	"4",			KB::ITFixed,	"Integer",	0			},
{	"5",			KB::ITFloat,	"Float",	FF_LENGTH|FF_PREC	},
{	"7",			KB::ITFloat,	"Float",	FF_LENGTH|FF_PREC	},

{	"8",		 	KB::ITDateTime,	"DateTime",	0			},

{	"10",			KB::ITString,	"Text",		FF_LENGTH		},
{	"12",			KB::ITString,	"Memo",		FF_LENGTH		},

{	"9",			KB::ITBinary,	"Binary",	FF_LENGTH		},
{	"11",			KB::ITBinary,	"LongBinary",	0			},
{	0,			KB::ITUnknown,	"",		0			}
}	;

static	UniSQLTypeMap	sys_typeMap[] =
{
/*	ident			itype		mtype		flags			*/
{	"boolean",		KB::ITBool,	"Bool",		0,			},
{	"i2",			KB::ITFixed,	"Integer",	FF_NOCREATE		},
{	"int",			KB::ITFixed,	"Integer",	0			},
{	"number",		KB::ITFloat,	"Float",	FF_LENGTH|FF_PREC
									 |FF_NOCREATE	},
{	"float",		KB::ITFloat,	"Float",	FF_LENGTH|FF_PREC	},
{	"dateTime",	 	KB::ITDateTime,	"DateTime",	0			},
{	"string",		KB::ITString,	"Text",		FF_LENGTH		},
{	"bin.hex",		KB::ITBinary,	"Binary",	FF_LENGTH		},
{	0,			KB::ITUnknown,	"",		0			}
}	;



/*  ------------------------------------------------------------------  */

/*  KBUniSQLType								*/
/*  KBUniSQLType	: Constructor for UniSQL type object			*/
/*  typeInfo	: UniSQLTypeMap * : Type information			*/
/*  length	: uint		 : Underlying database length		*/
/*  prec	: uint		 : Underlying database precision	*/
/*  nullOK	: bool		 : True if null is OK			*/
/*  (returns)	: KBUniSQLType	 :					*/

KBUniSQLType::KBUniSQLType
	(	UniSQLTypeMap	*typeInfo,
		uint	  	length,
		uint		prec,
		bool	  	nullOK
	)
	:
	KBType
	(	"UniSQL",
		typeInfo == 0 ? KB::ITUnknown : typeInfo->itype,
		length,
		prec,
		nullOK
	),
	m_typeInfo	(typeInfo)
{
}

/*  KBUniSQLType								*/
/*  isValid	: Test if value is valid for type			*/
/*  value	: const QString & : Value to check			*/
/*  pError	: KBValue &	  : Error return			*/
/*  where	: const QString & : Caller				*/
/*  (returns)	: bool		  : Valid				*/

bool	KBUniSQLType::isValid
	(	const QString	 &value,
		KBError		 &pError,
		const QString	&where
	)
{
	/* *** Specific type checking					*/
	return	KBType::isValid (value, pError, where) ;
}

/*  KBUniSQLType								*/
/*  getQueryText							*/
/*  value	: KBDataArray  * : Raw text of value to convert		*/
/*  d		: KBShared     * : Decoded representation		*/
/*  buffer	: KBDataBuffer & : Results buffer			*/
/*  (returns)	: void		 :					*/

void	KBUniSQLType::getQueryText
	(	KBDataArray	*value,
		KBShared	*d,
		KBDataBuffer	&buffer
	)
{
	/* **** Stuff like binary escaping				*/
	KBType::getQueryText (value, d, buffer) ;
}





/*  ------------------------------------------------------------------  */

/*  KBUniSQL								*/
/*  KBUniSQL	: Constructor for UniSQL database connection class	*/
/*  (returns)	: KBServer	:					*/

KBUniSQL::KBUniSQL ()
	:
	KBServer ()
{
	/* Assume that the rekall information table is not available;	*/
	/* this will be checked when the connection is made.		*/
	m_rekallTables	= false ;

	m_connected	= false	;

//	m_mapCRLF	= false ;
//	m_showSysTables	= false ;
	m_mapExpressions= false ;
}

/*  KBUniSQL								*/
/*  ~KBUniSQL	: Destructor for UniSQL database connection class	*/
/*  (returns)	:		:					*/

KBUniSQL::~KBUniSQL ()
{
}

bool	KBUniSQL::makeConnection
	(	bool		force,
		KBError		&pError
	)
{
	if (!m_connected && !force)
	{
		pError	= KBError
			  (	KBError::Error,
				QString(TR("No connection to UniverSQL server %1"))
					.arg(m_name),
				QString::null,
				__ERRLOCN
			  )	;
		return	false	;
	}

	KBUniSQLConnect	dConnect (m_connection, m_host, m_port) ;
	if (!dConnect.exec ())
	{
		pError	= KBError
			  (	KBError::Error,
				QString(TR("Failed to connect to UniverSQL server %1"))
					.arg(m_name),
				QString::null,
				__ERRLOCN
			  )	;
		return	false	;
	}

	m_connected = true	;
	return	true		;
}

/*  KBUniSQL								*/
/*  doConnect	: Open connection to database				*/
/*  svInfo	: KBServerInfo *: Server information			*/
/*  (returns)	: bool		: Success				*/

bool	KBUniSQL::doConnect
	(	KBServerInfo	*svInfo
	)
{
	if (svInfo->advanced() != 0)
		if (svInfo->advanced()->isType("unisql"))
		{
			KBUniSQLAdvanced *a = (KBUniSQLAdvanced *)svInfo->advanced() ;
			fprintf
			(	stderr,
				"KBUniSQL::doConnect: advanced=%p map=%d\n",
				(void *)a,
				a->m_mapExpressions
			)	;
//			m_mapCRLF	  = a->m_mapCRLF	;
//			m_showSysTables   = a->m_showSysTables  ;
			m_mapExpressions  = a->m_mapExpressions	;
//			m_odbcType	  = a->m_odbcType	;
		}
		else	/* If the type is wrong then show a warning.	*/
			/* This should never happen unless someone	*/
			/* hand edits the XML database file.		*/
			KBError::EError
			(	TR("Driver error"),
				TR("Invalid advanced options, ignoring"),
				__ERRLOCN
			)	;

	m_readOnly	= svInfo->readOnly  ()		;
	m_name		= svInfo->serverName()		;
	m_host		= svInfo->hostName  ()		;
	m_port		= svInfo->portNumber().toInt()	;

#ifndef	_WIN32
	/* The driver supports SSH tunneling. If there is a tunnel	*/
	/* target then attempt to open the tunnel; if this succeeds	*/
	/* then the host becomes the local host and the port is that	*/
	/* returned by the tunneler.					*/
	fprintf
	(	stderr,
		"KBPgSQL::doConnect: sshTarget=[%s]\n",
		(cchar *)m_sshTarget
	)	;

	if (!m_sshTarget.isEmpty())
	{
		int local = openSSHTunnel(6928) ;
		if (local < 0) return false ;

		m_host	= "127.0.0.1"	;
		m_port	= local		;
	}
#endif

	if (m_port == 0) m_port = 6928 ;

	if (!makeConnection (true, m_lError))
		return	false	;

	m_dsn	= QString("DSN=%1").arg(svInfo->database()) ;

	if (!svInfo->userName().isEmpty())
		m_dsn	+= ";UID=" + svInfo->userName() ;
	if (!svInfo->password().isEmpty())
		m_dsn	+= ";PWD=" + svInfo->password() ;


	QString		dummy	;
	KBUniSQLParser	*parser	;

	parser	= execSQL
		  (	"select TableName from __RekallTables where 0 = 1",
			dummy,
			0,
			0,
			0,
			"",
			m_lError
	   	  )	;

	if (parser != 0)
	{
		fprintf
		(	stderr,
			"KBUniSQL::doConnect: using rekall tables\n"
		)	;

		m_rekallTables	= true ;
		delete	parser	;
	}

	if (m_rekallTables)
		m_typeMap = rkl_typeMap ;
	else	m_typeMap = sys_typeMap ;

	m_dIdentToType.clear () ;

	for (uint idx = 0 ; m_typeMap[idx].mtype[0] != 0 ; idx += 1)
	{
		UniSQLTypeMap *m = &m_typeMap[idx] ;
		m_dIdentToType.insert (m->ident, m) ;
	}

	return	true	;
}

QString	KBUniSQL::mapExpression
	(	const QString	&expr
	)
{
	static	QString	spec ("_") ;

	return	m_mapExpressions ?
			doMapExpression (expr, "[", "]", spec) :
			expr ;
}

/*  KBUniSQL								*/
/*  execSQL	: Execute SQL statement					*/
/*  rawSql	: const QString &  : Statement with placeholders	*/
/*  subSql	: QString &	   : Substituted statement for logging	*/
/*  nvals	: uint		   : Number of substitution values	*/
/*  values	: KBValue *	   : Substitution values		*/
/*  codec	: QTextCodec *	   : Non-default codec			*/
/*  emsg	: cchar *	   : Error text on error		*/
/*  pError	: KBError &	   : Error return			*/
/*  identity	: bool		   : Add "select @@IDENTITY"		*/
/*  (returns)	: KBUniSQLParser * : Parser with results or null	*/

KBUniSQLParser
	*KBUniSQL::execSQL
	(	const QString	&rawSql,
		QString		&subSql,
		uint		nvals,
		const KBValue	*values,
		QTextCodec	*codec,
		cchar		*,
		KBError		&pError,
		bool		identity
	)
{
	KBDataBuffer	exeSql	;
	bool	rc	= true	;

	if (!subPlaceList (rawSql, nvals, values, exeSql, codec, pError))
		return 0	;

	subSql	= subPlaceList (rawSql, nvals, values, pError) ;
	if (subSql == QString::null)
		return 0	;

	if (m_connection.state() == QSocket::Idle)
		if (!makeConnection (false, pError))
			return	0 ;

	QString	request	 =
		"<?xml version=\"1.0\"?>"
		"<request><connectionstring>" + m_dsn  + "</connectionstring>"
		"<sql><![CDATA[" 	      + subSql +"]]></sql>"
		;

	/* Hack to retrieve insert identities. Has to be done this way	*/
	/* since it does not work as a separate request, presumably	*/
	/* the client disconnects or something.				*/
	if (identity)
		request	+= "<sql>select @@IDENTITY</sql>" ;

	request	+= "</request> " ;

	cchar	*raw	= (cchar *)request ;

	fprintf
	(	stderr,
		"[[[ %s ]]]\n",
		(cchar *)subSql
	)	;

	if (m_connection.writeBlock (raw, qstrlen(raw)) < 0)
	{
		pError	= KBError
			  (	KBError::Error,
			   	TR("Error sending command to UniverSQL server"),
				QString("Error code: %1").arg(errno),
				__ERRLOCN
			  )	;
		return	0	;
	}

	m_connection.flush ()	;

	long		bytes	;
	KBDataBuffer	buff	;
	bool		gotall	= false	;
	bool		error	= false	;

	while (!error && ((bytes = m_connection.waitForMore (10000)) > 0))
	{
		// fprintf (stderr, "bytes -> %d\n", bytes) ;

		while (bytes > 0)
		{
			char	b[4096] ;
			long	avail	= bytes	> (long)sizeof(b) ? sizeof(b) : bytes ;
			long	got	= m_connection.readBlock (b, avail) ;

			if (got < 0)
			{
				error	= true	;
				break	;
			}

			// fprintf (stderr, "got   -> %d\n", got) ;
			buff.append (b, got) ;
			bytes	-= got ;
		}

		if (strstr (buff.data(), "</xml>") != 0)
		{	gotall	= true	;
			break	;
		}
	}

	if (error || !gotall)
	{
		pError	= KBError
			  (	KBError::Error,
			   	TR("Error receiving reply from UniverSQL server"),
				QString::null,
				__ERRLOCN
			  )	;
		return	0	;
	}

	KBUniSQLParser	*parser	= new KBUniSQLParser ;

	FILE *log = fopen ("/tmp/unisql.res", "w") ;
	fprintf (log, "%s\n", buff.data()) ;
	fclose	(log) ;

	if (!parser->parseText (buff.array()))
	{
		pError	= KBError
			  (	KBError::Error,
			   	TR("Error parsing reply from UniverSQL server"),
				QString::null,
				__ERRLOCN
			  )	;
		delete	parser	;
		return	0	;
	}

	QString		errMsg	;
	if (parser->error (errMsg))
	{
		pError	= KBError
			  (	KBError::Error,
			   	TR("UniverSQL server error"),
				errMsg,
				__ERRLOCN
			  )	;
		delete	parser	;
		return	0	;
	}

	printQuery (rawSql, nvals, values, rc) ;
	return	parser	;
}

/*  KBUniSQL								*/
/*  qrySelect	: Open a select query					*/
/*  data	: bool		 : Query for data			*/
/*  select	: const QString& : Select query				*/
/*  update	: bool		 : Select for update			*/
/*  (returns)	: KBSQLSelect *  : Select query class or NULL on error	*/

KBSQLSelect
	*KBUniSQL::qrySelect
	(	bool		data,
		const QString	&select,
		bool
	)
{
	return	new KBUniSQLQrySelect (this, data, select) ;
}


/*  KBUniSQL								*/
/*  qryUpdate	: Open an update query					*/
/*  data	: bool		  : Query for data			*/
/*  update	: const QString & : Update query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  (returns)	: KNQryUpdate *   : Update query class or NULL on error	*/

KBSQLUpdate
	*KBUniSQL::qryUpdate
	(	bool		data,
		const QString	&update,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting update query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	new KBUniSQLQryUpdate (this, data, update, tabName) ;
}

/*  KBUniSQL								*/
/*  qryInsert	: Open an insert query					*/
/*  data	: bool		  : Query for data			*/
/*  insert	: const QString & : Insert query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  (returns)	: KNQryInsert *   : Insert query class or NULL on error	*/

KBSQLInsert
	*KBUniSQL::qryInsert
	(	bool		data,
		const QString	&insert,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting insert query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	new KBUniSQLQryInsert (this, data, insert, tabName) ;
}

/*  KBUniSQL								*/
/*  qryDelete	: Open an delete query					*/
/*  data	: bool		  : Query for data			*/
/*  _delete	: const QString & : Delete query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  (returns)	: KNQryDelete *   : Delete query class or NULL on error	*/

KBSQLDelete
	*KBUniSQL::qryDelete
	(	bool		data,
		const QString	&_delete,
		const QString	&tabName
	)
{
	if (m_readOnly)
	{
		m_lError = KBError
			   (	KBError::Error,
			 	TR("Database is read-only"),
			 	TR("Attempting delete query"),
			 	__ERRLOCN
			  )	;
		return	 0 ;
	}

	return	new KBUniSQLQryDelete (this, data, _delete, tabName) ;
}

/*  KBUniSQL								*/
/*  command	: Execute arbitrary SQL					*/
/*  data	: bool		  : Querying for data			*/
/*  query	: const QString & : Query text				*/
/*  nvals	: uint		  : Number of substitution values	*/
/*  values	: KBValue *	  : Substitution values			*/
/*  (returns)	: bool		  : Success				*/

bool	KBUniSQL::command
	(	bool		data,
		const QString	&rawqry,
		uint		nvals,
		KBValue		*values,
		KBSQLSelect	**select
	)
{
	QString		subqry	;
	KBUniSQLParser	*parser	;

	parser	= execSQL
		  (	rawqry,
			subqry,
			nvals,
			values,
			getCodec (data),
			"Query failed",
			m_lError
		  )	;
	if (parser == 0) return false ;

	if (select == 0)
	{
		delete	parser	;
		return	true	;
	}

	if (parser->rows() == 0)
	{
		*select	= 0	;
		delete	parser	;
		return	true	;
	}

	*select	= new KBUniSQLQrySelect (this, data, rawqry, parser) ;
	return	true	;
}


/*  KBUniSQL								*/
/*  listTypes	: Get list of types with information flags		*/
/*  (returns)	: QString	: List as bar-separated string		*/

QString	KBUniSQL::listTypes ()
{
	static	QString	typeList ;

	if (typeList.isNull())
	{
		typeList = "Primary Key,0|Foreign Key,0" ;

		for (uint idx = 0 ; m_typeMap[idx].mtype[0] != 0 ; idx += 1)
		{
			UniSQLTypeMap *m = &m_typeMap[idx] ;

			if ((m->flags & FF_NOCREATE) == 0)
				typeList += QString("|%1,%2").arg(m->mtype).arg(m->flags) ;
		}
	}

	return	typeList ;
}

/*  KBUniSQL								*/
/*  tableExists	: See if named table exists				*/
/*  table	: const QString & : Table name				*/
/*  exists	: bool &	  : True if table exists		*/
/*  (returns)	: bool		  : Success				*/

bool	KBUniSQL::tableExists
	(	const QString	&,
		bool		&exists
	)
{
	exists = false	; 
	return	 true	;
}

/*  KBUniSQL								*/
/*  doListTablesSys							*/
/*		: List tables in database using system tables		*/
/*  tabList	: KBTableDetailsList &					*/
/*				: Result list				*/
/*  alltables	: bool		: List all tables			*/
/*  type	: uint		: Get only this type			*/
/*  (returns)	: bool		: Success				*/

bool	KBUniSQL::doListTablesSys
	(	KBTableDetailsList		&tabList,
		bool				,
		uint	
	)
{
	QString		dummy	;
	KBUniSQLParser	*parser	;

	/* This version access the MSysObjects table to get all table	*/
	/* names.							*/
	parser	= execSQL
		  (	"select Name from MSysObjects where type = 1",
			dummy,
			0,
			0,
			0,
			"Error retrieving list of tables",
			m_lError
		  )	;

	if (parser == 0)
		return	false	;

	for (uint idx = 0 ; idx < parser->rows() ; idx += 1)
		tabList.append
		(	KBTableDetails
			(	parser->value (idx, 0),
				KB::IsTable,
				QP_SELECT|QP_INSERT|QP_UPDATE|QP_DELETE
		)	)	;

	delete	parser	;
	return	true	;
}

/*  KBUniSQL								*/
/*  doListTablesRkl							*/
/*		: List tables in database using rekall table		*/
/*  tabList	: KBTableDetailsList &					*/
/*				: Result list				*/
/*  alltables	: bool		: List all tables			*/
/*  type	: uint		: Get only this type			*/
/*  (returns)	: bool		: Success				*/

bool	KBUniSQL::doListTablesRkl
	(	KBTableDetailsList		&tabList,
		bool				,
		uint	
	)
{
	QString		dummy	;
	KBUniSQLParser	*parser	;

	/* Other version, gets the names from the pre-prepared		*/
	/* __RekallTables in the database.				*/
	parser	= execSQL
		  (	"select distinct TableName from __RekallTables",
			dummy,
			0,
			0,
			0,
			"Error retrieving list of tables",
			m_lError
		  )	;

	if (parser == 0)
		return	false	;

	for (uint idx = 0 ; idx < parser->rows() ; idx += 1)
		tabList.append
		(	KBTableDetails
			(	parser->value (idx, 0),
				KB::IsTable,
				QP_SELECT|QP_INSERT|QP_UPDATE|QP_DELETE
		)	)	;

	delete	parser	;
	return	true	;
}

/*  KBUniSQL								*/
/*  doListTables: List tables in database				*/
/*  tabList	: KBTableDetailsList &					*/
/*				: Result list				*/
/*  type	: uint		: Get only this type			*/
/*  (returns)	: bool		: Success				*/

bool	KBUniSQL::doListTables
	(	KBTableDetailsList	&tabList,
		uint			type
	)
{
	return	m_rekallTables ?
			doListTablesRkl (tabList, m_showAllTables, type) :
			doListTablesSys (tabList, m_showAllTables, type) ;
}

/*  KBUniSQL								*/
/*  doListFieldsSys							*/
/*		: List fields in table, default method			*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  (returns)	: bool		: Success				*/

bool	KBUniSQL::doListFieldsSys
	(	KBTableSpec	&tabSpec
	)
{
	QString		dummy	;
	KBUniSQLParser	*parser	;

	tabSpec.m_prefKey   = -1    ;
	tabSpec.m_keepsCase = false ;

	/* Jet SQL does not seem to have a "describe table" function,	*/
	/* so do the best we can with a dodgy select.			*/
	parser	= execSQL
		  (	"select * from [" + tabSpec.m_name + "] where 0 = 1",
			dummy,
			0,
			0,
			0,
			"Error retrieving list of columns",
			m_lError
		  )	;

	if (parser == 0)
		return	false	;

	uint	nCols	= parser->columns() ;
	for (uint col = 0 ; col < nCols ; col += 1)
	{
		QString		name	= parser->name  (col) ;
		QString		type	= parser->type  (col) ;
		int		len	= parser->length(col) ;
		uint		flags	= 0 ;

		UniSQLTypeMap	*m	= m_dIdentToType.find(type) ;

		QString		mtype	;
		KB::IType	itype	;

		if (m != 0)
		{
			mtype	= m->mtype	;
			itype	= m->itype	;
		}
		else
		{
			mtype	= QString("<Unknown %1>").arg(type) ;
			itype	= KB::ITUnknown	;
		}

		if (!parser->nullable(col)) flags |= KBFieldSpec::NotNull  ;
		if ( parser->autoinc (col)) flags |= KBFieldSpec::InsAvail |
						     KBFieldSpec::Serial   |
						     KBFieldSpec::ReadOnly ;
		if ( parser->keycol  (col)) flags |= KBFieldSpec::Primary  |
						     KBFieldSpec::Indexed  |
						     KBFieldSpec::Unique   ;

		tabSpec.m_fldList.append
			(	new KBFieldSpec
				(	col,
					name,
					mtype,
					itype,
					flags,
					len,
					0
			)	)	;

		if (parser->autoinc(col) && parser->keycol(col))
			tabSpec.m_prefKey = col ;
	}

	delete	parser	;
	return	true	;
}

/*  KBUniSQL								*/
/*  doListFieldsRlk							*/
/*		: List fields in table, using rekall table		*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  (returns)	: bool		: Success				*/

bool	KBUniSQL::doListFieldsRkl
	(	KBTableSpec	&tabSpec
	)
{
	QString		dummy	;
	KBUniSQLParser	*parser	;
	KBValue		tabName	(tabSpec.m_name) ;

	tabSpec.m_prefKey   = -1    ;
	tabSpec.m_keepsCase = false ;


	/* Interrogate the pre-prepared table on the server to get	*/
	/* information about the table.					*/
	parser	= execSQL
		  (	"select FieldName, FieldCode, FieldSize,  "
			"	FieldAttr, FieldIndx, FieldReqd,  "
			"	FieldDflt			  "
			"from	__RekallTables			  "
			"where	TableName = ?			  ",
			dummy,
			1,
			&tabName,
			0,
			"Error retrieving list of columns",
			m_lError
		  )	;

	if (parser == 0)
		return	false	;


	for (uint idx = 0 ; idx < parser->rows() ; idx += 1)
	{
		QString	fldName	= parser->value(idx, 0) ;
		QString	fldCode	= parser->value(idx, 1) ;
		int	fldSize	= parser->value(idx, 2).toInt() ;
		int	fldAttr	= parser->value(idx, 3).toInt() ;
		int	fldIndx	= parser->value(idx, 4).toInt() ;
		bool	fldReqd	= parser->value(idx, 5).toInt() != 0 ;
		QString	fldDefv	= parser->value(idx, 6) ;

		UniSQLTypeMap	*m	= m_dIdentToType.find(fldCode) ;

		QString		mtype	;
		KB::IType	itype	;
		uint		flags	= 0	;

		if (m != 0)
		{
			mtype	= m->mtype	;
			itype	= m->itype	;
		}
		else
		{
			mtype	= QString("<Unknown %1>").arg(fldCode) ;
			itype	= KB::ITUnknown	;
		}

		if (fldIndx == 1)
			flags	|= KBFieldSpec::Indexed	 ;
		if (fldIndx == 2)
			flags	|= KBFieldSpec::Primary	 | KBFieldSpec::Indexed ;
		if (fldReqd)
			flags	|= KBFieldSpec::NotNull	 ;
		if ((fldAttr & 16) != 0)
			flags	|= KBFieldSpec::InsAvail | KBFieldSpec::Serial | KBFieldSpec::ReadOnly ;

		const uint PrefFlags = KBFieldSpec::Primary|KBFieldSpec::InsAvail ;
		const uint PkeyFlags = KBFieldSpec::Primary|KBFieldSpec::InsAvail|KBFieldSpec::Serial ;

		if ((flags & PrefFlags) == PrefFlags)
		{
			tabSpec.m_prefKey = tabSpec.m_fldList.count() ;
		}
		if ((flags & PkeyFlags) == PkeyFlags)
		{
			mtype		  = "Primary Key" ;
		}

		if (fldDefv[0] == '=') fldDefv = fldDefv.mid(1) ;

		KBFieldSpec *fs = new KBFieldSpec
				  (	tabSpec.m_fldList.count(),
					fldName,
					mtype,
					itype,
					flags,
					fldSize,
					0
				  )	;
		fs->m_defval = fldDefv	;
		tabSpec.m_fldList.append (fs) ;
	}


	delete	parser	;
	return	true	;
}

/*  KBUniSQL								*/
/*  doListFields: List fields in table					*/
/*  tabSpec	: KBTableSpec &	: Table specification			*/
/*  (returns)	: bool		: Success				*/

bool	KBUniSQL::doListFields
	(	KBTableSpec	&tabSpec
	)
{
	return	m_rekallTables ?
			doListFieldsRkl (tabSpec) :
			doListFieldsSys (tabSpec) ;
}

/*  KBUniSQL								*/
/*  doCreateTable: Create a new table					*/
/*  tabSpec	 : KBTableSpec &: Table specification			*/
/*  assoc	 : bool		: Create associated stuff		*/
/*  best	 : bool		: Use best match			*/
/*  (returns)	 : bool		: Success				*/

bool	KBUniSQL::doCreateTable
	(	KBTableSpec	&,
		bool		,
		bool
	)
{
	m_lError = KBError
		   (	KBError::Error,
			"Not implemented",
			"create",
			__ERRLOCN
		   )	;
	return	false	;
}

/*  KBUniSQL								*/
/*  doRenameTable: Rename a  table					*/
/*  oldName	 : cchar *	: Current table name			*/
/*  newName	 : cchar *	: New table name			*/
/*  assoc	 : bool		: Rename associated stuff - none here	*/
/*  (returns)	 : bool		: Success				*/

bool	KBUniSQL::doRenameTable
	(	cchar	*,
		cchar	*,
		bool
	)
{
	m_lError = KBError
		   (	KBError::Error,
			"Not implemented",
			"rename",
			__ERRLOCN
		   )	;
	return	false	;
}

/*  KBUniSQL								*/
/*  doDropTable	: Drop a table						*/
/*  table	: cchar *	: Table name				*/
/*  assoc	: bool		: Drop associated stuff - none here	*/
/*  (returns)	: bool		: Success				*/

bool	KBUniSQL::doDropTable
	(	cchar	*,
		bool
	)
{
	m_lError = KBError
		   (	KBError::Error,
			"Not implemented",
			"drop",
			__ERRLOCN
		   )	;
	return	false	;
}

/*  ------------------------------------------------------------------- */

/*  KBUniSQLQrySelect							*/
/*  KBUniSQLQrySelect							*/
/*		: Constructor for select query object			*/
/*  server	: KBUniSQL *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Select query			*/
/*  (returns)	: KBUniSQLQrySelect:					*/

KBUniSQLQrySelect::KBUniSQLQrySelect
	(	KBUniSQL	*server,
		bool		data,
		const QString	&query
	)	
	:
	KBSQLSelect	(server, data, query),
	m_server	(server)
{
	m_nRows		= 0 ;
	m_nFields	= 0 ;
	m_parser	= 0 ;
}

/*  KBUniSQLQrySelect							*/
/*  KBUniSQLQrySelect							*/
/*		: Constructor for select query object			*/
/*  server	: KBUniSQL *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Select query			*/
/*  parser	: KBUniSQLParser *: Extant results parser		*/
/*  (returns)	: KBUniSQLQrySelect:					*/

KBUniSQLQrySelect::KBUniSQLQrySelect
	(	KBUniSQL	*server,
		bool		data,
		const QString	&query,
		KBUniSQLParser	*parser
	)	
	:
	KBSQLSelect	(server, data, query),
	m_server	(server)
{
	m_parser	= parser ;
	m_nRows		= m_parser->rows    () ;
	m_nFields	= m_parser->columns () ;

	m_fields.clear	() ;
	DELOBJ(m_types)	;
	m_types	= new KBType *[m_nFields] ;

	for (uint fld = 0 ; fld < m_nFields ; fld += 1)
	{
		const QString &type = m_parser->type(fld) ;

		if	(qstricmp (type, "int"    ) == 0) m_types[fld] = &_kbFixed  ;
		else if	(qstricmp (type, "float"  ) == 0) m_types[fld] = &_kbFloat  ;
		else if	(qstricmp (type, "number" ) == 0) m_types[fld] = &_kbFloat  ;
		else if (qstricmp (type, "boolean") == 0) m_types[fld] = &_kbBool   ;
		else					  m_types[fld] = &_kbString ;

		m_fields.append (m_parser->name(fld)) ;

		fprintf
		(	stderr,
			" ......[%.12s] [%s]\n",
			(cchar *)m_parser->name(fld),
			(cchar *)m_types[fld]->getDescrip()
		)	;
	}

}

/*  KBUniSQLQrySelect							*/
/*  ~KBUniSQLQrySelect							*/
/*		: Destructor for select query object			*/
/*  (returns)	:		:					*/

KBUniSQLQrySelect::~KBUniSQLQrySelect ()
{
	DELOBJ	(m_parser)	;
}

/*  KBUniSQLQrySelect							*/
/*  execute	: Execute query						*/
/*  nvals	: uint		: Number of substitution values		*/
/*  values	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBUniSQLQrySelect::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	DELOBJ	(m_parser)	;

	m_parser = m_server->execSQL
		   (	m_rawQuery,
			m_subQuery,
			nvals,
			values,
			m_codec,
			"Select query failed",
			m_lError
		   )	;

	if (m_parser == 0)
		return	false	;

	m_nRows		= m_parser->rows    () ;
	m_nFields	= m_parser->columns () ;

	m_fields.clear	() ;

	if (m_types == 0)
	{
	m_types	= new KBType *[m_nFields] ;

	for (uint fld = 0 ; fld < m_nFields ; fld += 1)
	{
		const QString &type = m_parser->type(fld) ;

		if	(qstricmp (type, "int"    ) == 0) m_types[fld] = &_kbFixed  ;
		else if	(qstricmp (type, "float"  ) == 0) m_types[fld] = &_kbFloat  ;
		else if	(qstricmp (type, "number" ) == 0) m_types[fld] = &_kbFloat  ;
		else if (qstricmp (type, "boolean") == 0) m_types[fld] = &_kbBool   ;
		else					  m_types[fld] = &_kbString ;

		m_fields.append (m_parser->name(fld)) ;

		fprintf
		(	stderr,
			" ......[%.12s] [%s]\n",
			(cchar *)m_parser->name(fld),
			(cchar *)m_types[fld]->getDescrip()
		)	;
	}
	}

	return	true	;
}

/*  KBUniSQLQrySelect							*/
/*  getField	: Get a specified field from the query results		*/
/*  qrow	: uint		  : Row number				*/
/*  qcol	: uint		  : Column number			*/
/*  vtrans	: KBValue::VTrans : Translation mode			*/
/*  (returns)	: KBValue	  : Value				*/

KBValue	KBUniSQLQrySelect::getField
	(	uint		qrow,
		uint		qcol,
		KBValue::VTrans
	)
{
	if (m_parser          == 0) return KBValue () ;
	if ((int)qrow >= m_nRows  ) return KBValue () ;
	if (qcol      >= m_nFields) return KBValue () ;

	QString	text	= m_parser->value(qrow, qcol) ;
	if (text.isNull()) return KBValue (m_types[qcol]) ;

	return	KBValue	(text, m_types[qcol]) ;
}

/*  KBUniSQLQrySelect							*/
/*  getFieldname: Get a specified field name from the query results	*/
/*  qcol	: uint		: Column number				*/
/*  (returns)	: QString	: Name					*/

QString	KBUniSQLQrySelect::getFieldName
	(	uint	qcol
	)
{
	return	qcol >= m_nFields ? QString() : m_fields[qcol] ;
}


/*  KBUniSQLQryUpdate							*/
/*  KBUniSQLQryUpdate							*/
/*		: Constructor for update query object			*/
/*  server	: KBUniSQL *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Update query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  (returns)	: KBUniSQLQryUpdate:					*/

KBUniSQLQryUpdate::KBUniSQLQryUpdate
	(	KBUniSQL		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLUpdate (server, data, query, tabName),
	m_server    (server)
{
	m_nRows	= 0 ;
}

/*  KBUniSQLQryUpdate							*/
/*  execute	: Execute query						*/
/*  nvals	: uint		: Number of substitution values		*/
/*  values	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBUniSQLQryUpdate::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	KBUniSQLParser	*parser	;

	parser	= m_server->execSQL
		  (	m_rawQuery,
			m_subQuery,
			nvals,
			values,
			m_codec,
			"Update query failed",
			m_lError
		  )	;

	if (parser == 0)
		return	false ;

	m_nRows	= 1	;
	return	true	;
}

/*  KBUniSQLQryUpdate							*/
/*  ~KBUniSQLQryUpdate							*/
/*		: Destructor for update query object			*/
/*  (returns)	:		:					*/

KBUniSQLQryUpdate::~KBUniSQLQryUpdate ()
{
}


/*  KBUniSQLQryInsert							*/
/*  KBUniSQLQryInsert							*/
/*		: Constructor for insert query object			*/
/*  server	: KBUniSQL *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  tabName	: const QString & : Table being updated			*/
/*  query	: const QString & : Insert query			*/
/*  (returns)	: KBUniSQLQryInsert:					*/

KBUniSQLQryInsert::KBUniSQLQryInsert
	(	KBUniSQL		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLInsert (server, data, query, tabName),
	m_server    (server)
{
	m_nRows	= 0 ;
}

/*  KBUniSQLQryInsert							*/
/*  ~KBUniSQLQryInsert							*/
/*		: Destructor for insert query object			*/
/*  (returns)	:		:					*/

KBUniSQLQryInsert::~KBUniSQLQryInsert ()
{
}

/*  KBUniSQLQryInsert							*/
/*  execute	: Execute query						*/
/*  nvals	: uint		: Number of substitution values		*/
/*  values	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBUniSQLQryInsert::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	QString		dummy	;
	KBUniSQLParser	*parser	;

	m_newKey = KBValue() ;

	parser	= m_server->execSQL
		  (	m_rawQuery,
			m_subQuery,
			nvals,
			values,
			m_codec,
			"Insert query failed",
			m_lError,
			true
		  )	;

	if (parser == 0)
		return	false ;

//	Not done here - see comments in ::execSQL
//
//	delete	parser	;
//
//	parser	= m_server->execSQL
//		  (	"select @@IDENTITY",
//			dummy,
//			0,
//			0,
//			0,
//			"Error retrieving inserted key",
//			m_lError
//		  )	;
//
//	if (parser == 0)
//		return	false	;

	m_newKey = KBValue (parser->value(0, 0), &_kbFixed) ;
	m_nRows	 = 1	;
	delete	parser	;

	fprintf
	(	stderr,
		"KBUniSQLQryInsert::execute: newkey is [%s]\n",
		(cchar *)m_newKey.getRawText()
	)	;

	return	 true	;
}

/*  KBUniSQLQryInsert							*/
/*  getNewKey	: Get new insert key					*/
/*  primary	: const QString & : Key column name			*/
/*  _newKey	: KBValue &	  : New key				*/
/*  prior	: bool		  : Pre-insert call			*/
/*  (returns)	: bool		  : Success				*/

bool	KBUniSQLQryInsert::getNewKey
	(	const QString	&,
		KBValue		&_newKey,
		bool		prior
	)
{
	if (prior)
	{
		_newKey	 = KBValue () ;
		return	true ; 
	}

	_newKey = m_newKey ;
	return	true ;
}

/*  KBUniSQLQryDelete							*/
/*  KBUniSQLQryDelete							*/
/*		: Constructor for delete query object			*/
/*  server	: KBUniSQL *	  : Server connection object		*/
/*  data	: bool		  : Query for data			*/
/*  query	: const QString & : Delete query			*/
/*  tabName	: const QString & : Table being updated			*/
/*  (returns)	: KBUniSQLQryDelete:					*/

KBUniSQLQryDelete::KBUniSQLQryDelete
	(	KBUniSQL		*server,
		bool		data,
		const QString	&query,
		const QString	&tabName
	)	
	:
	KBSQLDelete (server, data, query, tabName),
	m_server    (server)
{
	m_nRows	= 0 ;
}

/*  KBUniSQLQryDelete							*/
/*  ~KBUniSQLQryDelete							*/
/*		: Destructor for delete query object			*/
/*  (returns)	:		:					*/

KBUniSQLQryDelete::~KBUniSQLQryDelete ()
{
}

/*  KBUniSQLQryDelete							*/
/*  execute	: Execute query						*/
/*  nvals	: uint		: Number of substitution values		*/
/*  values	: KBValue *	: Substitution values			*/
/*  (returns)	: bool		: Success				*/

bool	KBUniSQLQryDelete::execute
	(	uint		nvals,
		const KBValue	*values
	)
{
	KBUniSQLParser	*parser	;

	parser	= m_server->execSQL
		  (	m_rawQuery,
			m_subQuery,
			nvals,
			values,
			m_codec,
			"Delete query failed",
			m_lError
		  )	;

	if (parser == 0)
		return	false ;

	m_nRows	= 1	;
	delete	parser	;
	return	true	;
}


#ifndef _WIN32
KBFACTORY
(	KBUniSQLFactory,
	"driver_unisql"
)

KBFACTORYIMPL
(	KBUniSQLFactory,
	driver_unisql,
	"Rekall UniSQL driver",
	"Plugin",
	"0",
	"7",
	"0"
)
#else
class	KBUniSQLFactory : public KBFactory
{
public:
	inline	KBUniSQLFactory() : KBFactory ()
	{
	}
	virtual	QObject	*create(QObject * = 0, const char *	= 0, const char * = 0, 
		const QStringList &	= QStringList());
};

extern	"C"	__declspec(dllexport) void *init_libkbase_driver_unisql()
{							
	return	new KBUniSQLFactory;				
}							
#endif

QObject	*KBUniSQLFactory::create
	(	QObject		  *parent,
		cchar		  *object,
		cchar		  *,
		const QStringList &
	)
{
	if ((parent != 0) && !parent->inherits ("QWidget"))
	{
		fprintf	(stderr, "KBUniSQLFactory: parent does not inherit QWidget\n") ;
		return	0  ;
	}

	if (strcmp (object, "driver"  ) == 0) return new KBUniSQL () ;

	if (strcmp (object, "advanced") == 0) return new KBUniSQLAdvanced () ;

	return	0 ;
}

cchar	*KBUniSQLFactory::ident ()
{
	return	__KB_BUILD_IDENT	;
}


/*  ------------------------------------------------------------------  */

KBUniSQLConnect::KBUniSQLConnect
	(	QSocket		&socket,
		const QString	&host,
		int		port
	)
	:
	QDialog		(0, "", true),
	m_lStatus	(this),
	m_bCancel	(this),
	m_socket	(socket),
	m_host		(host),
	m_port		(port)
{
	QHBoxLayout	*lay	= new QHBoxLayout (this) ;

	lay->addWidget	(&m_lStatus) ;
	lay->addWidget	(&m_bCancel) ;

	m_bCancel.setText (TR("Cancel")) ;
	m_lStatus.setMinimumWidth  (150) ;

	connect
	(	&m_bCancel,
		SIGNAL(clicked   ()),
		SLOT  (slotCancel())
	)	;

	connect
	(	&m_socket,
		SIGNAL(error(int)),
		SLOT  (slotError(int))
	)	;
	connect
	(	&m_socket,
		SIGNAL(connected()),
		SLOT  (slotConnected())
	)	;
	connect
	(	&m_socket,
		SIGNAL(connectionClosed()),
		SLOT  (slotClosed())
	)	;
	connect
	(	&m_socket,
		SIGNAL(hostFound()),
		SLOT  (slotHostFound())
	)	;
}

KBUniSQLConnect::~KBUniSQLConnect ()
{
}

void	KBUniSQLConnect::slotError
	(	int
	)
{
	done	(0) ;
}

void	KBUniSQLConnect::slotConnected ()
{
	done	(1) ;
}

void	KBUniSQLConnect::slotClosed ()
{
	done	(0) ;
}

void	KBUniSQLConnect::slotHostFound ()
{
	m_lStatus.setText ("Host Found") ;
}

void	KBUniSQLConnect::slotCancel ()
{
	done	(0) ;
}


int	KBUniSQLConnect::exec ()
{
	m_socket.connectToHost (m_host, m_port) ;
	m_lStatus.setText ("Connecting") ;

	return	QDialog::exec () ;
}



/*  ------------------------------------------------------------------  */

/*  KBUniSQLParser							*/
/*  parseText	: Parse reply from text					*/
/*  text	: QByteArray &	: Reply text				*/
/*  (returns)	: bool		: Success				*/

bool	KBUniSQLParser::parseText
	(	QByteArray	&text
	)
{
	QTextStream	 xmlText   (text, IO_ReadOnly) ;
	QXmlInputSource	 xmlSource (xmlText  )	;
	QXmlSimpleReader xmlReader		;

	m_state  = Start	;
	m_curCol = &m_dummy	;

	xmlReader.setContentHandler(this     )	;
	xmlReader.parse		   (xmlSource)	;

	return	true	;
}

cchar	*KBUniSQLParser::state ()
{
	switch (m_state)
	{
		case Error		: return "Error"		;
		case Start		: return "Start"		;
		case Result		: return "Result"		;
		case ColumnInfo		: return "ColumnInfo"		;
		case ColumnInfoDetail	: return "ColumnInfoDetail"	;
		case RowData		: return "RowData"		;
		case ErrorData		: return "ErrorData"		;
		default			: break	 ;
	}

	return	"Unknown" ;
}

/*  KBUniSQLParser							*/
/*  startElement: Handle start of element				*/
/*  URI		: const QString &	: Namespace URI			*/
/*  localName	: const QString &	:				*/
/*  qName	: const QString &	: Element name			*/
/*  attribs	: const QXmlAttributes &: Attribute list		*/
/*  (returns)	: bool			:				*/

bool	KBUniSQLParser::startElement
	(	const	QString		&,
		const	QString		&,
		const	QString		&qName,
		const	QXmlAttributes	&attribs
	)
{
//	fprintf
//	(	stderr,
//		"KBUniSQLParser::startElement: [%s][%s][%s] [%s]",
//		(cchar *)uri,
//		(cchar *)localName,
//		(cchar *)qName,
//		state()
//	)	;

	switch (m_state)
	{
		case Error	:
			break	;

		case Start	:
			if (qstricmp (qName, "result") == 0)
			{	m_state	= Result ;
				break	;
			}
			if (qstricmp (qName, "s:elementtype") == 0)
			{	m_state = ColumnInfo ;
				break	;
			}
			if (qstricmp (qName, "rs:data") == 0)
			{	m_state	= RowData ;
				break	;
			}

			break	;

		case Result	:
			if (qstricmp (qName, "error") == 0)
			{	m_state	= ErrorData ;
				break	;
			}

			break	;

		case ColumnInfo	:
			/* Column information. This comes in the	*/
			/* "s:attributetype" element, and the nested	*/
			/* "s:datatype" element.			*/
			if (qstricmp (qName, "s:attributetype") == 0)
			{
				/* Add a new column to the list of	*/
				/* columns, and set the column index in	*/
				/* the entry. This is used when data is	*/
				/* retrieved.				*/
				m_columns.append (UniColInfo()) ;
				m_curCol = &m_columns.last()	;
				m_curCol->m_slot = m_columns.count() - 1 ;

				for (int c = 0 ; c < attribs.length() ; c += 1)
				{
					QString	name	= attribs.qName(c) ;
					QString	value	= attribs.value(c) ;
					cchar	*cname	= name	;

					if (qstricmp (cname, "name") == 0)
					{
						m_curCol->m_tag      = value ;
						m_colMap.insert (value, m_curCol) ;
						continue ;
					}
					if (qstricmp (cname, "rs:basecolumn") == 0)
					{
						m_curCol->m_name     = value ;
						continue ;
					}
					if (qstricmp (cname, "rs:nullable") == 0)
					{
						m_curCol->m_nullable = value == "true" ;
						continue ;
					}
					if (qstricmp (cname, "rs:autoincrement") == 0)
					{
						m_curCol->m_autoinc  = value == "true" ;
						continue ;
					}
					if (qstricmp (cname, "rs:keycolumn") == 0)
					{
						m_curCol->m_keycol   = value == "true" ;
						continue ;
					}
				}

				break	;
			}

			if (qstricmp (qName, "s:datatype") == 0)
			{
				for (int c = 0 ; c < attribs.length() ; c += 1)
				{
					QString	name	= attribs.qName(c) ;
					QString	value	= attribs.value(c) ;
					cchar	*cname	= name	;

					if (qstricmp (cname, "dt:type") == 0)
					{
						m_curCol->m_type   = value ;
						continue ;
					}
					if (qstricmp (cname, "dt:maxLength") == 0)
					{
						m_curCol->m_length = value.toInt() ;
						continue ;
					}
				}

				break	;
			}

			break	;

		case RowData	:
			/* Now reading data from the reply. The values	*/
			/* are contained in attributes, with no		*/
			/* attribute where the value is null. Hence set	*/
			/* up an empty list of column values and then	*/
			/* populate with non-null values.		*/
			if (qstricmp (qName, "z:row") == 0)
			{
				QStringList	row	;
				UniColInfo	*info	;

				for (uint c1 = 0 ; c1 < m_columns.count () ; c1 += 1)
					row.append (QString::null) ;

				for (int  c2 = 0 ; c2 < attribs  .length() ; c2 += 1)
					if ((info = m_colMap.find  (attribs.qName(c2))) != 0)
						row[info->m_slot] = attribs.value(c2) ;

				m_values.append (row)	;
				break	;
			}

			break	;

		default	:
			break	;
	}

//	fprintf
//	(	stderr,
//		"->[%s]\n",
//		state()
//	)	;

	return	m_state != Error ;
}

/*  KBSAXHandler							*/
/*  endElement	: Handle end of element					*/
/*  URI		: const QString &	: Namespace URI			*/
/*  localName	: const QString &	:				*/
/*  qName	: const QString &	: Element name			*/
/*  (returns)	: bool			: Success			*/

bool	KBUniSQLParser::endElement
	(	const	QString		&,
		const	QString		&,
		const	QString		&qName
	)
{
//	fprintf
//	(	stderr,
//		"KBUniSQLParser::endElement:   [%s][%s][%s]\n",
//		(cchar *)uri,
//		(cchar *)localName,
//		(cchar *)qName
//	)	;

	if (qstricmp (qName, "s:elementtype") == 0)
	{
		m_state	= Start	;
		return	true	;
	}

	return	true	;
}

/*  KBUniSQLParser							*/
/*  characters	: Handle a block of characters				*/
/*  text	: const QString & : Characters				*/
/*  (returns)	: bool		  : Success				*/

bool	KBUniSQLParser::characters
	(	const QString		&text
	)
{
	if (m_state == ErrorData)
	{
		m_error	= text	;
		m_state	= Error	;
		return	true	;
	}

	return	true	;
}
