/***************************************************************************
    file	         : kb_database.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>

#include	<qstring.h>
#include	<qdict.h>

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"

#ifndef		_WIN32
#include	"kb_database.moc"
#else
#include	"kb_database.h"
#endif

#include	"kb_dbinfo.h"
#include	"kb_designinfo.h"


static	void	addFlagToXML
	(	QDomElement	&elem,
		cchar		*attr,
		bool		flag
	)
{
	elem.setAttribute (attr, flag ? "Yes" : "No") ;
}

static	void	addXMLToFlag
	(	const QDomElement	&elem,
		cchar			*attr,
		uint			&flag,
		uint			bit
	)
{
	if (elem.attribute (attr) == "Yes") flag |= bit ;
}

static	void	addDesignToXML
	(	QDomElement	&elem,
		cchar		*attr,
		KBDesignInfo	*di,
		uint		idx
	)
{
	elem.setAttribute (attr, di->getField(idx)->getRawText()) ;
}

/*  KBSQLSelect								*/
/*  getFieldType: Set field type for specified column			*/
/*  qcol	: uint		: Column				*/
/*  (returns)	: KBType *	: Type					*/

KBType	*KBSQLSelect::getFieldType
	(	uint	qcol
	)
{
	if (qcol >= m_nFields)
	{	fprintf	(stderr,
			 "KBSQLSelect::getFieldType(%u) with nFields=%u\n",
			 qcol,
			 m_nFields) ;
		exit	(1) ;
	}

	return	m_types[qcol] ;
}



/*  ------------------------------------------------------------------  */


KBDriverDetails::KBDriverDetails ()
	:
	m_flags		(0)
{
}

KBDriverDetails::KBDriverDetails
	(	const QString	&tag,
		const QString	&comment,
		const QString	&info,
		uint		flags
	)
	:
	m_tag		(tag),
	m_comment	(comment),
	m_info		(info),
	m_flags		(flags)
{
}

KBDriverDetails::KBDriverDetails
	(	const KBDriverDetails	&extant
	)
	:
	m_tag		(extant.m_tag),
	m_comment	(extant.m_comment),
	m_info		(extant.m_info),
	m_flags		(extant.m_flags)
{
}

/*  ------------------------------------------------------------------  */

/*  KBSQLQuery								*/
/*  KBSQLQuery	: Constructor for query base object			*/
/*  server	: KBServer *	 : Server connection object		*/
/*  data	: bool		 : Querying for data			*/
/*  query	: const QString &: Query text				*/
/*  codec	: QTextCodec *   : Non-default codec			*/
/*  (returns)	: KBSQLQuery	 :					*/

KBSQLQuery::KBSQLQuery
	(	KBServer	*server,
		bool		data,
		const QString	&query
	)
	:
	m_rawQuery (query),
	m_server   (server),
	m_codec	   (server->getCodec (data))
{
	m_vtrans = 0 ;
}

/*  KBSQLQuery								*/
/*  ~KBSQLQuery	: Destructor for query base object			*/
/*  (returns)	:		:					*/

KBSQLQuery::~KBSQLQuery ()
{
}

/*  KBSQLQuery								*/
/*  lastError	: Get last error					*/
/*  (returns)	: const KBError&: Last error if any			*/

const KBError	&KBSQLQuery::lastError ()
{
	return	m_lError	;
}



/*  ------------------------------------------------------------------  */

/*  KBFieldSpec								*/
/*  KBFieldSpec	: Field specificaton constructor			*/
/*  colno	: uint		: Table column number			*/
/*  name	: cchar *	: Column name				*/
/*  ftype	: cchar *	: Column type				*/
/*  flags	: uint		: Primary, etc. flags			*/
/*  length	: uint		: Field length				*/
/*  prec	: uint		: Precision				*/
/*  (returns)	: KBFieldSpec	:					*/

KBFieldSpec::KBFieldSpec
	(	uint		colno,
		cchar		*name,
		cchar		*ftype,
		KB::IType	itype,
		uint		flags,
		uint		length,
		uint		prec
	)
	:
	m_state		(KB::RSInSync),
	m_colno		(colno),
	m_name		(name),
	m_typeName	(ftype),
	m_typeIntl	(itype),
	m_flags		(flags),
	m_length	(length),
	m_prec		(prec)
{
	/* This constructor is used within the driver listFields	*/
	/* function, hence the specification is in sync with the	*/
	/* database.							*/
	m_dirty	 = true	;
	m_table	 = 0	;
	m_dbType = 0	;
}

/*  KBFieldSpec								*/
/*  KBFieldSpec	: Field specification constructor			*/
/*  colno	: uint		: Notional column number		*/
/*  (returns)	: KBFieldSpec	:					*/

KBFieldSpec::KBFieldSpec
	(	uint	colno
	)
	:
	m_state		(KB::RSInserted),
	m_colno		(colno),
	m_name		(""),
	m_typeName	(""),
	m_typeIntl	(KB::ITUnknown),
	m_flags		(0),
	m_length	(0),
	m_prec		(0)
{
	/* This constructor is ised when adding a new column during	*/
	/* table design, hence the state is inserted			*/
	m_dirty	 = true	;
	m_table	 = 0	;
	m_dbType = 0	;
}

/*  KBFieldSpec								*/
/*  KBFieldSpec	: Field specificaton constructor			*/
/*  spec	: KBFieldSpec &	: Extant specification			*/
/*  (returns)	: KBFieldSpec	:					*/

KBFieldSpec::KBFieldSpec
	(	KBFieldSpec	&spec
	)
	:
	m_state		(spec.m_state	),
	m_colno		(spec.m_colno	),
	m_name		(spec.m_name	),
	m_typeName	(spec.m_typeName),
	m_typeIntl	(spec.m_typeIntl),
	m_flags		(spec.m_flags	),
	m_length	(spec.m_length	),
	m_prec		(spec.m_prec	),
	m_defval	(spec.m_defval	),
	m_table		(spec.m_table   )
{
	m_dirty	 = true	;

	/* Copy the database type object pointer over. If this does	*/
	/* refer to an object then increment the reference count.	*/
	m_dbType = spec.m_dbType ;
	if (m_dbType != 0) m_dbType->ref() ;
}

/*  KBFieldSpec								*/
/*  KBFieldSpec	: Field specification constructor			*/
/*  _colno	: uint			: Notional column number	*/
/*  elem	: const QDomElement &	: Specification			*/
/*  (returns)	: KBFieldSpec	:					*/

KBFieldSpec::KBFieldSpec
	(	uint			_colno,
		const QDomElement	&elem
	)
{
	m_state   	= KB::RSInserted ;
	m_colno   	= _colno         ;
	m_table   	= 0              ;
	m_flags		= 0		 ;

	m_name		= elem.attribute ("name"     ) ;
	m_typeName	= elem.attribute ("ftype"    ) ;
	m_length	= elem.attribute ("length"   ).toInt () ;
	m_prec		= elem.attribute ("precision").toInt () ;
	m_defval	= elem.attribute ("defval"   ) ;
	m_typeIntl	= (KB::IType)elem.attribute ("itype").toInt () ;

	addXMLToFlag	  (elem, "primary",  m_flags, Primary ) ;
	addXMLToFlag	  (elem, "notnull",  m_flags, NotNull ) ;
	addXMLToFlag	  (elem, "indexed",  m_flags, Indexed ) ;
	addXMLToFlag	  (elem, "unique",   m_flags, Unique  ) ;
	addXMLToFlag	  (elem, "serial",   m_flags, Serial  ) ;

	m_dirty	 	= true	;
	m_dbType 	= 0	;
}

/*  KBFieldSpec								*/
/*  ~KBFieldSpec: Field specification destructor			*/
/*  (returns)	:		:					*/

KBFieldSpec::~KBFieldSpec ()
{
	if (m_dbType != 0) m_dbType->deref() ;
}

/*  KBFieldSpec								*/
/*  toXML	: Generate XML specification element			*/
/*  elem	: QDomElement  & : Specification			*/
/*  di		: KBDesignInfo * : Design information			*/
/*  (returns)	: void		 :					*/

void	KBFieldSpec::toXML
	(	QDomElement	&elem,
		KBDesignInfo	*di
	)
{
	elem.setAttribute ("name",    	m_name  ) ;
	elem.setAttribute ("length",    m_length) ;
	elem.setAttribute ("precision", m_prec  ) ;
	elem.setAttribute ("defval",    m_defval) ;

	addFlagToXML	  (elem, "primary",  m_flags & Primary ) ;
	addFlagToXML	  (elem, "notnull",  m_flags & NotNull ) ;
	addFlagToXML	  (elem, "indexed",  m_flags & Indexed ) ;
	addFlagToXML	  (elem, "unique",   m_flags & Unique  ) ;
	addFlagToXML	  (elem, "serial",   m_flags & Serial  ) ;

	if (di)
	{
		addDesignToXML (elem, "descr",   di, DI_DESCR  ) ;
		addDesignToXML (elem, "evalid",  di, DI_EVALID ) ;
		addDesignToXML (elem, "igncase", di, DI_IGNCASE) ;
		addDesignToXML (elem, "default", di, DI_DEFAULT) ;
		addDesignToXML (elem, "format",  di, DI_FORMAT ) ;
		addDesignToXML (elem, "link",    di, DI_LINK   ) ;
		addDesignToXML (elem, "width",   di, DI_WIDTH  ) ;
	}

	addFlagToXML (elem, "insavail", m_flags & InsAvail) ;

	elem.setAttribute ("itype", m_typeIntl) ;
	elem.setAttribute ("ftype", m_typeName) ;
	elem.setAttribute ("colno", m_colno   ) ;

	if	(m_state == KB::RSDeleted )
		elem.setAttribute ("state", "deleted" ) ;
	else if (m_state == KB::RSInserted)
		elem.setAttribute ("state", "inserted") ;
	else if (m_state == KB::RSChanged )
		elem.setAttribute ("state", "changed" ) ;
}

/*  KBFieldSpec								*/
/*  operator ==	: Compare specifications				*/
/*  other	: const KBFieldSpec &	: Other specification		*/
/*  (returns)	: bool			: Equality			*/

bool	KBFieldSpec::operator ==
	(	const KBFieldSpec	&other
	)	const
{
	return	(m_colno    == other.m_colno   ) &&
		(m_name     == other.m_name    ) &&
		(m_typeName == other.m_typeName) &&
		(m_typeIntl == other.m_typeIntl) &&
		(m_flags    == other.m_flags   ) &&
		(m_length   == other.m_length  ) &&
		(m_prec     == other.m_prec    ) ;
}

/*  ------------------------------------------------------------------  */

/*  KBTableDetails							*/
/*  KBTableDetails							*/
/*		: Constructor for table details object			*/
/*  name	: const QString & : Table name				*/
/*  type	: TableType	  : Type (table, view, ....)		*/
/*  perms	: uint		  : Table-level permissions		*/
/*  extra	: const QString & : Extra information			*/
/*  (returns)	: KBTableDetails  :					*/

KBTableDetails::KBTableDetails
	(	const QString	&name,
		KB::TableType	type,
		uint		perms,
		const QString	&extra
	)
	:
	m_name	(name),
	m_type	(type),
	m_perms	(perms),
	m_extra	(extra)
{
}

/*  KBTableDetails							*/
/*  KBTableDetails							*/
/*		: Constructor for table details object			*/
/*  (returns)	: KBTableDetails  :					*/

KBTableDetails::KBTableDetails ()
{
	m_perms	= 0 ;
}

/*  KBTableDetails							*/
/*  KBTableDetails							*/
/*		: Constructor for table details object			*/
/*  extant	: KBTableDetails & : Extant details			*/
/*  (returns)	: KBTableDetails   :					*/

KBTableDetails::KBTableDetails
	(	const KBTableDetails	&extant
	)
{
//	fprintf
//	(	stderr,
//		"KBTableDetails::KBTableDetails: copying ....\n"
//	)	;

	m_name	= extant.m_name	 ;
	m_type	= extant.m_type	 ;
	m_perms	= extant.m_perms ;
	m_extra	= extant.m_extra ;
}

/*  KBTableDetails							*/
/*  typeText	: Get type as text					*/
/*  (returns)	: QString	:					*/

QString	KBTableDetails::typeText ()
{
	switch (m_type)
	{
		case KB::IsTable	:
			return	TR("Table"   )	;

		case KB::IsView	:
			return	TR("View"    )	;

		case KB::IsSequence	:
			return	TR("Sequence")	;

		default	:
			break	;
	}

	return	TR("Unknown") ;
}



/*  ------------------------------------------------------------------  */


/*  KBTableSpec								*/
/*  KBTableSpec	: Constructor for a table specification object		*/
/*  name	: const QString & : Table name				*/
/*  (returns)	: KBTableSpec	  :					*/

KBTableSpec::KBTableSpec
	(	const QString	&name
	)
	:
	m_name	(name)
{
	m_prefKey   = -1	;
	m_keepsCase = true	;
	m_fakeKey   = 0		;
	m_type	    = KB::IsTable ;

	m_fldList.setAutoDelete (true) ;
}

/*  KBTableSpec								*/
/*  KBTableSpec	: Constructor for a table specification object		*/
/*  elem	: const QDomElement &					*/
/*				: Table specification			*/
/*  (returns)	: KBTableSpec	:					*/

KBTableSpec::KBTableSpec
	(	const QDomElement	&elem
	)
	:
	m_name	(elem.attribute("name"))
{
	QDomNode node	= elem.firstChild() ;
	uint	 colno	= 0 ;

	while (!node.isNull())
	{
		m_fldList.append (new KBFieldSpec (colno, node.toElement())) ;
		node	= node.nextSibling() ;
		colno  += 1 ;
	}

	m_prefKey   = -1   ;
	m_keepsCase = true ;
	m_fakeKey   = 0	   ;

	if (elem.attribute ("type") == "view")
		m_type	= KB::IsView  ;
	else	m_type	= KB::IsTable ;

	m_view	    = elem.attribute  ("view") ;

	m_fldList.setAutoDelete (true) ;
}

/*  KBTableSpec								*/
/*  KBTableSpec	: Constructor for a table specification object		*/
/*  spec	: KBTableSpec &	: Extant specification			*/
/*  (returns)	: KBTableSpec	:					*/

KBTableSpec::KBTableSpec
	(	const KBTableSpec	&spec
	)
{
	m_name	    = spec.m_name	;
	m_prefKey   = spec.m_prefKey	;
	m_fakeKey   = spec.m_fakeKey	;
	m_keepsCase = spec.m_keepsCase  ;
	m_type	    = spec.m_type	;
	m_view	    = spec.m_view	;

	LITER
	(	KBFieldSpec,
		spec.m_fldList,
		fSpec,

		m_fldList.append (new KBFieldSpec (*fSpec))
	)

	m_fldList.setAutoDelete (true) ;
}

KBTableSpec
	&KBTableSpec::operator =
	(	const KBTableSpec	&spec
	)
{
	m_name	    = spec.m_name	;
	m_prefKey   = spec.m_prefKey	;
	m_keepsCase = spec.m_keepsCase	;
	m_fakeKey   = spec.m_fakeKey	;

	m_fldList.clear () ;

	LITER
	(	KBFieldSpec,
		spec.m_fldList,
		fSpec,

		m_fldList.append (new KBFieldSpec (*fSpec))
	)

	return	*this	;
}

/*  KBTableSpec								*/
/*  reset	: Reset specification					*/
/*  _name	: const QString & : Table name				*/
/*  (returns)	: void		:					*/

void	KBTableSpec::reset
	(	const QString	&_name
	)
{
	m_name	    = _name ;
	m_prefKey   = -1    ;
	m_fakeKey   = 0	    ;
	m_keepsCase = true  ;
	
	m_fldList.clear ()  ;
}

/*  KBTableSpec								*/
/*  insertKeyAvail							*/
/*		: See if insert key is available for column		*/
/*  col		: int		: Required column			*/
/*  (returns)	: bool		: True if key available			*/

bool	KBTableSpec::insertKeyAvail
	(	int	col
	)
{
	if ((col < 0) || (col >= (int)m_fldList.count())) return false  ;
	return	(m_fldList.at(col)->m_flags & KBFieldSpec::InsAvail) != 0 ;
}

/*  KBTableSpec								*/
/*  findField	: Find named field					*/
/*  name	: const QString & : Field				*/
/*  (returns)	: KBFieldSpec *	  : Field specification or null		*/

KBFieldSpec
	*KBTableSpec::findField
	(	const QString	&name
	)
{
	LITER
	(	KBFieldSpec,
		m_fldList,
		fSpec,

		bool	same ;

		if (m_keepsCase)
			same = fSpec->m_name == name ;
		else	same = fSpec->m_name.lower() == name.lower() ;

		if (same) return fSpec ;
	)

	return	0 ;
}

/*  KBTableSpec								*/
/*  toXML	: Generate XML specification element			*/
/*  elem	: QDomElement &	  : Specification			*/
/*  di		: QList<KBDesignInfo> *					*/
/*				  : Design information			*/
/*  (returns)	: void		  :					*/

void	KBTableSpec::toXML
	(	QDomElement		&elem,
		QList<KBDesignInfo>	*di
	)
{
	elem.setAttribute ("name", m_name) ;
	elem.setAttribute ("type", m_type == KB::IsView ? "view" : "table") ;
	elem.setAttribute ("view", m_view) ;

	for (uint i = 0 ; i < m_fldList.count() ; i += 1)
	{	QDomElement col = elem.ownerDocument().createElement ("column") ;
		m_fldList.at(i)->toXML
		(	col,
			di == 0 ? 0 : di->at(i)
		)	;
		elem.appendChild (col) ;
	}
}

/*  KBTableSpec								*/
/*  findPrimary	: Find the primary column if any			*/
/*  (returns)	: KBFieldSpec *	: Column specification or null		*/

KBFieldSpec
	*KBTableSpec::findPrimary ()
{
	/* A primary key column will be the preferred column, so we	*/
	/* case use ::findUnique to get the a candidate column.		*/
	KBFieldSpec *fSpec = findUnique () ;

	return	fSpec == 0 ? 0 :
		(fSpec->m_flags & KBFieldSpec::Primary) == 0 ? 0 : fSpec ;
}


/*  KBTableSpec								*/
/*  findUnique	: Find a unique column if any				*/
/*  (returns)	: KBFieldSpec *	: Column specification or null		*/

KBFieldSpec
	*KBTableSpec::findUnique ()
{
	return	m_prefKey < 0 ? 0 : m_fldList.at(m_prefKey) ;
}


/*  ------------------------------------------------------------------  */

/*  KBSequenceSpec							*/
/*  KBSequenceSpec							*/
/*		: Constructor for a sequence specification object	*/
/*  name	: const QString & : Sequence name			*/
/*  (returns)	: KBSequenceSpec	  :				*/

KBSequenceSpec::KBSequenceSpec
	(	const QString	&name
	)
	:
	m_name	(name)
{
	m_increment	= 0 ;
	m_minValue	= 0 ;
	m_maxValue	= 0 ;
	m_start		= 0 ;
	m_flags		= 0 ;
}

/*  KBSequenceSpec							*/
/*  KBSequenceSpec							*/
/*		: Constructor for a table specification object		*/
/*  elem	: const QDomElement &	: Sequence specification	*/
/*  (returns)	: KBSequenceSpec	:				*/

KBSequenceSpec::KBSequenceSpec
	(	const QDomElement	&elem
	)
	:
	m_name	(elem.attribute("name"))
{
	m_increment	= elem.attribute("increment").toInt () ;
	m_minValue	= elem.attribute("minvalue" ).toInt () ;
	m_maxValue	= elem.attribute("maxvalue" ).toInt () ;
	m_start		= elem.attribute("start"    ).toInt () ;
	m_flags		= elem.attribute("flags"    ).toUInt() ;
}

/*  KBSequenceSpec							*/
/*  KBSequenceSpec							*/
/*		: Constructor for a table specification object		*/
/*  spec	: KBSequenceSpec &	: Extant specification		*/
/*  (returns)	: KBSequenceSpec	:				*/

KBSequenceSpec::KBSequenceSpec
	(	const KBSequenceSpec	&spec
	)
	:
	m_name	(spec.m_name)
{
	m_increment	= spec.m_increment	;
	m_minValue	= spec.m_minValue	;
	m_maxValue	= spec.m_maxValue	;
	m_start		= spec.m_start		;
	m_flags		= spec.m_flags		;
}

/*  KBSequenceSpec							*/
/*  toXML	: Generate XML specification element			*/
/*  elem	: QDomElement  & : Specification			*/
/*  (returns)	: void		 :					*/

void	KBSequenceSpec::toXML
	(	QDomElement	&elem
	)
{
	elem.setAttribute ("name",    	m_name	   ) ;
	elem.setAttribute ("increment",	m_increment) ;
	elem.setAttribute ("minvalue",	m_minValue ) ;
	elem.setAttribute ("maxvalue",	m_maxValue ) ;
	elem.setAttribute ("start",	m_start	   ) ;
	elem.setAttribute ("flags",	m_flags	   ) ;
}

/*  ------------------------------------------------------------------  */

KBDBAdvanced::KBDBAdvanced
	(	const QString	&dbType
	)
	:
	m_dbType (dbType)
{
}

KBDBAdvanced::~KBDBAdvanced ()
{
}



