/***************************************************************************
    file	         : kb_dbinfo.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<qdir.h>
#include	<qfile.h>
#include	<qtextstream.h>

#include	"kb_classes.h"
#include	"kb_location.h"
#include	"kb_dbinfo.h"
#include	"kb_dblink.h"



/*  KBDBInfo								*/
/*  KBDBInfo	: Constructor for database information class		*/
/*  dbName	: const QString &: Information file name		*/
/*  (returns)	: KBDBInfo	 : Database information			*/

KBDBInfo::KBDBInfo
	(	const QString	&dbName
	)
	:
	m_dbName	(dbName),
	m_version	(0)
{
	m_files		= 0		;
	m_wasBSF	= false		;
	m_version	= KB_HEXVER 	;
}

/*  KBDBInfo								*/
/*  init	: Initialise object					*/
/*  create	: bool		: Create new database			*/
/*  (returns)	: void		:					*/

void	KBDBInfo::init
	(	bool		create
	)
{
	/* Get the directory path corresponding to the database file.	*/
	/* It is this directory that we will scan for forms, reports,	*/
	/* etc.								*/
	int	sIdx	= m_dbName.findRev ('/') ;
	int	dIdx	= m_dbName.findRev ('.') ;

	if (dIdx < sIdx) dIdx = -1 ;

	if	(sIdx <  0) m_dbPath = QDir(".").absPath() ;
	else if (sIdx == 0) m_dbPath = "/" ;
	else		    m_dbPath = m_dbName.left(sIdx) ;

	if (dIdx >= 0)
		m_dbExtn = m_dbName.mid (dIdx + 1) ;
	else
		m_dbExtn = "rkl"	;

	fprintf	(stderr, "dbName: %s\n", (cchar *)m_dbName) ;
	fprintf	(stderr, "dbPath: %s\n", (cchar *)m_dbPath) ;
	fprintf	(stderr, "dbExtn: %s\n", (cchar *)m_dbExtn) ;

	m_servers.setAutoDelete (true) ;
	m_files	 = 0 	;
	m_wasBSF = false	;

	if (!create)
	{
		QFile	file	(m_dbName) ;
		if (file.open (IO_ReadOnly))
		{
			QTextStream stream (&file) ;
			QString	    text   = stream.read() ;

			if (text.at(0) == '<')
				loadDomFormat (text) ;
			else	loadBSFFormat (text) ;
		}
	}
	else	m_version = KB_HEXVER ;

	if (m_files == 0)
	{	m_files = newServerInfo () ;
		m_files->m_serverName = KBLocation::m_pFile ;
	}

	if (create) save (m_dbName) ;
}

/*  KBDBInfo								*/
/*  ~KBDBInfo	: Destructor for database information class		*/
/*  (returns)	:		:					*/

KBDBInfo::~KBDBInfo ()
{
	/* Save the table information for each server. This is *not*	*/
	/* done in the KBServerInfo destructors since by that stage	*/
	/* they have been removed from the server dictionary.		*/
	QDictIterator<KBServerInfo> svIter (m_servers) ;
	KBServerInfo *server ;

	while ((server = svIter.current()) != 0)
	{
		server ->saveTableInfo() ;
		svIter += 1  ;
	}

	if (m_files != 0)
		m_files->saveTableInfo() ;

	DELOBJ	(m_files) ;
}

/*  KBDBInfo								*/
/*  loadDomFormat							*/
/*		: Load from XML formatted text				*/
/*  text	: const QString & : text				*/
/*  (returns)	: void		  :					*/

void	KBDBInfo::loadDomFormat
	(	const QString	&text
	)
{
	QDomDocument doc ;
	doc.setContent (text) ;
	QDomElement  docElem = doc.documentElement() ;
	QDomNode     node    = docElem.firstChild () ;

	m_version = docElem.attribute("version").toUInt() ;

	DOMITER_BEGIN(docElem,"serverinfo",server)
	{
		KBServerInfo *n = newServerInfo (server) ;

		if (n->m_serverName == KBLocation::m_pFile)
		{
			m_files	= n ;
		}
		else
		{	m_servers.insert (n->m_serverName, n) ;
			m_svrList.append (n) ;
		}
	}
	DOMITER_END

	m_wasBSF = false ;
}

/*  KBDBInfo								*/
/*  loadBSFFormat							*/
/*		: Load from bar-separated formatted text			*/
/*  text	: const QString & : text				*/
/*  (returns)	: void		  :					*/

void	KBDBInfo::loadBSFFormat
	(	const QString	&text
	)
{
	QStringList sl = QStringList::split ("\n", text) ;

	for (QStringList::Iterator it = sl.begin() ; it != sl.end() ; ++it)
	{
		QString	line	= (*it) ;

		if (line.left(7) == "server=")
		{
			KBServerInfo *n = newServerInfo (line.mid(7)) ;
			m_servers.insert (n->m_serverName, n) ;
			m_svrList.append (n) ;
			continue ;
		}

		if (line.left(6) == "files=" )
		{
			m_files	= newServerInfo (line.mid(6)) ;
			m_files->m_serverName = KBLocation::m_pFile ;
			continue ;
		}
	}

	m_version = 0	 ;
	m_wasBSF  = true ;
}

/*  KBDBInfo								*/
/*  findServer	: Find server by name					*/
/*  svName	: const QString & : Server name				*/
/*  (returns)	: KBServerInfo  * : Server or null if not found		*/

KBServerInfo
	*KBDBInfo::findServer
	(	const QString &svName
	)
{
	return	svName == KBLocation::m_pFile ? m_files : m_servers.find (svName) ;
}

/*  KBDBInfo								*/
/*  findTableInfoSet							*/
/*		: Find table information by server			*/
/*  svName	: const QString  & : Server name			*/
/*  (returns)	: KBTableInfoSet * : Table information			*/

KBTableInfoSet
	*KBDBInfo::findTableInfoSet
	(	const QString	&svName
	)
{
	KBServerInfo *svInfo = findServer (svName) ;
	if (svInfo == 0) return 0 ;

	return	svInfo->tableInfoSet() ;
}

/*  KBDBInfo								*/
/*  findTableInfo : Find table information by server and table name	*/
/*  svName	  : const QString & : Server name			*/
/*  tabName	  : const QString & : Table name			*/
/*  (returns)	  : KBTableInfo  *  : Inforamation or null if not found	*/

KBTableInfo
	*KBDBInfo::findTableInfo
	(	const QString	&svName,
		const QString	&tabName
	)
{
	KBServerInfo *svInfo = findServer (svName) ;
	if (svInfo == 0) return 0 ;

	return	svInfo->tableInfoSet()->getTableInfo(tabName) ;
}

/*  KBDBInfo								*/
/*  save	: Save database information				*/
/*  _dbName	: const QString &: Database name			*/
/*  (returns)	: bool		 : Success				*/

bool	KBDBInfo::save
	(	const QString	&dbName
	)
{
	QFile file (dbName) ;

	if (m_wasBSF)
		KBError::EWarning
		(	TR("Saving database"),
			QString	(TR ("Converting database file to XML format\n"
				     "Original file will be stored as %1"))
				.arg(dbName + "~"),
			__ERRLOCN
			
		)	;

	if (file.exists ())
	{	QString	bak = dbName + "~" ;
		::rename (m_dbName.local8Bit(), bak.local8Bit()) ;
	}

	if (!file.open (IO_WriteOnly))
	{
		KBError::EError
		(
			TR("Saving database"),
			QString(TR("Cannot save database %1")).arg(m_dbName),
			__ERRLOCN
		)	;
		return false ;
	}

	QDomDocument doc ("rekallDB") ;
	QDomElement  docElem = doc.createElement ("servers"   ) ;

	docElem .setAttribute	  ("version", KB_HEXVER) ;
	doc	.appendChild	  (doc.createProcessingInstruction("xml", "version=\"1.0\" encoding=\"UTF=8\"")) ;
	doc	.appendChild	  (docElem) ;

	if (m_files != 0)
	{
		QDomElement  felem   = doc.createElement ("serverinfo") ;
		m_files->buildSpecElement (felem) ;
		docElem	.appendChild	  (felem) ;
	}

	QDictIterator<KBServerInfo> svIter (m_servers) ;
	KBServerInfo *svInfo ;
	QString	     svText  ;

	while ((svInfo = svIter.current()) != 0)
	{
		QDomElement  selem   = doc.createElement ("serverinfo") ;
		docElem.appendChild	 (selem) ;
		svInfo->buildSpecElement (selem) ;
		svIter += 1 ;
	}

	QTextStream s (&file)	;
	s << doc.toString()	;
	file.close	 ()	;

	m_wasBSF = false	;
	return	true		;
}

/*  KBDBInfo								*/
/*  remove	: Remove an entry					*/
/*  svInfo	: KBServerInfo * : Entry in question			*/
/*  del		: bool		 : Delete entry				*/
/*  (returns)	: void		 :					*/

void	KBDBInfo::remove
	(	KBServerInfo	*svInfo,
		bool		del
	)
{
	m_svrList.removeRef (svInfo) ;

	if (svInfo->m_serverName == KBLocation::m_pFile)
		m_files	= 0	;
	else	m_servers.take (svInfo->m_serverName) ;

	if (del) delete svInfo ;
}

/*  KBDBInfo								*/
/*  insert	: Insert an entry					*/
/*  svInfo	: KBServerInfo * : Entry in question			*/
/*  (returns)	: void		 :					*/

void	KBDBInfo::insert
	(	KBServerInfo	*svInfo
	)
{
	if (svInfo->m_serverName == KBLocation::m_pFile)
	{
		DELOBJ	(m_files)		;
		m_files	= svInfo	;
	}
	else
	{	m_servers.insert (svInfo->m_serverName, svInfo) ;
		m_svrList.append (svInfo) ;
	}
}

/*  KBDBInfo								*/
/*  getServerIter: Get an iterator on the list of servers		*/
/*  (returns)	 : (iterator)	:					*/

QListIterator<KBServerInfo> *KBDBInfo::getServerIter()
{
	QListIterator<KBServerInfo> *iter = new QListIterator<KBServerInfo> (m_svrList) ;
	return	iter ;
}

KBServerInfo
	*KBDBInfo::newServerInfo ()
{
	return	new KBServerInfo (this) ;
}

KBServerInfo
	*KBDBInfo::newServerInfo
	(	const QString	  &spec
	)
{
	return	new KBServerInfo (this, spec) ;
}

KBServerInfo
	*KBDBInfo::newServerInfo
	(	const QDomElement &elem
	)
{
	return	new KBServerInfo (this, elem) ;
}

