/***************************************************************************
    file	         : kb_conductor.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<errno.h>

#include	<qapp.h>
#include	<qtimer.h>
#include	<qfile.h>
#include	<qpixmap.h>
#include	<qcursor.h>

#ifndef		_WIN32
#include	"kb_conductor.moc"
#else
#include	"kb_conductor.h"
#endif

#include	"kb_node.h"

#if		__KB_KDE
#include	<X11/Xlib.h>
#include	<X11/Xatom.h>
#endif


static	KBConductor	*conductor;

/*  KBConductor								*/
/*  KBConductor	: Constructor for replay conductor class		*/
/*  (returns)	: KBConductor	:					*/

KBConductor::KBConductor ()
{
#if	__KB_KDE
	m_state		= StateIdle	;
	conductor	= this		;
	m_sending.setAutoDelete	 (true) ;

	m_process << "rekallCon"	;
#endif
}

/*  KBConductor								*/
/*  ~KBConductor: Destructor for replay conductor class			*/
/*  (returns)	:		:					*/

KBConductor::~KBConductor ()
{
#if	__KB_KDE
	conductor	= 0	;
#endif
}

/*  KBConductor								*/
/*  addPlayer	: Add a new player					*/
/*  player	: KBPlayer *	: The player				*/
/*  (returns)	: void		:					*/

void	KBConductor::addPlayer
	(	KBPlayer	*player
	)
{
#if	__KB_KDE
//	fprintf	(stderr, "KBConductor adds    %s:%s:%s\n",
//			 (cchar *)player->playerClass(),
//			 (cchar *)player->playerType (),
//			 (cchar *)player->playerName ()) ;

	m_players.append (player) ;
#endif
}

/*  KBConductor								*/
/*  remPlayer	: Remove a player					*/
/*  player	: KBPlayer *	: The player				*/
/*  (returns)	: void		:					*/

void	KBConductor::remPlayer
	(	KBPlayer	*player
	)
{
#if	__KB_KDE
//	fprintf	(stderr, "KBConductor removes %s:%s:%s\n",
//			 (cchar *)player->playerClass(),
//			 (cchar *)player->playerType (),
//			 (cchar *)player->playerName ()) ;

	m_players.removeRef (player) ;
#endif
}

/*  KBConductor								*/
/*  init	: Initialise a new score				*/
/*  scoreName	: const QString & : Score file name			*/
/*  error	: KBError &	  : Error details return		*/
/*  withDlg	: bool		  : Run with dialog			*/
/*  (returns)	: bool		  : Success				*/

bool	KBConductor::init
	(	const QString	&scoreName,
		KBError		&error,
		bool		withDlg
	)
{
#if	__KB_KDE
	m_stanza.clear()	;
	m_score .clear()	;
	m_state	= StateIdle	;

	/* Open thee score file, parse into a DOM document, and get the	*/
	/* first child. We assume that these top level children are	*/
	/* all stanza's and don;t bother to check the tags.		*/
	QFile	scoreFile (scoreName) ;

	if (!scoreFile.open (IO_ReadOnly))
	{
		error	= KBError
			  (	KBError::Error,
				"Cannot open score file",
				QString("%1: %2").arg(scoreName).arg(strerror(errno)),
				__ERRLOCN
			  )	;
		return	false	;
	}

	if (!m_score.setContent (&scoreFile))
	{
		m_score.clear ();

		error	= KBError
			  (	KBError::Error,
				"Cannot parse score",
				scoreName,
				__ERRLOCN
			  )	;
		return	false	;
	}

	QDomElement	docElem	;
	QDomNode	topNode	;

	docElem	= m_score.documentElement() ;
	if (!docElem.isNull()) topNode  = docElem.firstChild() ;
	if (!topNode.isNull()) m_stanza = topNode.toElement () ;

	if (m_stanza.isNull())
	{
		m_score.clear ();

		error	= KBError
			  (	KBError::Error,
				"Score is empty",
				scoreName,
				__ERRLOCN
			  )	;
		return	false	;
	}

	m_state	  	= StateIdle	;
	m_withDlg 	= withDlg	;
#endif
	return	  true	;
}

void	KBConductor::startDelay
	(	int		delay,
		State		nextState
	)
{
	QTimer::singleShot (delay, this, SLOT(nextStanza())) ;
#if	__KB_KDE
	m_state		= nextState	;
#endif
}

/*  KBConductor								*/
/*  start	: Start playing score					*/
/*  (returns)	: void							*/

void	KBConductor::start ()
{
#if	__KB_KDE
	if (!m_withDlg)
	{
		m_state	   = StateLeadin ;
		nextStanza () ;
		return	;
	}

	if (!m_process.isRunning())
		if (!m_process.start (KProcess::NotifyOnExit, (KProcess::Communication)(KProcess::Stdin|KProcess::Stdout)))
		{
			fprintf	(stderr, "KBConductor::start: process started failed\n") ;
			return	;
		}

	connect
	(	&m_process,
		SIGNAL(processExited (KProcess *)),
		SLOT  (processExited (KProcess *))
	)	;
	connect
	(	&m_process,
		SIGNAL(wroteStdin    (KProcess *)),
		SLOT  (wroteStdin    (KProcess *))
	)	;
	connect
	(	&m_process,
		SIGNAL(receivedStdout(KProcess *, char *, int)),
		SLOT  (receivedStdout(KProcess *, char *, int))
	)	;

	m_state	   = StateLeadin ;
	sendToDialog ('I', 0)    ;
#endif
}

/*  KBConductor								*/
/*  processExited: Notification that dialog process has exited		*/
/*  process	 : KBprocess *	: The process				*/
/*  (returns)	 : void		:					*/

void	KBConductor::processExited
	(	KProcess	*
	)
{
#if	__KB_KDE
	fprintf	(stderr, "Conductor process exited\n") ;
	m_state	= StateIdle	;
#endif
}

/*  KBConductor								*/
/*  receivedStdout: Notification of data from dialog process		*/
/*  process	  : KBprocess *	: The process				*/
/*  (returns)	  : void		:				*/

void	KBConductor::receivedStdout
	(	KProcess	*,
		char		*,
		int
	)
{
#if	__KB_KDE
	fprintf	(stderr, "Received from conductor process\n") ;
	nextStanza ()	;
#endif
}

/*  KBConductor								*/
/*  writeToDialog: Write message buffer to dialog process		*/
/*  msg		 : MsgBuff *	: The message buffer			*/
/*  (returns)	 : void		:					*/

void	KBConductor::writeToDialog
	(	MsgBuff		*msg
	)
{
#if	__KB_KDE
	/* Write the message. If there is an error then clear the	*/
	/* message queue and set idle, and warn the user.		*/
	if (!m_process.writeStdin ((char *)msg, sizeof(uint) + msg->len))
	{
		m_sending.clear  () ;
		m_state	= StateIdle ;

		KBError::EError
		(	"Conductor failed to write to dialog",
			strerror(errno),
			__ERRLOCN
		)	;
	}
#endif
}

/*  KBConductor								*/
/*  wroteStdin	: Notification of write completion			*/
/*  process	: KBprocess *	: The process				*/
/*  (returns)	: void		:					*/

void	KBConductor::wroteStdin
	(	KProcess	*
	)
{
#if	__KB_KDE
	/* Remove the head message, which is not written. If there are	*/
	/* further messages pending then send the next one.		*/
	m_sending.removeFirst() ;
	if (m_sending.count() > 0) writeToDialog (m_sending.at(0)) ;
#endif
}

/*  KBConductor								*/
/*  sendToDialog: Send a message to the dialog process			*/
/*  code	: uint		  : Message code			*/
/*  seq		: uint		  : Score sequence number		*/
/*  text	: const QString & : Message text			*/
/*  (returns)	: void		  :					*/

void	KBConductor::sendToDialog
	(	uint		code,
		uint		seq,
		const QString	&text
	)
{
#if	__KB_KDE
	/* The message is stored in a buffer along with its lenght, and	*/
	/* queued. If it becomes the only message on the queue then it	*/
	/* is send at once, otherwise it will wait until all prior	*/
	/* messages are sent.						*/
	cchar	*tmp	= text		;
	uint	tlen	= strlen(tmp)	;
	MsgBuff	*msg	= (MsgBuff *)new char[3 * sizeof(uint) + tlen + 1] ;

	msg->len	= 2 * sizeof(uint) + tlen + 1 ;
	msg->code	= code	  ;
	msg->seq	= seq	  ;
	strcpy	(msg->buff, tmp == 0 ? "" : tmp) ;
	

	m_sending.append (msg) ;
	if (m_sending.count() == 1) writeToDialog (msg) ;
#endif
}

/*  KBConductor								*/
/*  sendMessage	: Send possible text message to dialog process		*/
/*  seq		: uint		  : Score sequence number		*/
/*  (returns)	: bool		: True if a text message was sent	*/

bool	KBConductor::sendMessage
	(	uint	seq
	)
{
#if	__KB_KDE
	QDomNode child	= m_stanza.firstChild () ;
	QString	 text	;

	while (!child.isNull())
	{
		QDomElement elem = child.toElement () ;
		if (elem.tagName() == "text")
		{
			text	= elem.text() ;
			break	;
		}
		child	= child.nextSibling() ;
	}

	if (!text.isEmpty())
	{	sendToDialog ('M', seq, text) ;
		return	true  ;
	}
#endif

	return	false	;
}

/*  KBConductor								*/
/*  sendReady	: Send ready-for-next-step indication to dialog process	*/
/*  seq		: uint		  : Score sequence number		*/
/*  (returns)	: void		:					*/

void	KBConductor::sendReady
	(	uint	seq
	)
{
#if	__KB_KDE
	sendToDialog ('R', seq) ;
#endif
}

/*  KBConductor								*/
/*  sendSnapshot: Send ready-for-snapshot indication to dialog process	*/
/*  seq		: uint		  : Score sequence number		*/
/*  (returns)	: void		:					*/

void	KBConductor::sendSnapshot
	(	uint	seq
	)
{
#if	__KB_KDE
	sendToDialog ('S', seq) ;
#endif
}

/*  KBConductor								*/
/*  snapshot	: Take a snapshot					*/
/*  player	: KBPlayer *	  : Player to snapshot			*/
/*  file	: const QString & : Snapshot file name less extension	*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBConductor::snapshot
	(	KBPlayer	*player,
		const QString	&file,
		KBError		&pError
	)
{
#if	__KB_KDE
	QWidget	*widget	;
	WId	window	; 

	if ((widget = player->playerWidget()) == 0)
	{
		pError	= KBError
			  (	KBError::Error,
				TR("Error saving snapshot"),
				QString	(TR("Player %1/%2 does not have a widget"))
					.arg(player->playerClass())
					.arg(player->playerType ()),
				__ERRLOCN
			  )	;
		return	false	;
	}
	if ((window = widget->winId()) == 0)
	{
		pError	= KBError
			  (	KBError::Error,
				TR("Error saving snapshot"),
				QString	(TR("Player %1/%2 does not have a window"))
					.arg(player->playerClass())
					.arg(player->playerType ()),
				__ERRLOCN
			  )	;
		return	false	;
	}

	/* Raise the widget and make sure its window becomes active.	*/
	/* Then the nasty bit. QPinxmap::grabWindow() with the above	*/
	/* window *does not* get the window frame, at least not on KDE.	*/
	/* Hence the following code. I can't find a point->window X11	*/
	/* mapping function, so move the cursor to an appropriate place	*/
	widget->raise 		() ;
	widget->setActiveWindow () ;
	QCursor::setPos (widget->mapToGlobal (QPoint(widget->width()/2, widget->height()/2))) ;
	qApp->processEvents () ;

        Window	root;
        Window	child;
        uint	mask;
        int	rootX, rootY, winX, winY;

        XQueryPointer
	(	qt_xdisplay(),
		qt_xrootwin(),
		&root,
		&child,
		&rootX, &rootY, &winX, &winY,
		&mask
	)	;

	QPixmap snap	= QPixmap::grabWindow (child) ;
	if (!snap.save ("/tmp/" + file + ".png", "PNG"))
	{
		pError	= KBError
			  (	KBError::Error,
				"Error saving snapshot",
				QString	("%1: %3")
					.arg(file + ".png")
					.arg(strerror(errno)),
				__ERRLOCN
			  )	;
		return	false	;
	}
#endif
	return	true	;
}


/*  KBConductor								*/
/*  doPerform	: Perform the actiin in the current stanza		*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBConductor::doPerform
	(	KBError		&pError
	)
{
#if	__KB_KDE
	QString		_class	= m_stanza.attribute  ("class"   ) ;
	QString		type	= m_stanza.attribute  ("type"    ) ;
	QString		name	= m_stanza.attribute  ("name"    ) ;
	QString		action	= m_stanza.attribute  ("action"  ) ;
	QString		seq	= m_stanza.attribute  ("seq"     ) ;
	QDomNode	child	= m_stanza.firstChild () ;
	QStringList	args	;

	if (action == "none")
		return	true	;

	while (!child.isNull())
	{
		QDomElement elem = child.toElement () ;
		if (elem.tagName() == "arg") args.append (elem.text()) ;
		child	= child.nextSibling() ;
	}

	for (KBPlayer *player = m_players.first() ; player != 0 ; player = m_players.next())
		if (player->playerClass() == _class)
			if ( ((type == "*") || (player->playerType() == type)) &&
			     ((name == "*") || (player->playerName() == name)) )
			{
				if (!player->playerPerform (action, args, pError))
					return	false	;

				return	true	;
			}


	pError	= KBError
		  (	KBError::Error,
			"No matching player found in score",
			QString	("%1:%2:%3 - seq %4")
				.arg(_class)
				.arg(type  )
				.arg(name  )
				.arg(seq   ),
			__ERRLOCN
		  )	;
#endif
	return	false	;
}

/*  KBConductor								*/
/*  doSnapshot	: Get snapshot						*/
/*  pError	: KBError &	  : Error return				*/
/*  fileName	: const QString & : Snapshot file name			*/
/*  (returns)	: bool		  : Success				*/

bool	KBConductor::doSnapshot
	(	KBError		&pError,
		const QString	&fileName
	)
{
#if	__KB_KDE
	QString		_class	= m_stanza.attribute  ("class"   ) ;
	QString		type	= m_stanza.attribute  ("type"    ) ;
	QString		name	= m_stanza.attribute  ("name"    ) ;
	QString		seq	= m_stanza.attribute  ("seq"     ) ;

	for (KBPlayer *player = m_players.first() ; player != 0 ; player = m_players.next())
		if (player->playerClass() == _class)
			if ( ((type == "*") || (player->playerType() == type)) &&
			     ((name == "*") || (player->playerName() == name)) )
			{
				if (!snapshot (player, fileName, pError))
						return	false	;

				return	true	;
			}


	pError	= KBError
		  (	KBError::Error,
			"No matching player found in score",
			QString	("%1:%2:%3 - seq %4")
				.arg(_class)
				.arg(type  )
				.arg(name  )
				.arg(seq   ),
			__ERRLOCN
		  )	;
#endif
	return	false	;
}

/*  KBConductor								*/
/*  nextStanza	: Play the next stanza element				*/
/*  (returns)	: void		:					*/

void	KBConductor::nextStanza ()
{
#if	__KB_KDE
	KBError	error	;

	while (!m_stanza.isNull())
	{
		int	leadin	= m_stanza.attribute("leadin",  "").toInt() ;
		int	pause	= m_stanza.attribute("pause",   "").toInt() ;
		int	seq	= m_stanza.attribute("seq",     "").toInt() ;
		QString	preSnap	= m_stanza.attribute("presnap", "") ;
		QString	postSnap= m_stanza.attribute("postsnap","");

		switch (m_state)
		{
			case StateIdle	 :
				/* Idle, nothing to do, silently ignore	*/
				/* the call.				*/
				return	;

			case StateLeadin :
				/* Leadin time. First see if there is	*/
				/* a dialog and if so whether there is	*/
				/* a message.				*/
				if (m_withDlg && sendMessage (seq))
				{
					if (m_state != StateIdle)
						m_state	= StatePreSnap ;
					return	;
				}

				/* If the time is non-zero then change	*/
				/* state and delay, else if zero then	*/
				/* drop through.			*/
				if (leadin > 0)
				{
					startDelay (leadin, StatePreSnap) ;
					return	;
				}

			case StatePreSnap :
				/* PreSnap. If there is a snapshot then	*/
				/* notify the dialog if present.	*/
				if (!preSnap.isEmpty() && m_withDlg)
				{
					sendSnapshot (seq)    ;
					m_state	= StateStanza ;
					return	;
				}

			case StateStanza :
				/* Stanza state. First see if there is	*/
				/* a pre-snapshot.			*/
				if (!preSnap.isEmpty ())
					if (!doSnapshot (error, preSnap))
						error.DISPLAY () ;

				/* As the action may not return at once	*/
				/* (ok, dialogs) we have to use a timer	*/
				/* to move on and must do this first.	*/
				startDelay (pause > 0 ? pause : 0, StatePostSnap) ;

				if (!doPerform(error))
				{
					m_score .clear ()   ;
					m_stanza.clear ()   ;
					m_state	= StateIdle ;
					error.DISPLAY  ()   ;
					return	;
				}

				qApp->processEvents () ;
				return	;

			case StatePostSnap :
				/* PreSnap. If there is a snapshot then	*/
				/* notify the dialog if present.	*/
				if (!postSnap.isEmpty() && m_withDlg)
				{
					sendSnapshot (seq)    ;
					m_state	= StatePause  ;
					return	;
				}

			case StatePause	 :
				/* Pause state. First see if there is	*/
				/* a pre-snapshot.			*/
				if (!postSnap.isEmpty ())
					if (!doSnapshot (error, postSnap))
						error.DISPLAY () ;

				/* It is here that we move on through	*/
				/* the score, checking for the end.	*/
				if (m_stanza.nextSibling().isNull())
				{	m_score .clear ()   ;
					m_stanza.clear ()   ;
					m_state	= StateIdle ;
					return	;
				}

				m_stanza = m_stanza.nextSibling().toElement () ;
				m_state	 = StateLeadin ;

				if (m_withDlg)
				{
					sendReady (seq) ;
					return	  ;

				}
				    
				break	;

			default	:
				break	;
		}
	}
#endif
	return	;
}


KBConductor
	*KBConductor::self ()
{
	return	conductor == 0 ? new KBConductor() : conductor ;
}


/*  ------------------------------------------------------------------  */

/*  KBPlayer								*/
/*  KBPlayer	: Constructor for player object for scrores		*/
/*  class	: const QString & : Class of player			*/
/*  type	: const QString & : Type of player			*/
/*  widget	: QWidget *	  : Associated widget if any		*/
/*  node	: KBNode  *	  : Associated node if any		*/

KBPlayer::KBPlayer
	(	const QString	&_class,
		const QString	&_type,
		QWidget		*widget,
		KBNode		*node
	)
	:
	m_class	(_class),
	m_type	(_type),
	m_widget(widget),
	m_node	(node)
{
#if	__KB_KDE
	KBConductor::self()->addPlayer (this) ;
#endif
}

/*  KBPlayer								*/
/*  KBPlayer	: Destructor for player object for scrores		*/
/*  (returns)	:		:					*/

KBPlayer::~KBPlayer ()
{
#if	__KB_KDE
	KBConductor::self()->remPlayer (this) ;
#endif
}

QString	KBPlayer::playerName ()
{
	return	"notNamed" ;
}

bool	KBPlayer::playerPerform
	(	const QString		&action,
		const QStringList	&,
		KBError			&pError
	)
{
#if	__KB_KDE
	QString	info	;

	if (m_node == 0)
		info	= QString ("%1: %2")
				  .arg(playerName())
				  .arg(action) ;
	else
		info	= QString ("%1.%2 (%3): %4")
				  .arg(m_node->getElement())
				  .arg(playerName())
				  .arg(m_node->showingDesign() ? "design" :
				       m_node->showingData  () ? "data"   : "unknown")
				  .arg(action) ;

	pError	= KBError
		  (	KBError::Error,
			"Unknown or inappropriate action",
			info,
			__ERRLOCN
		  )	;
#endif
	return	false	;
}

