/***************************************************************************
    file	         : kb_copyfile.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<errno.h>
#include	<qdom.h>

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_location.h"

#include	"kb_param.h"

#include	"kb_copybase.h"
#include	"kb_copyfile.h"


/*  IOError	: Convert QT IO error status to message			*/
/*  status	: int		: Error status				*/
/*  (returns)	: QString	: Error message				*/

static	QString	IOError
	(	int	status
	)
{
	cchar	*qterr	;

	switch (status)
	{
		case IO_Ok		: qterr	= TR("No error")	; break ;
		case IO_ReadError	: qterr	= TR("Read error")	; break ;
		case IO_WriteError	: qterr	= TR("Write error")	; break ;
		case IO_FatalError	: qterr	= TR("Fatal error")	; break ;
		case IO_OpenError	: qterr	= TR("Open error")	; break ;
		case IO_AbortError	: qterr	= TR("Abort")		; break ;
		case IO_TimeOutError	: qterr	= TR("Time-out error")	; break ;
		default			: qterr	= TR("Unknown error")	; break ;
	}

	return	QString("%1: %2").arg(qterr).arg(strerror(errno)) ;
}

/*  QCharToQString							*/
/*  		: Convert character to string				*/
/*  ch		: QChar		: Character				*/
/*  (returns)	: QString	: String				*/

static /*LIBKBASE_API*/	QString	QCharToQString
	(	const QChar	&ch
	)
{
	/* This is needed else the XML generation fouls up on null	*/
	/* characters.							*/
	return	ch.isNull() ? QString("") : QString(ch) ;
}


/*  KBCopyFile								*/
/*  KBCopyFile	: Constructor for table copier				*/
/*  srce	: bool		: Source, else destination		*/
/*  location	: KBLocation &	: Database location			*/
/*  (returns)	: KBCopyFile	:					*/

KBCopyFile::KBCopyFile
	(	bool		srce,
		KBLocation	&location
	)
	:
	m_srce	   (srce),
	m_location (location)
{
	m_local	= 0 ;
}

/*  KBCopyFile								*/
/*  ~KBCopyFile: Destructor for table copier				*/
/*  (returns)	:		:					*/

KBCopyFile::~KBCopyFile ()
{
	if (m_local != 0) delete [] m_local ;
}

/*  KBCopyFile								*/
/*  tag		: Get tag for XML output				*/
/*  (returns)	: cchar *	: Tag					*/

cchar	*KBCopyFile::tag ()
{
	return	"file"	;
}

/*  KBCopyFile								*/
/*  setWhich	: Set which (fixed/delimited) is selected		*/
/*  which	: uint		: Which					*/
/*  (returns)	: void		:					*/

void	KBCopyFile::setWhich
	(	uint		which
	)
{
	m_which	= which ;
}

/*  KBCopyFile								*/
/*  setErrOpt	: Set line error option					*/
/*  erropt	: uint		: Option				*/
/*  (returns)	: void		:					*/

void	KBCopyFile::setErrOpt
	(	uint		erropt
	)
{
	m_erropt = erropt ;
}

/*  KBCopyFile								*/
/*  setFile	: Set source/destination file name			*/
/*  file	: const QString & : File name				*/
/*  (returns)	: void		  :					*/

void	KBCopyFile::setFile
	(	const QString	&file
	)
{
	m_file	= file  ;
}

/*  KBCopyFile								*/
/*  setDelim	: Set delimiter string					*/
/*  delim	: const QString & : Delimiter				*/
/*  (returns)	: void		  :					*/

void	KBCopyFile::setDelim
	(	const QString	&delim
	)
{
	m_delim	= delim == "<tab>" ? QChar('\t') : delim[0] ;
}

/*  KBCopyFile								*/
/*  setQualif	: Set qualifier string					*/
/*  qualif	: const QString & : Qualifier				*/
/*  (returns)	: void		  :					*/

void	KBCopyFile::setQualif
	(	const QString	&qualif
	)
{
	m_qualif = qualif[0] ;
}

/*  KBCopyFile								*/
/*  setCopyList	: Set cipy list string					*/
/*  qualif	: const QString & : Qualifier				*/
/*  (returns)	: void		  :					*/

void	KBCopyFile::setCopyList
	(	const QString	&copyList
	)
{
	m_copyList = copyList ;
}

/*  KBCopyFile								*/
/*  setHeader	: Set header option					*/
/*  header	: bool		: Headers enabled			*/
/*  skip	: uint		: Header skip count			*/
/*  (returns)	: void		:					*/

void	KBCopyFile::setHeader
	(	bool		header,
		uint		skip
	)
{
	m_header = header ;
	m_skip	 = skip	  ;
}

/*  KBCopyFile								*/
/*  addField	: Add fixed field definition				*/
/*  name	: const QString & : Nominal name			*/
/*  offset	: uint		  : Field offset			*/
/*  width	: uint		  : Field width				*/
/*  strip	: bool		  : Strip white space			*/
/*  (returns)	: void		  :					*/

void	KBCopyFile::addField
	(	const QString	&name,
		uint		offset,
		uint		width,
		bool		strip
	)
{
	m_names  .append (name  ) ;
	m_offsets.append (offset) ;
	m_widths .append (width ) ;
	m_strip  .append (strip ) ;
}

/*  KBCopyFile								*/
/*  getField	: Get specified field					*/
/*  idx		: uint		: Field index				*/
/*  name	: QString &	: Return notional name			*/
/*  offset	: uint &	: Return offset				*/
/*  width	: uint &	: Return width				*/
/*  strip	: bool &	: Strip white space			*/
/*  (returns)	: bool		: Indexed field exists			*/

bool	KBCopyFile::getField
	(	uint		idx,
		QString		&name,
		uint		&offset,
		uint		&width,
		bool		&strip
	)
{
	if (idx < m_names.count())
	{
		name	= *m_names.at(idx) ;
		offset	= m_offsets  [idx] ;
		width	= m_widths   [idx] ;
		strip	= m_strip    [idx] ;
		return	true	;
	}

	return	false	;
}

/*  KBCopyFile								*/
/*  set		: Set copier from definition				*/
/*  copy	: QDomElement & : Definition parent			*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: bool		: Success				*/

bool	KBCopyFile::set
	(	QDomElement	&copy,
		KBError		&
	)
{
	QDomElement	element	= copy.namedItem(tag()).toElement() ;

	if (element.isNull ()) return true ;
//	{
//		pError	= KBError
//			  (	KBError::Error,
//				QString("Document lacks %1 file part").arg(srce ? "source" : "destination"),
//				QString::null,
//				__ERRLOCN
//			  )	;
//		return	false	;
//	}

	reset	()	;

	setWhich   (element.attribute ("which"   ).toUInt()) ;
	setErrOpt  (element.attribute ("erropt"  ).toUInt()) ;
	setDelim   (element.attribute ("delim"   )) ;
	setQualif  (element.attribute ("qualif"  )) ;
	setCopyList(element.attribute ("copylist")) ;
	setFile    (element.attribute ("file"    )) ;

	setHeader 
	(	element.attribute ("header"  ).toUInt(),
		element.attribute ("skip"    ).toUInt()
	)	;

	QDomNodeList	dFields	= element.elementsByTagName ("field") ;
	for (uint idx = 0 ; idx < dFields.length() ; idx += 1)
	{	QDomElement e = dFields.item(idx).toElement() ;
		m_names  .append (e.attribute("name"  )) ;
		m_offsets.append (e.attribute("offset").toUInt()) ;
		m_widths .append (e.attribute("width" ).toUInt()) ;
		m_strip  .append (e.attribute("strip" ).toUInt()) ;
	}

	return	true	;
}

/*  KBCopyFile								*/
/*  valid	: Check validity					*/
/*  error	: KBError &	: Error return				*/
/*  (returns)	: bool		: Valid					*/

bool	KBCopyFile::valid
	(	KBError		&error
	)
{
	if (m_file.isEmpty())
	{
		error	= KBError
			  (	KBError::Error,
				TR("No source or destination file specified"),
				QString::null,
				__ERRLOCN
			  )	;
		return	false	;
	}

	if (m_which == UseDelimited)
	{
		if (m_delim.isNull())
		{
			error	= KBError
				  (	KBError::Error,
					TR("No delimiter set"),
					QString::null,
					__ERRLOCN
		 		  )	;
			return	false	;
		}

		return	true	;
	}

	if (m_which == UseFixed)
	{
		for (uint idx1 = 0 ; idx1 < m_names.count() ; idx1 += 1)
			if (m_widths.at(idx1) == 0)
			{
				error	= KBError
					  (	KBError::Error,
						TR("Zero-width fixed width field"),
						QString::null,
						__ERRLOCN
					  )	;
				return	false	;
			}

		bool	olap	= false	;
		for (uint idx2 = 0 ; idx2 < m_names.count() ; idx2 += 1)
			for (uint idx3 = 0 ; idx3 < m_names.count() ; idx3 += 1)
				if (idx2 != idx3)
					olap |= (m_offsets[idx2] + m_widths[idx2] > m_offsets[idx3]) &&
					    	(m_offsets[idx3] + m_widths[idx3] > m_offsets[idx2]) ;

		if (olap)
			KBError::EWarning
			(	TR("Some fixed-width fields overlap"),
				QString::null,
				__ERRLOCN
			)	;

		return	true	;
	}


	error	= KBError
		  (	KBError::Error,
			TR("File setting neither delimited nor fixed width"),
			QString::null,
			__ERRLOCN
		  )	;
	return	false	;
}

/*  KBCopyFile								*/
/*  def		: Get table copy definition				*/
/*  copy	: QDomElement &	: Element to which to attach		*/
/*  (returns)	: void		:					*/

void	KBCopyFile::def
	(	QDomElement	&copy
	)
{
	QDomElement	element	;

	copy   .appendChild  (element = copy.ownerDocument().createElement (tag())) ;

	element.setAttribute ("which",    m_which   ) ;
	element.setAttribute ("erropt",   m_erropt  ) ;
	element.setAttribute ("delim",    QCharToQString(m_delim )) ;
	element.setAttribute ("qualif",   QCharToQString(m_qualif)) ;
	element.setAttribute ("copylist", m_copyList) ;
	element.setAttribute ("file",     m_file    ) ;
	element.setAttribute ("header",   m_header  ) ;
	element.setAttribute ("skip",     m_skip    ) ;

	for (uint idx = 0 ; idx < m_names.count() ; idx += 1)
	{
		QDomElement	dField	;
		element.appendChild  (dField = element.ownerDocument().createElement ("field")) ;
		dField.setAttribute ("name",   *m_names.at(idx)) ;
		dField.setAttribute ("offset", m_offsets  [idx]) ;
		dField.setAttribute ("width",  m_widths   [idx]) ;
		dField.setAttribute ("strip",  m_strip    [idx]) ;
	}
}

/*  KBCopyFile								*/
/*  getColumnNames							*/
/*		: Get column names					*/
/*  names	: QStringList &	: List of column names			*/
/*  (returns)	: void		:					*/

void	KBCopyFile::getColumnNames
	(	QStringList	&names
	)
{
	if (m_header && m_srce)
		names	= m_columns ;
	else	names	= m_names   ;
}

/*  KBCopyFile								*/
/*  prepare	: Prepare for execution					*/
/*  paramDict	: const QDict<QString> &				*/
/*				: Paramater dictionary			*/
/*  other	: KBCopyBase *	: Other half of copier			*/
/*  (returns)	: bool		: Success				*/

bool	KBCopyFile::prepare
	(	const QDict<QString>	&paramDict,
		KBCopyBase		*other
	)
{
	m_io.close   ()	  ;
	m_io.setName (m_useFile = paramSub (m_file, paramDict)) ;

	if (!m_io.open (m_srce ? IO_ReadOnly : IO_WriteOnly|IO_Truncate))
	{
		m_lError = KBError
			   (	KBError::Error,
				QString (TR("Unable to open \"%1\"")).arg(m_useFile),
				IOError (m_io.status()),
				__ERRLOCN
			   )	;
		return	false	;
	}

	m_stream.setDevice (&m_io) ;

	m_nRows	 = 0 ;
	m_maxlen = 0 ;
	for (uint idx0 = 0 ; idx0 < m_names.count() ; idx0 += 1)
		if (m_offsets[idx0] + m_widths[idx0] > m_maxlen)
			m_maxlen = m_offsets[idx0] + m_widths[idx0] ;


	if (m_srce)
	{
		/* If using a delimited source file and there is a non-	*/
		/* empty copy list then convert this to a copy map, ie.	*/
		/* a list of column numbers to be copied.		*/
		if ((m_which != UseFixed) && !m_copyList.isEmpty())
		{
			m_copyMap.clear  () ;

			QStringList cl = QStringList::split (',', m_copyList) ;
			for (uint s = 0 ; s < cl.count() ; s += 1)
			{
				QStringList cp = QStringList::split ('-', cl[s]) ;
				uint	    fr = cp[0].toUInt() ;
				uint	    to = cp.count() > 1 ? cp[1].toUInt() : fr ;

				if (fr > 0) fr -= 1 ;
				if (to > 0) to -= 1 ;

				if (to >= 2048)
				{	m_lError = KBError
						   (	KBError::Error,
							"Copy list value exceeds 2048",
							QString::null,
							__ERRLOCN
						   )	;
					return	false	;
				}

				while (fr != to)
				{	m_copyMap.append (fr) ;
					fr = fr > to ? fr - 1 : fr + 1 ;
				}
				m_copyMap.append (fr) ;
			}

			fprintf	(stderr, "<<<< CopyList\n") ;
			for (uint j = 0 ; j < m_copyMap.count() ; j += 1)
				fprintf (stderr, "     %d\n", m_copyMap[j]) ;
			fprintf	(stderr, ">>>> CopyList\n") ;
		}

		/* If the first source line is a header then parse this	*/
		/* to get a set of column headers.			*/
		if (m_header)
		{
			KBValue	dummy	[1024] ;
			int	aCols	;

			m_columns.clear	  () ;
			m_buffer = m_stream .readLine() ;

			if	(m_which == UseFixed) aCols = fixedScan  (dummy, 1024) ;
			else if (m_qualif.isNull  ()) aCols = delimScan  (dummy, 1024) ;
			else			      aCols = qualifScan (dummy, 1024) ;

			if (aCols <= 0)
			{
				m_lError = KBError
					   (	KBError::Error,
						TR("Over 1024 (or no) columns in header line"),
						QString::null,
						__ERRLOCN
					   )	;
				return	 false	;
			}

			/* Check the copy map if any to remove any	*/
			/* entries which exceed the number of columns	*/
			/* that actually exist.				*/
			QValueListIterator<uint> iter = m_copyMap.begin() ;
			while (iter != m_copyMap.end())
				if ((int)(*iter) >= aCols)
					iter  = m_copyMap.remove (iter) ;
				else	iter ++ ;

			if (m_copyMap.count() == 0)
				for (int idx1 = 0 ; idx1 < aCols ; idx1 += 1)
					m_columns.append (dummy[idx1].getRawText().stripWhiteSpace()) ;
			else
				for (uint idx2 = 0 ; idx2 < m_copyMap.count() ; idx2 += 1)
					m_columns.append (dummy[m_copyMap[idx2]].getRawText().stripWhiteSpace()) ;
		}

		/* If the skip count is set the gobble the specified	*/
		/* number of lines.					*/
		if (m_skip > 0)
			for (uint l = 0 ; l < m_skip ; l += 1)
				m_stream.readLine () ;

//		fprintf	(stderr, "<<<< CopyList\n") ;
//		for (uint j = 0 ; j < m_copyMap.count() ; j += 1)
//			fprintf (stderr, "     %d [%s]\n", m_copyMap[j], (cchar *)m_columns[j]) ;
//		fprintf	(stderr, ">>>> CopyList\n") ;
	}

	if (m_header && !m_srce)
	{
		QString		buff	  ;
		QStringList	srceNames ;
		other->getColumnNames (srceNames) ;

		if (m_which == UseFixed)
		{

			buff.fill (' ', m_maxlen) ;

			for (uint idx = 0 ; idx < m_names.count() ; idx += 1)
			{
				QString	text    = srceNames[idx] ;
				uint	offset	= m_offsets[idx] ;
				uint	width	= m_widths [idx] ;

				while (text.length() < width) text.append (' ') ;
				text.truncate (width) ;

				buff.replace  (offset, width, text) ;
			}
		}
		else
		{

			for (uint idx = 0 ; idx < srceNames.count() ; idx += 1)
			{
				if (idx > 0) buff += m_delim ;

				if (!m_qualif.isNull()) buff += m_qualif ;
				buff += srceNames[idx];
				if (!m_qualif.isNull()) buff += m_qualif ;
			}
		}

		buff.append ('\n') ;
		m_stream << buff   ;
	}

	return	true	;
}

/*  KBCopyFile								*/
/*  getNumCols	: Get number of columns in copy				*/
/*  (returns)	: int		: Number of columns			*/

int	KBCopyFile::getNumCols ()
{
	if (m_which == UseFixed)
		return	m_names  .count() ;

	if (m_srce)
		return	m_copyMap.count() ;

	return	0 ;
}

/*  KBCopyFile								*/
/*  fixedScan	: Scan fixed format input				*/
/*  values	: KBValue *	: Vector for values			*/
/*  ncols	: uint		: Size of value vector			*/
/*  (returns)	: int		: Actual number of columns		*/

int	KBCopyFile::fixedScan
	(	KBValue		*values,
		uint		
	)
{
	bool	lineErr	= false ;

	/* Extract each field. If any field lies all or partly outside	*/
	/* the line then set the value to null and set the line error	*/
	/* flag.							*/
	for (uint idx = 0 ; idx < m_names.count() ; idx += 1)
	{
		uint	offset	= m_offsets[idx] ;
		uint	width	= m_widths [idx] ;

		if (m_buffer.length() < offset + width)
		{
			values[idx]	= KBValue() ;
			lineErr		= true	;
			continue ;
		}

		if (m_strip[idx])
			values[idx] = KBValue (m_buffer.mid(offset, width).stripWhiteSpace()) ;
		else	values[idx] = KBValue (m_buffer.mid(offset, width)) ;
	}

	/* If there was an error (a line did not encompas a field) then	*/
	/* see what to do ....						*/
	if (lineErr) switch (m_erropt)
	{
		case ErrAbort	:
			/* Abort: Set an error message and return	*/
			/* a negative result, indicating an error.	*/
			m_lError = KBError
				   (	KBError::Error,
					TR("Short source line"),
					QString::null,
					__ERRLOCN
				   )	;
			return	-1	;

		case ErrSkip	:
			/* Skip: This line will be ignored ...		*/
			return	0	;

		default	:
			/* The other case is pad with nulls, which will	*/
			/* have already been done in the loop.		*/
			break	;
	}

	return	m_names.count () ;
}

/*  KBCopyFile								*/
/*  delimScan	: Scan delimited input					*/
/*  values	: KBValue *	: Vector for values			*/
/*  nCols	: uint		: Size of value vector			*/
/*  (returns)	: int		: Actual number of columns		*/

int	KBCopyFile::delimScan
	(	KBValue		*values,
		uint		nCols
	)
{
	uint	bptr	= 0	;
	uint	aCols	= 0	;
	bool	allused	= false	;

	while ((bptr < m_buffer.length()) && (aCols < nCols))
	{
		int	eof	= m_buffer.find (m_delim, bptr) ;

		if (eof < 0)
		{	values[aCols] = KBValue(m_buffer.mid(bptr)) ;
			aCols	   += 1    ;
			allused	    = true ;
			break ;
		}

		values[aCols] = KBValue(m_buffer.mid(bptr, eof - bptr)) ;
		bptr	    = eof + 1 ;
		aCols	   += 1	 ;
	}

	if (!allused) switch (m_erropt)
	{
		case ErrAbort	:
			/* Abort: Set an error message and return	*/
			/* a negative result, indicating an error.	*/
			m_lError = KBError
				   (	KBError::Error,
					TR("Source line has excess data"),
					QString::null,
					__ERRLOCN
				   )	;
			return	-1	;

		case ErrSkip	:
			/* Skip: This line will be ignored ...		*/
			return	0	;

		default	:
			/* The other case is to ignore excess data, in	*/
			/* which case we will return whatever we have.	*/
			break	;
	}

	return	aCols ;
}

/*  KBCopyFile								*/
/*  nextQualified: Get next qualified token				*/
/*  bptr	 : uint &	: Buffer pointer			*/
/*  (returns)	 : QString	: Token or null on error		*/

QString	KBCopyFile::nextQualified
	(	uint		&bptr
	)
{
	/* See if the next character is a qualifier, in which case we	*/
	/* have a qualified string. If not then it is easy, just look	*/
	/* for the next delimiter.					*/
	if (m_buffer[bptr] != m_qualif)
	{
		int	bptr2	= m_buffer.find (m_delim, bptr) ;

		if (bptr2 < 0) bptr2 = m_buffer.length () ;
 
		QString	res	= m_buffer.mid (bptr, bptr2 - bptr) ;
		bptr	= bptr2	;
		return	res	;
	}

	bptr	+= 1	;

	/* OK, need to handle the qualified case. This may straddle	*/
	/* onto subsequent lines, so we will accumulate the result as	*/
	/* we go. Prime the loop by looking for the next qualifier.	*/
	QString	text	= ""	;
	int	bptr2	= m_buffer.find (m_qualif, bptr) ;

	for (;;)
	{
		/* If there is no next qualifier then append the reset	*/
		/* of the buffer to the result, and read the next line.	*/
		/* Check here for and end-of-file, which is an error.	*/
		if (bptr2 < 0)
		{
			text	 += m_buffer.mid (bptr) ;
			m_buffer =  m_stream.readLine() ;

			if (m_buffer.isNull())
			{
				m_lError = KBError
					   (	KBError::Error,
					  	TR("Source field lacks trailing qualifier"),
				  		QString::null,
				 	 	__ERRLOCN
				 	   )	;
				return	QString::null ;
			}

			text   += "\n"	;
			bptr	= 0	;
			bptr2	= m_buffer.find (m_qualif) ;
			continue  ;
		}

		/* Append the text up to the next qualifier, and then	*/
		/* check for the double-qualifier case, which is a	*/
		/* single escaped qualifier.				*/
		text	+= m_buffer.mid (bptr, bptr2 - bptr) ;
		bptr	 = bptr2 ;

		if (m_buffer[bptr+1] == m_qualif)
		{	text	+= m_qualif ;
			bptr	+= 2 ;
			bptr2	 = m_buffer.find (m_qualif, bptr) ;
			continue   ;
		}


		/* Hit a single qualifier, so we are at done. Just	*/
		/* advance the pointer to the putative next delimiter.	*/
		bptr	+= 1 ;
		break	;
	
	}

	return	text	;
}

/*  KBCopyFile								*/
/*  qualifScan	: Scan delimited input with qualifiers			*/
/*  values	: KBValue *	: Vector for values			*/
/*  ncols	: uint		: Size of value vector			*/
/*  (returns)	: int		: Actual number of columns		*/

int	KBCopyFile::qualifScan
	(	KBValue		*values,
		uint		nCols
	)
{
	uint	bptr	= 0	;
	uint	aCols	= 0	;
	bool	allused	= false	;
	QString	text	;

	while ((bptr < m_buffer.length()) && (aCols < nCols))
	{
		QString	text	= nextQualified (bptr) ;

		values[aCols] = KBValue(text) ;
		aCols	+= 1 ;

		if (bptr >= m_buffer.length ())
		{	allused	= true	;
			break	;
		}

		if (m_buffer[bptr] != m_delim)
		{
			m_lError = KBError
				   (	KBError::Error,
				  	TR("Delimiter missing from source file"),
				  	QString::null,
				  	__ERRLOCN
				   )	;
			return	-1 ;
		}

		bptr	+= 1 ;
	}

	if (!allused) switch (m_erropt)
	{
		case ErrAbort	:
			/* Abort: Set an error message and return	*/
			/* a negative result, indicating an error.	*/
			m_lError = KBError
				   (	KBError::Error,
					TR("Source line has excess data"),
					QString::null,
					__ERRLOCN
				   )	;
			return	-1	;

		case ErrSkip	:
			/* Skip: This line will be ignored ...		*/
			return	0	;

		default	:
			/* The other case is to ignore excess data, in	*/
			/* which case we will return whatever we have.	*/
			break	;
	}

	return	aCols ;
}

/*  KBCopyFile								*/
/*  getRow	: Get next row						*/
/*  values	: KBValue *	: Value vector				*/
/*  nCols	: uint		: Number of columns available		*/
/*  ok		: bool &	: Execute OK				*/
/*  (returns)	: int		: Actual number of columns		*/

int	KBCopyFile::getRow
	(	KBValue		*values,
		uint		nCols,
		bool		&ok
	)
{
	KBValue	*vp	;

	/* Check that this is a source copier, if not then there is an	*/
	/* implementation error.					*/
	if (!m_srce)
	{
		m_lError = KBError
			   (	KBError::Fault,
				TR("Attempt to fetch row from destination copier"),
				QString::null,
				__ERRLOCN
			   )	;
		ok	= false	;
		return	-1	;
	}

	int	aCols	;

	if (m_copyMap.count() > 0)
	{
		if (m_local == 0) m_local = new KBValue[500] ;
		vp	= m_local ;
		nCols	= 500	  ;
	}
	else	vp	= values  ;

	for (;;)
	{
		m_buffer = m_stream.readLine() ;

		if (m_buffer.isNull())
		{	ok	= true	;
			return	-1	;
		}

		if	(m_which == UseFixed) aCols = fixedScan  (vp, nCols) ;
		else if (m_qualif.isNull  ()) aCols = delimScan  (vp, nCols) ;
		else			      aCols = qualifScan (vp, nCols) ;

		if (aCols > 0) break	;

		if (aCols < 0)
		{	ok	= false	;
			return	-1	;
		}
	}

	if (m_copyMap.count() > 0)
	{
		aCols	= m_copyMap.count() ;
		for (int idx = 0 ; idx < aCols ; idx += 1)
			values[idx] = m_local[m_copyMap[idx]] ;
	}

	ok	= true	;
	return	aCols	;
}

/*  KBCopyFile								*/
/*  putRow	: Put a row of data					*/
/*  values	: KBValue *	: Value vector				*/
/*  aCols	: uint		: Actual number of columns		*/
/*  (returns)	: bool		: Success				*/

bool	KBCopyFile::putRow
	(	KBValue		*values,
		uint		aCols
	)
{
	/* Check that this is a destination copier. An error here is an	*/
	/* implementation problem.					*/
	if (m_srce)
	{
		m_lError = KBError
			   (	KBError::Fault,
				TR("Attempt to insert row into source copier"),
				QString::null,
				__ERRLOCN
			   )	;
		return	false	;
	}

	/* If there are no values this is the initial call before any	*/
	/* real data arrives. Nothing to do here.			*/
	if (values == 0)
		return	true	;

	/* Check that we have the right number of columns and take	*/
	/* action accordingly if not. Note the for delimited output,	*/
	/* any number of columns is OK.					*/
	if ((getNumCols() != 0) && (getNumCols() != (int)aCols))
		switch (m_erropt)
		{
			case ErrSkip	:
				/* Just skip this record.		*/
				return	true	;

			case ErrAbort	:
				/* Abort copy with a suitable error.	*/
				m_lError = KBError
					   (	KBError::Error,
						TR("Insufficient output columns"),
						QString	(TR("Expected %1, got %2"))
							.arg(getNumCols())
							.arg(aCols),
						__ERRLOCN
					   )	;
				return	false	;

			default	:
				/* Otherwise pad with nulls. The copier	*/
				/* object will have done this for us.	*/
				break	;
		}


	QString	buff	;

	if (m_which == UseFixed)
	{
		buff.fill (' ', m_maxlen) ;

		for (uint idx = 0 ; idx < m_names.count() ; idx += 1)
		{
			QString	text	= idx >= aCols ? QString::null :
							 values [idx].getRawText() ;
			uint	offset	= m_offsets[idx] ;
			uint	width	= m_widths [idx] ;

			while (text.length() < width) text.append (' ') ;
			text.truncate (width) ;

			buff.replace  (offset, width, text) ;
		}
	}
	else
	{

		for (uint idx = 0 ; idx < aCols ; idx += 1)
		{
			if (idx > 0) buff += m_delim ;

			if (!m_qualif.isNull()) buff += m_qualif ;
			buff += values[idx].getRawText() ;
			if (!m_qualif.isNull()) buff += m_qualif ;
		}
	}

	buff.append ('\n') ;
	m_stream << buff   ;

	if (m_io.status() != IO_Ok)
	{
		m_lError = KBError
			   (	KBError::Error,
				QString (TR("Error writing to \"%1\"")).arg(m_file),
				IOError (m_io.status()),
				__ERRLOCN
		  	   )	;
		return	false	;
	}

	m_nRows	+= 1	;
	return	true	;
}

/*  KBCopyFile								*/
/*  finish	: Finish execution					*/
/*  report	: QString &	: Completion report			*/
/*  (returns)	: bool		: Success				*/

bool	KBCopyFile::finish
	(	QString		&report
	)
{
	m_io.close () ;

	if (m_io.status() != IO_Ok)
	{
		m_lError = KBError
			   (	KBError::Error,
				QString (TR("Error closing \"%1\"")).arg(m_file),
				IOError (m_io.status()),
				__ERRLOCN
		  	   )	;
		return	false	;
	}

	report	= QString("Copied %1 rows").arg(m_nRows) ;
	return	true	;
}

/*  KBCopyFile								*/
/*  (returns)	: void		:					*/

void	KBCopyFile::reset ()
{
	m_which	 = 0	    ;
	m_delim	 = QChar () ;
	m_qualif = QChar () ;
	m_header = false    ;

	m_names  .clear  () ;
	m_offsets.clear  () ;
	m_widths .clear  () ;
	m_strip  .clear  () ;
}

