/***************************************************************************
    file	         : kb_paramsetdlg.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdarg.h>
#include	<stdlib.h>

#include	<qapplication.h>

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_location.h"
#include	"kb_dbinfo.h"
#include	"kb_node.h"
#include	"kb_object.h"
#include	"kb_dbinfo.h"
#include	"kb_param.h"
#include	"kb_docroot.h"
#include	"kb_script.h"

#ifndef 	_WIN32
#include 	"kb_paramsetdlg.moc"
#else
#include 	"kb_paramsetdlg.h"
#endif

#include	"tk_messagebox.h"



/*  KBParamSetDlg							*/
/*  KBParamSetDlg: Constructor for parameter values dialog		*/
/*  caption	: QString	 : Dialog box caption			*/
/*  params	: QDict<KBParamSet>&					*/
/*				 : List of all paramaters		*/
/*  docRoot	: KBDocRoot *	 : Associated document root		*/
/*  pError	: KBerror &	 : Initialisation error return		*/
/*  ok		: bool &	 : Initialisation success/failure	*/
/*  (returns)	: KBParamSetDlg	 :					*/

KBParamSetDlg::KBParamSetDlg
	(	QString			caption,
		QDict<KBParamSet>	&params,
		KBDocRoot		*docRoot,
		KBError			&pError,
		bool			&ok
	)
	:
	_KBDialog	(caption, true),
	m_docRoot	(docRoot),
	m_layMain	(this),
	m_layParams	(&m_layMain),
	m_layButtons	(&m_layMain),
	m_bOK		(this),
	m_bCancel	(this)
{
	m_layMain   .setMargin  (4) ;
	m_layMain   .setSpacing (8) ;
	m_layParams .setMargin  (4) ;
	m_layParams .setSpacing (8) ;
	m_layButtons.setMargin  (4) ;
	m_layButtons.setSpacing (8) ;

	m_layButtons.addStretch	()         ;
	m_layButtons.addWidget 	(&m_bOK    ) ;
	m_layButtons.addWidget 	(&m_bCancel) ;


	m_bOK	    .setText	(TR("OK"    )) ;
	m_bCancel   .setText	(TR("Cancel")) ;
	m_bOK	    .show	() ;
	m_bCancel   .show	() ;

	connect (&m_bOK,     SIGNAL(clicked ()), SLOT(clickOK     ())) ;
	connect (&m_bCancel, SIGNAL(clicked ()), SLOT(clickCancel ())) ;

	QDictIterator<KBParamSet> iter (params) ;
	KBParamSet		  *pSet 	;
	uint			  row	 = 0	;

	while ((pSet = iter.current()) != 0)
	{
		/* If the parameter has been set and is not marked as	*/
		/* prompt-anyway, then it is not added to the dialog.	*/
		if (pSet->m_set && !pSet->m_prompt)
		{
			iter	+= 1 ;
			continue ;
		}

		/* If the parameter has been set the use the value that	*/
		/* was set, otherwise use the specified default.	*/
		QString	  defVal = pSet->m_set ? pSet->m_value : pSet->m_defval	;

		/* Just in case there is no legend, default it to the	*/
		/* paramater name.					*/
		if (pSet->m_legend.isEmpty())
			pSet->m_legend = iter.currentKey() ;


		/* Assuming that we have a document root, then if the	*/
		/* default value starts with the equals character then	*/
		/* evaluate it as a script expression.			*/
		if ((m_docRoot != 0) && (defVal.at(0) == '='))
		{
			defVal	= getScriptValue (defVal.mid(1), pError, ok) ;
			if (!ok) return ;
		}

		QLabel	   *label = new QLabel    ( this) ;
		TKLineEdit *value = new TKLineEdit (this) ;

		label->setText (pSet->m_legend) ;
		value->setText (defVal) ;

		m_layParams.addWidget (label, row, 0) ;
		m_layParams.addWidget (value, row, 1) ;

		m_params   .append    (pSet ) ;
		m_values   .append    (value) ;

		row	+= 1 ;
		iter	+= 1 ;
	}

	if (row > 0)
	{
		m_values.at(0)->setFocus() ;
		m_anyParams	= true	;
	}
	else	m_anyParams	= false ;

	ok	= true	  ;
}

/*  KBParamSetDlg							*/
/*  ~KBParamSetDlg: Destructor for control properties dialog		*/
/*  (returns)	  :		:					*/

KBParamSetDlg::~KBParamSetDlg ()
{
}

/*  KBParamSetDlg							*/
/*  getScriptValue							*/
/*		: Get value from script expression			*/
/*  expr	: const QString & : Expression				*/
/*  pError	: KBError &	  : Error return			*/
/*  ok		: bool &	  : Success				*/
/*  (returns)	: QString	  : Value				*/

QString	KBParamSetDlg::getScriptValue
	(	const QString	&expr,
		KBError		&pError,
		bool		&ok
	)
{
	/* A script interface is needed to be able to execute the	*/
	/* expression code.						*/
	KBScriptIF *scrIface = m_docRoot->loadScripting (ok, pError) ;

	if (!ok)
		return	QString::null	;

	if (scrIface == 0)
	{
		pError	= KBError
			  (	KBError::Error,
				TR("No scripting language specified"),
				QString(TR("Trying to execute default %1")).arg(expr),
				__ERRLOCN
			  )	;
		ok	= false		;
		return QString::null	;
	}

	QString		eText	;
	QString		ePatt	;
	KBScriptCode	*code	;

	if ((code = scrIface->compileExpr
		    (	expr,
			QString::null,
			eText,
			ePatt,
			m_docRoot->getImports(),
			pError
		    )	) == 0)
	{
		/* Compile error in the inline code ...			*/
		/* message, and return null.				*/
		ok	= false		;
		return	QString::null	;
	}

	/* Execute the code, which can then be deleted (it not being	*/
	/* worth keeping, as for, say, KBAttrExpr).			*/
	KBValue		resval	;
	KBScript::ExeRC rc	= code->execute (0, 0, 0, resval) ;
	delete		code	;

	if ((rc == KBScript::ExeError) || (rc == KBScript::ExeFail))
	{
		QString	   errMsg   ;
		uint	   errLno   ;
		QString	   errText  ;
		KBLocation errLocn  = scrIface->exeError (errMsg, errLno, errText) ;

		pError	= KBError
			  (	KBError::Error,
				errMsg,
				QString ("%1, line %2\n%3")
					.arg(errLocn.docName)
					.arg(errLno)
					.arg(errText),
				__ERRLOCN
			  )	;
		ok	= false		;
		return	QString::null	;
	}

	/* Success ....							*/
	ok	= true	;
	return	resval.getRawText ()	;
}

/*  KBParamSetDlg							*/
/*  clickOK	: Handle click of the OK button				*/
/*  (returns)	: void		:					*/

void	KBParamSetDlg::clickOK ()
{
	for (uint idx = 0 ; idx < m_values.count() ; idx += 1)
	{
		KBParamSet	*pSet	= m_params.at(idx) ;
		QString		text	= m_values.at(idx)->text() ;
		QStringList	format	= QStringList::split (':', pSet->m_format, true) ;

		fprintf
		(	stderr,
			"KBParamSetDlg::clickOK: [%s][%s][%s]\n",
			(cchar *)text,
			(cchar *)format[0],
			(cchar *)format[1]
		)	;

		if (format.count() < 2)
		{
			pSet->m_value = text	;
			continue ;
		}


		KBType	*type	= 0 ;

		if	(format[0] == "String"  ) type = &_kbString   ;
		else if	(format[0] == "Fixed"   ) type = &_kbFixed    ;
		else if	(format[0] == "Float"   ) type = &_kbFloat    ;
		else if	(format[0] == "Date"    ) type = &_kbDate     ;
		else if	(format[0] == "Time"    ) type = &_kbTime     ;
		else if	(format[0] == "DateTime") type = &_kbDateTime ;
		else if	(format[0] == "Bool"	) type = &_kbBool     ;
		
		if (type == 0)
		{
			TKMessageBox::sorry
			(	0,
				QString(TR("Unrecognised paramater format for %1: %2"))
					.arg(pSet->m_legend)
					.arg(pSet->m_format),
				TR("Parameter formatting error")
			)	;
			return	;
		}

		KBValue	value	(text, type) ;
		bool	ok	;

		QString	result	= value.getText (pSet->m_format, &ok) ;

		if (!ok)
		{
			TKMessageBox::sorry
			(	0,
				QString(TR("Value not value for %1: must be %2"))
					.arg(pSet->m_legend)
					.arg(type->getDescrip()),
				TR("Parameter value error")
			)	;
			return	;
		}

		pSet->m_value = result	;
	}

	done	(true) ;
}

/*  KBParamSetDlg							*/
/*  clickCancel	: Handle click of the Cancel button			*/
/*  (returns)	: void		:					*/

void	KBParamSetDlg::clickCancel ()
{
	done	(false) ;
}

