/***************************************************************************
    file	         : kb_dumper.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<errno.h>

#include	<qstring.h>
#include	<qfile.h>
#include	<qtextstream.h>
#include	<qlayout.h>
#include	<qtimer.h>

#include	"tk_messagebox.h"

#include	"kb_database.h"
#include	"kb_dbinfo.h"
#include	"kb_dblink.h"

#include	"kb_dialog.h"

#include	"kb_copybase.h"
#include	"kb_copytable.h"
#include	"kb_copyxml.h"
#include	"kb_copyexec.h"
#include	"kb_paramsetdlg.h"

#ifdef		_WIN32
#include	"kb_dumper.h"
#else
#include	"kb_dumper.moc"
#endif



class	LIBKBASE_API	KBCopyExecDumper : public KBCopyExec
{
	KBDumper	*m_dumper	;

public	:

	KBCopyExecDumper
	(	KBCopyBase	*,
		KBCopyBase	*,
		KBDumper	*
	)	;

	virtual	bool	showProgress	(int)	;
}	;

KBCopyExecDumper::KBCopyExecDumper
	(	KBCopyBase	*srce,
		KBCopyBase	*dest,
		KBDumper	*dumper
	)
	:
	KBCopyExec	(srce, dest),
	m_dumper	(dumper)
{
}

bool	KBCopyExecDumper::showProgress
	(	int		nRows
	)
{
	return	m_dumper->showProgress (nRows) ;
}

/*  ------------------------------------------------------------------  */

/*  KBDumper								*/
/*  KBDumper	: Constructor for database dumper			*/
/*  dbInfo	: KBDBInfo *	  : Database information		*/
/*  server	: const QString & : Server name				*/
/*  dir		: const QString & : Output directory			*/
/*  (returns)	: KBDumper	  :					*/

KBDumper::KBDumper
	(	KBDBInfo	*dbInfo,
		const QString	&server,
		const QString	&dir
	)
	:
	_KBDialog	(TR("Dump database"), true, "KBDumper"),
	m_dumpAll	(this),
	m_dumpDefn	(this),
	m_dumpData	(this),
	m_tableList	(this),
	m_dumping	(this),
	m_record	(this),
	m_of		(this),
	m_bOK		(this, "ok"    ),
	m_bCancel	(this, "cancel"),
	m_dbInfo	(dbInfo),
	m_server	(server),
	m_dir		(dir)
{
	QVBoxLayout	*layMain = new QVBoxLayout (this) ;

	layMain->addWidget (&m_dumpAll  ) ;
	layMain->addWidget (&m_dumpDefn ) ;
	layMain->addWidget (&m_dumpData ) ;
	layMain->addWidget (&m_tableList) ;

	QHBoxLayout	*layProg = new QHBoxLayout (layMain) ;
	QLabel		*l1	 = new QLabel	   (this) ;
	QLabel		*l2	 = new QLabel	   (this) ;
	QLabel		*l3	 = new QLabel	   (this) ;

	layProg->addWidget (l1) ;
	layProg->addWidget (&m_dumping) ;
	layProg->addWidget (l2) ;
	layProg->addWidget (&m_record ) ;
	layProg->addWidget (l3) ;
	layProg->addWidget (&m_of     ) ;

	QHBoxLayout	*layButt = new QHBoxLayout (layMain) ;

	layButt->addStretch()		;
	layButt->addWidget (&m_bOK    ) ;
	layButt->addWidget (&m_bCancel) ;

#if	! __KB_EMBEDDED
	m_dumping.setMinimumWidth (200) ;
#endif
	m_dumpAll   .setText (TR("Dump all objects"	)) ;
	m_dumpDefn  .setText (TR("Dump definitions"	)) ;
	m_dumpData  .setText (TR("Dump data"		)) ;

	m_dumping.setFrameStyle (QFrame::Box|QFrame::Plain) ;
	m_dumping.setLineWidth  (1) ;
	m_record .setFrameStyle (QFrame::Box|QFrame::Plain) ;
	m_record .setLineWidth  (1) ;
	m_of	 .setFrameStyle (QFrame::Box|QFrame::Plain) ;
	m_of	 .setLineWidth  (1) ;

	l1->setText 	(TR("Object"  )) ;
	l1->setAlignment(Qt::AlignRight) ;

	l2->setText	(TR("Record"  )) ;
	l2->setAlignment(Qt::AlignRight) ;

	l3->setText 	(TR("Object"  )) ;
	l3->setAlignment(Qt::AlignRight) ;

	m_cancelled	= false	;
	m_finished	= false ;
	m_index		= 0	;
	m_currItem	= 0	;

	m_tableList.addColumn(TR("Name")) ;
	m_tableList.addColumn(TR("Type")) ;
}

/*  KBDumper								*/
/*  accept	: User clicks OK button					*/
/*  (returns)	: void		:					*/

void	KBDumper::accept ()
{
	if (!m_finished)
	{
		if (!m_dumpDefn.isChecked() && !m_dumpData.isChecked())
		{
			TKMessageBox::sorry
			(	0,
				TR("Please select definition and/or data dumping"),
				TR("Dump Database")
			)	;
			return	;
		}

		if (!m_dumpAll.isChecked())
		{
			bool	any = false ;
			QCheckListItem *i = (QCheckListItem *)m_tableList.firstChild() ;
			while (i != 0)
				if (i->isOn())
				{	any	= true ;
					break	;
				}
				else	i = (QCheckListItem *)i->nextSibling() ;

			if (!any)
			{
				TKMessageBox::sorry
				(	0,
					TR("No tables selected for dumping"),
					TR("Dump Database")
				)	;
				return	;
			}
		}

		m_dumpAll   .setEnabled (false) ;
		m_dumpDefn  .setEnabled (false) ;
		m_dumpData  .setEnabled (false) ;
		m_tableList .setEnabled (false) ;
		m_bOK	    .setEnabled (false) ;

		slotTimer () ;
		return	;
	}

	done	(1) ;
}


/*  KBDumper								*/
/*  reject	: User clicks cancel button				*/
/*  (returns)	: void		:					*/

void	KBDumper::reject ()
{
	if (!m_cancelled)
	{
		m_cancelled = true ;
		done	(0) ;
	}
}

/*  KBDumper								*/
/*  exec	: Execute dumper					*/
/*  (returns)	: int		: Exit code				*/

int	KBDumper::exec	()
{
	/* See if there are any likely looking files already in the	*/
	/* target directory, in which case check with the user.		*/
	QDir	dir ;

	dir.setPath	  (m_dir)	  ;
	dir.setFilter	  (QDir::Files)   ;
	dir.setNameFilter ("*.tabledef;*.tabledata;*.viewdef;*.seqdef") ;
	dir.setSorting	  (QDir::Name)    ;

	if (dir.entryList().count() > 0)
		if (TKMessageBox::questionYesNo
			(	0,
				TR("Directory already contains database dump files: continue anyway?"),
				TR("Dump Database")
			)
			!= TKMessageBox::Yes
		   )
		   return  0 ; 


	/* Link to the database and get the list of tables here and now	*/
	/* If OK then fire off a timer to start the dump cycle and	*/
	/* enter the execution loop.					*/
	if (!m_dbLink.connect (m_dbInfo, m_server))
	{	m_dbLink.lastError().DISPLAY() ;
		return 0 ;
	}

	if (!m_dbLink.listTables (m_tables, KB::IsAny))
	{	m_dbLink.lastError().DISPLAY() ;
		return 0 ;
	}

	m_tableList.setSorting (0) ;
	for (uint idx = 0 ; idx < m_tables.count() ; idx += 1)
		new KBDumperItem (&m_tableList, &m_tables[idx]) ;

	m_currItem = (KBDumperItem *)m_tableList.firstChild() ;
	m_index	   = 0 ;

	return	_KBDialog::exec () ;
}

/*  KBDumper								*/
/*  dumpTableDef: Dump table definition					*/
/*  tabSpec	: KBTableSpec &	  : Table definition			*/
/*  dir		: const QString & : Destination directory		*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBDumper::dumpTableDef
	(	KBTableSpec	&tabSpec,
		KBError		&pError
	)
{
	QDomDocument	doc ("tablelist") ;
	doc.appendChild
	(	doc.createProcessingInstruction
		(	"xml",
			"version=\"1.0\" encoding=\"UTF-8\""
	)	)	;

	QDomElement	docElem	= doc.createElement ("tablelist") ;
	QDomElement	tabElem	= doc.createElement ("table"    ) ;

	doc    .appendChild (docElem) ;
	docElem.appendChild (tabElem) ;
	tabSpec.toXML	 (tabElem, 0) ;

	QString	eName	= m_dir + "/" + tabSpec.m_name + ".tabledef" ;
	QFile	eFile	(eName) ;
	if (!eFile.open (IO_WriteOnly|IO_Truncate))
	{
		pError	= KBError
			  (	KBError::Error,
				QString	(TR("Cannot open \"%1\"")).arg(eName),
				strerror(errno),
				__ERRLOCN
			)	;
		return	false	;
	}

	QTextStream(&eFile) << doc.toString() ;
	return	true	;
}

/*  KBDumper								*/
/*  dumpViewDef	: Dump view definition					*/
/*  viewSpec	: KBTableSpec &	  : View definition			*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBDumper::dumpViewDef
	(	KBTableSpec	&viewSpec,
		KBError		&pError
	)
{
	QDomDocument	doc ("viewlist") ;
	doc.appendChild
	(	doc.createProcessingInstruction
		(	"xml",
			"version=\"1.0\" encoding=\"UTF-8\""
	)	)	;

	QDomElement	docElem	 = doc.createElement ("viewlist") ;
	QDomElement	viewElem = doc.createElement ("view"    ) ;

	doc    .appendChild (docElem ) ;
	docElem.appendChild (viewElem) ;
	viewSpec.toXML	 (viewElem, 0) ;

	QString	eName	= m_dir + "/" + viewSpec.m_name + ".viewdef" ;
	QFile	eFile	(eName) ;
	if (!eFile.open (IO_WriteOnly|IO_Truncate))
	{
		pError	= KBError
			  (	KBError::Error,
				QString	(TR("Cannot open \"%1\"")).arg(eName),
				strerror(errno),
				__ERRLOCN
			)	;
		return	false	;
	}

	QTextStream(&eFile) << doc.toString() ;
	return	true	;
}

/*  KBDumper								*/
/*  dumpSequenceDef							*/
/*		: Dump sequence definition				*/
/*  seqSpec	: KBSequenceSpec & : Sequence definition		*/
/*  pError	: KBError &	   : Error return			*/
/*  (returns)	: bool		   : Success				*/

bool	KBDumper::dumpSequenceDef
	(	KBSequenceSpec	&seqSpec,
		KBError		&pError
	)
{
	QDomDocument	doc ("sequencelist") ;
	doc.appendChild
	(	doc.createProcessingInstruction
		(	"xml",
			"version=\"1.0\" encoding=\"UTF-8\""
	)	)	;

	QDomElement	docElem	= doc.createElement ("sequencelist") ;
	QDomElement	seqElem	= doc.createElement ("sequence"    ) ;

	doc    .appendChild (docElem) ;
	docElem.appendChild (seqElem) ;
	seqSpec.toXML	    (seqElem) ;

	QString	eName	= m_dir + "/" + seqSpec.m_name + ".seqdef" ;
	QFile	eFile	(eName) ;
	if (!eFile.open (IO_WriteOnly|IO_Truncate))
	{
		pError	= KBError
			  (	KBError::Error,
				QString	(TR("Cannot open \"%1\"")).arg(eName),
				strerror(errno),
				__ERRLOCN
			)	;
		return	false	;
	}

	QTextStream(&eFile) << doc.toString() ;
	return	true	;
}

/*  KBDumper								*/
/*  dumpTableData: Dump table data					*/
/*  tabSpec	: KBTableSpec &	  : Table definition			*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: bool		  : Success				*/

bool	KBDumper::dumpTableData
	(	KBTableSpec	&tabSpec,
		KBError		&pError
	)
{
	/* This is done using a copier. Create a dummy location and	*/
	/* thence a table source and an XML destination. These can be	*/
	/* appropriately initialised with the table structure.		*/
	KBLocation	location   (m_dbInfo, "copier", m_server, "unnamed") ;
	KBCopyTable	*copyTable = new KBCopyTable (true,  location) ;
	KBCopyXML	*copyXML   = new KBCopyXML   (false, location) ;

	copyTable->setServer  (m_server) ;
	copyTable->setTable   (tabSpec.m_name) ;
	copyTable->setOption  (KBCopyTable::OptReplace, "") ;

	copyXML	 ->setMainTag (tabSpec.m_name) ;
	copyXML	 ->setRowTag  ("row") ;
	copyXML	 ->setErrOpt  (KBCopyXML::ErrPad) ;
	copyXML	 ->setFile    (m_dir + "/" + tabSpec.m_name + ".tabledata") ;

	for (uint idx = 0 ; idx < tabSpec.m_fldList.count() ; idx += 1)
	{
		KBFieldSpec *fSpec = tabSpec.m_fldList.at(idx) ;

		copyTable->addField (fSpec->m_name) ;
		copyXML  ->addField (fSpec->m_name, false) ;
	}

	/* Now create the actual copier and execute it. Most of the	*/
	/* arguments are dummy in this situation.			*/
	KBCopyExecDumper  copy   (copyTable, copyXML, this) ;
	QString	   	  report ;
	int	   	  nRows  ;
	QDict<QString>	  d1	 ;
	QDict<KBParamSet> d2	 ;
	if (!copy.execute
		(	report,
			pError,
			nRows,
			d1, d2,
			false
		))
	{
		return	false	;
	}

	return	true	;
}

bool	KBDumper::dumpDetails
	(	KBTableDetails	*details
	)
{
	if (details->m_type == KB::IsTable)
	{
		KBTableSpec	tabSpec (details->m_name) ;
		KBError		error	;

		m_dumping.setText (details->m_name) ;
		m_record .setText ("") ;

		m_of.setText
		(	QString(TR("%1 of %2"))
				.arg(m_index + 1)
				.arg(m_tables.count())
		)	;

		m_tableList.ensureItemVisible (m_currItem) ;
		m_tableList.setCurrentItem    (m_currItem) ;

		qApp->processEvents() ;

		/* In the event of an error, display the error and	*/
		/* simulate the user clicking the cancel button, to	*/
		/* terminate the dialog.				*/
		if (!m_dbLink.listFields (tabSpec))
		{	m_dbLink.lastError().DISPLAY() ;
			return	false ;
		}

		qApp->processEvents() ;

		if (m_dumpDefn.isChecked())
			if (!dumpTableDef  (tabSpec, error))
			{	error.DISPLAY() ;
				return	false	;
			}

		qApp->processEvents() ;

		if (m_dumpData.isChecked())
			if (!dumpTableData (tabSpec, error))
			{	error.DISPLAY() ;
				return	false	;;
			}

		qApp->processEvents() ;
		return	true	;
	}

	if (details->m_type == KB::IsView)
	{
		KBTableSpec	tabSpec (details->m_name) ;
		KBError		error	;

		if (!m_dumpDefn.isChecked()) return true  ;

		m_dumping.setText (details->m_name) ;
		m_record .setText ("") ;

		m_of.setText
		(	QString(TR("%1 of %2"))
				.arg(m_index + 1)
				.arg(m_tables.count())
		)	;

		m_tableList.ensureItemVisible (m_currItem) ;
		m_tableList.setCurrentItem    (m_currItem) ;

		qApp->processEvents() ;

		/* In the event of an error, display the error and	*/
		/* simulate the user clicking the cancel button, to	*/
		/* terminate the dialog.				*/
		if (!m_dbLink.listFields (tabSpec))
		{	m_dbLink.lastError().DISPLAY() ;
			return	false	;
		}

		qApp->processEvents() ;

		if (!dumpViewDef  (tabSpec, error))
		{	error.DISPLAY() ;
			return	false	;
		}

		qApp->processEvents() ;
		return	true	;
	}

	if (details->m_type == KB::IsSequence)
	{
		KBSequenceSpec	seqSpec (details->m_name) ;
		KBError		error	;

		if (!m_dumpDefn.isChecked()) return true  ;

		m_dumping.setText (details->m_name) ;
		m_record .setText ("") ;

		m_of.setText
		(	QString(TR("%1 of %2"))
				.arg(m_index + 1)
				.arg(m_tables.count())
		)	;

		m_tableList.ensureItemVisible (m_currItem) ;
		m_tableList.setCurrentItem    (m_currItem) ;

		qApp->processEvents() ;

		/* In the event of an error, display the error and	*/
		/* simulate the user clicking the cancel button, to	*/
		/* terminate the dialog.				*/
		if (!m_dbLink.descSequence (seqSpec))
		{	m_dbLink.lastError().DISPLAY() ;
			return	false	;
		}

		qApp->processEvents() ;

		if (!dumpSequenceDef  (seqSpec, error))
		{	error.DISPLAY() ;
			return	false	;
		}

		qApp->processEvents() ;
		return	true	;
	}

	return	true	;
}

/*  KBDumper								*/
/*  slotTimer	: Start dump of next object				*/
/*  (returns)	: void		:					*/

void	KBDumper::slotTimer ()
{
	/* Look for the next object to dump. The loop will exit if we	*/
	/* run out of objects before finding a candidate.		*/
	while (m_currItem != 0)
	{
		KBTableDetails *details = m_currItem->details() ;
		bool		pause	= false	;

		if (m_dumpAll.isChecked() || m_currItem->isOn())
		{
			if (!dumpDetails (details))
			{
				reject	() ;
				return	;
			}

			pause	= true	;
		}

		m_currItem = (KBDumperItem *)m_currItem->nextSibling() ;
		m_index   += 1 ;

		if (pause)
		{
			QTimer::singleShot (200, this, SLOT(slotTimer())) ;
			qApp->processEvents() ;
			return	;
		}
	}

	/* All done. Change the cancel button to "OK" to indicate this	*/
	/* to the user, and set the finished flag. This leaves the	*/
	/* dialog just awaiting user confirmation before terminating.	*/
	m_bCancel.setEnabled (false) ;
	m_bOK    .setEnabled (true ) ;
	m_finished = true ;
}

bool	KBDumper::showProgress
	(	int		nRows
	)
{
	if (nRows % 10 == 0)
	{
		m_record.setText (QString::number(nRows)) ;
		qApp->processEvents() ;
	}

	return	m_cancelled ;
}

/*  dumpDatabase: Dump database to a set of files			*/
/*  dbInfo	: KBDBInfo *	  : Database information		*/
/*  server	: const QString & : Server name				*/
/*  dir		: const QString & : Destination directory		*/
/*  (returns)	: void		  :					*/

LIBKBASE_API	void	dumpDatabase
	(	KBDBInfo	*dbInfo,
		const QString	&server,
		const QString	&dir
	)
{
	KBDumper (dbInfo, server, dir).exec() ;
}
