/***************************************************************************
    file	         : kb_eventdlg.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/


#include	<qlayout.h>
#include	<qregexp.h>
#include	<qpopupmenu.h>

#include	"kb_classes.h"
#include	"kb_event.h"
#include	"kb_object.h"
#include	"kb_options.h"
#include	"kb_docroot.h"
#include	"kb_prompt.h"

#ifndef		_WIN32
#include	"kb_eventdlg.moc"
#else
#include	"kb_eventdlg.h"
#endif

#include	"tk_messagebox.h"


/*  checkCompile: Check script compiles OK				*/
/*  parent	: KBObject *	: Parent object for script code		*/
/*  text	: QString	: Script				*/
/*  func	: cchar *	: Script entry function			*/
/*  (returns)	: void		:					*/

bool	checkCompile
	(	KBObject	*parent,
		QString		text,
		cchar		*func
	)
{
	bool		ok	 ;
	KBError		error	 ;
	KBDocRoot	*root	 = parent->getRoot()->isDocRoot() ;

	/* Locate the scripting interface for the document. If there is	*/
	/* none, then report an error.	If we have one then verify that	*/
	/* the script compiles OK.					*/
	KBScriptIF	*iface	= root->getScriptIF
				  (	QString::null,
					ok,
					error
				  )	;

	if (!ok)
	{	error.DISPLAY() ;
		return	false	;
	}

	QString		eText	;
	QString		ePatt	;
	KBScriptCode	*code	;

	if ((code = iface->compileFunc
		    (	text,
			parent->getPath(),
			func,
			eText,
			ePatt,
			root->getImports(),
			0,
			error
		    )	) == 0)
	{
		error.DISPLAY()	;
		return	false	;
	}

	/* Hey! Sucess!							*/
	DELOBJ	(code)	;
	return	true	;
}

/*  ------------------------------------------------------------------  */

/*  KBAttrEventItem							*/
/*  KBAttrEventItem							*/
/*		: Constructor for event attribute item			*/
/*  event	: KBEvent *	  : Associated event			*/
/*  (returns)	: KBAttrEventItem :					*/

KBAttrEventItem::KBAttrEventItem
	(	KBEvent		*event
	)
	:
	KBAttrItem	(event),
	m_event		(event)
{
	KBMacroExec *m	= event->getMacro    () ;
	m_macro		= m == 0 ? 0 : new KBMacroExec (m) ;
	m_breakpoints	= event->breakpoints () ;

	for (uint idx = 0 ; idx < m_breakpoints.count() ; idx += 1)
		fprintf
		(	stderr,
			"KBAttrEventItem::KBAttrEventItem: breakpoint %d\n",
			m_breakpoints[idx]
		)	;
}

/*  KBAttrEventItem							*/
/*  ~KBAttrEventItem							*/
/*		: Destructor for event attribute item			*/
/*  (returns)	: void		:					*/

KBAttrEventItem::~KBAttrEventItem ()
{
	if (m_macro != 0) delete m_macro ;
}

/*  KBAttrEventItem							*/
/*  displayValue: Get display value for event				*/
/*  (returns)	: QString	: Display text	*			*/

QString	KBAttrEventItem::displayValue ()
{
	return	m_macro == 0 ?
			KBAttrItem::displayValue () :
			QString("[Macro]") ;
}

/*  KBAttrEventItem							*/
/*  getAttrDlg	: Get attribute dialog for this item			*/
/*  parent	: QWidget *		: Parent widget			*/
/*  attrDict	: QDict<KBAttrItem> &	: Disctionary of all attributes	*/
/*  (returns)	: KBAttrDlg *		:				*/

KBAttrDlg
	*KBAttrEventItem::getAttrDlg
	(	QWidget			*parent,
		QDict<KBAttrItem>	&attrDict
	)
{
	return	new KBEventDlg (parent, m_event, this, attrDict) ;
}

/*  KBAttrEventItem							*/
/*  save	: Save value back to event				*/
/*  (returns)	: void		:					*/

void	KBAttrEventItem::save ()
{
	if (m_macro != 0)
	{	m_event->setMacro (m_macro) ;
		m_macro	= 0   ;
		return	;
	}

	m_event->setMacro (0) ;

	QString	text	= KBEvent::trimEvent (m_value) ;

	/* Check whether a non-empty script compiles. Note that we	*/
	/* don't stop the save, for instance in case the user has	*/
	/* forgotten to set the script language.			*/
	if (!text.isEmpty())
		if ((text.at(0) != '#') || !text.at(1).isLetter())
			checkCompile (m_event->getOwner()->isObject(), text, "eventFunc") ;

	m_event->setValue	(text) ;
	m_event->setBreakpoints (m_breakpoints) ;
}

/*  KBAttrEventItem							*/
/*  clear	: Clear value						*/
/*  (returns)	: void		:					*/

void	KBAttrEventItem::clear ()
{
	DELOBJ	(m_macro)   ;
	KBAttrItem::clear() ;
}

void	KBAttrEventItem::setBreakpoints
	(	QValueList<int>	&breakpoints
	)
{
	m_breakpoints = breakpoints ;

	for (uint idx = 0 ; idx < m_breakpoints.count() ; idx += 1)
		fprintf
		(	stderr,
			"KBAttrEventItem::setBreakpoints: breakpoint %d\n",
			m_breakpoints[idx]
		)	;
}

const QValueList<int>
	&KBAttrEventItem::breakpoints ()
{
	for (uint idx = 0 ; idx < m_breakpoints.count() ; idx += 1)
		fprintf
		(	stderr,
			"KBAttrEventItem::breakpoints: breakpoint %d\n",
			m_breakpoints[idx]
		)	;

	return	m_breakpoints ;
}



/*  ------------------------------------------------------------------  */

/*  KBEventDlg								*/
/*  KBEventDlg	: Constructor for event dialog				*/
/*  parent	: QWidget *		: Parent widget			*/
/*  event	: KBEvent *		: Actual event property		*/
/*  item	: KBAttrEventItem *	: Associated item		*/
/*  attrDict	: QDict<KBAttrItem> &	: Attribute dictionary		*/
/*  (returns)	: KBEventDlg		:				*/

KBEventDlg::KBEventDlg
	(	QWidget			*parent,
		KBEvent			*event,
		KBAttrEventItem		*item,
		QDict<KBAttrItem>	&attrDict
	)
	:
	KBAttrDlg	(parent, event, item, attrDict),
	m_editor	(new TKTextDocument
				(	_KBDialog::getTextManager(KBOptions::getScriptFont(),
					true,
					false
				)), this),
	m_mapper	(&m_editor),
	m_eventItem	(item)
{
	QVBoxLayout	*layout = new QVBoxLayout (this) ;
	layout->addWidget (&m_editor) ;

	KBNode	*node	= m_attr->getOwner  () ;
	m_language	= node  ->getRoot()->getAttrVal("language") ;

	if (m_language == "py")
		m_editor.setHighlight ("Python") ;
	else	m_editor.setHighlight ("NoHighlight") ;

	if (event != 0)
	{
		m_breakpoints	= event->breakpoints () ;

		for (uint idx = 0 ; idx < m_breakpoints.count() ; idx += 1)
			fprintf
			(	stderr,
				"KBEventDlg::KBEventDlg: breakpoint %d\n",
				m_breakpoints[idx]
			)	;
	}

	connect
	(	m_editor.textView(),
		SIGNAL(mouseRightClick (QPoint, int)),
		SLOT  (mouseRightClick (QPoint, int))
	)	;
}

/*  KBEventDlg								*/
/*  init	: Initialise for editing				*/
/*  (returns)	: bool		: True if modal dialog executed		*/

bool	KBEventDlg::init ()
{
	KBOptions::ScriptType type ;

	/* First need to decide whether this is a script or a macro.	*/
	/* First case is if there is no event item, in which case we	*/
	/* must be using scripting. The next check whether there is an	*/
	/* extant value, in which case use that irrespective of the	*/
	/* options setting.						*/
	if	(m_eventItem == 0)
	{
		type	= KBOptions::ScriptTypeAlwaysScript ;
	}
	else if	(m_eventItem->getMacro()    != 0)
	{
		type	= KBOptions::ScriptTypeAlwaysMacro  ;
	}
	else if	(!m_eventItem->value().isEmpty())
	{
		type	= KBOptions::ScriptTypeAlwaysScript ;
	}
	else
	{
		/* If the event is empty then use the option setting,	*/
		/* prompting the user if set to user choice.		*/
		type	= KBOptions::getScriptType() ;

		if (type == KBOptions::ScriptTypeUserChoice)
		{
			QStringList	opts	   ;
			opts.append (TR("Script")) ;
			opts.append (TR("Macro" )) ;
			static	QString	res	   ;

			KBChoiceDlg	dDlg
					(	TR("Event"),
						TR("Select script or macro"),
						opts,
						res
					)	;

			if (!dDlg.exec()) return true ;

			if (res == TR("Script"))
				type	= KBOptions::ScriptTypeAlwaysScript ;
			else	type	= KBOptions::ScriptTypeAlwaysMacro  ;
		}
	}
						

	/* First case is a macro, in which case run the macro editor as	*/
	/* a modal dialog. On return, the caller does not have anything	*/
	/* more to do.							*/
	if (type == KBOptions::ScriptTypeAlwaysMacro)
	{
		KBNode		*owner	= m_item->attr()->getOwner() ;
		KBLocation	locn	= owner->getDocRoot()->getLocation() ;
		KBEventMacroDlg mDlg	(locn.dbInfo, locn.docLocn)  ;
		KBError		error	;

		mDlg.startup (m_eventItem->getMacro(), error) ;
		if (mDlg.exec ())
		{
			KBMacroExec *m = mDlg.macro (owner, error) ;
			if (m == 0)
			{
				error.DISPLAY() ;
				return	true	;
			}

			m_eventItem->setMacro (m)	;
		}

		return	true ;
	}

	/* Using a script. The script is edited within the property	*/
	/* dialog. If the current text is empty then see if there is a	*/
	/* skeleton and if so, insert that and move the cursor to the	*/
	/* initial point.						*/
	if (type == KBOptions::ScriptTypeAlwaysScript)
	{
		return	init (m_eventItem == 0 ? QString::null : m_eventItem->value ()) ;
	}

	/* Should never get here ...					*/
	return	true	;
}

bool	KBEventDlg::init
	(	const QString	&value
	)
{
	extern	KBDictionary	*getAttrDict() ;

	QString	text	= value	;
	bool	mark	= false	;

	if (m_eventItem != 0)
	{
		QString	element	= m_eventItem->attr()->getOwner()->getElement() ;

		if (text.isEmpty())
		{
			text	= getAttrDict()->getAttrExtra
				  (	element,
					m_eventItem->attr()->getName(),
					"skeleton_" + m_language
				  )	;
			mark	= text.find("__MARK__") >= 0 ;

			if (element.left(2) == "KB")
				text.replace (QRegExp("__TYPE__"), element.mid(2).lower()) ;
		}
	}

	m_editor.setText (text)	;
	m_editor.setFocus()	;

	if (mark)
		if (m_editor.find ("__MARK__", false, false, false, false))
			m_editor.replaceFound ("") ;

	if (m_eventItem != 0)
	{
		m_breakpoints	= m_eventItem->breakpoints () ;

		for (uint idx = 0 ; idx < m_breakpoints.count() ; idx += 1)
			fprintf
			(	stderr,
				"KBEventDlg::KBEventDlg: breakpoint %d\n",
				m_breakpoints[idx]
			)	;
	}

	for (uint slot = 0 ; slot < m_breakpoints.count() ; slot += 1)
		m_editor.textDocument()->setMarked (m_breakpoints[slot], 0x01) ;

	return	false ;
}

void	KBEventDlg::save ()
{
	if (m_eventItem != 0)
	{
		m_eventItem->setValue	    (value()) ;
		m_eventItem->setBreakpoints (m_breakpoints) ;
	}
}

void	KBEventDlg::mouseRightClick
	(	QPoint		pos,
		int		lno
	)
{
	QPopupMenu pop ;

	pop.insertItem (TR("Cancel")) ;
	pop.insertItem (TR("Toggle breakpoint"), this, SLOT(toggleBreakpoint())) ;
	pop.insertItem (TR("Clear breakpoints"), this, SLOT(clearBreakpoints())) ;

	m_lastLno = lno ;
	pop.exec (pos)	;
}

void	KBEventDlg::toggleBreakpoint ()
{
	if (m_breakpoints.findIndex (m_lastLno) < 0)
	{
		m_editor.textDocument()->setMarked (m_lastLno, 0x01) ;
		m_breakpoints.append (m_lastLno) ;
		return	;
	}

	m_editor.textDocument()->setMarked (m_lastLno, 0x00) ;
	m_breakpoints.remove (m_lastLno) ;

}

void	KBEventDlg::clearBreakpoints ()
{
	for (uint slot = 0 ; slot < m_breakpoints.count() ; slot += 1)
		m_editor.textDocument()->setMarked (m_breakpoints[slot], 0x00) ;

	m_breakpoints.clear () ;
}

/*  KBEventDlg								*/
/*  value	: Get value						*/
/*  (returns)	: QString	:					*/

QString	KBEventDlg::value ()
{
	return	KBEvent::trimEvent (m_editor.text()) ;
}

/*  KBEventDlg								*/
/*  verify	: Verify current value					*/
/*  (returns)	: void		:					*/

void	KBEventDlg::verify  ()
{
	/* Default assued script text. Get the text and see if it	*/
	/* looks like a shortcut, in which case give the user a hint	*/
	/* ....								*/
	QString	text	= value () ;
	if ((text.at(0) == '#') && text.at(1).isLetter())
	{
		KBError::EWarning
		(	TR("Event shortcut, cannot compile"),
			TR("Event code should be in a script module"),
			__ERRLOCN
		)	;
		return	;
	}

	/* ... and quietly ignore it if it is empty.			*/
	if (text.isEmpty()) return ;

	/* OK, we have something that looks like a script, so see if it	*/
	/* compiles. 							*/
	if (checkCompile (m_attr->getOwner()->isObject(), text, "eventFunc"))
		TKMessageBox::information (0, TR("Event compiles OK")) ;
}



/*  ------------------------------------------------------------------  */

/*  KBEventMacroDlg							*/
/*  KBEventMacroDlg							*/
/*		: Constructor for inline macro editing dialog		*/
/*  dbInfo	: KBDBInfo *	  : Database information		*/
/*  svName	: const QString & : Server name				*/	
/*  (returns)	: KBEventMacroDlg	:				*/

KBEventMacroDlg::KBEventMacroDlg
	(	KBDBInfo	*dbInfo,
		const QString	&svName
	)
	:
	_KBDialog	("Macro", true, "KBEventMacroDlg"),
	m_editor	(this, dbInfo, svName),
	m_bOK		(this, "ok"	),
	m_bCancel	(this, "cancel"	)
{
	QVBoxLayout	*layMain = new QVBoxLayout (this) ;
	layMain->addWidget (&m_editor ) ;

	QHBoxLayout	*layButt = new QHBoxLayout (layMain) ;
	layButt->addStretch() ;
	layButt->addWidget (&m_bOK    ) ;
	layButt->addWidget (&m_bCancel) ;
}

/*  KBEventMacroDlg							*/
/*  accept	: User accepts changes					*/
/*  (returns)	: void		:					*/

void	KBEventMacroDlg::accept ()
{
	done	(1) ;
}

/*  KBEventMacroDlg							*/
/*  macro	: Get macro definition					*/
/*  owner	: KBNode *	: Owning node				*/
/*  error	: KBError &	: Error return				*/
/*  (returns)	: KBMacroExec *	: Macro or null on error		*/

KBMacroExec
	*KBEventMacroDlg::macro
	(	KBNode		*owner,
		KBError		&error
	)
{
	return	m_editor.macro (error, owner) ;
}
