/***************************************************************************
    file	         : kb_qrydata.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_location.h"

#include	"kb_database.h"
#include	"kb_queryset.h"
#include	"kb_qrylevel.h"

#ifndef 	_WIN32
#include 	"kb_qrydata.moc"
#else
#include 	"kb_qrydata.h"
#endif

#include	"kb_item.h"
#include	"kb_table.h"

/*  ------------------------------------------------------------------  */

static	QSize	qdSize	;

/*  KBQryDisplay							*/
/*  KBQryDisplay: Simple display of rich-text SQL			*/
/*  sql		: const QString &	: Text				*/
/*  reason	: const QString &	: Permissions reason		*/
/*  (returns)	: KBQryDisplay		:				*/

KBQryDisplay::KBQryDisplay
	(	const QString	&sql,
		const QString	&reason
	)
	:
	_KBDialog	("Query text", true),
	m_query		(this),
	m_reason	(this),
	m_bOK		(this)
{
	fprintf
	(	stderr,
		"KBQryDisplay::KBQryDisplay: [%s][%s]\n",
		(cchar *)sql,
		(cchar *)reason
	)	;

	m_layMain	= new QVBoxLayout (this) ;
	m_layMain->addWidget  (&m_query ) ;
	m_layMain->addWidget  (&m_reason) ;

	m_layButt	= new QHBoxLayout (m_layMain) ;
	m_layButt->addStretch () ;
	m_layButt->addWidget  (&m_bOK) ;

	m_bOK	  .setText    (TR("OK" )) ;
	m_query   .setText    (sql	) ;
	m_reason  .setText    (reason	) ;

	if (reason.isEmpty()) m_reason.hide () ;

	connect	(&m_bOK, SIGNAL(clicked()), SLOT(clickOK())) ;

#if	__KB_EMBEDDED
	m_showMaxed = true ;
#else
	if (qdSize == QSize()) qdSize = QSize (550, 250) ;
	resize	(qdSize) ;
#endif
}

/*  KBQryDisplay							*/
/*  clickOK	: User clicks OK button					*/
/*  (returns)	: void		:					*/

void	KBQryDisplay::clickOK ()
{
	qdSize	= size () ;
	done	(1) ;
}


/*  ------------------------------------------------------------------  */

/*  KBQryData								*/
/*  KBQryData	: Constructor for form query node			*/
/*  parent	: KBNode *	: Parent node				*/
/*  aList	: const QDict<QString> &				*/
/*				: List of attributes			*/
/*  element	: cchar *	: Element name				*/
/*  (returns)	: KBQryData	:					*/

KBQryData::KBQryData
	(	KBNode			*parent,
		const QDict<QString>	&aList,
		cchar			*element
	)
	:
	KBQryBase (parent, aList, element)
{
	qryLevels.setAutoDelete (true) ;
	wLError	= false	;
}

/*  KBQryData								*/
/*  KBQryData	: Constructor for form query node			*/
/*  parent	: KBForm *	: Parent node				*/
/*  element	: cchar *	: Element name				*/
/*  (returns)	: KBQryData	:					*/

KBQryData::KBQryData
	(	KBNode	*parent,
		cchar	*element
	)
	:
	KBQryBase (parent, element)
{
	qryLevels.setAutoDelete (true) ;
	wLError	= false	;
}

/*  KBQryData								*/
/*  KBQryData	: Constructor for form query node			*/
/*  _parent	: KBNode *	: Parent node				*/
/*  _query	: KBQryData *	: Extant form query			*/
/*  (returns)	: KBQryData	:					*/

KBQryData::KBQryData
	(	KBNode		*_parent,
		KBQryData	*_query
	)
	:
	KBQryBase	(_parent, _query)
{
	qryLevels.setAutoDelete (true) ;
	wLError	= false	;
}

/*  KBQryData								*/
/*  ~KBQryData	: Destructor for form query node			*/
/*  (returns)	:		:					*/

KBQryData::~KBQryData ()
{
}

/*  KBQryData								*/
/*  addItem	: Add a child form item					*/
/*  qlvl	: uint		: Query level				*/
/*  item	: KBItem *	: Child item				*/
/*  (returns)	: bool		: Item fetches from database		*/

bool	KBQryData::addItem
	(	uint	qlvl,
		KBItem	*item
	)
{
	return	getQryLevel(qlvl)->addItem (item) ;
}

/*  KBQryData								*/
/*  remItem	: Remove a child form item				*/
/*  qlvl	: uint		: Query level				*/
/*  item	: KBItem *	: Child form item			*/
/*  (returns)	: void		:					*/

void	KBQryData::remItem
	(	uint	qlvl,
		KBItem	*item
	)
{
	getQryLevel(qlvl)->remItem (item) ;
}

/*  KBQryData								*/
/*  resetData	: Reset data to original state				*/
/*  qlvl	: uint		: Caller's query level			*/
/*  qrow	: uint		: Caller's query row number		*/
/*  (returns)	: void		:					*/

void	KBQryData::resetData
	(	uint	qlvl,
		uint	qrow
	)
{
	getQryLevel(qlvl)->resetData (qrow) ;
}

/*  KBQryData								*/
/*  loadItems	: Load items from query					*/
/*  qlvl	: uint		: Caller's query level			*/
/*  qrow	: uint		: Caller's query row number		*/
/*  (returns)	: bool		: Success				*/

bool	KBQryData::loadItems
	(	uint	qlvl,
		uint	qrow
	)
{
	return	getQryLevel(qlvl)->loadItems (qrow) ;
}

/*  KBQryData								*/
/*  clearItems	: Clear all items in query				*/
/*  qlvl	: uint		: Caller's query level			*/
/*  qrow	: uint		: Caller's query row number		*/
/*  (returns)	: bool		: Success				*/

bool	KBQryData::clearItems
	(	uint	qlvl,
		uint	qrow
	)
{
	return	getQryLevel(qlvl)->clearItems (qrow) ;
}

/*  KBQryData								*/
/*  getRowState	: Get state of specified query row			*/
/*  qlvl	: uint		: Query level				*/
/*  qrow	: uint		: Row number				*/
/*  (returns)	: KB::RState	: State					*/

KB::RState KBQryData::getRowState
	(	uint	qlvl,
		uint	qrow
	)
{
	return	getQryLevel(qlvl)->getRowState (qrow) ;
}



/*  KBQryData								*/
/*  getField	: Get field value					*/
/*  qlvl	: uint		  : Query level				*/
/*  qrow	: uint		  : Row number				*/
/*  qcol	: uint		  : Column number			*/
/*  initial	: bool		  : Get initial value			*/
/*  (returns)	: KBValue	  : Value				*/

KBValue	KBQryData::getField
	(	uint	qlvl,
		uint	qrow,
		uint	qcol,
		bool	initial
	)
{
	return	getQryLevel(qlvl)->getField(qrow, qcol, initial) ;
}

/*  KBQryData								*/
/*  setField	: Set field value					*/
/*  qlvl	: uint		  : Query level				*/
/*  qrow	: uint		  : Row number				*/
/*  qcol	: uint		  : Column number			*/
/*  value	: const KBValue & : Value				*/
/*  (returns)	: void		  :					*/

void	KBQryData::setField
	(	uint		qlvl,
		uint		qrow,
		uint		qcol,
		const KBValue	&value
	)
{
	getQryLevel(qlvl)->setField(qrow, qcol, value) ;
}

/*  KBQryData								*/
/*  getNumRows	: Get number of rows in query set			*/
/*  qlvl	: uint		: Query level				*/
/*  (returns)	: uint		: Number of rows			*/

uint	KBQryData::getNumRows
	(	uint		qlvl
	)
{
	return	getQryLevel(qlvl)->getNumRows() ;
}

/*  KBQryData								*/
/*  getWidth	: Get nominal column width				*/
/*  qlvl	: uint		: Query level				*/
/*  qcol	: uint		: Query column				*/
/*  (returns)	: uint		: Nominal width				*/

uint	KBQryData::getWidth
	(	uint		qlvl,
		uint		qcol
	)
{
	return	getQryLevel(qlvl)->getWidth (qcol); 
}

bool	KBQryData::startUpdate
	(	uint		qlvl,
		uint		qrow,
		Locking		locking
	)
{
	KBError	error	;

	if (!getQryLevel(qlvl)->startUpdate (qrow, locking, error))
	{
		setError (error) ;
		return	 false	 ;
	}

	return	true	;
}

bool	KBQryData::endUpdate
	(	uint		qlvl,
		bool		commit
	)
{
	KBError	error	;

	if (!getQryLevel(qlvl)->endUpdate (commit, error))
	{
		setError (error) ;
		return	 false	 ;
	}

	return	true	;
}

KBQryBase::Locking
	KBQryData::lockingState
	(	uint		qlvl
	)
{
	return	getQryLevel(qlvl)->lockingState() ;
}

/*  KBQryData								*/
/*  saveRow	: Save row						*/
/*  qlvl	: uint		: Query level				*/
/*  qrow	: uint		: Query row number			*/
/*  (returns)	: bool		: Success				*/

bool	KBQryData::saveRow
	(	uint	qlvl,
		uint	qrow
	)
{
	KBError	error	;

	if (!getQryLevel(qlvl)->saveRow(qrow, error))
	{
		setError (error) ;
		return	 false	 ;
	}

	return	true	 ;
}

/*  KBQryData								*/
/*  deleteRow	: Delete row						*/
/*  qlvl	: uint		: Query level				*/
/*  qrow	: uint		: Query row number			*/
/*  (returns)	: bool		: Success				*/

bool	KBQryData::deleteRow
	(	uint	qlvl,
		uint	qrow
	)
{
	/* Note that deleteRow simply sets the deleted flag on the row.	*/
	/* The actual deletion will occur inside syncRow or syncAll.	*/
	getQryLevel(qlvl)->setRowState (qrow, KB::RSDeleted) ;
	return	true	;
}

/*  KBQryData								*/
/*  insertRow	: Insert row						*/
/*  qlvl	: uint		: Query level				*/
/*  qrow	: uint		: Query row number			*/
/*  (returns)	: bool		: Success				*/

bool	KBQryData::insertRow
	(	uint	qlvl,
		uint	qrow
	)
{
	getQryLevel(qlvl)->insertRow   (qrow) ;
	getQryLevel(qlvl)->setRowState (qrow, KB::RSInserted) ;
	return	true	;
}

/*  KBQryData								*/
/*  syncRow	: Synchronise row					*/
/*  qlvl	: uint		 : Query level				*/
/*  qrow	: uint		 : Query row number			*/
/*  pValue	: KBValue *	 : Parent value if any			*/
/*  cExpr	: const QString &: Child expression			*/
/*  block	: KBBlock *	 : Requesting block			*/
/*  oper	: KB::Action &	 : Operation performed			*/
/*  priKey	: KBValue &	 : Return row primary key		*/
/*  (returns)	: bool		 : Success				*/

bool	KBQryData::syncRow
	(	uint		qlvl,
		uint		qrow,
		KBValue		*pValue,
		const QString	&cExpr,
		KBBlock		*block,
		KB::Action	&oper,
		KBValue		&priKey
	)
{
	KBError	error	;

	if (!getQryLevel(qlvl)->syncRow
				(	qrow,
					pValue,
					cExpr,
					block,
					error,
					oper,
					priKey
				))
	{
		setError (error) ;
		return	 false	 ;
	}

	return	true	;
}

/*  KBQryData								*/
/*  syncAll	: Synchronise all rows					*/
/*  qlvl	: uint		 : Query level				*/
/*  pValue	: KBValue *	 : Parent value if any			*/
/*  cExpr	: const QString &: Child expression			*/
/*  block	: KBBlock *	 : Requesting block			*/
/*  (returns)	: bool		 : Success				*/

bool	KBQryData::syncAll
	(	uint		qlvl,
		KBValue		*pValue,
		const QString	&cExpr,
		KBBlock		*block
	)
{
	KBError	error	;

	if (!getQryLevel(qlvl)->syncAll (pValue, cExpr, block, error))
	{
		setError (error) ;
		return	 false	 ;
	}

	return	true	;
}

bool	KBQryData::getSelect
	(	uint		qlvl,
		KBSelect	&select
	)
{
	return	getQryLevel(qlvl)->getSelect (select) ;
}

/*  KBQryData								*/
/*  select	: Execute associated select statement			*/
/*  qlvl	: uint		 : Query level				*/
/*  pValue	: KBValue *	 : Parent value if any			*/
/*  cExpr	: const QString &: Child expression			*/
/*  filter	: const QString &: Filter if any			*/
/*  sorting	: const QString &: Sorting if any			*/
/*  query	: bool		 : True to add query terms		*/
/*  qrow	: uint		 : Query row number			*/
/*  (returns)	: bool		 : Success				*/

bool	KBQryData::select
	(	uint		qlvl,
		KBValue		*pValue,
		const QString	&cExpr,
		const QString	&filter,
		const QString	&sorting,
		bool		query,
		uint		qrow
	)
{
	if ((qlvl > 0) && !query) return true ;

	KBError	error	;

	if (!getQryLevel(qlvl)->doSelect (pValue, cExpr, filter, sorting, query, qrow, error))
	{
		setError (error) ;
		return	 false	 ;
	}

	return	true	;
}

/*  KBQryData								*/
/*  newRowEmpty	: Check if new row at end of query is empty		*/
/*  qlvl	: uint		: Query level				*/
/*  qrow	: uint		: Row being checked			*/
/*  (returns)	: bool		: Row is empty				*/

bool	KBQryData::newRowEmpty
	(	uint	qlvl,
		uint	qrow
	)
{
	return	getQryLevel(qlvl)->newRowEmpty (qrow) ;
}

/*  KBQryData								*/
/*  rowIsDirty	: Check if specified row is dirty			*/
/*  qlvl	: uint		: Query level				*/
/*  qrow	: uint		: Query row number			*/
/*  reset	: bool		: True to reset dirty flag		*/
/*  (returns)	: bool		: Dirty					*/

bool	KBQryData::rowIsDirty
	(	uint	qlvl,
		uint	qrow,
		bool	reset
	)
{
	return	getQryLevel(qlvl)->rowIsDirty (qrow, reset) ;
}

/*  KBQryData								*/
/*  getFieldList: Get list of fields from database			*/
/*  qlvl	: uint		: Query level				*/
/*  fldList	: QList<KBFieldSpec>&					*/
/*				: Field list result			*/
/*  pKey	: int &		: Primary key index			*/
/*  (returns)	: bool		: Success				*/

bool	KBQryData::getFieldList
	(	uint			qlvl,
		QList<KBFieldSpec>	&fldList,
		int			&pKey
	)
{
	KBError	error	;

	if (!getQryLevel(qlvl)->getFieldList (fldList, pKey, isQryTable() == 0, error))
	{
		setError (error) ;
		return	 false	 ;
	}

	return	true ;
}

/*  KBQryData								*/
/*  setCurrentRow: Set current query row				*/
/*  qlvl	 : uint		: Query level				*/
/*  qrow	 : uint		: Query row number			*/
/*  (returns)	 : uint		: Total rows here and below		*/

uint	KBQryData::setCurrentRow
	(	uint	qlvl,
		uint	qrow
	)
{
	return	getQryLevel(qlvl)->setCurrentRow(qrow) ;
}

/*  KBQryData								*/
/*  getPermission: Get permission flags					*/
/*  qlvl	 : uint		: Query level number			*/
/*  (returns)	 : uint		: Permission flags			*/

uint	KBQryData::getPermission
	(	uint	qlvl
	)
{
	return	getQryLevel(qlvl)->getPermission() ;
}

/*  KBQryData								*/
/*  sortByColumn: Sort data by column					*/
/*  qlvl	: uint		: Query level				*/
/*  qrow	: uint		: Query row number			*/
/*  asc		: bool		: Sort ascending			*/
/*  item	: KBItem *	: Item associated with column		*/
/*  (returns)	: uint		: Total rows here and below		*/

void	KBQryData::sortByColumn
	(	uint	qlvl,
		uint	qcol,
		bool	asc,
		KBItem	*item
	)
{
	getQryLevel(qlvl)->sortByColumn (qcol, asc, item) ;
}

/*  KBQryData								*/
/*  getQryLevel	: Get a specified query level				*/
/*  qlvl	: uint		: Query level number			*/
/*  (returns)	: KBQryLevel *	: Query level object			*/

KBQryLevel *KBQryData::getQryLevel
	(	uint	qlvl
	)
{
	/* If there are no loaded query levels then now is the time to	*/
	/* load them. If this fails (for instance, a specified KBase-	*/
	/* level query does not exist) then report the error, and	*/
	/* create a dummy level.					*/
	if ((qryLevels.count() == 0) && !loadQuery ())
	{
		lastError().DISPLAY() ;

		KBTable	   *dTab = new KBTable    (this, "__Unknown", "", "__Unknown", "", "", "", "", "", "") ;
		KBQryLevel *dLvl = new KBQryLevel (getParent(), 0, dbLink, 0, dTab) ;

		qryLevels.append   (dLvl) ;
	}

	if (qlvl >= qryLevels.count())
	{
		if (!wLError)
		{
			KBError::EError
			(	QString	(TR("Request for query level %1 with only %2"))
					.arg(qlvl)
					.arg(qryLevels.count()),
				QString::null,
				__ERRLOCN
			)	;
			wLError	= true	;
		}

		qlvl	= 0	;
	}

	return	qryLevels.at (qlvl) ;
}

/*  KBQryData								*/
/*  setRowMarked: Set row marking					*/
/*  qlvl	: uint		: Query level				*/
/*  qrow	: uint		: Query row				*/
/*  op		: KB::MarkOp	: Mark operation			*/
/*  (returns)	: void		:					*/

void	KBQryData::setRowMarked
	(	uint		qlvl,
		uint		qrow,
		KB::MarkOp	op
	)
{
	getQryLevel(qlvl)->setRowMarked (qrow, op) ;
}

/*  KBQryData								*/
/*  getRowMarked: Set row mark state					*/
/*  qlvl	: uint		: Query level				*/
/*  qrow	: uint		: Query row				*/
/*  (returns)	: bool		: True if marked			*/

bool	KBQryData::getRowMarked
	(	uint		qlvl,
		uint		qrow
	)
{
	return	getQryLevel(qlvl)->getRowMarked (qrow) ;
}

/*  KBQryData								*/
/*  deleteAllMarked							*/
/*		: Delete all marked rows				*/
/*  qlvl	: uint		: Query level				*/
/*  nrows	: uint &	: Return number of rows deleted		*/
/*  (returns)	: bool		: Success				*/

bool	KBQryData::deleteAllMarked
	(	uint		qlvl,
		uint		&nrows
	)
{
	/* Note that this actually only sets row states to be deleted;	*/
	/* the actual deletion occurs when the query set is		*/
	/* synchronised.						*/
	KBError	error	;
	if (!getQryLevel(qlvl)->deleteAllMarked (nrows, error))
	{
		setError (error) ;
		return	 false	 ;
	}

	return	true	;
}

/*  KBQryData								*/
/*  getSQLText	: Get the SQL text for the query			*/
/*  qryLvl	: uint		: Query level				*/
/*  pretty	: bool		: Get in pretty format			*/
/*  (returns)	: QString	: Text					*/

QString	KBQryData::getSQLText
	(	uint		qryLvl,
		bool		pretty
	)
{
	/* Note that this returns the prettified text, for display	*/
	/* purposes.							*/
	QString	prefix	= qryLvl == 0 ?
				"" : "<i>[Top level query]</i><br/><br/>" ;

	return	prefix + getQryLevel(0)->getSQLText(pretty) ;
}

/*  KBQryData								*/
/*  getSQLReason: Get reason for permissions				*/
/*  pretty	: bool		: Get in pretty format			*/
/*  (returns)	: QString	: Text					*/

QString	KBQryData::getSQLReason
	(	uint		qryLvl
	)
{
	return	getQryLevel(qryLvl)->getSQLReason () ;
}
