/*
    Copyright (C) 2001-2002 by theKompany.com <www.thekompany.com>
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
    See COPYING.GPL file.

    In addition, as a special exception, theKompany.com gives permission
    to link the code of this program with the tkwidgets library (or with
    modified versions of tkwidgets that use the same license as tkwidgets),
    and distribute linked combinations including the two.  You must obey
    the GNU General Public License in all respects for all of the code used
    other than tkwidgets.  If you modify this file, you may extend this
    exception to your version of the file, but you are not obligated to do so.
    If you do not wish to do so, delete this exception statement from your
    version.

    This license grants you the ability to use tkwidgets with Rekall only
    and may not be used outside of Rekall.
    See also http://www.rekall.a-i-s.co.uk/products/license.txt for details.
*/


#ifndef _WIN32
#include "tktextdoc.moc"
#else
#include "tktextdoc.h"
#endif

#include "tktexteditor.h"
#include "tktextmanager.h"
#include "tktextline.h"
#include "tkhighlight.h"

#include <qapplication.h>
#include <qclipboard.h>
#include <qfont.h>
#include <qfontmetrics.h>
#include <qpainter.h>
#include <qtextcodec.h>
#include <qregexp.h>
#include <qpixmap.h>
#include <qcolor.h>
#include <qintdict.h>
#include <qprinter.h>

#define CP debug(" %s:%d", __FILE__, __LINE__);

TKTextLineArray::TKTextLineArray()
: autoDelete(false)
{
}

void TKTextLineArray::append(TKTextLine *line)
{
    resize(size() + 1);
    at(size() - 1) = line;
}

void TKTextLineArray::insert(int index, TKTextLine *line)
{
    resize(size() + 1);
    int id;
    for (id = size() - 1; id != index; --id)
        at(id) = at(id - 1);
    at(id) = line;
}

void TKTextLineArray::remove(int index)
{
    int id = index;
    if (autoDelete)
      delete at(id);
    for (; id < (int)size() - 1; id++)
        at(id) = at(id + 1);
    resize(size() - 1);
}

void TKTextLineArray::clear()
{
    if (!isEmpty() && autoDelete) {
        for (int id = size() - 1; id >= 0; --id)
            delete at(id);
    }
    resize(0);
}

void TKTextLineArray::setAutoDelete(bool f)
{
  autoDelete = f;
}
/**************************************************************/
TKTextDocument::TKTextDocument(TKTextEditorManager *manager)
: QObject(), m(manager)
{
    rcontents.setAutoDelete(true);

    endRecordLock = false;
    startRecordLock = false;

    paint = 0;
    buffer = 0;
    hl = TKTextHighlightManager::highlight(manager);
    tabChars = m->tabWidth();
    updateFontData();

    newDocGeometry = false;
    modified = false;

    undoList.setAutoDelete(true);
    undoSteps = 500;

    buffer = 0;
    paint = 0;
    readonly = false;
    shouldWrap = false;
    nWrapMode = wrapLine;
    wordBound = " \"'\\:;,./?!`";

    clear();
}

TKTextDocument::~TKTextDocument()
{
    contents.clear();
    delete hl;
    delete paint;
    delete buffer;
}

void TKTextDocument::lockRecordStart(bool f)
{
  startRecordLock = f;
}

void TKTextDocument::lockRecordEnd(bool f)
{
  endRecordLock = f;
}

int TKTextDocument::lastLine() const
{
    return contents.count() - 1;
}

int TKTextDocument::lineNum(int line)
{
    int l = 0;
    int num = -1;
    while(l <= line)
    {
        if(! (lineOf(l)->status() & TKTextLine::Wrapped))
            num++;
        l++;
    }
    return QMAX(num, 0);
}

int TKTextDocument::lineCol(int line, int column)
{
    int l = line-1;
    int delta = 0;
    if( (lineOf(line)->status() & TKTextLine::Wrapped) )
    {
        while((l > 0) && (lineNum(l) == lineNum(line))
            && (lineOf(l)->status() & TKTextLine::Wrapped))
        {
            delta += lineOf(l)->length();
            l--;
        }
        if(l >= 0)
            delta += lineOf(l)->length();
        column += delta;
    }
    return column;
}


TKTextLine *TKTextDocument::lineOf(int line)
{
//*******************
/*    if (line > lastLine()) {
        int zero = 0;
        debug("%d", 10/zero);
    }
*/
    return contents.at(line);
}

void TKTextDocument::tagLine(int line)
{
    updateRegion.expand(line);
}

void TKTextDocument::tagLines(int start, int end)
{
    updateRegion.expand(start, end);
}

void TKTextDocument::tagAll()
{
    updateRegion.expand(0, lastLine());
}

void TKTextDocument::registerView(TKTextView *view)
{
    views.append(view);
}

void TKTextDocument::removeView(TKTextView *view)
{
    views.remove(view);
}

bool TKTextDocument::load(QTextStream &ts)
{
    clear();
    TKTextLine *textLine = contents.at(0);
    int num = 1;
    do {
        QString line = ts.readLine();
#ifdef KOBOL
		bool isNum=true;
		for (int i=0;i<6;i++)
		if (!line[i].isDigit())
			isNum=false;
		if (isNum && !line[6].isDigit())
			line.remove(0,6);
#endif

		if (line.isNull())
            break;
#ifdef KOBOL
		if (line[0]!='*' && line[0]!='-' && line[0]!=' ' )
		    line = ' '+line;
#endif    
	    
	    
        if (textLine) {
            textLine->set(line);
        }
        else {
            textLine = new TKTextLine();
            textLine->setNumber(num);
            textLine->setInternalNumber(num);
            textLine->set(line);
            contents.append(textLine);
            rcontents.append(textLine);
            num++;
        }
        textLine = 0;
    } while (true);

    updateLines();
    updateViews();
    setReadOnly(false);
    return true;
}

bool TKTextDocument::save(QTextStream &ts)
{
    if (isReadOnly())
        return false;

    int line = 0;
    int last = rcontents.size() - 1;
	ts << *rcontents.at(0);
    while (line != last)
	{
        if( !(rcontents.at(++line)->status() & TKTextLine::Wrapped) )
            ts << endl;
		QString s=*rcontents.at(line);
        ts << s;
	}

    documentChanged(false);
    return true;
}

bool TKTextDocument::saveWith6Numbers(QTextStream &ts)
{
#ifdef KOBOL
    if (isReadOnly())
        return false;

    int line = 0;
    int last = rcontents.size() - 1;
		if (m->fLinuNumberMarginZero)
    	ts << "000000"+*rcontents.at(0);
		else
    	ts << "000001"+*rcontents.at(0);
		
    while (line != last)
	{
		QString s=*rcontents.at(++line);
		QString number;
		if (m->fLinuNumberMarginZero)
			number.setNum(line);
		else
			number.setNum(line+1);
		while (number.length()<6) number="0"+number;
		s=number+s;
        ts << endl << s;
	}

    documentChanged(false);
#endif
    return true;
}

QString TKTextDocument::text()
{
    QString result;
    result.append(*rcontents.at(0));
    int line = 0;
    int last = rcontents.size() - 1;

    // mike: this bit based on code in ::save(QTextStream &)
    while (line != last)
	{
        if( !(rcontents.at(++line)->status() & TKTextLine::Wrapped) )
            result.append("\n") ;
	result.append(*rcontents.at(line));
	}

    return result;
}

void TKTextDocument::setReadOnly(bool f)
{
    readonly = f;
}

bool TKTextDocument::isReadOnly()
{
    return readonly;
}

void TKTextDocument::setWrap(bool wrap)
{
    if( shouldWrap != wrap)
    {
	    shouldWrap = wrap;
        for (TKTextView *view = views.first(); view; view = views.next())
        {
            if(wrap)
                rewrap(view);
            else
            {
                int vc = view->cursor->column();
                int vl = view->cursor->line();

                for(int i=0;i<lastLine();i++)
                    unwrap(view, i, &vl, &vc);

                view->setCursorPosition(vl, vc);
            }
            updateLines(0,lastLine());

            view->updateView(newDocGeometry);
        }
    }
}

bool TKTextDocument::getWrap()
{
  return shouldWrap;
}

void TKTextDocument::setCustomHighlight(const QString &hlModePath)
{
    setHighlight(TKTextHighlightManager::highlight(m, hlModePath));
}

void TKTextDocument::setHighlight(TKTextHighlight *h)
{
    TKTextHighlight *temp = hl;
    hl = h;
    updateLines();
    updateViews();
    delete temp;
}

void TKTextDocument::updateFontData()
{
    int maxAscent = 0;
    int maxFontHeight = 0;

    for (int z = 0; z < 4; z++) {
        QFontMetrics fm(m->font(z));
        if (z == 0)
            tabWidth = fm.width("w");
        maxAscent = QMAX(maxAscent, fm.ascent());
        maxFontHeight = QMAX(maxFontHeight, fm.height());
    }

    fontHeight = maxFontHeight;
    fontAscent = maxAscent;
    tabsWidth = tabChars * tabWidth;
    longestLine = 0;
    updateMaxLength(0);
    resizeBuffer();
    tagAll();
    updateViews();
}

void TKTextDocument::setTabWidth(int chars)
{
    tabChars = QMIN(QMAX(chars, 1), 16);
    tabsWidth = tabChars * tabWidth;
    longestLine = 0;
    updateMaxLength(0);
    resizeBuffer();
    tagAll();
    updateViews();
}

void TKTextDocument::setMarked(int line, int m)
{
    while((line>0) && (lineOf(line)->status() & TKTextLine::Wrapped))
        line--;
    contents.at(line)->setMarked(m);
    for (TKTextView *view = views.first(); view; view = views.next())
        view->repaintMargins();
}

int TKTextDocument::marked(int line)
{
    while((line>0) && (lineOf(line)->status() & TKTextLine::Wrapped))
        line--;
    return contents.at(line)->marked() ;
}

void TKTextDocument::updateLines()
{
    updateLines(0, lastLine());
}

void TKTextDocument::updateLine(int line)
{
    updateLines(line, line);
}

void TKTextDocument::updateLines(int startLine, int endLine)
{
    int line = startLine;
    int ctxNum = line != 0 ? contents.at(line - 1)->context() : 0;

    while (line <= endLine) {
        ctxNum = hl->doHighlight(ctxNum, contents.at(line));
        contents.at(line)->setContext(ctxNum);
        updateMaxLength(contents.at(line));
        ++line;
    }

    while (line <= lastLine()) {
        ++line;
        ctxNum = hl->doHighlight(ctxNum, contents.at(line - 1));
        if (contents.at(line - 1)->context() == ctxNum)
            break;
        contents.at(line - 1)->setContext(ctxNum);
    }

    tagLines(startLine, line - 1);
}

void TKTextDocument::updateMaxLength(TKTextLine *textLine)
{
    if (longestLine && lineWidth(textLine) > maxLength) {
        newDocGeometry = true;
        longestLine = textLine;
        maxLength = lineWidth(textLine);
    } else
        if (!longestLine || textLine == longestLine) {
        newDocGeometry = true;
        maxLength = -1;
        for (int l = 0; l <= lastLine(); l++)
            if (maxLength < lineWidth(contents.at(l))) {
                maxLength = lineWidth(contents.at(l));
                longestLine = contents.at(l);
            }
    }
}

void TKTextDocument::resizeBuffer()
{
    delete paint;
    paint = 0;
    delete buffer;
    int w = 0;
    for (TKTextView *view = views.first(); view; view = views.next()) {
        w = QMAX(w, view->width());
        view->lineHeightChanged();
    }
    buffer = new QPixmap(w, lineHeight());
    if (buffer->width() != 0 && buffer->height() != 0)
        paint = new QPainter(buffer);
}

void TKTextDocument::updateViews()
{
    for (TKTextView *view = views.first(); view; view = views.next())
        view->updateView(newDocGeometry);

    newDocGeometry = false;
    updateRegion.clear();
}

int TKTextDocument::lineWidth(TKTextLine *textLine)
{
    int w = QFontMetrics(m->font(3)).width('W');
    int x = 0;
    for (uint p = 0; p < textLine->length(); p++)
        x += textLine->ref(p) == '\t' ? tabsWidth - (x % tabsWidth) : w;
    return x;
}

int TKTextDocument::textWidth(int line, int length)
{
    TKTextLine *textLine = lineOf(line);
    if (!textLine->isHighlighted())
        hl->highlight(textLine);

    int x = 0;
    int p;
    for ( p = 0; p < length && p < (int)textLine->length(); p++)
        if (textLine->ref(p) == '\t')
            x += tabsWidth - (x % tabsWidth);
        else {
            TKTextAttribute *a = hl->attribute(textLine->attribute(p));
            x += m->width(a->fontType, textLine->ref(p));
        }

    int space = m->width(0, QString(" ").ref(0));
    while (p < length) {
        x += space;
        p++;
    }
    return x;
}

int TKTextDocument::textLength(int line)
{
    return lineOf(line)->length();
}

int TKTextDocument::textLength(int line, int width)
{
    TKTextLine *textLine = lineOf(line);
    if (!textLine->isHighlighted())
        hl->highlight(textLine);

    int x = 0;
    uint p = 0;
    while (x < width && p < textLine->length()) {
        if (textLine->ref(p) == '\t')
            x += tabsWidth - (x % tabsWidth);
        else {
            TKTextAttribute *a = hl->attribute(textLine->attribute(p));
            x += m->width(a->fontType, textLine->ref(p));
        }
        p++;
    }
    int space = m->width(0, QString(" ").ref(0));
    while (x < width) {
        x += space;
        p++;
    }
    return p == 0 ? 0 : --p;
}

int TKTextDocument::documentWidth()
{
    return maxLength;
}

int TKTextDocument::documentHeight()
{
    return linePosition(lastLine()) + fontHeight;
}

void TKTextDocument::append(const QString &s)
{
    if (isReadOnly())
        return;

    insert(0, s, lastLine(), textLength(lastLine()));
}

QString TKTextDocument::html(int line)
{
    QString result;
    TKTextLine *textLine = lineOf(line);

    if (!textLine->isHighlighted())
        hl->highlight(textLine);

    int x = 0;
    int l = textLine->length();

    while (x < l) {
        int a = textLine->attribute(x);
        TKTextAttribute *set = hl->attribute(a);

        result += QString("<font color=\"%1\">").arg(set->color.name());
        if (set->fontType & 1)
            result += "<i>";
        if (set->fontType & 2)
            result += "<b>";

        int s = x;
        do {
            x++;
        } while (x < l && textLine->attribute(x) == a);

        QString str(&textLine->ref(s), x - s);
        int pos = str.find("\t");
        while (pos != -1) {
#if QT_VERSION >= 300
            str.replace(pos, 1, QString().fill(' ', tabChars - pos % tabChars));
#else
            QString temp;
            temp.fill(' ', tabChars - pos % tabChars);
            str.replace(pos, 1, temp);
#endif
            pos = str.find("\t");
        }

        str.replace(QRegExp("<"), "&lt;");
        result += str;

        if (set->fontType & 2)
            result += "</b>";
        if (set->fontType & 1)
            result += "</i>";
        result += "</font>";
    }

    return result;
}

void TKTextDocument::paintLine(TKTextView *view, int line)
{
    TKTextLine *textLine = lineOf(line);

    if (!textLine->isHighlighted())
        hl->highlight(textLine);

    buffer->fill(m->normalBackgroundColor);
    paint->translate(-view->xPos+2, 0);

    int p = 0;
    int x = 0;
    int l = textLine->length();
    TKTextAttribute *set = 0;

    if (l == 0 && line > 1) {
        int ln = line-1;
        TKTextLine *tL1 = lineOf(ln);
        while (tL1->length()==0 && ln>1) {
            ln--;tL1=lineOf(ln);
        }

        if (tL1->length()>0) {
            set = hl->attribute(tL1->attribute(tL1->length()-1));
#ifndef KOBOL
            if (set->usebg) paint->fillRect(0, 0, buffer->width()+view->xPos, fontHeight, set->bgcolor);
#endif
        }
    }

#ifdef KOBOL
    bool fte = false;
#endif
    while (p < l) {
        int a = textLine->attribute(p);

        int s = p;
        do {
            //if (textLine->ref(p) == '\t')
            //  break;
            p++;
        } while (p < l && textLine->attribute(p) == a);

        QConstString str(&textLine->ref(s), p - s);

        set = hl->attribute(a);

        int tl = m->width(set->fontType, str.string());
        if (set->usebg)
          paint->fillRect(x, 0, tl, fontHeight, set->bgcolor);

        x += tl;

        if (p < l && textLine->ref(p) == '\t') {
            p++;
            x += tabsWidth - (x % tabsWidth);
        }
#ifdef KOBOL
	if (p==l && textLine->ref(p-1)=='.') fte=true;
    }
    if (set && set->usebg){
	if (fte) paint->fillRect(x, 0, buffer->width()-x+view->xPos, fontHeight, set->bgcolor);
	fte = false;
#else
	if (set && set->usebg) paint->fillRect(x, 0, buffer->width()-x+view->xPos, fontHeight, set->bgcolor);
#endif
    }

    QRect frect;
/*************************************************************************************************************/
    frect = view->selected;
    if (frect.topLeft() != frect.bottomRight()) {
        QColor color = m->selectedBackgroundColor;
        bool fs = m->fVerticalSelection;
        QRect nr = frect.normalize();
        int sw = frect.width();
        int sh = frect.height();

        QRect rr = nr.intersect(QRect(0, line, nr.right() + 1, 1));
        if (!rr.isEmpty()) {
            int x = textWidth(line, rr.left());
            int w = textWidth(line, rr.right()) - x;
            if (fs || nr.height() == 1) {
                paint->fillRect(x, 0, w, fontHeight, color);
            } else {
                bool f = (sw > 0 && sh > 0) || (sw <= 0 && sh <= 0);
                int start = f ? x : x + w;
                start = start ? start : -2;
                int end = f ? x + w : x;
                if (line == nr.top())
                    paint->fillRect(start, 0, buffer->width() - start + view->xPos, fontHeight, color);
                else
                    if (line == nr.bottom()) {
                        if (end != 0)
                            paint->fillRect(-2, 0, end + 2, fontHeight, color);
                    } else
                        buffer->fill(color);
            }
        }
    }
/*************************************************************************************************************/
    frect = view->found;
    if (frect.topLeft() != frect.bottomRight()) {
        QColor color = m->foundBackgroundColor;
        bool fs = false;
        QRect nr = frect.normalize();
        int sw = frect.width();
        int sh = frect.height();

        QRect rr = nr.intersect(QRect(0, line, nr.right() + 1, 1));
        if (!rr.isEmpty()) {
            int x = textWidth(line, rr.left());
            int w = textWidth(line, rr.right()) - x;
            if (fs || nr.height() == 1) {
                paint->fillRect(x, 0, w, fontHeight, color);
            } else {
                bool f = (sw > 0 && sh > 0) || (sw <= 0 && sh <= 0);
                int start = f ? x : x + w;
                start = start ? start : -2;
                int end = f ? x + w : x;
                if (line == nr.top())
                    paint->fillRect(start, 0, buffer->width() - start + view->xPos, fontHeight, color);
                else
                    if (line == nr.bottom()) {
                        if (end != 0)
                            paint->fillRect(-2, 0, end + 2, fontHeight, color);
                    } else
                        buffer->fill(color);
            }
        }
    }
/*************************************************************************************************************/

/*
#ifndef QT_NO_PRINTER
    paint->setPen(Qt::lightGray);
    paint->drawLine(m->printWidth, 0, m->printWidth, fontHeight);
#endif
*/
    p = 0;
    x = 0;
    l = textLine->length();

    while (p < l) {
        int a = textLine->attribute(p);

        int s = p;
        do {
            if (textLine->ref(p) == '\t')
                break;
            p++;
        } while (p < l && textLine->attribute(p) == a);

        QConstString str(&textLine->ref(s), p - s);

        TKTextAttribute *set = hl->attribute(a);

        paint->setPen(set->color);
        paint->setFont(m->font(set->fontType));

        paint->drawText(x, fontAscent - 1, str.string());
        x += m->width(set->fontType, str.string());

        if (p < l && textLine->ref(p) == '\t') {
            p++;
            x += tabsWidth - (x % tabsWidth);
        }
    }

    paint->translate(view->xPos-2, 0);
}

void TKTextDocument::paintEmptyArea(TKTextView *view, const QRect &area)
{
    QPainter paint(view);
    paint.fillRect(area, m->normalBackgroundColor);
/*
#ifndef QT_NO_PRINTER
    paint.setPen(Qt::lightGray);
    paint.translate(-view->xPos+2, 0);
    paint.drawLine(m->printWidth, area.y(), m->printWidth, area.bottom());
#endif
*/
}

uint TKTextDocument::viewsCount()
{
    return views.count();
}

void TKTextDocument::setHighlight(const QString &s)
{
    setHighlight(TKTextHighlightManager::findHighlight(m, s));
}

void TKTextDocument::indent(TKTextView *view)
{
    if (isReadOnly())
        return;

    int line = view->cursor->line();
    int last = line + 1;
    QRect rect = view->selected;
    if (rect.topLeft() != rect.bottomRight()) {
        QRect normal = rect.normalize();

        line = normal.top();
        last = normal.bottom() + 1;

        bool f = (rect.width() > 0 && rect.height() > 0) ||
                 (rect.width() <= 0 && rect.height() <= 0);

        int end = f ? normal.right() : normal.left();
        if (end == 0 && normal.height() > 1)
            --last;
    }

#if QT_VERSION >= 300
    QString ins = m->fSaveTab ? QString("\t") : QString().fill(' ', tabChars);
#else
    QString temp;
    temp.fill(' ', tabChars);
    QString ins = m->fSaveTab ? QString("\t") : temp;
#endif
    recordStart(view->cursor->line(), view->cursor->column());
    for (; line < last; line++)
        doAction(TKEditorAction::insert(line, 0, ins));

    recordEnd(0, view->cursor->line(), view->cursor->column());
    view->editor->notifyChange(TKTextEditor::UndoRedo);
}

void TKTextDocument::unindent(TKTextView *view)
{
    if (isReadOnly())
        return;

    int line = view->cursor->line();
    int last = line + 1;
    QRect rect = view->selected;
    if (rect.topLeft() != rect.bottomRight()) {
        QRect normal = rect.normalize();

        line = normal.top();
        last = normal.bottom() + 1;

        bool f = (rect.width() > 0 && rect.height() > 0) ||
                 (rect.width() <= 0 && rect.height() <= 0);

        int end = f ? normal.right() : normal.left();
        if (end == 0 && normal.height() > 1)
            --last;
    }

#if QT_VERSION >= 300
    QString ins = m->fSaveTab ? QString("\t") : QString().fill(' ', tabChars);
#else
    QString temp;
    temp.fill(' ', tabChars);
    QString ins = m->fSaveTab ? QString("\t") : temp;
#endif
    recordStart(view->cursor->line(), view->cursor->column());
    for (; line < last; line++) {
        int length = 0;
        if (lineOf(line)->length() > 0 && lineOf(line)->ref(0) == '\t')
            length = 1;
        else
            length = QMIN(tabChars, lineOf(line)->firstChar());

        if (length > 0)
            doAction(TKEditorAction::remove(line, 0, length));
    }

    recordEnd(0, view->cursor->line(), view->cursor->column());
    view->editor->notifyChange(TKTextEditor::UndoRedo);
}

QString TKTextDocument::selectionCommand(TKTextView *view, SelectionCommand command)
{
  if (view->found.topLeft() != view->found.bottomRight())
    return selectionCommand(view, command, view->found, false);

  return selectionCommand(view, command, view->selected, m->fVerticalSelection);
}

QString TKTextDocument::selectionCommand(TKTextView *view, SelectionCommand command, const QRect &rect, bool vs)
{
    if (isReadOnly()) {
        if (command == SelectionRemove)
            return QString::null;
        if (command == SelectionCut)
            command = SelectionCopy;
    }

    QString result;

    if (rect.topLeft() == rect.bottomRight())
        return QString::null;

    QRect normal = rect.normalize();

    int line = normal.y();
    int last = normal.bottom();

    bool f = (rect.width() > 0 && rect.height() > 0) ||
             (rect.width() <= 0 && rect.height() <= 0);

    int start = f ? normal.left() : normal.right();
    int end = f ? normal.right() : normal.left();

    if (vs || normal.height() == 1) {
        if (command & SelectionRemove)
            recordStart(normal.bottom(), normal.right());

        int length = normal.width() - 1;
        int column = normal.x();
        while (line <= last && line <= lastLine() ) {
            if ((command & SelectionCopy || command & SelectionText) && column < (int)lineOf(line)->length())
                result += QConstString(&lineOf(line)->ref(column), QMIN(length, (int)lineOf(line)->length() - column)).string() + '\n';
            if (command & SelectionRemove)
                doAction(TKEditorAction::remove(line, column, length));
            ++line;
        }

        if (command & SelectionRemove)
            recordEnd(view, normal.y(), normal.x());

        if ((command & SelectionCopy || command & SelectionText) && !result.isEmpty())
            result.truncate(result.length() - 1);

    } else {
        if (command & SelectionRemove)
            recordStart(normal.bottom(), end);

        if ((command & SelectionCopy || command & SelectionText) && start < (int)lineOf(line)->length())
            result += QConstString(&lineOf(line)->ref(start), lineOf(line)->length() - start).string() + '\n';

        if (command & SelectionRemove)
            doAction(TKEditorAction::remove(line, start, lineOf(line)->length() - start));

        line++;

        int current = line;

        while (line < last && current <= lastLine()) {
            if (command & SelectionCopy || command & SelectionText)
                result += *lineOf(current) + '\n';

            if (command & SelectionRemove) {
                doAction(TKEditorAction::remove(current, 0, lineOf(current)->length()));
                doAction(TKEditorAction::removeLine(current));
            } else
                current++;

            line++;
        }


        if (current <= lastLine()) {
            if ((command & SelectionCopy || command & SelectionText) && !lineOf(current)->isEmpty())
                result += QConstString(&lineOf(current)->ref(0), QMIN(end, (int)lineOf(current)->length())).string();

            if (command & SelectionRemove) {
                doAction(TKEditorAction::remove(current, 0, end));
                doAction(TKEditorAction::removeLineBreak(current - 1));
            }

        }

        if (command & SelectionRemove)
            recordEnd(view, normal.y(), start);
    }

    if (command & SelectionCopy && !result.isEmpty())
#ifndef QT_NO_CLIPBOARD
        QApplication::clipboard()->setText(result);
#else
        m->clipboard = result;
#endif
    if( shouldWrap )
        rewrap(view);

    return command & SelectionText ? result : QString::null;
}

void TKTextDocument::insert(TKTextView *view, const QString &s, int line, int column)
{
    if (isReadOnly())
        return;

    QString ins;
    if (view) {
        line = view->cursor->line();
        column = view->cursor->column();
    }
    recordStart(line, column);

    QStringList lines = QStringList::split("\n", s, true);
    QStringList::Iterator l = lines.begin();
    while (l != lines.end()) {
        QStringList tabs = m->fSaveTab ? QStringList(*l) : QStringList::split("\t", *l, true);
        QStringList::Iterator t = tabs.begin();
        ins.truncate(0);
        while (t != tabs.end()) {
            ins += *t;
            t++;
            if (t != tabs.end()) { // tab
                TKTextLine *textLine = lineOf(line);
                int pos = 0;
                for (int x = 0; x < column; x++)
                    if (x < (int)textLine->length() && textLine->ref(x) == '\t')
                        pos += tabChars - (pos % tabChars);
                    else
                        pos++;

#if QT_VERSION >= 300
                ins += QString().fill(' ', tabChars - (pos + ins.length()) % tabChars);
#else
                QString temp;
                temp.fill(' ', tabChars - (pos + ins.length()) % tabChars);
                ins += temp;
#endif
            }
        }

        doAction(TKEditorAction::replace(line, column, m->fOvr ? ins.length() : 0, ins));
        l++;
        if (l != lines.end()) { // new line
            if (m->fVerticalSelection) {
                line++;
                if (line > lastLine())
                    doAction(TKEditorAction::makeLine(line));
            } else {
                column += ins.length();
                doAction(TKEditorAction::insertLineBreak(line, column));
                line++;
                column = 0;
            }
        } else
            column += ins.length();
    }
    // glueAll();
    
    recordEnd(view, line, column);
    if( shouldWrap && (s.length() > 1))
        rewrap(view);
}

void TKTextDocument::insertLineBreak(TKTextView *view, int line, int column)
{
    if (isReadOnly())
        return;

    if (view) {
        line = view->cursor->line();
        column = view->cursor->column();
    }
    recordStart(line, column);

    TKTextLine *textLine = lineOf(line);
    int pos = QMIN(column, textLine->firstChar());

    if (pos == -1)
        pos = column;

    doAction(TKEditorAction::insertLineBreak(line, column));
    line++;
    column = 0;

    if (pos > 0 && m->fAutoIndent) {
        QString ins;
        if ((int)textLine->length() < pos) {
            if (m->fSaveTab) {
                ins.fill('\t', pos / tabChars);
#if QT_VERSION >= 300
                ins += QString().fill(' ', pos % tabChars);
#else
                QString temp;
                temp.fill(' ', pos % tabChars);
                ins += temp;
#endif
            } else
                ins.fill(' ', pos);
        } else
            ins = textLine->left(pos);

        doAction(TKEditorAction::insert(line, column, ins));
        column = ins.length();
    }

    recordEnd(view, line, column);
}

void TKTextDocument::removeLine(TKTextView *view, int line, int column)
{
    if (isReadOnly())
        return;

    if (view) {
        line = view->cursor->line();
        column = view->cursor->column();
    }
    if (lastLine() == 0) {
        if (lineOf(0)->length() != 0) {
            recordStart(line, column);
            column = 0;
            doAction(TKEditorAction::remove(line, column, lineOf(line)->length()));
            recordEnd(view, line, column);
        }
    } else {
        recordStart(line, column);
        column = 0;
        doAction(TKEditorAction::remove(line, column, lineOf(line)->length()));
        doAction(TKEditorAction::removeLine(line));
        recordEnd(view, line, column);
    }
}

void TKTextDocument::del(TKTextView *view, int line, int column)
{
    if (isReadOnly())
        return;

    if (view) {
        line = view->cursor->line();
        column = view->cursor->column();
    }

    if(shouldWrap && (column == (int)lineOf(line)->length()))
    {
        if(line < lastLine())
        {
            line++;
            column = 0;
        }
    }

    if ((column < (int)lineOf(line)->length()))
    {
        recordStart(line, column);
        doAction(TKEditorAction::remove(line, column, 1));
        recordEnd(view, line, column);
    } else {
        if (line < lastLine()) {
            recordStart(line, column);
            doAction(TKEditorAction::removeLineBreak(line));
            recordEnd(view, line, column);
        }
    }
    if( shouldWrap )
    {
        int vc = view->cursor->column();
        int vl = view->cursor->line();
        int l = unwrap(view, line, &vl, &vc);
        if(l>=0)
            wrap(view, l, &vl, &vc);

        view->setCursorPosition(vl, vc);
        line = vl;
        column = vc;
    }
}

void TKTextDocument::backspace(TKTextView *view, int line, int column)
{
    if (isReadOnly())
        return;

    if (view) {
        line = view->cursor->line();
        column = view->cursor->column();
    }
    if (column <= 0 && line <= 0)
        return;

    recordStart(line, column);
    if (column > 0) {
        TKTextLine *textLine = lineOf(line);
        int pos = textLine->firstChar();
        int length = 1;

        if (m->fBackspaceIndent && (pos == -1 || pos >= column)) {
            pos = 0;
            int prev = line;
            while (prev != 0 && (textLine = contents.at(--prev))) {
                pos = textLine->firstChar();
                if (pos != -1 && pos < column)
                    break;
            }
            length = column - QMAX(0, pos);
            if (length == 0)
                length = pos;
        }
        column -= length;
        doAction(TKEditorAction::remove(line, column, length));
    } else {
        line--;
        column = lineOf(line)->length();
        doAction(TKEditorAction::removeLineBreak(line));
    }
    if( shouldWrap )
    {
        int vc = column;
        int vl = line;
        int l = unwrap(view, line, &vl, &vc);
        if(l>=0)
            wrap(view, l, &vl, &vc);

        line = vl;
        column = vc;
    }
    recordEnd(view, line, column);
}

int TKTextDocument::wrap(TKTextView *view, int line, int * vl, int *vc)
{

    int cline=0, rl=0;
    int column=1, c;
    int linecount = 0;
    if (view) {
        rl = cline = *vl;
        column = *vc;
    }

    if (column < 0 && cline < 0)
        return 0;

    if (!shouldWrap)
        return 0;

    // mike : don't bother if we are not visible
    //        maybe we need to trap QShowEvent in TKTextView and do a
    //        rewrap for (spontaneous) events
    if (!view->isVisible())
        return 0 ;

    TKTextLine *textLine = lineOf(line);

    // mike : paranoia rules!
    int vw = view->width() ;
    if (vw < 30)
        return 0 ;

    c = textLine->length();
    while( ((textWidth(line, c) + tabWidth) > vw))
    {
       c = QMAX((vw / tabWidth) -1, 1);
       // TODO: Optimize search
       while((c < (int)textLine->length()) && ((textWidth(line, c) + tabWidth) < vw))
           c++;
       c--;
       if (c >= (int)textLine->length()) break ;
       if(c < 0)
           c = 0;
       int oldc = c;
       if(nWrapMode == wrapWord)
       {
           // now let's find closest word boundary
           while((c > 0) && !wordBound.contains(textLine->ref(c)))
               c--;

           // not found :-/
           if(c == 0)
               c = oldc;
           else
           {
                // at least one delimiter found
                // we should keep it in the wrapping line
                if(c < oldc)
                    c++;
           }

       }

       insertLine(line + 1);
       /*
       line++;
        TKTextLine *l = new TKTextLine();
        int rn = line == 0 ? 0 : contents.at(line - 1)->number() + 1;
        l->setNumber(rn);
        l->setInternalNumber(line);
        contents.insert(line, l);
        rcontents.insert(rn, l);
        line--;
        */


        if (c < (int)textLine->length()) {
            lineOf(line + 1)->set(QConstString(&textLine->ref(c), textLine->length() - (c)).string());
            lineOf(line+1)->setStatus(lineOf(line)->status() | TKTextLine::Wrapped);
            linecount++;

            // copying attributes

            for(unsigned int i = 0; i< textLine->length() - c; i++)
                 lineOf(line+1)->setAttributes(textLine->attribute(c+i), i,i+1);

            lineOf(line+1)->doneHighlight();
            textLine->truncate(c);
        }
        // if(update)
        updateLines(line, line+1);

        if(cline > line)
            cline++;
        else 
        if((cline == line) && (column > c))
        {
            column -= c;
            cline = line+1;
        }

        line++;
        textLine = lineOf(line);
        c = textLine->length();
    }

    // view->cursor->set(rl, column, update);
    *vc = column;
    *vl = cline;
    return linecount;
}

void TKTextDocument::qwrap(TKTextView *view, int line, int column)
{

    if (view) {
        line = view->cursor->line();
        column = view->cursor->column();
    }

    if (column < 0 && line < 0)
        return;

    if (!shouldWrap)
        return;

    TKTextLine *textLine = lineOf(line);
    int pos = QMIN(column, textLine->firstChar());

    if (pos == -1)
        pos = column;
    
    /*
    if ( (column < (int)textLine->length()) 
            || ((line < lastLine()) && (lineOf(line+1)->status() & TKTextLine::Wrapped)))
            */
    {
        int vc = view->cursor->column();
        int vl = view->cursor->line();
        int l = unwrap(view, line, &vl, &vc);
        if(l>=0)
            wrap(view, l, &vl, &vc);
         view->setCursorPosition(vl, vc);
        vc = view->cursor->column();
        vl = view->cursor->line();
         line = vl;
         column = vc;
    } /*else
    {

        if( ((textWidth(line, column) + tabWidth) >= view->width()))
        {
            int c = textLine->length()-1;
            if ( c > 0 )
            {
                if( (line+1) <= lastLine() && 
                    (lineOf(line+1)->status() & TKTextLine::Wrapped) )
                {
                    line++;
                    column = 1;
                }
                else
                {
                    recordStart(line, column);
                    doAction(TKEditorAction::insertLineBreak(line, c+1));
                    line++;
                    column = lineOf(line)->length();
                    lineOf(line)->setStatus(lineOf(line)->status() | TKTextLine::Wrapped);
                    renumLines(line-1, false);
                    column = lineOf(line)->length();
                    recordEnd(view, line, column);
                }
            }
        }

    }
    */
    if(view)
    {
        if((line != view->cursor->line()) ||
            (column != view->cursor->column()))
        view->setCursorPosition(line, column);
    }

}

int TKTextDocument::unwrap(TKTextView *view, int line, int * vl, int *vc)
{

    if(!view)
        return -1;
    
    int column = *vc;
    int rl = *vl;
    
    if (line < 0 || column < 0)
        return -1;

    if((lineOf(line)->status() & TKTextLine::Wrapped))
        while((line >=  0) && (lineOf(line)->status() & TKTextLine::Wrapped))
            line --;
    if(line<0)
        line = 0;

    if(lineNum(line) == lineNum(rl))
        column = lineCol(rl, column);

    while((line+1 <=  lastLine()) && (lineOf(line+1)->status() & TKTextLine::Wrapped))
    {
        TKTextLine *textLine = lineOf(line);
        textLine->append(*lineOf(line + 1));
        tagLines(line, QMIN(lastLine(),line+view->height()/fontHeight)+1);
       
        /*
        TKTextLine *l = contents.at(line+1);
        if (longestLine == l)
            longestLine = 0;
        contents.remove(line+1);
        int rn = l->number();
        rcontents.remove(rn);
        */


        removeLine(line + 1);
        if(rl > line)
            rl--;
    }
    *vc = column;
    *vl = rl;
    hl->highlight(lineOf(line));
    return line;
}

void TKTextDocument::rewrap(TKTextView *view)
{
    if(!shouldWrap)
        return;

    // mike: without this, TKTextEditor::setText(....) will display
    //       unwrapped text in any views until the use does something
    if (view == 0)
    {   for (uint v = 0 ; v < views.count() ; v += 1)
            rewrap (views.at(v)) ;
	return ;
    }

	int i;

    int vc = view->cursor->column();
    int vl = view->cursor->line();
    int maxl = view->height() / fontHeight;
    int sl = QMAX(vl-maxl,0);
    int ss = QMIN(vl+maxl+1,lastLine()+1);
    int wsl=0, wss=0;
    for(i=sl; (i<ss) && (i <=  lastLine());i++)
    {
        int cl = unwrap(view, i, &vl, &vc);
        if(i==sl)
            wsl = cl;
        wss = cl;
    }

    int delta = 0;
    for(i=wsl;i<=wss;i++)
    {
        delta = wrap(view, i, &vl, &vc);
        i += delta;
        wss += delta;
    }
    updateLines(wsl,wss);
    view->setCursorPosition(vl, vc);

}


void TKTextDocument::clear()
{
    tagAll();

    undoList.clear();
    currentUndo = 0;

    contents.clear();
    rcontents.clear();

    insertLine(0);

    longestLine = 0;
    updateLine(0);

    for (TKTextView *view = views.first(); view; view = views.next())
        view->setCursorPosition(0, 0);

    documentChanged(false);
}

void TKTextDocument::insertLine(int line)
{
    TKTextLine *l = new TKTextLine();
    int rn = line == 0 ? 0 : contents.at(line - 1)->number() + 1;
    l->setNumber(rn);
    l->setInternalNumber(line);
    contents.insert(line, l);
    rcontents.insert(rn, l);

    renumLines(line, true);
    renumLines(rn, false);
    newDocGeometry = true;
    tagLines(line, lastLine());
}

void TKTextDocument::removeLine(int line)
{
    tagLines(line, lastLine());

    TKTextLine *l = contents.at(line);
    if (longestLine == l)
        longestLine = 0;
    contents.remove(line);
    int rn = l->number();
    rcontents.remove(rn);

    renumLines(line, true);
    renumLines(rn, false);
    newDocGeometry = true;
}

void TKTextDocument::renumLines(int start, bool internal)
{
  int last = internal ? contents.size() : rcontents.size();
  int id = start;
  if (internal) {
    while (id < last) {
        contents.at(id)->setInternalNumber( id );
        /* if(contents.at(id)->status() & TKTextLine::Wrapped)
            contents.at(id)->setInternalNumber( contents.at(id-1)->internalNumber() );
        else
            if( id > 0)
                contents.at(id)->setInternalNumber( contents.at(id-1)->internalNumber() +1 );
            else
                contents.at(id)->setInternalNumber( id );
                */

      id++;
    }
  } else {
    while (id < last) {
        rcontents.at(id)->setNumber( id );
        /* if(contents.at(id)->status() & TKTextLine::Wrapped)
            contents.at(id)->setNumber( contents.at(id-1)->number() );
        else
            if( id > 0)
                contents.at(id)->setNumber( contents.at(id-1)->number()+1 );
            else
                contents.at(id)->setNumber( id );
                */
        id++;
    }
  }
}

void TKTextDocument::doAction(TKEditorAction *a)
{
    if (isReadOnly())
        return;

    TKTextLine *textLine = lineOf(a->line);
    switch (a->action) {
    case TKEditorAction::Replace: {
            int length = QMIN((int)textLine->length() - a->column, a->length);
            QString old = length > 0 ? QConstString(&textLine->ref(a->column), length).string(): QString::null;
            textLine->replace(a->column, a->length, a->text);
            a->length = a->text.length();
            a->text = old;
            break;}
    case TKEditorAction::InsertLineBreak:
        insertLine(a->line + 1);
        if (a->column < (int)textLine->length()) {
            lineOf(a->line + 1)->set(QConstString(&textLine->ref(a->column), textLine->length() - a->column).string());
            textLine->truncate(a->column);
        }
        if(((textLine->status() & TKTextLine::Slave) ||
                (textLine->status() & TKTextLine::Mater)) && 
                !(textLine->status() & TKTextLine::Last))
            lineOf(a->line + 1)->setStatus( lineOf(a->line + 1)->status() |
                    TKTextLine::Slave);
        a->action = TKEditorAction::RemoveLineBreak;
        break;
    case TKEditorAction::RemoveLineBreak:
        a->column = textLine->length();
        textLine->append(*lineOf(a->line + 1));
        removeLine(a->line + 1);
        a->action = TKEditorAction::InsertLineBreak;
        break;
    case TKEditorAction::MakeLine:
        insertLine(a->line);
        a->action = TKEditorAction::RemoveLine;
        break;
    case TKEditorAction::RemoveLine:
        removeLine(a->line);
        a->action = TKEditorAction::MakeLine;
        break;
    case TKEditorAction::Fold: {
        int status = textLine->status();
        if (status & TKTextLine::Mater)
          if (status & TKTextLine::Close)
            unfoldLine(a->line);
          else
            foldLine(a->line);
        break; }
    }
    updateLine(a->line);

    actionsStack->push(a);
}

void TKTextDocument::recordStart(int line, int column)
{
    if (startRecordLock)
      return;

    while ((int)undoList.count() > currentUndo)
        undoList.removeLast();

    while ((int)undoList.count() > undoSteps) {
        undoList.removeFirst();
        currentUndo--;
    }

    undoList.append(new TKEditorActionGroup());

    undoList.getLast()->setUndoCursor(line, column);
    actionsStack = undoList.getLast()->actions;
    currentUndo++;
}

void TKTextDocument::recordEnd(TKTextView *view, int line, int column)
{
    if (view)
        view->setCursorPosition(line, column);

    if (endRecordLock)
      return;

    undoList.getLast()->setRedoCursor(line, column);
    actionsStack = 0;
    documentChanged(true);

    if (view)
      view->editor->notifyChange(TKTextEditor::UndoRedo);
}

void TKTextDocument::doActionGroup(TKEditorActionGroup *g)
{
    actionsStack = new QPtrStack<TKEditorAction>;
    while (g->actions->current())
        doAction(g->actions->pop());

    delete g->actions;
    g->actions = actionsStack;
    actionsStack = 0;
}

bool TKTextDocument::canUndo()
{
    return currentUndo != 0;
}

void TKTextDocument::undo(TKTextView *view)
{
    if (currentUndo != 0) {
        currentUndo--;
        TKEditorActionGroup *a = undoList.at(currentUndo);
        doActionGroup(a);
        view->setCursorPosition(a->undoLine, a->undoColumn);
        documentChanged(true);
        view->editor->notifyChange(TKTextEditor::UndoRedo);
    }
}

bool TKTextDocument::canRedo()
{
    return currentUndo != (int)undoList.count();
}

void TKTextDocument::redo(TKTextView *view)
{
    if (currentUndo != (int)undoList.count()) {
        TKEditorActionGroup *a = undoList.at(currentUndo);
        currentUndo++;
        doActionGroup(a);
        view->setCursorPosition(a->redoLine, a->redoColumn);
        documentChanged(true);
        view->editor->notifyChange(TKTextEditor::UndoRedo);
    }
}

void TKTextDocument::documentChanged(bool isModified)
{
    TKTextView *view;
    for ( view = views.first(); view; view = views.next() )
        view->editor->notifyChange(TKTextEditor::Modification);

    if (modified != isModified) {
        modified = isModified;
        for ( view = views.first(); view; view = views.next() )
            view->editor->notifyChange(TKTextEditor::Modification);
    }
    updateViews();
}

void TKTextDocument::makeFold(TKTextView *, int fromLine, int toLine)
{
  if (fromLine > lastLine() || toLine > lastLine() || fromLine >= toLine)
    return;

  TKTextLine *start = lineOf(fromLine);
  start->setStatus(start->status() | TKTextLine::Mater | TKTextLine::Open);

  TKTextLine *line = 0;
  int num = fromLine + 1;
  while (num <= toLine) {
    line = lineOf(num++);
    line->setStatus(line->status() | TKTextLine::Slave);
  }
  line->setStatus(line->status() | TKTextLine::Last);

  tagLines(fromLine, toLine);
  updateViews();
}

void TKTextDocument::fold(TKTextView *view, int line)
{
  recordStart(view->cursor->line(), view->cursor->column());
  doAction(TKEditorAction::fold(line));
  recordEnd(view, line, 0);
}

void TKTextDocument::foldLine(int line)
{
  TKTextLine *start = lineOf(line);
  start->setStatus((start->status() ^ TKTextLine::Open) | TKTextLine::Close);

  int toLine = line;
  do {
    start->appendChild(lineOf(++toLine));
  } while (!(lineOf(toLine)->status() & TKTextLine::Last));

  tagLines(line, lastLine());

  int id = line + 1;
  int num = toLine + 1;
  while (num != (int)contents.size())
    contents.at(id++) = contents.at(num++);
  contents.resize(id);

  renumLines(line, true);

  longestLine = 0;
  updateMaxLength(0);
}

void TKTextDocument::unfoldLine(int line)
{
  TKTextLine *start = lineOf(line);
  QPtrList<TKTextLine> *sublines = start->children();
  if (!sublines)
    return;

  start->setStatus((start->status() ^ TKTextLine::Close) | TKTextLine::Open);

  int id = line + 1;
  int source = contents.size();
  int count = (int)sublines->count();
  contents.resize(contents.size() + count);
  int dest = contents.size();

  while (source != id)
    contents.at(--dest) = contents.at(--source);

  for (TKTextLine *sub = sublines->first(); sub; sub = sublines->next())
    contents.at(id++) = sub;

  start->removeChildren();

  renumLines(line, true);
  tagLines(line, lastLine());
  longestLine = 0;
  updateMaxLength(0);
}


void TKTextDocument::setWrapMode(enum wrapMode mode)
{
    if(nWrapMode != mode)
        nWrapMode = mode;
}

void TKTextDocument::setWordBound(QString bound)
{
    wordBound = bound;
}
