/***************************************************************************
    file	         : tk_helpwindow.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<unistd.h>
#include	<errno.h>
#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/wait.h>

#include	<qtextbrowser.h>
#include	<qfile.h>
#include	<qtextstream.h>
#include	<qdom.h>

#include	<kapp.h>
#include	<kconfig.h>
#include	<klocale.h>
#include	<kaboutdata.h>
#include	<kcmdlineargs.h>

#include	"kb_classes.h"
#include	"tk_helpwindow.moc"
#include	"tk_helpproxy.h"


static	const	KCmdLineOptions	options[] =
{
	{	"+helpDir",	I18N_NOOP("Help directory"),	0	},
	{	0,		0,				0	}
}	;



/*  ------------------------------------------------------------------  */

/*  TKHelpPage								*/
/*  TKHelpPage	: Constructor for a help page				*/
/*  parent	: QWidget *	 : Parent widget			*/
/*  helpWin	: TKHelpWindow * : Owning help window			*/
/*  (returns)	: TKHelpPage	 : Help page				*/

TKHelpPage::TKHelpPage
	(	QWidget		*parent,
		TKHelpWindow	*helpWin
	)
	:
	QTextBrowser	(parent),
	m_helpWin	(helpWin)
{
	/* All this does is construct the widget and saves a pointer	*/
	/* at the help window. The page and the text will be set later	*/
	/* by the caller.						*/
}

/*  TKHelpPage								*/
/*  ~TKHelpPage	: Destructor for a help page				*/
/*  (returns)	:		:					*/

TKHelpPage::~TKHelpPage ()
{
}

/*  TKHelpPage								*/
/*  setSource	: Handled for user clicking link			*/
/*  link	: const QString & : Link text				*/
/*  (returns)	: void		  :					*/

void	TKHelpPage::setSource
	(	const QString	&link
	)
{
	m_helpWin->showHelpText (this, link) ;
}



/*  ------------------------------------------------------------------  */

/*  TKHelpWindow							*/
/*  TKHelpWindow: Constructor for help window main widget		*/
/*  helpDir	: const QString & : Directory containing help files	*/
/*  (returns)	: TKHelpWindow	  :					*/

TKHelpWindow::TKHelpWindow
	(	const QString	&helpDir
	)
	:
	QDialog		(0, 0, false, WDestructiveClose),
	m_helpDir	(helpDir),
	m_tabber	(this),
	m_bConfig	(this),
	m_bDismiss	(this),
	m_bClose	(this)
{
	m_layMain    = new QVBoxLayout (this) ;
	m_layMain    ->setMargin  (4) ;
	m_layMain    ->setSpacing (8) ;
	m_layMain    ->addWidget  (&m_tabber) ;

	m_layControls = new QHBoxLayout (m_layMain) ;
	m_layControls ->setMargin  (4) ;
	m_layControls ->setSpacing (8) ;
	m_layControls ->addWidget  (&m_bConfig ) ;
	m_layControls ->addStretch ()	     ;
	m_layControls ->addWidget  (&m_bDismiss) ;
	m_layControls ->addWidget  (&m_bClose  ) ;

	m_bDismiss.setEnabled 	 (false) ;
	m_bDismiss.setText	 (TR("Dismiss" )) ;
	m_bClose  .setText	 (TR("Close"   )) ;
	m_bConfig .setText	 (TR("Config"  )) ;

	KConfig *config	= kapp->config() ;
	config->setGroup  ("Options")    ;

	m_linkToNew = config->readBoolEntry 	   ("linkToNew", false) ;
	m_maxPages  = config->readUnsignedNumEntry ("maxPages",      5) ;

	m_pCache  .setMaxCost    (m_maxPages) ;
	m_pCache  .setAutoDelete (true) ;

	connect	(&m_bClose,   SIGNAL(clicked()), SLOT(slotClose      ())) ;
	connect	(&m_bDismiss, SIGNAL(clicked()), SLOT(slotDismiss    ())) ;
	connect	(&m_bConfig,  SIGNAL(clicked()), SLOT(slotConfig     ())) ;

	m_socket.setSocket (0) ;
	connect	(&m_socket,   SIGNAL(connectionClosed()), SLOT(slotLinkClosed ())) ;
	connect	(&m_socket,   SIGNAL(readyRead	     ()), SLOT(slotReadyRead  ())) ;

	m_tabber.setMinimumSize (500, 250) ;

	QSize	size = config->readSizeEntry ("size") ;
	if (size != QSize()) resize (size) ;
	show	() ;
}

/*  TKHelpWindow							*/
/*  ~TKHelpWindow: Destructor for help window main widget		*/
/*  (returns)	 :		:					*/

TKHelpWindow::~TKHelpWindow ()
{
	/* Don't delete the tap pages twice ...				*/
	m_pCache.setAutoDelete (false) ;

	KConfig *config	 = kapp->config() ;
	config->setGroup   ("Options")    ;
	config->writeEntry ("size", size()) ;
}

/*  TKHelpWindow							*/
/*  bringToTop	: Bring the help window to the top			*/
/*  (returns)	: void		:					*/

void	TKHelpWindow::bringToTop ()
{
	/* This seems to cover most eventualities ....			*/
	show		() ;
	raise		() ;
	setActiveWindow () ;
}

/*  TKHelpWindow							*/
/*  showHelpText: Show a specified help page				*/
/*  page	: const QString & : Page name or error text		*/
/*  (returns)	: void		  :					*/

void	TKHelpWindow::showHelpText
	(	const QString	&page
	)
{
	showHelpText	(0, page) ;
}

void	TKHelpWindow::loadHelpText
	(	const QString	&page,
		QString		&legend,
		QString		&text
	)
{
	/* If the page name begins with the '<' character then treat	*/
	/* is as XML, otherwise we will get the XML from the help file	*/
	/* for thee page.						*/
	if (page.at(0) == '<')
	{
		legend	= "Error" ;
		text	= page	  ;
		return	;
	}

	QFile	file	(m_helpDir + page + ".qt") ;
	if (!file.open (IO_ReadOnly))
	{
		legend	= "Error" ;
		text	= QString ("<qt>Cannot open help<br/>(File <i>%2</i>)</qt>")
				.arg(m_helpDir + page) ;
		return	;
	}

	QTextStream	stream (&file) ;
	QDomDocument	doc    ;

	doc.setContent (stream.read()) ;
	QDomElement	root   = doc.documentElement() ;

	for (QDomNode child = root.firstChild() ; !child.isNull() ; child = child.nextSibling())
	{
		QDomElement e = child.toElement() ;

		if (e.tagName() == "legend")
		{
			legend = e.text() ;
			continue ;
		}

		if (e.tagName() == "qt")
		{
			QTextStream  out (&text, IO_WriteOnly) ;
			e.save (out, 0) ;
			continue ;
		}
	}
}

/*  TKHelpWindow							*/
/*  showHelpText: Show a specified help page				*/
/*  browser	: TKHelpWindow * : Help window				*/
/*  page	: QString &	 : Page					*/
/*  (returns)	: void		 :					*/

void	TKHelpWindow::showHelpText
	(	TKHelpPage	*browser,
		const QString	&page
	)
{
	/* First see if we have the page on display already. If so	*/
	/* then show that page. The lookup moves it to the front of the	*/
	/* cache if it is found.					*/
	TKHelpPage	*ptr ;
	if ((ptr = m_pCache.find (page)) != 0)
	{
		m_tabber.showPage (ptr) ;
		bringToTop () ;
		return	;
	}

	if ((browser != 0) && !m_linkToNew)
	{
		/* We are reusing the same page, in which case we need	*/
		/* to re-enter it into the cache under the new page	*/
		/* name.						*/
		m_pCache.take 	    (browser->page()) ;
		m_pCache.insert     (page, browser) ;
	}
	else
	{
		/* Either no page given or linking opens a new page. In	*/
		/* this case we allocate a new help page.		*/
		browser = new TKHelpPage (&m_tabber, this) ;
		m_tabber .addTab    (browser, "") ;
		m_tabber .showPage  (browser) ;
		m_pCache .insert    (page, browser) ;
	}

	QString	legend	;
	QString	text	;
	loadHelpText (page, legend, text) ;

	browser  ->setPage    (page) ;
	browser  ->setText    (text) ;
	browser  ->show       () ;
	m_tabber  .changeTab  (browser, legend) ;
	m_bDismiss.setEnabled (true) ;

	bringToTop () ;
}

/*  TKHelpWindow							*/
/*  slotConfig	: User clicks configuration button			*/
/*  (returns)	: void		:					*/

void	TKHelpWindow::slotConfig ()
{
	TKHelpDialog hDlg (this, m_linkToNew, m_maxPages) ;
	if (hDlg.exec())
	{
		KConfig *config	= kapp->config() ;
		config->setGroup  ("Options")    ;

		config->writeEntry ("linkToNew", m_linkToNew) ;
		config->writeEntry ("maxPages",  m_maxPages ) ;

		m_pCache.setMaxCost (m_maxPages) ;
	}
}

/*  TKHelpWindow							*/
/*  slotClose	: User clicks close button				*/
/*  (returns)	: void		:					*/

void	TKHelpWindow::slotClose ()
{
	close	()	;
}

/*  TKHelpWindow							*/
/*  slotDismiss	: User clicks dismiss button				*/
/*  (returns)	: void		:					*/

void	TKHelpWindow::slotDismiss ()
{
	m_pCache.remove (((TKHelpPage *)m_tabber.currentPage())->page()) ;
	m_bDismiss.setEnabled (m_pCache.count() > 0) ;
}

/*  TKHelpWindow							*/
/*  slotLinkClosed : Link to parent process terminates			*/
/*  (returns)	   : void	:					*/

void	TKHelpWindow::slotLinkClosed ()
{
	_exit	(0) ;
}

void	TKHelpWindow::lruLimitChanged
	(	int	limit
	)
{
	m_pCache.setMaxCost (limit) ;
}

/*  TKHelpWindow							*/
/*  slotReadyRead: Help display request has arrived			*/
/*  (returns)	 : void		:					*/

void	TKHelpWindow::slotReadyRead ()
{
	TKHelpRequest	req	;

	if (m_socket.readBlock ((char *)&req, sizeof(req)) < 0)
	{
		fprintf	(stderr, "TKHelpWindow::slotReadyRead() error\n") ;
		_exit	(0) ;
	}

	showHelpText (req.page) ;
}


/*  ------------------------------------------------------------------  */

/*  TKHelpDialog							*/
/*  TKHelpDialog: Constructor for help window configuration dialog	*/
/*  parent	: QWidget *	: Parent widget				*/
/*  linkToNew	: bool &	: Link-to-new setting			*/
/*  maxPages	: uint &	: Maximum pages setting			*/
/*  (returns)	: TKHelpDialog	:					*/

TKHelpDialog::TKHelpDialog
	(	QWidget		*parent,
		bool		&linkToNew,
		uint		&maxPages
	)
	:
	QDialog		(parent, 0, true),
	m_linkCheck	(this),
	m_maxSpin	(this),
	m_linkLabel	(this),
	m_maxLabel	(this),
	m_bOK		(this),
	m_bCancel	(this),
	m_layout	(this),
	m_linkToNew	(linkToNew),
	m_maxPages	(maxPages)
{
	m_layout.addWidget (&m_linkLabel, 0, 0) ;
	m_layout.addWidget (&m_maxLabel,  1, 0) ;
	m_layout.addWidget (&m_linkCheck, 0, 1) ;
	m_layout.addWidget (&m_maxSpin,   1, 1) ;
	m_layout.addWidget (&m_bOK,       0, 2) ;
	m_layout.addWidget (&m_bCancel,   1, 2) ;
	m_layout.setMargin (4) ;
	m_layout.setSpacing(8) ;

	m_linkCheck.setChecked	(m_linkToNew) ;
	m_linkLabel.setText	(TR("Open links in new tab")) ;
	m_maxLabel .setText	(TR("Maximum pages")) ;
	m_maxSpin  .setRange	(2, 100) ;
	m_maxSpin  .setValue	(maxPages) ;
	m_bOK	   .setText	(TR("OK"    )) ;
	m_bCancel  .setText	(TR("Cancel")) ;

	setCaption ("RekallHelp Configuration") ;

	connect	(&m_bOK,     SIGNAL(clicked()), SLOT(clickOK    ())) ;
	connect	(&m_bCancel, SIGNAL(clicked()), SLOT(clickCancel())) ;
}

void	TKHelpDialog::clickOK ()
{
	m_linkToNew = m_linkCheck.isChecked() ;
	m_maxPages  = m_maxSpin  .value    () ;

	done	(1) ;
}

void	TKHelpDialog::clickCancel ()
{
	done	(0) ;
}

/*  ------------------------------------------------------------------  */

int	main
	(	int	argc,
		char	*argv[]
	)
{
	KAboutData about ("rekallHelp",
			  I18N_NOOP("RekallHelp"),
			  "1.0",
			  I18N_NOOP("Help display for Rekall"),
			  KAboutData::License_Unknown,
			  "(c) 2001 by Mike Richardson",
			  "(c) 2001 by theKompany.com"
			 ) ;
	about.addAuthor  ("Mike Richardson", 0, "mike@quaking.demon.co.uk") ; 

	KCmdLineArgs::init		(argc, argv, &about) ;
	KCmdLineArgs::addCmdLineOptions (options) ;

	KApplication app   	(argc, argv) ;
	KCmdLineArgs *args	= KCmdLineArgs::parsedArgs () ;

	freopen	("/dev/tty", "w", stderr) ;

	if (args->count() == 0)
	{	fprintf	(stderr, "Missing help directory\n") ;
		_exit	(1) ;
	}

	app.setMainWidget (new TKHelpWindow (args->url(0).path())) ;
	return	app.exec  () ;
}



