/***************************************************************************
    file	         : kb_kjsscript.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<math.h>

#include	<qintdict.h>

#include	"kb_classes.h"
#include	"kb_node.h"

#include	"kb_kjsscript.moc"

#include	"kb_idbase.h"
#include	"kb_objectproxy.h"
#include	"kb_rekallmain.h"
#include	"kb_kjdebug.h"
#include	"kb_kjsregister.h"


#define	EXPRFUNC	"exprFunc"
#define	EXPRWRAP	"function " EXPRFUNC " () { return %1 ; }"


static	KBKJSScriptIF	*scrIface	;


QString	kjsStringArg
	(	ExecState	*exec,
		const List	&args,
		int		index
	)
{
	return	index >= args.size()  ?
			QString::null :
			args.at(index).toString(exec).qstring() ;
}


namespace KJS
{
	UString::UString
		(	const QString	&text
		)
	{
		uint	length	= text.length ()    ;
		UChar	*data	= new UChar[length] ;

		memcpy	(data, text.unicode(), length * sizeof(UChar)) ;
		rep	= UString::Rep::create(data, length) ;
	}

	QString	UString::qstring () const
	{
		return	QString	((QChar *)data(), size()) ;
	}
}


/*  ------------------------------------------------------------------  */


static	QIntDict<KBKJSScriptCode>	scriptCodeMap	;

static	QString				errMsg		;
static	uint				errLno		;
static	QString				errText		;
static	int				errSourceID	;

/*  ------------------------------------------------------------------  */

/*  KBKJSScriptCode							*/
/*  KBKJSScriptCode							*/
/*		: Constructor for inline script object			*/
/*  interpreter	: KJS::Interpreter *	: KJS interpreter object	*/
/*  script	: const QString &	: Script code			*/
/*  fname	: const QString &	: Entry function name		*/
/*  location	: KBLocation &		: Nominal location		*/
/*  ok		: bool &		: Return success		*/
/*  (returns)	: KBKJSScriptCode	: Inline script object		*/

KBKJSScriptCode::KBKJSScriptCode
	(	KJS::Interpreter	*interpreter,
		const QString		&script,
		const QString		&fname,
		KBLocation		&location,
		bool			&ok
	)
	:
	m_interpreter	(interpreter),
	m_location	(location)
{
	/* Evaluate the code. This should result in the named function	*/

	/* as a property of the interpreters global object. Check first	*/
	/* that the execution is successful.				*/
	KJS::Completion compVal	= m_interpreter->evaluate
				  (	script,
					m_interpreter->globalObject()
				  )	;

	fprintf
	(	stderr,
		"KBKJSScriptCode::KBKJSScriptCode: got ct=%d val=%d\n",
		(int)compVal.complType(),
		(int)compVal.value().isValid()
	)	;

	switch (compVal.complType())
	{
		case KJS::Normal	:
		case KJS::ReturnValue	:
			break	;

		case KJS::Break		:
		case KJS::Continue	:
		case KJS::Throw		:
			ok	= false	;
			return	;

		default	:
			ok	= false	;
			return	;
	}

	/* Retrieve the function; checking that we get a valid value,	*/
	/* that it is an object, and that it implements "call" (if not	*/
	/* then it is not a function).					*/
	KJS::Value	value	= m_interpreter->globalObject().get(m_interpreter->globalExec(), fname) ;
	if (!value.isValid())
	{
		fprintf
		(	stderr,
			"KBKJSScriptCode::KBKJSScriptCode: no function\n"
		)	;
		ok	= false	;
		return	;
	}

	fprintf
	(	stderr,
		"KBKJSScriptCode::KBKJSScriptCode: got vt=%d\n",
		(int    )value.type()
	)	;

	if (!value.type() == KJS::ObjectType)
	{
		fprintf
		(	stderr,
			"KBKJSScriptCode::KBKJSScriptCode: not object type\n"
		)	;
		ok	= false	;
		return	;
	}

	m_kjsFunc = value.toObject(m_interpreter->globalExec()) ;
	if (!m_kjsFunc.implementsCall())
	{
		fprintf
		(	stderr,
			"KBKJSScriptCode::KBKJSScriptCode: call not implemented\n"
		)	;
		ok	= false	;
		return	;
	}

	m_sourceID = KBKJSDebugger::self()->lastSourceID() ;
	scriptCodeMap.insert (m_sourceID, this) ;

	ok	= true	;
}

/*  KBKJSScriptCode							*/
/*  ~KBKJSScriptCode							*/
/*		: Destructor for inline script object			*/
/*  (returns)	:		:					*/

KBKJSScriptCode::~KBKJSScriptCode ()
{
	scriptCodeMap.remove (m_sourceID) ;
}


/*  KBKJSScriptCode							*/
/*  execute	: Execute function					*/
/*  node	: KBNode *	  : Invoking rekall object		*/
/*  argc	: uint		  : Argument count			*/
/*  argv	: const KBValue * : Argument vector			*/
/*  resval	: KBValue *	  : Function return value		*/
/*  (return)	: KBScript::ExeRC : Execution return code		*/

KBScript::ExeRC
	KBKJSScriptCode::execute
	(	KBNode		*node,
		uint		argc,
		const KBValue	*argv,
		KBValue		&resval
	)
{
	KJS::List   	args  	;
	KJS::ExecState	*exec	= m_interpreter->globalExec() ;
	KBObjectProxy	*proxy	= makeProxy (m_interpreter, node) ;

	if (proxy == 0)
	{
		KBError::EFault
		(	QString(TR("Failed to locate KJS class for %1")).arg(node->getElement()),
			QString::null,
			__ERRLOCN
		)	;
		return	KBScript::ExeError ;
	}

	KJS::Object object (proxy) 	  ;
	proxy->addBindings (exec, object) ;
	args  .append	   (object)	  ;

	/* Convert the KBValue argument values to KJS values and stack them	*/
	/* up in the argument list object.					*/
	for (uint idx = 0 ; idx < argc ; idx += 1)
	{
		switch (argv[idx].getType()->getIType())
		{
			case KB::ITFixed    :
				args.append (KJS::Number (argv[idx].getRawText().toInt   ())) ;
				break	;

			case KB::ITFloat    :
			case KB::ITDecimal  :
				args.append (KJS::Number (argv[idx].getRawText().toDouble())) ;
				break	;

			case KB::ITString   :
			case KB::ITBinary   :
				args.append (KJS::String (argv[idx].getRawText())) ;
				break	;

			case KB::ITBool     :
				args.append (KJS::Boolean(argv[idx].isTrue())) ;
				break	;

			case KB::ITDate	    :
			case KB::ITTime	    :
			case KB::ITDateTime :
				args.append (KJS::String (argv[idx].getDateTime()->defFormat (argv[idx].getType()->getIType()))) ;
				break	;

			case KB::ITUnknown  :
			case KB::ITRaw	    :
			case KB::ITDriver   :
				args.append (KJS::String (argv[idx].getRawText())) ;
				break	;

			default	:
				return	KBScript::ExeError ;
		}
	}

	/* Call the function. We pass a dummy object as the object	*/
	/* argument, since the invoking object is passed as the first	*/
	/* argument.							*/
	KJS::Object 	dummy	;
	KJS::Value 	value	= m_kjsFunc.call(exec, dummy, args) ;

	delete	proxy	;

	if (exec->hadException())
	{
		fprintf
		(	stderr,
			"KBKJSScriptCode::execute: Execption occurred\n"
		)	;

		errSourceID	= KBKJSDebugger::self()->lastSourceID() ;
		errLno		= KBKJSDebugger::self()->lastLine    () ;
		errText		= ""		;
		errMsg		= "Exception"	;

		exec->clearException ()	   ;
		return	KBScript::ExeError ;
	}

	/* The value returned is converted back to a KBValue ..		*/
	switch (value.type())
	{
		case KJS::UnspecifiedType :
		case KJS::UndefinedType	  :
		case KJS::NullType	  :
			resval	= KBValue() ;
			break	;

		case KJS::BooleanType	:
			resval	= KBValue
				  (	value.toBoolean(exec),
					&_kbBool
				  )	;
			break	;

		case KJS::StringType	:
			resval	= KBValue
				  (	value.toString (exec).qstring(),
					&_kbString
				  )	;
			break	;

		case KJS::NumberType	:
		{
			double	h ;
			double	d = value.toNumber (exec) ;
			double	f = modf (d, &h) ;

			fprintf	(stderr, "[%f][%f][%f]\n", d, h, f) ;

			if (f == 0.0)
				resval	= KBValue ((int)h, &_kbFixed) ;
			else	resval	= KBValue (d,	   &_kbFloat) ;

			break	;
		}

		case KJS::ObjectType	:
		case KJS::ReferenceType	:
		case KJS::ListType	:
		case KJS::CompletionType:
			resval	= KBValue() ;
			break	;

		default	:
			resval	= KBValue() ;
			break	;
	}

	return	KBScript::ExeTrue ;
}



/*  ------------------------------------------------------------------  */


KBScript::ExeRC
	KBKJSScriptCode::execute
	(	KBNode		*,
		KBNode		*,
		const QString	&,
		uint		,
		const KBValue	*,
		KBValue		&
	)
{
	return	KBScript::ExeError ;
}



KJS::Value
	KBKJSScriptIF::compileText
	(	KBLocation		&,
		const QString		&,
		QString			&,
		QString			&,
		KBError			&
	)
{
	return	KJS::Value() ;
}


KBScriptCode
	*KBKJSScriptIF::compileInline
	(	const QString		&,
		const QString		&,
		QString			&,
		QString			&,
		const QStringList	&,
		KBError			&
	)
{
	return	0 ;
}

KJS::Value
	KBKJSScriptIF::findFunction
	(	const QStringList	&,
		const QString		&
	)
{
	return	KJS::Value() ;
}


KBKJSScriptIF::KBKJSScriptIF ()
{
	scrIface	= this ;
	m_interpreter	= new KJS::Interpreter () ;

	KJS::ExecState *exec = m_interpreter->globalExec  	    () ;
	KJS::Object    glob  = m_interpreter->globalObject	    () ;
	KJS::Object    objp  = m_interpreter->builtinObjectPrototype() ;

	glob.put
	(
		exec,
		"RekallMain",
		KJS::Object(new RekallMainObjectImp (exec, static_cast<ObjectPrototypeImp *>(objp.imp()))),
		DontEnum
	)	;

	registerClasses () ;

	KBKJSDebugger::self()->attach(m_interpreter) ;
}

KBKJSScriptIF::~KBKJSScriptIF ()
{
	scrIface 	= 0	;
	delete	m_interpreter	;
}


bool	KBKJSScriptIF::load
	(	KBLocation		&,
		QString			&,
		QString			&,
		KBError			&,
		bool			&
	)
{
	return	false	;
}


void	KBKJSScriptIF::setApp 
	(	KBNotifier	*,
		KBCallback	*
	)
{
}


bool	KBKJSScriptIF::compile
	(	KBLocation		&,
		QString			&,
		QString			&,
		KBError			&
	)
{
	return	false	;
}

/*  KBKJSScriptIF							*/
/*  compileFunc	: Compile an inline function				*/
/*  script	: const QString & : Function text			*/
/*  fname	: const QString & : Function name			*/
/*		: QString & 	  :					*/
/*		: QString & 	  :					*/
/*		: QStringList &   :					*/
/*  pError	: KBError &	  : Error return			*/
/*  (returns)	: KBScriptCode *  : Inl;ine script object		*/

KBScriptCode
	*KBKJSScriptIF::compileFunc
	(	const QString		&script,
		const QString		&fname,
		QString			&,
		QString			&,
		const QStringList	&,
		KBError			&pError
	)
{
	if (!m_interpreter->checkSyntax (script))
	{
		pError	= KBError
			  (	KBError::Error,
				"Syntax error",
				script,
				__ERRLOCN
			  )	;
		return	0 ;
	}

	KBLocation	inlineLoc
			(	0,
				"script",
				KBLocation::m_pInline,
				fname,
				script
			)	;

	bool	ok ;
	KBKJSScriptCode *code	= new KBKJSScriptCode (m_interpreter, script, fname, inlineLoc, ok) ;

	if (!ok)
	{	delete	code	;
		return	0	;
	}

	return	code	;
}

KBScriptCode
	*KBKJSScriptIF::compileExpr
	(	const QString		&script,
		QString			&,
		QString			&,
		const QStringList	&,
		KBError			&pError
	)
{
	if (!m_interpreter->checkSyntax (script))
	{
		pError	= KBError
			  (	KBError::Error,
				"Syntax error",
				script,
				__ERRLOCN
			  )	;
		return	0 ;
	}

	KBLocation	inlineLoc
			(	0,
				"expr",
				KBLocation::m_pInline,
				EXPRFUNC,
				script
			)	;

	bool	ok ;
	KBKJSScriptCode *code	= new KBKJSScriptCode
				  (	m_interpreter,
					QString(EXPRWRAP).arg(script),
					EXPRFUNC,
					inlineLoc,
					ok
				  )	;

	if (!ok)
	{	delete	code	;
		return	0	;
	}

	return	code	;
}

bool	KBKJSScriptIF::load
	(	KBLocation		&,
		QString			&,
		QString			&,
		KBError			&
	)
{
	return	false	;
}

KBScript::ExeRC
	KBKJSScriptIF::execute
	(	const QStringList	&,
		const QString		&,
		KBNode			*,
		uint,
		const KBValue		*,
		KBValue 		&
	)
{
	return	KBScript::ExeError ;
}

KBLocation
	KBKJSScriptIF::exeError
	(	QString		&_errMsg,
		uint		&_errLno,
		QString 	&_errText
	)
{
	_errMsg	 = errMsg  ;
	_errLno	 = errLno  ;
	_errText = errText ;

	/* If the error source is empty then it must be a failure	*/
	/* to locate the error. In this case we have nothing to go on.	*/
	if (errSourceID == 0)
		return	KBLocation
			(	0,
				"script",
				KBLocation::m_pInline,
				"[unknown]",
				"kjs"
			) ;

	/* If the error was in inline code then return just enough to	*/
	/* correspond to this. The original caller should know where	*/
	/* it is.							*/
	if (scriptCodeMap.find (errSourceID) != 0)
		return	KBLocation
			(	0,
				"script",
				KBLocation::m_pInline,
				KBLocation::m_pInline,
				"kjs"
			) ;

	/* OK, got a real looking identifier. It should exist in the	*/
	/* module map, in which case return the location for the map	*/
	/* entry.							*/
//	KBPYModule *module ;
//	if ((module = loadMap.find (errIdent)) != 0)
//	{	fprintf	(stderr, "KBPYScriptIF::exeError -> [%s]\n",
//				 (cchar *)module->location.ident()) ;
//		return	module->location ;
//	}

	/* Should not get here!						*/
	fprintf
	(	stderr,
		"KBKJSScriptIF::exeError failed for [%d]\n",
		errSourceID
	)	;
	return	KBLocation () ;
}

bool	KBKJSScriptIF::unlink
	(	KBLocation	&,
		KBError		&
	)
{
	return	false	;
}

bool	KBKJSScriptIF::rename
	(	KBLocation	&,
		const QString	&,
		KBError		&
	)
{
	return	false	;
}

void	KBKJSScriptIF::editorInit
	(	TKTextEditor	*
	)
{
}

KBDebug	*KBKJSScriptIF::showDebug
	(	TKToggleAction *
	)
{
	return	0 ;
}

bool	KBKJSScriptIF::debugScript
	(	KBLocation  	&,
		KBError 	&
	)
{
	return	false	;
}


KBKJSScriptIF
	*KBKJSScriptIF::getIface ()
{
	return	scrIface	;
}

QString	KBKJSScriptIF::ident ()
{
	return	QString	("%1").arg(__KB_BUILD_IDENT) ;
}



KBFACTORYIMPL
(	KBKJSScriptFactory,
	script_kjs,
	"Rekall KJS scripting",
	"Plugin",
	"0",
	"7",
	"0"
)

/*  KBPYScriptFactory							*/
/*		: Get an instance of the appropriate interpreter	*/
/*  parent	: QObject *	: Parent object				*/
/*  (returns)	: QObject *	: Interpreter instance			*/

QObject	*KBKJSScriptFactory::create
	(	QObject		  *,
		cchar		  *,
		cchar		  *,
		const QStringList &
	)
{
	return	new KBKJSScriptIF ()	;
}

cchar	*KBKJSScriptFactory::ident ()
{
	return	__KB_BUILD_IDENT ;
}

