/*
 * Copyright (c) 1999 - 2001 Phil Thompson <phil@river-bank.demon.co.uk>
 *
 * SIP library code.
 */


#include <stdarg.h>
#include <string.h>
#include <Python.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "sip.h"
#include "sipint.h"


#define	PARSE_OK	0x00000000		/* Parse is Ok so far. */
#define	PARSE_MANY	0x10000000		/* Too many arguments. */
#define	PARSE_FEW	0x20000000		/* Too few arguments. */
#define	PARSE_TYPE	0x30000000		/* Argument with a bad type. */
#define	PARSE_MASK	0xf0000000


static int needInit = TRUE;			/* Set if we need to init. */
static PyObject *thisName = NULL;		/* "this" as a Python object. */
static sipObjectMap cppPyMap;			/* The C/C++ to Python map. */
static const void *cppPending = NULL;		/* A C/C++ object to be wrapped. */
static int cppPendingFlags = 0;			/* Flags for pending object. */
static getattrofunc savedClassGetAttr;		/* Saved class getattr(). */
static getattrofunc savedInstanceGetAttr;	/* Saved instance getattr(). */
static sipModuleDef *modulesList = NULL;	/* List of registered modules. */
static PyThreadState *savedThreadState = NULL;	/* Saved thread state. */
static PyObject *weakRefMeth = NULL;		/* The _weakref.ref method. */


static PyMethodDef *searchVarTable(PyMethodDef **,PyObject *);
static PyObject *sipDtor(PyObject *,PyObject *);
static PyObject *classGetAttr(PyObject *,PyObject *);
static PyObject *instanceGetAttr(PyObject *,PyObject *);
static PyObject *handleLazyAttr(PyObject *,PyObject *,PyClassObject *,PyObject *);
static PyMethodDef *findLazyAttr(PyClassObject *,char *);
static int checkPointer(const void *);
static void badArgs(int,char *,char *);
static void finalise(void);


/*
 * Register a module.
 */

int sipRegisterModule(sipModuleDef *sm)
{
	PyObject *dictofmods, *mod;

	if ((dictofmods = PyImport_GetModuleDict()) == NULL)
		return -1;

	/* Initialise the library if it hasn't already been done. */

	if (needInit)
	{
		static PyMethodDef dtorBuiltin[] = {
			{"__sipDtor__", sipDtor, METH_VARARGS, NULL},
			{NULL}
		};

		PyObject *dict;

		Py_AtExit(finalise);

		if ((thisName = PyString_FromString("sipThis")) == NULL)
			return -1;

		/* Get the __builtin__ module dictionary. */

		if ((mod = PyDict_GetItemString(dictofmods,"__builtin__")) == NULL ||
		    (dict = PyModule_GetDict(mod)) == NULL)
			return -1;

		/*
		 * Add the dtor as a builtin.  This makes sure it is always
		 * available to instance dtors even when the module has been
		 * deleted.
		 */

		if (sipAddFunctions(dict,dtorBuiltin) < 0)
			return -1;

		/*
		 * This truly dreadful hack is needed so that we can resolve
		 * Class.method references for lazy attributes.
		 */

		savedClassGetAttr = PyClass_Type.tp_getattro;
		PyClass_Type.tp_getattro = classGetAttr;

		/* We might as well extend the hack to instances. */

		savedInstanceGetAttr = PyInstance_Type.tp_getattro;
		PyInstance_Type.tp_getattro = instanceGetAttr;

		/* Initialise the object map. */

		sipOMInit(&cppPyMap);

#ifdef SIP_QT_SUPPORT
		sipQObjectClass = NULL;
#endif

		needInit = FALSE;
	}

	/* Get the module dictionary. */

	if ((mod = PyDict_GetItemString(dictofmods,sm -> md_name)) == NULL ||
	    (sm -> md_dict = PyModule_GetDict(mod)) == NULL)
	{
		PyErr_Format(PyExc_SystemError,"Unable to find module dictionary for %s",sm -> md_name);

		return -1;
	}

	/* Add to the list of modules. */

	sm -> md_next = modulesList;
	modulesList = sm;

	return 0;
}


/*
 * Called by the interpreter to do any final clearing up, just in case the
 * interpreter will re-start.
 */

static void finalise(void)
{
	/* Release all memory we've allocated directly. */

	sipOMFinalise(&cppPyMap);

	/* Re-initialise those globals that (might) need it. */

	cppPending = NULL;
	cppPendingFlags = 0;
	modulesList = NULL;
	savedThreadState = NULL;
	weakRefMeth = NULL;

	/* We need to re-initialise if we are called again. */

	needInit = TRUE;
}


/*
 * A Python 1.5 style memory allocator that supports Python 1.5 and 1.6.
 */

void *sipMalloc(size_t nbytes)
{
#if PY_VERSION_HEX >= 0x01060000
	void *mem;

	if ((mem = PyMem_Malloc(nbytes)) == NULL)
		PyErr_NoMemory();

	return mem;
#else
	return Py_Malloc(nbytes);
#endif
}


/*
 * A Python 1.5 style memory de-allocator that supports Python 1.5 and 1.6.
 */

void sipFree(void *mem)
{
#if PY_VERSION_HEX >= 0x01060000
	PyMem_Free(mem);
#else
	Py_Free(mem);
#endif
}


/*
 * Release the interpreter lock and save the current Python thread state.
 */

void sipReleaseLock()
{
	savedThreadState = PyEval_SaveThread();
}


/*
 * Acquire the interpreter lock and restore the Python thread state.
 */

void sipAcquireLock()
{
	PyEval_RestoreThread(savedThreadState);
	savedThreadState = NULL;
}


/*
 * Release the interpreter lock, if it was previously acquired, and save the
 * Python thread state.
 */

void sipCondReleaseLock(int relLock)
{
	if (relLock)
		sipReleaseLock();
}


/*
 * Acquire the interpreter lock, if it isn't already acquired, and restore the
 * Python thread state.
 */

int sipCondAcquireLock()
{
	if (savedThreadState != NULL)
	{
		sipAcquireLock();
		return 1;
	}

	return 0;
}


/*
 * Parse the arguments to a C/C++ function without any side effects.
 */

int sipParseArgs(int *argsParsedp,PyObject *sipArgs,char *fmt,...)
{
	int valid, decref, compulsory, nrargs, argnr, nrparsed;
	va_list va;

	if ((nrargs = PyTuple_Size(sipArgs)) < 0)
		return 0;

	va_start(va,fmt);

	/* See if we need to dispose of the arguments on a successful parse. */

	decref = TRUE;

	if (*fmt == '-')
	{
		decref = FALSE;
		++fmt;
	}

	/* Parse the arguments. */

	valid = PARSE_OK;
	nrparsed = 0;
	compulsory = TRUE;
	argnr = 0;

	/* Just in case. */

	PyErr_Clear();

	do
	{
		char ch;
		PyObject *arg;

		/* See if the following arguments are optional. */

		if ((ch = *fmt++) == '|')
		{
			compulsory = FALSE;
			ch = *fmt++;
		}

		/* See if we run out of arguments. */

		if (argnr == nrargs)
		{
			/* Invalid if we still expect compulsory arguments. */

			if (ch != '\0' && compulsory)
				valid = PARSE_FEW;

			break;
		}

		/* See if we don't expect anything else. */

		if (ch == '\0')
		{
			/* Invalid if there are still arguments. */

			if (argnr < nrargs)
				valid = PARSE_MANY;

			break;
		}

		/* Get the argument. */

		arg = PyTuple_GET_ITEM(sipArgs,argnr);
		++argnr;

		switch (ch)
		{
		case 's':
			{
				/* String or None. */

				char **p = va_arg(va,char **);

				if (arg == Py_None)
					*p = NULL;
				else if (PyString_Check(arg))
					*p = PyString_AS_STRING(arg);
				else
					valid = PARSE_TYPE;

				break;
			}

		case 'S':
			{
				/* Slot name, return the name. */

				if (PyString_Check(arg))
				{
					char *s = PyString_AS_STRING(arg);

					if (*s == '1' || *s == '2' || *s == '9')
						*va_arg(va,char **) = s;
					else
						valid = PARSE_TYPE;
				}
				else
					valid = PARSE_TYPE;

				break;
			}

		case 'G':
			{
				/* Signal name, return the name. */

				if (PyString_Check(arg))
				{
					char *s = PyString_AS_STRING(arg);

					if (*s == '2' || *s == '9')
						*va_arg(va,char **) = s;
					else
						valid = PARSE_TYPE;
				}
				else
					valid = PARSE_TYPE;

				break;
			}

		case 'I':
			{
				/* Class instance. */

				typedef int (*convfunc)(PyObject *);

				convfunc func = va_arg(va,convfunc);
				PyObject **ap = va_arg(va,PyObject **);

				if (arg == Py_None || (*func)(arg))
					*ap = arg;
				else
					valid = PARSE_TYPE;

				break;
			}

		case 'O':
			{
				/* Python object of any type. */

				*va_arg(va,PyObject **) = arg;
				break;
			}

		case 'T':
			{
				/* Python object of given type. */

				PyTypeObject *type = va_arg(va,PyTypeObject *);
				PyObject **p = va_arg(va,PyObject **);

				if (arg -> ob_type == type)
					*p = arg;
				else
					valid = PARSE_TYPE;

				break;
			}

		case 'R':
			{
#ifdef SIP_QT_SUPPORT
				/* Sub-class of QObject. */

				if (sipIsSubClassInstance(arg,sipQObjectClass))
					*va_arg(va,PyObject **) = arg;
				else
					valid = PARSE_TYPE;
#else
				valid = PARSE_TYPE;
#endif

				break;
			}

		case 'F':
			{
				/* Python callable object. */

				if (PyCallable_Check(arg))
					*va_arg(va,PyObject **) = arg;
				else
					valid = PARSE_TYPE;

				break;
			}

		case 'a':
			{
				/* Byte array or None. */

				char **p = va_arg(va,char **);
				int *szp = va_arg(va,int *);

				if (arg == Py_None)
				{
					*p = NULL;
					*szp = 0;
				}
				else if (PyString_Check(arg))
				{
					*p = PyString_AS_STRING(arg);
					*szp = PyString_GET_SIZE(arg);
				}
				else
					valid = PARSE_TYPE;

				break;
			}

		case 'c':
			{
				/* Character. */

				if (PyString_Check(arg) && PyString_GET_SIZE(arg) == 1)
					*va_arg(va,char *) = *PyString_AS_STRING(arg);
				else
					valid = PARSE_TYPE;

				break;
			}

		case 'i':
			{
				/* Integer. */

				long v = PyInt_AsLong(arg);

				if (PyErr_Occurred())
					valid = PARSE_TYPE;
				else
					*va_arg(va,int *) = (int)v;

				break;
			}

		case 'h':
			{
				/* Short integer. */

				long v = PyInt_AsLong(arg);

				if (PyErr_Occurred())
					valid = PARSE_TYPE;
				else
					*va_arg(va,short *) = (short)v;

				break;
			}

		case 'l':
			{
				/* Long integer. */

				long v = PyInt_AsLong(arg);

				if (PyErr_Occurred())
					valid = PARSE_TYPE;
				else
					*va_arg(va,long *) = v;

				break;
			}

		case 'f':
			{
				/* Float. */

				double v = PyFloat_AsDouble(arg);

				if (PyErr_Occurred())
					valid = PARSE_TYPE;
				else
					*va_arg(va,float *) = (float)v;

				break;
			}

		case 'd':
			{
				/* Double float. */

				double v = PyFloat_AsDouble(arg);

				if (PyErr_Occurred())
					valid = PARSE_TYPE;
				else
					*va_arg(va,double *) = v;

				break;
			}

		case 'v':
			{
				/* Void pointer. */

				void *v = sipConvertToVoidPtr(arg);

				if (PyErr_Occurred())
					valid = PARSE_TYPE;
				else
					*va_arg(va,void **) = v;

				break;
			}
		}

		if (valid == PARSE_OK)
			++nrparsed;
	}
	while (valid == PARSE_OK);

	va_end(va);

	if (valid == PARSE_OK)
	{
		if (decref)
			Py_DECREF(sipArgs);
	}
	else if ((*argsParsedp & PARSE_MASK) == PARSE_OK || (*argsParsedp & ~PARSE_MASK) < nrparsed)
		*argsParsedp = valid | nrparsed;

	/* Just in case. */

	PyErr_Clear();

	return (valid == PARSE_OK);
}


/*
 * Carry out actions common to all ctors.
 */

void sipCommonCtor(sipMethodCache *cache,int nrmeths)
{
	while (nrmeths-- > 0)
		cache++ -> mcflags = 0;
}


/*
 * Carry out actions common to all dtors.
 */

void sipCommonDtor(sipThisType *sipThis)
{
	if (sipThis != NULL)
	{
		int relLock = sipCondAcquireLock();

		sipOMRemoveObject(&cppPyMap,sipThis -> u.cppPtr,sipThis);

		/* This no longer points to anything useful. */

		sipThis -> u.cppPtr = NULL;

		/* Delete the Python object if C++ has responsibility. */

		if (!sipIsPyOwned(sipThis))
			Py_DECREF(sipThis -> sipSelf);

		sipCondReleaseLock(relLock);
	}
}


/*
 * A convenience function to convert a C/C++ void pointer from a Python object.
 */

void *sipConvertToVoidPtr(PyObject *obj)
{
	return (obj == Py_None) ? NULL : PyCObject_AsVoidPtr(obj);
}


/*
 * A convenience function to convert a C/C++ void pointer to a Python object.
 */

PyObject *sipConvertFromVoidPtr(void *val)
{
	if (val != NULL)
		return PyCObject_FromVoidPtr(val,NULL);

	Py_INCREF(Py_None);

	return Py_None;
}


/*
 * A convenience function to convert a C/C++ boolean to a Python object.
 */

PyObject *sipConvertFromBool(int val)
{
	PyObject *pyobj;

	pyobj = (val ? Py_True : Py_False);

	Py_INCREF(pyobj);

	return pyobj;
}


/*
 * Call the ctor for any of a module's classes.
 */

PyObject *sipCallCtor(sipModuleDef *sm,PyObject *args)
{
	int c;
	PyObject *pySelf, *ctorargs;

	if (!PyArg_ParseTuple(args,"iOO",&c,&pySelf,&ctorargs))
		return NULL;

	return (*sm -> md_classes[c].cd_thisctor)(pySelf,ctorargs);
}


/*
 * Handle the dtor for any class.  This ensures that the wrapped object is
 * deleted after all the other things in the instance.  This more closely
 * mimics what C++ does and can avoid some very, very subtle problems.
 */

static PyObject *sipDtor(PyObject *ignore,PyObject *args)
{
	PyInstanceObject *pySelf;
	PyObject *sipThis;

	if (!PyArg_ParseTuple(args,"O!",&PyInstance_Type,&pySelf))
		return NULL;

	if ((sipThis = PyDict_GetItem(pySelf -> in_dict,thisName)) != NULL)
	{
		/* Keep this alive. */

		Py_INCREF(sipThis);

		/* Kill everything else. */

		PyDict_Clear(pySelf -> in_dict);

		/*
		 * Finally try and kill this. However, it may not succeed if
		 * Python is still keeping a reference to it.  I think the only
		 * way it can do this is if it has a reference to one of the
		 * class's lazy methods which itself will have a reference to
		 * sipThis.  Therefore we NULL the self pointer so that we know
		 * that the Python instance has gone.
		 */

		((sipThisType *)sipThis) -> sipSelf = NULL;

		Py_DECREF(sipThis);
	}

	Py_INCREF(Py_None);
	return Py_None;
}


/*
 * Register a module's classes.
 */

int sipRegisterClasses(sipModuleDef *sm,int qobjclass)
{
	int c;
	sipClassDef *cd = sm -> md_classes;

	/* Get the class objects from the module dictionary. */

	for (c = 0; c < sm -> md_nrclasses; ++c)
	{
		if (cd -> cd_name != NULL && (*cd -> cd_classptr = PyDict_GetItemString(sm -> md_dict,cd -> cd_name)) == NULL)
		{
			PyErr_Format(PyExc_SystemError,"Unable to find class object for %s",cd -> cd_name);

			return -1;
		}

		++cd;
	}

	/* Check we can handle Qt if needed. */

	if (qobjclass >= 0)
	{
#ifdef SIP_QT_SUPPORT
#if PY_VERSION_HEX >= 0x02010000
		/*
		 * If this is at least Python 2.1 and we might be using Qt
		 * signals, then the weak references module will be useful (but
		 * not essential).
		 */

		PyObject *wrmod;

		if ((wrmod = PyImport_ImportModule("_weakref")) != NULL)
			weakRefMeth = PyObject_GetAttrString(wrmod,"ref");

		/* Just in case. */

		PyErr_Clear();
#endif

		if (sipQObjectClass != NULL)
		{
			PyErr_Format(PyExc_RuntimeError,"SIP - module \"%s\" implements QObject but it has already been implemented",sm -> md_name);
			return -1;
		}

		sipQObjectClass = *sm -> md_classes[qobjclass].cd_classptr;
#else
		PyErr_Format(PyExc_RuntimeError,"SIP - module \"%s\" requires Qt support from the SIP library",sm -> md_name);
		return -1;
#endif
	}

	return 0;
}


/*
 * Get "this" and modify the argument tuple for a method called as
 * Class.Method(self,...) rather than self.Method(...).
 */

sipThisType *sipGetThis(PyObject *thisObj,PyObject **argp,PyObject *cls)
{
	int nrargs;
	sipThisType *sipThis;
	PyObject *arg;

	arg = *argp;

	/* See if was the normal self.Method(...). */

	if (thisObj != NULL)
	{
		/* sipParseArgs() will decrement the reference count. */

		Py_INCREF(arg);
		return (sipThisType *)thisObj;
	}

	/* Get self from the arguments. */

	sipThis = NULL;

	if ((nrargs = PyTuple_Size(arg)) >= 1)
	{
		PyObject *sipSelf = PyTuple_GET_ITEM(arg,0);

		if (sipIsSubClassInstance(sipSelf,cls))
			sipThis = sipMapSelfToThis(sipSelf);
	}

	if (sipThis == NULL)
	{
		PyErr_SetString(PyExc_TypeError,"unbound method must be called with class instance 1st argument");

		return NULL;
	}

	/* Copy the arguments without self. */

	if ((*argp = PyTuple_GetSlice(arg,1,nrargs)) == NULL)
		return NULL;

	return sipThis;
}


/*
 * This is the replacement class getattro function.  We want to create a
 * class's methods on demand because creating all of them at the start slows
 * the startup time and consumes too much memory, especially for scripts that
 * use a small subset of a large class library.  We used to define a
 * __getattr__ function for each class, but this doesn't work when you want to
 * use the standard technique for calling a base class's method from the same
 * method in a derived class, ie. Base.method(self,...).
 */

static PyObject *classGetAttr(PyObject *op,PyObject *nameobj)
{
	PyObject *attr;

	if ((attr = (*savedClassGetAttr)(op,nameobj)) != NULL)
		return attr;

	return handleLazyAttr(op,nameobj,(PyClassObject *)op,NULL);
}


/*
 * This is the replacement instance getattro function.  This could be
 * handled using a Python __getattr__ method - but seeing has we need to do
 * it this way for classes, we might as well do it this way for instances.
 */

static PyObject *instanceGetAttr(PyObject *op,PyObject *nameobj)
{
	PyObject *attr;

	if ((attr = (*savedInstanceGetAttr)(op,nameobj)) != NULL)
		return attr;

	return handleLazyAttr(op,nameobj,((PyInstanceObject *)op) -> in_class,op);
}


/*
 * Handle the result of a call to the Python class/instance getattr functions.
 */

static PyObject *handleLazyAttr(PyObject *op,PyObject *nameobj,PyClassObject *clsobj,PyObject *instance)
{
	char *name;
	PyMethodDef *pmd;
	PyObject *attr, *sipThis, *dict;

	/* If it was an error, propagate it. */

	if (!PyErr_ExceptionMatches(PyExc_AttributeError))
		return NULL;

	PyErr_Clear();

	/* See if it was a lazy attribute. */

	if ((name = PyString_AsString(nameobj)) == NULL)
		return NULL;

	if ((pmd = findLazyAttr(clsobj,name)) == NULL)
	{
		PyErr_SetObject(PyExc_AttributeError,nameobj);
		return NULL;
	}

	/*
	 * We don't cache methods called using class.method(self,...).  If we
	 * did then, because we use the Python class and instance getattr
	 * functions before we see if an attribute is lazy, then in the case
	 * that we were looking for a method that is being called using
	 * self.method(...) for the first time - and after it has been called
	 * using class.method(self,...) - the Python instance getattr function
	 * would find the unbound method which would cause an exception.
	 * Because the most common calling method is cached, the performance
	 * penalty should be negligable.
	 */

	if (instance == NULL)
		return PyCFunction_New(pmd,NULL);

	/* Get "this". */

	dict = ((PyInstanceObject*)instance) -> in_dict;

	if ((sipThis = PyDict_GetItem(dict,thisName)) == NULL)
	{
		PyErr_SetObject(PyExc_AttributeError,thisName);
		return NULL;
	}

	/* Convert it to a function. */

	if ((attr = PyCFunction_New(pmd,sipThis)) == NULL)
		return NULL;

	/* Cache it in the instance dictionary. */

	if (PyDict_SetItem(dict,nameobj,attr) < 0)
	{
		Py_DECREF(attr);
		return NULL;
	}

	return attr;
}


/*
 * Find the method definition for a lazy class attribute.
 */

static PyMethodDef *findLazyAttr(PyClassObject *op,char *name)
{
	int b, nrbases;
	sipModuleDef *sm;
	PyMethodDef *pmd;

	/* See if this is a SIP class. */

	for (sm = modulesList; sm != NULL; sm = sm -> md_next)
	{
		int c;
		sipClassDef *cd = sm -> md_classes;

		for (c = 0; c < sm -> md_nrclasses; ++c)
		{
			if (cd -> cd_name != NULL && *cd -> cd_classptr == (PyObject *)op)
			{
				for (pmd = cd -> cd_attrtab; pmd -> ml_name != NULL; ++pmd)
					if (strcmp(pmd -> ml_name,name) == 0)
						return pmd;

				break;
			}

			++cd;
		}

		if (c < sm -> md_nrclasses)
			break;
	}

	/* Check the base classes. */

	nrbases = PyTuple_Size(op -> cl_bases);

	for (b = 0; b < nrbases; ++b)
		if ((pmd = findLazyAttr((PyClassObject *)PyTuple_GET_ITEM(op -> cl_bases,b),name)) != NULL)
			return pmd;

	return NULL;
}


/*
 * Report a constructor with invalid argument types.
 */

void sipNoCtor(int argsParsed,char *classname)
{
	badArgs(argsParsed,NULL,classname);
}


/*
 * Report a function with invalid argument types.
 */

void sipNoFunction(int argsParsed,char *func)
{
	badArgs(argsParsed,NULL,func);
}


/*
 * Report a method/function/signal with invalid argument types.
 */

void sipNoMethod(int argsParsed,char *classname,char *method)
{
	badArgs(argsParsed,classname,method);
}


/*
 * Handle error reporting for bad arguments to various things.
 */

static void badArgs(int argsParsed,char *classname,char *method)
{
	char *sep;
	int nrparsed = argsParsed & ~PARSE_MASK;

	if (classname != NULL)
		sep = ".";
	else
	{
		classname = "";
		sep = "";
	}

	switch (argsParsed & PARSE_MASK)
	{
	case PARSE_FEW:
		PyErr_Format(PyExc_TypeError,"Insufficient number of arguments to %s%s%s()",classname,sep,method);
		break;

	case PARSE_MANY:
		PyErr_Format(PyExc_TypeError,"Too many arguments to %s%s%s(), %d at most expected",classname,sep,method,nrparsed);
		break;

	case PARSE_TYPE:
		PyErr_Format(PyExc_TypeError,"Argument %d of %s%s%s() has an invalid type",nrparsed + 1,classname,sep,method);
		break;
	}
}


/*
 * Report a Python object that cannot be converted to a particular class.
 */

void sipBadClass(char *classname)
{
	PyErr_Format(PyExc_TypeError,"Cannot convert Python object to an instance of %s",classname);
}


/*
 * Check a None argument for a class pointer that we might dereference.
 */

void sipCheckNone(int willDeref,int *isErr,char *classname)
{
	if (willDeref)
	{
		PyErr_Format(PyExc_RuntimeError,"Cannot pass None as a %s argument in this call",classname);

		*isErr = 1;
	}
}


/*
 * Report a Python class variable with an unexpected type.
 */

void sipBadSetType(char *classname,char *var)
{
	PyErr_Format(PyExc_TypeError,"Invalid type for variable %s.%s",classname,var);
}


/*
 * Report a Python member function with an unexpected return type.
 */

void sipBadVirtualResultType(char *classname,char *method)
{
	PyErr_Format(PyExc_TypeError,"Invalid result type from %s.%s()",classname,method);
}


/*
 * Transfer ownersip of a class instance from Python to C/C++, or vice versa.
 */

void sipTransferSelf(PyObject *sipSelf,int toCpp)
{
	if (sipSelf != NULL && sipSelf != Py_None)
	{
		sipThisType *sipThis;

		/*
		 * There is a legitimate case where we try to transfer a
		 * PyObject that may not be a SIP generated class.  The virtual
		 * handler code calls this function to keep the C/C++ instance
		 * alive when it gets rid of the Python object returned by the
		 * Python method.  A class may have handwritten code that
		 * converts a regular Python type - so we can't assume that
		 * the lack of a "sipThis" is an error.
		 */

		if ((sipThis = sipMapSelfToThis(sipSelf)) != NULL)
			if (toCpp)
			{
				sipResetPyOwned(sipThis);
				Py_INCREF(sipSelf);
			}
			else
			{
				sipSetPyOwned(sipThis);
				Py_DECREF(sipSelf);
			}
	}
}


/*
 * Add the global void pointer instances to a dictionary.
 */

int sipAddVoidPtrInstances(PyObject *dict,sipVoidPtrInstanceDef *vi)
{
	while (vi -> vi_name != NULL)
	{
		int rc;
		PyObject *w;

		if ((w = sipConvertFromVoidPtr(vi -> vi_val)) == NULL)
			return -1;

		rc = PyDict_SetItemString(dict,vi -> vi_name,w);
		Py_DECREF(w);

		if (rc < 0)
			return -1;

		++vi;
	}

	return 0;
}


/*
 * Add the global char instances to a dictionary.
 */

int sipAddCharInstances(PyObject *dict,sipCharInstanceDef *ci)
{
	while (ci -> ci_name != NULL)
	{
		int rc;
		PyObject *w;

		if ((w = PyString_FromStringAndSize(&ci -> ci_val,1)) == NULL)
			return -1;

		rc = PyDict_SetItemString(dict,ci -> ci_name,w);
		Py_DECREF(w);

		if (rc < 0)
			return -1;

		++ci;
	}

	return 0;
}


/*
 * Add the global string instances to a dictionary.
 */

int sipAddStringInstances(PyObject *dict,sipStringInstanceDef *si)
{
	while (si -> si_name != NULL)
	{
		int rc;
		PyObject *w;

		if ((w = PyString_FromString(si -> si_val)) == NULL)
			return -1;

		rc = PyDict_SetItemString(dict,si -> si_name,w);
		Py_DECREF(w);

		if (rc < 0)
			return -1;

		++si;
	}

	return 0;
}


/*
 * Add the global long instances to a dictionary.
 */

int sipAddLongInstances(PyObject *dict,sipLongInstanceDef *li)
{
	while (li -> li_name != NULL)
	{
		int rc;
		PyObject *w;

		if ((w = PyInt_FromLong(li -> li_val)) == NULL)
			return -1;

		rc = PyDict_SetItemString(dict,li -> li_name,w);
		Py_DECREF(w);

		if (rc < 0)
			return -1;

		++li;
	}

	return 0;
}


/*
 * Add the global double instances to a dictionary.
 */

int sipAddDoubleInstances(PyObject *dict,sipDoubleInstanceDef *di)
{
	while (di -> di_name != NULL)
	{
		int rc;
		PyObject *w;

		if ((w = PyFloat_FromDouble(di -> di_val)) == NULL)
			return -1;

		rc = PyDict_SetItemString(dict,di -> di_name,w);
		Py_DECREF(w);

		if (rc < 0)
			return -1;

		++di;
	}

	return 0;
}


/*
 * Add enum instances to a dictionary.
 */

int sipAddEnumInstances(PyObject *dict,sipEnumValueInstanceDef *evi)
{
	while (evi -> evi_name != NULL)
	{
		int rc;
		PyObject *w;

		if ((w = PyInt_FromLong(evi -> evi_val)) == NULL)
			return -1;

		rc = PyDict_SetItemString(dict,evi -> evi_name,w);
		Py_DECREF(w);

		if (rc < 0)
			return -1;

		++evi;
	}

	return 0;
}


/*
 * Wrap the global class instances and add them to a dictionary.
 */

int sipAddClassInstances(PyObject *dict,sipClassInstanceDef *ci)
{
	while (ci -> ci_name != NULL)
	{
		int rc;
		PyObject *w;

		if ((w = sipNewCppToSelf(ci -> ci_ptr,ci -> ci_class,ci -> ci_flags)) == NULL)
			return -1;

		rc = PyDict_SetItemString(dict,ci -> ci_name,w);
		Py_DECREF(w);

		if (rc < 0)
			return -1;

		++ci;
	}

	return 0;
}


/*
 * Get the C/C++ pointer for a complex object.
 */

const void *sipGetComplexCppPtr(sipThisType *w)
{
	if (sipIsSimple(w))
	{
		PyErr_SetString(PyExc_RuntimeError,"No access to protected functions or signals for object not created from Python");

		return NULL;
	}

	if (checkPointer(w -> u.cppPtr) < 0)
		return NULL;

	return w -> u.cppPtr;
}


/*
 * Create a number of functions and add them to a dictionary.
 */

int sipAddFunctions(PyObject *dict,PyMethodDef *md)
{
	while (md -> ml_name != NULL)
	{
		PyObject *func;

		if ((func = PyCFunction_New(md,NULL)) == NULL ||
		    PyDict_SetItemString(dict,md -> ml_name,func) < 0)
			return -1;

		++md;
	}

	return 0;
}


/*
 * Search a variable hierachy table for a name.
 */

static PyMethodDef *searchVarTable(PyMethodDef **vhiertab,PyObject *nameobj)
{
	PyMethodDef *vt;

	while ((vt = *vhiertab++) != NULL)
		while (vt -> ml_name != NULL)
		{
			if (strcmp(vt -> ml_name,PyString_AS_STRING(nameobj)) == 0)
				return vt;

			++vt;
		}

	return NULL;
}


/*
 * Handle the getting of class variables.
 */

PyObject *sipGetVar(sipModuleDef *sm,PyObject *args)
{
	int c;
	PyObject *sipSelf, *nameobj;
	PyMethodDef *at;

	if (!PyArg_ParseTuple(args,"iOS",&c,&sipSelf,&nameobj))
		return NULL;

	if ((at = searchVarTable(sm -> md_classes[c].cd_vartab,nameobj)) == NULL)
	{
		PyErr_SetObject(PyExc_AttributeError,nameobj);
		return NULL;
	}

	return (*at -> ml_meth)((PyObject *)sipMapSelfToThis(sipSelf),NULL);
}


/*
 * Handle the setting of class variables.
 */

PyObject *sipSetVar(sipModuleDef *sm,PyObject *args)
{
	int c;
	PyObject *sipSelf, *nameobj, *valobj = NULL;
	PyMethodDef *at;

	if (!PyArg_ParseTuple(args,"iOS|O",&c,&sipSelf,&nameobj,&valobj))
		return NULL;

	if ((at = searchVarTable(sm -> md_classes[c].cd_vartab,nameobj)) == NULL)
	{
		if (sipSetInstanceAttr(sipSelf,nameobj,valobj) < 0)
			return NULL;

		Py_INCREF(Py_None);
		return Py_None;
	}

	if (valobj == NULL)
	{
		PyErr_Format(PyExc_AttributeError,"%s cannot be deleted",at -> ml_name);
		return NULL;
	}

	return (*at -> ml_meth)((PyObject *)sipMapSelfToThis(sipSelf),valobj);
}


/*
 * Get the class of a Python instance.
 */

PyObject *sipGetClass(PyObject *sipSelf)
{
	return (PyObject*)((PyInstanceObject *)sipSelf) -> in_class;
}


/*
 * Find the Python member function corresponding to a C/C++ virtual function,
 * if any.
 */

int sipIsPyMethod(sipMethodCache *pymc,sipThisType *sipThis,char *cname,char *mname,int *relLockp)
{
	int relLock;

	/*
	 * It's possible that the Python object has been deleted but the
	 * underlying (complex) C/C++ instance is still working and trying to
	 * handle virtual functions.  If this is the case then say there is no
	 * Python method.
	 */

	if (sipThis == NULL)
		return FALSE;

	/* See if we have already looked for the Python method. */

	if (!sipFoundMethod(pymc))
	{
		PyObject *methobj;

		relLock = sipCondAcquireLock();

		/*
		 * Using PyMethod_Check() rather than PyCallable_Check() has
		 * the added benefits of ensuring the (common) case of there
		 * being no Python method is handled as a direct call to C/C++
		 * (rather than converted to Python and then back to C/C++) and
		 * makes sure that abstract virtuals are trapped.
		 */

		if ((methobj = PyObject_GetAttrString(sipGetClass(sipThis -> sipSelf),mname)) != NULL)
		{
			if (PyMethod_Check(methobj))
			{
				sipSetIsMethod(pymc);
				sipSaveMethod(&pymc -> pyMethod,methobj);
			}

			Py_DECREF(methobj);
		}

		sipSetFoundMethod(pymc);

		/* Release the lock if we know we don't need it. */

		if (!sipIsMethod(pymc))
		{
			sipCondReleaseLock(relLock);
			relLock = 0;
		}
	}
	else if (sipIsMethod(pymc))
		relLock = sipCondAcquireLock();
	else
		relLock = 0;

	*relLockp = relLock;

	if (!sipIsMethod(pymc) && cname != NULL)
		PyErr_Format(PyExc_NameError,"%s.%s() is abstract and must be overridden",cname,mname);

	return sipIsMethod(pymc);
}


/*
 * Set (or delete) a Python class instance attribute.
 */

int sipSetInstanceAttr(PyObject *sipSelf,PyObject *name,PyObject *value)
{
	if (value != NULL)
		return PyDict_SetItem(((PyInstanceObject *)sipSelf) -> in_dict,name,value);

	return PyDict_DelItem(((PyInstanceObject *)sipSelf) -> in_dict,name);
}


/*
 * Create a new C/C++ object wrapper.
 */

sipThisType *sipCreateThis(PyObject *sipSelf,const void *cppPtr,PyTypeObject *pyType,int initflags,sipExtraType *extraType)
{
	sipThisType *sipThis;
	PyObject *obj;

	/* Check "this" doesn't already exist. */

	if ((obj = PyDict_GetItem(((PyInstanceObject *)sipSelf) -> in_dict,thisName)) != NULL)
	{
		Py_DECREF(obj);

		PyErr_Format(PyExc_TypeError,"Cannot sub-class from more than one wrapped class");

		return NULL;
	}

	if ((sipThis = PyObject_NEW(sipThisType,pyType)) == NULL)
		return NULL;

	/* Save it as an attribute of the instance. */

	if (sipSetInstanceAttr(sipSelf,thisName,(PyObject *)sipThis) < 0)
	{
		Py_DECREF(sipThis);
		return NULL;
	}

	Py_DECREF(sipThis);

	sipThis -> u.cppPtr = cppPtr;
	sipThis -> flags = initflags;
	sipThis -> sipSelf = sipSelf;
	sipThis -> pySigList = NULL;
	sipThis -> xType = extraType;

	sipOMAddObject(&cppPyMap,cppPtr,sipThis);

	/*
	 * If C++ owns the object, give the Python object an extra reference to
	 * keep it alive.
	 */

	if (!sipIsPyOwned(sipThis))
		Py_INCREF(sipSelf);

	return sipThis;
}


/*
 * Delete a C/C++ object wrapper.
 */

void sipDeleteThis(sipThisType *sipThis)
{
	/* Ignore any errors.  Don't indirect. */

	if (sipThis -> u.cppPtr != NULL)
		sipOMRemoveObject(&cppPyMap,sipThis -> u.cppPtr,sipThis);

	while (sipThis -> pySigList != NULL)
	{
		sipPySig *ps;
		sipPySigRx *psrx;

		/* Take this one out of the list. */

		ps = sipThis -> pySigList;
		sipThis -> pySigList = ps -> next;

		while ((psrx = ps -> rxlist) != NULL)
		{
			ps -> rxlist = psrx -> next;

			if (psrx -> rx.name != NULL)
				sipFree((void *)psrx -> rx.name);

			sipFree((void *)psrx);
		}

		sipFree((void *)ps -> name);
		sipFree((void *)ps);
	}

	PyMem_DEL(sipThis);
}


/*
 * Convert a Python class instance to a C/C++ object wrapper.
 */

sipThisType *sipMapSelfToThis(PyObject *sipSelf)
{
	PyObject *sipThis;

	/*
	 * Get the value of "sipThis" from the instance.  The only reason it
	 * might not be found is if the programmer has further sub-classed and
	 * forgot to call the super-class constructor.
	 */

	sipThis = PyObject_GetAttr(sipSelf,thisName);
	Py_XDECREF(sipThis);

	return (sipThisType *)sipThis;
}


/*
 * Get the address of any C/C++ object waiting to be wrapped.
 */

const void *sipGetPending(int *fp)
{
	if (cppPending != NULL)
		*fp = cppPendingFlags;

	return cppPending;
}


/*
 * Convert a new C/C++ pointer to a Python instance.
 */

PyObject *sipNewCppToSelf(const void *cppPtr,PyObject *pyClass,int initflags)
{
	PyObject *w;

	if (cppPtr == NULL)
	{
		PyErr_SetString(PyExc_RuntimeError,"Attempt to create a Python instance for a NULL pointer");

		return NULL;
	}

	cppPending = cppPtr;
	cppPendingFlags = initflags;

	w = PyInstance_New(pyClass,NULL,NULL);

	cppPending = NULL;

	return w;
}


/*
 * Convert a C/C++ pointer to a Python instance.  If the C/C++ pointer is
 * recognised and it is an instance of a sub-class of the expected class then
 * the previously wrapped instance is returned.  Otherwise a new Python
 * instance is created with the expected class.  The instance comes with a
 * reference.
 */

PyObject *sipMapCppToSelf(const void *cppPtr,PyObject *pyClass)
{
	const sipThisType *sipThis;
	PyObject *sipSelf;

	if (cppPtr == NULL)
	{
		Py_INCREF(Py_None);
		return Py_None;
	}

	/* See if we have already wrapped it. */

	/* Don't indirect. */

	if ((sipThis = sipGetThisWrapper(cppPtr,pyClass)) != NULL)
	{
		sipSelf = sipThis -> sipSelf;
		Py_INCREF(sipSelf);
	}
	else
		sipSelf = sipNewCppToSelf(cppPtr,pyClass,SIP_SIMPLE);

	return sipSelf;
}


/*
 * Convert a C/C++ pointer to the object that wraps it.
 */

sipThisType *sipGetThisWrapper(const void *cppPtr,PyObject *pyClass)
{
	return sipOMFindObject(&cppPyMap,cppPtr,pyClass);
}


/*
 * Get the C/C++ pointer from a wrapper and cast it to the required type.
 */

const void *sipGetCppPtr(sipThisType *w,PyObject *toClass)
{
	const void *ptr;

	if (sipIsAccessFunc(w))
		ptr = (*w -> u.afPtr)();
	else if (sipIsIndirect(w))
		ptr = *((const void **)w -> u.cppPtr);
	else
		ptr = w -> u.cppPtr;

	if (checkPointer(ptr) < 0)
		return NULL;

	return (*w -> xType -> castfunc)(ptr,toClass);
}


/*
 * Check that a pointer is non-NULL.
 */

static int checkPointer(const void *ptr)
{
	if (ptr == NULL)
	{
		PyErr_SetString(PyExc_RuntimeError,"Underlying C/C++ object has been deleted");
		return -1;
	}

	return 0;
}


/*
 * See if a Python object is an instance of a sub-class of a given base class.
 */

int sipIsSubClassInstance(PyObject *inst,PyObject *baseclass)
{
	return (PyInstance_Check(inst) && PyClass_IsSubclass(sipGetClass(inst),baseclass));
}


/*
 * Convert a Python instance of a class to a C/C++ object pointer, checking
 * that the instance's class is derived from a given base class.
 */

const void *sipConvertToCpp(PyObject *sipSelf,PyObject *baseclass,int *iserrp)
{
	sipThisType *sipThis;
	const void *ptr;

	if ((sipThis = sipMapSelfToThis(sipSelf)) == NULL || (ptr = sipGetCppPtr(sipThis,baseclass)) == NULL)
	{
		*iserrp = TRUE;
		return NULL;
	}

	return ptr;
}


/*
 * Save the components of a Python method.
 */

void sipSaveMethod(sipPyMethod *pm,PyObject *meth)
{
	pm -> mfunc = PyMethod_GET_FUNCTION(meth);
	pm -> mself = PyMethod_GET_SELF(meth);
	pm -> mclass = PyMethod_GET_CLASS(meth);
}


/*
 * Return a weak reference to the given object.
 */

PyObject *sipGetWeakRef(PyObject *obj)
{
	PyObject *args, *wr;

	if (obj == NULL || weakRefMeth == NULL)
		return NULL;

	if ((args = Py_BuildValue("(O)",obj)) == NULL)
		return NULL;

	if ((wr = PyEval_CallObject(weakRefMeth,args)) == NULL)
		PyErr_Clear();

	Py_DECREF(args);

	return wr;
}


/*
 * Call a Python method.
 */

PyObject *sipEvalMethod(const sipPyMethod *pm,PyObject *args)
{
	PyObject *meth, *res;

	meth = PyMethod_New(pm -> mfunc,pm -> mself,pm -> mclass);

	if (meth == NULL)
                return NULL;
 
	res = PyEval_CallObject(meth,args);
 
	Py_DECREF(meth); 

	return res;
}


/*
 * Call a hook.
 */

void sipCallHook(char *hookname)
{
	PyObject *dictofmods, *mod, *dict, *hook, *res;
 
	/* Get the dictionary of modules. */
 
	if ((dictofmods = PyImport_GetModuleDict()) == NULL)
		return;
 
	/* Get the __builtin__ module. */
 
	if ((mod = PyDict_GetItemString(dictofmods,"__builtin__")) == NULL)
		return;
 
	/* Get it's dictionary. */
 
	if ((dict = PyModule_GetDict(mod)) == NULL)
		return;
 
	/* Get the function hook. */
 
	if ((hook = PyDict_GetItemString(dict,hookname)) == NULL)
		return;
 
	/* Call the hook and discard any result. */
 
	res = PyObject_CallObject(hook,NULL);
 
	Py_XDECREF(res);
}
