#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
rename-flac takes the information from FLAC metadata to batch rename
the files according to a file-naming scheme.

Usage:
    rename-flac [--verbose] <scheme> <directory>
    rename-flac (-h | --help)
    rename-flac --version

Arguments:
    <scheme>         The filenaming scheme. Has to be between quotation marks
    <directory>      The path to the directory containing the album

Options:
    -h  --help       Shows the help screen
    --version        Outputs version information
    --verbose        Runs the program as verbose mode

    These are the options you can use to define the filenaming scheme:
      %a = Artist  |  %b = Album        |  %c = Composer  |  %d = Date
      %g = Genre   |  %n = Tracknumber  |  %t = Title
"""
import sys

try:
    from docopt import docopt  # Creating command-line interface
except ImportError:
    sys.stderr.write("""
        %s is not installed: this program won't run correctly.
        """ % ("docopt"))

import subprocess
import re
import os

__version__ = "2.1.0"

TAGS = dict(a='artist', b='album', c='composer', d='date',
            g='genre', n='tracknumber', t='title')


def scheme_processing(args):
    """Process the scheming arguments."""
    scheme = args["<scheme>"]
    if not re.search("%[tn]", scheme):  # To have a unique filename
        sys.exit("Error: %t or %n has to be present in <scheme>")
    scheme = re.sub('%%([%s])' % ''.join(TAGS.keys()),
                    lambda m: '%%(%s)s' % TAGS[m.group(1)],
                    scheme)
    return scheme


def fetch_metadata(filepath):
    """Fetch metadata and format it."""
    args = ["--show-tag=%s" % tag for tag in TAGS.values()]
    tags = ["%s=" % tag for tag in TAGS.values()]
    # make sure the keys exist. Prevents failure in case of incomplete metadata
    metadata = dict.fromkeys(list(TAGS.values()), '')
    pipe = subprocess.Popen(["metaflac"] + args + [filepath],
                            stdout=subprocess.PIPE)
    output, error = pipe.communicate()
    if pipe.returncode:
        raise IOError("metaflac failed: %s" % error)
    output = output.decode('utf-8') # binary to string
    output = output.splitlines()
    for tag in tags:
        for item in output:
            x = re.compile(re.escape(tag), re.IGNORECASE)
            if bool(re.match(x, item)):
                tag = tag.replace("=", "")
                if tag == "tracknumber":
                    metadata[tag] = x.sub("", item).zfill(2)
                else:
                    metadata[tag] = x.sub("", item)
    return metadata


def rename(scheme, dirname, filename, args):
    """Renames the files."""
    filepath = os.path.join(dirname, filename)
    new_filename = scheme % fetch_metadata(filepath) + ".flac"
    if new_filename == filename:
        if args["--verbose"]:
            print('"%s" is already named correctly' % filename)
    else:
        new_filepath = os.path.join(dirname, new_filename)
        os.rename(filepath, new_filepath)
        if args["--verbose"]:
            print('"%s" --> "%s"' % (filename, new_filename))


def main():
    """Main function."""
    args = docopt(__doc__, version="rename-flac %s" % __version__)
    scheme = scheme_processing(args)
    if not os.path.isdir(args["<directory>"]):
        sys.exit("Error: %s is not a valid directory" % args["<directory>"])
    counter = 0
    for dirname, _, filenames in os.walk(args["<directory>"], topdown=False):
        for filename in sorted(filenames):
            if filename.endswith('.flac'):
                counter += 1
                try:
                    rename(scheme, dirname, filename, args)
                except KeyboardInterrupt:
                    raise
                except OSError:
                    sys.exit("Error: %s title contains /. Please rename it"
                             % filename)
    if counter < 1:
        sys.exit("Error: No FLAC files found in %s" % args["<directory>"])


if __name__ == "__main__":
    main()
