genRepos <- function (repName, urlBase, urlPath="",
                      repType=c("package", "vignette"),
                      repRelLevel=c("release", "devel"), dir =".",
                      HTML=TRUE, functionDesc=FALSE) {
    require("tools") || stop("Need package tools")
    repType <- match.arg(repType)
    repRelLevel <- match.arg(repRelLevel)

    outFile <- file.path(dir, "replisting")
    out <- paste("repname: ", repName, "\nreptype: ", repType,
                 "\nrepaddrBase: ", urlBase, "\nrepaddrPath: ",
                 urlPath, "\nreprellevel: ", repRelLevel, "\n\n",
                 sep = "")
    cat(out, file = outFile)

    if (!file.exists(dir)) {
        stop(paste("Directory parameter",dir,"does not exist"))
    }

    switch(repType,
           "package" = genPkgRepos(dir, HTML, functionDesc=functionDesc),
           "vignette" = genVigRepos(dir),
           stop("Repository type ", repType, " not currently supported"))

    repTXML = file.path(dir, "repThemes.xml")
    repTrda = file.path(dir, "repThemes.rda")
    repDrda = file.path(dir, "repdatadesc.rda")
    parseThemesXML(repTXML, repTrda, repDrda)

    ## Create a CRAN-style PACKAGES file
    write_PACKAGES(dir)

    return(list(repName=repName, repType = repType, repaddrBase =
                urlBase, repaddrPath=urlPath,
                repRelLevel=repRelLevel, repDir=dir))
}

genPkgRepos <- function(dataDir, HTML=TRUE, functionDesc) {

    ## Create a temporary directory for unpacking packages
    tmpDir <- tempfile()
    on.exit(unlink(tmpDir,recursive=TRUE),add=TRUE)
    dir.create(tmpDir)

    tmpInfo <- file.path(tmpDir,"info")
    dir.create(tmpInfo)

    curDir <- getwd()
    on.exit(setwd(curDir),add=TRUE)

    ## check to make sure this works
    setwd(dataDir)

    ## Get list of packages to unpack
    pkgs <- dir(pattern=".*\\.tar\\.gz|.*\\.tgz|.*\\.zip")
    PACKin <- SrcStat <- WinStat <- NULL
    for (pkg in pkgs) {
        ext <- getExt(pkg)
        DESC <- try(getPackageDescriptionAsMatrix(pkg))
        if (inherits(DESC, "try-error")) {
            badPkg <- paste(pkg, "BAD", sep=".")
            warning("Skipping ivalid package ", sQuote(pkg),
                    " and renaming to ", sQuote(badPkg))
            file.rename(pkg, badPkg)
            next
        }
        if (!is.null(DESC)) {
            samePack <- which(PACKin[,"Package"]==DESC[,"Package"])
            if ((length(samePack) == 0) ||
                (all(PACKin[samePack,"Version",drop=FALSE]!=DESC[,"Version"]))) {
                PACKin <- rbind(PACKin, DESC)
            }
            if (!is.na(DESC[,"Built"])) {
                parts <- strsplit(DESC[,"Built"],"; ")
                pkgRvers <- strsplit(parts[[1]][1]," ")[[1]][2]
                pkgDate <- parts[[1]][3]
            }
            else {
                pkgRvers <- ""
                pkgDate <- date()
            }
            newStatus <- c(DESC[,"Package"],
                           DESC[,"Version"],
                           pkg,
                           "OK", pkgRvers, pkgDate)
            switch(ext,
                   "gz"=SrcStat<-rbind(SrcStat,newStatus),
                   "tgz"=SrcStat<-rbind(SrcStat,newStatus),
                   "zip"=WinStat<-rbind(WinStat,newStatus))
        }
    }
    ##rbind seems to add unwanted rownames
    if( !is.null(SrcStat) ) row.names(SrcStat) <- NULL
    if( !is.null(WinStat) ) row.names(WinStat) <- NULL
    fields <- c("Package", "Version", "File", "Status", "Rvers","Date")

    write.dcf(PACKin, file.path(tmpInfo, "PACKAGES.in"))

    if (!is.null(SrcStat)) {
        colnames(SrcStat) <- fields
        write.dcf(SrcStat, file.path(tmpInfo,"Source.status"))
    }
    if (!is.null(WinStat)) {
        colnames(WinStat) <- fields
        write.dcf(WinStat, file.path(tmpInfo,"Win32.status"))
    }

    df <- buildPkgDf(infoDir=tmpInfo)
    saveDfRda(df)
    if (HTML)
        genReposHTML(functionDesc=functionDesc)
}

genVigRepos <- function(dataDir) {
    ## Create a temporary directory for holding temp data
    tmpDir <- tempfile()
    on.exit(unlink(tmpDir,recursive=TRUE),add=TRUE)
    dir.create(tmpDir)
    tmpInfo <- file.path(tmpDir,"info")
    dir.create(tmpInfo)
    curDir <- getwd()
    on.exit(setwd(curDir),add=TRUE)

    setwd(dataDir)
    ## Get list of vignettes to use
    vigList <- getPkgVigList(".",baseVigDesc,".",pkgVers=FALSE)
    vigList<- mergeVigListStrings(vigList)
    vigList <- filterVigList(vigList)
    if (length(vigList) == 0) {
        warning("Didn't find any vignettes to work with")
        return(FALSE)
    }
    ## Push this into a data frame
    tmpVigDf <- data.frame(I(vigList[[1]]))
    if (length(vigList) > 1) {
        for (i in 2:length(vigList))
          tmpVigDf <- cbind(tmpVigDf,I(vigList[[i]]))
    }
    tmpVigDf <- t(tmpVigDf)
    rownames(tmpVigDf) <- 1:nrow(tmpVigDf)
    tmpVigDf <- as.data.frame(tmpVigDf)
    ## !! Temporarily just remove those w/o a PDFpath.
    ## !!! Need to try to build their PDFs
    tmpVigDf <- tmpVigDf[!is.na(tmpVigDf$PDFpath),]
    ## Write to DCF file
    write.dcf(tmpVigDf,file=file.path(tmpInfo,"Vignettes.in"))

    vigDf <- buildVigDf(infoDir=tmpInfo)
    setwd(curDir)
    saveDfRda(vigDf)
}

getPackageDescriptionAsMatrix <- function(pkg) {
    con <- getDesciptionCon(pkg)
    fields <- c("Package","Version", "Keywords", "Depends", "Title",
                "Suggests", "Imports", "Replaces", "Description", "URL",
                "Author", "Maintainer", "License", "Status",
                "Priority", "Built","Bundle","BundleDescription",
                "Contains", "SystemRequirements", "ReleaseLevel")
    descMat <- read.dcf(con, fields)
    close(con)
    if (!is.na(descMat[1,"Bundle"])) {
        ## then we have a package bundle treat it like a normal
        ## package by faking it.
        descMat[1,"Package"] <- descMat[1,"Bundle"]
        descMat[1,"Description"] <- descMat[1,"BundleDescription"]
    }
    ## Subset the result of unpackExtract as R-devel now attaches
    ## a line to the end of DESCRIPTION, but if there were
    ## blank lines at the end of the normal block, this will
    ## cause there to be multiple rows in the matrix
    descMat <- descMat[1, , drop=FALSE]
    descMat
}

parsePackageName <- function(pkg) {
    pkgName <- strsplit(pkg, "_")[[1]]
    if (length(pkgName) > 2)
      stop(pkg, " implies an invalid package name.")
    pkgName[1]
}

getExt <- function(path) {
    ## Helper to get the file extension
    parts <- strsplit(path, "\\.")[[1]]
    last <- parts[length(parts)]
    last
}

getDesciptionCon <- function(packageArchive) {
    ## Return a connection to the DESCRIPTION file in
    ## the packageArchive.
    ##
    ## packageArchive can be .tar.gz or .zip and can
    ## represent either a regular package or a bundle.
    pkgName <- parsePackageName(basename(packageArchive))
    ext <- getExt(packageArchive)
    if (ext == "gz")
      getCon <- untgz
    else if (ext == "zip")
      getCon <- unz
    else
      stop("Unknown archive extension: ", ext)
    descPath <-  file.path(pkgName, "DESCRIPTION")
    con <- try(getCon(packageArchive, descPath, open="r"), silent=TRUE)
    if (inherits(con, "try-error")) {
        ## Maybe it is a zip bundle
        con <- try(getCon(packageArchive, "DESCRIPTION", open="r"), silent=TRUE)
        if (inherits(con, "try-error"))
          stop("Unable to find DESCRIPTION in ", sQuote(packageArchive))
    }
    ##print(paste("Found DESCRIPTION for", sQuote(packageArchive)))
    con
}

untgz <- function(description, filename, open="", checkExists=TRUE) {
    ## Return a connection to the contents of filename within the
    ## .tar.gz archive given in description.
    ##
    ## If checkExists, stop if filename isn't in the archive.  This
    ## requires reading a list of the entire archive contents.
    ##
    ## The idea is to provide a .tar.gz equivalent to the unz()
    ## built-in.
    if (checkExists) {
        fileListCon <- pipe(paste("tar", "-ztf", description))
        fileList <- readLines(fileListCon)
        close(fileListCon)
        if (! filename %in% fileList)
          stop(sQuote(filename), " not found in archive ", sQuote(description))
    }
    tarCmd <- paste("tar", "-zxOf")
    con <- pipe(paste(tarCmd, description, filename), open=open)
    con
}

unpackSourcePkg <- function(pkg) {
    ## Decompress and untar in two steps to avoid use of pipes
    ## that may be related to the tar hang bug
    pkgtar <- strsplit(pkg, ".gz")[[1]]
    ret <- system(paste("gunzip", pkg, ";sleep 0.1"))
    if (ret != 0)
      stop("gunzip of ", pkg, " returned non-zero exit status")
    ret <- system(paste("tar", "-xf", pkgtar))
    if (ret !=0)
      stop("tar -xf ", pkgtar, " returned non-zero exit status in", getwd())
    ret
}

unpackZipPkg <- function(pkg) {
    OST <- .Platform$OS.type
    if (OST == "unix") {
        zip <- getOption("unzip")
        system(paste(zip,"-q",pkg))
    }
    else {
        zip.unpack(pkg,".")
    }

    return(0)
}


getArgRepos <- function(rep) {
    ## Determine if rep is a directory or an online repository
    ## used for functions where one can pass in a rep or a local
    ## directory that is a rep (but might not be online)
    if (is(rep, "character")) {
        if ((!file.exists(file.path(rep,"replisting")))||
            (!file.exists(file.path(rep,"repdatadesc.rda")))) {
            stop(paste(rep,
                       "does not seem to be a valid repository directory"))
        }
        else {
            load(file.path(rep,"repdatadesc.rda"))
            repL <- read.dcf(file.path(rep,"replisting"))
            rep <- buildReposEntry(new("replisting",replisting=repL),
                                   new("repdatadesc",
                                       repdatadesc=reposDF))
        }
    }
    else if (!is(rep,"ReposEntry"))
        stop(paste("Don't know how to handle passed rep argument:",rep))

    return(rep)
}

genReposHTML <- function (rep=".", filename="index.html", outDir="html",
                          headerInfo="", functionDesc=FALSE)
{
    rep <- getArgRepos(rep)
    if (!file.exists(outDir))
        dir.create(outDir)
    GenPkgListingHTML(file.path(outDir,filename), rep,
                      headerInfo, functionDesc)
}

##idea here is to write a function that creates the package listings
GenPkgListingHTML <- function(filename, rep, headerInfo="",
                              upFile, functionDesc)
{
  if( missing(filename) )
    filename <- "packagelistingindex.html"

  outFile <- file(filename, "w")
  on.exit(close(outFile))
  rep <- getArgRepos(rep)
  ## Write header
  t1 <- paste("<TITLE>",repName(rep),"</TITLE>")
  cat("<html>", "<head>", t1, headerInfo,
        "</head> <body>", file = outFile, sep = "\n")
  cat("<h1>", repName(rep), "</h1>\n", sep="", file=outFile, append=TRUE)
  cat(file=outFile, "<table border=0 align=left>",
         "<tr> <td><b>Package</b></td> <td><b>Title</b></td>",
           "<td><b>Version</b></td> </tr>", sep="\n", append=TRUE)
  reposDF <- repdataframe(rep)
  pfilenames <- vector("character", length=nrow(reposDF))
  pN <- rownames(reposDF)
  pNames <- reposDF$Package

  for(i in 1:nrow(reposDF) ) {
    cat(file=outFile, "<tr>")
    pfilenames[i] <- genPkgListing(pN[i], reposDF = reposDF,
                                   upFile=basename(filename), rep=rep,
                                   outDir=dirname(filename), functionDesc)
    cat(file=outFile, "<td><a href=\"" , basename(pfilenames[i]), "\"> ", pNames[i], "</a></td>\n",
      sep="", append=TRUE)
    ## we need a short description
    cat(file=outFile,"<td>", as.character(reposDF[i,"Title"][[1]]),
         "</td>\n", sep="", append=TRUE)
    cat(file=outFile, "<td>", as.character(reposDF[i,"Version"][[1]]),
         "</td>\n", sep="", append=TRUE)
     cat(file=outFile, "</tr>", append=TRUE)
  }
  if( !missing(upFile) )
    cat(file=outFile, "<h3 align=left> <a href=", upFile,
      ">Return to Index</a></h3><br> \n", sep="", append=TRUE)

  cat(file=outFile, "</table>", "</body>",  "</html>", sep="\n", append=TRUE )
}

genFuncDescHTML <- function(srcPath, pkgN, linkDir) {
    ## unpack srcPath to a temp location, run Rdindex()
    ## on it, then HTMLize.  Store in ./descrips/
    ## adds a *lot* of processing time to web generation
    ## but we don't really care about that (would be better
    ## to extract earlier on but at this point wer'e onily
    ## dealing with repositories, and this shouldn't be in
    ## repos
    require("tools") || stop("Need package tools")
    cDir <- getwd()
    
    descDir <- file.path(linkDir, "descrips")
    if (!file.exists(descDir))
        dir.create(descDir)

    fileName <- basename(srcPath)

    result <- NULL
    if (file.exists(fileName)) {
        tmpDir <- tempfile()
        dir.create(tmpDir)

        outFile <- paste(pkgN,"Desc.html",sep="")
        outFile <- file.path(cDir,descDir,outFile)

        file.copy(srcPath, file.path(tmpDir,fileName))
        setwd(tmpDir)

        cat("unpacking ", fileName, " for Rd indexing...")
        z <- system(paste("tar -xzf", fileName))
        cat("    done\n")
        if (z == 0) {
            setwd(dirname(srcPath))
            Rdfile <- tempfile()
            Rdindex(strsplit(fileName,"_")[[1]][1], outFile=Rdfile)
            cat(Rdindex2html(Rdfile), file=outFile)
            result <- paste("<a href=\"descrips/", basename(outFile),
                            "\">Function Descriptions</a><br>\n",sep="")
        } else {
            warning("Could not open ",fileName," properly")
        }
        setwd(cDir)
        unlink(tmpDir, recursive=TRUE)
    }
    return(result)
}

Rdindex2html <- function(indexFile, headerInfo="") {
    Rdcontents <- read.00Index(indexFile)
    colnames(Rdcontents) <- c("Name","Title")

    out <- paste("<html>","<title>","Package Description",
                 "</title>",headerInfo, "<body>", sep="\n")
    out <- paste(out,"<table border=0 align=left>\n",
        "<tr>\n<td><b>Function</b></td>\n<td><b>Description",
        "</b></td>\n</tr>\n",sep="")

    descs <-  paste("<tr>\n<td>",Rdcontents[,"Name"],"</td>\n<td>",
                    Rdcontents[,"Title"],"</td>\n</tr>",
                    sep="",collapse="\n")
    out <- paste(out,descs,"</table>\n</body>\n",sep="")
}

genPkgListing <- function(pkgN, reposDF, filename, upFile, rep,
                          outDir=".", functionDesc=FALSE)
{
    nVers <- strsplit(pkgN, ":v:")[[1]]
    repRow <- which(row.names(reposDF)==pkgN)
    if( missing(filename) )
       filename <- paste(paste(nVers[1], nVers[2], sep="v"), ".html", sep="")
    ##for now we overwrite
    filename <- file.path(outDir, filename)
    outfile <- file(filename, "w")
    on.exit(close(outfile))
    cat(file=outfile, "<html>",  "<title>", nVers[1], "</title>", "<body>",
      sep="\n")
    if( !missing(upFile) )
       cat(file=outfile, "<h3 align=left> <a href=", upFile,
         ">Return to package listing</a></h3><br> \n", sep="", append=TRUE)
    cat(file=outfile, " <h3 align=left> Package: ", nVers[1],
          "</h3> <div align=center> </div> \n <br> \n", sep="", append=TRUE)
    if (reposDF[pkgN, "Description"] == "NA")
        desc <- "None available"
    else
        desc <- reposDF[pkgN, "Description"]
    cat(file=outfile, "<b>Description:</b> ", desc, "<br>\n", sep="",
        append=TRUE)
    cat(file=outfile, "<b>Version:</b> ", nVers[2], "<br>\n", sep="",
        append=TRUE)
    if (reposDF[pkgN, "Author"] == "NA")
        author <- "None available"
    else
        author <- reposDF[pkgN, "Author"]

    cat(file=outfile,"<b>Author:</b> ", author, "<br>\n",sep="", append=TRUE)

    if (reposDF[pkgN, "Maintainer"] == "NA")
        maint <- "None available"
    else
        maint <- reposDF[pkgN, "Maintainer"]

    cat(file=outfile,"<b>Maintainer:</b> ", maint, "<br>\n",sep="", append=TRUE)

    if ((length(reposDF[pkgN, "Depends"][[1]]) == 0) ||
        ((length(reposDF[pkgN,"Depends"][[1]]) < 2)&&
        (reposDF[pkgN, "Depends"][[1]] == "NA")))
        deps <- "None"
    else
        deps <- reposDF[pkgN, "Depends"][[1]]
    cat(file=outfile, "<b>Dependencies:</b> ", paste(deps,collapse=", "),
         "<br>\n",sep="", append=TRUE)

    if ((length(reposDF[pkgN, "Suggests"][[1]]) == 0)||
         (length(reposDF[pkgN, "Suggests"][[1]]) < 2)&&
        (reposDF[pkgN, "Suggests"][[1]] == "NA"))
        sugs <- "None"
    else
        sugs <- reposDF[pkgN, "Suggests"][[1]]
    cat(file=outfile, "<b>Suggests:</b> ", paste(sugs, collapse=", "),
        "<br>\n", sep="", append=TRUE)

    if (reposDF[pkgN, "SystemRequirements"] == "NA")
        sr <- "None"
    else
        sr <- reposDF[pkgN, "SystemRequirements"]
    cat(file=outfile, "<b>SystemRequirements:</b> ",
        sr, "<br>\n",sep="", append=TRUE)

    if (reposDF[pkgN, "License"] == "NA")
        license <- "None available"
    else
        license <- reposDF[pkgN, "License"]

    cat(file=outfile,"<b>License:</b> ", license, "<br>\n",sep="", append=TRUE)

    if (reposDF[pkgN, "URL"] == "NA")
        url <- "None available"
    else
        url <- reposDF[pkgN, "URL"]

    cat(file=outfile,"<b>URL:</b> ", url, "<br><br><br>\n",sep="", append=TRUE)

    ## Set download links
    if (length(repRow) > 0) {
        osS <- reposDF$OSspecific[[repRow]]
        if (length(osS) > 0)
            for (i in 1:length(osS)) {
                curOS <- osS[[i]]
                if (curOS$Status == "OK") {
                    if (names(osS)[i] == "Source") {
                        if (functionDesc) {
                            fLink <- genFuncDescHTML(curOS$File, nVers[1],
                                                     dirname(filename))
                            if (!is.null(fLink))
                              cat(file=outfile, fLink, append=TRUE)
                        }
                    }

                    cat(file=outfile,"<a href=\"",
                        file.path(repURL(rep), curOS$File),
                        "\">",names(osS)[i],
                        " package download</a><br>", sep="", append=TRUE)
                }
            }

    }
    cat(file=outfile, "</body> \n </html>", append=TRUE)
    return(filename)
}

filterVigList <- function(vigList) {
    newVigList <- list()
    fields <- c("VignetteIndexEntry","VignettePackage", "VignetteTitle",
                   "VignetteVersion", "VignetteDepends",
                   "VignetteKeywords", "PDFpath")
    for (i in 1:length(vigList)) {
        tmp <- list()
        for (j in 1:length(fields)) {
            tmp[[j]] <- as.character(vigList[[i]][fields[j]])
        }
        names(tmp) <- fields
        newVigList[[i]] <- tmp
    }
    names(newVigList) <- names(vigList)
    return(newVigList)
}

mergeVigListStrings <- function(vList) {
    for (i in 1:length(vList)) {
        if ((!is.null(vList[[i]]$VignetteDepends))&&
            (!is.na(vList[[i]]$VignetteDepends))) {
            if (length(vList[[i]]$VignetteDepends > 1)) {
                vList[[i]]$VignetteDepends <-
                    paste(vList[[i]]$VignetteDepends, collapse=", ")
            }
        }
        else {
            vList[[i]]$VignetteDepends <- NA
        }
        if ((!is.null(vList[[i]]$VignetteKeywords))&&
            (!is.na(vList[[i]]$VignetteKeywords))) {
            if (length(vList[[i]]$VignetteKeywords > 1)) {
                vList[[i]]$VignetteKeywords <-
                    paste(vList[[i]]$VignetteKeywords, collapse=", ")
            }
        }
        else {
            vList[[i]]$VignetteKeywords <- NA
        }
    }
    return(vList)
}

