/*  This file is part of "reprepro"
 *  Copyright (C) 2003,2004,2005 Bernhard R. Link
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02111-1301  USA
 */
#include <config.h>

#include <errno.h>
#include <assert.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <getopt.h>
#include <string.h>
#include <strings.h>
#include <malloc.h>
#include <fcntl.h>
#include "error.h"
#define DEFINE_IGNORE_VARIABLES
#include "ignore.h"
#include "mprintf.h"
#include "strlist.h"
#include "dirs.h"
#include "names.h"
#include "md5sum.h"
#include "chunks.h"
#include "files.h"
#include "target.h"
#include "packages.h"
#include "reference.h"
#include "binaries.h"
#include "sources.h"
#include "release.h"
#include "aptmethod.h"
#include "updates.h"
#include "upgradelist.h"
#include "signature.h"
#include "extractcontrol.h"
#include "checkindeb.h"
#include "checkindsc.h"
#include "checkin.h"
#include "downloadcache.h"
#include "terms.h"
#include "tracking.h"
#include "optionsfile.h"


#ifndef STD_BASE_DIR
#define STD_BASE_DIR "."
#endif
#ifndef STD_METHOD_DIR
#define STD_METHOD_DIR "/usr/lib/apt/methods"
#endif

/* global options */
static char /*@only@*/ /*@notnull@*/ // *g*
	*mirrordir = NULL ,
	*distdir = NULL,
	*dbdir = NULL,
	*listdir = NULL,
	*confdir = NULL,
	*overridedir = NULL,
	*methoddir = NULL;
static char /*@only@*/ /*@null@*/ 
	*section = NULL,
	*priority = NULL,
	*component = NULL,
	*architecture = NULL,
	*packagetype = NULL;
static int	delete = D_COPY;
static int	force = 0;
static bool_t	nothingiserror = FALSE;
static bool_t	nolistsdownload = FALSE;
static bool_t	keepunreferenced = FALSE;
static bool_t	keepunneededlists = FALSE;
static bool_t	keepdirectories = FALSE;
static bool_t	onlyacceptsigned = FALSE;
static bool_t	askforpassphrase = FALSE;
static bool_t	skipold = TRUE;
int		verbose = 0;

/* define for each config value an owner, and only higher owners are allowed
 * to change something owned by lower owners. */
enum config_option_owner config_state,
#define O(x) owner_ ## x = CONFIG_OWNER_DEFAULT
O(mirrordir), O(distdir), O(dbdir), O(listdir), O(confdir), O(overridedir), O(methoddir), O(section), O(priority), O(component), O(architecture), O(packagetype), O(nothingiserror), O(nolistsdownload), O(keepunreferenced), O(keepunneededlists), O(keepdirectories), O(onlyacceptsigned),O(askforpassphrase), O(skipold);
#undef O
	
#define CONFIGSET(variable,value) if(owner_ ## variable <= config_state) { \
					owner_ ## variable = config_state; \
					variable = value; }
#define CONFIGDUP(variable,value) if(owner_ ## variable <= config_state) { \
					owner_ ## variable = config_state; \
					free(variable); \
					variable = strdup(value); \
					if( variable == NULL ) { \
						fputs("Out of Memory!",stderr); \
						exit(EXIT_FAILURE); \
					} }

static inline retvalue removeunreferencedfiles(references refs,filesdb files,struct strlist *dereferencedfilekeys) {
	int i;
	retvalue result,r;
	result = RET_OK;

	for( i = 0 ; i < dereferencedfilekeys->count ; i++ ) {
		const char *filekey = dereferencedfilekeys->values[i];

		r = references_isused(refs,filekey);
		if( r == RET_NOTHING ) {
			r = files_deleteandremove(files,filekey,!keepdirectories);
		}
		RET_UPDATE(result,r);
	}
	return result;
}

#define ACTION_N(name) static retvalue action_n_ ## name ( \
			UNUSED(references dummy1), 	\
			UNUSED(filesdb dummy2), 	\
			UNUSED(struct strlist* dummy3), \
			int argc,const char *argv[])

#define ACTION_U_R(name) static retvalue action_r_ ## name ( \
			references references, 		\
			UNUSED(filesdb dummy2), 	\
			UNUSED(struct strlist* dummy3), \
			int argc,UNUSED(const char *dummy4[]))

#define ACTION_R(name) static retvalue action_r_ ## name ( \
			references references, 		\
			UNUSED(filesdb dummy2), 	\
			UNUSED(struct strlist* dummy3), \
			int argc,const char *argv[])

#define ACTION_U_F(name) static retvalue action_f_ ## name ( \
			UNUSED(references dummy1), 	\
			filesdb filesdb,		\
			UNUSED(struct strlist* dummy3), \
			int argc,UNUSED(const char *dummy4[]))

#define ACTION_F(name) static retvalue action_f_ ## name ( \
			UNUSED(references dummy1), 	\
			filesdb filesdb,		\
			UNUSED(struct strlist* dummy3), \
			int argc,const char *argv[])

#define ACTION_RF(name) static retvalue action_rf_ ## name ( \
			references references, 		\
			filesdb filesdb,		\
			UNUSED(struct strlist* dummy3), \
			int argc,const char *argv[])

#define ACTION_U_RF(name) static retvalue action_rf_ ## name ( \
			references references, 		\
			filesdb filesdb,		\
			UNUSED(struct strlist* dummy3), \
			int argc,UNUSED(const char *dumym4[]))

#define ACTION_D(name) static retvalue action_d_ ## name ( \
			references references, 		\
			filesdb filesdb,		\
			struct strlist* dereferenced, 	\
			int argc,const char *argv[])

#define ACTION_D_U(name) static retvalue action_d_ ## name ( \
			references references, 		\
			UNUSED(filesdb dummy2),		\
			struct strlist* dereferenced, 	\
			int argc,const char *argv[])

ACTION_N(printargs) {
	int i;

	fprintf(stderr,"argc: %d\n",argc);
	for( i=0 ; i < argc ; i++ ) {
		fprintf(stderr,"%s\n",argv[i]);
	}
	return 0;
}
ACTION_N(extractcontrol) {
	retvalue result;
	char *control;

	if( argc != 2 ) {
		fprintf(stderr,"reprepro __extractcontrol <.deb-file>\n");
		return RET_ERROR;
	}

	result = extractcontrol(&control,argv[1]);
	
	if( RET_IS_OK(result) ) 
		printf("%s\n",control);
	return result;
}


ACTION_U_F(addmd5sums) {
	char buffer[2000],*c,*m;
	retvalue result,r;

	if( argc != 1 ) {
		fprintf(stderr,"reprepro _addmd5sums < <data>\n");
		return RET_ERROR;
	}

	result = RET_NOTHING;
	
	while( fgets(buffer,1999,stdin) != NULL ) {
		c = strchr(buffer,'\n');
		if( c == NULL ) {
			fprintf(stderr,"Line too long\n");
			return RET_ERROR;
		}
		*c = '\0';
		m = strchr(buffer,' ');
		if( m == NULL ) {
			fprintf(stderr,"Malformed line\n");
			return RET_ERROR;
		}
		*m = '\0'; m++;
		r = files_add(filesdb,buffer,m);
		RET_UPDATE(result,r);

	}
	return result;
}


ACTION_R(removereferences) {

	if( argc != 2 ) {
		fprintf(stderr,"reprepro _removereferences <identifier>\n");
		return RET_ERROR;
	}
	return references_remove(references,argv[1]);
}


ACTION_U_R(dumpreferences) {

	if( argc != 1 ) {
		fprintf(stderr,"reprepro dumpreferences\n");
		return RET_ERROR;
	}
	return references_dump(references);
}

struct fileref { /*@temp@*/references refs; };

static retvalue checkifreferenced(void *data,const char *filekey,UNUSED(const char *md5sum)) {
	struct fileref *dist = data;
	retvalue r;

	r = references_isused(dist->refs,filekey);
	if( r == RET_NOTHING ) {
		printf("%s\n",filekey);
		return RET_OK;		
	} else if( RET_IS_OK(r) ) {
		return RET_NOTHING;
	} else
		return r;
}

ACTION_U_RF(dumpunreferenced) {
	retvalue result;
	struct fileref dist;

	if( argc != 1 ) {
		fprintf(stderr,"reprepro dumpunreferenced\n");
		return RET_ERROR;
	}
	dist.refs = references;
	result = files_foreach(filesdb,checkifreferenced,&dist);
	dist.refs = NULL;
	return result;
}

struct filedelref { /*@temp@*/references references; /*@temp@*/filesdb filesdb; };

static retvalue deleteifunreferenced(void *data,const char *filekey,UNUSED(const char *md5sum)) {
	struct filedelref *dist = data;
	retvalue r;

	r = references_isused(dist->references,filekey);
	if( r == RET_NOTHING ) {
		r = files_deleteandremove(dist->filesdb,filekey,!keepdirectories);
		return r;
	} else if( RET_IS_OK(r) ) {
		return RET_NOTHING;
	} else
		return r;
}

ACTION_U_RF(deleteunreferenced) {
	retvalue result;
	struct filedelref dist;

	if( argc != 1 ) {
		fprintf(stderr,"reprepro deleteunreferenced\n");
		return RET_ERROR;
	}
	if( keepunreferenced ) {
		fprintf(stderr,"Calling deleteunreferenced with --keepunreferencedfiles does not really make sense, does it?\n");
		return RET_ERROR;
	}
	dist.references = references;
	dist.filesdb = filesdb;
	result = files_foreach(filesdb,deleteifunreferenced,&dist);
	dist.references = NULL;
	dist.filesdb = NULL;
	return result;
}

ACTION_R(addreference) {

	if( argc != 3 ) {
		fprintf(stderr,"reprepro _addreference <reference> <referee>\n");
		return RET_ERROR;
	}
	return references_increment(references,argv[1],argv[2]);
}


struct remove_args {/*@temp@*/references refs; int count; /*@temp@*/ const char * const *names; bool_t *gotremoved; int todo;/*@temp@*/struct strlist *removedfiles;/*@temp@*/struct trackingdata *trackingdata;};

static retvalue remove_from_target(/*@temp@*/void *data, struct target *target) {
	retvalue result,r;
	int i;
	struct remove_args *d = data;

	result = target_initpackagesdb(target,dbdir);
	if( RET_WAS_ERROR(result) ) {
		return result;
	}

	result = RET_NOTHING;
	for( i = 0 ; i < d->count ; i++ ){
		r = target_removepackage(target,d->refs,d->names[i],d->removedfiles,d->trackingdata);
		if( RET_IS_OK(r) ) {
			if( ! d->gotremoved[i] )
				d->todo--;
			d->gotremoved[i] = TRUE;
		}
		RET_UPDATE(result,r);
	}
	r = target_closepackagesdb(target);
	RET_ENDUPDATE(result,r);
	return result;
}

ACTION_D_U(remove) {
	retvalue result,r;
	struct distribution *distribution;
	struct remove_args d;
	trackingdb tracks;
	struct trackingdata trackingdata;

	if( argc < 3  ) {
		fprintf(stderr,"reprepro [-C <component>] [-A <architecture>] [-T <type>] remove <codename> <package-names>\n");
		return RET_ERROR;
	}
	r = distribution_get(&distribution,confdir,argv[1]);
	assert( r != RET_NOTHING);
	if( RET_WAS_ERROR(r) ) {
		return r;
	}

	if( distribution->tracking != dt_NONE ) {
		r = tracking_initialize(&tracks,dbdir,distribution);
		if( RET_WAS_ERROR(r) ) {
			(void)distribution_free(distribution);
			return r;
		}
		r = trackingdata_new(tracks,&trackingdata);
		if( RET_WAS_ERROR(r) ) {
			(void)distribution_free(distribution);
			(void)tracking_done(tracks);
			return r;
		}
		d.trackingdata = &trackingdata;
	} else {
		d.trackingdata = NULL;
	}

	d.count = argc-2;
	d.names = argv+2;
	d.todo = d.count;
	d.gotremoved = calloc(d.count,sizeof(*d.gotremoved));
	d.refs = references;
	d.removedfiles = dereferenced;
	if( d.gotremoved == NULL )
		result = RET_ERROR_OOM;
	else
		result = distribution_foreach_part(distribution,component,architecture,packagetype,remove_from_target,&d,force);
	d.refs = NULL;
	d.removedfiles = NULL;

	if( d.todo < d.count ) {
		r = distribution_export(distribution,confdir,dbdir,distdir,TRUE);
		RET_ENDUPDATE(result,r);
	}
	if( d.trackingdata != NULL ) {
		trackingdata_done(d.trackingdata);
		r = tracking_done(tracks);
		RET_ENDUPDATE(result,r);
	}
	r = distribution_free(distribution);
	RET_ENDUPDATE(result,r);
	if( verbose >= 0 && !RET_WAS_ERROR(result) && d.todo > 0 ) {
		(void)fputs("Not removed as not found: ",stderr);
		while( d.count > 0 ) {
			d.count--;
			assert(d.gotremoved != NULL);
			if( ! d.gotremoved[d.count] ) {
				(void)fputs(d.names[d.count],stderr);
				d.todo--;
				if( d.todo > 0 )
					(void)fputs(", ",stderr);
			}
		}
		(void)fputc('\n',stderr);
	}
	free(d.gotremoved);
	return result;
}

static retvalue list_in_target(void *data, struct target *target) {
	retvalue r,result;
	const char *packagename = data;
	char *control,*version;

	result = target_initpackagesdb(target,dbdir);
	if( RET_WAS_ERROR(result) ) {
		return result;
	}
	result = packages_get(target->packages,packagename,&control);
	if( RET_IS_OK(result) ) {
		r = (*target->getversion)(target,control,&version);
		if( RET_IS_OK(r) ) {
			printf("%s: %s %s\n",target->identifier,packagename,version);
			free(version);
		} else {
			printf("Could not retrieve version from %s in %s\n",packagename,target->identifier);
		}
		free(control);
	}
	r = target_closepackagesdb(target);
	RET_ENDUPDATE(result,r);
	return result;
}

ACTION_N(list) {
	retvalue r,result;
	struct distribution *distribution;

	if( argc != 3  ) {
		fprintf(stderr,"reprepro [-C <component>] [-A <architecture>] [-T <type>] list <codename> <package-name>\n");
		return RET_ERROR;
	}
	r = distribution_get(&distribution,confdir,argv[1]);
	assert( r != RET_NOTHING);
	if( RET_WAS_ERROR(r) ) {
		return r;
	}

	result = distribution_foreach_part(distribution,component,architecture,packagetype,list_in_target,(void*)argv[2],force);
	r = distribution_free(distribution);
	RET_ENDUPDATE(result,r);
	return result;
}


struct listfilter { /*@temp@*/ struct target *target; /*@temp@*/ term *condition; };

static retvalue listfilterprint(/*@temp@*/void *data,const char *packagename,const char *control){
	struct listfilter *d = data;
	char *version;
	retvalue r;

	r = term_decidechunk(d->condition,control);
	if( RET_IS_OK(r) ) {
		r = (*d->target->getversion)(d->target,control,&version);
		if( RET_IS_OK(r) ) {
			printf("%s: %s %s\n",d->target->identifier,packagename,version);
			free(version);
		} else {
			printf("Could not retrieve version from %s in %s\n",packagename,d->target->identifier);
		}
	}
	return r;
}

static retvalue listfilter_in_target(/*@temp@*/void *data, /*@temp@*/struct target *target) {
	retvalue r,result;
	/*@temp@*/ struct listfilter d;

	result = target_initpackagesdb(target,dbdir);
	if( RET_WAS_ERROR(result) ) {
		return result;
	}
	d.target = target;
	d.condition = data;
	result = packages_foreach(target->packages,listfilterprint,&d,force);
	d.target = NULL;
	d.condition = NULL;

	r = target_closepackagesdb(target);
	RET_ENDUPDATE(result,r);
	return result;
}

ACTION_N(listfilter) {
	retvalue r,result;
	struct distribution *distribution;
	term *condition;

	if( argc != 3  ) {
		fprintf(stderr,"reprepro [-C <component>] [-A <architecture>] [-T <type>] listfilter <codename> <term to describe which packages to list>\n");
		return RET_ERROR;
	}
	r = distribution_get(&distribution,confdir,argv[1]);
	assert( r != RET_NOTHING);
	if( RET_WAS_ERROR(r) ) {
		return r;
	}
	result = term_compile(&condition,argv[2],T_OR|T_BRACKETS|T_NEGATION|T_VERSION|T_NOTEQUAL);
	if( RET_WAS_ERROR(result) ) {
		r = distribution_free(distribution);
		RET_ENDUPDATE(result,r);
		return result;
	}

	result = distribution_foreach_part(distribution,component,architecture,packagetype,listfilter_in_target,condition,force);
	term_free(condition);
	r = distribution_free(distribution);
	RET_ENDUPDATE(result,r);
	return result;
}

ACTION_F(detect) {
	char buffer[5000],*nl;
	int i;
	retvalue r,ret;

	ret = RET_NOTHING;
	if( argc > 1 ) {
		for( i = 1 ; i < argc ; i++ ) {
			r = files_detect(filesdb,argv[i]);
			RET_UPDATE(ret,r);
		}

	} else
		while( fgets(buffer,4999,stdin) != NULL ) {
			nl = strchr(buffer,'\n');
			if( nl == NULL ) {
				return RET_ERROR;
			}
			*nl = '\0';
			r = files_detect(filesdb,buffer);
			RET_UPDATE(ret,r);
		} 
	return ret;
}

ACTION_F(forget) {
	char buffer[5000],*nl;
	int i;
	retvalue r,ret;

	ret = RET_NOTHING;
	if( argc > 1 ) {
		for( i = 1 ; i < argc ; i++ ) {
			r = files_remove(filesdb,argv[i]);
			RET_UPDATE(ret,r);
		}

	} else
		while( fgets(buffer,4999,stdin) != NULL ) {
			nl = strchr(buffer,'\n');
			if( nl == NULL ) {
				return RET_ERROR;
			}
			*nl = '\0';
			r = files_remove(filesdb,buffer);
			RET_UPDATE(ret,r);
		} 
	return ret;
}

ACTION_U_F(listmd5sums) {
	if( argc != 1 ) {
		fprintf(stderr,"reprepro _listmd5sums \n");
		return RET_ERROR;
	}
	return files_printmd5sums(filesdb);
}

static retvalue printout(UNUSED(void *data),const char *package,const char *chunk){
	printf("'%s' -> '%s'\n",package,chunk);
	return RET_OK;
}

ACTION_N(dumpcontents) {
	retvalue result,r;
	packagesdb packages;

	if( argc != 2 ) {
		fprintf(stderr,"reprepro _dumpcontents <identifier>\n");
		return RET_ERROR;
	}

	result = packages_initialize(&packages,dbdir,argv[1]);
	if( RET_WAS_ERROR(result) )
		return result;

	result = packages_foreach(packages,printout,NULL,force);

	r = packages_done(packages);
	RET_ENDUPDATE(result,r);
	
	return result;
}

ACTION_N(export) {
	retvalue result,r;
	struct distribution *distributions,*d;

	if( argc < 1 ) {
		fprintf(stderr,"reprepro export [<distributions>]\n");
		return RET_ERROR;
	}
	
	result = distribution_getmatched(confdir,argc-1,argv+1,&distributions);
	assert( result != RET_NOTHING);
	if( RET_WAS_ERROR(result) )
		return result;
	result = RET_NOTHING;
	for( d = distributions ; d != NULL ; d = d->next ) {
		if( verbose > 0 ) {
			fprintf(stderr,"Exporting %s...\n",d->codename);
		}

		r = distribution_export(d,confdir,dbdir,distdir,FALSE);
		RET_UPDATE(result,r);
		if( RET_WAS_ERROR(r) && force<= 0 )
			return r;
	}
	r = distribution_freelist(distributions);
	RET_ENDUPDATE(result,r);
	return result;
}

/***********************update********************************/

ACTION_D(update) {
	retvalue result,r;
	bool_t doexport;
	struct update_pattern *patterns;
	struct distribution *distributions;
	struct update_distribution *u_distributions;

	if( argc < 1 ) {
		fprintf(stderr,"reprepro update [<distributions>]\n");
		return RET_ERROR;
	}

	result = dirs_make_recursive(listdir);	
	if( RET_WAS_ERROR(result) ) {
		return result;
	}

	result = distribution_getmatched(confdir,argc-1,argv+1,&distributions);
	assert( result != RET_NOTHING );
	if( RET_WAS_ERROR(result) )
		return result;

	result = updates_getpatterns(confdir,&patterns);
	if( RET_WAS_ERROR(result) ) {
		r = distribution_freelist(distributions);
		RET_ENDUPDATE(result,r);
		return result;
	}
	assert( RET_IS_OK(result) );

	result = updates_calcindices(listdir,patterns,distributions,&u_distributions);
	if( RET_WAS_ERROR(result) ) {
		updates_freepatterns(patterns);
		r = distribution_freelist(distributions);
		RET_ENDUPDATE(result,r);
		return result;
	}
	assert( RET_IS_OK(result) );

	if( !keepunneededlists ) {
		result = updates_clearlists(listdir,u_distributions);
	}
	if( !RET_WAS_ERROR(result) )
		result = updates_update(dbdir,methoddir,filesdb,references,u_distributions,force,nolistsdownload,skipold,dereferenced);
	updates_freeupdatedistributions(u_distributions);
	updates_freepatterns(patterns);

	doexport = force>0 || RET_IS_OK(result);
	if( doexport && verbose >= 0 )
		fprintf(stderr,"Exporting indices...\n");
	if( doexport )
		r = distribution_exportandfreelist(distributions,confdir,dbdir,distdir);
	else
		r = distribution_freelist(distributions);
	RET_ENDUPDATE(result,r);

	return result;
}

ACTION_D(iteratedupdate) {
	retvalue result,r;
	bool_t doexport;
	struct update_pattern *patterns;
	struct distribution *distributions;
	struct update_distribution *u_distributions;

	if( argc < 1 ) {
		fprintf(stderr,"reprepro iteratedupdate [<distributions>]\n");
		return RET_ERROR;
	}

	result = dirs_make_recursive(listdir);	
	if( RET_WAS_ERROR(result) ) {
		return result;
	}

	result = distribution_getmatched(confdir,argc-1,argv+1,&distributions);
	assert( result != RET_NOTHING );
	if( RET_WAS_ERROR(result) )
		return result;

	result = updates_getpatterns(confdir,&patterns);
	if( RET_WAS_ERROR(result) ) {
		r = distribution_freelist(distributions);
		RET_ENDUPDATE(result,r);
		return result;
	}

	result = updates_calcindices(listdir,patterns,distributions,&u_distributions);
	if( RET_WAS_ERROR(result) ) {
		updates_freepatterns(patterns);
		r = distribution_freelist(distributions);
		RET_ENDUPDATE(result,r);
		return result;
	}

	if( !keepunneededlists ) {
		result = updates_clearlists(listdir,u_distributions);
	}
	if( !RET_WAS_ERROR(result) )
		result = updates_iteratedupdate(confdir,dbdir,distdir,methoddir,filesdb,references,u_distributions,force,nolistsdownload,skipold,dereferenced);
	updates_freeupdatedistributions(u_distributions);
	updates_freepatterns(patterns);

	doexport = force>0 || RET_IS_OK(result);
	if( doexport && verbose >= 0 )
		fprintf(stderr,"Exporting indices...\n");
	if( doexport )
		r = distribution_exportandfreelist(distributions,confdir,dbdir,distdir);
	else
		r = distribution_freelist(distributions);
	RET_ENDUPDATE(result,r);

	return result;
}

ACTION_N(checkupdate) {
	retvalue result,r;
	struct update_pattern *patterns;
	struct distribution *distributions;
	struct update_distribution *u_distributions;

	if( argc < 1 ) {
		fprintf(stderr,"reprepro checkupdate [<distributions>]\n");
		return RET_ERROR;
	}

	result = dirs_make_recursive(listdir);	
	if( RET_WAS_ERROR(result) ) {
		return result;
	}

	result = distribution_getmatched(confdir,argc-1,argv+1,&distributions);
	assert( result != RET_NOTHING);
	if( RET_WAS_ERROR(result) )
		return result;

	result = updates_getpatterns(confdir,&patterns);
	if( RET_WAS_ERROR(result) ) {
		r = distribution_freelist(distributions);
		RET_ENDUPDATE(result,r);
		return result;
	}

	result = updates_calcindices(listdir,patterns,distributions,&u_distributions);
	if( RET_WAS_ERROR(result) ) {
		updates_freepatterns(patterns);
		r = distribution_freelist(distributions);
		RET_ENDUPDATE(result,r);
		return result;
	}

	result = updates_checkupdate(dbdir,methoddir,u_distributions,force,nolistsdownload,skipold);

	updates_freeupdatedistributions(u_distributions);
	updates_freepatterns(patterns);
	r = distribution_freelist(distributions);
	RET_ENDUPDATE(result,r);

	return result;
}


/***********************rereferencing*************************/
struct data_binsrcreref { /*@temp@*/const struct distribution *distribution; /*@temp@*/references refs;};

static retvalue reref(void *data,struct target *target) {
	retvalue result,r;
	struct data_binsrcreref *d = data;

	result = target_initpackagesdb(target,dbdir);
	if( !RET_WAS_ERROR(result) ) {
		result = target_rereference(target,d->refs,force);
		r = target_closepackagesdb(target);
		RET_ENDUPDATE(result,r);
	}
	return result;
}


ACTION_R(rereference) {
	retvalue result,r;
	struct distribution *distributions,*d;

	if( argc < 1 ) {
		fprintf(stderr,"reprepro rereference [<distributions>]\n");
		return RET_ERROR;
	}

	result = distribution_getmatched(confdir,argc-1,argv+1,&distributions);
	assert( result != RET_NOTHING );
	if( RET_WAS_ERROR(result) ) {
		return result;
	}
	result = RET_NOTHING;
	for( d = distributions ; d != NULL ; d = d->next ) {
		struct data_binsrcreref dat;

		if( verbose > 0 ) {
			fprintf(stderr,"Referencing %s...\n",d->codename);
		}
		dat.distribution = d;
		dat.refs = references;

		r = distribution_foreach_part(d,component,architecture,packagetype,
				reref,&dat,force);
		dat.refs = NULL;
		dat.distribution = NULL;
		RET_UPDATE(result,r);
		if( RET_WAS_ERROR(r) && force <= 0 )
			break;
	}
	r = distribution_freelist(distributions);
	RET_ENDUPDATE(result,r);

	return result;
}
/***************************retrack****************************/
struct data_binsrctrack { /*@temp@*/const struct distribution *distribution; /*@temp@*/references refs; trackingdb tracks;};

static retvalue retrack(void *data,struct target *target) {
	retvalue result,r;
	struct data_binsrctrack *d = data;

	result = target_initpackagesdb(target,dbdir);
	if( !RET_WAS_ERROR(result) ) {
		result = target_retrack(target,d->tracks,d->refs,force);
		r = target_closepackagesdb(target);
		RET_ENDUPDATE(result,r);
	}
	return result;
}

ACTION_R(retrack) {
	retvalue result,r;
	struct distribution *distributions,*d;

	if( argc < 1 ) {
		fprintf(stderr,"reprepro retrack [<distributions>]\n");
		return RET_ERROR;
	}

	result = distribution_getmatched(confdir,argc-1,argv+1,&distributions);
	assert( result != RET_NOTHING );
	if( RET_WAS_ERROR(result) ) {
		return result;
	}
	result = RET_NOTHING;
	for( d = distributions ; d != NULL ; d = d->next ) {
		struct data_binsrctrack dat;

		if( verbose > 0 ) {
			fprintf(stderr,"Chasing %s...\n",d->codename);
		}
		dat.distribution = d;
		dat.refs = references;
		r = tracking_initialize(&dat.tracks,dbdir,d);
		if( RET_WAS_ERROR(r) ) {
			RET_UPDATE(result,r);
			if( RET_WAS_ERROR(r) && force <= 0 )
				break;
			continue;
		}
		r = tracking_clearall(dat.tracks);
		RET_UPDATE(result,r);
		r = references_remove(references,d->codename);
		RET_UPDATE(result,r);

		r = distribution_foreach_part(d,component,architecture,packagetype,
				retrack,&dat,force);
		RET_UPDATE(result,r);
		dat.refs = NULL;
		dat.distribution = NULL;
		r = tracking_done(dat.tracks);
		RET_ENDUPDATE(result,r);
		if( RET_WAS_ERROR(result) && force <= 0 )
			break;
	}
	r = distribution_freelist(distributions);
	RET_ENDUPDATE(result,r);

	return result;
}

ACTION_D_U(removetrack) {
	retvalue result,r;
	struct distribution *distribution;
	trackingdb tracks;

	if( argc != 4 ) {
		fprintf(stderr,"reprepro removetrack <distribution> <sourcename> <version>\n");
		return RET_ERROR;
	}
	result = distribution_get(&distribution,confdir,argv[1]);
	assert( result != RET_NOTHING );
	if( RET_WAS_ERROR(result) )
		return result;
	r = tracking_initialize(&tracks,dbdir,distribution);
	if( RET_WAS_ERROR(r) ) {
		distribution_free(distribution);
		return r;
	}

	result = tracking_remove(tracks,argv[2],argv[3],references,dereferenced);

	r = tracking_done(tracks);
	RET_ENDUPDATE(result,r);
	r = distribution_free(distribution);
	RET_ENDUPDATE(result,r);
	return result;
}
	
ACTION_R(cleartracks) {
	retvalue result,r;
	struct distribution *distributions,*d;

	if( argc < 1 ) {
		fprintf(stderr,"reprepro cleartracks [<distributions>]\n");
		return RET_ERROR;
	}

	result = distribution_getmatched(confdir,argc-1,argv+1,&distributions);
	assert( result != RET_NOTHING );
	if( RET_WAS_ERROR(result) ) {
		return result;
	}
	result = RET_NOTHING;
	for( d = distributions ; d != NULL ; d = d->next ) {
		trackingdb tracks;

		if( verbose > 0 ) {
			fprintf(stderr,"Deleting all tracks for %s...\n",d->codename);
		}
		r = tracking_initialize(&tracks,dbdir,d);
		if( RET_WAS_ERROR(r) ) {
			RET_UPDATE(result,r);
			if( RET_WAS_ERROR(r) && force <= 0 )
				break;
			continue;
		}
		r = tracking_clearall(tracks);
		RET_UPDATE(result,r);
		r = references_remove(references,d->codename);
		RET_UPDATE(result,r);
		r = tracking_done(tracks);
		RET_ENDUPDATE(result,r);
		if( RET_WAS_ERROR(result) && force <= 0 )
			break;
	}
	r = distribution_freelist(distributions);
	RET_ENDUPDATE(result,r);

	return result;
}
ACTION_N(dumptracks) {
	retvalue result,r;
	struct distribution *distributions,*d;

	if( argc < 1 ) {
		fprintf(stderr,"reprepro dumptracks [<distributions>]\n");
		return RET_ERROR;
	}

	result = distribution_getmatched(confdir,argc-1,argv+1,&distributions);
	assert( result != RET_NOTHING );
	if( RET_WAS_ERROR(result) ) {
		return result;
	}
	result = RET_NOTHING;
	for( d = distributions ; d != NULL ; d = d->next ) {
		trackingdb tracks;

		r = tracking_initialize(&tracks,dbdir,d);
		if( RET_WAS_ERROR(r) ) {
			RET_UPDATE(result,r);
			if( RET_WAS_ERROR(r) && force <= 0 )
				break;
			continue;
		}
		r = tracking_printall(tracks);
		RET_UPDATE(result,r);
		r = tracking_done(tracks);
		RET_ENDUPDATE(result,r);
		if( RET_WAS_ERROR(result) && force <= 0 )
			break;
	}
	r = distribution_freelist(distributions);
	RET_ENDUPDATE(result,r);

	return result;
}
/***********************checking*************************/
struct data_check { /*@temp@*/const struct distribution *distribution; /*@temp@*/references references; /*@temp@*/filesdb filesdb;};

static retvalue check_target(void *data,struct target *target) {
	struct data_check *d = data;
	retvalue result,r;

	r = target_initpackagesdb(target,dbdir);
	if( RET_WAS_ERROR(r) )
		return r;
	result = target_check(target,d->filesdb,d->references,force);
	r = target_closepackagesdb(target);
	RET_ENDUPDATE(result,r);
	return result;
}

ACTION_RF(check) {
	retvalue result,r;
	struct distribution *distributions,*d;

	if( argc < 1 ) {
		fprintf(stderr,"reprepro check [<distributions>]\n");
		return RET_ERROR;
	}

	result = distribution_getmatched(confdir,argc-1,argv+1,&distributions);
	assert( result != RET_NOTHING );
	if( RET_WAS_ERROR(result) ) {
		return result;
	}
	result = RET_NOTHING;
	for( d = distributions ; d != NULL ; d = d->next ) {
		struct data_check dat;

		if( verbose > 0 ) {
			fprintf(stderr,"Checking %s...\n",d->codename);
		}

		dat.distribution = d;
		dat.references = references;
		dat.filesdb = filesdb;

		r = distribution_foreach_part(d,component,architecture,packagetype,check_target,&dat,force);
		dat.references = NULL;
		dat.filesdb = NULL;
		dat.distribution = NULL;
		RET_UPDATE(result,r);
		if( RET_WAS_ERROR(r) && force <= 0 )
			break;
	}
	r = distribution_freelist(distributions);
	RET_ENDUPDATE(result,r);

	return result;
}

ACTION_F(checkpool) {

	if( argc < 1 || argc > 2 || (argc == 2 && strcmp(argv[1],"fast") != 0)) {
		fprintf(stderr,"reprepro checkpool [fast] \n");
		return RET_ERROR;
	}

	return files_checkpool(filesdb,argc == 2);
}
/*****************reapplying override info***************/

static retvalue reoverride_target(void *data,struct target *target) {
	const struct alloverrides *ao = data;
	retvalue result,r;

	r = target_initpackagesdb(target,dbdir);
	if( RET_WAS_ERROR(r) )
		return r;
	result = target_reoverride(target,ao);
	r = target_closepackagesdb(target);
	RET_ENDUPDATE(result,r);
	return result;
}

ACTION_N(reoverride) {
	retvalue result,r;
	struct distribution *distributions,*d;

	if( argc < 1 ) {
		fprintf(stderr,"reprepro [-T ...] [-C ...] [-A ...] reoverride [<distributions>]\n");
		return RET_ERROR;
	}

	result = distribution_getmatched(confdir,argc-1,argv+1,&distributions);
	assert( result != RET_NOTHING );
	if( RET_WAS_ERROR(result) ) {
		return result;
	}
	result = RET_NOTHING;
	for( d = distributions ; d != NULL ; d = d->next ) {
		struct alloverrides ao;

		if( verbose > 0 ) {
			fprintf(stderr,"Reapplying override to %s...\n",d->codename);
		}

		r = override_readall(overridedir,&ao,d->deb_override,d->udeb_override,d->dsc_override);
		if( RET_IS_OK(r) ) {
			r = distribution_foreach_part(d,component,architecture,packagetype,reoverride_target,&ao,FALSE);
			override_free(ao.deb);
			override_free(ao.udeb);
			override_free(ao.dsc);
		} else if( r == RET_NOTHING ) {
			fprintf(stderr,"No override files, thus nothing to do for %s.\n",d->codename);
		}
		RET_UPDATE(result,r);
		if( RET_WAS_ERROR(r) && force <= 0 )
			break;
	}
	if( RET_IS_OK(result) )
		r = distribution_exportandfreelist(distributions,confdir,dbdir,distdir);
	else
		r = distribution_freelist(distributions);
	RET_ENDUPDATE(result,r);

	return result;
}

/***********************include******************************************/

ACTION_D(includedeb) {
	retvalue result,r;
	struct distribution *distribution;
	struct overrideinfo *override;
	bool_t isudeb;
	const char *overridefile;
	trackingdb tracks;

	if( argc != 3 ) {
		fprintf(stderr,"reprepro [--delete] include[u]deb <distribution> <package>\n");
		return RET_ERROR;
	}
	if( onlyacceptsigned ) {
		fprintf(stderr,"include[u]deb together with --onlyacceptsigned is not yet possible,\n as .[u]deb files cannot be signed yet.\n");
		return RET_ERROR;
	}
	if( strcmp(argv[0],"includeudeb") == 0 ) {
		isudeb = TRUE;
		if( packagetype != NULL && strcmp(packagetype,"udeb") != 0 ) {
			fprintf(stderr,"Calling includeudeb with a -T different from 'udeb' makes no sense!\n");
			return RET_ERROR;
		}
	} else if( strcmp(argv[0],"includedeb") == 0 ) {
		isudeb = FALSE;
		if( packagetype != NULL && strcmp(packagetype,"deb") != 0 ) {
			fprintf(stderr,"Calling includedeb with -T something where something is not 'deb' makes no sense!\n");
			return RET_ERROR;
		}

	} else {
		fprintf(stderr,"Internal error with command parsing!\n");
		return RET_ERROR;
	}

	result = distribution_get(&distribution,confdir,argv[1]);
	assert( result != RET_NOTHING );
	if( RET_WAS_ERROR(result) ) {
		return result;
	}

	overridefile = isudeb?distribution->udeb_override:distribution->deb_override;
	if( overridefile != NULL ) {
		result = override_read(overridedir,overridefile,&override);
		if( RET_WAS_ERROR(result) ) {
			r = distribution_free(distribution);
			RET_ENDUPDATE(result,r);
			return result;
		}
	} else
		override = NULL;

	// TODO: same for component? (depending on type?)
	if( architecture != NULL && !strlist_in(&distribution->architectures,architecture) ){
		fprintf(stderr,"Cannot force into the architecture '%s' not available in '%s'!\n",architecture,distribution->codename);
		override_free(override);
		(void)distribution_free(distribution);
		return RET_ERROR;
	}

	if( distribution->tracking != dt_NONE ) {
		result = tracking_initialize(&tracks,dbdir,distribution);
		if( RET_WAS_ERROR(result) ) {
			r = distribution_free(distribution);
			RET_ENDUPDATE(result,r);
			override_free(override);
			return result;
		}
	} else {
		tracks = NULL;
	}

	result = deb_add(dbdir,references,filesdb,component,architecture,
			section,priority,isudeb?"udeb":"deb",distribution,argv[2],
			NULL,NULL,override,delete,
			dereferenced,tracks);

	override_free(override);

	r = tracking_done(tracks);
	RET_ENDUPDATE(result,r);

	r = distribution_export(distribution,confdir,dbdir,distdir,TRUE);
	RET_ENDUPDATE(result,r);

	r = distribution_free(distribution);
	RET_ENDUPDATE(result,r);

	return result;
}


ACTION_D(includedsc) {
	retvalue result,r;
	struct distribution *distribution;
	struct overrideinfo *srcoverride;
	trackingdb tracks;

	if( argc != 3 ) {
		fprintf(stderr,"reprepro [--delete] includedsc <distribution> <package>\n");
		return RET_ERROR;
	}

	if( architecture != NULL && strcmp(architecture,"source") != 0 ) {
		fprintf(stderr,"Cannot put a source-package anywhere else than in architecture 'source'!\n");
		return RET_ERROR;
	}
	if( packagetype != NULL && strcmp(packagetype,"dsc") != 0 ) {
		fprintf(stderr,"Cannot put a source-package anywhere else than in type 'dsc'!\n");
		return RET_ERROR;
	}

	result = distribution_get(&distribution,confdir,argv[1]);
	assert( result != RET_NOTHING );
	if( RET_WAS_ERROR(result) )
		return result;
	srcoverride = NULL;
	if( distribution->dsc_override != NULL ) {
		result = override_read(overridedir,distribution->dsc_override,&srcoverride);
		if( RET_WAS_ERROR(result) ) {
			r = distribution_free(distribution);
			RET_ENDUPDATE(result,r);
			return result;
		}
	}

	if( distribution->tracking != dt_NONE ) {
		result = tracking_initialize(&tracks,dbdir,distribution);
		if( RET_WAS_ERROR(result) ) {
			r = distribution_free(distribution);
			RET_ENDUPDATE(result,r);
			override_free(srcoverride);
			return result;
		}
	} else {
		tracks = NULL;
	}

	result = dsc_add(dbdir,references,filesdb,component,section,priority,distribution,argv[2],NULL,NULL,NULL,NULL,srcoverride,delete,dereferenced,onlyacceptsigned,tracks);
	
	override_free(srcoverride);
	r = tracking_done(tracks);
	RET_ENDUPDATE(result,r);
	r = distribution_export(distribution,confdir,dbdir,distdir,TRUE);
	RET_ENDUPDATE(result,r);
	r = distribution_free(distribution);
	RET_ENDUPDATE(result,r);

	return result;
}

ACTION_D(include) {
	retvalue result,r;
	struct distribution *distribution;
	struct alloverrides ao;
	trackingdb tracks;

	if( argc != 3 ) {
		fprintf(stderr,"reprepro [--delete] include <distribution> <.changes-file>\n");
		return RET_ERROR;
	}

	result = distribution_get(&distribution,confdir,argv[1]);
	assert( result != RET_NOTHING );
	if( RET_WAS_ERROR(result) )
		return result;

	result = override_readall(overridedir,&ao,distribution->deb_override,distribution->udeb_override,distribution->dsc_override);
	if( RET_WAS_ERROR(result) ) {
		r = distribution_free(distribution);
		RET_ENDUPDATE(result,r);
		return result;
	}

	if( distribution->tracking != dt_NONE ) {
		result = tracking_initialize(&tracks,dbdir,distribution);
		if( RET_WAS_ERROR(result) ) {
			r = distribution_free(distribution);
			RET_ENDUPDATE(result,r);
			override_free(ao.deb);override_free(ao.udeb);override_free(ao.dsc);
			return result;
		}
	} else {
		tracks = NULL;
	}

	result = changes_add(dbdir,tracks,references,filesdb,packagetype,component,architecture,section,priority,distribution,&ao,argv[2],delete,dereferenced,onlyacceptsigned);

	override_free(ao.deb);override_free(ao.udeb);override_free(ao.dsc);

	r = tracking_done(tracks);
	RET_ENDUPDATE(result,r);
	r = distribution_export(distribution,confdir,dbdir,distdir,TRUE);
	RET_ENDUPDATE(result,r);
	r = distribution_free(distribution);
	RET_ENDUPDATE(result,r);

	return result;
}

/***********************createsymlinks***********************************/

ACTION_N(createsymlinks) {
	retvalue result,r;
	struct distribution *distributions,*d,*d2;

	r = dirs_make_recursive(distdir);
	if( RET_WAS_ERROR(r) )
		return r;

	result = distribution_getmatched(confdir,argc-1,argv+1,&distributions);
	assert( result != RET_NOTHING );
	if( RET_WAS_ERROR(result) ) {
		return result;
	}
	result = RET_NOTHING;
	for( d = distributions ; d != NULL ; d = d->next ) {
		char *linkname,*buffer;
		size_t bufsize;
		int ret;
		
		if( d->suite == NULL )
			continue;
		r = RET_NOTHING;
		for( d2 = distributions ; d2 != NULL ; d2 = d2->next ) {
			if( d!=d2 && d2->suite!=NULL &&strcmp(d->suite,d2->suite)==0) {
				fprintf(stderr,
"Not linking %s->%s due to conflict with %s->%s\n",
					d->suite,d->codename,
					d2->suite,d2->codename);
				r = RET_ERROR;
			} else if( strcmp(d->suite,d2->codename)==0) {
				fprintf(stderr,
"Not linking %s->%s due to conflict with %s\n",
					d->suite,d->codename,d2->codename);
				r = RET_ERROR;
			}
		}
		if( RET_WAS_ERROR(r) ) {
			RET_UPDATE(result,r);
			continue;
		}

		linkname = calc_dirconcat(distdir,d->suite);
		bufsize = strlen(d->codename)+10;
		buffer = calloc(1,bufsize);
		if( linkname == NULL || buffer == NULL ) {
			free(linkname);free(buffer);
			fputs("Out of Memory!\n",stderr);
			return RET_ERROR_OOM;
		}

		ret = readlink(linkname,buffer,bufsize-4);
		if( ret < 0 && errno == ENOENT ) {
			ret = symlink(d->codename,linkname);
			if( ret != 0 ) {
				r = RET_ERRNO(errno);
				fprintf(stderr,"Error creating symlink %s->%s: %d=%m\n",linkname,d->codename,errno);
				RET_UPDATE(result,r);
			} else {
				if( verbose > 0 ) {
					printf("Created %s->%s\n",linkname,d->codename);
				}
				RET_UPDATE(result,RET_OK);
			}
		} else if( ret >= 0 ) {
			buffer[ret] = '\0';
			if( ret >= ((int)bufsize)-4 ) {
				buffer[bufsize-4]='.';
				buffer[bufsize-3]='.';
				buffer[bufsize-2]='.';
				buffer[bufsize-1]='\0';
			}
			if( strcmp(buffer,d->codename) == 0 ) {
				if( verbose > 2 ) {
					printf("Already ok: %s->%s\n",linkname,d->codename);
				}
				RET_UPDATE(result,RET_OK);
			} else {
				if( delete <= 0 ) {
					fprintf(stderr,"Cannot create %s as already pointing to %s instead of %s,\n use --delete to delete the old link before creating an new one.\n",linkname,buffer,d->codename);
					RET_UPDATE(result,RET_ERROR);
				} else {
					unlink(linkname);
					ret = symlink(d->codename,linkname);
					if( ret != 0 ) {
						r = RET_ERRNO(errno);
						fprintf(stderr,"Error creating symlink %s->%s: %d=%m\n",linkname,d->codename,errno);
						RET_UPDATE(result,r);
					} else {
						if( verbose > 0 ) {
							printf("Replaced %s->%s\n",linkname,d->codename);
						}
						RET_UPDATE(result,RET_OK);
					}

				}
			}
		} else {
			r = RET_ERRNO(errno);
			fprintf(stderr,"Error checking %s, perhaps not a symlink?: %d=%m\n",linkname,errno);
			RET_UPDATE(result,r);
		}

		RET_UPDATE(result,r);
	}
	r = distribution_freelist(distributions);
	RET_ENDUPDATE(result,r);
	return result;
}


/**********************/
/* lock file handling */
/**********************/

static retvalue acquirelock(const char *dbdir) {
	char *lockfile;
	int fd;
	retvalue r;

	// TODO: create directory 
	r = dirs_make_recursive(dbdir);
	if( RET_WAS_ERROR(r) )
		return r;

	lockfile = calc_dirconcat(dbdir,"lockfile");
	if( lockfile == NULL )
		return RET_ERROR_OOM;
	fd = open(lockfile,O_WRONLY|O_CREAT|O_EXCL|O_NOFOLLOW|O_NOCTTY,S_IRUSR|S_IWUSR);
	if( fd < 0 ) {
		int e = errno;
		fprintf(stderr,"Error creating lockfile '%s': %d=%m!\n",lockfile,e);
		free(lockfile);
		if( e == EEXIST ) {
			fprintf(stderr,
"The lockfile already exists, there might be another instance with the\n"
"same database dir running. To avoid locking overhead, only one process\n"
"can access the database at the same time. Only delete the lockfile if\n"
"you are sure no other version is still running!\n");

		}
		return RET_ERRNO(e);
	}
	// TODO: do some more locking of this file to avoid problems
	// with the non-atomity of O_EXCL with nfs-filesystems...
	if( close(fd) != 0 ) {
		int e = errno;
		fprintf(stderr,"Error creating lockfile '%s': %d=%m!\n",lockfile,e);
		(void)unlink(lockfile);
		free(lockfile);
		return RET_ERRNO(e);
	
	}
	free(lockfile);
	return RET_OK;
}

static void releaselock(const char *dbdir) {
	char *lockfile;

	lockfile = calc_dirconcat(dbdir,"lockfile");
	if( lockfile == NULL )
		return;
	if( unlink(lockfile) != 0 ) {
		int e = errno;
		fprintf(stderr,"Error deleting lockfile '%s': %d=%m!\n",lockfile,e);
		(void)unlink(lockfile);
	}
	free(lockfile);
}

/*********************/
/* argument handling */
/*********************/

#define NEED_REFERENCES 1
#define NEED_FILESDB 2
#define NEED_DEREF 4
#define A_N(w) action_n_ ## w, 0
#define A_F(w) action_f_ ## w, NEED_FILESDB
#define A_R(w) action_r_ ## w, NEED_REFERENCES
#define A_RF(w) action_rf_ ## w, NEED_FILESDB|NEED_REFERENCES
/* to dereference files, one needs files and references database: */
#define A_D(w) action_d_ ## w, NEED_FILESDB|NEED_REFERENCES|NEED_DEREF 

static const struct action {
	char *name;
	retvalue (*start)(
			/*@null@*/references references, 
			/*@null@*/filesdb filesdb,
			/*@null@*/struct strlist *dereferencedfilekeys,
			int argc,const char *argv[]);
	int needs;
} all_actions[] = {
	{"__d", 		A_N(printargs)},
	{"__extractcontrol",	A_N(extractcontrol)},
	{"_detect", 		A_F(detect)},
	{"_forget", 		A_F(forget)},
	{"_listmd5sums",	A_F(listmd5sums)},
	{"_addmd5sums",		A_F(addmd5sums)},
	{"_dumpcontents", 	A_N(dumpcontents)},
	{"_removereferences", 	A_R(removereferences)},
	{"_addreference", 	A_R(addreference)},
	{"remove", 		A_D(remove)},
	{"list", 		A_N(list)},
	{"listfilter", 		A_N(listfilter)},
	{"createsymlinks", 	A_N(createsymlinks)},
	{"export", 		A_N(export)},
	{"check", 		A_RF(check)},
	{"reoverride", 		A_N(reoverride)},
	{"checkpool", 		A_F(checkpool)},
	{"rereference", 	A_R(rereference)},
	{"dumpreferences", 	A_R(dumpreferences)},
	{"dumpunreferenced", 	A_RF(dumpunreferenced)},
	{"deleteunreferenced", 	A_RF(deleteunreferenced)},
	{"retrack",	 	A_R(retrack)},
	{"dumptracks",	 	A_N(dumptracks)},
	{"cleartracks",	 	A_R(cleartracks)},
	{"removetrack",		A_D(removetrack)},
	{"update",		A_D(update)},
	{"iteratedupdate",	A_D(iteratedupdate)},
	{"checkupdate",		A_N(checkupdate)},
	{"includedeb",		A_D(includedeb)},
	{"includeudeb",		A_D(includedeb)},
	{"includedsc",		A_D(includedsc)},
	{"include",		A_D(include)},
	{NULL,NULL,0}
};
#undef A_N
#undef A_F
#undef A_R
#undef A_RF
#undef A_F

static retvalue callaction(const struct action *action,int argc,const char *argv[]) {
	retvalue result;
	references references;
	filesdb filesdb;
	struct strlist dereferencedfilekeys;
	bool_t deletederef;

	assert(action != NULL);
	
	deletederef = ISSET(action->needs,NEED_DEREF) && !keepunreferenced;
	
	result = acquirelock(dbdir);
	if( !RET_IS_OK(result) )
		return result;

	if( ISSET(action->needs,NEED_REFERENCES) )
		result = references_initialize(&references,dbdir);
	else
		references = NULL;

	assert( result != RET_NOTHING );
	if( RET_IS_OK(result) ) {

		if( ISSET(action->needs,NEED_FILESDB) )
			result = files_initialize(&filesdb,dbdir,mirrordir);
		else
			filesdb = NULL;

		assert( result != RET_NOTHING );
		if( RET_IS_OK(result) ) {

			if( deletederef ) {
				assert( ISSET(action->needs,NEED_REFERENCES) );
				assert( ISSET(action->needs,NEED_REFERENCES) );
				result = strlist_init(&dereferencedfilekeys);
			}

			assert( result != RET_NOTHING );
			if( RET_IS_OK(result) ) {
				result = action->start(references,filesdb,
					deletederef?&dereferencedfilekeys:NULL,
					argc,argv);

				if( deletederef ) {
					if( dereferencedfilekeys.count > 0 ) {
					    if( RET_IS_OK(result) ) {
						retvalue r;

						assert(filesdb!=NULL);
						assert(references!=NULL);

						if( verbose >= 0 )
					  	    fprintf(stderr,
"Deleting files no longer referenced...\n");
						r = removeunreferencedfiles(
							references,filesdb,
							&dereferencedfilekeys);
						RET_UPDATE(result,r);
					    } else {
						    fprintf(stderr,
"Not deleting possibly left over files due to previous errors.\n"
"(To avoid the files in the still existing index files vanishing)\n"
"Use dumpunreferenced/deleteunreferenced to show/delete files without referenes.\n");
					    }
					}
					strlist_done(&dereferencedfilekeys);
				}
			}
			if( ISSET(action->needs,NEED_FILESDB) ) {
				retvalue r = files_done(filesdb);
				RET_ENDUPDATE(result,r);
			}
		}

		if( ISSET(action->needs,NEED_REFERENCES) ) {
			retvalue r = references_done(references);
			RET_ENDUPDATE(result,r);
		}
	}
	releaselock(dbdir);
	return result;
}


#define LO_DELETE 1
#define LO_KEEPUNREFERENCED 2
#define LO_KEEPUNNEEDEDLISTS 3
#define LO_NOTHINGISERROR 4
#define LO_NOLISTDOWNLOAD 5
#define LO_ONLYACCEPTSIGNED 6
#define LO_ASKPASSPHRASE 7
#define LO_KEEPDIRECTORIES 8
#define LO_SKIPOLD 9
#define LO_NODELETE 21
#define LO_NOKEEPUNREFERENCED 22
#define LO_NOKEEPUNNEEDEDLISTS 23
#define LO_NONOTHINGISERROR 24
#define LO_LISTDOWNLOAD 25
#define LO_NOONLYACCEPTSIGNED 26
#define LO_NOASKPASSPHRASE 27
#define LO_NOKEEPDIRECTORIES 28
#define LO_NOSKIPOLD 29
#define LO_DISTDIR 10
#define LO_DBDIR 11
#define LO_LISTDIR 12
#define LO_OVERRIDEDIR 13
#define LO_CONFDIR 14
#define LO_METHODDIR 15
#define LO_VERSION 20
#define LO_UNIGNORE 30
static int longoption = 0;
const char *programname;

static void handle_option(int c,const char *optarg) {
	retvalue r;

	switch( c ) {
		case 'h':
			printf(
"reprepro - Produce and Manage and Debian package repository\n\n"
"options:\n"
" -h, --help:                        Show this help\n"
" -i  --ignore <flag>:               Ignore errors of type <flag>.\n"
"     --keepunreferencedfiles:       Do not delete files no longer needed.\n"
"     --delete:                      Delete included files if reasonable.\n"
" -b, --basedir <dir>:               Base-dir (will overwrite prior given\n"
"                                                        -d, -D, -L, -c).\n"
"     --distdir <dir>:               Directory to place the \"dists\" dir in.\n"
"     --dbdir <dir>:                 Directory to place the database in.\n"
"     --listdir <dir>:               Directory to place downloaded lists in.\n"
"     --confdir <dir>:               Directory to search configuration in.\n"
"     --overridedir <dir>:           Directory to search override files in.\n"
"     --methodir <dir>:              Use instead of /usr/lib/apt/methods/\n"
" -S, --section <section>:           Force include* to set section.\n"
" -P, --priority <priority>:         Force include* to set priority.\n"
" -C, --component <component>: 	     Add,list or delete only in component.\n"
" -A, --architecture <architecture>: Add,list or delete only to architecture.\n"
" -T, --type <type>:                 Add,list or delete only type (dsc,deb,udeb).\n"
"\n"
"actions (selection, for more see manpage):\n"
" dumpreferences:    Print all saved references\n"
" dumpunreferenced:   Print registered files withour reference\n"
" deleteunreferenced: Delete and forget all unreferenced files\n"
" checkpool:          Check if all files in the pool are still in proper shape.\n"
" check [<distributions>]\n"
"       Check for all needed files to be registered properly.\n"
" export [<distributions>]\n"
"	Force (re)generation of Packages.gz/Packages/Sources.gz/Release\n"
" update [<distributions>]\n"
"	Update the given distributions from the configured sources.\n"
" remove <distribution> <packagename>\n"
"       Remove the given package from the specified distribution.\n"
" include <distribution> <.changes-file>\n"
"       Inlcude the given upload.\n"
" includedeb <distribution> <.deb-file>\n"
"       Inlcude the given binary package.\n"
" includeudeb <distribution> <.udeb-file>\n"
"       Inlcude the given installer binary package.\n"
" includedsc <distribution> <.dsc-file>\n"
"       Inlcude the given source package.\n"
" list <distribution> <package-name>\n"
"       List all packages by the given name occuring in the given distribution.\n"
" listfilter <distribution> <condition>\n"
"       List all packages in the given distribution matching the condition.\n"
"\n");
			exit(EXIT_SUCCESS);
		case '\0':
			switch( longoption ) {
				case LO_UNIGNORE:
					r = set_ignore(optarg,FALSE,config_state);
					if( RET_WAS_ERROR(r) ) {
						exit(EXIT_FAILURE);
					}
					break;
				case LO_DELETE:
					delete++;
					break;
				case LO_NODELETE:
					delete--;
					break;
				case LO_ONLYACCEPTSIGNED:
					CONFIGSET(onlyacceptsigned,TRUE);
					break;
				case LO_NOONLYACCEPTSIGNED:
					CONFIGSET(onlyacceptsigned,FALSE);
					break;
				case LO_KEEPUNREFERENCED:
					CONFIGSET(keepunreferenced,TRUE);
					break;
				case LO_NOKEEPUNREFERENCED:
					CONFIGSET(keepunreferenced,FALSE);
					break;
				case LO_KEEPUNNEEDEDLISTS:
					CONFIGSET(keepunneededlists,TRUE);
					break;
				case LO_NOKEEPUNNEEDEDLISTS:
					CONFIGSET(keepunneededlists,FALSE);
					break;
				case LO_KEEPDIRECTORIES:
					CONFIGSET(keepdirectories,TRUE);
					break;
				case LO_NOKEEPDIRECTORIES:
					CONFIGSET(keepdirectories,FALSE);
					break;
				case LO_NOTHINGISERROR:
					CONFIGSET(nothingiserror,TRUE);
					break;
				case LO_NONOTHINGISERROR:
					CONFIGSET(nothingiserror,FALSE);
					break;
				case LO_NOLISTDOWNLOAD:
					CONFIGSET(nolistsdownload,TRUE);
					break;
				case LO_LISTDOWNLOAD:
					CONFIGSET(nolistsdownload,FALSE);
					break;
				case LO_ASKPASSPHRASE:
					CONFIGSET(askforpassphrase,TRUE);
					break;
				case LO_NOASKPASSPHRASE:
					CONFIGSET(askforpassphrase,FALSE);
					break;
				case LO_SKIPOLD:
					CONFIGSET(skipold,TRUE);
					break;
				case LO_NOSKIPOLD:
					CONFIGSET(skipold,FALSE);
					break;
				case LO_DISTDIR:
					CONFIGDUP(distdir,optarg);
					break;
				case LO_DBDIR:
					CONFIGDUP(dbdir,optarg);
					break;
				case LO_LISTDIR:
					CONFIGDUP(listdir,optarg);
					break;
				case LO_OVERRIDEDIR:
					CONFIGDUP(overridedir,optarg);
					break;
				case LO_CONFDIR:
					CONFIGDUP(confdir,optarg);
					break;
				case LO_METHODDIR:
					CONFIGDUP(methoddir,optarg);
					break;
				case LO_VERSION:
					fprintf(stderr,"%s: This is " PACKAGE " version " VERSION "\n",programname);
					exit(EXIT_SUCCESS);
				default:
					fprintf (stderr,"Error parsing arguments!\n");
					exit(EXIT_FAILURE);
			}
			longoption = 0;
			break;
		case 'v':
			verbose++;
			break;
		case 'V':
			verbose+=5;
			break;
		case 'f':
			force++;
			break;
		case 'b':
			CONFIGDUP(mirrordir,optarg);
			break;
		case 'i':
			r = set_ignore(optarg,TRUE,config_state);
			if( RET_WAS_ERROR(r) ) {
				exit(EXIT_FAILURE);
			}
			break;
		case 'C':
			if( component != NULL && 
					strcmp(component,optarg) != 0) {
				fprintf(stderr,"Multiple '-C' are not supported!\n");
				exit(EXIT_FAILURE);
			}
			CONFIGDUP(component,optarg);
			break;
		case 'A':
			if( architecture != NULL && 
					strcmp(architecture,optarg) != 0) {
				fprintf(stderr,"Multiple '-A's are not supported!\n");
				exit(EXIT_FAILURE);
			}
			CONFIGDUP(architecture,optarg);
			break;
		case 'T':
			if( packagetype != NULL && 
					strcmp(packagetype,optarg) != 0) {
				fprintf(stderr,"Multiple '-T's are not supported!\n");
				exit(EXIT_FAILURE);
			}
			CONFIGDUP(packagetype,optarg);
			break;
		case 'S':
			if( section != NULL && 
					strcmp(section,optarg) != 0) {
				fprintf(stderr,"Multiple '-S' are not supported!\n");
				exit(EXIT_FAILURE);
			}
			CONFIGDUP(section,optarg);
			break;
		case 'P':
			if( priority != NULL && 
					strcmp(priority,optarg) != 0) {
				fprintf(stderr,"Multiple '-P's are mpt supported!\n");
				exit(EXIT_FAILURE);
			}
			CONFIGDUP(priority,optarg);
			break;
		case '?':
			/* getopt_long should have already given an error msg */
			exit(EXIT_FAILURE);
		default:
			fprintf (stderr,"Not supported option '-%c'\n", c);
			exit(EXIT_FAILURE);
	}
}


int main(int argc,char *argv[]) {
	static struct option longopts[] = {
		{"delete", no_argument, &longoption,LO_DELETE},
		{"nodelete", no_argument, &longoption,LO_NODELETE},
		{"basedir", required_argument, NULL, 'b'},
		{"ignore", required_argument, NULL, 'i'},
		{"unignore", required_argument, &longoption, LO_UNIGNORE},
		{"noignore", required_argument, &longoption, LO_UNIGNORE},
		{"methoddir", required_argument, &longoption, LO_METHODDIR},
		{"distdir", required_argument, &longoption, LO_DISTDIR},
		{"dbdir", required_argument, &longoption, LO_DBDIR},
		{"listdir", required_argument, &longoption, LO_LISTDIR},
		{"overridedir", required_argument, &longoption, LO_OVERRIDEDIR},
		{"confdir", required_argument, &longoption, LO_CONFDIR},
		{"section", required_argument, NULL, 'S'},
		{"priority", required_argument, NULL, 'P'},
		{"component", required_argument, NULL, 'C'},
		{"architecture", required_argument, NULL, 'A'},
		{"type", required_argument, NULL, 'T'},
		{"help", no_argument, NULL, 'h'},
		{"verbose", no_argument, NULL, 'v'},
		{"version", no_argument, &longoption, LO_VERSION},
		{"nothingiserror", no_argument, &longoption, LO_NOTHINGISERROR},
		{"nolistsdownload", no_argument, &longoption, LO_NOLISTDOWNLOAD},
		{"keepunreferencedfiles", no_argument, &longoption, LO_KEEPUNREFERENCED},
		{"keepunneededlists", no_argument, &longoption, LO_KEEPUNNEEDEDLISTS},
		{"keepdirectories", no_argument, &longoption, LO_KEEPDIRECTORIES},
		{"onlyacceptsigned", no_argument, &longoption, LO_ONLYACCEPTSIGNED},
		{"ask-passphrase", no_argument, &longoption, LO_ASKPASSPHRASE},
		{"nonothingiserror", no_argument, &longoption, LO_NONOTHINGISERROR},
		{"nonolistsdownload", no_argument, &longoption, LO_LISTDOWNLOAD},
		{"listsdownload", no_argument, &longoption, LO_LISTDOWNLOAD},
		{"nokeepunreferencedfiles", no_argument, &longoption, LO_NOKEEPUNREFERENCED},
		{"nokeepunneededlists", no_argument, &longoption, LO_NOKEEPUNNEEDEDLISTS},
		{"nokeepdirectories", no_argument, &longoption, LO_NOKEEPDIRECTORIES},
		{"noonlyacceptsigned", no_argument, &longoption, LO_NOONLYACCEPTSIGNED},
		{"noask-passphrase", no_argument, &longoption, LO_NOASKPASSPHRASE},
		{"skipold", no_argument, &longoption, LO_SKIPOLD},
		{"noskipold", no_argument, &longoption, LO_NOSKIPOLD},
		{"nonoskipold", no_argument, &longoption, LO_SKIPOLD},
		{"force", no_argument, NULL, 'f'},
		{NULL, 0, NULL, 0}
	};
	const struct action *a;
	retvalue r;
	int c;

	programname = argv[0];

	init_ignores();

	config_state = CONFIG_OWNER_CMDLINE;

	while( (c = getopt_long(argc,argv,"+fVvhb:P:i:A:C:S:T:",longopts,NULL)) != -1 ) {
		handle_option(c,optarg);
	}
	if( optind >= argc ) {
		fprintf(stderr,"No action given. (see --help for available options and actions)\n");
		exit(EXIT_FAILURE);
	}

	/* only for this CONFIG_OWNER_ENVIRONMENT is a bit stupid,
	 * but perhaps it gets more... */
	config_state = CONFIG_OWNER_ENVIRONMENT;
	if( mirrordir == NULL && getenv("REPREPRO_BASE_DIR") != NULL ) {
		CONFIGDUP(mirrordir,getenv("REPREPRO_BASE_DIR"));
	}
	if( confdir == NULL && getenv("REPREPRO_CONFIG_DIR") != NULL ) {
		CONFIGDUP(confdir,getenv("REPREPRO_CONFIG_DIR"));
	}

	if( mirrordir == NULL ) {
		mirrordir=strdup(STD_BASE_DIR);
	}
	if( confdir == NULL && mirrordir != NULL )
		confdir=calc_dirconcat(mirrordir,"conf");

	if( mirrordir == NULL || confdir == NULL ) {
		(void)fputs("Out of Memory!\n",stderr);
		exit(EXIT_FAILURE);
	}

	config_state = CONFIG_OWNER_FILE;
	optionsfile_parse(confdir,longopts,handle_option);

	/* basedir might have changed, so recalculate */
	if( owner_confdir == CONFIG_OWNER_DEFAULT ) {
		free(confdir);
		confdir=calc_dirconcat(mirrordir,"conf");
	}

	if( methoddir == NULL )
		methoddir = strdup(STD_METHOD_DIR);
	if( distdir == NULL )
		distdir=calc_dirconcat(mirrordir,"dists");
	if( dbdir == NULL )
		dbdir=calc_dirconcat(mirrordir,"db");
	if( listdir == NULL )
		listdir=calc_dirconcat(mirrordir,"lists");
	if( overridedir == NULL )
		overridedir=calc_dirconcat(mirrordir,"override");
	if( distdir == NULL || dbdir == NULL || listdir == NULL 
			|| confdir == NULL || overridedir == NULL || methoddir == NULL) {
		(void)fputs("Out of Memory!\n",stderr);
		exit(EXIT_FAILURE);
	}
	a = all_actions;
	while( a->name != NULL ) {
		if( strcasecmp(a->name,argv[optind]) == 0 ) {
			signature_init(askforpassphrase);
			r = callaction(a,argc-optind,(const char**)argv+optind);
			/* yeah, freeing all this stuff before exiting is
			 * stupid, but it makes valgrind logs easier 
			 * readable */
			signatures_done();
			free(dbdir);
			free(distdir);
			free(listdir);
			free(confdir);
			free(overridedir);
			free(mirrordir);
			free(methoddir);
			free(component);
			free(architecture);
			free(packagetype);
			free(section);
			free(priority);
			if( RET_WAS_ERROR(r) ) {
				if( r == RET_ERROR_OOM )
					fputs("Out of Memory!\n",stderr);
				else if( verbose >= 0 )
					fputs("There have been errors!\n",stderr);
			}
			exit(EXIT_RET(r));
		} else
			a++;
	}

	fprintf(stderr,"Unknown action '%s'. (see --help for available options and actions)\n",argv[optind]);
	signatures_done();
	free(dbdir);
	free(distdir);
	free(listdir);
	free(confdir);
	free(overridedir);
	free(mirrordir);
	free(methoddir);
	free(component);
	free(architecture);
	free(packagetype);
	free(section);
	free(priority);
	exit(EXIT_FAILURE);
}

