# Copyright (c) 2002 Red Hat, Inc. All rights reserved.
#
# This software may be freely redistributed under the terms of the
# GNU General Public License.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
# Application menu and related functions
#
# Written for Red Hat Inc. by Neil Padgett <npadgett@redhat.com>
#
# Component of: Red Hat Database GUI Administration tool

package require Itcl
package require Itk
package require Iwidgets
package provide SQLLogWindow 1.0

proc createSQLLogWindow {} {

    # Create the window

    # Note:
    # The name cannot be .SQLlog as window names may not start with
    # a capital letter

    toplevel .sqllog

    wm withdraw .sqllog

    wm title .sqllog "SQL Transmission Log"

    wm protocol .sqllog WM_DELETE_WINDOW hideSQLLogWindow

    iwidgets::scrolledtext .sqllog.st \
	    -hscrollmode dynamic \
	    -vscrollmode dynamic \
	    -state disabled

    [.sqllog.st component vertsb] configure -takefocus 0

    pack .sqllog.st -fill both -expand yes 

    # Create Clear button
    button .sqllog.clearButton -text "Clear" -underline 4

    # Create Save button 
    button .sqllog.saveButton -text "Save" -underline 0 \
	-command "saveSQLOutput" -state disabled

    # Create close button
    button .sqllog.closeButton -text "Close" -underline 0 -command hideSQLLogWindow

    .sqllog.clearButton configure -command \
	".sqllog.st configure -state normal
	.sqllog.st clear
	.sqllog.st configure -state disabled
	.sqllog.clearButton configure -state disabled
	.sqllog.saveButton configure -state disabled
	" -state disabled

    bind .sqllog <Alt-s> {saveSQLOutput; break}
    bind .sqllog <Alt-c> {hideSQLLogWindow; break}
    bind .sqllog <Alt-r> {.sqllog.st configure -state normal; .sqllog.st clear; .sqllog.st configure -state disabled; .sqllog.clearButton configure -state disabled; .sqllog.saveButton configure -state disabled; break}
    bind .sqllog.clearButton <Return> {.sqllog.st configure -state normal; .sqllog.st clear; .sqllog.st configure -state disabled; .sqllog.clearButton configure -state disabled; .sqllog.saveButton configure -state disabled; break}
    bind .sqllog.saveButton <Return> {saveSQLOutput; break}
    bind .sqllog.closeButton <Return> {hideSQLLogWindow; break}
    bind .sqllog <Escape> {hideSQLLogWindow; break}

    pack .sqllog.st -fill both -expand yes 
    pack .sqllog.clearButton .sqllog.saveButton -side left
    pack .sqllog.closeButton -side right
}

proc showSQLLogWindow {} {
    wm deiconify .sqllog
#    wm transient .sqllog .
}

proc hideSQLLogWindow {} {
    wm withdraw .sqllog
    catch {destroy .saveSQLOutput}
}

proc logSQLStatement { connId target statement } {
    .sqllog.st configure -state normal
    .sqllog.st insert end "$connId:($target) ->\n\t$statement\n\n"
    .sqllog.st configure -state disabled
    .sqllog.st see end

    # enable the clear and save button after queries have been written to
    # the log window
    .sqllog.clearButton configure -state normal
    .sqllog.saveButton configure -state normal
}

proc saveSQLOutput {} {
    # Create the save output window
    toplevel .saveSQLOutput

    # Withdraw the window while we pack it
    wm withdraw .saveSQLOutput

    # Disable the SQL log window
    wm transient .saveSQLOutput .sqllog
    grab .saveSQLOutput

    # Fileselection box to save the output
    iwidgets::fileselectionbox .saveSQLOutput.fileSelection \
        -textbackground white -directory ~/ -childsitepos n

    [[.saveSQLOutput.fileSelection component dirs] component horizsb] \
	configure -takefocus 0
    [[.saveSQLOutput.fileSelection component dirs] component vertsb] \
	configure -takefocus 0

    [[.saveSQLOutput.fileSelection component files] component horizsb] \
	configure -takefocus 0
    [[.saveSQLOutput.fileSelection component files] component vertsb] \
	configure -takefocus 0

    # Create Save/Cancel button
    button .saveSQLOutput.saveButton -text "Save" -underline 0 \
	-command "saveSQLToFile"

    button .saveSQLOutput.cancelButton -text "Cancel" -underline 0 \
        -command "destroy .saveSQLOutput"

    grid configure .saveSQLOutput.fileSelection -row 0 -column 0 -sticky nwe \
        -columnspan 4
    grid configure .saveSQLOutput.saveButton -row 1 -column 2 -sticky nwe
    grid configure .saveSQLOutput.cancelButton -row 1 -column 3 -sticky nwe

    bind .saveSQLOutput <Alt-s> {saveSQLToFile; break}
    bind .saveSQLOutput <Alt-c> {destroy .saveSQLOutput; break}
    bind .saveSQLOutput.saveButton <Return> {saveSQLToFile; break}
    bind .saveSQLOutput.cancelButton <Return> {destroy .saveSQLOutput; break}
    bind .saveSQLOutput <Escape> {destroy .saveSQLOutput; break}

    wm title .saveSQLOutput "Save SQL Queries To File"
    wm deiconify .saveSQLOutput
}


proc saveSQLToFile {} {
    set filename [file nativename [.saveSQLOutput.fileSelection get]]
    if {$filename == ""} {
        tk_messageBox -icon error -title "Error" \
            -message "Please enter a filename" -parent .saveSQLOutput
        return
    }
    # if the string doesn't start with a /, add the directory in front
    # of it
    if {[string index $filename 0] != "/" && [string index $filename 0] != "~"} {
   	set filename "[file dirname \
	    [.saveSQLOutput.fileSelection.filter get]]/$filename"
    }
    if {[file exists $filename]} {
        set overwrite [tk_messageBox -icon warning -title "Overwrite" \
            -message "$filename already exists. Overwrite $filename?" \
            -type yesnocancel -parent .saveSQLOutput -default no]
        if {$overwrite == "no"} {
            return
        }
        if {$overwrite == "cancel"} {
            destroy .saveSQLOutput
            return
        }
    }
    catch {.sqllog.st export $filename} res
    if {$res != ""} {
        tk_messageBox -icon error -title "Error" -message "$res" \
	    -parent .saveSQLOutput
        return
    }  
    destroy .saveSQLOutput
    wm deiconify .sqllog
}

# End of file.
