#ifndef _RHEO_CAD_H
#define _RHEO_CAD_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:cad
NAME: @code{cad} - the boundary definition class
@cindex  boundary geometry
@cindex  Bezier patches
@cindex  RHEOPATH environment variable
@clindex cad
@fiindex @file{.cad}
DESCRIPTION:       
  @noindent
  The @code{cad} class defines a container for the
  description of the boundary of a geometry, by using
  Bezier patches.
 
  @noindent
  The aim is to handle high-level polynomial interpolation
  together with curved boundaries (non-polynomial). So, the
  description bases on Bezier curves and surfaces.

  Also, the adaptive mesh loop requieres interpolation on the
  boundary, and this class should facilitates such procedures.
 
  This class is actually under development.
FILE FORMAT:       
  Under definition.
FILE FORMAT CONVERSION:       
  Under development.
  Conversion to geomview and vtk for visualization.
  Also to and from qmg, grummp, modulef/ghs, opencascade for compatibility.
AUTHORS:
    LMC-IMAG, 38041 Grenoble cedex 9, France
   | Pierre.Saramito@imag.fr
DATE:   28 february 2001
METHODS: @geo
End:
*/

#include "rheolef/cad_domain.h"
#include "rheolef/point.h"
#include "rheolef/occurence.h"
namespace rheolef { 

// --------------------------------------------------------------------
// cad_rep
// --------------------------------------------------------------------
class cad_rep : public occurence {
public:
// typedefs:

	typedef cad_domain::size_type size_type;

// allocator:

	cad_rep();
	cad_rep (const std::string& file_name);

// accessors:

	size_type dimension() const { return _dim; }
	size_type n_face() const;

	const point& xmin() const;
  	const point& xmax() const;

	void  eval (const cad_element& S, const point& ref_coord, point & real_coord) const;
	point eval (const cad_element& S, const point& ref_coord) const;

// input/output:
	
	void put_rheo    (std::ostream&) const;
	void put_geomview(std::ostream&, size_type subdiv, bool adapt_degree) const;

	void get_rheo (std::istream&);
	void get_qmg  (std::istream&);

// graphics:
	
	int gnuplot (std::string basename, bool execute, bool clean, bool verbose, 
		size_type subdiv, bool adapt_degree) const;
	int geomview(std::string basename, bool execute, bool clean, bool verbose, 
		size_type subdiv, bool adapt_degree) const;

// data:
//protected:
        size_type               _dim;
	std::vector<point>	_x;
        point                   _xmin;
        point                   _xmax;

	std::vector<cad_domain>	_p;
	std::vector<cad_domain>	_e;
	std::vector<cad_domain>	_s;
	std::vector<cad_domain>	_v;

	friend int cad_qmg_parse();
};
std::istream& operator >> (std::istream& is, cad_rep& x);
std::ostream& operator << (std::ostream& os, const cad_rep& x);

// --------------------------------------------------------------------
// inlined
// --------------------------------------------------------------------
inline
cad_rep::cad_rep()
  : _dim(0), 
    _x(), 
    _xmin(), 
    _xmax(), 
    _p(), 
    _e(), 
    _s(), 
    _v()
{
}
inline
point
cad_rep::eval (const cad_element& S, const point& ref_coord) const
{
    point real_coord;
    eval (S, ref_coord, real_coord);
    return real_coord;
}
inline
const point& 
cad_rep::xmin() const
{
    return _xmin;
}
inline
const point& 
cad_rep::xmax() const
{
    return _xmax;
}
// --------------------------------------------------------------------
// cad
// --------------------------------------------------------------------
//<cad:
class cad : public smart_pointer<cad_rep> {
public:
// typedefs:

	typedef cad_rep::size_type size_type;


// allocator:

	cad();
	cad (const std::string& file_name);

// accessors:

	size_type dimension() const;
	size_type n_face() const;

	const point& xmin() const;
  	const point& xmax() const;

	void  eval (const cad_element& S, const point& ref_coord, point & real_coord) const;
	point eval (const cad_element& S, const point& ref_coord) const;

// input/output:
	
	friend std::istream& operator >> (std::istream& is, cad& x);
	friend std::ostream& operator << (std::ostream& os, const cad& x);
};
//>cad:

// --------------------------------------------------------------------
// inlined
// --------------------------------------------------------------------
inline
cad::cad ()
  : smart_pointer<cad_rep> (new_macro(cad_rep)) {}

inline
cad::cad(const std::string& file_name)
  : smart_pointer<cad_rep> (new_macro(cad_rep(file_name))) {}

inline
cad::size_type 
cad::dimension() const
{
	return data().dimension();
}
inline
cad::size_type 
cad::n_face() const
{
	return data().n_face();
}
inline
void  
cad::eval (const cad_element& S, const point& ref_coord, point & real_coord) const
{
    data().eval(S, ref_coord, real_coord);
}
inline
point
cad::eval (const cad_element& S, const point& ref_coord) const
{
    return data().eval(S, ref_coord);
}
inline
const point& 
cad::xmin() const
{
    return data().xmin();
}
inline
const point& 
cad::xmax() const
{
    return data().xmax();
}
inline
std::istream&
operator >> (std::istream& s, cad& x)
{
  return s >> x.data();
}
inline
std::ostream&
operator << (std::ostream& s, const cad& x)
{
  return s << x.data();
}
}// namespace rheolef
#endif // _RHEO_CAD_H
