///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is sequential in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/config.h"

#ifdef _RHEOLEF_HAVE_MPI
#include "rheolef/domain.h"
#include "rheolef/geo.h"
#include "rheolef/geo_connectivity.h"

namespace rheolef {

// ----------------------------------------------------------------------------
/// @brief domain_mpi i/o in format version 2
// ----------------------------------------------------------------------------
/**
 Implementation notes:

 The domain is implemented as an array of elements, e.g. edges or faces,
 but it could also be vertices or volume elements, for volumic domains.
 Says a domain of "subgeo" for simplicity. 
 The orientation is important for boundaries or interfaces
 and the direction of the normal has a sense in domains.
 It is described as an array of a pair: an orientation
 and an index of subgeo, denoted in brief as "oisg".
 The index part of the pair, in distributed environment, is denoted
 as "par_isg".
 An index that navigates in this array of pair is denoted
 as ioisg, a short name for an index of (a pair of) orientation and
 index of a subgeo.

 1) Initial numbering ("ini")

 The array of oisg pairs that constitutes the domain contains "par_noisg" entries:
 this array is denoted as "ini_oisg".
 Its distributor is denoted as ini_ioisg_ownership,
 and constructed simply from par_noisg and the communicator obtained from
 the geometry omega.
	par_ini_isg = ini_oisg[ini_ioisg].index

 2) First renumbering ("old")

 Indexes of subgeo (the par_isg part of the pair oisg), that are read from the
 input file, may be changed to a new numbering:
 the subgeos has been renumbered after the partition of the geometry.
 The new subgeo numbering is also used for computations, see e.g. the "space" class.
 The initial numbering is refered in the geometry omega as,
 e.g. par_old_iedg or par_old_ifac, says here par_old_isg.
 The new side numbering is obtained by:
      	par_isg = omega.old_isg2par_isg (old_isg)
 There is a first drawback in distributed environment:
 this call is valid when old_isg is local to the current processor,
 while only the global par_old_isg index is valid. The old_isg are
 distributed according to old_isg_ownership (e.g. old_edge_ownership
 or old_face_ownership), as specified in the geometry omega.
 Thus, the array of osig pairs may be fisrt re-distributed from
 ini_ioisg_ownership to another distributor, denoted as old_ioisg_ownership,
 so that the index part of the pair matches the subgeo old_isg_ownership
 distribution. After redistribution of the array of pairs, these pairs
 have been reordered, i.e. renumbered. The permutation and inverse
 permutation arrays are denoted by:
	ini_ioisg2par_old_ioisg,	according to ini_ioisg_ownership
	old_ioisg2par_ini_ioisg,	according to old_ioisg_ownership
 Also, the initial "ini_oisg" array elements are reordered and re-distributed
 in a new array denoted as "old_oisg". The index of subgeo part of an
 element writes:
	par_old_isg = old_oisg [old_ioisg].index
 It can be converted into the
 local subgeo numbering as "old_isg" and is then suitable for the obtention
 of the new subgeo numbering used for computations
      	par_isg = omega.old_isg2par_isg (old_isg)

 3) Second renumbering (no prefix)

 The par_isg index obtained by this way refers to a subgeo that is not owned in
 general by the current processor.
 So, the array of pairs may be a second time redistributed
 accordingly to the renumbered subgeo, i.e. the "isg_ownership" distributor
 (e.g. edge_ownership or face_ownership), as specified by the geometry omega.

 An array of pairs, containing the orientation and the new subgeo index "par_isg"
 is created and denoted as "tmp_oisg" :
	tmp_oisg[old_ioisg] = pair(orient, par_isg)
 Notice that this array is based on the old_isg_ownership distributor.
 A new distribution follows the distribution of subgeos and is denoted as "oisg":
	oisg[ioisg] = pair(orient,par_isg)
 such that the locally owned part of oisg contains indexes of subgeos
 that are also locally owned by the geometry.
 The permutation and inverse permutation arrays are denoted by:
	old_ioisg2par_ioisg,	according to old_ioisg_ownership
	ioisg2par_old_ioisg,	according to     ioisg_ownership

 The "oisg" array is stored directly in the class domain: domain is 
 implemented as a derived class of the array class which represents "oisg".
 Thus "oisg" is unamed in the domain class.

 4) Back to "ini" renumbering

 In order to write domains into files with the initial ordering and indexes
 of subgeo, we also build the direct permutation and inverse permutation arrays:

 for (old_ioisg=0..) {
    par_ini_ioisg = old_ioisg2par_ini_ioisg [old_ioisg]
    par_ioisg     = old_ioisg2par_ioisg     [old_ioisg]
    ini_ioisg2par_ioisg.set (par_ini_ioisg, par_ioisg)
    ioisg2par_ini_ioisg.set (par_ioisg,     par_ini_ioisg)
 }
 ioisg2par_ini_ioisg.assembly()
 ini_ioisg2par_ioisg.assembly()

 5) Output domains

 for (ioisg=0...)
    par_ini_ioisg = ioisg2par_ini_ioisg [ioisg]
    orient = oisg [ioisg].orient
    isg    = oisg [ioisg].index
    par_old_isg = omega.isubgeo2par_old_isubgeo (isg)
    ini_oisg.set (par_ini_ioisg, pair(orient, par_old_isg)
 }
 ini_oisg.assembly
 ini_oisg.put

*/
template<class T>
iparstream&
domain_mpi_rep<T>::get (
    iparstream&                     ips,
    const geo_basic<T,distributed>& smart_ptr_omega)
{
  warning_macro ("get domain...");
  base::_ptr_omega = &smart_ptr_omega;
  warning_macro ("get domain[0.1]...");
  const geo_mpi_rep<T>& omega = smart_ptr_omega.data();
  warning_macro ("get domain[0.2]...");

  const size_type unset = std::numeric_limits<size_type>::max();
  // ---------------------
  // 1) get initial array
  // ---------------------
  // 1.1) get header
  warning_macro ("get domain [1.1]...");
  communicator comm = omega.comm();
  if ( ! par_scatch (ips, comm, "\ndomain")) {
    warning_macro ("get domain: none");
    ips.is().setstate (std::ios::badbit);
    return ips;
  }
  size_type version, par_noisg;
  ips >> base::_name
      >> version >> base::_dim >> par_noisg;
  check_macro (version == 2, "unsupported version="<<version<<" domain format");

  // 1.2) get data
  warning_macro ("get domain "<<base::_name<<" [1.2]: par_size="<<par_noisg<<"...");
  distributor ini_ioisg_ownership (par_noisg, comm);
  _ini_oisg.resize (ini_ioisg_ownership);
  warning_macro ("get domain "<<base::_name<<" [1.2]: par_size="<<par_noisg<<" size⁼"
		<< ini_ioisg_ownership.size() << "...");
  _ini_oisg.get (ips);

  // ---------------------------
  // 2) first renumbering (old)
  // ---------------------------
  // 2.1) compute old_ownership for each oriented-side
  warning_macro ("get domain "<<base::_name<<" [2.1]: old ownership...");
  distributor old_isg_ownership = omega.subgeo_old_ownership (base::_dim);
  array<size_type>  old_partition (ini_ioisg_ownership, unset);
  for (size_type ini_ioisg = 0, ini_noisg = ini_ioisg_ownership.size();
	ini_ioisg < ini_noisg; ini_ioisg++) {
    size_type old_isg = _ini_oisg [ini_ioisg].index();
    old_partition [ini_ioisg] = old_isg_ownership.find_owner (old_isg);
  }
  // 2.2) old repartition
  warning_macro ("get domain "<<base::_name<<" [2.2]: old repartition...");
  array<domain_pair_type>  old_oisg;
  array<size_type>  ini_ioisg2par_old_ioisg;
  array<size_type>  old_ioisg2par_ini_ioisg;
  _ini_oisg.repartition (
	old_partition,
        old_oisg,
  	old_ioisg2par_ini_ioisg,
  	ini_ioisg2par_old_ioisg);

  // ---------------------
  // 3) first renumbering
  // ---------------------
  // 3.1) subgeo renumbering
  warning_macro ("get domain "<<base::_name<<" [3.1]: subgeo renumbering...");
  distributor old_ioisg_ownership = old_oisg.ownership();
  array<domain_pair_type>  tmp_oisg (old_ioisg_ownership);
  for (size_type old_ioisg = 0, old_noisg = old_ioisg_ownership.size();
	old_ioisg < old_noisg; old_ioisg++) {
    orientation_type orient      = old_oisg [old_ioisg].orientation();
    size_type        par_old_isg = old_oisg [old_ioisg].index();
    size_type        old_isg     = par_old_isg - old_isg_ownership.first_index();
    size_type        par_isg     = omega.old_isubgeo2par_isubgeo (base::_dim, old_isg);
    tmp_oisg [old_ioisg].set (orient, par_isg);
  }
  // 3.2) compute ownership for each oriented-side
  warning_macro ("get domain "<<base::_name<<" [3.2]: ownership...");
  distributor isg_ownership = omega.subgeo_ownership (base::_dim);
  array<size_type>  partition (old_ioisg_ownership, unset);
  for (size_type old_ioisg = 0, old_noisg = old_ioisg_ownership.size();
	old_ioisg < old_noisg; old_ioisg++) {
    size_type isg = tmp_oisg [old_ioisg].index();
    partition [old_ioisg] = isg_ownership.find_owner (isg);
  }
  // 3.3) repartition
  warning_macro ("get domain "<<base::_name<<" [3.3]: repartition...");
#ifdef TO_CLEAN
  array<domain_pair_type>  oisg;
#endif // TO_CLEAN
  array<size_type>  old_ioisg2par_ioisg;
  array<size_type>  ioisg2par_old_ioisg;
  tmp_oisg.repartition (
	partition,
        *this,
  	ioisg2par_old_ioisg,
  	old_ioisg2par_ioisg);

  // -----------------------------
  // 4) Back to "ini" renumbering
  // -----------------------------
  warning_macro ("get domain "<<base::_name<<" [4]: back to ini renumbering...");
  distributor ioisg_ownership = array<domain_pair_type>::ownership();
  _ini_ioisg2par_ioisg.resize (ini_ioisg_ownership, unset);
  _ioisg2par_ini_ioisg.resize (    ioisg_ownership, unset);
  for (size_type old_ioisg = 0, old_noisg = old_ioisg_ownership.size();
	old_ioisg < old_noisg; old_ioisg++) {
    size_type par_ini_ioisg = old_ioisg2par_ini_ioisg [old_ioisg];
    size_type par_ioisg     = old_ioisg2par_ioisg     [old_ioisg];
    _ini_ioisg2par_ioisg.set (par_ini_ioisg, par_ioisg);
    _ioisg2par_ini_ioisg.set (par_ioisg,     par_ini_ioisg);
 }
 _ioisg2par_ini_ioisg.assembly();
 _ini_ioisg2par_ioisg.assembly();

  warning_macro ("get domain "<<base::_name<<" done: status="<<ips.good());
  return ips;
}
template<class T>
oparstream&
domain_mpi_rep<T>::put (
    oparstream&           ops,
    const geo_mpi_rep<T>& omega) const
{
  using namespace std;
  ops << "domain" << endl
      << base::_name << endl
      << "2 " << base::_dim << " " << _ini_oisg.par_size() << endl;
  warning_macro ("put domain "<<base::_name<<"...");
  distributor ioisg_ownership = array<domain_pair_type>::ownership();
  distributor ini_ioisg_ownership = _ini_ioisg2par_ioisg.ownership();
  distributor isg_ownership = omega.subgeo_ownership (base::_dim);
  array<domain_pair_type> ini_oisg (ini_ioisg_ownership);
  for (size_type ioisg = 0, noisg = ioisg_ownership.size();
	ioisg < noisg; ioisg++) {
    size_type par_ini_ioisg  = _ioisg2par_ini_ioisg [ioisg];
    orientation_type orient  = operator[] (ioisg).orientation();
    size_type        par_isg = operator[] (ioisg).index();
    size_type        isg = par_isg - isg_ownership.first_index();
    size_type   par_old_isg = omega.isubgeo2par_old_isubgeo (base::_dim, isg);
    ini_oisg.set (par_ini_ioisg, domain_pair_type (orient, par_old_isg));
  }
  ini_oisg.assembly();
  ini_oisg.put (ops);
  warning_macro ("put domain "<<base::_name<<" done");
  return ops;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class domain_mpi_rep<Float>;

} // namespace rheolef
#endif // _RHEOLEF_HAVE_MPI
