#ifndef _RHEOLEF_ELEMENT_H
#define _RHEOLEF_ELEMENT_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

#include "rheolef/compiler.h"
#include "rheolef/basis.h"
#include "rheolef/numbering.h"
#include "rheolef/geo.h"

namespace rheolef {

/*Class:element
NAME:  @code{element} - the finite element method
@cindex  finite element method
@clindex element
@clindex space
DESCRIPTION:
  Defines the finite element method in use: it is a combination of three
  choosen ingredients: a geometrical element, a polynomial @code{basis}
  (@pxref{basis class})
  and a set of momentums.
  The finite element method is defined on various @code{reference_element}s 
  (@pxref{reference_element class})
  and then
  transformed on any concrete @code{geo_element} of a @code{geo} by using
  the Piola transformation
  (@pxref{geo_element class}, @ref{geo class}).
  The @code{element} class defines also how the basis and momentums are also transformed
  by the Piola transformation.
NOTE:
  This class is under development in the parallel branch of rheolef development.
TO DO:
  The @code{element} class is a virtual class, basis of a hierarchy
  of classes such as @code{lagrange}, @code{hermitte}, etc.
  It is used by the @code{space} class for assembling @code{field}s and @code{form}s
  (@pxref{space class}, @ref{field class}, @ref{form class}).
DATE:   20 dec 2010
End:
*/
template<class T>
class element_rep {
public:
    typedef typename basis::size_type size_type;

    element_rep (std::string name = "");
    virtual ~element_rep () {}

    std::string name() const { return _name; }
    const basis& get_basis() const { return _b; }
    const numbering& get_numbering() const { return _numb; }
    const basis&  get_p1_transformation  () const { return _tr_p1; }  
// data:
protected:
    std::string _name;
    basis       _b; 		// TODO: basis<T> instead of basis
    numbering   _numb;
    basis       _tr_p1;		// TODO: or P2 piola transformation
// internal:
public:
    static element_rep<T>* make_ptr (std::string name);
};
//<verbatim:
template<class T>
class element : public smart_pointer<element_rep<T> > {
public:

// allocators:

    element (std::string name = "");

// accessors:

    std::string name() const;
    const basis& get_basis() const;
    const basis&  get_p1_transformation() const;
};
//>verbatim:
// -----------------------------------------------------------
// inlined
// -----------------------------------------------------------
template<class T>
inline
element<T>::element (std::string name)
 : smart_pointer<element_rep<T> > (element_rep<T>::make_ptr(name))
{
}
template<class T>
inline
std::string
element<T>::name() const
{
    return smart_pointer<element_rep<T> >::data().name();
}
template<class T>
inline
const basis& 
element<T>::get_basis() const
{
    return smart_pointer<element_rep<T> >::data().get_basis();
}
template<class T>
inline
const basis& 
element<T>::get_p1_transformation() const
{
    return smart_pointer<element_rep<T> >::data().get_p1_transformation();
}
// -----------------------------------------------------------
// the lagrange element
// -----------------------------------------------------------
template<class T>
class lagrange : public element_rep<T> {
public:
    lagrange (std::string name) 
     : element_rep<T>(name)
    {
    }
    ~lagrange()
    {
    }
};
// -----------------------------------------------------------
// the P1 Lagrange momentum
// -----------------------------------------------------------
template <class T, class Function>
/* typename Function::result_type */
T
momentum(
    const element<T>& method,
    Function f,
    typename geo_element::size_type loc_idof,
    const geo_element& K,
    const geo_basic<T>& omega)
{
    typedef typename geo_element::size_type size_type;
    size_type par_idof = K [loc_idof];
    size_type idof = par_idof - omega.vertex_ownership().first_index();
    const basic_point<T>& x = omega.vertex (idof);
    return f(x);
}

} // namespace rheolef
#endif // _RHEOLEF_ELEMENT_H
